//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================

// This program converts TrueType fonts to special bitmapped fonts.  These
// bitmap fonts constists (a) of an image file containing the individual
// character glyphs and (b) a simplified font metrics file describing
// the dimension of the characters and their coordinates inside the
// image.
//
// The image file is a regular 32bit BMP, the font metrics file is a text
// file with each line describing one particular character.  Each line
// contains four numbers, separated by white space:
//
//      * character number (ASCII/ISO code of the character)
//	* x coordinate of the character glyph (in pixels)
//	* width of the character glyph (in pixels)
//	* logical width of the character (in pixels)
//
// For a list of command line options, call `ttf2bmf --help'.

#include <SDL/SDL_ttf.h>
#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <stdlib.h>
#include "getopt.h"

using namespace std;

void usage() 
{
    cout << "ttf2bmf [options] file.ttf\n\n"
         << "-o name      basename of the output files\n"
         << "-a           use antialiasing\n"
         << "-c rgb       foreground color as RGB (e.g., 0xff0000 is red)\n"
         << "-s pxsize    size of the generated font in pixels\n"
         << "-h           print this help message\n";
}

int main(int argc, char **argv)
{
    SDL_Color fgcol = {0x0ff,0xff, 0xff};
    SDL_Color bgcol = {0,0,0};
    string ofile;               // basename of the output file
    string ttfname;             // filename of the True Type font
    int ttfsize = 12;

    bool antialias = false;

    int c;

    opterr = 0;
    while ((c = getopt(argc, argv, "ao:c:s:h")) != -1) {
        switch (c) {
        case 'a':
            antialias=true;
            break;
        case 'o':
            ofile = optarg;
            break;
        case 'c':
            { 
                int color;
                sscanf(optarg, "%x", &color);
                fgcol.r = (color >> 16) & 0xff;
                fgcol.g = (color >> 8) & 0xff;
                fgcol.b = color & 0xff;
            }
            break;
        case 's':
            ttfsize = atoi(optarg);
            break;
        case 'h':
            usage();
            return 0;
        case '?':
            cerr << "Unknown option `-" << optopt << "'.\n";
            return 1;
        default:
            abort();
        }
    }
    if (optind + 1 < argc) {
        cerr << "More than one input file specified.\n";
        return 1;
    } else if (optind == argc) {
        cerr << "No input file specified.\n";
        return 1;
    }
    ttfname = argv[optind];

    if (ofile == "") {
        char xx[10];
        sprintf(xx, "%d", ttfsize);
        ofile = string(ttfname, 0, ttfname.find(".ttf"));
        ofile += xx;
    }

    // Initialize True Type library
    if (TTF_Init() < 0) {
	cerr << "cannot initialize SDL_ttf: " << TTF_GetError() << endl;
	return 1;
    }
    TTF_Font *font = TTF_OpenFont(ttfname.c_str(), ttfsize);
    if (!font) {
	cerr << "cannot open font: " << TTF_GetError() << endl;
	return 1;
    }



    ofstream fontinfo((ofile + ".bmf").c_str());

    const int firstc = 32;
    const int lastc = 127;

    char str[2];
    str[1] = 0;

    int width=0, height=0;
    for (int c=firstc; c<=lastc; ++c) {
        int w,h;
        str[0]=c;
        TTF_SizeText(font, str, &w, &h);
	width += w;
    }
    height = TTF_FontHeight(font);
    SDL_Surface *glyphs = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                               width, height, 32, 0,0,0,0);
    SDL_FillRect(glyphs, 0, SDL_MapRGB(glyphs->format,0xff,0,0xff));

    int xoff=0;
    for (int c=firstc; c<=lastc; ++c) 
    {
	SDL_Surface *glyph = 0;
        str[0] = c;
        if (antialias) {
            glyph = TTF_RenderText_Shaded(font, str, fgcol, bgcol);
            SDL_SetColorKey(glyph, SDL_SRCCOLORKEY,0);
        }
        else
            glyph = TTF_RenderText_Solid(font, str, fgcol);
        assert(glyph);

	int minx, maxx, miny, maxy, advance;
	TTF_GlyphMetrics(font, c, &minx, &maxx, &miny, &maxy, &advance);

        SDL_Rect dr;
        dr.x = xoff;
	dr.y = 0;
        dr.w = glyph->w;
        dr.h = glyph->h;

	SDL_BlitSurface(glyph, NULL, glyphs, &dr);
        SDL_FreeSurface(glyph);

        fontinfo << c << " "<<xoff<<" "<<glyph->w<<" " <<advance<<endl;
	xoff += glyph->w;
    }
    SDL_SaveBMP(glyphs, (ofile + ".bmp").c_str());
}
