//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "tools.hh"
#include "px/buffer.hh"
#include "px/tools.hh"

#include <cassert>
#include <iostream>
#include <map>
#include <utility>
#include <list>
#include <algorithm>
#include <functional>

using namespace std;
using namespace tools;

using px::Buffer;
using px::Assert;

 // Value implementation.
Value::Value(const char* str) 
    : type(STRING)
{
    val.str = new char[strlen(str)+1];
    strcpy(val.str, str);
}

void Value::assign(const char* s)
{
    clear();
    type = STRING;
    val.str = new char[strlen(s)+1];
    strcpy(val.str, s);
}

void Value::clear()
{
    if (type == STRING)
	delete[] val.str;
    type = NIL;
}

Value::Value(const Value& other) : type(NIL)
{
    this->operator=(other);
}

Value& 
Value::operator=(const Value& other)
{
    if (this != &other) {
        clear();
        if (other.type == STRING) {
            assign(other.val.str);
        } else {
            type = other.type;
            val = other.val;
        }
    }
    return *this;
}

double Value::get_double() const throw (TypeError)
{
    Assert<TypeError>(type == DOUBLE);
    return val.dval;
}

const char* Value::get_string() const throw (TypeError)
{
    Assert<TypeError>(type == STRING);
    return val.str;
}

Buffer& tools::operator<<(Buffer& buf, const Value& val)
{
    buf << Uint8(val.get_type());

    switch (val.get_type()) {
    case Value::NIL:
        break;
    case Value::DOUBLE:
        buf << val.get_double();
        break;
    case Value::STRING:
        {
            const char* str = val.get_string();
            buf << (Uint16)strlen(str);
            buf.write(str, strlen(str));
        } break;
    case Value::VECTOR3:
        break;
    }
    return buf;
}

Buffer& tools::operator>>(Buffer& buf, Value& val)
{
    Uint8 type = Value::NIL;
    buf >> type;

    switch (type) {
    case Value::NIL:
        // ## fixme
        break;
    case Value::DOUBLE:
        {
            double tmp;
            if (buf >> tmp)
                val = Value(tmp);
        } break;
    case Value::STRING:
        {
            Uint16 len;
            if (buf >> len) {
                char* tmp = new char[len+1];
                tmp[len] = 0;
                if (buf.read(tmp, len))
                    val.assign(tmp);
                delete[] tmp;
            }
        } break;
    }
    return buf;
}

int 
tools::to_int(const Value &v) 
{
    switch (v.get_type()) {
    case Value::DOUBLE: return int(v.get_double());
    case Value::STRING: return atoi(v.get_string());
    default: return 0;
    }
}

bool 
tools::to_bool(const Value &v)
{
    return (v.get_type() != Value::NIL);
}

double 
tools::to_double(const Value &v)
{
    switch (v.get_type()) {
    case Value::DOUBLE: return v.get_double();
    case Value::STRING: return atof(v.get_string());
    default: return 0;
    }
}

const char *
tools::to_string(const Value &v)
{
    switch (v.get_type()) {
    case Value::NIL: return "";
    case Value::DOUBLE: return ""; //v.get_double();
    case Value::STRING: return v.get_string();
    default: return 0;
    }
}


ostream& tools::operator<<(ostream& os, const Value& val)
{
    switch (val.get_type()) {
    case Value::NIL:   os << "nil"; break;
    case Value::DOUBLE: os << val.get_double(); break;
    case Value::STRING: os << val.get_string(); break;
    case Value::VECTOR3: break;
    }
    return os;
}

 // ValueMap implementation.
namespace tools
{
    class ValueMap_Impl {
      public:
        typedef map<string, Value> Map;
        Map bindings;
    };
}

class ValueMap_Itor : public ValueMap::Itor 
{
    typedef ValueMap_Impl::Map Map;
  public:
    ValueMap_Itor(Map& m) : _map(m) {}

    // Itor interface.
    const string* first() {
        _cur = _map.begin();
        return (_cur != _map.end()) ? &(_cur->first) : 0;
    }
    const string* next() {
        ++_cur;
        return (_cur != _map.end()) ? &(_cur->first) : 0;
    }
  private:
    Map&            _map;
    Map::iterator   _cur;
};

ValueMap::ValueMap() : pimpl(new ValueMap_Impl)
{}

ValueMap::~ValueMap()
{
    delete pimpl;
}

void 
ValueMap::assign(const string& name, const Value& val)
{
    pimpl->bindings[name] = val;
}

bool 
ValueMap::boundp(const string& name) const
{
    return pimpl->bindings.find(name) != pimpl->bindings.end();
}

const Value& 
ValueMap::value(const string& name) const
{
    ValueMap_Impl::Map::iterator i = pimpl->bindings.find(name);
    if (i == pimpl->bindings.end())
        throw UnboundName();
    return i->second;
}

ValueMap::Itor* 
ValueMap::itor() const
{
    return new ValueMap_Itor(pimpl->bindings);
}

//----------------------------------------------------------------------
 // Timer
//----------------------------------------------------------------------

Timer::Alarm::Alarm(TimeHandler* h, double i, bool r) 
    : handler(h), interval(i), timeleft(i), repeatp(r) 
{}

void Timer::Alarm::tick(double dtime) 
{
    timeleft -= dtime;
    if (timeleft <= 0) {
        handler->alarm();
        if (repeatp)
            timeleft += interval;
    }
}



void Timer::deactivate(TimeHandler* th) 
{
    handlers.remove(th);
}

void Timer::set_alarm(TimeHandler* th, double interval, bool repeatp) 
{
    if (interval == 0) {
        if (find(handlers.begin(), handlers.end(), th) == handlers.end())
            handlers.push_back(th);
    } else
        alarms.push_back(Alarm(th, interval, repeatp));
}

void Timer::tick(double dtime) 
{
    for_each(handlers.begin(), handlers.end(),
             bind2nd(mem_fun(&TimeHandler::tick), dtime));

    // Explicit loop since `mem_fun_ref' requires a constant member function
    for (list<Alarm>::iterator i = alarms.begin(); i != alarms.end(); ++i)
        i->tick(dtime);
    alarms.remove_if(mem_fun_ref(&Alarm::expired));
}
