//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================

// Class `Drawable' is an abstract interface to nearly everything that
// can be drawn to -- screens, images, etc.  The interface provided is
// very low-level; only a few primitives can be drawn and no clipping
// is performed.
//
// `GC' is a high-level interface to Drawables.  These ``graphics
// contexts'' have an internal state that may influence the appearance
// of drawn object.  The most important of these parameters are the
// clipping rectangle and the current color.

#ifndef PX_DRAWABLE_HH
#define PX_DRAWABLE_HH

#include <SDL/SDL_types.h>
#include "geom.hh"
#include "pxfwd.hh"

namespace px
{
    class Drawable {
    public:
	virtual ~Drawable() {}
    protected:
        friend class GC;

        virtual void lock() = 0;
        virtual void unlock() = 0;

        virtual Uint32 map_color(int r, int g, int b) = 0;
        virtual Uint32 map_color(int r, int g, int b, int a) = 0;

	// Drawable interface.
	virtual void blit(int x, int y, Surface* s) = 0;
	virtual void blit(int x, int y, Surface* s, const Rect& r) = 0;

	virtual Uint32 get_pixel(int x, int y) = 0;
	virtual void set_pixel(int x, int y, Uint32 color) = 0;

	virtual void set_pixels(int n, const int* x, const int* y, Uint32 color);
	virtual void hline(int x, int y, int w, Uint32 color);
	virtual void vline(int x, int y, int h, Uint32 color);
	virtual void box(int x, int y, int w, int h, Uint32 color);
        virtual void line(int x1, int y1, int x2, int y2, Uint32 color);

        virtual Rect size() const = 0;
    };
}

namespace px
{
    class GC {
    public:
        GC(Drawable* d) : drawable(d), cliprect(d->size()), color(0) 
        { xoff = yoff = 0; }
        void clip(const Rect& r) { cliprect.assign(r.x+xoff, r.y+yoff, r.w, r.h); }
        void noclip() { cliprect = drawable->size(); }

        void set_color(int r, int g, int b, int a) 
        {color = drawable->map_color(r, g, b, a);}
        void set_color(int r, int g, int b) 
        {color = drawable->map_color(r, g, b);}

        void set_pixel(int x, int y) {
            x+=xoff; y+=yoff;
            if (cliprect.contains(x, y))
                drawable->set_pixel(x, y, color);
        }
        void set_offset(int xo, int yo) { xoff=xo; yoff=yo; }

        void blit(int x, int y, Surface* s);
        void box(const Rect& r);
        void hline(int x, int y, int w);
        void vline(int x, int y, int h);
        void line(int x1, int y1, int x2, int y2);
    private:
        Drawable*   drawable;
        Rect        cliprect;   // current clipping rectangle
        Uint32      color;      // current color
        int         xoff, yoff;
    };

    inline void clip(GC &gc, const Rect& r) { gc.clip(r); }

    inline void set_color(GC & gc, int r, int g, int b) 
    { gc.set_color(r, g, b); }

    inline void set_color(GC & gc, int r, int g, int b, int a)
    { gc.set_color(r, g, b, a); }

    inline void blit(GC & gc, int x, int y, Surface *s)
    { gc.blit(x, y, s); }

    inline void set_pixel(GC & gc, int x, int y) { gc.set_pixel(x, y); }
    inline void hline(GC & gc, int x, int y, int w) { gc.hline(x, y, w); }
    inline void vline(GC & gc, int x, int y, int h) { gc.vline(x, y, h); }

    void frame(GC & gc, int x, int y, int w, int h);
    inline void frame(GC & gc, const Rect& r) { frame(gc, r.x, r.y, r.w, r.h);}

    inline void box(GC & gc, const Rect& r) { gc.box(r); }
    inline void box(GC & gc, int x, int y, int w, int h)
    { gc.box(Rect(x, y, w, h)); }
}
#endif
