//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#ifndef OBJECTS_HH
#define OBJECTS_HH

#include "fwd.hh"
#include "world.hh"
#include "display.hh"
#include "tools.hh"

#include "px/pxfwd.hh"
#include "px/math.hh"
#include <string>

//
// Object is the base class for all, well, ``objects'' in the world.
// The most important facilities this class provides are:
//
// (1) A way to clone() and dispose() objects.  This is mainly used in
//     function MakeObject to create new objects of a given type.
//
// (2) A way to pass messages between unrelated objects via message().
//     This allows us to send messages to objects from LUA and to
//     decouple objects types as much as possible.
//
// (3) A way to get and set attributes.  These attributes are quite
//     similar to instance variables, but they can be easily modified
//     from LUA.  This makes it possible to modify certain object
//     parameters (such as the text on a piece of paper or the color
//     of an oxyd stone) in level descriptions.
//

namespace world
{
    using tools::Value;

    class UndefinedAttrib {
    public:
        const char *what() const { return "undefined attribute"; }
    };
        
    
    class Object {
    public:
        Object() : kind_(), refcnt(0) {}
        Object(const string& kind);
        virtual ~Object() {}

        // Object interface
        virtual void message(const string& msg, const Value &val) {}
        virtual void set_attrib(const string& key, const Value &val);
        virtual const Value* get_attrib(const string& key) const;

        const char *string_attrib(const string &name) const;
        int int_attrib(const string &name) const;
        double double_attrib(const string &name) const;

        // Increase and decrease reference count
        void incref() { refcnt++; }
        void decref() { if (--refcnt == 0) delete this; }

        virtual Object *clone()=0;
        virtual void dispose()=0;

        const std::string &get_kind() const { return kind_; }
    private:
        std::string kind_;      // the kind of object this is
        tools::ValueMap attribs;
        unsigned refcnt;
    };

    void SendMessage(Object *o, const std::string &msg);
}


// GridObject is the base class for everything that can only be placed
// on "The Grid" (thinking of it, "The Matrix" might have been a funnier name:-)
namespace world
{
    class GridObject : public Object {
    public:
        GridObject() {}
        GridObject(const std::string kind) : Object(kind) {}

        void creation(GridPos p);
        void removal() { on_removal(); }

        virtual bool on_laserhit(Direction d) { return false; }

        // GridObject interface
        virtual void actor_enter(const ActorInfo &ai) {}
        virtual void actor_leave(const ActorInfo &ai) {}

    protected:
        GridPos get_pos() const { return pos; }

    private:
        virtual void on_creation() = 0;
        virtual void on_removal() = 0;
        GridPos pos;
    };
}

//
// Floor
//
namespace world
{
    class Floor : public GridObject {
    public:
        Floor(const string &kind, double friction, double mfactor);

        // Object interface
        Object *clone();
        void dispose();


        // Floor interface
        virtual px::V3 get_force(Actor* a, const px::V3 &mouseforce) 
        { return px::V3(); }

        virtual void on_drop(Item* it) {}
        virtual void on_pickup(Item* it) {}

        virtual void stone_change(Stone *st) {}

        void actor_hit(const ActorInfo &ai) { on_actorhit(ai); }

        double friction() const { return _friction; }
        double mousefactor() const { return _mousefactor; }
    protected:
        Floor() {}
        void set_model(const string& name);
    private:
        // GridObject interface
        virtual void on_creation();
        virtual void on_removal();

        virtual void on_actorhit(const ActorInfo &ai) {}

        double _friction;
        double _mousefactor;
    };
}

//
// Item
//
namespace world
{
    // What may happen to an item _after_ it was being activated?
    enum ItemAction {
        ITEM_DROP, // drop it on the floor
        ITEM_KILL, // remove it from the inventory and kill it
        ITEM_KEEP  // keep it in the inventory; do nothing further
    };

    class Item : public GridObject {
    public:
        Item(const string &kind);

        // Item interface
        virtual void on_drop(Actor* a) {}
        virtual void on_pickup(Actor* a) {}

        virtual void stone_change(Stone *st) {}

        // Return true if the item should be picked up.
        virtual bool actor_hit(const ActorInfo &ai);

        // The model used for displaying this item in an inventory 
        virtual std::string get_inventory_model();

        // Item is activated by the player
        virtual ItemAction activate(Actor *ac, GridPos p)
        { return ITEM_DROP; }

    protected:
        Item() {}
        display::ModelId set_model(const std::string &name);
    private:
        virtual void on_creation();
        virtual void on_removal();

    };
}

//
// Stone
//
namespace world
{
    class Stone : public GridObject {
    public:
        Stone(const string & kind);

        // Object interface
        Object *clone();
        void dispose();

        // Stone interface
        virtual StoneResponse actor_hit(const StoneContact &sc);
        virtual void on_move() {}
    protected:
        Stone() {}
        display::ModelId set_model(const std::string &mname);
        virtual void set_model();

        virtual void on_creation();
        virtual void on_removal();
    };
}

    
//
// Actor
//
namespace world 
{
    class Actor : public Object, public px::Nocopy {
    public:
        // Actor interface.
        virtual void on_hit(Actor* a) {}
        virtual void on_creation(const px::V3 &pos);
        void move(const px::V3& newpos);

        virtual bool is_dead() = 0;

        double get_mass() const { return mass; }
        double get_charge() const { return charge; }
        double get_radius() const { return radius; }

    protected:
        Actor(const string & name, const px::V3& pos_, 
              double radius=0.5, double mass_ = 1, double charge_ = 0);

        void set_model(const string & mname, const px::V3 &pos);
        void set_model(const std::string &);
        void set_attrib(const string &key, const Value &val);

        const px::V3 &get_pos() const { return pos; }
        display::SpriteId get_spriteid() const { return sprite_id; }

    private:
        virtual void on_motion(px::V3 newpos) {}

        double      mass;
        double      charge;
        double      radius;
        px::V3      pos;
        display::SpriteId sprite_id;
    };
}

namespace world
{
    Object *MakeObject(const std::string &kind);
    void DisposeObject(Object *o);

    void recalc_light();
    void recalc_light_now();
}

#endif
