//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "enigma.hh"
#include "objects.hh"
#include "display.hh"
#include "player.hh"
#include "lua.hh"
#include "sound.hh"

#include "px/buffer.hh"
#include "px/tools.hh"

#include <algorithm>
#include <map>
#include <string>
#include <stdlib.h>
#include <iostream>

using namespace std;
using namespace px;
using namespace world;
using namespace enigma;

using tools::Value;

namespace
{
    void maybe_recalc_light(GridPos p);
    bool light_from(GridPos p, Direction d);

    string to_suffix(enigma::Direction d) 
    {
	switch(d) {
	case enigma::NORTH: return "-n";
	case enigma::SOUTH: return "-s";
	case enigma::EAST: return "-e";
	case enigma::WEST: return "-w";
	case enigma::NODIR: return "";
	}
        return "";
    }
    
    player::Inventory *
    get_inventory(const ActorInfo *ai)
    {
        Actor *a = ai->actor;
        if (const Value *v = a->get_attrib("player"))
            return player::GetInventory(to_int(*v));
        return 0;
    }

    bool wielded_item_is(const ActorInfo *ai, const string &kind)
    {
        if (player::Inventory *inv = get_inventory(ai)) 
            if (Item *it = inv->get_item(0))
                return it->get_kind() == kind;
        return false;
    }


    // This function is used by all triggers, switches etc. that
    // perform some particular action when activated (like opening
    // doors or switching lasers on and off).
    void perform_action(Object *o, bool onoff)
    {
        try {
            string action = o->string_attrib("action");
            string target = o->string_attrib("target");
            if (action == "callback") {
                  lua::CallFunc(target, Value(onoff));
            }
            else if (action == "idle")
                ;
            else if (Object *o = GetNamedObject(target)) {
                SendMessage(o, action);
            }
        } 
        catch (UndefinedAttrib &e) {
            cerr << "Trigger: undefined attrib " << endl;
        }
    }
}



//----------------------------------------
// Object impl.
//----------------------------------------

Object::Object(const string& kind)
{ 
    kind_ = kind;
    set_attrib("kind", Value(kind.c_str()));
}

void 
Object::set_attrib(const string& key, const Value& val)
{
    attribs.assign(key, val);
}

const Value*
Object::get_attrib(const string& key) const
{
    try {
        return &attribs.value(key);
    }
    catch (tools::UnboundName) {
        return 0;
    }
}

const char *
Object::string_attrib(const string &name) const
{
    if (const Value *v = get_attrib(name)) {
        const char *s = to_string(*v);
        if (s == 0)
            throw UndefinedAttrib();
        return s;
    } else
        throw UndefinedAttrib();
}

int 
Object::int_attrib(const string &name) const
{
    if (const Value *v = get_attrib(name))
        return to_int(*v);
    else
        throw UndefinedAttrib();
}

double
Object::double_attrib(const string &name) const
{
    if (const Value *v = get_attrib(name))
        return to_double(*v);
    else
        throw UndefinedAttrib();
}

//----------------------------------------
// GridObject
//----------------------------------------
void GridObject::creation(GridPos p) {
    pos = p;
    maybe_recalc_light(p);
    on_creation();
}

namespace
{
    // This interface must be implemented by all items and stones that
    // are capable of emitting light.
    class LaserEmitter {
    public:
	virtual ~LaserEmitter() {}
	virtual DirectionBits emission_directions() const = 0;
    };
}

//----------------------------------------
// PhotoCell
//----------------------------------------
namespace
{
    // Photocells are objects that are sensitive to laser light.
    class PhotoCell {
    public:
        virtual ~PhotoCell() {
            photo_deactivate();
        }

        static void notify_start();
        static void notify_finish();

        virtual void on_recalc_start() = 0;
        virtual void on_recalc_finish() = 0;
    protected:
        void photo_activate() { 
            vector<PhotoCell*>::iterator i;
            i = find(instances.begin(), instances.end(), this);
            if (i != instances.end())
                cerr << "Photocell activated twice\n";
            else
                instances.push_back(this); 
        }
        void photo_deactivate() {
            vector<PhotoCell*>::iterator i;
            i = find(instances.begin(), instances.end(), this);
            if (i != instances.end())
                instances.erase(i);
        }

    private:
        static vector<PhotoCell*> instances;
    };

}

vector<PhotoCell*> PhotoCell::instances;

void
PhotoCell::notify_start() {
    for(unsigned i=0; i<instances.size(); ++i)
        instances[i]->on_recalc_start();
}

void
PhotoCell::notify_finish() {
    for(unsigned i=0; i<instances.size(); ++i)
        instances[i]->on_recalc_finish();
}


//======================================================================
// FLOORS
//======================================================================

Floor::Floor(const string &kind, double friction, double mfactor)
    : GridObject(kind) , _friction(friction)
{
    _mousefactor = mfactor;
}

void Floor::set_model(const string & mname)
{
    SetModel(display::GRID_FLOOR, get_pos().x, get_pos().y, mname);
}

void Floor::on_creation() 
{
    set_model(get_kind());
}

void Floor::on_removal()
{
    KillModel(display::GRID_FLOOR, get_pos().x, get_pos().y);
}

Object *Floor::clone() {
    return new Floor(get_kind(), _friction, _mousefactor);
}

void Floor::dispose() {
    delete this;
}

template <class T> 
class CloneFloor : public virtual Floor {
public:
    Object* clone() { return new T; }
    void dispose() { delete this; }
};


//----------------------------------------
// Abyss
//----------------------------------------
namespace
{
    class Abyss : public CloneFloor<Abyss> {
    public:
        Abyss() : Floor("fl-abyss", 20, 1) {}
    private:
        void actor_enter(const ActorInfo &ai) {
            SendMessage(ai.actor, "fall");
        }
        void on_actorhit(const ActorInfo &ai) {
            SendMessage(ai.actor, "fall");
        }
    };
}

#if 0

class WhiteTile : public Floor {
public:
    V3 get_force(Actor *a, V3 mouseforce) {
        if (a->get_attrib("blackball"))
            return Floor::get_force(a, 0);
        else
            return Floor::get_force(a, mouseforce);
    }
};

class BlackTile : public Floor {
public:
    V3 get_force(Actor *a, V3 mouseforce) {
        if (a->get_attrib("whiteball"))
            return Floor::get_force(a, 0);
        else
            return Floor::get_force(a, mouseforce);
    }
};

// ForceField created by a black hole
class BlackHole_FF : public ForceField {
public:
    BlackHole_FF(const V3 & center) {
    }
    V3 get_force() {
    }
};

class BlackHole : public Item {
public:
    BlackHole() : Item("it-blackhole") {}

    void set_attrib(cstring &name, const Value &val)
    {
        if (name == "warpdest") {
            if (val.get_type() == Value::VECTOR3) {
            }
            else
                cerr << "BlackHole`warpdest must be a vector\n";
        }
    }

    void on_creation() {
        center = make_v3(x+0.5,y+0.5,0);
        ForceField *ff = new BlackHole_FF(center);
        world::AddForceField(ff);
    }
    void on_hit(Actor *a, V3 pos) {
        if (length(pos-center) < 0.2) {
            // play animation
            world::WarpActor(a, warpdest);
        }
    }
private:
    V3 center;
    V3 warpdest;
};


class Spade : public Item {
public:
    Spade();
private:
    ItemAction on_activation(Actor *a, int x, int y)
    {
        if (Item *it = world::GetItem(x,y)) {
            kind = it->get_kind();
            if (kind == "smallmound" || kind=="smallhollow")
                ;
            else if (kind == "mound")
                ;
            else if (kind == "hollow")
                ;
            else
                ; // play annoying sound
            return ITACTION_KEPP;
        }
        else
            return ITACTION_DROP;
    }
    void on_laserhit(int x, int y) {
    }
    void on_stonehit(Stone *st, int x, int y) {
    }
};
#endif


//======================================================================
// ITEMS
//======================================================================

Item::Item(const string & name) 
    : GridObject(name)
{}

void Item::on_creation()
{
    set_model(get_kind());
    if (get_kind() != "it-laserbeam")
        maybe_recalc_light(get_pos());
}

void Item::on_removal()
{
    KillModel(display::GRID_ITEMS, get_pos().x, get_pos().y);
    if (get_kind() != "it-laserbeam")
        maybe_recalc_light(get_pos());
}

display::ModelId
Item::set_model(const string & mname)
{
    return SetModel(display::GRID_ITEMS, get_pos().x, get_pos().y, mname);
}

string 
Item::get_inventory_model() {return get_kind();}

bool Item::actor_hit(const ActorInfo &ai)
{
    const double item_radius = 0.3;
    int x = get_pos().x;
    int y = get_pos().y;
    return (length(ai.pos-make_v3(x+0.5, y+0.5, ai.pos[2])) < item_radius);
}

//----------------------------------------
// Item Mixins
//----------------------------------------
namespace
{
    class StaticItem : public virtual Item {
    public:
        bool actor_hit(const ActorInfo &ai) { return false; }
    };
  
    template <class T> 
    class CloneItem : public virtual Item {
    public:
        Object* clone() { return new T; }
        void dispose() { delete this; }
    };
}

//----------------------------------------
// Document.
//
// This item looks like a piece of paper and contains text messages
// that can be displayed by activating the item.
//
// Attributes:
//
// :text     The message to be displayed
//----------------------------------------

namespace
{
    class Document : public CloneItem<Document> {
    public:
        Document() : Item("it-document") {
            set_attrib("text", "demo");
        }

        ItemAction activate(Actor *a, GridPos)
        {
            display::ShowText(to_string(*get_attrib("text")));
            return ITEM_KILL;
        }
    };
}
//----------------------------------------
// Explosion
//----------------------------------------
namespace
{
    class Explosion : public CloneItem<Explosion>,
                      public display::AnimCallback
    {
    public:
        Explosion() : Item("it-explosion") {}
    private:
        void on_creation() {
            SetCallback(set_model("expl"), this);
        }
        bool actor_hit(const ActorInfo &ai) {
            SendMessage(ai.actor, "shatter");
            return false;
        }
        void animation_finished() {
            world::KillItem(get_pos());
        }
    };
}

//----------------------------------------
// Dynamite.
//----------------------------------------
class Dynamite 
    : public CloneItem<Dynamite>, public display::AnimCallback
{
public:
    Dynamite() : Item("it-dynamite"), state(IDLE) {}
private:
    enum State { IDLE, BURNING };
    State state;

    void change_state(State newstate) {
        if (newstate==BURNING && state==IDLE) {
            state = BURNING;
            SetCallback(set_model("it-dynamite-burning"), this);
        }
    }

    void explode() {
        GridPos p = get_pos();
        SendMessage(GetItem(move(p,NORTH)), "ignite");
        SendMessage(GetItem(move(p,SOUTH)), "ignite");
        SendMessage(GetItem(move(p,EAST)), "ignite");
        SendMessage(GetItem(move(p,WEST)), "ignite");
        SetItem(p, new Explosion);
    }

    void animation_finished() {explode();}
    
    void message(const string &msg, const Value &val) {
        if (msg == "ignite")
            change_state(BURNING);
        else if (msg == "explode")
            explode();
    }

    bool on_laserhit(Direction) {
        change_state(BURNING);
        return false;           // block light
    }

    void on_drop(Actor *a) {change_state(BURNING);}

    bool actor_hit(const ActorInfo &ai) {
        // don't pick up burning dynamite
        return (state == IDLE); 
    }
};


//----------------------------------------
// BlackBomb.
//----------------------------------------
namespace
{
    class BlackBomb 
        : public CloneItem<BlackBomb>, public StaticItem,
          public display::AnimCallback
    {
    public:
        BlackBomb() : Item("it-blackbomb"), state(IDLE) {}

    private:
        enum State { IDLE, BURNING };
        State state;

        void message(const string &msg, const Value &val) {
            if (msg == "ignite")
                change_state(BURNING);
            else if (msg == "explode")
                explode();
        }

        void explode() {
            GridPos p = get_pos();
            SendMessage(GetItem(move(p,NORTH)), "ignite");
            SendMessage(GetItem(move(p,SOUTH)), "ignite");
            SendMessage(GetItem(move(p,EAST)), "ignite");
            SendMessage(GetItem(move(p,WEST)), "ignite");
            SetItem(p, new Explosion);
        }

        void change_state(State newstate) {
            if (newstate==BURNING && state==IDLE) {
                state = BURNING;
                SetCallback(set_model("it-blackbomb-burning"), this);
            }
        }

        void animation_finished() { explode(); }
        bool on_laserhit(Direction) {
            change_state(BURNING);
            return false;           // block light
        }
    };
}

//----------------------------------------
// ExtraLife.
//----------------------------------------
class ExtraLife : public CloneItem<ExtraLife> {
public:
    ExtraLife() : Item("it-extralife") {}
private:
    std::string get_inventory_model() { return string("inv-blackball"); }
};

//----------------------------------------
// Hammer.
//----------------------------------------
class Hammer : public CloneItem<Hammer> {
public:
    Hammer() : Item("it-hammer") {}
};

//----------------------------------------
// MagicWand.
//----------------------------------------
class MagicWand : public CloneItem<MagicWand> {
public:
    MagicWand() : Item("it-magicwand") {}
};

//----------------------------------------
// Trigger.
//----------------------------------------
namespace
{
    class Trigger : public CloneItem<Trigger>, public StaticItem {
        int actorcnt;               // no actors on the trigger
    public:
        Trigger() : Item("it-trigger"), actorcnt(0) {}
    private:
        void actor_enter(const ActorInfo &) 
        {
            if (++actorcnt == 1) {
                set_model("it-trigger1");
                perform_action(this, true);
                sound::PlaySound("it-triggerdown");
            }
        }
        void actor_leave(const ActorInfo &) {
            if (--actorcnt == 0) {
                set_model("it-trigger");
                perform_action(this, false);
                sound::PlaySound("it-triggerup");
            }
        }
    };
}

//----------------------------------------
// LaserBeam
// 
// The implementation of laser beams is a little tricky because, in
// spite of being implemented as Items, lasers aren't localized and
// changes to any part of the beam may affect beam items elsewhere.  A
// `change' may be anything from moving a stone in or out of the beam,
// rotating or moving one of the mirrors, to making a stone in the
// beam transparent.
//
// Here are a couple of facts about laser beams in Enigma:
//
// - Laser beams are static. Once calculated they do not change until
//   they are completely recalculated
//   
// - LaserBeam::emit_from() is the only way to emit laser beams. A new
//   beam will propagate automatically and stops only if it comes
//   across a stone or or item that returns `false' from
//   on_laserhit().
//
// - `on_laserhit()' is called for objects in the beam *whenever*
//   the beam is recalculated.  For objects that need to be notified
//   when the laser goes on or off, use the `PhotoStone'
//   mixin.
//
// More bits of information can be found in the description of laser
// stones and mirrors.
//----------------------------------------
namespace
{
    class LaserBeam : public Item, public LaserEmitter {
    public:
        static void emit_from(GridPos p, Direction d);
        static void kill_all();
	
	// LaserEmitter interface
        DirectionBits emission_directions() const { return directions; }
    private:
        LaserBeam(Direction dir) : Item("it-laserbeam") {
            directions = to_bits(dir);
        }

        bool on_laserhit(Direction dir) {
            DirectionBits dirbit = to_bits(dir);
            if (!(directions & dirbit)) {
                // `dir' not in `directions' ?
                directions = DirectionBits(directions | dirbit);
                emit_from(get_pos(), dir);
                set_model();
            }
            return false;
        }
        void on_creation() { 
            if (directions & EASTBIT) emit_from(get_pos(), EAST);
            if (directions & WESTBIT) emit_from(get_pos(), WEST);
            if (directions &NORTHBIT) emit_from(get_pos(), NORTH);
            if (directions &SOUTHBIT) emit_from(get_pos(), SOUTH);
            set_model(); 
        }
        void set_model();

        void on_removal() {
            KillModel(display::GRID_ITEMS, get_pos().x, get_pos().y);
        }

        bool actor_hit(const ActorInfo &ai);

        Object *clone() {
            // new LaserBeams may only created inside `emit_from'.
            assert(0); return 0; 
        }
        void dispose() {
            instances.erase(find(instances.begin(), instances.end(), this));
            delete this;
        }

        // Variables
        DirectionBits directions;
        static vector<LaserBeam*> instances;
    };
}

vector<LaserBeam*> LaserBeam::instances;

void 
LaserBeam::set_model() {
    string modelname = "it-laser";
    if (directions & (EASTBIT | WESTBIT))
        modelname += "h";
    if (directions & (NORTHBIT | SOUTHBIT))
        modelname += "v";
    Item::set_model(modelname);
}

void
LaserBeam::emit_from(GridPos p, Direction d) 
{
    bool may_pass = true;

    p.move(d);
    if (Stone *st = GetStone(p))
        may_pass = st->on_laserhit(d);

    if (may_pass) {
        Item *it = GetItem(p);
        if (!it) {
            LaserBeam *lb = new LaserBeam(d);
            SetItem(p, lb);
            instances.push_back(lb);
        } else
            it->on_laserhit(d);
    }
}

bool 
LaserBeam::actor_hit(const ActorInfo &ai)
{
    double r = ai.actor->get_radius();
    V3 p = ai.pos;
    GridPos gp = get_pos();

    // distance of actor from center of the grid
    double dx = fabs(p[0] - gp.x - 0.5) - r;
    double dy = fabs(p[1] - gp.y - 0.5) - r;

    if ((directions & (EASTBIT | WESTBIT) && dy<0.1) ||
        (directions & (NORTHBIT | SOUTHBIT)) && dx<0.1)
    {
        SendMessage(ai.actor, "shatter");
    }

    return false; // laser beams can't be picked up
}

void
LaserBeam::kill_all()
{
    while (!instances.empty())
        world::KillItem(instances[0]->get_pos());
}

//----------------------------------------
// Umbrella.
//----------------------------------------
class Umbrella : public CloneItem<Umbrella> {
public:
    Umbrella() : Item("it-umbrella") {}
};

//----------------------------------------
// ShogunDot
//
// Attributes:
//
// :size            1..3  (smallest..largest)
// :target,action   as usual
//----------------------------------------
namespace
{
    class ShogunDot : public CloneItem<ShogunDot>, public StaticItem {
    public:
        ShogunDot() : Item("it-shogun"), activated(false) {
            set_size(1);
        }
    private:
        void set_size(int s) { set_attrib("size", s); }
        int get_size() const { return int_attrib("size"); }

        void on_creation();

        void message(const string &str, const Value &v);
        void stone_change(Stone *st) {
            if (activated && st == 0) {
                activated = false;
                perform_action(this, false);
            } 
        }

        bool activated;
    };
}

void
ShogunDot::on_creation() {
    switch(get_size()) {
    case 1: Item::set_model("it-shogun1"); break;
    case 2: Item::set_model("it-shogun2"); break;
    case 3: Item::set_model("it-shogun3"); break;
    default: assert(0);
    }
}

void 
ShogunDot::message(const string &str, const Value &v) 
{
    int size=get_size();
    if (activated) {
        if (str=="noshogun") {
            activated=false;
            perform_action(this, false);
        }
    }
    else {
        if ((size==1 && str == "shogun1") ||
            (size==2 && str == "shogun2") ||
            (size==3 && str == "shogun3"))
        {
            activated=true;
            perform_action(this, true);
        }
    }
}

//======================================================================
// STONE MIXINS
//======================================================================

//----------------------------------------
// CloneStone
//----------------------------------------
namespace
{
    template <class T>
    class CloneStone : public virtual Stone {
    public:
        Object *clone() { return new T; }
        void dispose() { delete this; }
    };
}

//----------------------------------------
// MovableStone
//
// This mixin class tries to move the stone to a neighboring field
// when hit hard enough.
//----------------------------------------
namespace
{
    Direction contact_face(const StoneContact &sc)
    {
        if (sc.normal == make_v3(-1,0,0)) 
            return WEST;
        else if (sc.normal == make_v3(1,0,0)) 
            return EAST;
        else if (sc.normal == make_v3(0,-1,0))
            return NORTH;
        else if (sc.normal == make_v3(0,1,0))
            return SOUTH;
        else 
            return NODIR;
    }

    class MovableStone : public virtual Stone {
    public:
        StoneResponse actor_hit(const StoneContact &sc)
        {
            const ActorInfo &ai = *sc.actorinfo;
            
            // If the speed component towards the face of the stone is large
            // enough (and actually pointing towards the stone), consider
            // moving the stone.
            if (ai.vel * sc.normal < -4)
                MaybeMoveStone(get_pos(), reverse(contact_face(sc)));
            return STONE_REBOUND;
        }
    };
}
    

//----------------------------------------
// PhotoStone
//
// Most stones are indifferent to laser beams: They either block the
// light completely or they let it pass, but they do not change their
// internal state when they are hit by light.  Certain kinds of stones
// need to be notified whenever the `light' goes on or off -- these
// can be derived from this mixin class.
//----------------------------------------
namespace
{
    class PhotoStone : public virtual Stone, public PhotoCell {
    public:
        PhotoStone() {illuminated=false;}
        virtual ~PhotoStone() {}
    private:
        bool illuminated;

        void on_recalc_start() {}
        void on_recalc_finish() { check_state(); }

        void check_state() {
            GridPos p = get_pos();
            bool illu = (light_from(p, NORTH) | light_from(p, EAST) 
                         | light_from(p, WEST) | light_from(p, SOUTH));
            if (illu != illuminated) {
                if (illu)
                    notify_laseron();
                else
                    notify_laseroff();
                illuminated = illu;
            }
        }

        virtual void notify_laseron() = 0;
        virtual void notify_laseroff() = 0;
    };
}

//======================================================================
 // STONES
//======================================================================


Stone::Stone(const string & kind) 
    : GridObject(kind) 
{}

void Stone::on_creation() {
    set_model();
}

void Stone::set_model() {
    set_model(get_kind());
}

display::ModelId 
Stone::set_model(const string & mname)
{
    return SetModel(display::GRID_STONES, 
                    get_pos().x, get_pos().y, mname);
}

void Stone::on_removal()
{
    KillModel(display::GRID_STONES, get_pos().x, get_pos().y);
    maybe_recalc_light(get_pos());
}

Object *Stone::clone()
{
    return new Stone(get_kind());
}

void Stone::dispose()
{
    delete this;
}

StoneResponse 
Stone::actor_hit(const StoneContact &sc)
{
    return STONE_REBOUND;
}

//----------------------------------------
// One Way Stone
//
// This stone can only be passed in one direction. (To be exact, it
// can be left in all directions, but can only be entered through one
// of the four faces of the stone).
//----------------------------------------
namespace
{
    class OneWayStone : public CloneStone<OneWayStone> {
    public:
	OneWayStone() : Stone("st-oneway") {
	    set_orientation(SOUTH);
	}
    private:
	void set_model() {
	    Stone::set_model(string("st-oneway") 
			     + to_suffix(get_orientation()));
	}
	Direction get_orientation() const {
	    return Direction(int_attrib("orientation"));
	}
        void set_orientation(Direction dir) {
            set_attrib("orientation", Value(dir));
        }
	StoneResponse actor_hit(const StoneContact&);
    };
}

StoneResponse
OneWayStone::actor_hit(const StoneContact &sc)
{
    Direction d=contact_face(sc);
    Direction o=get_orientation();
    if (d == reverse(o))
	return STONE_PASS;
    
    if (d == o) {
        // magic wand as first item?
        if (wielded_item_is(sc.actorinfo, "it-magicwand")) {
            set_orientation(reverse(o));
            set_model();
        }
    }
    return STONE_REBOUND;
}

//----------------------------------------
// Chameleon Stone
//
// This stone takes on the look of the floor beneath it.  Actors can
// move through it, so these stones are perfect for hiding stuff under
// them...
//----------------------------------------
namespace
{
    class ChameleonStone : public CloneStone<ChameleonStone> {
    public:
        ChameleonStone() : Stone("st-chameleon"){}
    private:
        void set_model() {
            string modelname = "fl-gray";
            if (Floor *fl = GetFloor(get_pos()))
                modelname = fl->get_kind();
            Stone::set_model(modelname);
        }
        StoneResponse actor_hit(const StoneContact &sc) {
            return STONE_PASS;
        }
    };
}

//----------------------------------------
// SwapStone
//
// This stone can exchange its position with other neighboring stones
// if it is hit hard enough.  In a way, this makes swap stones a kind
// of "movable stone", except that they can be only exchanged with
// other stones and may not be moved on empty fields.
//----------------------------------------
namespace
{
    class SwapStone : public CloneStone<SwapStone> {
    public:
        SwapStone() : Stone("st-swap") {}

        StoneResponse actor_hit(const StoneContact &sc)
        {
            const ActorInfo &ai = *sc.actorinfo;
            Direction hit_dir = reverse(contact_face(sc));

            if (ai.vel * sc.normal < -4 && hit_dir!=NODIR) {
                GridPos p = get_pos();
                GridPos newp = move(p, hit_dir);
                if (GetStone(newp))
                    SwapStones(p,newp);
            }
            return STONE_REBOUND;
        }
    };
}

//----------------------------------------
// Brownie
//----------------------------------------
namespace
{
    class Brownie : public CloneStone<Brownie>, public MovableStone {
    public:
        Brownie() : Stone("st-brownie") {}
        StoneResponse actor_hit(const StoneContact &sc) {
            sound::PlaySound("st-thud");
            return MovableStone::actor_hit(sc);
        }
    };
}

//----------------------------------------
// PuzzleStones
//
// These stones can be connected to other stones of the same type.
// Any of the four faces of the stone can have ``socket''.  If the
// touching faces of two adjacent stones both happen to have a socket,
// these two stones link up and henceforth move as group.  
//
// A cluster of puzzle stones may for example look like this:
// 
// +---+---+---+---+
// |   |   |   |   |
// | --+-+-+---+-+ |
// |   | | |   | | |
// +---+-+-+---+-+-+
//     | | |   | | |
//     | | |   | | |
//     |   |   |   |
//     +---+   +---+
//
// This example actually presents the special case of a ``complete''
// cluster.  A cluster is complete if none of its stones has an
// unconnected socket.  If such a group is touched with a magic wand,
// all its constituents explode.
//
// Attributes:
//
// :connections     1..16; each bit in (connections-1) corresponds
//                  to a socket on one of the four faces.  You will
//                  normally simply use one of the Lua constants
//                  PUZ_0000 to PUZ_1111.
//----------------------------------------
namespace
{
    class PuzzleStone : public Stone {
    public:
        PuzzleStone() : Stone("st-puzzle") {
            set_attrib("connections", 0.0);
        }
    private:
        typedef vector<PuzzleStone*> InstanceList;
        typedef vector<GridPos> Cluster;

        DirectionBits get_connections() {
            int conn=int_attrib("connections") - 1;
            if (conn >=0 && conn <16)
                return DirectionBits(conn);
            else
                return NODIRBIT;
        }

        bool visited;           // flag for DFS
        static InstanceList instances;

        static bool visit_dir(vector<GridPos> &stack, 
                              GridPos curpos, Direction dir);
        bool find_cluster(Cluster &);
        void find_row_or_column_cluster(Cluster &c, 
                                        GridPos startpos, Direction dir);
        bool cluster_complete();
        void maybe_move_cluster(Cluster &c, Direction dir);
        void rotate_cluster(const Cluster &c);

        void set_model();
        StoneResponse actor_hit(const StoneContact &sc);

        Object *clone() {
            PuzzleStone *o = new PuzzleStone;
            instances.push_back(o);
            return o;
        }
        void dispose() {
            instances.erase(find(instances.begin(), instances.end(), this));
            delete this;
        }
    };
}

PuzzleStone::InstanceList PuzzleStone::instances;

bool
PuzzleStone::visit_dir(vector<GridPos> &stack, GridPos curpos, Direction dir)
{
    GridPos newpos = move(curpos, dir);
    PuzzleStone *pz = dynamic_cast<PuzzleStone*>(GetStone(newpos));
    
    if (!pz)
        return false;


    if (has_dir(pz->get_connections(), reverse(dir))) {
        // Puzzle stone at newpos is connected to stone at curpos
        if (!pz->visited)
            stack.push_back(newpos);
        return true;
    } else {
        // The two stones are adjacent but not connected
        return false;
    }
}

// Use a depth first search to determine the group of all stones that
// are connected to the current stone.  Returns true if the cluster is
// ``complete'' in the sense defined above.
bool
PuzzleStone::find_cluster(Cluster &cluster)
{
    for (unsigned i=0; i<instances.size(); ++i)
        instances[i]->visited=false;

    vector<GridPos> pos_stack;
    bool is_complete = true;
    pos_stack.push_back(get_pos());
    while (!pos_stack.empty()) {
        GridPos curpos = pos_stack.back();
        pos_stack.pop_back();

        PuzzleStone *pz = dynamic_cast<PuzzleStone*>(GetStone(curpos));
        assert(pz);

        cluster.push_back(curpos);
        pz->visited = true;
        DirectionBits cfaces = pz->get_connections();

        if (has_dir(cfaces, NORTH))
            is_complete &= visit_dir(pos_stack, curpos, NORTH);
        if (has_dir(cfaces, EAST))
            is_complete &= visit_dir(pos_stack, curpos, EAST);
        if (has_dir(cfaces, SOUTH))
            is_complete &= visit_dir(pos_stack, curpos, SOUTH);
        if (has_dir(cfaces, WEST))
            is_complete &= visit_dir(pos_stack, curpos, WEST);
    }
    return is_complete;
}

void
PuzzleStone::find_row_or_column_cluster(Cluster &c, GridPos startpos, Direction dir)
{
    GridPos p = startpos;
    while (dynamic_cast<PuzzleStone*>(GetStone(p))) {
        c.push_back(p);
        p.move(dir);
    }
}

void 
PuzzleStone::maybe_move_cluster(Cluster &c, Direction dir)
{
    // First remove all stones in the cluster from the world
    vector<Stone*> clusterstones;
    for (unsigned i=0; i<c.size(); ++i) {
        clusterstones.push_back(YieldStone(c[i]));
        c[i] = move(c[i], dir);
    }

    // Now check whether all stones can be placed at their new
    // position
    bool move_ok = true;
    for (unsigned i=0; i<c.size(); ++i) {
        if (GetStone(c[i]) != 0)
            move_ok = false;
    }

    // Now either move all the stones or place them at their old
    // positions
    for (unsigned i=0; i<c.size(); ++i) {
        if (move_ok)
            SetStone(c[i], clusterstones[i]);
        else
            SetStone(move(c[i], reverse(dir)), clusterstones[i]);
    }
    if (move_ok) 
        sound::PlaySound("st-move");
}

bool 
PuzzleStone::cluster_complete()
{
    Cluster c;
    return find_cluster(c);
}

void 
PuzzleStone::set_model()
{
    string modelname = "st-puzzle";
    char x[10];

    // every combination of ``sockets'' has its own model
    int modelno = int_attrib("connections");
    sprintf(x, "%d", modelno);
    Stone::set_model(modelname+x);
}

void
PuzzleStone::rotate_cluster(const Cluster &c)
{
    if (c.size() > 1) {
        vector<Stone*> stones;
        for (unsigned i=0; i<c.size(); ++i)
            stones.push_back(YieldStone(c[i]));

        // exclude last list item
        for (unsigned i=0; i<c.size()-1; ++i)
            SetStone(c[i+1], stones[i]);
        SetStone(c[0], stones.back());
    }
}

StoneResponse 
PuzzleStone::actor_hit(const StoneContact &sc)
{
    const ActorInfo &ai = *sc.actorinfo;

    bool move_impulse= (ai.vel*sc.normal < -4);
    Direction hit_dir=reverse(contact_face(sc));

    if (wielded_item_is(&ai, "it-magicwand")) {
        if (cluster_complete()) {
            // make all cluster stones explode
            return STONE_REBOUND;
        } 
        else if (hit_dir != NODIR) {
            Cluster c;
            find_row_or_column_cluster(c, get_pos(), hit_dir);

            if (move_impulse)
                maybe_move_cluster(c, hit_dir);
            else
                rotate_cluster(c);
        }
    } 
    else if (move_impulse && hit_dir != NODIR) {
        Cluster c;
        find_cluster(c);
        maybe_move_cluster(c, hit_dir);
    }

    return STONE_REBOUND;
}


//----------------------------------------
// Door
//
// Attributes:
//
// :type        h or v for a door that opens horizontally or vertically
//----------------------------------------
namespace
{
    class Door : public CloneStone<Door>, public display::AnimCallback {
    public:
        Door() : Stone("st-door") {
            set_attrib("type", "h");
            state = CLOSED;
        }
    private:
        enum State { OPEN, CLOSED, OPENING, CLOSING };
        State state;
        display::ModelId model_id;

        string get_type() const {
            return string_attrib("type");
        }

        void animation_finished();

        void set_model();
        bool on_laserhit(Direction dir) {
            if (get_type() == "h")
                return state==OPEN || dir==EAST || dir==WEST;
            else
                return state==OPEN || dir==NORTH || dir==SOUTH;
        }

        void message(const string &m, const Value &);
        void change_state(State newstate);
        StoneResponse actor_hit(const StoneContact &sc);
    };
}

void 
Door::set_model() {
    string mname = string("st-door")+get_type();
    if (state == CLOSED) 
        mname += "-closed";
    else if (state==OPEN)
        mname += "-open";
    Stone::set_model(mname);
}

void 
Door::animation_finished()
{
    if (state == OPENING)
        change_state(OPEN);
    else if (state == CLOSING)
        change_state(CLOSED);
}

void 
Door::message(const string &m, const Value &)
{
    if (m == "open" && (state==CLOSED || state==CLOSING)) 
        change_state(OPENING);
    else if (m=="close" && (state==OPEN || state==OPENING))
        change_state(CLOSING);
    else if (m=="openclose") {
        if (state==OPEN || state==OPENING)
            change_state(CLOSING);
        else 
            change_state(OPENING);
    }
}

void 
Door::change_state(State newstate) 
{
    string basename = string("st-door") + get_type();
    switch (newstate) {
    case OPEN: 
        Stone::set_model(basename+"-open"); 
        maybe_recalc_light(get_pos());
        break;
    case CLOSED: 
        Stone::set_model(basename+"-closed"); 
        maybe_recalc_light(get_pos());
        break;
    case OPENING: 
        if (state == CLOSING)
            ReverseAnimation(model_id);
        else {
            model_id = Stone::set_model(basename+"-opening");
            SetCallback(model_id, this);
        }
        break;
    case CLOSING:
        if (state == OPENING)
            ReverseAnimation(model_id);
        else {
            model_id = Stone::set_model(basename+"-closing");
            SetCallback(model_id, this);
        }
        break;
    }
    state = newstate;
}


StoneResponse 
Door::actor_hit(const StoneContact &sc)
{
    Direction cf = contact_face(sc);
    if (state == OPEN)
        return STONE_PASS;
    else {
        string t = get_type();
        if ((t == "v" && (cf==WEST || cf==EAST)) ||
            (t == "h" && (cf==SOUTH || cf==NORTH)))
        {
            return STONE_REBOUND;
        } else
            return STONE_PASS;
    }
}


//----------------------------------------
// TimerStone
//
// Attributes:
//
// :interval        seconds between two "ticks"
// :action,target   as usual
//----------------------------------------
namespace
{
    class TimerStone : public CloneStone<TimerStone> {
    public:
        TimerStone() : Stone("st-timer") {
            set_attrib("interval", 1.0);
            set_attrib("on", 1.0);
        }
    private:
        double get_interval() const { return double_attrib("interval"); }

        void message(const string &m, const Value &) {
            
        }
    };

}

//----------------------------------------
// Switch
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------
namespace
{
    class SwitchStone : public CloneStone<SwitchStone> {
    public:
        SwitchStone() : Stone("st-switch") {
            set_attrib("on", false);
        }
    private:
        bool is_on() { return int_attrib("on")==1; }

        void set_model() {
            Stone::set_model(is_on() ? "st-switch1" : "st-switch0");
        }

        StoneResponse actor_hit(const StoneContact &sc) {
            sound::PlaySound("st-metal");
            set_attrib("on", !is_on());
            perform_action(this, is_on());
            set_model();
            return STONE_REBOUND;
        }
    };
}

//----------------------------------------
// ShogunStone
//
// Attributes:
//
// :holes            1..7
//----------------------------------------
namespace
{
    class ShogunStone : public CloneStone<ShogunStone> {
    public:
        ShogunStone() : Stone("st-shogun") {
            set_holes(SMALL);
        }
    private:
        enum Holes { SMALL=1, MEDIUM=2, LARGE=4 };

        void set_holes(Holes h) { set_attrib("holes", h); }
        Holes get_holes() const { 
            int h=int_attrib("holes"); 
            if (h>=1 && h<=7)
                return Holes(h);
            else {
                cerr << "Wrong `holes' attribute\n";
                return SMALL;
            }
        }

        void notify_item();

        void add_hole(Holes h) {
            set_attrib("holes", get_holes() | h);
            notify_item();
            set_model();
        }

        void on_creation() { 
            set_model(); 
            notify_item();
        }

        void set_model() {
            char x[20];
            sprintf(x, "st-shogun%d", int(get_holes()));
            Stone::set_model(x);
        }

        static Holes smallest_hole(Holes s) {
            if (s & SMALL)
                return SMALL;
            if (s & MEDIUM)
                return MEDIUM;
            if (!(s & LARGE))
                assert(0);
            return LARGE;
        }

        Holes transfer_smallest_hole(GridPos p);
        StoneResponse actor_hit(const StoneContact &sc);
    };
}

void 
ShogunStone::notify_item() 
{
    if (Item *it=GetItem(get_pos())) {
        switch (get_holes()) {
        case SMALL:
            SendMessage(it,"shogun1");
            break;
        case (MEDIUM | SMALL):
            SendMessage(it,"shogun2");
            break;
        case (LARGE | MEDIUM | SMALL):
            SendMessage(it,"shogun3");
            break;
        default:
            SendMessage(it, "noshogun");
            break;
        }
    }
}

ShogunStone::Holes
ShogunStone::transfer_smallest_hole(GridPos p)
{
    bool hole_transferred=false;
    Holes h = get_holes();
    Holes sh = smallest_hole(h);


    if (Stone *st=GetStone(p)) {
        if (ShogunStone *sst = dynamic_cast<ShogunStone*>(st)) {
            // Transfer the smallest hole to the other shogun stone.
            // Holes must be ``stacked'' properly
            if (sh < smallest_hole(sst->get_holes())) {
                sst->add_hole(sh);
                hole_transferred=true;
            }
        }
    }
    else {
        // empty field? Simply create a new stone on it
        ShogunStone *sst = new ShogunStone();
        sst->set_holes(sh);
        SetStone(p, sst);
        hole_transferred=true;
    }

    return (hole_transferred) ? Holes(h & ~sh) : h;
}

StoneResponse 
ShogunStone::actor_hit(const StoneContact &sc)
{
    const ActorInfo &ai = *sc.actorinfo;
    bool move_impulse= (ai.vel*sc.normal < -4);
    Direction hit_dir=reverse(contact_face(sc));

    if (hit_dir == NODIR || !move_impulse)
        return STONE_REBOUND;

    GridPos newpos=move(get_pos(), hit_dir);
    Holes holes = get_holes();
    Holes newholes = transfer_smallest_hole(newpos);

    if (newholes!=holes) {
        sound::PlaySound("st-move");
        if (newholes == 0)
            KillStone(get_pos());
        else {
            set_holes(newholes);
            notify_item();
            set_model();
        }
    }
    return STONE_REBOUND;
}


//----------------------------------------
// FakeOxydStone
//
// These look like real oxyd stones, but they only blink a little when
// touched and do not open or have other special abilities.
//----------------------------------------
namespace
{
    class FakeOxydStone : public CloneStone<FakeOxydStone>,
                          public display::AnimCallback
    {
    public:
        FakeOxydStone() : Stone("st-fakeoxyd"), state(IDLE) {}
    private:
        enum State { IDLE, BLINKING } state;
        StoneResponse actor_hit(const StoneContact &sc) {
            if (state == IDLE) {
                SetCallback(set_model("st-fakeoxyd-blink"), 
                            this);
                state = BLINKING;
            }
            sound::PlaySound("st-fakeoxyd");
            return STONE_REBOUND;
        }
        void animation_finished() {
            set_model("st-fakeoxyd");
            state = IDLE;
        }
    };
}

//----------------------------------------
// OxydStone
//
// Oxyd stones are characterized by two attributes: Their flavor and
// their color.  The flavor only affects the visual representation of
// the stone; it can be either 'a' (opening like a flower) or 'b'
// (displaying a fade-in animation).  The color attribute determines
// the shape on the oxyd stone.
//
// [only flavor 'b' is currently implemented]
//
// Attributes:
//
// :flavor      "a" or "b"
// :color       number between 0 and 7
//----------------------------------------

namespace
{
    class OxydStone : public PhotoStone, public display::AnimCallback 
    {
    public:
        OxydStone();

    private:
        enum State { CLOSED, OPEN, OPENING, CLOSING, BLINKING };
        State state;
        display::ModelId openclose_model;
    
        typedef vector<OxydStone*> InstanceList;
        static InstanceList instances;

        static bool blinking_or_opening(OxydStone *a) { 
            return (a->state==BLINKING || a->state==OPENING); 
        }
        static bool not_open(OxydStone *a) {
            return a->state!=OPEN;
        }

        // Stone interface
        StoneResponse actor_hit(const StoneContact &sc);
        void on_creation();
        void on_removal();

        // Object interface
        Object *clone();
        void dispose();

        // AnimCallback interface
        void animation_finished();

        void maybe_open_stone();
        void change_state(State newstate);

        void notify_laseron() { maybe_open_stone(); }
        void notify_laseroff() {}
    };
}

vector<OxydStone*> OxydStone::instances;

OxydStone::OxydStone() 
    : Stone("st-oxyd"), state(CLOSED)
{
    set_attrib("flavor", "b");
    set_attrib("color", "0");
}

void
OxydStone::change_state(State newstate)
{
    string modelname = "st-oxyd";
    modelname += string_attrib("flavor");
    modelname += string_attrib("color");

    switch (newstate) {
    case CLOSED:
        set_model(string("st-oxyd")+string_attrib("flavor"));
        break;

    case BLINKING:
        set_model(modelname + "-blink");
        break;
	
    case OPEN:
	if (state == CLOSED) {
            sound::PlaySound("st-oxydopen");
            sound::PlaySound("st-oxydopened");
            SetCallback(set_model(modelname+"-opening"), this);
        } else {
            set_model(modelname + "-open");

            // If this was the last closed oxyd stone, finish the
            // level
            if (find_if(instances.begin(),instances.end(),not_open)
                ==instances.end())
            {
                enigma::FinishLevel();
            }
        }
        break;
	
    case OPENING:
        sound::PlaySound("st-oxydopen");
	if (state == CLOSED) {
            openclose_model = set_model(modelname + "-opening");
            SetCallback(openclose_model, this);
	}
	else if (state == CLOSING)
            ReverseAnimation(openclose_model);
	break;
	
    case CLOSING:
        if (state == OPENING)
            ReverseAnimation(openclose_model);
	else if (state == BLINKING) {
            openclose_model = set_model(modelname + "-closing");
            SetCallback(openclose_model, this);
	}
        break;
    }
    state = newstate;
}

void OxydStone::animation_finished()
{
    if (state == CLOSING)
        change_state(CLOSED);
    else if (state == OPENING)
        change_state(BLINKING);
    else if (state == OPEN)
        change_state(OPEN); // set the right model
}

void
OxydStone::maybe_open_stone()
{
    if (state == CLOSED || state == CLOSING) {
        int mycolor = int_attrib("color");

        // Is another oxyd stone currently blinking?
        InstanceList::iterator i;
        i=find_if(instances.begin(), instances.end(), blinking_or_opening);

        if (i != instances.end()) {
            // If colors match, open both stones. Close one of them
            // otherwise
            if (mycolor == (*i)->int_attrib("color")) {
                change_state(OPEN);
                (*i)->change_state(OPEN);
            } else {
                (*i)->change_state(CLOSING);
                change_state(OPENING);
            }
        } 
        else {
            // no blinking stone? -> make this one blink
            change_state(OPENING);
        }
    }
}

StoneResponse 
OxydStone::actor_hit(const StoneContact &sc)
{
    maybe_open_stone();
    sound::PlaySound("st-metal");
    return STONE_REBOUND;
}

void OxydStone::on_creation()
{
    string modelname = "st-oxyd";
    modelname += string_attrib("flavor");
//    modelname += string_attrib("color");
    set_model(modelname);
    photo_activate();
}

void OxydStone::on_removal()
{
    photo_deactivate();
    Stone::on_removal();
}

Object *OxydStone::clone() 
{
    OxydStone *o = new OxydStone;
    instances.push_back(o);
    return o;
}

void OxydStone::dispose() 
{
    instances.erase(find(instances.begin(),
                         instances.end(), this));
    delete this;
}


//----------------------------------------
// Grate1
//----------------------------------------
namespace
{
    class Grate1 : public CloneStone<Grate1> {
    public:
        Grate1() : Stone("st-grate1") {}
        StoneResponse actor_hit(const StoneContact &sc) {
            return STONE_PASS;
        }
        bool on_laserhit(Direction) { return true; }
    };
}

//----------------------------------------
// Laser stones
//
// These stones emit a laser beam in a specified direction when
// activated.  They are the only objects in the game that can act as
// primary light sources (mirrors can also emit light but they require
// an incoming beam).
//
// Attributes:
//
// :on      1 if laser in active, 0 if not
// :dir     the Direction in which light is emitted
//----------------------------------------
namespace
{
    class LaserStone : public Stone, public LaserEmitter {
    public:
        LaserStone() : Stone("st-laser") {
            set_attrib("on", Value(1));
            set_attrib("dir", Value(EAST));
        }
        static void reemit_all();

        // LaserEmitter interface
        DirectionBits emission_directions() const {
            return to_bits(get_dir());
        }
    private:
        void emit_light();

        void on_creation() {
            if (is_on())
                LaserBeam::emit_from(get_pos(), get_dir());
            set_model();
        }
        void set_model() {
            if (is_on())
                Stone::set_model("st-laseron" + to_suffix(get_dir()));
            else
                Stone::set_model("st-laser" + to_suffix(get_dir()));
        }

        bool is_on() const { return int_attrib("on"); }
        Direction get_dir() const {return Direction(int_attrib("dir"));}

        void message(const string &m, const Value &);

        Object *clone() {
            LaserStone *st = new LaserStone;
            instances.push_back(st);
            return st;
        }
        void dispose() {
            instances.erase(find(instances.begin(),
                                 instances.end(), this));
            delete this;
        }

        // Variables
        static vector<LaserStone*> instances;

    };
}

vector<LaserStone*> LaserStone::instances;

void 
LaserStone::message(const string &m, const Value &)
{
    bool newon;
    if (m == "on")
        newon = true;
    else if (m=="off")
        newon = false;
    else if (m=="onoff")
        newon = !is_on();
    if (newon != is_on()) {
        set_attrib("on", Value(newon));
        set_model();
        recalc_light();
    }
}

void
LaserStone::reemit_all()
{
    for (unsigned i=0; i<instances.size(); ++i) {
        instances[i]->emit_light();
    }
}

void 
LaserStone::emit_light()
{
    if (is_on())
        LaserBeam::emit_from(get_pos(), get_dir());
}

//----------------------------------------
// Mirror stones
//
// Mirrors exist in two basic flavors: plane mirrors and triangle
// mirrors.  They differ in the way they reflect light.
//
// They have the following two attributes: "transparent" determines
// whether the mirror is semitransparent or not.  If it is, the light
// also passes the stone in addition to being deflected.  "movable" is
// set for mirror stones that can be moved by the player.
//
// Attributes:
//
// :transparent  boolean
// :movable      boolean
// :orientation  number 1 <= x <= 4
//----------------------------------------
namespace
{
    class MirrorStone : public virtual Stone, public MovableStone,
                        public LaserEmitter, public PhotoCell
    {
    protected:
        MirrorStone() {
            set_attrib("transparent", Value());
            set_attrib("movable", Value());
            set_attrib("orientation", Value(1));
        }

        bool is_transparent() const { return int_attrib("transparent"); }
        bool is_movable() const { return int_attrib("movable"); }

        void set_orientation(int o) {set_attrib("orientation", o);}
        int get_orientation() {return int_attrib("orientation");}

        void emit_light(Direction dir) {
            LaserBeam::emit_from(get_pos(), dir);
            outdirs = DirectionBits(outdirs | to_bits(dir));
        }

        void set_model() {
            string mname = get_kind();
            mname += is_movable() ? "-m" : "-s";
            mname += is_transparent() ? "t" : "o";
            mname += char('0' + get_orientation());
            Stone::set_model(mname);
        }
    private:
        // LaserEmitter interface
        DirectionBits emission_directions() const {
            return outdirs;
        }

        // PhotoCell interface
        void on_recalc_start() { outdirs = NODIRBIT; }
        void on_recalc_finish() {}

        // Stone interface
        StoneResponse actor_hit(const StoneContact &sc) {
            if (is_movable())
                MovableStone::actor_hit(sc);
            rotate_right();
            return STONE_REBOUND;
        }
        void on_creation() {
            set_model();
            photo_activate();
        }
        void on_removal() {
            photo_deactivate();
            Stone::on_removal();
        }

        // Private methods
        void rotate_right() {
            set_orientation(1+(get_orientation() % 4));
            set_model();
            maybe_recalc_light(get_pos());
            sound::PlaySound("st-mirrorturn");
        }

        // Variables
        DirectionBits outdirs;
    };
}

//----------------------------------------
// PlaneMirror
//----------------------------------------
namespace
{
    class PlaneMirror : 
        public CloneStone<PlaneMirror>, public MirrorStone
    {
    public:
        PlaneMirror() : Stone("st-pmirror") {
            set_orientation('/');
        }
    private:
        void set_orientation(char o) {
            char *a = " -\\|/";
            MirrorStone::set_orientation(strchr(a,o)-a);
        }
        char get_orientation() {
            char *a = " -\\|/";
            return a[MirrorStone::get_orientation()];
        }
        bool on_laserhit(Direction dir);
    };
}

bool 
PlaneMirror::on_laserhit(Direction dir)
{
    switch (get_orientation()) {
    case '|':
        if (dir==EAST || dir==WEST)
            emit_light(reverse(dir));
        break;
    case '-': 
        if (dir==NORTH || dir==SOUTH)
            emit_light(reverse(dir));
        break;
    case '/': 
        switch(dir) {
        case EAST: emit_light(NORTH); break;
        case SOUTH: emit_light(WEST); break;
        case NORTH: emit_light(EAST); break;
        case WEST: emit_light(SOUTH); break;
        case NODIR: break;
        }
        break;
    case '\\': 
        switch(dir) {
        case EAST: emit_light(SOUTH); break;
        case SOUTH: emit_light(EAST); break;
        case NORTH: emit_light(WEST); break;
        case WEST: emit_light(NORTH); break;
        case NODIR: break;
        }
        break;
    }
    if (is_transparent())
        emit_light(dir);
    return false;
}

//----------------------------------------
// TriangleMirror
//----------------------------------------
namespace
{
    class TriangleMirror : 
        public CloneStone<TriangleMirror>, public MirrorStone
    {
    public:
        TriangleMirror() : Stone("st-3mirror") {}
    private:
        Direction get_orientation() {
            Direction a[] = {NODIR, SOUTH, WEST, NORTH, EAST};
            return a[MirrorStone::get_orientation()];
        }
        bool on_laserhit(Direction dir);
    };
}

bool 
TriangleMirror::on_laserhit(Direction dir) 
{
    Direction orient = get_orientation();

    if (dir == orient) 
        emit_light(reverse(dir));
    else if (dir == reverse(orient)) {
        // this is the "complicated" case where the light falls
        // on the tip of the triangle
        switch (dir) {
        case SOUTH: case NORTH: 
            emit_light(EAST); emit_light(WEST); break;
        case WEST: case EAST: 
            emit_light(SOUTH); emit_light(NORTH); break;
        case NODIR: break;
        }
    } else 
        emit_light(orient);
    if (is_transparent())
        emit_light(dir);
    return false;
}

//----------------------------------------
// Laser recalculation
//----------------------------------------
namespace
{
    bool light_recalc_scheduled = false;

    bool light_from(GridPos p, Direction dir)
    {
        p.move(dir);
        if (LaserEmitter *le = dynamic_cast<LaserEmitter*>(GetStone(p)))
            if (has_dir(le->emission_directions(), reverse(dir)))
                return true;
        if (LaserEmitter *le = dynamic_cast<LaserEmitter*>(GetItem(p)))
            return (has_dir(le->emission_directions(), reverse(dir)));
        return false;
    }
    void maybe_recalc_light(GridPos p)
    {
        if (light_recalc_scheduled)
            return;
        light_recalc_scheduled = 
            (light_from(p, NORTH) || light_from(p, SOUTH) || 
             light_from(p, WEST) || light_from(p, EAST));
    }
}

void 
world::recalc_light()
{
    light_recalc_scheduled = true;
}

void
world::recalc_light_now()
{
    if (light_recalc_scheduled) {
        PhotoCell::notify_start();
        LaserBeam::kill_all();
        LaserStone::reemit_all();
        PhotoCell::notify_finish();
        light_recalc_scheduled = false;
    }
}


//----------------------------------------
// BombStone
//
// These stones add a bomb to the player's inventory when touched.
//----------------------------------------
namespace
{
    class BombStone : public CloneStone<BombStone> {
    public:
        BombStone() : Stone("st-bombs") {}
    private:
        StoneResponse actor_hit(const StoneContact &sc);
    };
}

StoneResponse 
BombStone::actor_hit(const StoneContact &sc)
{
    Actor *a = sc.actorinfo->actor;
    if (const Value *v = a->get_attrib("player"))
    {
        int iplayer = to_int(*v);
        player::Inventory *inv = player::GetInventory(iplayer);
        if (!inv->is_full())
        {
            Item *it = dynamic_cast<Item*>(MakeObject("it-blackbomb"));
            inv->add_item(it);
        }
    }
    sound::PlaySound("st-stone");
    return STONE_REBOUND;
}

//----------------------------------------
// GlassStone
//
// Laser light may pass this stone, but actors may not.
//----------------------------------------
namespace
{
    class GlassStone : public CloneStone<GlassStone> {
    public:
        GlassStone() : Stone("st-glass") {}
        bool on_laserhit(Direction dir) { return true; }
    };
}

//======================================================================
 // ACTORS
//======================================================================

Actor::Actor(const string &kind, const px::V3& pos_,
             double radius_, double mass_, double charge_)
    : Object(kind)
{
    charge = charge_;
    mass = mass_;
    radius = radius_;
    sprite_id = 0;
}

void Actor::set_attrib(const string &key, const Value &val)
{
    if (key == "player")
    {
        int iplayer = to_int(val);
        player::SetMainActor(iplayer, this);
    }
    Object::set_attrib(key, val);
}

void Actor::on_creation(const V3 &p)
{
    set_model(get_kind(), p);
}

void Actor::move(const V3& newpos)
{
    display::MoveSprite(sprite_id, newpos);
    pos = newpos;
    on_motion(newpos);
}

void Actor::set_model(const string & mname, const V3 &pos)
{
    sprite_id = AddSprite(display::SPRITE_ACTOR, pos, mname);
}

void Actor::set_model(const string &name)
{
    display::ReplaceSprite(sprite_id, name);
}

//----------------------------------------
// BlackBall
//
// Attributes:
//
// :color       0=black, 1=white
// 
// Messages:
//
// :"shatter"
// :"fall"
//----------------------------------------
namespace
{
    class BlackBall : public Actor, public display::AnimCallback {
    public:
        BlackBall() : Actor("ac-blackball", px::V3(), 21.0/64.0), state(NORMAL)
        {
            fallcnt = oldfallcnt = 0;
            set_attrib("mouseforce", Value(true));
            set_attrib("color", Value(0.0));
        }
    private:
        enum State { NORMAL, SHATTERING, DROWNING, FALLING, DEAD };
        State state;
        int fallcnt, oldfallcnt;

        bool is_dead() { return state == DEAD; }

        void on_motion(V3 newpos) {
            if (fallcnt == oldfallcnt) 
                fallcnt = 0;
//        oldfallcnt = fallcnt;
        }

        void on_creation(const V3 &p)
        {
            set_model("ac-blackball", p);
            display::FollowSprite(get_spriteid());
        }

        void change_state(State newstate);
        void animation_finished();
        void message(const string &m, const Value &);

        Object* clone() { return new BlackBall; }
        void dispose() { delete this; }
    };
}

void 
BlackBall::change_state(State newstate)
{
    switch (newstate) {
    case NORMAL: break;
    case SHATTERING:
        sound::PlaySound("shatter");
        world::GrabActor(this);
        set_model("ac-blackball-shatter");
        SetCallback(get_spriteid(), this);
        break;
    case DROWNING:
        break;
    case FALLING:
        set_attrib("mouseforce", Value(0.0));
        world::GrabActor(this);
        set_model("ac-blackball-fall");
        SetCallback(get_spriteid(), this);
        break;
    case DEAD: break;
    }
    state = newstate;
}

void 
BlackBall::animation_finished() 
{
    switch (state) {
    case SHATTERING:
        set_model("ac-blackball-shattered");
        change_state(DEAD);
        break;
    case FALLING:
        sound::PlaySound("shatter");
        set_model("ac-blackball-fallen");
        change_state(DEAD);
        break;
    default:
        break;
    }
}

void 
BlackBall::message(const string &m, const Value &)
{
    if (state == NORMAL) {
        if (m == "shatter")
            change_state(SHATTERING);
        else if (m == "fall") {
            if (++fallcnt > 10)
                change_state(FALLING);
        }
    }
}


//----------------------------------------------------------------------
// Local variables
//----------------------------------------------------------------------

namespace
{
    typedef map<string, Object*> ObjectMap;
    ObjectMap objmap;           // repository of object templates
}

namespace
{
    bool has_templ(const string &name)
    {
        return objmap.find(name) != objmap.end();
    }

    // Add an new Object template to `objmap'. 
    void add_templ(Object *o)
    {
        string kind = o->get_kind();
        if (has_templ(kind))
            cerr << "add_objtempl: redefinition of object `" <<kind<< "'.\n";
        else
            objmap[kind] = o;
    }

    void clear_repos()
    {
        delete_map(objmap.begin(), objmap.end());
    }

    // Initialize the Object repository in `objmap'. 
    void init_repos()
    {
        // Floors
        add_templ(new Abyss);

        // Floor(kind, friction, mousefactor)
        add_templ(new Floor("fl-dunes",      0.3,   1.0));
        add_templ(new Floor("fl-sand",       3.6,   2.0));
        add_templ(new Floor("fl-wood",       0.4,   1.0));
        add_templ(new Floor("fl-hay",        5.0,   1.5));
        add_templ(new Floor("fl-bluegreen",  6.0,   2.0));
        add_templ(new Floor("fl-bluegreenx", 0.4,   1.0));
        add_templ(new Floor("fl-gray",       3.0,  4.0));
        add_templ(new Floor("fl-space",      0.0,   0.0));
        add_templ(new Floor("fl-leaves",     5.0,   2.0));
        add_templ(new Floor("fl-inverse",    3.0,  -2.0));

        // Items
        add_templ(new BlackBomb);
        add_templ(new Document);
        add_templ(new Dynamite);
        add_templ(new ExtraLife);
        add_templ(new Hammer);
        add_templ(new MagicWand);
        add_templ(new Trigger);
        add_templ(new Umbrella);
        add_templ(new ShogunDot);

        // Stones
        add_templ(new Brownie);
        add_templ(new ChameleonStone);
        add_templ(new Grate1);
        add_templ(new BombStone);
        add_templ(new SwapStone);
        add_templ(new FakeOxydStone);
        add_templ(new OxydStone);
        add_templ(new Stone("st-marble"));
        add_templ(new Stone("st-rock2"));
        add_templ(new Stone("st-greenbrown"));
        add_templ(new OneWayStone);
        add_templ(new PuzzleStone);
        add_templ(new Door);
        add_templ(new ShogunStone);
        add_templ(new SwitchStone);
        add_templ(new GlassStone);

        add_templ(new LaserStone);
        add_templ(new TriangleMirror);
        add_templ(new PlaneMirror);

        // Actors
        add_templ(new BlackBall);
    }
}

Object *
world::MakeObject(cstring &kind)
{
    static bool initialized = false;
    if (!initialized) {
        init_repos();
	atexit(&clear_repos);
        initialized = true;
    }

    if (!has_templ(kind)) {
        cerr << "MakeObject: unkown object name `" <<kind<< "'.\n";
        return 0;
    } else
        return objmap[kind]->clone();
}

void
world::DisposeObject(Object *o)
{ 
    if (o != 0)
        o->dispose(); 
}

void 
world::SendMessage(Object *o, const std::string &msg)
{
    if (o)
        o->message(msg, Value());
}
