------------------------------------------------------------------------
-- Copyright (C) 2002 Daniel Heck
--
-- This program is free software; you can redistribute it and/or
-- modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation; either version 2
-- of the License, or (at your option) any later version.
--  
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--  
-- You should have received a copy of the GNU General Public License along
-- with this program; if not, write to the Free Software Foundation, Inc.,
-- 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
------------------------------------------------------------------------


----------------------
-- Helper functions --
----------------------

modeltag = 0
function unique_modelname()
    modeltag = modeltag + 1
    return "xmodel" .. modeltag
end

--
-- Define a new image model.  'name' is the name of the model being created
-- and 'opt' contains optional information about the image file.  The following
-- fields can be set in 'opt':
--
--     * filename
--     * xoff,yoff (hotspot coordinates inside the image)
--
-- Suitable default values are used for all options
--
function def_image(name, opt)
    opt = opt or {}
    name = name or unique_modelname()
    fname = (opt.filename or name)
    opt = opt or {}
    enigma.DefineImage(name, fname, opt.xoff or 0, opt.yoff or 0)
    return name
end

function def_images(names, opt)
    opt = opt or {}
    for i,name in names do
	def_image(name,opt)
    end
end

function def_stone(name, shmodel, opt)
    opt = opt or {}
    shmodel = shmodel or "sh-solid"
    opt.filename = opt.filename or name
    enigma.DefineShmodel(name, def_image(nil, opt), shmodel)
end

function def_shmodel(name, fg, bg)
    enigma.DefineShmodel(name, fg, bg)
end

function def_stones(names)
    for i,name in names do def_stone(name) end
end

function def_overlay(name, imglist)
    enigma.DefineOverlayImage(name, imglist)
end

function def_solidstone(name, front)
    enigma.DefineShmodel(name, front, "sh-solid")
end

----------------
-- Animations --
----------------

--
-- Generate a list of frame names by appending increasing number prefixes
-- to a base name.  For example, 'framenames("hello", 1,2)' yields
-- {"hello_0001", "hello_0002"}.
--
function framenames(prefix, first, last)
    local fn = {}
    for i=first,last do
	tinsert(fn, prefix .. format("_%04d", i))
    end
    return fn
end

-- Build a list of frames from a list of model names and a constant duration
function buildframes(names, msec)
    local a={}
    for i,n in names do a[i] = {n, msec} end
    return a
end

-- Build a list of frames from (1) a list of names and (2) a list of frame 
-- durations.  These two list are assumed to have the same number of entries,
-- or, more generally, to have the same index set.
function composeframes(namelist, mseclist)
    local a={}
    for i,n in namelist do a[i] = {n, mseclist[i]} end
    return a
end


--
-- Given a list of frames, this function generates a new framelist with
-- for a ping-pong style animation.
--
function pingpong(framelist)
    local a=framelist
    local n=getn(framelist)
    for i = 0,n-1 do
	a[n+i+1] = framelist[n-i]
    end
    return a
end

function repeat_frames(framelist, blocksize, cnt)
    local a={}
    for i=1,getn(framelist),blocksize do
	for j=1,cnt do
	    for k=i,i+blocksize-1 do
		tinsert(a,framelist[k])
	    end
	end
    end
    return a
end

function repeatanim(framelist)
    return repeat_frames(framelist, getn(framelist), 2)
end

function reverseframes(framelist)
    local a={}
    for i=getn(framelist),1,-1 do
	tinsert(a, framelist[i])
    end
    return a
end

function def_anim_images(name, frames, opt)
    opt = opt or {}
    enigma.DefineAnim(name, opt.loop)
    for i=1,getn(frames) do
	local frame=frames[i]
	opt.filename = frame[1]
	local immodel = def_image(nil, opt)
	enigma.AddFrame(name, immodel, frame[2])
    end
end

function def_anim(name, frames, loop)
    enigma.DefineAnim(name, loop)
    for i=1,getn(frames) do
	local frame = frames[i]
	enigma.AddFrame(name, frame[1], frame[2])
    end
end
    

----------------------
-- General settings --
----------------------
enigma.SetTileSize(32, 32)

------------------
-- Actor models --
------------------

def_image("sh-blackball", {xoff=-6, yoff=-6})

enigma.DefineShmodel("ac-blackball",
    def_image(nil, {filename="ac-blackball",xoff=-12,yoff=-12}),
    "sh-blackball"
)
enigma.DefineShmodel("ac-whiteball",
    def_image(nil, {filename="ac-whiteball",xoff=-12,yoff=-12}),
    "sh-blackball"
)


-- shattering black ball
def_anim_images("ac-blackball-shatter", 
                buildframes(framenames("ac-bshatter", 1,5), 60),
                {xoff=-20,yoff=-20})

def_image("ac-blackball-shattered", 
          {filename="ac-bshatter_0005", xoff=-20, yoff=-20})

-- falling black ball

def_anim_images("ac-blackball-fall", 
                composeframes(framenames("ac-blackfall", 1,11), 
                              {70,65,60,55,50,50,50,50,50,50,50}),
                {xoff=-12,yoff=-12})

def_image("ac-blackball-fallen", 
          {filename="invisible", xoff=-4, yoff=-4})

def_image("inv-blackball")


------------------
-- Floor models --
------------------

function def_floors(floorlist)
    for i,name in floorlist do
	def_image(name)
    end
end

function def_randfloor(name, imagelist)
    def_floors(imagelist)
    enigma.DefineRandmodel(name, imagelist)
end

def_image("fl-rock", {filename="fl-rock1"})
def_floors{"fl-dunes", "fl-sand", "fl-wood", "fl-bluegreen", "fl-bluegreenx"}
def_floors{"fl-abyss", "fl-inverse"}

def_randfloor("fl-hay",{"fl-hay1", "fl-hay2"});
def_randfloor("fl-leaves", {"fl-leaves1", "fl-leaves2", "fl-leaves3"})
def_randfloor("fl-gray", {"fl-gray1","fl-gray2","fl-gray3","fl-gray4","fl-gray5"})
def_randfloor("fl-space", {"fl-space1", "fl-space2", "fl-space3"})


-----------------
-- Item models --
-----------------

def_images{"it-blackbomb", "it-document", "it-dynamite", "it-extralife",
    "it-magicwand", "it-laserh", "it-laserv", "it-laserhv",
    "it-hammer",
    "it-umbrella",
    "it-trigger", "it-trigger1"}

dyn_frames=buildframes(framenames("it-dynamite-burning", 1,15), 100)
def_anim_images("it-dynamite-burning", repeat_frames(dyn_frames,3,2))

dyn_frames=buildframes(framenames("it-blackbomb-burning", 1,9), 100)
def_anim_images("it-blackbomb-burning", repeat_frames(dyn_frames,3,2))

def_anim_images("expl", {{"expl", 1000}})


shogun1_frames=pingpong(buildframes(framenames("it-shogun",1,6),80))
shogun2_frames=pingpong(buildframes(framenames("it-shogun",7,12),80))
shogun3_frames=pingpong(buildframes(framenames("it-shogun",13,18),80))

def_anim_images("it-shogun1", shogun1_frames, {loop=1})
def_anim_images("it-shogun2", shogun2_frames, {loop=1})
def_anim_images("it-shogun3", shogun3_frames, {loop=1})

------------------
-- STONE MODELS --
------------------

-- stone shades
def_image("sh-solid")
def_images{"sh-grate1"}
def_image("sh-glass")

-- stone models
def_stone("st-glass", "sh-glass")
def_stone("st-rock2")
def_stone("st-brownie")
def_stone("st-marble")
def_stone("st-grate1", "sh-grate1")
def_stone("st-swap")
def_stone("st-bombs", nil, {filename="st-bombs2"})
def_stone("st-greenbrown")
def_stone("st-switch0", nil, {filename="st-switch2"})
def_stone("st-switch1", nil, {filename="st-switch1"})

-----------------
-- Oxyd Stones --
-----------------

-- Oxyd stones occur in different colors and different flavors and
-- for each combination of these two attributes we need animations
-- for the following internal states or state transitions:
--
--   (a) opening and closing
--   (b) blinking
--   (c) opened
--
-- Creating the required image files by hand would be tedious, so
-- we use a couple of specialized functions and overlays to create 
-- them automatically.

oxbaseimg = "st-oxydbtempl_0001"
oxcolorspots = framenames("st-oxydbtempl", 2,9)
oxbulges = framenames("st-oxydbtempl", 10,14)


-- Define "fading in" and "fading out" animations for oxyd stones.
-- These two animations are combined with the stone images to 
-- produce the opening and closing animations for oxyd stones.

oxb_fadein=buildframes(framenames("oxydb-fadein", 1,14), 50)
oxb_fadeout=reverseframes(oxb_fadein)
def_anim_images("oxydb-fadein", oxb_fadein)
def_anim_images("oxydb-fadeout", oxb_fadeout)

function make_oxydopenclose(flavor, color)
    basename = "st-oxyd" .. flavor .. color
    def_overlay(basename .. "-base", {oxbaseimg, oxcolorspots[color+1]})
    def_solidstone(basename, basename.."-base")
    enigma.DefineComposite(basename .. "-opening", basename, "oxydb-fadein")
    enigma.DefineComposite(basename .. "-closing", basename, "oxydb-fadeout")
end


function make_oxydblink(flavor, color)
    basename = "st-oxyd" .. flavor .. color .. "-blink"
    def_overlay(basename .. 1, 
                {oxbaseimg, oxcolorspots[color+1], "st-oxyd-questmark"})
    def_overlay(basename .. 2, {oxbaseimg, oxcolorspots[color+1]})
    def_anim(basename.."-anim", 
             buildframes({basename .. 1,basename .. 2}, 500), 1)
    enigma.DefineShmodel(basename, basename .. "-anim", "sh-solid")
end

function make_oxydopened(flavor, color)
    basename = "st-oxyd" .. flavor .. color .. "-open"
    names = {}
    for i=1,getn(oxbulges) do
	names[i] = basename .. format("_%04d", i)
	enigma.DefineOverlayImage(names[i], 
                                 {oxbaseimg, oxcolorspots[color+1],oxbulges[i]})
    end
    
    -- compose these images into an animation
    frames = pingpong(buildframes(names, 100))
    def_anim(basename .. "-anim", frames, 1)

    -- and finally add a shadow to make the model complete
    def_solidstone(basename, basename.."-anim")
end

for color=0,7 do
    make_oxydopenclose("b", color)
    make_oxydblink("b", color)
    make_oxydopened("b", color)
end

def_stone("st-oxydb")
def_stone("st-oxyda")

-----------------
-- Laser stone --
-----------------
function make_laser(dir)
    laseron=framenames("st-laser"..dir, 1, 9)

    -- deactivated laser
    enigma.DefineOverlayImage("laser"..dir, {"st-laser-base", laseron[1]})
    def_solidstone("st-laser"..dir, "laser"..dir)

    -- activated laser
    names = {}
    for i=1,getn(laseron) do
	names[i] = "st-laseron"..dir .. format("_%04d", i)
	enigma.DefineOverlayImage(names[i], {"st-laser-base", laseron[i]})
    end
    frames = buildframes(names, 100)
    def_anim("st-laseron-anim"..dir, frames, 1)
    def_solidstone("st-laseron"..dir, "st-laseron-anim"..dir)
end

make_laser("-e")
make_laser("-s")
make_laser("-w")
make_laser("-n")

---------------------
-- Fake oxyd stone --
---------------------
def_stone("st-fakeoxyd", nil, {filename="st-oxydb"})
def_anim_images("fakeoxyd-blink",
                repeatanim(pingpong
                           (buildframes(framenames("st-fakeoxyd-blink", 1,4), 
                                        50))))
def_solidstone("st-fakeoxyd-blink", "fakeoxyd-blink")


-------------
-- Mirrors -- 
-------------

-- naming scheme for mirror models: 
--
--	st-{3mirror,pmirror}-{m,s}{o,t}[1234]
--
-- {m,s} -> movable or static
-- {o,t} -> opaque or transparent
-- 
-- The numbers map to actual orientations as follows:
--
--   NUMBER    TRIANG.M.   PLANE M.
--	1	south	    "-"
--	2	west	    "\"
--	3	north	    "|"
--	4	east	    "/"

function make_mirror(basename, baseimg, overlays)
    for i=1,4 do
	mname = basename .. i
	enigma.DefineOverlayImage(mname .. "-ovl", {baseimg, overlays[i]})
	enigma.DefineShmodel(mname, mname .. "-ovl", "sh-solid")
    end
end

mirror3_opaque = framenames("st-mirrortempl", 1, 4)
mirror3_transp = framenames("st-mirrortempl", 5, 8)
mirrorp_opaque = framenames("st-mirrortempl", 9, 12)
mirrorp_transp = framenames("st-mirrortempl", 13, 16)

make_mirror("st-3mirror-mo", "st-mirror-movable", mirror3_opaque)
make_mirror("st-3mirror-so", "st-mirror-static",  mirror3_opaque)
make_mirror("st-3mirror-mt", "st-mirror-movable", mirror3_transp)
make_mirror("st-3mirror-st", "st-mirror-static",  mirror3_transp)

make_mirror("st-pmirror-mo", "st-mirror-movable", mirrorp_opaque)
make_mirror("st-pmirror-so", "st-mirror-static",  mirrorp_opaque)
make_mirror("st-pmirror-mt", "st-mirror-movable", mirrorp_transp)
make_mirror("st-pmirror-st", "st-mirror-static",  mirrorp_transp)

-- OneWay --
def_stones{"st-oneway-n", "st-oneway-e", "st-oneway-w", "st-oneway-s"}

-- Puzzle stones --
function make_puzzle(no)
    def_stone("st-puzzle"..no, nil, 
              {filename="st-puzzletempl_"..format("%04d", no)})
end
for i=1,16 do make_puzzle(i) end

-- Door --
def_image("st-doorh-open", {filename="st-doorh_0007"})
def_image("st-doorh-closed", {filename="st-doorh_0001"})
doorhframes = buildframes(framenames("st-doorh",1,7),50)
def_anim_images("st-doorh-opening", doorhframes)
def_anim_images("st-doorh-closing", reverseframes(doorhframes))

def_image("st-doorv-open", {filename="st-doorv_0007"})
def_image("st-doorv-closed", {filename="st-doorv_0001"})
doorvframes = buildframes(framenames("st-doorv",1,7),50)
def_anim_images("st-doorv-opening", doorvframes)
def_anim_images("st-doorv-closing", reverseframes(doorvframes))

-- Shogun stones --
def_images{"sh-shogun1","sh-shogun2", "sh-shogun4"}

def_stone("st-shogun1", "sh-shogun1")
def_stone("st-shogun2", "sh-shogun2")
def_stone("st-shogun4", "sh-shogun4")

def_overlay("st-shogun3o", {"st-shogun1", "st-shogun2"})
def_shmodel("st-shogun3", "st-shogun3o", "sh-shogun1")

def_overlay("st-shogun5o", {"st-shogun1", "st-shogun4"})
def_shmodel("st-shogun5", "st-shogun5o", "sh-shogun1")

def_overlay("st-shogun6o", {"st-shogun2", "st-shogun4"})
def_shmodel("st-shogun6", "st-shogun6o", "sh-shogun2")

def_overlay("st-shogun7o", {"st-shogun1", "st-shogun2", "st-shogun4"})
def_shmodel("st-shogun7", "st-shogun7o", "sh-shogun1")


-------------
-- Effects --
-------------

ring_anim_frames = buildframes(framenames("ring-anim", 1,8), 40)
def_anim_images("ring-anim", ring_anim_frames, {xoff=-16,yoff=-16})
