//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
//#include "config.h"
#include "lua.hh"
#include "world.hh"
#include "objects.hh"
#include "enigma.hh"

extern "C" {
#include "lua/include/lualib.h"
}
#include "px/px.hh"
#include <string>
#include <iostream>

using namespace lua;
using namespace std;

using world::GridPos;


namespace lua
{
    class LuaState {
    public:
    private:
        lua_State *state;
    };
}

namespace lua
{
    lua_State *state = 0;       // global LUA state
    int object_tag;             // LUA tag for `Object's
}

//----------------------------------------------------------------------
// Helper routines
//----------------------------------------------------------------------

static void
push_value(lua_State *L, const tools::Value &val)
{
    using tools::Value;

    switch (val.get_type()) {
    case Value::NIL: lua_pushnil(L); break;
    case Value::DOUBLE: lua_pushnumber(L, to_double(val)); break;
    case Value::STRING: lua_pushstring(L, to_string(val)); break;
    case Value::VECTOR3: assert(0); break; // FIXME
    }
}

static tools::Value
to_value(lua_State *L, int idx)
{
    switch (lua_type(L, idx)) {
    case LUA_TNIL: return tools::Value();
    case LUA_TNUMBER: return tools::Value(lua_tonumber(L,idx));
    case LUA_TSTRING: return tools::Value(lua_tostring(L,idx));
    default: lua_error(L,"Cannot convert type to Value.");
    }
    return tools::Value();
}

static bool
is_object(lua_State *L, int idx)
{
    return lua_isuserdata(L,idx) && lua_tag(L,idx)==object_tag;
}

static world::Object *
to_object(lua_State *L, int idx)
{
    if (!is_object(L,idx)) {
        lua_error(L, "Cannot convert type to an Object");
        return 0;
    }
    return static_cast<world::Object*>(lua_touserdata(L,idx));
}


//----------------------------------------------------------------------
// Enigma interface routines
//----------------------------------------------------------------------
static int 
en_make_object (lua_State *L)
{
    world::Object *obj = world::MakeObject(lua_tostring(L, 1));
    lua_pushusertag(L, obj, object_tag);
    return 1;
}

static int
en_set_attrib(lua_State *L)
{
    world::Object *obj = to_object(L,1);
    const char *key = lua_tostring(L,2);
    if (obj && key)
        obj->set_attrib(key, to_value(L, 3));
    else
        lua_error(L, "Illegal argument");
    return 0;
}

enum ObjType { FLOOR, ITEM, STONE };

static int
en_set_floor(lua_State *L)
{
    GridPos p(lua_tonumber(L,1), int(lua_tonumber(L,2)));
    world::Floor *fl=0;

    if (lua_type(L, 3) == LUA_TNIL)
        fl = 0;
    else if (is_object(L,3))
        fl = static_cast<world::Floor*>(lua_touserdata(L,3));
    else
        lua_error(L, "argument 3 must be an Object or nil");
    world::SetFloor(p, fl);
    return 0;
}

static int
en_set_item(lua_State *L)
{
    GridPos p(lua_tonumber(L,1), int(lua_tonumber(L,2)));
    world::Item *it = dynamic_cast<world::Item*>(to_object(L, 3));
    world::SetItem(p, it);
    return 0;
}
static int
en_set_stone(lua_State *L)
{
    GridPos p(lua_tonumber(L,1), int(lua_tonumber(L,2)));
    world::Stone *st = dynamic_cast<world::Stone*>(to_object(L, 3));
    world::SetStone(p, st);
    return 0;
}

static int
en_set_actor(lua_State *L)
{
    double x = lua_tonumber(L,1);
    double y = lua_tonumber(L,2);
    world::Actor *ac = dynamic_cast<world::Actor*>(to_object(L, 3));
    world::SetActor(x, y, ac);
    return 0;
}


static int
en_send_message(lua_State *L)
{
    world::Object *obj = to_object(L, 1);
    const char *msg = lua_tostring(L, 2);
    if (obj && msg)
        obj->message(msg, to_value(L, 3));
    else
        lua_error(L, "Illegal argument");
    return 0;
}

static int
en_name_object(lua_State *L)
{
    world::Object *obj = to_object(L, 1);
    const char *name = lua_tostring(L,2);

    world::NameObject(obj, name);

    return 0;
}

static int
en_get_named_object(lua_State *L)
{
    world::Object *o = world::GetNamedObject(lua_tostring(L,1));
    if (o)
        lua_pushusertag(L, o, object_tag);
    else
        cerr << "GetNamedObject: no object of this name\n";
    return 1;
}

static int
en_finish_level(lua_State *L)
{
    enigma::FinishLevel();
    return 0;
}

static int
clear_level_list(lua_State *L)
{
    enigma::ClearLevelList();
    return 0;
}

static int
add_level(lua_State *L)
{
    const char *n = lua_tostring(L,1);
    const char *d = lua_tostring(L,2);
    if (n && d) 
        enigma::AddLevel(n,d);
    else
        lua_error(L, "invalid parameters");
    return 0;
}

 // LUA functions

static int
create_world(lua_State *L)
{
    const char *name = lua_tostring(L, 1);
    int w = int(lua_tonumber(L, 2));
    int h = int(lua_tonumber(L, 3));

    if (name && w>0 && h>0) {
        enigma::SetCaption(string("Enigma - ") + name);
        world::Create(w, h);
    }
    else
        lua_error(L, "invalid arguments");
    return 0;
}

static int
find_data_file(lua_State *L)
{
    const char *filename = lua_tostring(L, 1);
    string absfile = enigma::FindDataFile(filename);
    lua_pushstring(L, absfile.c_str());
    return 1;
}

static CFunction luafuncs[] = {
    {en_set_attrib,     "SetAttrib"},
    {en_make_object,    "MakeObject"},
    {en_set_floor,      "SetFloor"},
    {en_set_item,       "SetItem"},
    {en_set_stone,      "SetStone"},
    {en_set_actor,      "SetActor"},
    {en_send_message,   "SendMessage"},
    {en_name_object,    "NameObject"},
    {en_get_named_object,"GetNamedObject"},
    {en_finish_level,   "FinishLevel"},
    {clear_level_list,  "ClearLevelList"},
    {add_level,         "AddLevel"},
    {create_world,  "CreateWorld"},
    {find_data_file,    "FindDataFile"},
    {0,0}
};

//----------------------------------------------------------------------
// lua:: functions
//----------------------------------------------------------------------

void 
lua::RegisterFuncs(CFunction *funcs)
{
    lua_getglobal(state, "enigma");
    for (unsigned i=0; funcs[i].func; ++i) {
        lua_pushstring(state, funcs[i].name);
        lua_pushcfunction(state, funcs[i].func);
        lua_settable(state, -3);
    }
    lua_pop(state, 1);
}

void 
lua::set_str(const char *varname, const char *str)
{
    lua_getglobal(state, "enigma");
    lua_pushstring(state, varname);
    lua_pushstring(state, str);
    lua_settable(state, -3);
    lua_pop(state, 1);
}

const char *
lua::get_str(const char *varname)
{
    lua_getglobal(state, "enigma");
    lua_pushstring(state, varname);
    lua_gettable(state, -2);
    const char *str = lua_tostring(state, -1);
    lua_pop(state, 2);
    return str;
}

void lua::set_num(const char *varname, double num)
{
    lua_getglobal(state, "enigma");
    lua_pushstring(state, varname);
    lua_pushnumber(state, num);
    lua_settable(state, -3);
    lua_pop(state, 1);
}

double lua::get_num(const char *varname)
{
    lua_getglobal(state, "enigma");
    lua_pushstring(state, varname);
    lua_gettable(state, -2);
    double num = lua_tonumber(state, -1);
    lua_pop(state, 2);
    return num;
}

void 
lua::CallFunc(const std::string &funcname, const tools::Value& arg)
{
    lua_getglobal(state, funcname.c_str()); // get function
    push_value(state, arg);
    switch (lua_call(state, 1, 0)) {
    default:
        ;
    }
}

void 
lua::Dofile(const string &filename)
{
    int oldtop = lua_gettop(state);
    switch (lua_dofile(state, filename.c_str())) {
    case 0:
        break;
    default:
        throw lua::Error();
    }
    lua_settop(state, oldtop);
}

void 
lua::Init()
{
    state = lua_open(0);
    lua_baselibopen(state);
    lua_strlibopen(state);
    lua_mathlibopen(state);
    lua_iolibopen(state);

    // Create a new tag for world::Object objects
    object_tag = lua_newtag(state);

    // Create a new table to store all variables and functions
    // accessible from LUA.
    lua_newtable(state);
    lua_setglobal(state, "enigma");

    // Register all functions
    RegisterFuncs(luafuncs);

//    set_str("datapath", DEFAULT_DATA_PATH);
    set_str("Playername", "unknown");
    set_num("Sensitivity", 2.0);

    set_num("NORTH", enigma::NORTH);
    set_num("SOUTH", enigma::SOUTH);
    set_num("WEST", enigma::WEST);
    set_num("EAST", enigma::EAST);
}

void lua::Shutdown()
{
    lua_close(state);
}
