static char rcsid[] = "@(#)$Id: pgp_decode.c,v 1.4.4.4 1999/10/12 11:01:13 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.4.4.4 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "melib.h"
#include "s_me.h"
#include "s_elm.h"

#ifdef USE_PGP
#include <sys/time.h>
#include <errno.h>

extern int errno;

extern int pgp_keeppassfor; /* 5 minutes */
char pgp_passphrase[PGP_NUM][STRING];
int pgp_expires;

static const char * pgp_names[PGP_NUM] = {
    "*none*",
    "PGP 2",
    "PGP 5",
    "GnuPG"
};

/* if v >= pgp2 returns gives available pgp version */

enum pgp_version have_pgp (v)
     enum pgp_version v;
{
    enum pgp_version return_value = v;
    
    switch(v) {
    case pgp_none:
	return_value = pgp_none;
	;;
    case pgp2:
	if (strcmp(pgp2_path,"none") == 0 || 
	    pgp2_path[0] == '\0') {
	    return_value = pgp_none;
	} else if (pgp2_path[0] == '/') {
	    if (-1 == access(pgp2_path,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  pgp2_path, error_description(err));
	    
		dprint(5,(debugfile,"have_pgp: no access %s: %s\n",pgp2_path,
			  error_description(err)));
		sleep_message();
		return_value = pgp_none;
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      pgp2_path);
	    dprint(5,(debugfile,"have_pgp: bad path: %s\n",pgp2_path));
	    sleep_message();
	    return_value = pgp_none;
	}
	if (return_value != pgp_none)
	    break;

	return_value = pgp5;
	/* FALLTROUGH */    
    case pgp5: try_pgp5:
	if (strcmp(pgp5_dir,"none") == 0 || 
	    pgp5_dir[0] == '\0') {
	    return_value = pgp_none;
	} else if (pgp5_dir[0] == '/') {
	    char * path = elm_message(FRM("%s/pgpv"),pgp5_dir);
	    if (-1 == access(path,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  path, error_description(err));
		
		dprint(5,(debugfile,"have_pgp: no access %s: %s\n",path,
			  error_description(err)));
		sleep_message();
		return_value = pgp_none;
	    }
	    free(path);
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      pgp5_dir);
	    dprint(5,(debugfile,"have_pgp: bad path: %s\n",pgp5_dir));
	    sleep_message();
	    return_value = pgp_none;
	}
	break;
    case gpg:
	if (strcmp(gpg_path,"none") == 0 || 
	    gpg_path[0] == '\0') {
	    return_value = pgp_none;
	} else if (gpg_path[0] == '/') {
	    if (-1 == access(gpg_path,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  gpg_path, error_description(err));
	    
		dprint(5,(debugfile,"have_pgp: no access %s: %s\n",gpg_path,
			  error_description(err)));
		sleep_message();
		return_value = pgp_none;
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      gpg_path);
	    dprint(5,(debugfile,"have_pgp: bad path: %s\n",gpg_path));
	    sleep_message();
	    return_value = pgp_none;
	}
	if (return_value != pgp_none)
	    break;

	return_value = pgp5;
	goto try_pgp5;
    }
    dprint(5,(debugfile,"have_pgp(%d)=%d\n",v,return_value));
    return return_value;
}

void pgp_void_passphrase ()
{
    int i;
    enum pgp_version v;
    
    for (v = pgp_none; v < PGP_NUM; v++) {
	for (i = 0 ; i < sizeof pgp_passphrase[v] ; i++)
	    pgp_passphrase[v][i] = '\0';
	pgp_expires = 0;
    }
    return;
}

static int QueryExpirePassphrase P_((void));

static int QueryExpirePassphrase()
{
    struct timeval now;

    /* negative implies never expire */
    if (pgp_keeppassfor < 0)
	return(0);
    gettimeofday(&now, 0);
    if (now.tv_sec < pgp_expires) {
	pgp_expires = now.tv_sec + pgp_keeppassfor;
	return(0);
    }
    pgp_void_passphrase ();
    return(1);
}

static int GetPassphrase P_((enum pgp_version v));

static int GetPassphrase (v)
     enum pgp_version v;
{
    struct timeval now;
    int status;

    dprint (2, (debugfile,"GetPassphrase(%d) -- pgp_names[%d]=%s\n",
		v,v,pgp_names[v]));

redraw:
    PutLineX(elm_LINES-2, 0, 
	     CATGETS(elm_msg_cat, MeSet, MePgpEnterPassphrase,
		     "Please enter your %s passphrase: "),
	     pgp_names[v]);
    CleartoEOS();
    
    status = optionally_enter(pgp_passphrase[v], elm_LINES-2, 37, 
			      OE_PASSWD|OE_REDRAW_MARK, 
			      sizeof pgp_passphrase[v]);
    if (REDRAW_MARK == status)
	goto redraw;

    if (status != 0)
	return(0);
    gettimeofday(&now, 0);
    if (pgp_keeppassfor > 0)
	pgp_expires = now.tv_sec + pgp_keeppassfor;
    return(pgp_passphrase[v][0] != '\0');
}

int pgp_goodPassphrase(v)
     enum pgp_version v;
{
    if (pgp_passphrase[v][0] == '\0' || QueryExpirePassphrase())
	return(GetPassphrase(v));
    else
	return(1);
}

static void close_them P_((struct run_state *rs));
static void close_them(rs)
     struct run_state *rs;
{
    int *array = rs->ext_init_data;

    close(array[0]);
    close(array[1]);
    if (array[2] != -1)
	close(array[2]);
}

/* opens up a PGP process as a child and returns its stdin and stdout */
int pgp_decrypt_init (fpin, fpout, opts, v, rs)
     FILE **fpin, **fpout;
     int opts;              /* PGP_MESSAGE, PGP_SIGNED_MESSAGE, or
			       PGP_PUBLIC_KEY */
     enum pgp_version v;    /* 'minimum version' */
     struct run_state *rs;
{
    int pgp_child_in[2];
    int pgp_child_out[2];
    int array[3];       
    char * argv[10];
    char * env[2];
    int passpipe[2];
    int usepass=FALSE;
    int code;

    char cmd[STRING];
    
    dprint (2, (debugfile, "pgp_descrypt_init() called with opts=%d\n", opts));
    

    rs->save_errno = 0;
    
    if (pipe(pgp_child_in) == -1) {
	rs->save_errno = errno;    
	return(0);
    }
    if (pipe(pgp_child_out) == -1) {
	rs->save_errno = errno;    
	close(pgp_child_in[0]);
	close(pgp_child_in[1]);
	return(0);    
    }
    if ((opts & PGP_MESSAGE) && pgp_keeppass) {
	if (pipe(passpipe) == -1) {
	    rs->save_errno = errno;
    
	    close(pgp_child_in[0]);
	    close(pgp_child_out[0]);
	    close(pgp_child_in[1]);
	    close(pgp_child_out[1]);
	    return(0);
	}
	usepass = TRUE;
    }
    dprint (3, (debugfile, "usepass = %d.\n", usepass));
  
    Raw(OFF);
    ClearScreen();
    
    /* Tell the user why they are waiting */
    if (opts & PGP_MESSAGE) {
	Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpDecryptMes,
				"Running pgp: Decrypting message...\n"));
    } else if (opts & PGP_SIGNED_MESSAGE) {
	Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpCheckSig,
				"Running pgp: Checking signature...\n"));
    } else if (opts & PGP_PUBLIC_KEY) {
	Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpExtractKeys,
				"Running pgp: Extracting keys...\n"));
    } else {
	Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpRun,
				"Running pgp ...\n"));
    }
    
    array[0] = pgp_child_in[1];
    array[1] = pgp_child_out[0];
    array[2] = -1;

    rs->ext_init_data = &array;
    rs->ext_init      = close_them;
    
    env[0] = NULL;
    rs->ext_env = env; 

    switch(v) {
    case pgp2:
    case pgp5:
	if (usepass) {
	    static char buffer[20];
	    elm_sfprintf(buffer, sizeof buffer,FRM("PGPPASSFD=%d"),
			 passpipe[0]);
	    env[0] = buffer;
	    array[2] = passpipe[1];
	}
	env[1] = NULL;
    }

    switch(v) {
	static char path[1000];
	int n;
    case pgp2:	
	n = 0;
	argv[n++] = pgp2_path;
	argv[n++] = "-f";
	argv[n++] = "+verbose=0";
	argv[n++] = "+KEEPBINARY=OFF";
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "+batchmode";
	if (opts == PGP_PUBLIC_KEY)
	    argv[n++] = "-ka";
	argv[n] = NULL;
	break;
    case pgp5:
	n = 0;
	elm_sfprintf(path, sizeof path,FRM("%s/pgpv"),pgp5_dir);
	argv[n++] = path;
	if (opts != PGP_PUBLIC_KEY)
	    argv[n++] = "-f";
	argv[n++] = "+verbose=0";
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "+batchmode";
	argv[n] = NULL;
	break;
    case gpg:	
	n = 0;
	argv[n++] = gpg_path;
	if (usepass) {
	    static char buffer[10];
	    argv[n++] = "--passphrase-fd";
	    elm_sfprintf(buffer, sizeof buffer,FRM("%d"),
			 passpipe[0]);
	    argv[n++] = buffer;
	    array[2] = passpipe[1];
	}	
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "--batch";
	if (opts == PGP_PUBLIC_KEY)
	    argv[n++] = "--import";
	else 
	    argv[n++] = "--decrypt";
	argv[n] = NULL;
	break;
    }

    code = start_run(rs, SY_RUN_STATE_INIT|SY_RUN_STATE_ENV , argv ,
		     pgp_child_in[0],pgp_child_out[1]);
	  
    close (pgp_child_in[0]);
    close (pgp_child_out[1]);
  
    if (!code)
	return 0;

    /* now send the passphrase if needed */
    if (usepass) {
	dprint(3,(debugfile,"pgp_decrypt_init: sending pgp passphrase.\n"));
	close (passpipe[0]);
	write (passpipe[1], pgp_passphrase[v], 
	       strlen(pgp_passphrase[v]));
	write (passpipe[1], "\n", 1);
	close (passpipe[1]);
    }
  
    if ((*fpin = fdopen(pgp_child_out[0], "r")) == 0) {
	int tmp;
	int err = errno;

	dprint(3,(debugfile,"pgp_decrypt_init: fdopen(%d) failed, errno=%d\n",
		  pgp_child_out[0],err));

	kill(rs->pid,SIGTERM);
	wait_end(rs,&tmp);	    
	rs->save_errno = err;
	Raw(ON);

	return(0);
    }
    if ((*fpout = fdopen(pgp_child_in[1], "w")) == 0) {
	int err = errno;
	int tmp;

	dprint(3,(debugfile,"pgp_decrypt_init: fdopen(%d) failed, errno=%d\n",
		  pgp_child_out[0],err));

	kill(rs->pid,SIGTERM);
	wait_end(rs,&tmp);	    
	rs->save_errno = err;
	Raw(ON);

	return(0);
    }
  
    return(code);
}

static int pgp_mime_opts P_((char *));

static int pgp_mime_opts (s)
     char *s;
{
    char value[STRING];
    
    if (s) {
	if (mime_get_param ("format", value, s, sizeof(value))) {
	    if (istrcmp (value, "keys-only") ==  0)
		return PGP_PUBLIC_KEY;
	}
	if (mime_get_param ("x-action", value, s, sizeof (value))) {
	    if (istrcmp (value, "encryptsign") == 0)
		return (PGP_MESSAGE | PGP_SIGNED_MESSAGE);
	    else if (istrcmp (value, "encrypt") == 0)
		return PGP_MESSAGE;
	    else if (istrcmp (value, "sign") == 0)
		return PGP_SIGNED_MESSAGE;
	    else if (istrcmp (value, "signclear") == 0)
		return PGP_SIGNED_MESSAGE;
	}
    }
    return PGP_MESSAGE;
}


void pgp_decode (m, s_in, s_out)
     mime_t *m;
     in_state_t  *s_in;
     out_state_t *s_out;
{
    /* This procedure implements the de-facto standard for using PGP with MIME.
     * Content-Type: application/pgp
     * Required-Parameters: none
     * Optional parameters: format, x-action
     *     format = mime | text | keys-only
     *         mime : indicates that the signed/encrypted body contains a MIME
     *                compliant body and should be parsed recursively.
     *         text : [DEFAULT if there is no format option].  This option
     *                means that the encrypted/signed data should be presented
     *                to the user after processing, no additional processing
     *                needed.
     *         keys-only:
     *                The data in the body represents public key data only
     *     x-action = encryptsign | encrypt | sign
     *         This keyword is meant to be helpful to the application, but is
     *         not required, and may not even be necessary to look at.
     *
     *         encryptsign : the application/pgp data is both signed and
     *                       encrypted.
     *         encrypt     : the data is encrypted only
     *         sign        : the data is signed only
     */
    
    char buffer[LONG_STRING];
    FILE *pgpout, *pgpin, *tmpfp=NULL, *decode_fp = NULL;
    int
	code = 0,
	inbody = FALSE,
	opts,
	len,
	raw,
	stat = -1,
	bytes = 0,
	nested = FALSE; /* PGP output should be parsed as a MIME body */
    mime_t *tmpmt;
    enum pgp_version v = pgp2;
    
    struct run_state RS;
    
    int was_binary = 0;
    in_state_t newstate2;
    enum pgp_version version;

    in_state_clear(&newstate2,STATE_in_file);


    raw = RawState ();
    
    if (mime_get_param ("format", buffer, m->type_opts, sizeof (buffer)) &&
	(istrcmp (buffer, "mime") == 0)) {
	char * tempfile = elm_message(FRM("%selmPT%d"), 
				      temp_dir, getpid ());
	
	nested = TRUE;
	dprint (3, (debugfile, "pgp_decode: format=mime\n"));
	
	
	if (NULL == (tmpfp = safeopen_rdwr(tempfile))) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
			      "Failed to create file for decoding."));
	    state_putc('[',s_out);
	    state_puts(catgets(elm_msg_cat, MeSet, MePgpCantCreate,
			       "Failed to create file for decoding."),
		       s_out);      
	    state_puts("]\n",s_out);
	    free(tempfile);
	    return;
	}
	
	unlink (tempfile); /* Filename is no longer needed ... */
	free(tempfile);
    }
    else
	tmpfp = NULL;

    opts = pgp_mime_opts (m->type_opts);
        
    buffer[0] = '\0';
    if (opts & PGP_PUBLIC_KEY) {
	state_puts (catgets(elm_msg_cat, MeSet, MePgpPublicKeys,
			    "(** This message contains PGP public key(s) **)\n\n"),
		    s_out);
    }

    /* Decode Content-transfer-encoding */
    
    if (decode_fp = arrange_decoded(m,s_in,s_out,&newstate2,NULL)) {  
	/* If arrange_decoded was treated this as text, don't consider
	 * about binary PGP files -- that happens when type is text/x-pgp
	 */
	int was_text = is_text_type (mime_types[m->type], m->subtype, 
				     m->encoding);
	
	dprint (4, (debugfile, "pgp_decode: was_text = %d\n", was_text));
	
	/* Print text before PGP armor 
	 *
	 */

	len = state_getl (buffer, sizeof (buffer), &newstate2);    
	if ( len < 1) {
	    state_puts(catgets(elm_msg_cat, MeSet, MePgpNoText1,
			       "[No text in PGP section.]\n"),
		       s_out);
	} else {
	    int c = (unsigned char) buffer[0],i;
	    
	    if (!pgp_noarmor && opts != PGP_PUBLIC_KEY) {
		was_binary = 1; /* default assumption */
		goto pgp_found;
	    }
	    
	    if (!was_text) {
		/* Check if that is binary PGP file */
		if (c & 0x80) {
		    /* In PGP block have type byte which have higgest bit set 
		     * always. so if in first byte have higgest bit set assume
		     * PGP binary file.
		     */
		    dprint (4, (debugfile, 
				"pgp_decode: first byte = %d -- assume binary PGP file\n",
				c));
		    was_binary = 1; 
		    goto pgp_found;
		}
		/* Another check */
		for (i = 0; i < len; i++) {
		    if (buffer[i] == 0) {
			dprint (4, (debugfile, 
				    "pgp_decode: byte idx=%d is zero -- binary file?\n",
				    i));
			state_puts(catgets(elm_msg_cat, MeSet, MePgpBinary,
					   "[Binary file, but does not look like PGP]\n"),
				   s_out);
			was_binary = 1; 
			goto pgp_found;
		    }
		}
	    }
	    if (strncmp(buffer, "-----BEGIN PGP", 14) == 0) 
		goto pgp_found;
	    
	    state_puts(catgets(elm_msg_cat, MeSet, MePgpBefore,
			       "[There is text before PGP section.]\n"),
		       s_out);

	    if (set_filter(m,s_out)) { /* Character set filtering */
		do {
		    state_add_prefix(s_out);

		    /* Take care of CRLF => LF conversion in here because
		     * arrange_decoded() is treating application/pgp as
		     * binary type (it need to treate it as binary because
		     * it may be binary)
		     */
		    if (!was_text &&
			len > 1 &&
			buffer[len - 1] == '\n' &&
			buffer[len - 2] == '\r') {
			buffer[len - 2] = '\n';
			buffer[len - 1] = '\0';
			len--;
		    }

		    state_put(buffer,len,s_out);
		    
		} while ((len = state_getl (buffer, sizeof (buffer), 
					    &newstate2)) > 0
			 && strncmp(buffer, "-----BEGIN PGP", 14) != 0);
		
	    } else {
		dprint (4, (debugfile, 
			    "pgp_decode: Unsupported character set?\n"));
	    }
	    
	pgp_found:
	    s_out -> filter = NULL_filter;
	}

    } else { 
	char * msg = NULL;
	if (opts & PGP_MESSAGE)
	    msg = elm_message(CATGETS(elm_msg_cat, MeSet, 
				      MePgpEncodedCantEncoding,
				      "-- Start of PGP encoded section -- can't decode content-transfer-encoding\n"));
	else if (opts & PGP_SIGNED_MESSAGE)
	    msg = elm_message(CATGETS(elm_msg_cat, MeSet, 
				      MePgpSignedCantEncoding,
				      "-- Start of PGP signed section -- can't decode content-transfer-encoding\n"));
	else
	    msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpCantEncoding,
				      "-- Start of PGP section -- can't decode content-transfer-encoding\n"));
	state_puts(msg,s_out);    
	free(msg);
	return;
    }

  if ( len < 1) {
      char * msg = NULL;
      if (opts & PGP_MESSAGE)
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpNoTextEncoded,
				    "[No text in PGP encoded section]\n"));
      else if (opts & PGP_SIGNED_MESSAGE)
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpNoTextSigned,
				    "[No text in PGP signed section]\n"));
    else
	msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpNoText,
				  "[No text in PGP section]\n"));
      state_puts(msg,s_out);    
      free(msg);
      return;
  }
  
  if (!was_binary) {
      int l = in_state_ftell(&newstate2);
      int len1;
      char buf[STRING];

      /* 
       * On PGP 2 messages these is empty line immediately after
       * -----BEGIN PGP SIGNED MESSAGE----
       * 
       * If there is something other such as 
       * Hash: SHA1
       * PGP 2 does not understand message.
       */

      while ((len1 = state_getl (buf, sizeof (buf), &newstate2)) > 0) {
	  if (len1 == 1 && buf[0] == '\n' ||
	      len1 == 2 && buf[0] == '\r' && buf[1] == '\n')
	      break;
	    if (opts == PGP_SIGNED_MESSAGE) {
		dprint (4, (debugfile, 
			    "pgp_decode: peek: Header on armor -- requires PGP 5 or GnuPG\n" ));
		v = gpg;
		break;
	    }
	    if (0 == strncmp("Version: ",buf,9)) {
		char *c = buf+9;
		
		if (0 == strncmp(c,"2.",2)) {
		    dprint (4, (debugfile,
				"pgp_decode: peek: Version -- PGP 2\n"));
		    v = pgp2;
		    break;
		} else if (0 == strncmp(c,"GnuPG",5)) {
		    dprint (4, (debugfile,
				"pgp_decode: peek: Version -- GnuPG\n"));
		    v = gpg;
		    break;
		} else {
		    dprint (4, (debugfile,
				"pgp_decode: peek: Version -- PGP 5?\n"));
		    if ('/' == pgp5_dir[0])
			v = pgp5;
		    else
			v = send_pgp_version;
		    break;
		}
	    }
      }

      in_state_fseek(&newstate2,l);
  }

  version = have_pgp(v);
  if (!version) {
      if (was_binary) {
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpNotAvailSkipping,
			      "[PGP not available, skipping...]\n"), 
		      s_out);
	  return;
      } else 
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpNotAvailRawdata,
			      "[PGP not available, raw data follows]\n"), 
		      s_out);
      goto fail;
  }

  if ((opts & PGP_MESSAGE) && pgp_keeppass) {
      if (!pgp_goodPassphrase(version)) {
	  lib_error(CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
			    "Decrypting message... Bad PGP passphrase."));
	  state_putc('[',s_out);
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpBadPassphrase,
			     "Decrypting message... Bad PGP passphrase."),
		     s_out);
	  state_puts("]\n",s_out);
	  return;
      }
  }

  if (!pgp_decrypt_init (&pgpout, &pgpin, opts, version, &RS)) {
      if (was_binary) 
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpInternalSkipping,
			      "[Internal error while calling pgp, skipping...]\n"), 
		      s_out);
      else {
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpInternalRawdata,
			      "[Internal error while calling pgp, raw data follows]\n"), 
		      s_out);
	  
      fail:

	  do {
	      state_add_prefix(s_out);
	
	      if (len > 1 &&
		  buffer[len - 1] == '\n' &&
		  buffer[len - 2] == '\r') {
		  buffer[len - 2] = '\n';
		  buffer[len - 1] = '\0';
		  len--;
	      }
	      
	      state_put(buffer,len,s_out);
	      bytes += len;
	  }  while ((len = state_getl (buffer, sizeof (buffer), 
				       &newstate2)) > 0);
	  
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpRawEnd,
			      "[End of raw data]\n"), 
		      s_out);
      }
      return;
  }
  
  do {
      fwrite(buffer,1,len,pgpin);
      bytes += len;
  }  while ((len = state_getl (buffer, sizeof (buffer), 
			       &newstate2)) > 0);
    
  fclose (pgpin);
  
  code = run_already_done(&RS,&stat);

  if (code != 0) {
      char * msg = NULL;
      if (opts & PGP_MESSAGE)
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded1,
				    "-- Start of PGP encoded section%s\n"),
			    code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						       MePgpFail,
						       ", PGP failed!") : ".");
      else if (opts & PGP_SIGNED_MESSAGE)
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpStartSigned1,
				    "-- Start of PGP signed section%s\n"),
			    code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						       MePgpFail,
						       ", PGP failed!") : ".");
      else if (opts & PGP_PUBLIC_KEY)
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpStartOutput1,
				    "-- Start of PGP output%s\n"),
			    code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						       MePgpFail,
						       ", PGP failed!") : ".");
      else
	  msg = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpStart1,
				    "-- Start of PGP section%s\n"),
			    code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						       MePgpFail,
						       ", PGP failed!") : ".");
      state_puts(msg,s_out);
      free(msg);
  } else {
      if (opts & PGP_MESSAGE)
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpStartEncoded,
			     "-- Start of PGP encoded section.\n"),
		     s_out);
      else if (opts & PGP_SIGNED_MESSAGE)
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpStartSigned,
			     "-- Start of PGP signed section.\n"),
		     s_out);
      else if (opts & PGP_PUBLIC_KEY)
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpStartOutput,
			     "-- Start of PGP output.\n"),
		     s_out);
      else
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpStart,
			     "-- Start of PGP section.\n"),
		     s_out);
  }

  bytes = 0;
  while ((len = mail_gets (buffer, sizeof (buffer), pgpout)) > 0) {
      if (nested) {
	  if (buffer[0] == '\n' || (buffer[0] == '\r' && buffer[1] == '\n'))
	      inbody = TRUE;
	  fputs (buffer, tmpfp);
	  if (inbody)
	      bytes += len;
      } else {
	  state_add_prefix(s_out);
	  state_puts(buffer,s_out);
      }
  }
  fclose (pgpout);

  if (nested) {
      struct in_state s2_in;
      in_state_clear(&s2_in,STATE_in_file);
      
      dprint (3, (debugfile, "pgp_decode: parsing decrypted data as MIME\n"));
      
      if (EOF == fflush(tmpfp)) {
	  lib_error(CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
			    "Error when flushing temporary file."));
	  state_putc('[',s_out);
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpErrorFlush,
			     "Error when flushing temporary file."),
		     s_out);
	  state_puts("]\n",s_out);
      }
      rewind(tmpfp); /* Rewind it for reading */
      
      tmpmt = mime_read_header (tmpfp, 0);
      tmpmt->length = bytes;
      
      if (tmpmt->flags & MIME_RFC822) {
	  dprint(9,(debugfile,"pgp_decode- (parsing) RFC822\n"));
	  fseek (tmpfp, tmpmt->offset, SEEK_SET);
	  tmpmt->parts = rfc822_parse (tmpfp, tmpmt->length);
      }
      else if (tmpmt->type == MIME_TYPE_MULTIPART) {
	  char subbound[80];
	  fseek (tmpfp, tmpmt->offset, SEEK_SET);
	  mime_get_boundary (subbound, tmpmt->type_opts, sizeof (subbound));
	  dprint(9,(debugfile,
		    "pgp_decode- (parsing) MULTIPART; boundary=%s\n",
		    subbound));
	  tmpmt->parts = multipart_parse (tmpfp, tmpmt->length, subbound, 
					  tmpmt->flags);
      }
      
      set_in_state_file(tmpfp,&s2_in);
      
      mime_decode (tmpmt, &s2_in, s_out);
      mime_destroy (tmpmt);
      
      in_state_destroy(&s2_in);
      
      fclose (tmpfp);
  }
  
  if (0 == code)
      code = wait_end(&RS,&stat);

  PressAnyKeyToContinue();

  if (raw)
      Raw (ON);

  if (opts & PGP_MESSAGE)
      elm_sfprintf(buffer,sizeof buffer,
		   CATGETS(elm_msg_cat, MeSet, MePgpEndEncoded,
			   "-- End of PGP encoded section%s\n"),
		   code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
					      ", PGP failed!") : ".");
  else if (opts & PGP_SIGNED_MESSAGE)
      elm_sfprintf(buffer,sizeof buffer,
		   CATGETS(elm_msg_cat, MeSet, MePgpEndSigned,
			   "-- End of PGP signed section%s\n"),
		   code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
					       ", PGP failed!") : ".");
  else if (opts & PGP_PUBLIC_KEY)
      elm_sfprintf(buffer,sizeof buffer,
		   CATGETS(elm_msg_cat, MeSet, MePgpEndOutput,
			   "-- End of PGP output%s\n"),
		   code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
					       ", PGP failed!") : ".");
  else
      elm_sfprintf(buffer,sizeof buffer,
		   CATGETS(elm_msg_cat, MeSet, MePgpEnd,
			   "-- End of PGP section%s\n"),
		   code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
				  ", PGP failed!") : ".");
  state_puts(buffer,s_out);

  in_state_destroy(&newstate2);
}

void pgp_SG_decoder(body,sign,state_in,state_out,micalg)
     mime_t *body; 
     mime_t *sign;
     in_state_t   *state_in;
     out_state_t  *state_out;     
     char         *micalg;
{
    int exit_code = -1;
    FILE *binary_fp;
    char *body_name = NULL;
    in_state_t newstate2;
    char * buffer;
    enum pgp_version v = pgp2;
    enum pgp_version version;

    in_state_clear(&newstate2,STATE_in_file);

    if (0 == strcasecmp(micalg,"pgp-md5")) {
	dprint (4, (debugfile, 
		    "pgp_SG_decoder: micalg=%s, PGP 2\n",
		    micalg));
	v = pgp2;
    } else { 
	dprint (4, (debugfile, 
		    "pgp_SG_decoder: micalg=%s, PGP 5\n",
		    micalg));
	v = pgp5;
    }

    if ((version=have_pgp(v))) {
	if (binary_fp = arrange_binary(body,state_in,state_out,&newstate2,
				       &body_name)) {
	    
	    in_state_t newstate3;
	    FILE *decode_fp = NULL;
	    char *sign_name = NULL;
	    
	    in_state_clear(&newstate3,STATE_in_file);
	    
	    if (in_state_fseek(state_in,sign->offset) != 0) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpSGSeekFail,
				  "pgp_SG_decoder: seek failed"));
		state_putc('[',state_out);
		state_puts(catgets(elm_msg_cat, MeSet, MePgpSGSeekFail,
				   "pgp_SG_decoder: seek failed"),
			   state_out);
		state_puts("]\n",state_out);	    
		sleep_message();
	    }
	    
	    
	    if (decode_fp = arrange_decoded(sign,state_in,state_out,&newstate3,
					    &sign_name)) {  
		struct run_state RS;
		int stat;
		char * argv[10];
		int raw = RawState ();
		
		switch(version) {
		    static char path[1000];
		case pgp2:	
		    argv[0] = pgp2_path;
		    argv[1] = "+batchmode";
		    argv[2] = "+TEXTMODE=off";
		    argv[3] = "+CHARSET=noconv";
		    argv[4] = sign_name;
		    argv[5] = body_name;
		    argv[6] = NULL;
		    break;
		pgp5:
		    elm_sfprintf(path, sizeof path,FRM("%s/pgpv"),pgp5_dir);
		    argv[0] = path;
		    argv[1] = "-f";
		    argv[2] = "+batchmode";
		    argv[3] = "+TEXTMODE=off";
		    argv[4] = "+CHARSET=noconv";
		    argv[5] = sign_name;
		    argv[6] = body_name;
		    argv[7] = NULL;
		    break;
		gpg:
		    argv[0] = path;
		    argv[1] = "--batch";
		    argv[2] = "--verify";
		    argv[3] = sign_name;
		    argv[4] = body_name;
		    argv[5] = NULL;
		    break;
		}
		    
		Raw(OFF);
		ClearScreen();
		Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpCheckSig,
					"Running pgp: Checking signature...\n"));

		stat = start_run(&RS,0,argv,-1,-1);
		if (stat)
		    stat = wait_end(&RS,&exit_code);
		
		PressAnyKeyToContinue();
		
		if (raw)
		    Raw (ON);
		
		if (stat) {
		    if (state_out->displaying) {
			state_puts("\n[",state_out);		
			if (exit_code == 0)
			    state_puts(catgets(elm_msg_cat, MeSet,MePgpSigOK,
					       "Checking application/pgp-signature: OK"),
				       state_out);
			else
			    state_puts(catgets(elm_msg_cat, MeSet,MePgpSigFAILURE,
					       "Checking application/pgp-signature: FAILURE"),
				       state_out);		
			state_puts("]\n\n",state_out);
		    } else if (exit_code != 0) {
			state_puts("\n",state_out);
			state_puts(catgets(elm_msg_cat, MeSet,MePgpSigFAILURE,
					   "Checking application/pgp-signature: FAILURE"),
				   state_out);		
			state_puts("\n",state_out);
		    }
		} else {
		    if (RS.save_errno) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
					  "Failed: %.30s: %.40s"),
				  argv[0],error_description(RS.save_errno));
		    } else
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantStart,
				      "Can't start %.30s"),
				  argv[0]);
		    sleep_message();
		}
		unlink(sign_name);
		free(sign_name);
	    in_state_destroy(&newstate3);
	    }
	    unlink(body_name);
	    free(body_name);
	    in_state_destroy(&newstate2);
	}
    }
    state_puts(catgets(elm_msg_cat, MeSet, MePgpStartSigned,
		       "-- Start of PGP signed section.\n"),
	       state_out);
    mime_decode(body,state_in,state_out);
    buffer = elm_message(CATGETS(elm_msg_cat, MeSet, MePgpEnd,
				 "-- End of PGP section%s\n"),
			 exit_code ? catgets(elm_msg_cat, MeSet, MePgpFail,
					     ", PGP failed!") : ".");
    state_puts(buffer,state_out);
    free(buffer);
}  

void pgp_EC_decoder(init,data,state_in,state_out)
     mime_t *init;
     mime_t *data;
     in_state_t   *state_in;
     out_state_t  *state_out;
{
    char * tempfile = NULL;
    FILE *decode_fp;
    FILE *tmpfp;
    in_state_t newstate2;

    in_state_clear(&newstate2,STATE_in_file);

    if (in_state_fseek(state_in,data->offset) != 0) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MePgpECSeekFail,
			  "pgp_EC_decoder: seek failed"));
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MePgpECSeekFail,
			   "pgp_EC_decoder: seek failed"),
		   state_out);
	state_puts("]\n",state_out);	    
	sleep_message();
    }

    tempfile = elm_message(FRM("%selmPT%d"), 
			   temp_dir, getpid ());
	
    if (NULL == (tmpfp = safeopen_rdwr(tempfile))) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
			  "Failed to create file for decoding."));
	state_putc('[',state_out);
	state_puts(catgets(elm_msg_cat, MeSet, MePgpCantCreate,
			   "Failed to create file for decoding."),
		   state_out);      
	state_puts("]\n",state_out);
	free(tempfile);
	return;
    }
    unlink (tempfile); /* Filename is no longer needed ... */
    free(tempfile); tempfile = NULL;


    if (decode_fp = arrange_decoded(data,state_in,state_out,&newstate2,
				    NULL)) {  
	int i = 0,len1;
	char buf[STRING];
	enum pgp_version v = pgp2;
	enum pgp_version version;
	struct in_state s2_in;
	struct run_state RS;
	FILE *pgpout,*pgpin;
	int code;
	mime_t *tmpmt;
	int stat;
	long bytes = 0;
	int inbody = FALSE;
	int raw = RawState ();
	in_state_clear(&s2_in,STATE_in_file);

	while ((len1 = state_getl (buf, sizeof (buf), &newstate2)) > 0) {
	    if (i == 0) {
		if (strncmp(buf, "-----BEGIN PGP", 14) != 0)
		    break;
		i++;
		continue;
	    }
	    if (len1 == 1 && buf[0] == '\n' ||
		len1 == 2 && buf[0] == '\r' && buf[1] == '\n')
		break;
	    if (0 == strncmp("Version: ",buf,9)) {
		char *c = buf+9;
		
		if (0 == strncmp(c,"2.",2)) {
		    dprint (4, (debugfile,
				"pgp_EC_decoder: peek: Version -- PGP 2\n"));
		    v = pgp2;
		    break;
		} else if (0 == strncmp(c,"GnuPG",5)) {
		    dprint (4, (debugfile,
				"pgp_EC_decoder: peek: Version -- GnuPG\n"));
		    v = gpg;
		    break;
		} else {
		    dprint (4, (debugfile,
				"pgp_EC_decoder: peek: Version -- PGP 5?\n"));
		    if ('/' == pgp5_dir[0])
			v = pgp5;
		    else
			v = send_pgp_version;
		    break;
		}
	    }
	}
	in_state_fseek(&newstate2,0);
      
	version = have_pgp(v);
	if (!version) {
	  state_puts (catgets(elm_msg_cat, MeSet, MePgpNotAvailSkipping,
			      "[PGP not available, skipping...]\n"), 
		      state_out);
	  return;
	}

	if (!pgp_goodPassphrase(version)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
			      "Decrypting message... Bad PGP passphrase."));
	    state_putc('[',state_out);
	    state_puts(catgets(elm_msg_cat, MeSet, MePgpBadPassphrase,
			       "Decrypting message... Bad PGP passphrase."),
		       state_out);
	    state_puts("]\n",state_out);
	    return;
	}

	if (!pgp_decrypt_init (&pgpout, &pgpin, PGP_MESSAGE, version, &RS)) {
	    state_puts (catgets(elm_msg_cat, MeSet, MePgpInternalSkipping,
				"[Internal error while calling pgp, skipping...]\n"), 
			state_out);
	}

	while ((len1 = state_getl (buf, sizeof (buf), 
				   &newstate2)) > 0) {
	    fwrite(buf,1,len1,pgpin);
	}  
	fclose (pgpin);
	
	code = run_already_done(&RS,&stat);	

	if (code != 0) {
	    char * msg = elm_message(CATGETS(elm_msg_cat, MeSet, 
					   MePgpStartEncoded1,
					   "-- Start of PGP encoded section%s\n"),
				   code < 0 || stat ? catgets(elm_msg_cat, 
							      MeSet, 
							      MePgpFail,
							      ", PGP failed!") 
				   : ".");
	    state_puts(msg,state_out);
	    free(msg);
	} else
	  state_puts(catgets(elm_msg_cat, MeSet, MePgpStartEncoded,
			     "-- Start of PGP encoded section.\n"),
		     state_out);
	
	bytes = 0;
	while ((len1 = mail_gets (buf, sizeof (buf), pgpout)) > 0) {
	  if (buf[0] == '\n' || (buf[0] == '\r' && buf[1] == '\n'))
	      inbody = TRUE;
	  fputs (buf, tmpfp);
	  if (inbody)
	      bytes += len1;
	}
	fclose(pgpout);

	if (EOF == fflush(tmpfp)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
			      "Error when flushing temporary file."));
	    state_putc('[',state_out);
	    state_puts(catgets(elm_msg_cat, MeSet, MePgpErrorFlush,
			       "Error when flushing temporary file."),
		       state_out);
	    state_puts("]\n",state_out);
	}
	rewind(tmpfp); /* Rewind it for reading */
      
	tmpmt = mime_read_header (tmpfp, 0);
	tmpmt->length = bytes;

	if (tmpmt->flags & MIME_RFC822) {
	    dprint(9,(debugfile,"pgp_EC_decoder- (parsing) RFC822\n"));
	    fseek (tmpfp, tmpmt->offset, SEEK_SET);
	    tmpmt->parts = rfc822_parse (tmpfp, tmpmt->length);
	}
	else if (tmpmt->type == MIME_TYPE_MULTIPART) {
	    char subbound[80];
	    fseek (tmpfp, tmpmt->offset, SEEK_SET);
	    mime_get_boundary (subbound, tmpmt->type_opts, sizeof (subbound));
	    dprint(9,(debugfile,
		      "pgp_EC_decoder- (parsing) MULTIPART; boundary=%s\n",
		      subbound));
	    tmpmt->parts = multipart_parse (tmpfp, tmpmt->length, subbound, 
					    tmpmt->flags);
	}
      
	set_in_state_file(tmpfp,&s2_in);
	
	mime_decode (tmpmt, &s2_in, state_out);
	mime_destroy (tmpmt);
	
	in_state_destroy(&s2_in);
	
	if (0 == code)
	    code = wait_end(&RS,&stat);

	PressAnyKeyToContinue();

	if (raw)
	    Raw (ON);
	
	elm_sfprintf(buf,sizeof buf,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndEncoded,
			     "-- End of PGP encoded section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");
	
	state_puts(buf,state_out);
	
	in_state_destroy(&newstate2);
	
    } else {
	char * msg = elm_message(CATGETS(elm_msg_cat, MeSet, 
					 MePgpEncodedCantEncoding,
					 "-- Start of PGP encoded section -- can't decode content-transfer-encoding\n"));
	state_puts(msg,state_out);    
	free(msg);
    }
    fclose (tmpfp);

}

#endif /* USE_PGP */


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
