#!/usr/local/bin/python2.0

""" Distutils Extensions needed for the mx Extensions.

    Copyright (c) 1997-2000, Marc-Andre Lemburg; mailto:mal@lemburg.com
    Copyright (c) 2000-2001, eGenix.com Software GmbH; mailto:info@egenix.com
    See the documentation for further information on copyrights,
    or contact the author. All Rights Reserved.
"""
import string, types, glob, os, sys
from distutils.errors import DistutilsError, DistutilsExecError, CompileError
from distutils.core import setup, Extension, Command
from distutils.sysconfig import get_config_h_filename, parse_config_h
from distutils.msvccompiler import MSVCCompiler
from distutils.util import execute
from distutils.spawn import spawn, find_executable
from distutils.dist import Distribution
from distutils.command.config import config
from distutils.command.build import build
from distutils.command.build_ext import build_ext
from distutils.command.build_py import build_py
from distutils.command.install_data import install_data

# distutils extensions
if sys.version[:3] >= '2.0':
    try:
        from distutils.command import bdist_ppm
    except ImportError:
        bdist_ppm = None
    try:
        from distutils.command import GenPPD
    except ImportError:
        GenPPD = None
else:
    bdist_ppm = None
    GenPPD = None

### Globals

__version__ = '1.2.0'

#
# mx Auto-Configuration command
#

class mx_autoconf(config):

    """ Auto-configuration class which adds some extra configuration
        settings to the packages.

    """
    # Command description
    description = "auto-configuration build step (for internal use only)"

    # Command line options
    user_options = config.user_options + [
        ('enable-debugging', None,
         'compile with debugging support'),
        ]

    # User option defaults
    enable_debugging = 0

    # C APIs to check
    api_checks = (('strftime', ['time.h']),
                  ('strptime', ['time.h']),
                  ('timegm', ['time.h']),
                  )

    def initialize_options(self):

        config.initialize_options(self)
        self.noisy = 0
        self.dump_source = 0

    def finalize_options(self):

        config.finalize_options(self)

    def run(self):

        # Setup .compiler instance
        self._check_compiler()
        self.compiler.define_macro('_GNU_SOURCE', '1')

        # Parse config.h
        config_h = get_config_h_filename()
        try:
            fp = open(config_h)
        except IOError,why:
            self.warn('could not open %s file' % config_h)
            configuration = {}
        else:
            configuration = parse_config_h(fp)
            fp.close()

        # Build lists of #defines and #undefs
        define = []
        undef = []

        # Check APIs
        for apiname, includefiles in self.api_checks:
            macro_name = 'HAVE_' + string.upper(apiname)
            if not configuration.has_key(macro_name):
                if self.check_func(apiname, includefiles):
                    define.append((macro_name, '1'))
                else:
                    undef.append(macro_name)

        # Enable debugging support
        if self.enable_debugging:
            define.append(('MAL_DEBUG', None))

        self.announce('macros to define: %s' % define)
        self.announce('macros to undefine: %s' % undef)

        # Reinitialize build_ext command with extra defines
        build_ext = self.distribution.reinitialize_command('build_ext')
        build_ext.ensure_finalized()
        # We set these here, since distutils 1.0.2 introduced a
        # new incompatible way to process .define and .undef
        if build_ext.define:
            print repr(build_ext.define)
            if type(build_ext.define) is types.StringType:
                # distutils < 1.0.2 needs this:
                l = string.split(build_ext.define, ',')
                build_ext.define = map(lambda symbol: (symbol, '1'), l)
            build_ext.define = build_ext.define + define
        else:
            build_ext.define = define
        if build_ext.undef:
            print repr(build_ext.undef)
            if type(build_ext.undef) is types.StringType:
                # distutils < 1.0.2 needs this:
                build_ext.undef = string.split(build_ext.undef, ',')
            build_ext.undef = build_ext.undef + undef
        else:
            build_ext.undef = undef

#
# mx MSVC Compiler extension
#
# We want some extra options for the MSVCCompiler, so we add them
# here. This is an awful hack, but there seems to be no other way to
# subclass a standard distutils C compiler class...

# remember old __init__
old_MSVCCompiler__init__ = MSVCCompiler.__init__

def mx_msvccompiler__init__(self, *args, **kws):

    apply(old_MSVCCompiler__init__, (self,) + args, kws)
    self.compile_options.extend(['/O2','/Gf','/GB','/GD'])

# "install" new __init__
MSVCCompiler.__init__ = mx_msvccompiler__init__

#
# mx Distribution class
#

class mx_Distribution(Distribution):

    """ Distribution class which knows about our distutils extensions.
        
    """
    # List of UnixLibrary instances
    unixlibs = None

    def has_unixlibs(self):
        return self.unixlibs and len(self.unixlibs) > 0

#
# mx Build command
#

class mx_build(build):

    """ build command which knows about our distutils extensions.
        
    """
    def has_unixlibs(self):
        return self.distribution.has_unixlibs()

    if len(build.sub_commands) > 4:
        raise DistutilsError, 'incompatible distutils version !'
    sub_commands = [('build_py',      build.has_pure_modules),
                    ('build_clib',    build.has_c_libraries),
                    ('build_unixlib', has_unixlibs),
                    ('mx_autoconf',   build.has_ext_modules),
                    ('build_ext',     build.has_ext_modules),
                    ('build_scripts', build.has_scripts),
                   ]

#
# mx Build Extensions command
#

class mx_build_ext(build_ext):

    """ build_ext command which runs mx_autoconf command before
        trying to build anything.
        
    """
    def run(self):

        if self.distribution.has_unixlibs():
            build_unixlib = self.get_finalized_command('build_unixlib')
            paths, libs = build_unixlib.get_unixlib_lib_options()
            self.libraries[:0] = libs
            self.library_dirs[:0] = paths
        self.run_command('mx_autoconf')
        build_ext.run(self)

#
# mx Build Python command
#

class mx_build_py(build_py):

    """ build_py command which also allows removing Python source code
        after the byte-code compile process.
        
    """
    user_options = build_py.user_options + [
        ('without-source', None, "only include Python byte-code"),
        ]

    boolean_options = build_py.boolean_options + ['without-source']

    # Omit source files ?
    without_source = 0
    
    def run(self):

        if self.without_source:
            # --without-source implies byte-code --compile
            if not self.compile and \
               not self.optimize:
                self.compile = 1

        # Build the Python code
        build_py.run(self)

        if self.without_source:
            self.announce('removing Python source files (--without-source)')
            verbose = self.verbose
            dry_run = self.dry_run
            for file in self.get_outputs(include_bytecode=0):
                # Remove source code
                execute(os.remove, (file,), "removing %s" % file,
                        verbose=verbose, dry_run=dry_run)
                # Remove .pyc files (if not requested)
                if not self.compile:
                    filename = file + "c"
                    if os.path.isfile(filename):
                        execute(os.remove, (filename,),
                                "removing %s" % filename,
                                verbose=verbose, dry_run=dry_run)
                # Remove .pyo files (if not requested)
                if self.optimize == 0:
                    filename = file + "o"
                    if os.path.isfile(filename):
                        execute(os.remove, (filename,),
                                "removing %s" % filename,
                                verbose=verbose, dry_run=dry_run)

#
# mx Build Unix Libs command
#
class UnixLibrary:

    """ Container for library configuration data.
    """
    # Name of the library
    libname = ''

    # Source tree where the library lives
    sourcetree = ''

    # List of library files and where to install them in the
    # build tree
    libfiles = None

    # Name of the configure script
    configure = 'configure'

    # Configure options
    configure_options = None

    # Make options
    make_options = None
    
    def __init__(self, libname, sourcetree, libfiles,
                 configure=None, configure_options=None,
                 make_options=None):

        self.libname = libname
        self.sourcetree = sourcetree
        # Check for 2-tuples...
        for libfile, targetdir in libfiles:
            pass
        self.libfiles = libfiles

        # Optional settings
        if configure:
            self.configure = configure
        if configure_options:
            self.configure_options = configure_options
        else:
            self.configure_options = []
        if make_options:
            self.make_options = make_options
        else:
            self.make_options = []
            
    def get(self, option, alternative=None):

        return getattr(self, option, alternative)

class mx_build_unixlib(Command):

    """ This command compiles external libs using the standard Unix
        procedure for this:
        
        ./configure
        make

    """
    description = "build Unix libraries used by Python extensions"

    # make program to use
    make = None
    
    user_options = [
        ('build-lib=', 'b',
         "directory to store built Unix libraries in"),
        ('build-temp=', 't',
         "directory to build Unix libraries to"),
        ('make=', None,
         "make program to use"),
        ('makefile=', None,
         "makefile to use"),
        ('force', 'f',
         "forcibly reconfigure"),
        ]
    
    boolean_options = ['force']

    def initialize_options (self):

        self.build_lib = None
        self.build_temp = None
        self.make = None
        self.makefile = None
        self.force = 0

    def finalize_options (self):

        self.set_undefined_options('build',
                                   ('verbose', 'verbose'),
                                   ('build_lib', 'build_lib'),
                                   ('build_temp', 'build_temp')
                                   )
        if self.make is None:
            self.make = 'make'
        if self.makefile is None:
            self.makefile = 'Makefile'

        # For debugging we are always in very verbose mode...
        self.verbose = 2

    def run_script(self, script, options=[]):

        if options:
            l = []
            for k, v in options:
                if v is not None:
                    l.append('%s=%s' % (k, v))
                else:
                    l.append(k)
            script = script + ' ' + string.join(l, ' ')
        if self.verbose > 1:
            self.announce('executing script %s' % repr(script))
        if self.dry_run:
            return 0
        try:
            rc = os.system(script)
        except DistutilsExecError, msg:
            raise CompileError, msg
        return rc
    
    def run_configure(self, options=[], dir=None, configure='configure'):

        """ Run the configure script using options is given.

            Options must be a list of tuples (optionname,
            optionvalue).  If an option should not have a value,
            passing None as optionvalue will have the effect of using
            the option without value.

            dir can be given to have the configure script execute in
            that directory instead of the current one.

        """
        cmd = './%s' % configure
        if dir:
            cmd = 'cd %s; ' % dir + cmd
        if self.verbose:
            self.announce('running %s in %s' % (configure, dir or '.'))
        rc = self.run_script(cmd, options)
        if rc != 0:
            raise CompileError, 'configure script failed'

    def run_make(self, targets=[], dir=None, make='make', options=[]):

        """ Run the make command for the given targets.

            Targets must be a list of valid Makefile targets.

            dir can be given to have the make program execute in that
            directory instead of the current one.

        """
        cmd = '%s' % make
        if targets:
            cmd = cmd + ' ' + string.join(targets, ' ')
        if dir:
            cmd = 'cd %s; ' % dir + cmd
        if self.verbose:
            self.announce('running %s in %s' % (make, dir or '.'))
        rc = self.run_script(cmd, options)
        if rc != 0:
            raise CompileError, 'make failed'

    def build_unixlib(self, unixlib):

        # Get options
        configure = unixlib.configure
        configure_options = unixlib.configure_options
        make_options = unixlib.make_options
        sourcetree = unixlib.sourcetree
        buildtree = os.path.join(self.build_temp, sourcetree)
        libfiles = unixlib.libfiles
        if not libfiles:
            raise DistutilsError, \
                  'no libfiles defined for unixlib "%s"' % unixlib.name
        if self.verbose:
            self.announce('building C lib %s in %s' % (unixlib.libname,
                                                       buildtree))
        # Prepare build
        if self.verbose:
            self.announce('preparing build of %s' % unixlib.libname)
        self.mkpath(buildtree)
        self.copy_tree(sourcetree, buildtree)
        
        # Run configure to build the Makefile
        if not os.path.exists(os.path.join(buildtree, self.makefile)) or \
           self.force:
            self.run_configure(configure_options,
                               dir=buildtree,
                               configure=configure)
        elif self.verbose:
            self.announce('skipping configure: %s is already configured' %
                          unixlib.libname)

        # Run make
        self.run_make(dir=buildtree,
                      make=self.make,
                      options=make_options)

        # Copy libs to destinations
        for sourcefile, destination in libfiles:
            if not destination:
                continue
            sourcefile = os.path.join(self.build_temp, sourcefile)
            destination = os.path.join(self.build_lib, destination)
            if not os.path.exists(sourcefile):
                raise CompileError, \
                      'library "%s" failed to build' % sourcefile
            self.mkpath(destination)
            self.copy_file(sourcefile, destination)

    def build_unixlibs(self, unixlibs):

        for unixlib in unixlibs:
            self.build_unixlib(unixlib)

    def get_unixlib_lib_options(self):

        libs = []
        paths = []
        for unixlib in self.distribution.unixlibs:
            for sourcefile, destination in unixlib.libfiles:
                if not destination:
                    # direct linking
                    sourcefile = os.path.join(self.build_temp, sourcefile)
                    libs.append(sourcefile)
                else:
                    # linking via link path and lib name
                    sourcefile = os.path.basename(sourcefile)
                    libname = os.path.splitext(sourcefile)[0]
                    if libname[:3] == 'lib':
                        libname = libname[3:]
                    libs.append(libname)
                    destination = os.path.join(self.build_lib, destination)
                    paths.append(destination)
        #print paths, libs
        return paths, libs

    def run(self):

        if not self.distribution.unixlibs:
            return
        self.build_unixlibs(self.distribution.unixlibs)

#
# mx Install Data command
#

class mx_install_data(install_data):

    def finalize_options(self):

        if self.install_dir is None:
            installobj = self.distribution.get_command_obj('install')
            self.install_dir = installobj.install_platlib
        install_data.finalize_options(self)

    def run (self):

        if not self.dry_run:
            self.mkpath(self.install_dir)
        data_files = self.get_inputs()
        for entry in data_files:
            if type(entry) == types.StringType:
                if 1 or os.name != 'posix':
                    # Unix- to platform-convention conversion
                    entry = string.join(string.split(entry, '/'), os.sep)
                filenames = glob.glob(entry)
                for filename in filenames:
                    dst = os.path.join(self.install_dir, filename)
                    dstdir = os.path.split(dst)[0]
                    if not self.dry_run:
                        self.mkpath(dstdir)
                        outfile = self.copy_file(filename, dst)[0]
                    else:
                        outfile = dst
                    self.outfiles.append(outfile)
            else:
                raise ValueError, 'tuples in data_files not supported'

#
# mx Uninstall command
#
# Credits are due to Thomas Heller for the idea and the initial code
# base for this command (he posted a different version to
# distutils-sig@python.org) in 02/2001.
#

class mx_uninstall(Command):

    description = "uninstall the package files and directories"

    user_options = []

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass
        
    def run(self):

        # Execute build
        self.announce('determining installation files')
        self.announce('(re)building package')
        savevalue = self.distribution.dry_run
        self.distribution.dry_run = 0
        self.run_command('build')

        # Execute install in dry-run mode
        self.announce('dry-run package install')
        self.distribution.dry_run = 1
        self.run_command('install')
        self.distribution.dry_run = savevalue
        build = self.get_finalized_command('build')
        install = self.get_finalized_command('install')

        # Remove all installed files
        self.announce("removing files")
        dirs = {}
        filenames = install.get_outputs()
        for filename in filenames:
            if not os.path.isabs(filename):
                raise DistutilsError,\
                      'filename %s from .get_output() not absolute' % \
                      filename

            if os.path.isfile(filename):
                self.announce("removing %s" % filename)
                if not self.dry_run:
                    try:
                        os.remove(filename)
                    except OSError, details:
                        self.warn("Could not remove file: %s" % details)
                    dir = os.path.split(filename)[0]
                    if not dirs.has_key(dir):
                        dirs[dir] = 1
                    if os.path.splitext(filename)[1] == '.py':
                        # Remove byte-code files as well
                        try:
                            os.remove(filename + 'c')
                        except OSError:
                            pass
                        try:
                            os.remove(filename + 'o')
                        except OSError:
                            pass

            elif os.path.isdir(filename):
                # This functionality is currently not being used by distutils
                if not dirs.has_key(dir):
                    dirs[filename] = 1

            elif not os.path.splitext(filename)[1] in ('.pyo', '.pyc'):
                self.announce("skipping removal of %s (not found)" %
                              filename)

        # Remove the installation directories
        self.announce("removing directories")
        dirs = dirs.keys()
        dirs.sort(); dirs.reverse() # sort descending
        for dir in dirs:
            self.announce("removing directory %s" % dir)
            if not self.dry_run:
                try:
                    os.rmdir(dir)
                except OSError, details:
                    self.warn("could not remove directory: %s" % details)

def run_setup(configurations):

    """ Run distutils setup.

        The parameters passed to setup() are extracted from the list
        of modules, classes or instances given in configurations.

        Names with leading underscore are removed from the parameters.
        Parameters which are not strings, lists or tuples are removed
        as well.  Configurations which occur later in the
        configurations list override settings of configurations
        earlier in the list.

    """
    # Build parameter dictionary
    kws = {}
    for configuration in configurations:
        kws.update(vars(configuration))
    for name, value in kws.items():
        if name[:1] == '_' or \
           type(value) not in (types.StringType,
                               types.ListType,
                               types.TupleType):
            del kws[name]

    # Add setup extensions
    kws['distclass'] = mx_Distribution
    extensions = {'build': mx_build,
                  'build_unixlib': mx_build_unixlib,
                  'mx_autoconf': mx_autoconf,
                  'build_ext': mx_build_ext,
                  'build_py': mx_build_py,
                  'install_data': mx_install_data,
                  'uninstall': mx_uninstall,
                  }
    if bdist_ppm is not None:
        extensions['bdist_ppm'] = bdist_ppm.bdist_ppm
    if GenPPD is not None:
        extensions['GenPPD'] = GenPPD.GenPPD
    kws['cmdclass'] = extensions

    # Invoke distutils setup
    apply(setup, (), kws)

