/* $Id: proto_xboard.cc,v 1.25 2002/06/17 03:24:31 bergo Exp $ */

/*

    eboard - chess client
    http://eboard.sourceforge.net
    Copyright (C) 2000-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include "eboard.h"
#include "global.h"
#include "network.h"
#include "protocol.h"
#include "chess.h"
#include "status.h"
#include "util.h"
#include "stl.h"
#include "tstring.h"

XBoardProtocol::XBoardProtocol() {
  EngineWhite=0;
  MoveNumber=1;
  MoveBlack=0;
  Variant=REGULAR;
  want_path_pane=1;
  supports_setboard=false;
  timeout_id=-1;
  need_handshake=true; // send xboard/protover 2

  got_init_pos=false;

  strcpy(ComputerName,"Computer");
  strcpy(EngineCommandLine,"gnuchess");
  strcpy(EngineRunDir,"~/.eboard/eng-out");

  Features.set("*feature*");
  IllegalMove.set("Illegal move*");

  Moved[0].set("move *");
  Moved[1].set("%n*... *");

  WhiteWin[0].set("1-0 {*}*");
  BlackWin[0].set("0-1 {*}*");
  Drawn[0].set   ("1/2-1/2 {*}*");
  WhiteWin[1].set("result 1-0*");
  BlackWin[1].set("result 0-1*");
  Drawn[1].set   ("result 1/2-1/2*");

  Dialect[0].set("White resigns*");
  Dialect[1].set("Black resigns*");
  Dialect[2].set("White %s*");
  Dialect[3].set("Black %s*");
  Dialect[4].set("Draw*");
  Dialect[5].set("computer mates*");
  Dialect[6].set("opponent mates*");
  Dialect[7].set("computer resigns*");
  Dialect[8].set("game is a draw*");
  Dialect[9].set("checkmate*");

  CurrentPosition.setStartPos();
  LegalityBackup.setStartPos();

  ebm=0;
}

void XBoardProtocol::setInitialPosition(Position *p) {

  initpos=(*p);
  got_init_pos=true;

}

// the "dialect" for game ends
char * XBoardProtocol::xlateDialect(char *s) {
  static char result[128];

  if (Dialect[0].match(s)) { strcpy(result,"0-1 { White resigns }"); return result; }
  if (Dialect[1].match(s)) { strcpy(result,"1-0 { Black resigns }"); return result; }
  if (Dialect[2].match(s)) { strcpy(result,"1-0 { White mates }"); return result; }
  if (Dialect[3].match(s)) { strcpy(result,"0-1 { Black mates }"); return result; }
  if (Dialect[4].match(s)) { strcpy(result,"1/2-1/2 { Draw }"); return result; }
  if (Dialect[5].match(s)) { strcpy(result,EngineWhite?"1-0 { White mates }":"0-1 { Black mates }"); return result; }
  if (Dialect[6].match(s)) { strcpy(result,EngineWhite?"0-1 { Black mates }":"1-0 { White mates }"); return result; }
  if (Dialect[7].match(s)) { strcpy(result,EngineWhite?"0-1 { White resigns }":"1-0 { Black resigns }"); return result; }
  if (Dialect[8].match(s)) { strcpy(result,"1/2-1/2 { Draw }"); return result; }
  if (Dialect[9].match(s)) { strcpy(result,EngineWhite?"1-0 { White mates }":"0-1 { Black mates }"); return result; }
  return 0;
}

void XBoardProtocol::receiveString(char *netstring) {
  ChessGame *refgame;
  char *p;
  char a[64],b[64];
  int i,moved;

  p=xlateDialect(netstring);
  if (p)
    netstring=p;

  if (Features.match(netstring)) {
    parseFeatures();
    goto just_print;
  }

  if (IllegalMove.match(netstring)) {

    if ((CurrentPosition.sidehint && EngineWhite) ||
	(!CurrentPosition.sidehint && !EngineWhite) ) {
      // engine sent illegal move string during HIS move
      // happens when FEN string is sent to GNU Chess 5 and
      // it doesn't understand the castling permissions
      global.output->append("engine claimed illegal move but we didn't move, ignoring it.", global.Colors.TextBright, IM_IGNORE);
      return;
    }

    CurrentPosition=LegalityBackup;
    refgame=getGame();
    backMove();
    refgame->retreat(1);
    refgame->updatePosition(CurrentPosition, MoveNumber,
			    EngineWhite?1:0, 
                            EngineWhite?0:CLOCK_UNCHANGED,
			    EngineWhite?CLOCK_UNCHANGED:0,
			    "illegal move!");
    global.status->setText(netstring);
    global.BoardList.front()->setCanMove(true);
    return;
  }

  moved=-1;
  if (Moved[0].match(netstring)) moved=0;
  if (Moved[1].match(netstring)) moved=1;

  if (moved>=0) {
    LegalityBackup=CurrentPosition;
    CurrentPosition.SANstring(Moved[moved].getStarToken(moved),a);
    CurrentPosition.moveAnyNotation(Moved[moved].getStarToken(moved),
                                    (MoveBlack?BLACK:WHITE),Variant);
    if (EngineWhite)
      sprintf(b,"%d. %s",MoveNumber,a);
    else
      sprintf(b,"%d. ... %s",MoveNumber,a);

    refgame=getGame();
    refgame->updatePosition(CurrentPosition, MoveNumber, 
                            !MoveBlack, 
			    EngineWhite?CLOCK_UNCHANGED:0,
			    EngineWhite?0:CLOCK_UNCHANGED,
			    b, true);
    advanceMove();
    global.BoardList.front()->setCanMove(true);
    global.opponentMoved();
    return;
  }

  for(i=0;i<2;i++) {
    if (WhiteWin[i].match(netstring)) {
      gameOver(WhiteWin[i],WHITE_WIN);
      goto just_print;
    }
    if (BlackWin[i].match(netstring)) {
      gameOver(BlackWin[i],BLACK_WIN);
      goto just_print;
    }
    if (Drawn[i].match(netstring)) {
      gameOver(Drawn[i],DRAW);
      goto just_print;
    }
  }
  
 just_print:
  global.output->append(netstring, global.Colors.Engine, IM_IGNORE);
}


void XBoardProtocol::advanceMove() {
  if (MoveBlack)
    MoveNumber++;
  MoveBlack=!MoveBlack;
}

void XBoardProtocol::backMove() {
  if (!MoveBlack)
    MoveNumber--;
  MoveBlack=!MoveBlack;
}

void XBoardProtocol::sendMove(int x1,int y1,int x2,int y2,int prom) {
  ChessGame *refgame;
  char move[6];
  char mymove[12],ppfmm[18]; // [p]retty [p]rint [f]or [m]y [m]ove
  piece pp;

  refgame=getGame();
  LegalityBackup=CurrentPosition;

  pp = prom ? global.promotion->getPiece() : EMPTY;

  CurrentPosition.stdNotationForMove(x1,y1,x2,y2,pp,mymove,refgame->Variant);
  CurrentPosition.moveCartesian(x1,y1,x2,y2,Variant,true);

  if (EngineWhite)
    sprintf(ppfmm,"%d. ... %s",MoveNumber,mymove);
  else
    sprintf(ppfmm,"%d. %s",MoveNumber,mymove);

  refgame->updatePosition(CurrentPosition,
                          MoveNumber,
                          !MoveBlack,
			  EngineWhite?0:CLOCK_UNCHANGED,
			  EngineWhite?CLOCK_UNCHANGED:0,
			  ppfmm);
  move[4]=0;
  move[5]=0;
  move[0]='a'+x1;
  move[1]='1'+y1;
  move[2]='a'+x2;
  move[3]='1'+y2;

  if (prom) {
    switch(pp) {
    case QUEEN:   move[4]='q'; break;
    case ROOK:    move[4]='r'; break;
    case BISHOP:  move[4]='b'; break;
    case KNIGHT:  move[4]='n'; break;
    case KING:    move[4]='k'; break;
    }
  }

  global.network->writeLine(move);
  advanceMove();
  global.BoardList.front()->setCanMove(false);
}

void XBoardProtocol::finalize() {
  ChessGame *refgame;
  refgame=getGame();
  if (refgame) {
    refgame->GameNumber=global.nextFreeGameId(XBOARD_GAME+1);
    refgame->endGame("interrupted",UNDEF);
  }
  if (global.network)
    global.network->close();
}

void XBoardProtocol::gameOver(ExtPatternMatcher &pm,GameResult gr,
                                    int hasreason=1)
{
  char reason[64];
  ChessGame *refgame;  

  if (hasreason) {
    reason[63]=0;
    strncpy(reason,pm.getStarToken(0),64);
  } else
    strcpy(reason,"Game over"); // I can't think of anything better...

  refgame=getGame();
  refgame->endGame(reason,gr);

  if (global.AppendPlayed) {
    if  (refgame->savePGN(global.AppendFile,true)) {
      char z[128];
      sprintf(z,"Game appended to %s",global.AppendFile);
      global.status->setText(z);
    }
  }

  global.network->close();
}

void XBoardProtocol::resign() {
  global.network->writeLine("resign");
  global.status->setText("Resigned.");
}

void XBoardProtocol::draw() {
  global.network->writeLine("draw");
  global.status->setText("<XBoard Protocol> draw request sent");
}

void XBoardProtocol::adjourn() {
  global.status->setText("<XBoard Protocol> Adjourning not supported");
}

void XBoardProtocol::abort() {
  finalize();
  global.status->setText("<XBoard Protocol> Session Aborted");
}

ChessGame * XBoardProtocol::getGame() {
  ChessGame *refgame;
  refgame=global.getGame(XBOARD_GAME);
  if (!refgame) {
    cerr << "* game not found: " << XBOARD_GAME << endl;
    exit(5);
  }
  return refgame;
}

int XBoardProtocol::run() {
  createDialog();
  if (!runDialog()) {
    destroyDialog();
    return 0;
  }

  ebm=0;

  readDialog();

  if (ebm!=0)
    global.addEngineBookmark(ebm);

  ebm=0;

  destroyDialog();
  return(loadEngine());
}

int XBoardProtocol::run(EngineBookmark *bm) {  
  loadBookmark(bm);
  return(loadEngine());
}

gboolean xb_initengine_timeout(gpointer data) {
  XBoardProtocol *me;
  me=(XBoardProtocol *) data;

  global.status->setText("Initializing engine");

  me->initEngine();
  return FALSE;
}

int XBoardProtocol::loadEngine() {
  PipeConnection *link;
  global.debug("XBoardProtocol","loadEngine",EngineCommandLine);

  if (strlen(EngineRunDir))
    sprintf(FullCommand,"cd %s ; %s",EngineRunDir,EngineCommandLine);
  else
    strcpy(FullCommand,EngineCommandLine);

  link=new PipeConnection("/bin/sh","-c",FullCommand,0,0);

  // only gnu chess 4 has trouble with it
  if (need_handshake)
    link->setHandshake("xboard\nprotover 2\n");

  if (link->open()==0) { 
    global.status->setText("Engine started (2 sec timeout)");
    global.network=link;

    // run initEngine after 2 seconds
    timeout_id=gtk_timeout_add(2000,xb_initengine_timeout,(gpointer) this);

    if (global.network->isConnected())
      return 1;
    else {
      global.network=0;      
      dumpGame();
    }
  }

  global.status->setText("Failed to run engine.");
  delete link;
  return 0;
}

void XBoardProtocol::initEngine() {
  char z[256];
  createGame();
  // xboard/protover already sent in open() call
  global.network->writeLine("nopost");

  if (ThinkAlways)
    global.network->writeLine("hard");
  else
    global.network->writeLine("easy");    

  global.network->writeLine("new");

  if (SecsPerMove > 0) {
    sprintf(z,"st %d",SecsPerMove);
    global.network->writeLine(z);
  }

  if (MaxDepth > 0) {
    sprintf(z,"sd %d",SecsPerMove);
    global.network->writeLine(z);
  }

  lateInit();
  endInit();
}

void XBoardProtocol::endInit() {
  char z[256];
  sprintf(z,"%s engine started.", ComputerName);
  global.status->setText(z);
}

void XBoardProtocol::lateInit() {
  if (got_init_pos) {    

    global.network->writeLine("force");

    setupPosition();

    // the a2a3 thing suggested in the protocol spec 
    // makes GNU chess 4 go nuts

    if (initpos.sidehint)
      global.network->writeLine("white");
    else
      global.network->writeLine("black");

    global.network->writeLine("go");
    
  } else {
    
    if (EngineWhite) {
      global.network->writeLine("white");
      global.network->writeLine("go");
    }
    
  }
}

void XBoardProtocol::setupPosition() {
  char z[256];

  if (!got_init_pos) return;

  if (supports_setboard) {
    string s;

    s="setboard ";
    s+=initpos.getFEN();
    strcpy(z,s.c_str());
    global.network->writeLine(z);    

  } else {
    char assoc[128];
    int i,j;
    piece p;

    global.network->writeLine("edit");
    global.network->writeLine("#");

    assoc[PAWN]  ='P';
    assoc[KNIGHT]='N';
    assoc[BISHOP]='B';
    assoc[ROOK]  ='R';
    assoc[QUEEN] ='Q';
    assoc[KING]  ='K';

    for(i=0;i<8;i++)
      for(j=0;j<8;j++) {
	p=initpos.getPiece(i,j);
	if ((p&COLOR_MASK) == WHITE) {
	  sprintf(z,"%c%c%d",assoc[p&PIECE_MASK],'a'+i,1+j);
	  global.network->writeLine(z);
	}
      }
    
    global.network->writeLine("c");

    for(i=0;i<8;i++)
      for(j=0;j<8;j++) {
	p=initpos.getPiece(i,j);
	if ((p&COLOR_MASK) == BLACK) {
	  sprintf(z,"%c%c%d",assoc[p&PIECE_MASK],'a'+i,1+j);
	  global.network->writeLine(z);
	}
      }

    global.network->writeLine(".");
  }

}

void XBoardProtocol::createGame() {
  ChessGame *cg;
  Position *pos;
  global.debug("XBoardProtocol","createGame");

  MoveNumber=1;
  MoveBlack=0;

  cg=new ChessGame();
  cg->GameNumber=XBOARD_GAME;
  cg->source = GS_Engine;
  strcpy(cg->PlayerName[EngineWhite?0:1],getComputerName());
  strcpy(cg->PlayerName[EngineWhite?1:0],global.env.User.c_str());

  cg->MyColor=EngineWhite?BLACK:WHITE;

  sprintf(cg->info0,"%d seconds per move",SecsPerMove);

  cg->clock_regressive=0;
  cg->Rated=0;
  cg->Variant=Variant;
  global.appendGame(cg);
  global.BoardList.front()->reset();
  cg->setBoard(global.BoardList.front());
  global.BoardList.front()->setGame(cg);
  global.BoardList.front()->pop();
  global.BoardList.front()->setCanMove(true);
  global.BoardList.front()->repaint();
  if (EngineWhite)
    global.BoardList.front()->setFlipped(true);
  cg->acknowledgeInfo();
  cg->fireWhiteClock(0,0);

  pos=new Position();

  if (got_init_pos) {
    (*pos)=initpos;
    CurrentPosition = initpos;
    LegalityBackup  = initpos;
    MoveBlack = pos->sidehint ? 0 : 1;
  }

  cg->updatePosition(*pos,1,pos->sidehint ? 0:1,0,0,"0. startpos");
  delete pos;
}

void XBoardProtocol::dumpGame() {
  ChessGame *cg;

  cg=global.getGame(XBOARD_GAME);
  if (!cg) return;

  if (!cg->isOver()) cg->endGame("*",UNDEF);
  global.BoardList.front()->reset();

}

void XBoardProtocol::parseFeatures() {
  ExtPatternMatcher sf,nf;
  ChessGame *cg;
  char *cand,*p;
  int n;
  bool gotdone=false;

  sf.set("%b%s=\"*\"*");
  nf.set("%b%s=%n*");
  cand=Features.getStarToken(1);
  
  while(1) {
    if (sf.match(cand)) {
      p=sf.getSToken(0);
      if (!strcmp(p,"myname")) {
	ComputerName[63]=0;
	strncpy(ComputerName,sf.getStarToken(0),64);
	global.status->setText(ComputerName);
	cg=global.getGame(XBOARD_GAME);
	if (cg) {
	  strncpy(cg->PlayerName[EngineWhite?0:1],ComputerName,64);
	  cg->acknowledgeInfo();
	}
      }
      cand=sf.getStarToken(1);
      continue;
    }
    if (nf.match(cand)) {
      p=nf.getSToken(0);

      if (!strcmp(p,"setboard")) {
	n=atoi(nf.getNToken(0));
	if (n) supports_setboard=true;
      }

      if (!strcmp(p,"done")) {
	n=atoi(nf.getNToken(0));
	if (n)
	  gotdone=true;
	else {
	  // engine asked more time
	  global.status->setText("Engine asked more time to startup, waiting.");
	  if (timeout_id >= 0)
	    gtk_timeout_remove(timeout_id);
	  
	  // 1 hour, as per Mann's doc
	  timeout_id=gtk_timeout_add(3600000, xb_initengine_timeout,
				     (gpointer) this);
	}
	
      }

      cand=nf.getStarToken(0);
      continue;
    }
    break;
  }

  if (gotdone && timeout_id >= 0) {
    gtk_timeout_remove(timeout_id);
    timeout_id=-1;
    global.status->setText("Engine loaded.");
    initEngine();
  }
}

char * XBoardProtocol::getDialogName() {
  return("Play against Engine");
}

char * XBoardProtocol::getComputerName() {
  return(ComputerName);
}

// ----- the bloated world of GUI -------

void XBoardProtocol::createDialog() {
  GtkWidget *v,*v2;
  GtkWidget *cfr,*rd[2],*ok,*cancel,*bhb,*hs;
  GtkWidget *f2,*tabl,*l2,*spm,*tl1,*tl2,*te1,*tv;
  GSList *rg;  

  GtkWidget *bp_l,*bp_v, *pp_l, *pp_v;
  GtkWidget *h3,*l3,*path3,*b31,*l31,*h4,*l4,*path4;

  eng_dlg=gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(eng_dlg),getDialogName());
  gtk_window_set_position(GTK_WINDOW(eng_dlg),GTK_WIN_POS_CENTER);
  gtk_container_set_border_width(GTK_CONTAINER(eng_dlg),6);
  gtk_widget_realize(eng_dlg);

  v=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(eng_dlg),v);

  eng_book=gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(v),eng_book,TRUE,TRUE,0);
  
  // basic pane
  bp_v=gtk_vbox_new(FALSE,0);
  bp_l=gtk_label_new("Side & Time");
  gtk_container_set_border_width(GTK_CONTAINER(bp_v),4);
  cfr=gtk_frame_new("Side Selection");
  gtk_frame_set_shadow_type(GTK_FRAME(cfr),GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(bp_v),cfr,FALSE,TRUE,4);
  v2=gtk_vbox_new(TRUE,2);
  gtk_container_add(GTK_CONTAINER(cfr),v2);
  rd[0]=gtk_radio_button_new_with_label( 0, "Human White vs. Computer Black" );
  rg=gtk_radio_button_group(GTK_RADIO_BUTTON(rd[0]));
  rd[1]=gtk_radio_button_new_with_label(rg, "Computer White vs. Human Black" );
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rd[0]), TRUE);
  gtk_box_pack_start(GTK_BOX(v2),rd[0],FALSE,TRUE,2);
  gtk_box_pack_start(GTK_BOX(v2),rd[1],FALSE,TRUE,2);
  // time per move
  f2=gtk_frame_new("Time Control");
  gtk_frame_set_shadow_type(GTK_FRAME(f2),GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(bp_v),f2,FALSE,TRUE,4);  
  tv=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(f2),tv);
  tl1=gtk_label_new("Set any of these fields to zero to\nuse the engine's default value.");
  gtk_box_pack_start(GTK_BOX(tv),tl1,FALSE,FALSE,2);
  tabl=gtk_table_new(3,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(tabl),4);
  gtk_table_set_row_spacings(GTK_TABLE(tabl),2);
  gtk_table_set_col_spacings(GTK_TABLE(tabl),2);
  gtk_box_pack_start(GTK_BOX(tv),tabl,FALSE,FALSE,2);
  l2=gtk_label_new("Seconds per move:");
  spm=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(spm),"20");
  gtk_table_attach_defaults(GTK_TABLE(tabl),l2,0,1,0,1);
  gtk_table_attach_defaults(GTK_TABLE(tabl),spm,1,2,0,1);
  tl2=gtk_label_new("Depth Limit:");
  te1=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(te1),"0");
  gtk_table_attach_defaults(GTK_TABLE(tabl),tl2,0,1,1,2);
  gtk_table_attach_defaults(GTK_TABLE(tabl),te1,1,2,1,2);

  think=gtk_check_button_new_with_label("Think on opponent's time");
  gtk_table_attach_defaults(GTK_TABLE(tabl),think,0,2,2,3);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(think), 1);

  bookmark=gtk_check_button_new_with_label("Add to Peer/Engine Bookmarks menu");
  gtk_box_pack_start(GTK_BOX(v),bookmark,FALSE,FALSE,2);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bookmark), 1);

  gtk_widget_show(bookmark);
  gtk_widget_show(think);
  gtk_widget_show(te1);
  gtk_widget_show(tl2);
  gtk_widget_show(tl1);
  gtk_widget_show(tabl);
  gtk_widget_show(tv);
  gtk_widget_show(f2);
  gtk_widget_show(l2);
  gtk_widget_show(spm);
  gtk_widget_show(rd[1]);
  gtk_widget_show(rd[0]);
  gtk_widget_show(v2);
  gtk_widget_show(cfr);
  gtk_notebook_append_page(GTK_NOTEBOOK(eng_book),bp_v,bp_l);
  gtk_widget_show(bp_v);
  gtk_widget_show(bp_l);

  // path pane
  if (want_path_pane) {
    pp_v=gtk_vbox_new(FALSE,0);
    pp_l=gtk_label_new("Engine Command");
    gtk_container_set_border_width(GTK_CONTAINER(pp_v),4);
    h3=gtk_hbox_new(FALSE,0);
    l3=gtk_label_new("Engine command line");
    gtk_box_pack_start(GTK_BOX(pp_v),h3,FALSE,FALSE,2);
    gtk_box_pack_start(GTK_BOX(h3),l3,FALSE,FALSE,0);
  }
  path3=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(path3),EngineCommandLine);  
  if (want_path_pane) {
    gtk_box_pack_start(GTK_BOX(pp_v),path3,FALSE,TRUE,2);
    h4=gtk_hbox_new(FALSE,0);
    l4=gtk_label_new("Directory to run from (e.g., where book files are)");
    gtk_box_pack_start(GTK_BOX(pp_v),h4,FALSE,FALSE,2);
    gtk_box_pack_start(GTK_BOX(h4),l4,FALSE,FALSE,0);
  }
  path4=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(path4),EngineRunDir);
  if (want_path_pane) {
    gtk_box_pack_start(GTK_BOX(pp_v),path4,FALSE,TRUE,2);
    b31=gtk_hbox_new(FALSE,0);
    l31=gtk_label_new("The engine will be run with\n/bin/sh -c 'cd directory ; command line'");
    gtk_label_set_justify(GTK_LABEL(l31),GTK_JUSTIFY_LEFT);
    gtk_box_pack_start(GTK_BOX(b31),l31,FALSE,FALSE,2);
    gtk_box_pack_start(GTK_BOX(pp_v),b31,FALSE,FALSE,2);
    gtk_widget_show(h3);
    gtk_widget_show(l3);
    gtk_widget_show(l31);
    gtk_widget_show(b31);
    gtk_widget_show(path3);
    gtk_widget_show(h4);
    gtk_widget_show(l4);
    gtk_widget_show(path4);
    gtk_notebook_append_page(GTK_NOTEBOOK(eng_book),pp_v,pp_l);
    gtk_widget_show(pp_v);
    gtk_widget_show(pp_l);
  }

  // bottom box
  hs=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(v),hs,FALSE,TRUE,4);

  bhb=gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(bhb), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(bhb), 5);
  gtk_box_pack_start(GTK_BOX(v),bhb,FALSE,FALSE,4);
  ok=gtk_button_new_with_label    ("OK");
  GTK_WIDGET_SET_FLAGS(ok,GTK_CAN_DEFAULT);
  cancel=gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS(cancel,GTK_CAN_DEFAULT);
  gtk_box_pack_start(GTK_BOX(bhb),ok,TRUE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(bhb),cancel,TRUE,TRUE,0);
  gtk_widget_grab_default(ok);
  gtk_widget_show(ok);
  gtk_widget_show(cancel);
  gtk_widget_show(bhb);
  gtk_widget_show(hs);
  gtk_widget_show(v);

  gtk_signal_connect(GTK_OBJECT(ok), "clicked",
                     GTK_SIGNAL_FUNC(xboard_eng_ok),this);
  gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
                     GTK_SIGNAL_FUNC(xboard_eng_cancel),this);
  gtk_signal_connect(GTK_OBJECT(eng_dlg), "delete_event",
                     GTK_SIGNAL_FUNC(xboard_eng_delete),0);

  gtk_widget_show(eng_book);
  gtk_notebook_set_page(GTK_NOTEBOOK(eng_book),0);

  engctl_spm=spm;
  engctl_ply=te1;
  engctl_ewhite=rd[1];
  engctl_engcmd=path3;
  engctl_engdir=path4;

  if (got_init_pos) {
    if (initpos.sidehint) // white to move
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rd[1]), TRUE);
    else
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rd[0]), TRUE);

    gtk_widget_set_sensitive(rd[0],FALSE);
    gtk_widget_set_sensitive(rd[1],FALSE);
  }
}

void XBoardProtocol::readDialog() {

  EngineWhite=0;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(engctl_ewhite)))
    EngineWhite=1;
  SecsPerMove=atoi(gtk_entry_get_text(GTK_ENTRY(engctl_spm)));
  MaxDepth=atoi(gtk_entry_get_text(GTK_ENTRY(engctl_ply)));
  strncpy(EngineCommandLine,gtk_entry_get_text(GTK_ENTRY(engctl_engcmd)),512);
  strncpy(EngineRunDir,gtk_entry_get_text(GTK_ENTRY(engctl_engdir)),256);
  ThinkAlways=(bool)gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(think));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(bookmark))) {
    ebm=new EngineBookmark();
    ebm->secspermove=SecsPerMove;
    ebm->maxply     =MaxDepth;
    ebm->humanwhite =1-EngineWhite;
    ebm->think      =ThinkAlways?1:0;
    ebm->directory  =EngineRunDir;
    ebm->cmdline    =EngineCommandLine;
    ebm->mode       =REGULAR;
    ebm->proto      =0; // base xboard protocol

    makeBookmarkCaption();

    // addition is done in run() to allow all virtual versions of this
    // to run before doing it

  } else {
    ebm=0; // derivated classes should check ebm!=0 before doing oddities
  }
}

void XBoardProtocol::makeBookmarkCaption() {
  tstring t,u;
  string x, *p;
  char caption[1024];

  if (!ebm) return;

  t.set(EngineCommandLine);
  x=EngineCommandLine;
  if (x.empty()) x="<blank?>";
  
  p=t.token(" ");
  if (p != 0) {
    u.set(*p);
    while( (p=u.token("/")) != 0 )
      x=*p;
  }
  
  sprintf(caption,"Play %s as %s vs. %s (%d secs/move, maxdepth %d, think always: %s)",
	  ChessGame::variantName(ebm->mode),
	  ebm->humanwhite?"White":"Black",
	  x.c_str(),
	  ebm->secspermove,
	  ebm->maxply,
	  ebm->think?"yes":"no");
  ebm->caption = caption;
}

void XBoardProtocol::loadBookmark(EngineBookmark *bm) {
  SecsPerMove = bm->secspermove;
  MaxDepth    = bm->maxply;
  ThinkAlways = (bm->think!=0);
  EngineWhite = 1-bm->humanwhite;
  Variant     = bm->mode;
  strcpy(EngineRunDir, bm->directory.c_str() );
  strcpy(EngineCommandLine, bm->cmdline.c_str() );

  if (got_init_pos)
    EngineWhite=initpos.sidehint ? 1 : 0;
}

int XBoardProtocol::runDialog() {
  GotResult=0;
  gtk_widget_show(eng_dlg);
  gtk_grab_add(eng_dlg);
  while(!GotResult)
    global.WrappedMainIteration();
  gtk_grab_remove(eng_dlg);
  return(GotResult==1);
}

void XBoardProtocol::destroyDialog() {
  gtk_widget_destroy(eng_dlg);
}

void xboard_eng_ok(GtkWidget *w,gpointer data) {
  XBoardProtocol *me;
  me=(XBoardProtocol *)data;
  me->GotResult=1;
}

void xboard_eng_cancel(GtkWidget *w,gpointer data) {
  XBoardProtocol *me;
  me=(XBoardProtocol *)data;
  me->GotResult=-1;
}

gboolean xboard_eng_delete(GtkWidget *w,GdkEvent *e,gpointer data) {
  return TRUE;
}
