/* browser.c - 2000/04/28 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/* Some parts of this browser are taken from:
 * XMMS - Cross-platform multimedia player
 * Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas,
 * Thomas Nilsson and 4Front Technologies
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "easytag.h"
#include "browser.h"
#include "et_core.h"
#include "msgbox.h"
#include "bar.h"
#include "misc.h"
#include "setting.h"
#include "i18n.h"

/* Pixmaps */
#include "../pixmaps/opened_folder.xpm"
#include "../pixmaps/closed_folder.xpm"
#include "../pixmaps/closed_folder_locked.xpm"
#include "../pixmaps/parent_folder.xpm"

/* Returns the GList item for the nth row (taken from gtkclist.c) */
#define    ROW_ELEMENT(clist, row) (((row) == (clist)->rows - 1) ? \
                                   (clist)->row_list_end : \
                                   g_list_nth ((clist)->row_list, (row)))


/****************
 * Declarations *
 ****************/
static GdkPixmap *opened_folder_pixmap, *closed_folder_pixmap, *closed_folder_locked_pixmap;
static GdkBitmap *opened_folder_mask,   *closed_folder_mask,   *closed_folder_locked_mask;

GtkWidget *BrowserTree;    // Tree of directories
GtkWidget *BrowserList;    // List of files
GtkWidget *BrowserEntry;
GtkWidget *BrowserLabel;
GtkWidget *BrowserButton;
gchar     *BrowserCurrentPath = NULL; // Path selected in the browser area (BrowserEntry or BrowserTree)

GtkCTreeNode *RootNode;
GtkWidget *RenameDirectoryWindow = NULL;
GtkWidget *RunProgramTreeWindow = NULL;
GtkWidget *RunProgramListWindow = NULL;

guint blrs_idle_handler_id = 0;
guint blru_idle_handler_id = 0;
guint bl_row_selected;


GdkColor LIGHT_BLUE = {0, 0xddd1, 0xeeec, 0xffff}; // Light blue
GdkColor RED        = {0, 0xffff, 0x0000, 0x0000}; // Red
GdkColor LIGHT_RED  = {0, 0xffff, 0x8888, 0x8888}; // Light red


/**************
 * Prototypes *
 **************/

static gint Browser_Tree_Key_Press        (GtkWidget *ctree, GdkEvent *event);
void        Browser_Tree_Set_Node_Visible (GtkCTree *ctree, GtkCTreeNode *node);
void        Browser_Tree_Disable          (void);
void        Browser_Tree_Enable           (void);
void        Browser_Tree_Initialize       (void);
void        Browser_Tree_Node_Selected    (GtkCTree *tree, GtkCTreeNode *node, gint column, gpointer data);
void        Browser_Tree_Rename_Directory (gchar *last_path, gchar *new_path);

static gint Browser_List_Key_Press       (GtkWidget *clist, GdkEvent *event);
static gint Browser_List_Button_Press    (GtkWidget *clist, GdkEventButton *event);
void        Browser_List_Set_Row_Visible (GtkCList *clist, gint row);
void        Browser_List_Disable         (void);
void        Browser_List_Enable          (void);
void        Browser_List_Row_Selected    (GtkCList *clist, gint row, gint column,
                                          GdkEventButton *event, gpointer data);
void        Browser_List_Row_Unselected  (GtkCList *clist, gint row, gint column,
                                          GdkEventButton *event, gpointer data);
void        Browser_List_Row_Selected_Idle      (void);
void        Browser_List_Row_Unselected_Idle    (void);
void        Browser_List_Select_All_Files       (void);
void        Browser_List_Unselect_All_Files     (void);
void        Browser_List_Invert_File_Selection  (void);

void        Browser_Entry_Activated (void);
void        Browser_Entry_Disable   (void);
void        Browser_Entry_Enable    (void);

void        Browser_Button_Clicked  (void);

gchar      *Browser_Get_Current_Path       (void);
void        Browser_Update_Current_Path    (gchar *path);
void        Browser_Load_Home_Directory    (void);
void        Browser_Load_Default_Directory (void);
void        Browser_Reload_Directory       (void);

static gboolean check_for_subdir   (gchar *path);
static void     destroy_ctb        (gpointer data);
static void     destroy_clb        (gpointer data);
gint            Find_Children_Node (gconstpointer a, gconstpointer b);

gboolean    Check_For_Access_Permission (gchar *path);
static void expand_cb   (GtkWidget *widget, GtkCTreeNode *parent_node);
static void collapse_cb (GtkWidget *ctree, GtkCTreeNode *parent_node);
static int  filetreeent_compare_func (const void *a, const void *b);

static gint sort_tree_compare_func (GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2);

/* Pop up menus */
gboolean    Browser_Popup_Menu (GtkMenu *menu, GdkEventButton *event);
GtkWidget  *Create_Browser_Tree_Popup_Menu (GtkCTree *ctree);
GtkWidget  *Create_Browser_List_Popup_Menu (GtkCList *clist);

/* For window to rename a directory */
void        Browser_Open_Rename_Directory_Window (void);
void        Destroy_Rename_Directory_Window (void);
void        Rename_Directory (GtkObject *combobox);
void        Rename_Directory_Window_Key_Press (GtkWidget *window, GdkEvent *event);

/* For window to run a program with the directory */
void        Browser_Open_Run_Program_Tree_Window (void);
void        Destroy_Run_Program_Tree_Window (void);
void        Run_Program_Tree_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void        Run_Program_With_Directory (GtkObject *combobox);
/* For window to run a program with the file */
void        Browser_Open_Run_Program_List_Window (void);
void        Destroy_Run_Program_List_Window (void);
void        Run_Program_List_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void        Run_Program_With_Selected_Files (GtkObject *combobox);

gboolean    Run_Program (gchar *program_name, GList *args_list);



/*************
 * Functions *
 *************/
/*
 * Load home directory
 */
void Browser_Load_Home_Directory (void)
{
    Browser_Tree_Select_Dir(HOME_VARIABLE);
}


/*
 * Load default directory
 */
void Browser_Load_Default_Directory (void)
{
    if (DEFAULT_PATH_TO_MP3 && strlen(DEFAULT_PATH_TO_MP3)>0)
        Browser_Tree_Select_Dir(DEFAULT_PATH_TO_MP3);
    else
        Browser_Tree_Select_Dir(HOME_VARIABLE);
}


/*
 * Get the path from the selected node (row) in the browser
 * Warning: return NULL if no row selected int the tree.
 */
gchar *Browser_Tree_Get_Path_Of_Selected_Node (void)
{
    GList *SelectedNode = NULL;
    DirNode *dirnode = NULL;

    SelectedNode = ((GtkCList*)BrowserTree)->selection;
    if (!SelectedNode) 
        return NULL;
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(BrowserTree),SelectedNode->data);

    if (!dirnode) 
        return NULL;
    else
        return dirnode->path;
}


/*
 * Set the 'path' within the variable BrowserCurrentPath.
 */
void Browser_Update_Current_Path (gchar *path)
{
    /* Be sure that we aren't passing 'BrowserCurrentPath' as parameter of the function : 
     * to avoid some memory problems */
    if (path == NULL || path == BrowserCurrentPath) return;

    if (BrowserCurrentPath != NULL)
        g_free(BrowserCurrentPath);
    BrowserCurrentPath = g_strdup(path);
    
    if (!strcmp("/",BrowserCurrentPath))
        gtk_widget_set_sensitive(BrowserButton,FALSE);
    else
        gtk_widget_set_sensitive(BrowserButton,TRUE);
}


/*
 * Return the current path
 */
gchar *Browser_Get_Current_Path (void)
{
    return BrowserCurrentPath;
}


/*
 * Reload the current directory.
 */
void Browser_Reload_Directory (void)
{
    if (BrowserCurrentPath != NULL)
        Browser_Tree_Select_Dir(BrowserCurrentPath);
}


/*
 * Set the current path (selected node) in browser as default path (within config variable).
 */
void Set_Current_Path_As_Default (void)
{
    if (DEFAULT_PATH_TO_MP3 != NULL)
        g_free(DEFAULT_PATH_TO_MP3);
    DEFAULT_PATH_TO_MP3 = g_strdup(BrowserCurrentPath);
    Statusbar_Message(_("New default path for files selected"),TRUE);
}


/*
 * When you press the key 'enter' in the BrowserEntry to validate the text (browse the directory)
 */
void Browser_Entry_Activated (void)
{
    gchar *path;

    Add_To_Combo_Box_History(GTK_OBJECT(BrowserEntry));
    path = gtk_entry_get_text_1(BrowserEntry);
    Browser_Tree_Select_Dir(path);
}

/*
 * Set a text into the BrowserEntry (and don't activate it)
 */
void Browser_Entry_Set_Text (gchar *text)
{
    if (text && BrowserEntry)
    {
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(BrowserEntry)->entry),text);
        /* Justify to the rigth, text into BrowserEntry */
        gtk_editable_set_position(GTK_EDITABLE(GTK_COMBO(BrowserEntry)->entry),-1);
    }
}


/*
 * Button to go to parent directory
 */
void Browser_Button_Clicked (void)
{
    gchar *parent_dir, *path;
    
    parent_dir = g_strdup(Browser_Get_Current_Path());
    if (strlen(parent_dir)>1)
    {
        if ( parent_dir[strlen(parent_dir)-1]=='/' )
            parent_dir[strlen(parent_dir)-1] = '\0';
        path = g_dirname(parent_dir);
        
        Browser_Tree_Select_Dir(path);
        g_free(path);
    }
    g_free(parent_dir);
}


/*
 * Set a text into the BrowserLabel
 */
void Browser_Label_Set_Text (gchar *text)
{
    if (BrowserLabel && text)
        gtk_label_set_text(GTK_LABEL(BrowserLabel),text?text:"");
}


/*
 * Key Press events into browser tree
 */
static gint Browser_Tree_Key_Press (GtkWidget *ctree, GdkEvent *event)
{
    GdkEventKey *kevent;
    GList *SelectedNode;

    if (event && event->type==GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        SelectedNode = ((GtkCList*)ctree)->selection;
        if (!SelectedNode) 
            return FALSE;
        switch(kevent->keyval)
        {
            case GDK_KP_Enter:    /* Enter key in Num Pad */
            case GDK_Return:      /* 'Normal' Enter key */
            case GDK_t:           /* Expand/Collapse node */
            case GDK_T:           /* Expand/Collapse node */
                gtk_ctree_toggle_expansion(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;

            case GDK_e:           /* Expand node */
            case GDK_E:           /* Expand node */
                gtk_ctree_expand(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;

            case GDK_c:           /* Collapse node */
            case GDK_C:           /* Collapse node */
                gtk_ctree_collapse(GTK_CTREE(ctree),SelectedNode->data);
                return TRUE;
                break;
            //default:
            //    gtk_widget_event(MainWindow,(GdkEvent *)event);
            //    break;
        }
    }
    return FALSE;
}


/*
 * Key Press events into browser list
 */
gboolean Browser_List_Stop_Timer (guint *flag)
{
    *flag = FALSE;
    return FALSE;
}
static gint Browser_List_Key_Press (GtkWidget *clist, GdkEvent *event)
{
    gchar *string, *current_filename;
    static gchar *key_string = NULL;
    static guint BrowserListTimerId = 0;
    static gboolean timer_is_running = FALSE;
    gint row;
    FileRow *filerow = NULL;
    GdkEventKey *kevent;

    kevent = (GdkEventKey *)event;
    if (event && event->type==GDK_KEY_PRESS)
    {
        if ( ((GtkCList*)clist)->selection )
        {
            switch(kevent->keyval)
            {
                case GDK_Delete:
                    Action_Delete_Selected_Files();
                    return TRUE;
            }
        }
    }

    /*
     * Tries to select file corresponding to the character sequence entered
     */
    if ( strlen(kevent->string)>0 ) // Alphanumeric key?
    {
        // If the timer is running: concatenate the character of the pressed key, else starts a new string
        string = key_string;
        if (timer_is_running)
            key_string = g_strconcat(key_string,kevent->string,NULL);
        else
            key_string = g_strdup(kevent->string);
        if (string) g_free(string);

        // Remove the current timer
        if (BrowserListTimerId)
        {
            gtk_timeout_remove(BrowserListTimerId);
            BrowserListTimerId = 0;
        }
        // Start a new timer of 500ms
        BrowserListTimerId = gtk_timeout_add(500,(GtkFunction)Browser_List_Stop_Timer,&timer_is_running);
        timer_is_running = TRUE;

        // Browse the list keeping the current classification
        for (row=0; row<((GtkCList *)clist)->rows; row++)
        {
            filerow = gtk_clist_get_row_data(GTK_CLIST(clist),row);
            current_filename = g_basename( ((File_Name *)filerow->ETFile->FileNameCur->data)->value );
            if (strncasecmp(current_filename,key_string,strlen(key_string))==0 )
            {
                gtk_clist_select_row(GTK_CLIST(clist),row,0);
                break;
            }
        }
    }
    return FALSE;
}


/*
 * Action for double/triple click
 */
static gint Browser_List_Button_Press (GtkWidget *clist, GdkEventButton *event)
{
    if (!event)
        return FALSE;
    
    if (event->type==GDK_2BUTTON_PRESS && event->button==1)
    {
        // Select files of the same directory (usefull when browsing sub-directories)
        GList *etfilelist = NULL;
        gchar *path_ref = NULL;
        gchar *patch_check = NULL;
        
        // File taken as reference...
        path_ref = g_dirname( ((File_Name *)ETFileDisplayed->FileNameCur->data)->value );

        // Search and select files of the same directory
        etfilelist = g_list_first(ETFileList);
        while (etfilelist)
        {
            patch_check = g_dirname( ((File_Name *)((ET_File *)etfilelist->data)->FileNameCur->data)->value );

            if ( path_ref && patch_check && strcmp(path_ref,patch_check)==0 )
            {
                Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
            }
            etfilelist = g_list_next(etfilelist);
            if (patch_check) g_free(patch_check);
        }
        if (path_ref) g_free(path_ref);
    }else if (event->type==GDK_3BUTTON_PRESS && event->button==1)
    {
        // Select all files of the list
        Action_Select_All_Files();
    }
    return TRUE;
}


/*
 * Collapse (close) tree recursively up to the root node.
 */
void Browser_Tree_Collapse (void)
{
    gtk_ctree_collapse_recursive(GTK_CTREE(BrowserTree),RootNode);
    /* But keep the main directory opened */
    gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
}


/*
 * Set a row (or node) visible in the ctree (by scrolling the tree) if the row is hidden
 */
void Browser_Tree_Set_Node_Visible (GtkCTree *ctree, GtkCTreeNode *node)
{
    GtkVisibility visible;

    visible = gtk_ctree_node_is_visible(ctree,node);
    switch (visible)
    {
        case GTK_VISIBILITY_NONE:
        case GTK_VISIBILITY_PARTIAL:
            gtk_ctree_node_moveto(ctree,node,0,0.1,0.0);
            break;
        case GTK_VISIBILITY_FULL:
            break;
    }
}


/*
 * Set a row visible in the clist (by scrolling the list) if the row is hidden
 */
void Browser_List_Set_Row_Visible (GtkCList *clist, gint row)
{
    GtkVisibility visible;

    visible = gtk_clist_row_is_visible(clist,row);
    switch (visible)
    {
        case GTK_VISIBILITY_NONE:
        case GTK_VISIBILITY_PARTIAL:
            gtk_clist_moveto(clist,row,0,0.5,0.0);
            break;
        case GTK_VISIBILITY_FULL:
            break;
    }
}


//void Browser_Tree_Node_Selected (GtkCTree *tree, GList *node, gint column ATTRIBUTE_UNUSED, gpointer data ATTRIBUTE_UNUSED)
void Browser_Tree_Node_Selected (GtkCTree *tree, GtkCTreeNode *node, gint column ATTRIBUTE_UNUSED, gpointer data ATTRIBUTE_UNUSED)
{
    DirNode *dirnode = NULL;
    static int counter = 0;


    /* Open the node */
    if (OPEN_SELECTED_BROWSER_NODE)
        gtk_ctree_expand(GTK_CTREE(tree),node);

    /* Don't start a new reading, if an other one is running...*/
    if (ReadingDirectory == TRUE) return;

    Browser_Tree_Set_Node_Visible(tree,node);
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(tree),node);
    if (dirnode && dirnode->path)
    {
        /* Save the current displayed data */
        ET_Save_File_Data_From_UI(ETFileDisplayed);
        Update_Command_Buttons_Sensivity(); // Not clean to put this here...

        /* Check if all files have been saved before changing the directory */
        if (ET_Check_If_All_Files_Are_Saved() != TRUE)
        {
            GtkWidget *msgbox = NULL;
            gint button;

            msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
                "saved...\nDo you want to save them before changing the directory?"),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            switch(button)
            {
                case BUTTON_YES:
                    if (Save_Selected_Files_With_Answer()==-1) return;
                    break;
                case BUTTON_NO:
                    break;
                case BUTTON_CANCEL:
                case -1:
                    return;
            }
        }
        
        /* Memorize the current path */
         Browser_Update_Current_Path(dirnode->path);
        
        /* Display the selected path into the BrowserEntry */
        gtk_entry_set_text_1(BrowserEntry,dirnode->path);

        /* Start to read the directory */
        /* The first time 'counter' is equal to zero and if we don't want to load 
         * directory on startup, we skip the 'reading', but we must read it */
        if (LOAD_ON_STARTUP || counter)
            Read_Directory(dirnode->path);
        else
            /* As we don't use the function 'Read_Directory' we must add this function here */
            Update_Command_Buttons_Sensivity();
        counter++;
    }
}



/*
 * Browser_Tree_Select_Dir: Select the directory corresponding to the 'path' in
 * the tree browser, but it doesn't read it!
 * Check if path is correct before selecting it. And returns 1 on success, else 0.
 */
gint Browser_Tree_Select_Dir (gchar *current_path)
{
    struct stat stbuf;
    GtkCTreeNode *node, *nextnode;
    gchar *currentdir,*pos,*tpath,*tpathnew;
    gboolean leaf;
    gboolean path_status; // Returned value : TRUE if path is correct

    /* If path is invalid: inform the user, but load the first directories 
     * of the full path while parent directories are valid */
    if ( stat(current_path,&stbuf)==-1)
    {
        GtkWidget *msgbox;
        gchar *msg;

        msg = g_strdup_printf(_("The entered path is invalid!:\n%s\n(%s)"),
            current_path,g_strerror(errno));
        msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        path_status = FALSE;
    }else
    {
        /* The path is correct */
        Browser_Update_Current_Path(current_path);
        path_status = TRUE;
    }

    /* Load current_path */
    if(current_path && *current_path)
    {
        currentdir=g_strdup(current_path);
        tpath=g_strdup("/");
        gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
        pos=(gchar*)strtok(currentdir,"/");

        node=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),NULL,"/",filetreeent_compare_func);
        do
        {
            tpathnew=g_strconcat(tpath,pos,"/",NULL);
            g_free(tpath);
            tpath=tpathnew;
            nextnode=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),node,tpath,filetreeent_compare_func);
            if(!nextnode)
                break;
            node=nextnode;
            if (pos) // if already NUL don't do it again else it returns something not NUL!
                pos=(gchar*)strtok(NULL,"/");
            gtk_ctree_get_node_info(GTK_CTREE(BrowserTree),node,NULL,NULL,NULL,NULL,NULL,NULL,&leaf,NULL);
            //Browser_Tree_Set_Node_Visible(GTK_CTREE(BrowserTree),node);
            if(!leaf && pos)
            {
                gtk_ctree_expand(GTK_CTREE(BrowserTree),node);
            }else
            {
                gtk_ctree_select(GTK_CTREE(BrowserTree),node);
                break;
            }
        }
        while(pos);
        g_free(tpath);
        g_free(currentdir);
    }else
    {
        gtk_ctree_expand(GTK_CTREE(BrowserTree),RootNode);
    }

    return path_status;
}


/*
 * Callback to select-row event
 */
void Browser_List_Row_Selected (GtkCList *clist, gint row, gint column ATTRIBUTE_UNUSED,
                                GdkEventButton *event ATTRIBUTE_UNUSED, gpointer data ATTRIBUTE_UNUSED)
{
    bl_row_selected = row; // To display row of the last selected_row event (not the first row selected in a multiple selection mode)
    
    // On normal state, when we select severals lines, it displays each item and takes a lot of time.
    // So we attach an idle event to display only the first file selected
    if (blrs_idle_handler_id == 0)
        blrs_idle_handler_id = gtk_idle_add((GtkFunction)Browser_List_Row_Selected_Idle,NULL);
}


/*
 * Display the file when it have no other select_row event to send
 */
void Browser_List_Row_Selected_Idle (void)
{
    FileRow *filerow = NULL;
    
    filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),bl_row_selected);
    if (filerow)
        Action_Select_Nth_File_By_Etfile(filerow->ETFile);
    
    gtk_idle_remove(blrs_idle_handler_id);
    blrs_idle_handler_id = 0;
}


/*
 * Callback to unselect-row event
 */
void Browser_List_Row_Unselected (GtkCList *clist, gint row, gint column ATTRIBUTE_UNUSED,
                                GdkEventButton *event ATTRIBUTE_UNUSED, gpointer data ATTRIBUTE_UNUSED)
{
    // Note : if you select severals lines, then if you select only one line of these ones, none "select" 
    // event was emitted. So to simulate this, we emit an event when unselecting the other lines.
    if (blru_idle_handler_id == 0)
        blru_idle_handler_id = gtk_idle_add((GtkFunction)Browser_List_Row_Unselected_Idle,NULL);
}


/*
 * Display the file when it have no other select_row event to send
 */
void Browser_List_Row_Unselected_Idle (void)
{
    GList   *SelectedRow         = NULL;
    
    SelectedRow = ((GtkCList*)BrowserList)->selection;
    if (SelectedRow)
    {
        gtk_signal_emit_by_name(GTK_OBJECT(BrowserList),"select_row",GPOINTER_TO_INT(SelectedRow->data));
    }
    
    gtk_idle_remove(blru_idle_handler_id);
    blru_idle_handler_id = 0;
}


/*
 * Load the list of supported files into the browser clist.
 */
static void destroy_clb(gpointer data)
{
    FileRow *row = data;
    // Data in structure 'row' aren't allocated!
    g_free(row);
}
void Browser_List_Load_Files (void)
{
    gboolean activate_bg_color = 0;
    GList *etfilelist;

    if (!BrowserList) return;
        
    gtk_clist_freeze(GTK_CLIST(BrowserList));
    gtk_clist_clear(GTK_CLIST(BrowserList));
    etfilelist = g_list_first(ETFileList);
    while (etfilelist)
    {
        FileRow *filerow;
        gchar   *row_text[] = {NULL};
        gint     row;
        gchar   *current_filename = ((File_Name *)((ET_File *)etfilelist->data)->FileNameCur->data)->value;
            
        /* Clist displays the current filename (name on HD) */
        row_text[0] = g_strdup(g_basename(current_filename));
        row = gtk_clist_append(GTK_CLIST(BrowserList),row_text);
        g_free(row_text[0]);

        /* Data attached to each row */
        filerow = g_malloc0(sizeof(FileRow));
        filerow->ETFile = (ET_File *)etfilelist->data;
        gtk_clist_set_row_data_full(GTK_CLIST(BrowserList),row,filerow,destroy_clb);

        /* Change background color when changing directory (the first row must not be changed) */
        if (row != 0)
        {
            gchar *dir1;
            gchar *dir2;
            gchar *previous_filename = ((File_Name *)((ET_File *)etfilelist->prev->data)->FileNameCur->data)->value;

            if (strcmp(dir1=g_dirname(previous_filename),
                       dir2=g_dirname(current_filename))!=0 )
            {
                activate_bg_color = !activate_bg_color;
            }
            if (dir1) g_free(dir1);
            if (dir2) g_free(dir2);
        }
        filerow->colour_bg = activate_bg_color;

        // Set color of the row
        Browser_List_Set_Row_Color(row);

        etfilelist = g_list_next(etfilelist);
    }
    gtk_clist_thaw(GTK_CLIST(BrowserList));
}


/*
 * Update state of files in the list after changes (without clearing the clist!)
 *  - Refresh filename is file saved,
 *  - Change color is something change on the file
 */
void Browser_List_Refresh_Whole_List (void)
{
    FileRow *filerow;
    guint row;
    gchar *current_filename;

    if (!ETFileList || !BrowserList || GTK_CLIST(BrowserList)->rows==0)
        return;

    gtk_clist_freeze(GTK_CLIST(BrowserList));
    
    // Browse the full list for changes
    for (row=0;row<(guint)GTK_CLIST(BrowserList)->rows;row++)
    {
        // Refresh filename
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
        current_filename = g_basename( ((File_Name *)filerow->ETFile->FileNameCur->data)->value );
        gtk_clist_set_text(GTK_CLIST(BrowserList),row,0,current_filename);
        
        // Set color of the row
        Browser_List_Set_Row_Color(row);
    }

    gtk_clist_thaw(GTK_CLIST(BrowserList));
}


/*
 * Update state of one file in the list after changes (without clearing the clist!)
 *  - Refresh filename is file saved,
 *  - Change color is something change on the file
 */
void Browser_List_Refresh_File_In_List (ET_File *ETFile)
{
    GList   *SelectedRow = NULL;
    FileRow *filerow = NULL;
    gboolean row_found = FALSE;
    guint    row = 0;
    gchar   *current_filename;
    
    if (!ETFileList || !BrowserList || GTK_CLIST(BrowserList)->rows==0)
        return;

    // Found the row of the modified file (when found: row_found=TRUE)
    // 1/3. Try with the selected file in clist
    SelectedRow = ((GtkCList*)BrowserList)->selection;
    if (SelectedRow && SelectedRow->data!=NULL)
    {
        row = (gulong)SelectedRow->data;
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
        if (filerow && (filerow->ETFile==ETFile))
            row_found = TRUE;
    }

    // 2/3. Fails, often it's just the row before (if we aren't on the first row)
    if (row_found==FALSE && row>0)
    {
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),--row);
        if (filerow && (filerow->ETFile==ETFile))
            row_found = TRUE;
    }

    // 2bis. We should add an other stage faster than 3/3:
    // Get position of ETFile in ETFileList : row=ETFileList_Length - g_list_length(ETFile in list)
    
    // 3/3. Fails, now we browse the full list to find it
    if (row_found==FALSE)
    {
        for (row=0;row<(guint)GTK_CLIST(BrowserList)->rows;row++)
        {
            filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
            if (filerow && (filerow->ETFile==ETFile))
            {
                row_found = TRUE;
                break;
            }
        }
    }

    // Error somewhere...
    if (row_found==FALSE)
        return;


    gtk_clist_freeze(GTK_CLIST(BrowserList));
    current_filename = g_basename( ((File_Name *)filerow->ETFile->FileNameCur->data)->value );
    gtk_clist_set_text(GTK_CLIST(BrowserList),row,0,current_filename);

    // Set color of the row
    Browser_List_Set_Row_Color(row);

    gtk_clist_thaw(GTK_CLIST(BrowserList));
}


/*
 * Set the color of the row
 */
void Browser_List_Set_Row_Color (guint row)
{
    FileRow *filerow = NULL;
    GtkStyle *style;

    filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),row);
    if (filerow==NULL || filerow->ETFile==NULL)
        return;

    // Create a new style for the rows
    style = gtk_style_copy(gtk_widget_get_style(BrowserList));;
    
    // Set color to red if filename or tag changed
    if ( ET_Check_If_File_Is_Saved(filerow->ETFile) == FALSE )
    {
        style->fg[GTK_STATE_NORMAL]   = RED;
        style->bg[GTK_STATE_SELECTED] = RED;
        if (filerow->colour_bg)
            style->base[GTK_STATE_NORMAL] = LIGHT_BLUE;
    }else
    {
        if (filerow->colour_bg)
            style->base[GTK_STATE_NORMAL] = LIGHT_BLUE;
    }
    gtk_clist_set_row_style(GTK_CLIST(BrowserList),row,style);
}


void Browser_List_Remove_File (ET_File *ETFile)
{
    if (ETFile)
    {
        guint row = ETFile->IndexKey - 1;
        gtk_clist_remove(GTK_CLIST(BrowserList),row);
    }
}


/*
 * Select the specified file into clist, whitout sending the signal.
 */
void Browser_List_Select_File (ET_File *ETFile, gboolean select_it)
{
    if (ETFile)
    {
        guint row = ETFile->IndexKey - 1; // Note : first key as value 1
        
        Browser_List_Select_File_By_Row(row,select_it);
    }
}
void Browser_List_Select_File_By_Row (guint row, gboolean select_it)
{
    if (select_it)
    {
        gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
        gtk_clist_select_row(GTK_CLIST(BrowserList),row,0);
        gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
    }
    Browser_List_Set_Row_Visible(GTK_CLIST(BrowserList),row);
}

/*
 * Unselect the specified file into clist, whitout sending the signal.
 */
void Browser_List_Unselect_File (ET_File *ETFile)
{
    if (ETFile)
    {
        guint row = ETFile->IndexKey - 1; // Note : first key as value 1
        
        Browser_List_Unselect_File_By_Row(row);
    }
}
void Browser_List_Unselect_File_By_Row (guint row)
{
    gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Unselected),NULL);
    gtk_clist_unselect_row(GTK_CLIST(BrowserList),row,0);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Unselected),NULL);

    //Browser_List_Set_Row_Visible(GTK_CLIST(BrowserList),row);
}


void Browser_List_Scroll_Vertical (GtkCList *clist, GtkScrollType scroll_type ATTRIBUTE_UNUSED,
                                  gfloat position ATTRIBUTE_UNUSED, gpointer user_data ATTRIBUTE_UNUSED)
{
    if (!clist) return;
    
    /* As this function was called by signal gtk_signal_..._after..., "clist->focus_row" contains
     * the number of the row which receive the signal */
    gtk_clist_select_row(GTK_CLIST(clist),(gint)clist->focus_row,0);
}


void Browser_List_Clear (void)
{
    gtk_clist_clear(GTK_CLIST(BrowserList));
}

/*
 * Return a list ETFile corrsponding to the selected lines in the BrowserList
 */
GList *Browser_List_Get_Selection (void)
{
    GList   *SelectedRow         = NULL;
    GList   *Selected_ETFileList = NULL;
    FileRow *filerow             = NULL;
    
    SelectedRow = ((GtkCList*)BrowserList)->selection;
    while (SelectedRow)
    {
        filerow = gtk_clist_get_row_data(GTK_CLIST(BrowserList),GPOINTER_TO_INT(SelectedRow->data));
        if (filerow && filerow->ETFile)
            Selected_ETFileList = g_list_append(Selected_ETFileList,filerow->ETFile);
        SelectedRow = SelectedRow->next;
    }

    return Selected_ETFileList;
}


void Browser_List_Select_All_Files (void)
{
    
    // Must block the select signal to avoid selecting all files (one by one) in the main files list
    gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
    gtk_clist_select_all(GTK_CLIST(BrowserList));
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
}

void Browser_List_Unselect_All_Files (void)
{
    
    gtk_clist_unselect_all(GTK_CLIST(BrowserList));
}

void Browser_List_Invert_File_Selection (void)
{
    guint row;
    GList *list;
    
    // Must block the select signal to avoid selecting all files (one by one) in the main files list
    gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
    list = g_list_first(GTK_CLIST(BrowserList)->row_list);
    for (row=0;row<GTK_CLIST(BrowserList)->rows;row++)
    {
        if (GTK_CLIST_ROW(list)->state == GTK_STATE_SELECTED)
            gtk_clist_unselect_row(GTK_CLIST(BrowserList),row,0);
        else
            gtk_clist_select_row(GTK_CLIST(BrowserList),row,0);
        list = list->next;
    }
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserList),GTK_SIGNAL_FUNC(Browser_List_Row_Selected),NULL);
}



/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the browser area (Tree + List + Entry)
 */
void Browser_Area_Set_Sensitive (gboolean activate)
{
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserEntry),activate);
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserTree), activate);
    gtk_widget_set_sensitive(GTK_WIDGET(BrowserList), activate);
}


/*
 * Browser_Popup_Menu_Handler : displays the corresponding menu
 * Create_Browser_Tree_Popup_Menu: Create a popup menu for the tree browser
 * Create_Browser_List_Popup_Menu: Create a popup menu for the list of files of browser
 */
gboolean Browser_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}
GtkWidget *Create_Browser_Tree_Popup_Menu (GtkCTree *ctree ATTRIBUTE_UNUSED)
{
    GtkWidget *BrowserPopupMenu;
    GtkWidget *MenuItem;


    BrowserPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(ctree),"button_press_event",
        (GtkSignalFunc)Browser_Popup_Menu_Handler,GTK_OBJECT(BrowserPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Run Audio Player"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Run_Audio_Player_Using_Directory,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Go to Home Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Load_Home_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Go to Default Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Load_Default_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Set Current Path as Default"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Set_Current_Path_As_Default,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Rename Directory ..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Open_Rename_Directory_Window,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Reload Directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Reload_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Browse Directory with ..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Open_Run_Program_Tree_Window,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    CheckMenuItemBrowseSubdirPopupMenu = gtk_check_menu_item_new_with_label(_("Browse Sub-directories"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(CheckMenuItemBrowseSubdirPopupMenu),BROWSE_SUBDIR);
    gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(CheckMenuItemBrowseSubdirPopupMenu),TRUE);
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),CheckMenuItemBrowseSubdirPopupMenu);
    gtk_signal_connect(GTK_OBJECT(CheckMenuItemBrowseSubdirPopupMenu),"toggled",
        (GtkSignalFunc)Check_Menu_Item_Toggled_Browse_Subdir,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Collapse Tree"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Tree_Collapse,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Refresh Tree"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    /* '..._connect_object' to send NULL parameter to Browser_Tree_Rebuild */
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Tree_Rebuild,NULL);

    gtk_widget_show_all(BrowserPopupMenu);
    return BrowserPopupMenu;
}
GtkWidget  *Create_Browser_List_Popup_Menu (GtkCList *clist ATTRIBUTE_UNUSED)
{
    GtkWidget *BrowserPopupMenu,*BrowserPopupSubMenuSorting;
    GtkWidget *MenuItem;

    /* Note: there is an interesting example with GtkItemFactoryEntry and
     * gtk_item_factory_set_translate_func in XMMS sources xmms/playlistwin.c */
    BrowserPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(clist),"button_press_event",
        (GtkSignalFunc)Browser_Popup_Menu_Handler,GTK_OBJECT(BrowserPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Select All Files"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Action_Select_All_Files,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Unselect All Files"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Action_Unselect_All_Files,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Invert Files Selection"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Action_Invert_Files_Selection,NULL);
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Run Audio Player"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Run_Audio_Player_Using_Selection,NULL);
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Search File(s) ..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Open_Search_File_Window,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Delete File(s)"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Action_Delete_Selected_Files,NULL);
    
    MenuItem = gtk_menu_item_new_with_label(_("Reload the directory"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Reload_Directory,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Open File(s) with ..."));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Browser_Open_Run_Program_List_Window,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by filename"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILENAME));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by filename"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILENAME));
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by creation date"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_CREATION_DATE));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by creation date"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_CREATION_DATE));
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by track number"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_TRACK_NUMBER));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by track number"));
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_TRACK_NUMBER));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);


    // Items of sub menu for other way of sorting
    BrowserPopupSubMenuSorting = gtk_menu_new();
    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by title"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_TITLE));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by title"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_TITLE));
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by artist"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_ARTIST));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by artist"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_ARTIST));
    
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by album"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_ALBUM));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by album"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_ALBUM));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by year"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_YEAR));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by year"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_YEAR));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by genre"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_GENRE));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by genre"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_GENRE));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by comment"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_COMMENT));

    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by comment"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_COMMENT));

    // Sub menu for other way of sorting from tag
    MenuItem = gtk_menu_item_new_with_label(_("Other sorting (from tag)"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(MenuItem),BrowserPopupSubMenuSorting);
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);


    // Items of sub menu for other way of sorting
    BrowserPopupSubMenuSorting = gtk_menu_new();
    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by file type"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILE_TYPE));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by file type"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILE_TYPE));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by file size"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILE_SIZE));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by file size"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILE_SIZE));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by duration"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILE_DURATION));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by duration"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILE_DURATION));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by bitrate"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILE_BITRATE));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by bitrate"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILE_BITRATE));

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Sort list ascending by samplerate"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_ASCENDING_FILE_SAMPLERATE));
    
    MenuItem = gtk_menu_item_new_with_label(_("Sort list descending by samplerate"));
    gtk_menu_append(GTK_MENU(BrowserPopupSubMenuSorting),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)ET_Sort_File_List_And_Update_UI,GINT_TO_POINTER(SORTING_BY_DESCENDING_FILE_SAMPLERATE));

    // Sub menu for other way of sorting from tag
    MenuItem = gtk_menu_item_new_with_label(_("Other sorting (from properties)"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(MenuItem),BrowserPopupSubMenuSorting);
    gtk_menu_append(GTK_MENU(BrowserPopupMenu),MenuItem);


    gtk_widget_show_all(BrowserPopupMenu);
    return BrowserPopupMenu;
}


gint Find_Node (gconstpointer a, gconstpointer b)
{
    gchar *data1;
    gchar *data2;

    if ( !a || !b || !((DirNode*)a)->path ) return -1;
    data1 = g_strdup((gchar *)((DirNode*)a)->path);
    data2 = g_strdup((gchar *)b);
    if (strlen(data1)>2 && data1[strlen(data1)-1]=='/') data1[strlen(data1)-1] = 0;
    if (strlen(data2)>2 && data2[strlen(data2)-1]=='/') data2[strlen(data2)-1] = 0;

    if ( strcmp(data1,data2)==0 )
    {
        g_free(data1);
        g_free(data2);
        return 0;
    }else
    {
        g_free(data1);
        g_free(data2);
        return -1;
    }
    
    
    return strcmp(data1,data2);
}
gint Find_Children_Node (gconstpointer a, gconstpointer b)
{
    gchar *data1;
    gchar *data2;

    if ( !a || !b || !((DirNode*)a)->path ) return -1;
    data1 = g_strdup((gchar *)((DirNode*)a)->path);
    data2 = g_strdup((gchar *)b);

    if ( strncmp(data1,data2,strlen(data2))==0 && strlen(data1)>strlen(data2) )
    {
        g_free(data1);
        g_free(data2);
        return 0;
    }else
    {
        g_free(data1);
        g_free(data2);
        return -1;
    }
}

/*
 * Destroy the whole tree up to the root node
 */
void Browser_Tree_Initialize (void)
{
    GtkCTreeNode *node;
    DirNode *dirnode;
    gchar *node_text = "dummy";

    gtk_clist_freeze(GTK_CLIST(BrowserTree));
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(BrowserTree),RootNode);
    if (!dirnode) return;
    // Delete children nodes of the root node
    while ( (node=gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),NULL,dirnode->path,Find_Children_Node)) )
    {
        gtk_ctree_remove_node(GTK_CTREE(BrowserTree),node);
    }
    // Reinit node data
    dirnode->scanned = 0;
    // Insert dummy node to display the hot spot
    node = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),RootNode,NULL,&node_text,4,
        NULL,NULL,NULL,NULL,TRUE,TRUE);
    gtk_clist_thaw(GTK_CLIST(BrowserTree));
}

/*
 * Browser_Tree_Rebuild: Refresh the tree browser by destroying it and rebuilding it.
 * Opens tree nodes corresponding to 'path_to_load' if this parameter isn't NULL.
 * If NULL, selects the current path.
 */
void Browser_Tree_Rebuild (gchar *path_to_load)
{
    if (path_to_load != NULL)
    {
        Browser_Tree_Initialize();
        Browser_Tree_Select_Dir(path_to_load);
    }else
    {
        gchar *current_path = NULL;

        /* Memorize the current path to load it again at the end */
        if (Browser_Tree_Get_Path_Of_Selected_Node() != NULL)
            current_path = g_strdup(Browser_Tree_Get_Path_Of_Selected_Node());
        /* If no node selected, get path from BrowserEntry or default path */
        else if (BrowserCurrentPath != NULL)
            current_path = g_strdup(BrowserCurrentPath);
        else if (strlen(gtk_entry_get_text_1(BrowserEntry)) > 0)
            current_path = g_strdup(gtk_entry_get_text_1(BrowserEntry));
        else
            current_path = g_strdup(DEFAULT_PATH_TO_MP3);
        
        Browser_Tree_Initialize();
        /* Reload the memorized path */
//        gtk_signal_handler_block_by_func(GTK_OBJECT(BrowserTree),GTK_SIGNAL_FUNC(Browser_Tree_Node_Selected),NULL);
        Browser_Tree_Select_Dir(current_path);
//        gtk_signal_handler_unblock_by_func(GTK_OBJECT(BrowserTree),GTK_SIGNAL_FUNC(Browser_Tree_Node_Selected),NULL);
        if (current_path) g_free(current_path);
    }
}

/*
 * Rename directories in the tree
 * The both paths must have the same number of directories and must terminate by '/'
 */
void child_node_func(GtkCTree *ctree, GtkCTreeNode *node, gpointer data)
{
    DirNode      *dirnode, *parent_dirnode;
    GtkCTreeNode *parent_node;
    
    parent_node    = GTK_CTREE_ROW(node)->parent;
    dirnode        = gtk_ctree_node_get_row_data(GTK_CTREE(ctree),node);
    parent_dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(ctree),parent_node);

    if (dirnode && parent_dirnode)
    {
        gchar *tmp = NULL;
        if (dirnode->path)
        {
            if (dirnode->path[strlen(dirnode->path)-1]=='/')
                dirnode->path[strlen(dirnode->path)-1] = '\0';
            tmp = g_strdup(g_basename(dirnode->path));
            g_free(dirnode->path);
        }
        dirnode->path = g_strconcat(parent_dirnode->path,tmp,"/",NULL);
        if (tmp) g_free(tmp);
    }
}
void Browser_Tree_Rename_Directory (gchar *last_path, gchar *new_path)
{
    GtkCTreeNode *node, *child_node;
    DirNode *dirnode;
    gchar *node_text = NULL;
    guint8 spacing;
    GdkPixmap *pixmap;
    GdkBitmap *mask;

    if (!last_path || !new_path)
        return;
    
    gtk_clist_freeze(GTK_CLIST(BrowserTree));

    // Process the "root" node : change node name and path in attached data
    node = gtk_ctree_find_by_row_data_custom(GTK_CTREE(BrowserTree),NULL,last_path,Find_Node);
    if (node)
    {
        // Rename the selected directory
        gtk_ctree_node_get_pixtext(GTK_CTREE(BrowserTree),node,0,NULL,&spacing,&pixmap,&mask);
        node_text = g_basename(new_path);
        gtk_ctree_node_set_pixtext(GTK_CTREE(BrowserTree),node,0,node_text,spacing,pixmap,mask);

        // Update path of the selected directory
        dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(BrowserTree),node);
        if (dirnode)
        {
            if (dirnode->path) g_free(dirnode->path);
            if (strlen(new_path)>2 && new_path[strlen(new_path)-1]=='/')
            {
                dirnode->path = g_strdup(new_path);
            }else
            {
                dirnode->path = g_strconcat(new_path,"/",NULL);
            }
        }

        // Update path of the children node
        child_node = GTK_CTREE_ROW(node)->children;
        while ( child_node != NULL )
        {
            gtk_ctree_pre_recursive(GTK_CTREE(BrowserTree),child_node,child_node_func,NULL);
            child_node = GTK_CTREE_ROW(child_node)->sibling;
        }
        
        // Resort the tree with the new directory name
        gtk_ctree_sort_node(GTK_CTREE(BrowserTree),GTK_CTREE_ROW(node)->parent);

    }

    // Update the variable of the current path
    Browser_Update_Current_Path(Browser_Tree_Get_Path_Of_Selected_Node());

    gtk_clist_thaw(GTK_CLIST(BrowserTree));
}


static gboolean check_for_subdir(gchar *path)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *npath;
    
    if( (dir=opendir(path)) )
    {
        while( (dirent=readdir(dir)) )
        {
            if(dirent->d_name[0]!='.')
            {
                npath = g_strconcat(path,dirent->d_name,"/",NULL);
                stat(npath,&statbuf);
                g_free(npath);
                if(S_ISDIR(statbuf.st_mode))
                {
                    closedir(dir);
                    return TRUE;
                }
            }
        }
        closedir(dir);
    }
    return FALSE;
}

/*
 * Check if you have access permissions for directory path. Returns 1 if ok, else 0.
 */
gboolean Check_For_Access_Permission (gchar *path)
{
    DIR *dir;

    if( (dir=opendir(path)) == NULL )
    {
        if (errno == EACCES)
            return FALSE;
    }else
    {
        closedir(dir);
    }
    return TRUE;
}


/*
 * Destroy data attached to each node of the tree
 */
static void destroy_ctb(gpointer data)
{
    DirNode *node = data;
    if (node || node->path) g_free(node->path);
    if (node)               g_free(node);
}

static void expand_cb(GtkWidget *ctree, GtkCTreeNode *parent_node)
{
    DIR *dir;
    struct dirent *dirent;
    gchar *path;
    gchar *text;
    gchar *dummy = "dummy";
    struct stat statbuf;
    GtkCTreeNode *node, *sub_node;
    DirNode *parent_dirnode, *dirnode;
    gboolean has_subdir = FALSE;
    
    
    parent_dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(ctree), parent_node);
    if(parent_dirnode && !parent_dirnode->scanned)
    {
        gtk_clist_freeze(GTK_CLIST(ctree));
        node = gtk_ctree_find_by_row_data(GTK_CTREE(ctree), parent_node, NULL);
        gtk_ctree_remove_node(GTK_CTREE(ctree),node);
        if( (dir=opendir(parent_dirnode->path)) )
        {
            while( (dirent=readdir(dir)) )
            {
                path = g_strconcat(parent_dirnode->path,dirent->d_name,NULL);
                stat(path,&statbuf);
                if(S_ISDIR(statbuf.st_mode)&&dirent->d_name[0]!='.')
                {
                    dirnode = g_malloc0(sizeof(DirNode));
                    dirnode->path = g_strconcat(path,"/",NULL);
                    text = dirent->d_name;

                    if(check_for_subdir(dirnode->path))
                        has_subdir=TRUE;
                    else
                        has_subdir=FALSE;

                    /* Select pixmap for accessible/unaccessible directory */
                    if (Check_For_Access_Permission(path))
                    {
                        node = gtk_ctree_insert_node(GTK_CTREE(ctree),
                            parent_node,NULL,&text,4, 
                            closed_folder_pixmap,closed_folder_mask,
                            opened_folder_pixmap,opened_folder_mask, 
                            !has_subdir, FALSE);
                    }
                    else
                    {
                        node = gtk_ctree_insert_node(GTK_CTREE(ctree),
                            parent_node,NULL,&text,4, 
                            closed_folder_locked_pixmap,closed_folder_locked_mask,
                            opened_folder_pixmap,opened_folder_mask, 
                            !has_subdir, FALSE);
                    }

                    gtk_ctree_node_set_row_data_full(GTK_CTREE(ctree),node,dirnode,destroy_ctb);

                    if(has_subdir)
                        sub_node=gtk_ctree_insert_node(GTK_CTREE(ctree),
                            node,NULL,&dummy,4,NULL,NULL,NULL,NULL,FALSE,FALSE);
                }
                g_free(path);
            }
            closedir(dir);
            gtk_ctree_sort_node(GTK_CTREE(ctree),parent_node);
        }
        gtk_clist_thaw(GTK_CLIST(ctree));    
        parent_dirnode->scanned=TRUE;
    }
}

static void collapse_cb(GtkWidget *ctree, GtkCTreeNode *parent_node)
{
    DirNode *dirnode;
    GtkCTreeNode *dummy_node, *child_node;
    GtkCTreeRow *row;
    gchar *dummy = "dummy";

    if (KEEP_TREE_BROWSER_IN_MEMORY) return;
    
    dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(ctree),parent_node);
    if (dirnode && dirnode->scanned)
    {
        gtk_signal_handler_block_by_func(GTK_OBJECT(ctree),GTK_SIGNAL_FUNC(collapse_cb),NULL);
        gtk_clist_freeze(GTK_CLIST(ctree));

        row = GTK_CTREE_ROW(parent_node);
        // Remove all childrens
        while ( (child_node=row->children) )
            gtk_ctree_remove_node(GTK_CTREE(ctree),child_node);
        // Insert a dummy node to show the hot spot '+' (Note: by definition, this node has sub nodes...)
        dummy_node=gtk_ctree_insert_node(GTK_CTREE(ctree),parent_node,NULL,&dummy,4,NULL,NULL,NULL,NULL,FALSE,FALSE);
        // Reinitialize attached data to rescan the dir when expanding the node
        dirnode->scanned = 0;

        gtk_clist_thaw(GTK_CLIST(ctree));    
        gtk_signal_handler_unblock_by_func(GTK_OBJECT(ctree),GTK_SIGNAL_FUNC(collapse_cb),NULL);
    }
}

static int filetreeent_compare_func(const void *a, const void *b)
{
    if(!a || !b || !((DirNode*)a)->path)
        return -1;
    return strcmp(((DirNode*)a)->path,(gchar*)b);
}

/*
 * We replace the default function to sort the tree : sort without using character case...
 */
static gint sort_tree_compare_func (GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2)
{
    GtkCListRow *row1 = (GtkCListRow *)ptr1;
    GtkCListRow *row2 = (GtkCListRow *)ptr2;
    
    if (row1->data && row2->data)
        return strcasecmp(((DirNode *)row1->data)->path,((DirNode *)row2->data)->path);
    else
        return 0;
}


/*
 * Create item of the browser (Entry + Tree + List).
 */
GtkWidget *Create_Browser_Items (GtkWidget *parent)
{
    GtkWidget *VerticalBox;
    GtkWidget *HBox;
    GtkWidget *ScrollWindowCTree;
    GtkWidget *ScrollWindowCList;
    gchar *RootText = "/";
    gchar *node_text = "dummy";
    DirNode *dirnode;
    GtkCTreeNode *node;
    GtkTooltips *Tips;
    GtkWidget *PopupMenu;
    GList *History_List;


    Tips = gtk_tooltips_new_1();

    VerticalBox = gtk_vbox_new(FALSE,2);
    gtk_container_set_border_width(GTK_CONTAINER(VerticalBox),2);


    // HBox for BrowserEntry + BrowserLabel
    HBox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(VerticalBox),HBox,FALSE,TRUE,0);
    
    
    /*
     * The button tot go to the parent directory
     */
    BrowserButton = Create_Button_With_Icon_And_Label(parent_folder_xpm,NULL);
    gtk_box_pack_start(GTK_BOX(HBox),BrowserButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(BrowserButton),GTK_RELIEF_NONE);
    gtk_signal_connect(GTK_OBJECT(BrowserButton),"clicked",(GtkSignalFunc)Browser_Button_Clicked,NULL);
    gtk_tooltips_set_tip(Tips,BrowserButton,_("Go to parent directory"),NULL);

    /*
     * The entry box for displaying path
     */
    BrowserEntry = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(BrowserEntry));
    gtk_combo_set_case_sensitive(GTK_COMBO(BrowserEntry),TRUE);
    /* History list */
    History_List = Load_Path_Entry_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(BrowserEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(BrowserEntry),"History",History_List);

    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(BrowserEntry)->entry),"activate",GTK_SIGNAL_FUNC(Browser_Entry_Activated),NULL);
    gtk_box_pack_start(GTK_BOX(HBox),BrowserEntry,TRUE,TRUE,1);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(BrowserEntry)->entry,_("Enter a directory to browse."),NULL);


    /*
     * The label for displaying number of files in path (without subdirs)
     */
    BrowserLabel = gtk_label_new(" ... ");
    gtk_box_pack_start(GTK_BOX(HBox),BrowserLabel,FALSE,FALSE,0);

    
    /*
     * The ScrollWindow and the CTree
     */
    ScrollWindowCTree = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowCTree),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    //gtk_box_pack_start(GTK_BOX(VerticalBox),ScrollWindowCTree,TRUE,TRUE,0);
    //gtk_widget_set_usize (GTK_WIDGET(ScrollWindowCTree),100,-1);

    /* The tree */
    BrowserTree = gtk_ctree_new(1,0);
    gtk_container_add(GTK_CONTAINER(ScrollWindowCTree),BrowserTree);
    gtk_ctree_set_indent(GTK_CTREE(BrowserTree),13);
    gtk_ctree_set_line_style(GTK_CTREE(BrowserTree),    BROWSER_LINE_STYLE);
    gtk_ctree_set_expander_style(GTK_CTREE(BrowserTree),BROWSER_EXPANDER_STYLE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(BrowserTree),0,TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(BrowserTree),GTK_SELECTION_SINGLE);
    //gtk_clist_set_selection_mode(GTK_CLIST(BrowserTree),GTK_SELECTION_EXTENDED);
    gtk_clist_set_compare_func(GTK_CLIST(BrowserTree),sort_tree_compare_func);

    /* Create pixmaps */
    if(!opened_folder_pixmap)
    {
        opened_folder_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &opened_folder_mask,NULL,opened_folder_xpm);
        closed_folder_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &closed_folder_mask,NULL,closed_folder_xpm);
        closed_folder_locked_pixmap = gdk_pixmap_create_from_xpm_d(parent->window,
                    &closed_folder_locked_mask,NULL,closed_folder_locked_xpm);
    }
    
    /* Signals */
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_expand",    (GtkSignalFunc)expand_cb,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_collapse",  (GtkSignalFunc)collapse_cb,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"tree_select_row",(GtkSignalFunc)Browser_Tree_Node_Selected,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserTree),"key_press_event",(GtkSignalFunc)Browser_Tree_Key_Press,NULL);
    gtk_signal_connect_after(GTK_OBJECT(BrowserTree),"scroll_vertical",(GtkSignalFunc)Browser_List_Scroll_Vertical,NULL);

    /* Create the root node */
    RootNode = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),NULL,NULL,&RootText,4, 
                                     closed_folder_pixmap,closed_folder_mask,
                                     opened_folder_pixmap,opened_folder_mask, 
                                     FALSE, FALSE);

    dirnode = g_malloc0(sizeof(DirNode));
    dirnode->path = g_strdup("/");
    gtk_ctree_node_set_row_data_full(GTK_CTREE(BrowserTree),RootNode,dirnode,destroy_ctb);

    /* Insert dummy node to display: '+' */
    node = gtk_ctree_insert_node(GTK_CTREE(BrowserTree),RootNode,NULL,&node_text,4, 
                    NULL,NULL,NULL,NULL,TRUE,TRUE);


    /* 
     * Create Popup Menu on browser ctree
     */
    PopupMenu = Create_Browser_Tree_Popup_Menu(GTK_CTREE(BrowserTree));


    /*
     * The ScrollWindow and the CList
     */
    ScrollWindowCList = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowCList),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);

    /* The clist */
    BrowserList = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(ScrollWindowCList),BrowserList);
    gtk_clist_set_column_auto_resize(GTK_CLIST(BrowserList),0,TRUE);
    gtk_clist_set_reorderable(GTK_CLIST(BrowserList),FALSE);
    //gtk_clist_set_selection_mode(GTK_CLIST(BrowserList),GTK_SELECTION_SINGLE);
    gtk_clist_set_selection_mode(GTK_CLIST(BrowserList),GTK_SELECTION_EXTENDED);

    gtk_signal_connect(GTK_OBJECT(BrowserList),"key_press_event",(GtkSignalFunc)Browser_List_Key_Press,NULL);
    gtk_signal_connect_after(GTK_OBJECT(BrowserList),"button_press_event",(GtkSignalFunc)Browser_List_Button_Press,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserList),"select_row",(GtkSignalFunc)Browser_List_Row_Selected,NULL);
    gtk_signal_connect(GTK_OBJECT(BrowserList),"unselect_row",(GtkSignalFunc)Browser_List_Row_Unselected,NULL);
    /* Note: "_after" for this signal to obtain the new focus_row of clist and not the previous one */
    gtk_signal_connect_after(GTK_OBJECT(BrowserList),"scroll_vertical",(GtkSignalFunc)Browser_List_Scroll_Vertical,NULL);

    /* 
     * Create Popup Menu on browser ctree
     */
    PopupMenu = Create_Browser_List_Popup_Menu(GTK_CLIST(BrowserList));


    /*
     * The pane for the CTree and the CList
     */
    BrowserHPaned = gtk_hpaned_new();
    gtk_box_pack_start(GTK_BOX(VerticalBox),BrowserHPaned,TRUE,TRUE,0);
    gtk_paned_set_handle_size(GTK_PANED(BrowserHPaned),8);
    gtk_paned_set_gutter_size(GTK_PANED(BrowserHPaned),6);                       
    gtk_paned_pack1(GTK_PANED(BrowserHPaned),ScrollWindowCTree,TRUE,TRUE); // Left side
    gtk_paned_pack2(GTK_PANED(BrowserHPaned),ScrollWindowCList,TRUE,TRUE); // Right side
    if (SET_PANE_HANDLE_POSITION2)
        gtk_paned_set_position(GTK_PANED(BrowserHPaned),PANE_HANDLE_POSITION2);

    gtk_widget_show_all(VerticalBox);
    
    /* Set home variable as current path */
    Browser_Update_Current_Path(HOME_VARIABLE);
    
    return VerticalBox;
}



/*
 * The window to Rename a directory into the browser.
 */
void Browser_Open_Rename_Directory_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *Label;
    GtkWidget *RenameDirectoryCombo;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GList *combo_list = NULL;
    gchar *directory_parent = NULL;
    gchar *directory_name = NULL;
    gchar *address = NULL;
    gchar *string;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (RenameDirectoryWindow != NULL) 
    {
        gdk_window_raise(RenameDirectoryWindow->window);
        return;
    }

    /* We get the full path but we musn't display the parent directories */
    directory_parent = g_strdup(BrowserCurrentPath);
    if (!directory_parent || strlen(directory_parent)==0)
        return;
    if (strlen(directory_parent)>1 && directory_parent[strlen(directory_parent)-1]=='/')
        directory_parent[strlen(directory_parent)-1]=0;
    address = strrchr(directory_parent,'/');
    if (!address) return;
    directory_name = g_strdup(address+1);
    *(address+1) = 0;
    if (!directory_name || strlen(directory_name)==0)
        return;
    
    
    RenameDirectoryWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(RenameDirectoryWindow),_("Rename the directory"));
    gtk_window_set_transient_for(GTK_WINDOW(RenameDirectoryWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(RenameDirectoryWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"destroy",(GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"delete_event",(GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RenameDirectoryWindow),"key_press_event",(GtkSignalFunc)Rename_Directory_Window_Key_Press,NULL);

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(RenameDirectoryWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    string = g_strdup_printf(_("Rename the directory '%s' to : "),directory_name);
    Label = gtk_label_new(_(string));
    g_free(string);
    gtk_box_pack_start(GTK_BOX(VBox),Label,TRUE,TRUE,0);
    gtk_label_set_line_wrap(GTK_LABEL(Label),TRUE);

    /* The combobox to rename the directory */
    RenameDirectoryCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(VBox),RenameDirectoryCombo,TRUE,TRUE,0);
    gtk_combo_set_value_in_list(GTK_COMBO(RenameDirectoryCombo),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(RenameDirectoryCombo),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(RenameDirectoryCombo),300,-1);
    /* Set the directory into the combobox */
    combo_list = g_list_append(combo_list,"");
    combo_list = g_list_append(combo_list,directory_name);
    gtk_combo_set_popdown_strings(GTK_COMBO(RenameDirectoryCombo),combo_list);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(RenameDirectoryCombo)->entry),directory_name);
    /* We attach usefull data to the combobox */
    gtk_object_set_data(GTK_OBJECT(RenameDirectoryCombo),"Parent_Directory",directory_parent);
    gtk_object_set_data(GTK_OBJECT(RenameDirectoryCombo),"Current_Directory",directory_name);
    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(GTK_COMBO(RenameDirectoryCombo)->entry));

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to save: rename directory */
    Button = Create_Button_With_Pixmap(BUTTON_APPLY);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Rename_Directory,GTK_OBJECT(RenameDirectoryCombo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(RenameDirectoryCombo)->entry),"changed",(GtkSignalFunc)Entry_Changed_Disable_Object,GTK_OBJECT(Button));
    
    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    //gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Rename_Directory_Window,NULL);

    gtk_widget_show(RenameDirectoryWindow);

    /* Note : the window must be shown before moving it.
     * The window is centered over the Browser Area */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(RenameDirectoryWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (RenameDirectoryWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (RenameDirectoryWindow->allocation.height)/2);

    // To avoid/minimize 'flicker'
    gtk_widget_show_all(RenameDirectoryWindow);
}
void Destroy_Rename_Directory_Window (void)
{
    if (RenameDirectoryWindow)
    {
        gtk_widget_destroy(RenameDirectoryWindow);
        RenameDirectoryWindow = (GtkWidget *)NULL;
    }
}
void Rename_Directory (GtkObject *combobox)
{
    DIR   *dir;
    gchar *directory_parent;
    gchar *directory_last_name;
    gchar *directory_new_name;
    gchar *last_path;
    gchar *tmp_path;
    gchar *new_path;
    gint   fd_tmp;


    if (!GTK_IS_COMBO(combobox)) return;
    
    directory_parent    = gtk_object_get_data(GTK_OBJECT(combobox),"Parent_Directory");
    directory_last_name = gtk_object_get_data(GTK_OBJECT(combobox),"Current_Directory");
    directory_new_name  = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1);

    /* Check if a name for the directory have been supplied */
    if (!directory_new_name || strlen(directory_new_name)<1)
    {
        GtkWidget *msgbox;

        msgbox = msg_box_new (_("Error..."),_("You must type a directory name!"),MSG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return;
    }

    /* If the directory name haven't been changed, we do nothing! */
    if (directory_last_name && directory_new_name && strcmp(directory_last_name,directory_new_name)==0)
    {
        Destroy_Rename_Directory_Window();
        g_free(directory_new_name);
        return;
    }

    /* Build the current and new absolute paths */
    last_path = g_strconcat(directory_parent,directory_last_name,NULL);
    new_path  = g_strconcat(directory_parent,directory_new_name,NULL);

    /* Check if the new directory name doesn't already exists, and detect if it's
     * only a case change (needs for vfat) */
    if ( (dir=opendir(new_path))!=NULL )
    {
        gchar *msg;
        GtkWidget *msgbox;

        closedir(dir);
        if ( strcasecmp(last_path,new_path)!=0 )
        {
            msg = g_strdup_printf(_("Can't rename because this directory name "
                                    "already exists!\n(%s)"),new_path); 
            msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
            g_free(msg);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);

            return;
        }
    }

    /* Temporary path (useful when changing only string case) */
    tmp_path = g_strdup_printf("%s.XXXXXX",last_path);
    if ( (fd_tmp = mkstemp(tmp_path)) >= 0 )
    {
        close(fd_tmp);
        unlink(tmp_path);
    }

    /* Rename the directory from 'last name' to 'tmp name' */
    if ( rename(last_path,tmp_path)!=0 )
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't rename directory \n'%s'\n to \n'%s'!\n(%s)"),
                    last_path,tmp_path,g_strerror(errno)); 
        msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        return;
    }
    /* Rename the directory from 'tmp name' to 'new name' (final name) */
    if ( rename(tmp_path,new_path)==0 )
    {
        ET_Update_Directory_Name_Into_File_List(last_path,new_path);
        Browser_Tree_Rename_Directory(last_path,new_path);
        
        // To update file path in the browser entry
        if (ETFileList)
            ET_Display_File_Data_To_UI(ETFileDisplayed);
        else
            Browser_Entry_Set_Text(Browser_Get_Current_Path());
        
    }else
    {
        gchar *msg;
        GtkWidget *msgbox;

        msg = g_strdup_printf(_("Can't rename directory \n'%s'\n to \n'%s'!\n(%s)"),
                    tmp_path,new_path,g_strerror(errno)); 
        msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        return;
    }

    Destroy_Rename_Directory_Window();
    g_free(last_path);
    g_free(new_path);
    g_free(tmp_path);
    g_free(directory_new_name);

    Statusbar_Message(_("Directory renamed"),TRUE);
}
void Rename_Directory_Window_Key_Press (GtkWidget *window ATTRIBUTE_UNUSED, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Rename_Directory_Window();
                break;
        }
    }
}





/*
 * Window where is typed the name of the program to run, which
 * receives the current directory as parameter.
 */
void Browser_Open_Run_Program_Tree_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *Label;
    GtkWidget *RunProgramCombo;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkTooltips *Tips;
    GList *History_List;
    gchar *current_directory = NULL;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (RunProgramTreeWindow != NULL) 
    {
        gdk_window_raise(RunProgramTreeWindow->window);
        return;
    }

    // Current directory
    current_directory = g_strdup(BrowserCurrentPath);
    if (!current_directory || strlen(current_directory)==0)
        return;
    
    RunProgramTreeWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(RunProgramTreeWindow),_("Browse Directory with ..."));
    gtk_window_set_transient_for(GTK_WINDOW(RunProgramTreeWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(RunProgramTreeWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(RunProgramTreeWindow),"destroy",(GtkSignalFunc)Destroy_Run_Program_Tree_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramTreeWindow),"delete_event",(GtkSignalFunc)Destroy_Run_Program_Tree_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramTreeWindow),"key_press_event",(GtkSignalFunc)Run_Program_Tree_Window_Key_Press,NULL);

    Tips = gtk_tooltips_new_1();

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(RunProgramTreeWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    Label = gtk_label_new(_("Program to run :"));
    gtk_box_pack_start(GTK_BOX(VBox),Label,TRUE,TRUE,0);
    gtk_label_set_line_wrap(GTK_LABEL(Label),TRUE);

    /* The combobox to enter the program to run */
    RunProgramCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(VBox),RunProgramCombo,TRUE,TRUE,0);
    gtk_combo_set_value_in_list(GTK_COMBO(RunProgramCombo),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(RunProgramCombo),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(RunProgramCombo),150,-1);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(RunProgramCombo)->entry,_("Enter the program to run. "
        "It will receive the current directory as parameter."),NULL);

    /* History list */
    History_List = Load_Run_Program_With_Directory_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(RunProgramCombo),History_List);
    gtk_object_set_data(GTK_OBJECT(RunProgramCombo),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(RunProgramCombo)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(RunProgramCombo));

    /* We attach usefull data to the combobox (into Run_Program_With_Directory) */
    gtk_object_set_data(GTK_OBJECT(RunProgramCombo),"Current_Directory",current_directory);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to execute */
    Button = Create_Button_With_Pixmap(BUTTON_EXECUTE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Run_Program_With_Directory,GTK_OBJECT(RunProgramCombo));
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(RunProgramCombo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(RunProgramCombo)->entry),"changed",(GtkSignalFunc)Entry_Changed_Disable_Object,GTK_OBJECT(Button));
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(RunProgramCombo)->entry),"changed",NULL);

    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Run_Program_Tree_Window,NULL);

    gtk_widget_show_all(RunProgramTreeWindow);

    /* Note : the window must be shown before moving it.
     * The window is centered over the Browser Area */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(RunProgramTreeWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (RunProgramTreeWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (RunProgramTreeWindow->allocation.height)/2);
    
}
void Destroy_Run_Program_Tree_Window (void)
{
    if (RunProgramTreeWindow)
    {
        gtk_widget_destroy(RunProgramTreeWindow);
        RunProgramTreeWindow = (GtkWidget *)NULL;
    }
}
void Run_Program_Tree_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Run_Program_Tree_Window();
                break;
        }
    }
}
void Run_Program_With_Directory (GtkObject *combobox)
{
    gchar *program_name;
    gchar *current_directory;
    GList *args_list = NULL;


    if (!GTK_IS_COMBO(combobox)) return;
    
    program_name      = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1);
    current_directory = gtk_object_get_data(GTK_OBJECT(combobox),"Current_Directory");

    // List of parameters (here only one! : the current directory)
    args_list = g_list_append(args_list,current_directory);

    Run_Program(program_name,args_list);
    g_list_free(args_list);

    // Save list attached to the combobox
    Save_Run_Program_With_Directory_List(gtk_object_get_data(GTK_OBJECT(combobox),"History"));

    Destroy_Run_Program_Tree_Window();
}






/*
 * Window where is typed the name of the program to run, which
 * receives the current file as parameter.
 */
void Browser_Open_Run_Program_List_Window (void)
{
    GtkWidget *Frame;
    GtkWidget *VBox;
    GtkWidget *Label;
    GtkWidget *RunProgramCombo;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkTooltips *Tips;
    GList *History_List;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (RunProgramListWindow != NULL) 
    {
        gdk_window_raise(RunProgramListWindow->window);
        return;
    }

    RunProgramListWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(RunProgramListWindow),_("Open File with ..."));
    gtk_window_set_transient_for(GTK_WINDOW(RunProgramListWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(RunProgramListWindow),FALSE,TRUE,TRUE);
    gtk_signal_connect(GTK_OBJECT(RunProgramListWindow),"destroy",(GtkSignalFunc)Destroy_Run_Program_List_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramListWindow),"delete_event",(GtkSignalFunc)Destroy_Run_Program_List_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(RunProgramListWindow),"key_press_event",(GtkSignalFunc)Run_Program_List_Window_Key_Press,NULL);

    Tips = gtk_tooltips_new_1();

    Frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(RunProgramListWindow),Frame);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    Label = gtk_label_new(_("Program to run :"));
    gtk_box_pack_start(GTK_BOX(VBox),Label,TRUE,TRUE,0);
    gtk_label_set_line_wrap(GTK_LABEL(Label),TRUE);

    /* The combobox to enter the program to run */
    RunProgramCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(VBox),RunProgramCombo,TRUE,TRUE,0);
    gtk_combo_set_value_in_list(GTK_COMBO(RunProgramCombo),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(RunProgramCombo),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(RunProgramCombo),150,-1);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(RunProgramCombo)->entry,_("Enter the program to run. "
        "It will receive the current file as parameter."),NULL);

    /* History list */
    History_List = Load_Run_Program_With_File_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(RunProgramCombo),History_List);
    gtk_object_set_data(GTK_OBJECT(RunProgramCombo),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(RunProgramCombo)->entry)),"activate",
        Add_To_Combo_Box_History,GTK_OBJECT(RunProgramCombo));

    /* We attach usefull data to the combobox (into Run_Program_With_Directory) */
    //gtk_object_set_data(GTK_OBJECT(Combo),"Current_File",current_file);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(VBox),ButtonBox,FALSE,FALSE,0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(ButtonBox),GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(ButtonBox),10);

    /* Button to execute */
    Button = Create_Button_With_Pixmap(BUTTON_EXECUTE);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Run_Program_With_Selected_Files,GTK_OBJECT(RunProgramCombo));
    gtk_signal_connect_object(GTK_OBJECT(Button),"clicked",Add_To_Combo_Box_History,GTK_OBJECT(RunProgramCombo));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(RunProgramCombo)->entry),"changed",(GtkSignalFunc)Entry_Changed_Disable_Object,GTK_OBJECT(Button));
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(RunProgramCombo)->entry),"changed",NULL);

    /* Button to cancel */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(Button);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Destroy_Run_Program_List_Window,NULL);

    gtk_widget_show_all(RunProgramListWindow);

    /* Note : the window must be shown before moving it.
     * The window is centered over the Browser Area */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(RunProgramListWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (RunProgramListWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (RunProgramListWindow->allocation.height)/2);
    
}
void Destroy_Run_Program_List_Window (void)
{
    if (RunProgramListWindow)
    {
        gtk_widget_destroy(RunProgramListWindow);
        RunProgramListWindow = (GtkWidget *)NULL;
    }
}
void Run_Program_List_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Destroy_Run_Program_List_Window();
                break;
        }
    }
}
void Run_Program_With_Selected_Files (GtkObject *combobox)
{
    gchar   *program_name;
    GList   *etfilelist;
    GList   *args_list = NULL;

    
    if (!ETFileList)
        return;

    // Programe name to run
    program_name = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(combobox)->entry),0,-1);

    // List of files to pass as parameters
    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {
        ET_File *etfile = (ET_File *)etfilelist->data;
        gchar *filename = ((File_Name *)etfile->FileNameCur->data)->value;
        args_list = g_list_append(args_list,filename);
        etfilelist = etfilelist->next;
    }
   
    Run_Program(program_name,args_list);
    g_list_free(args_list);

    // Save list attached to the combobox
    Save_Run_Program_With_File_List(gtk_object_get_data(GTK_OBJECT(combobox),"History"));

    Destroy_Run_Program_List_Window();
}


/*
 * Run a program with a list of parameters
 */
gboolean Run_Program (gchar *program_name, GList *args_list)
{
    gchar *msg;
    pid_t pid;


    /* Check if a name for the program have been supplied */
    if (!program_name || strlen(program_name)<1)
    {
        GtkWidget *msgbox;

        msgbox = msg_box_new (_("Error..."),_("You must type a program name!"),MSG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return FALSE;
    }

    pid = fork();
    switch(pid)
    {
        case -1:
            g_warning(_("Can't fork another process!\n"));
            break;
        case 0:
        {
            gchar **argv;
            gint    argv_index = 0;
            gchar **argv_user;
            gint    argv_user_number;

            argv_user = g_strsplit(program_name," ",0); // the string may contains arguments, space is the delimiter
            // Number of arguments into 'argv_user'
            for (argv_user_number=0;argv_user[argv_user_number];argv_user_number++);

            argv = g_new0(gchar *,argv_user_number + g_list_length(args_list) + 1); // 1 for NULL

            // Load 'user' arguments (program name and more...)
            while (argv_user[argv_index])
            {
                argv[argv_index] = argv_user[argv_index];
                argv_index++;
            }
            // Load arguments from 'args_list'
            while (args_list)
            {
                argv[argv_index] = (gchar *)args_list->data;
                argv_index++;
                args_list = args_list->next;
            }
            argv[argv_index] = NULL;

            // Execution ...
            execvp(argv[0],argv);

            msg = g_strdup_printf(_("Executed command : '%s %s'"),program_name,"...");
            Statusbar_Message(msg,TRUE);
            g_free(msg);
            break;
        }
        default:
            break;
    }
    return TRUE;
}
