/* et_core.c - 2001/10/21 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000-2002  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <ctype.h>
#include "errno.h"

#include "easytag.h"
#include "et_core.h"
#include "mpeg_header.h"
#include "flac_header.h"
#include "id3_tag.h"
#ifdef ENABLE_OGG
#    include "ogg_tag.h"
#endif
#include "bar.h"
#include "browser.h"
#include "misc.h"
#include "setting.h"
#include "i18n.h"



/***************
 * Declaration *
 ***************/



/**************
 * Prototypes *
 **************/

//gboolean ET_File_Is_Supported (gchar *filename);
gchar               *ET_Get_File_Extension                  (gchar *filename);
ET_File_Description *ET_Get_File_Description                (gchar *filename);
ET_File_Description *ET_Get_File_Description_From_Extension (gchar *extension);

gboolean ET_Free_File_List      (void);
gboolean ET_Free_File_List_Item (ET_File *ETFile);
gboolean ET_Free_File_Name_List (GList *FileNameList);
gboolean ET_Free_File_Tag_List  (GList *FileTagList);
gboolean ET_Free_File_Name_Item (File_Name *FileName);
gboolean ET_Free_File_Tag_Item  (File_Tag  *FileTag);
gboolean ET_Free_File_Tag_Item_Other_Field (File_Tag *FileTag);

//File_Name *ET_File_Name_Item_New (void);
//File_Tag  *ET_File_Tag_Item_New  (void);
ET_File   *ET_File_Item_New      (void);

void     ET_Initialize_File_Item      (ET_File *ETFile);
void     ET_Initialize_File_Tag_Item  (File_Tag *FileTag);
void     ET_Initialize_File_Name_Item (File_Name *FileName);

//gboolean ET_Copy_File_Tag_Item       (ET_File *ETFile, File_Tag *FileTag);
gboolean ET_Copy_File_Tag_Item_Other_Field (ET_File *ETFile, File_Tag *FileTag);
//gboolean ET_Set_Field_File_Name_Item (gint *FileNameField, gchar *value);
//gboolean ET_Set_Field_File_Tag_Item  (gint *FileTagField, gchar *value);

gulong   ET_File_Key_New (void);
gulong   ET_Undo_Key_New (void);

void     ET_Display_File_And_List_Status_To_UI (gchar *filename);
void     ET_Display_Filename_To_UI             (gchar *filename);
gboolean ET_Display_File_Tag_To_UI             (ET_File *ETFile);
gboolean ET_Display_File_Info_To_UI            (ET_File_Info *ETFileInfo);

gboolean ET_Save_File_Name_From_UI  (ET_File *ETFile, File_Name *FileName);
gboolean ET_Save_File_Name_Internal (ET_File *ETFile, File_Name *FileName);
gboolean ET_Save_File_Tag_From_UI   (File_Tag *FileTag);
gboolean ET_Save_File_Tag_Internal  (ET_File *ETFile, File_Tag *FileTag);

void     ET_Mark_File_Tag_As_Saved  (ET_File *ETFile);
void     ET_Mark_File_Name_As_Saved (ET_File *ETFile);

gboolean ET_Manage_Changes_Of_File_Data (ET_File *ETFile, File_Name *FileName, File_Tag *FileTag);
gboolean ET_Detect_Changes_Of_File_Name (File_Name *FileName1, File_Name *FileName2);
gboolean ET_Detect_Changes_Of_File_Tag  (File_Tag  *FileTag1,  File_Tag  *FileTag2);
gboolean ET_Add_File_Name_To_List       (ET_File *ETFile, File_Name *FileName);
gboolean ET_Add_File_Tag_To_List        (ET_File *ETFile, File_Tag  *FileTag);
gboolean ET_Add_File_To_History_List    (ET_File *ETFile);

gboolean ET_Read_File_Info (gchar *filename, ET_File_Info *ETFileInfo);
//gboolean ET_File_Name_Convert_Character (gchar *filename);


/*******************
 * Basic functions *
 *******************/

/*
 * Returns the extension of the file
 */
gchar *ET_Get_File_Extension (gchar *filename)
{
    if (filename)
        return (gchar *)strrchr(filename,'.');
    else
        return NULL;
}
    

/*
 * Determine description of file using his extension.
 * If extension is NULL or not found into the tab, it returns the last entry for UNKNOWN_FILE.
 */
ET_File_Description *ET_Get_File_Description_From_Extension (gchar *extension)
{
    gint i;
    
    if (!extension)
        return &ETFileDescription[ET_FILE_DESCRIPTION_SIZE];
    
    for (i=0; i<ET_FILE_DESCRIPTION_SIZE; i++)
        if ( strcasecmp(extension,ETFileDescription[i].Extension)==0 )
            return &ETFileDescription[i];

    /* If not found in the list */
    return &ETFileDescription[ET_FILE_DESCRIPTION_SIZE];
}


/*
 * Determines description of file.
 * Determines first the extension. If extension is NULL or not found into the tab,
 * it returns the last entry for UNKNOWN_FILE.
 */
ET_File_Description *ET_Get_File_Description (gchar *filename)
{
    return ET_Get_File_Description_From_Extension(ET_Get_File_Extension(filename));
}


/*
 * Returns TRUE if the file is supported, else returns FALSE
 */
gboolean ET_File_Is_Supported (gchar *filename)
{
    if (ET_Get_File_Description(filename)->FileType != UNKNOWN_FILE)
        return TRUE;
    else
        return FALSE;
}




/**************************
 * Initializing functions *
 **************************/

/*
 * Create a new File_Name structure
 */
File_Name *ET_File_Name_Item_New (void)
{
    File_Name *FileName;

    FileName = g_malloc0(sizeof(File_Name));
    ET_Initialize_File_Name_Item(FileName);
    
    return FileName;
}


/*
 * Create a new File_Tag structure
 */
File_Tag *ET_File_Tag_Item_New (void)
{
    File_Tag *FileTag;

    FileTag = g_malloc0(sizeof(File_Tag));
    ET_Initialize_File_Tag_Item(FileTag);
    
    return FileTag;
}


/*
 * Create a new ET_File structure
 */
ET_File *ET_File_Item_New (void)
{
    ET_File *ETFile;

    ETFile = g_malloc0(sizeof(ET_File));
    ET_Initialize_File_Item(ETFile);
    
    return ETFile;
}


void ET_Initialize_File_Name_Item (File_Name *FileName)
{
    if (FileName)
    {
        FileName->key    = ET_Undo_Key_New();
        FileName->saved  = FALSE;    
        FileName->value  = NULL;
    }
}


void ET_Initialize_File_Tag_Item (File_Tag *FileTag)
{
    if (FileTag)
    {
        FileTag->key         = ET_Undo_Key_New();
        FileTag->saved       = FALSE;    
        FileTag->title       = NULL;
        FileTag->artist      = NULL;
        FileTag->album       = NULL;
        FileTag->track       = NULL;
        FileTag->track_total = NULL;
        FileTag->year        = NULL;
        FileTag->genre       = NULL;
        FileTag->comment     = NULL;
        FileTag->other       = NULL;
    }
}


void ET_Initialize_File_Item (ET_File *ETFile)
{
    if (ETFile)
    {
        ETFile->ETFileKey         = 0;
        ETFile->ETFileDescription = NULL;    
        ETFile->ETFileInfo        = NULL;
        ETFile->FileNameCur       = NULL;    
        ETFile->FileNameNew       = NULL;
        ETFile->FileNameList      = NULL;
        ETFile->FileNameListBak   = NULL;
        ETFile->FileTag           = NULL;
        ETFile->FileTagList       = NULL;
        ETFile->FileTagListBak    = NULL;
    }
}


/* Key for each item of ETFileList */
gulong ET_File_Key_New (void)
{
    static gulong ETFileKey = 0;
    return ++ETFileKey;
}

/* Key for Undo */
gulong ET_Undo_Key_New (void)
{
    static gulong ETUndoKey = 0;
    return ++ETUndoKey;
}




/**********************************
 * File adding/removing functions *
 **********************************/

/*
 * ET_Add_File_To_File_List: Add a file to the "main" list. And get all informations of the file.
 */
GList *ET_Add_File_To_File_List (gchar *filename)
{
    ET_File_Description *ETFileDescription;
    ET_File      *ETFile;
    File_Name    *FileName;
    File_Tag     *FileTag;
    ET_File_Info *ETFileInfo;
    gulong        ETFileKey;
    gulong        undo_key;

    
    if (!filename)
        return ETFileList;

    /* Primary Key for this file */
    ETFileKey = ET_File_Key_New ();

    /* Get description of the file */
    ETFileDescription = ET_Get_File_Description(filename);

    /* Fill the File_Name structure for FileNameList */
    FileName = ET_File_Name_Item_New();
    FileName->saved = TRUE;    /* The file haven't been changed, so it's saved */
    FileName->value = g_strdup(filename);

    /* Fill the File_Tag structure for FileTagList */
    FileTag = ET_File_Tag_Item_New();
    FileTag->saved = TRUE;    /* The file haven't been changed, so it's saved */
    
    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            Id3tag_Read_File_Tag(filename,FileTag);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            Ogg_Tag_Read_File_Tag(filename,FileTag);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,filename);
            break;
    }

    /* Fill the ET_File_Info structure */
    ETFileInfo = g_malloc0(sizeof(ET_File_Info));

    switch (ETFileDescription->FileType)
    {
        case MP3_FILE:
        case MP2_FILE:
            Mpeg_Header_Read_File_Info(filename,ETFileInfo);
            break;
#ifdef ENABLE_OGG
        case OGG_FILE:
            Ogg_Header_Read_File_Info(filename,ETFileInfo);
            break;
#endif
        case FLAC_FILE:
            Flac_Header_Read_File_Info(filename,ETFileInfo);
            //ET_Read_File_Info(filename,ETFileInfo); // To get al least the file size
            break;
        case UNKNOWN_FILE:
        default:
            g_print("ETFileInfo: Undefined file type %d for file %s.\n",ETFileDescription->FileType,filename);
            break;
    }

    /* Attach all data defined above to this ETFile item */
    ETFile = ET_File_Item_New();
    ETFile->ETFileKey         = ETFileKey;
    ETFile->ETFileDescription = ETFileDescription;
    ETFile->FileNameList      = g_list_append(NULL,FileName);
    ETFile->FileNameCur       = ETFile->FileNameList;
    ETFile->FileNameNew       = ETFile->FileNameList;
    ETFile->FileTagList       = g_list_append(NULL,FileTag);
    ETFile->FileTag           = ETFile->FileTagList;
    ETFile->ETFileInfo        = ETFileInfo;

    /* Add the item to the "main list" */
    ETFileList = g_list_append(ETFileList,ETFile);

    /* Sort the file list */
    ET_Sort_File_List(SORTING_FILE_MODE);


    /*
     * Process the filename and tag to generate undo if needed...
     * The undo key must be the same for FileName and FileTag => changed in the same time
     */
    undo_key = ET_Undo_Key_New();

    FileName = ET_File_Name_Item_New();
    FileName->key = undo_key;
    ET_Save_File_Name_Internal(ETFile,FileName);

    FileTag = ET_File_Tag_Item_New();
    FileTag->key = undo_key;
    ET_Save_File_Tag_Internal(ETFile,FileTag);

    /*
     * Generate undo for the file and the main undo list.
     * If no changes detected, FileName and FileTag item are deleted.
     */
    ET_Manage_Changes_Of_File_Data(ETFile,FileName,FileTag);

    /* Refresh file into browser list */
    //Browser_List_Refresh_File_In_List(ETFile);

    return ETFileList;
}




/**************************
 * File sorting functions *
 **************************/

/*
 * Sort the 'ETFileList' following the 'Sorting_Type'
 */
void ET_Sort_File_List (ET_Sorting_Type Sorting_Type)
{
    switch (Sorting_Type)
    {
        case SORTING_UNKNOWN:
        case SORTING_BY_ASCENDING_FILENAME:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Filename);
            break;
        case SORTING_BY_DESCENDING_FILENAME:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Filename);
            break;
        case SORTING_BY_ASCENDING_TRACK_NUMBER:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Track_Number);
            break;
        case SORTING_BY_DESCENDING_TRACK_NUMBER:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Track_Number);
            break;
        case SORTING_BY_ASCENDING_CREATION_DATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Creation_Date);
            break;
        case SORTING_BY_DESCENDING_CREATION_DATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Creation_Date);
            break;
        case SORTING_BY_ASCENDING_TITLE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Title);
            break;
        case SORTING_BY_DESCENDING_TITLE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Title);
            break;
        case SORTING_BY_ASCENDING_ARTIST:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Artist);
            break;
        case SORTING_BY_DESCENDING_ARTIST:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Artist);
            break;
        case SORTING_BY_ASCENDING_ALBUM:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Album);
            break;
        case SORTING_BY_DESCENDING_ALBUM:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Album);
            break;
        case SORTING_BY_ASCENDING_YEAR:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Year);
            break;
        case SORTING_BY_DESCENDING_YEAR:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Year);
            break;
        case SORTING_BY_ASCENDING_GENRE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Genre);
            break;
        case SORTING_BY_DESCENDING_GENRE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Genre);
            break;
        case SORTING_BY_ASCENDING_COMMENT:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_Comment);
            break;
        case SORTING_BY_DESCENDING_COMMENT:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_Comment);
            break;
        case SORTING_BY_ASCENDING_FILE_TYPE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_File_Type);
            break;
        case SORTING_BY_DESCENDING_FILE_TYPE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_File_Type);
            break;
        case SORTING_BY_ASCENDING_FILE_SIZE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_File_Size);
            break;
        case SORTING_BY_DESCENDING_FILE_SIZE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_File_Size);
            break;
        case SORTING_BY_ASCENDING_FILE_DURATION:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_File_Duration);
            break;
        case SORTING_BY_DESCENDING_FILE_DURATION:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_File_Duration);
            break;
        case SORTING_BY_ASCENDING_FILE_BITRATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_File_Bitrate);
            break;
        case SORTING_BY_DESCENDING_FILE_BITRATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_File_Bitrate);
            break;
        case SORTING_BY_ASCENDING_FILE_SAMPLERATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Ascending_File_Samplerate);
            break;
        case SORTING_BY_DESCENDING_FILE_SAMPLERATE:
            ETFileList = g_list_sort(ETFileList,(GCompareFunc)ET_Comp_Func_Sort_File_By_Descending_File_Samplerate);
            break;
    }
}


/*
 * Sort the list of files following the 'Sorting_Type' value. The new sorting is displayed in the UI.
 */
void ET_Sort_File_List_And_Update_UI (ET_Sorting_Type Sorting_Type)
{
    if (!ETFileList) return;
    
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    /* Go to the first item of the list */
    ET_File_List_First();
    ET_Sort_File_List(Sorting_Type);
    
    /* Reload files in browser list */
    Browser_List_Load_Files();

    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    Update_Command_Buttons_Sensivity();
}


/*
 * Comparison function for sorting by ascending file name.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Filename (ET_File *ETFile1, ET_File *ETFile2)
{
    gchar *filename1  = ((File_Name *)((GList *)ETFile1->FileNameCur)->data)->value;
    gchar *filename2  = ((File_Name *)((GList *)ETFile2->FileNameCur)->data)->value;
    gchar *dirname1   = g_dirname(filename1);
    gchar *dirname2   = g_dirname(filename2);
    gint cmp_dirname  = strcmp(dirname1,dirname2);
    gint cmp_filename = SORTING_FILE_CASE_SENSITIVE?strcmp(filename1,filename2):strcasecmp(filename1,filename2);
    
    g_free(dirname1);
    g_free(dirname2);
    if ( cmp_dirname==0 )    /* Files are in the same directory */
        return cmp_filename;
    else 
        return cmp_dirname;
}

/*
 * Comparison function for sorting by descending file name.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Filename (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending track number.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Track_Number (ET_File *ETFile1, ET_File *ETFile2)
{
    gint track1, track2;
    
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->track )
        track1 = 0;
    else
        track1 = atoi( ((File_Tag *)ETFile1->FileTag->data)->track );

    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->track )
        track2 = 0;
    else
        track2 = atoi( ((File_Tag *)ETFile2->FileTag->data)->track );

    // Second criterion
    if (track1 == track2)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (track1 - track2);
}

/*
 * Comparison function for sorting by descending track number.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Track_Number (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Track_Number(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending creation date.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Creation_Date (ET_File *ETFile1, ET_File *ETFile2)
{
    struct stat statbuf1;
    struct stat statbuf2;
    gchar *filename1 = ((File_Name *)ETFile1->FileNameCur->data)->value;
    gchar *filename2 = ((File_Name *)ETFile2->FileNameCur->data)->value;

    lstat(filename1, &statbuf1);
    lstat(filename2, &statbuf2);

    // Second criterion
    if (statbuf1.st_ctime == statbuf2.st_ctime)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (statbuf1.st_ctime - statbuf2.st_ctime); // Creation date
    //return (statbuf1.st_mtime - statbuf2.st_mtime); // Modification date
}

/*
 * Comparison function for sorting by descending creation date.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Creation_Date (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Creation_Date(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending title.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Title (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->title )
        return -1;
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->title )
        return 1;

    if (SORTING_FILE_CASE_SENSITIVE)
    {
        if ( strcmp(((File_Tag *)ETFile1->FileTag->data)->title,((File_Tag *)ETFile2->FileTag->data)->title) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcmp(((File_Tag *)ETFile1->FileTag->data)->title,((File_Tag *)ETFile2->FileTag->data)->title);
    }else
    {
        if ( strcasecmp(((File_Tag *)ETFile1->FileTag->data)->title,((File_Tag *)ETFile2->FileTag->data)->title) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcasecmp(((File_Tag *)ETFile1->FileTag->data)->title,((File_Tag *)ETFile2->FileTag->data)->title);
    }
}

/*
 * Comparison function for sorting by descending title.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Title (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Title(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending artist.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Artist (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->artist )
        return -1;
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->artist )
        return 1;

    if (SORTING_FILE_CASE_SENSITIVE)
    {
        if ( strcmp(((File_Tag *)ETFile1->FileTag->data)->artist,((File_Tag *)ETFile2->FileTag->data)->artist) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcmp(((File_Tag *)ETFile1->FileTag->data)->artist,((File_Tag *)ETFile2->FileTag->data)->artist);
    }else
    {
        if ( strcasecmp(((File_Tag *)ETFile1->FileTag->data)->artist,((File_Tag *)ETFile2->FileTag->data)->artist) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcasecmp(((File_Tag *)ETFile1->FileTag->data)->artist,((File_Tag *)ETFile2->FileTag->data)->artist);
    }
}

/*
 * Comparison function for sorting by descending artist.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Artist (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Album(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending album.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Album (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->album )
        return -1;
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->album )
        return 1;

    if (SORTING_FILE_CASE_SENSITIVE)
    {
        if ( strcmp(((File_Tag *)ETFile1->FileTag->data)->album,((File_Tag *)ETFile2->FileTag->data)->album) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcmp(((File_Tag *)ETFile1->FileTag->data)->album,((File_Tag *)ETFile2->FileTag->data)->album);
    }else
    {
        if ( strcasecmp(((File_Tag *)ETFile1->FileTag->data)->album,((File_Tag *)ETFile2->FileTag->data)->album) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcasecmp(((File_Tag *)ETFile1->FileTag->data)->album,((File_Tag *)ETFile2->FileTag->data)->album);
    }
}

/*
 * Comparison function for sorting by descending album.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Album (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Album(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending year.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Year (ET_File *ETFile1, ET_File *ETFile2)
{
    gint year1, year2;

    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->year )
        year1 = 0;
    else
        year1 = atoi( ((File_Tag *)ETFile1->FileTag->data)->year );

    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->year )
        year2 = 0;
    else
        year2 = atoi( ((File_Tag *)ETFile2->FileTag->data)->year );

    // Second criterion
    if (year1 == year2)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (year1 - year2);
}

/*
 * Comparison function for sorting by descending year.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Year (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Year(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending genre.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Genre (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->genre ) return -1;
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->genre ) return 1;

    if (SORTING_FILE_CASE_SENSITIVE)
    {
        if ( strcmp(((File_Tag *)ETFile1->FileTag->data)->genre,((File_Tag *)ETFile2->FileTag->data)->genre) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcmp(((File_Tag *)ETFile1->FileTag->data)->genre,((File_Tag *)ETFile2->FileTag->data)->genre);
    }else
    {
        if ( strcasecmp(((File_Tag *)ETFile1->FileTag->data)->genre,((File_Tag *)ETFile2->FileTag->data)->genre) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcasecmp(((File_Tag *)ETFile1->FileTag->data)->genre,((File_Tag *)ETFile2->FileTag->data)->genre);
    }
}

/*
 * Comparison function for sorting by descending genre.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Genre (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Genre(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending comment.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_Comment (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->FileTag->data || !((File_Tag *)ETFile1->FileTag->data)->comment )
        return -1;
    if ( !ETFile2->FileTag->data || !((File_Tag *)ETFile2->FileTag->data)->comment )
        return 1;

    if (SORTING_FILE_CASE_SENSITIVE)
    {
        if ( strcmp(((File_Tag *)ETFile1->FileTag->data)->comment,((File_Tag *)ETFile2->FileTag->data)->comment) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcmp(((File_Tag *)ETFile1->FileTag->data)->comment,((File_Tag *)ETFile2->FileTag->data)->comment);
    }else
    {
        if ( strcasecmp(((File_Tag *)ETFile1->FileTag->data)->comment,((File_Tag *)ETFile2->FileTag->data)->comment) == 0 )
            // Second criterion
            return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);
        else
            // First criterion
            return strcasecmp(((File_Tag *)ETFile1->FileTag->data)->comment,((File_Tag *)ETFile2->FileTag->data)->comment);
    }
}

/*
 * Comparison function for sorting by descending comment.
 */
gint ET_Comp_Func_Sort_File_By_Descending_Comment (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_Comment(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending file type (mp3, ogg, ...).
 */
gint ET_Comp_Func_Sort_File_By_Ascending_File_Type (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->ETFileDescription ) return -1;
    if ( !ETFile2->ETFileDescription ) return 1;

    // Second criterion
    if (ETFile1->ETFileDescription->FileType == ETFile2->ETFileDescription->FileType)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (ETFile1->ETFileDescription->FileType - ETFile2->ETFileDescription->FileType);
}

/*
 * Comparison function for sorting by descending file type (mp3, ogg, ...).
 */
gint ET_Comp_Func_Sort_File_By_Descending_File_Type (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_File_Type(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending file size.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_File_Size (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->ETFileInfo ) return -1;
    if ( !ETFile2->ETFileInfo ) return 1;

    // Second criterion
    if (ETFile1->ETFileInfo->size == ETFile2->ETFileInfo->size)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (ETFile1->ETFileInfo->size - ETFile2->ETFileInfo->size);
}

/*
 * Comparison function for sorting by descending file size.
 */
gint ET_Comp_Func_Sort_File_By_Descending_File_Size (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_File_Size(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending file duration.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_File_Duration (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->ETFileInfo ) return -1;
    if ( !ETFile2->ETFileInfo ) return 1;

    // Second criterion
    if (ETFile1->ETFileInfo->duration == ETFile2->ETFileInfo->duration)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (ETFile1->ETFileInfo->duration - ETFile2->ETFileInfo->duration);
}

/*
 * Comparison function for sorting by descending file duration.
 */
gint ET_Comp_Func_Sort_File_By_Descending_File_Duration (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_File_Duration(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending file bitrate.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_File_Bitrate (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->ETFileInfo ) return -1;
    if ( !ETFile2->ETFileInfo ) return 1;

    // Second criterion
    if (ETFile1->ETFileInfo->bitrate == ETFile2->ETFileInfo->bitrate)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (ETFile1->ETFileInfo->bitrate - ETFile2->ETFileInfo->bitrate);
}

/*
 * Comparison function for sorting by descending file bitrate.
 */
gint ET_Comp_Func_Sort_File_By_Descending_File_Bitrate (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_File_Bitrate(ETFile2,ETFile1);
}


/*
 * Comparison function for sorting by ascending file samplerate.
 */
gint ET_Comp_Func_Sort_File_By_Ascending_File_Samplerate (ET_File *ETFile1, ET_File *ETFile2)
{
    if ( !ETFile1->ETFileInfo ) return -1;
    if ( !ETFile2->ETFileInfo ) return 1;

    // Second criterion
    if (ETFile1->ETFileInfo->samplerate == ETFile2->ETFileInfo->samplerate)
        return ET_Comp_Func_Sort_File_By_Ascending_Filename(ETFile1,ETFile2);

    // First criterion
    return (ETFile1->ETFileInfo->samplerate - ETFile2->ETFileInfo->samplerate);
}

/*
 * Comparison function for sorting by descending file samplerate.
 */
gint ET_Comp_Func_Sort_File_By_Descending_File_Samplerate (ET_File *ETFile1, ET_File *ETFile2)
{
    return ET_Comp_Func_Sort_File_By_Ascending_File_Samplerate(ETFile2,ETFile1);
}




/***************************
 * List handling functions *
 ***************************/

/*
 * Returns the first item of the "main list"
 */
GList *ET_File_List_First (void)
{
    ETFileList = g_list_first(ETFileList);
    ETFileList_Index = 1;
    return ETFileList;
}

/*
 * Returns the previous item of the "main list". When no more item, it returns NULL.
 */
GList *ET_File_List_Previous (void)
{
    if (ETFileList && ETFileList->prev)
    {
        ETFileList_Index--;
        return ETFileList = ETFileList->prev;
    }else
    {
        ETFileList_Index = 1;
        return NULL;
    }
}

/*
 * Returns the next item of the "main list".
 * When no more item, it returns NULL to don't "overwrite" the list.
 */
GList *ET_File_List_Next (void)
{
    if (ETFileList && ETFileList->next)
    {
        ETFileList_Index++;
        return ETFileList = ETFileList->next;
    }else
    {
        ETFileList_Index = ETFileList_Length;
        return NULL;
    }
}

/*
 * Returns the last item of the "main list"
 */
GList *ET_File_List_Last (void)
{
    ETFileList = g_list_last(ETFileList);
    ETFileList_Index = ETFileList_Length;
    return ETFileList;
}

/*
 * Remove an item from the "main list"
 */
GList *ET_File_List_Remove (GList *item_to_remove)
{
    GList *etfilelist;

    if (!item_to_remove)
        return ETFileList;

    /* Get the item that will become the new current item, when 'item_to_remove' will be removed */
    if (item_to_remove->next)
        etfilelist = item_to_remove->next;
    else if (item_to_remove->prev)
        etfilelist = item_to_remove->prev;
    else
        etfilelist = NULL;

    /* Remove infos of the file */
    ETFileList_TotalSize     -= ((ET_File_Info *)((ET_File *)item_to_remove->data)->ETFileInfo)->size;
    ETFileList_TotalDuration -= ((ET_File_Info *)((ET_File *)item_to_remove->data)->ETFileInfo)->duration;
    
    /* Free data of the file */
    g_list_remove_link(g_list_first(ETFileList),item_to_remove);
    ET_Free_File_List_Item(item_to_remove->data);

    ETFileList = etfilelist;
     
    /* Recalculate length of list */
    ET_File_List_Get_Length();
    if (ETFileList && ETFileList->data)
    {
        /* Select the new file (synchronize index,...) */
        ET_File_List_Nth_By_Key(((ET_File *)ETFileList->data)->ETFileKey);
    }else
    {
        /* Reinit the tag and file area */
        Clear_File_Entry_Field();
        Clear_Header_Fields();
        Clear_Tag_Entry_Fields();
        //ETFileList_Index = ETFileList_Length = 0;
        gtk_label_set_text(GTK_LABEL(FileIndex),"0/0:");
        Update_Command_Buttons_Sensivity();
    }

    return ETFileList;
}

/*
 * Returns the item of the "main list" which correspond to the given 'etfilekey' (was used into browser list).
 */
GList *ET_File_List_Nth_By_Key (gulong etfilekey)
{
    ET_File_List_First();
    while (ETFileList)
    {
        if ( etfilekey == ((ET_File *)ETFileList->data)->ETFileKey )
            break;
        if (ET_File_List_Next() == NULL)
            return NULL;    /* Not found */
    }
    return ETFileList;
}

/*
 * Returns the length of the "main list"
 */
guint ET_File_List_Get_Length (void)
{
    GList *list = NULL;
    
    list = g_list_first(ETFileList);
    ETFileList_Length = g_list_length(list);
    return ETFileList_Length;
}




/*********************
 * Freeing functions *
 *********************/

/*
 * Frees the full main list of files: GList *ETFileList and reinitialize it.
 */
gboolean ET_Free_File_List (void)
{
    if (!ETFileList) return FALSE;
    
    ETFileList = g_list_last(ETFileList);
    while (ETFileList)
    {
        ET_Free_File_List_Item((ET_File *)ETFileList->data);
        if (!ETFileList->prev) break;
        ETFileList = ETFileList->prev;
    }

    g_list_free(ETFileList);
    ETFileList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees one item of the full main list of files.
 */
gboolean ET_Free_File_List_Item (ET_File *ETFile)
{
    if (ETFile)
    {
        /* Frees the lists */
        ET_Free_File_Name_List(ETFile->FileNameList);
        ET_Free_File_Name_List(ETFile->FileNameListBak);
        ET_Free_File_Tag_List (ETFile->FileTagList);
        ET_Free_File_Tag_List (ETFile->FileTagListBak);
        /* Frees infos of ETFileInfo */
        if ( (ET_File_Info *)ETFile->ETFileInfo )
            g_free( (ET_File_Info *)ETFile->ETFileInfo );
        g_free(ETFile);
        ETFile = NULL;
    }
    return TRUE;
}


/*
 * Frees the full list: GList *FileNameList.
 */
gboolean ET_Free_File_Name_List (GList *FileNameList)
{
    GList *list;
    
    if (!FileNameList) return FALSE;
    
    list = g_list_last(FileNameList);
    while (list)
    {
        if ( (File_Name *)list->data )
            ET_Free_File_Name_Item( (File_Name *)list->data );

        if (!list->prev) break;
        list = list->prev;
    }
    g_list_free(list);
    FileNameList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees a File_Name item.
 */
gboolean ET_Free_File_Name_Item (File_Name *FileName)
{
    if (!FileName) return FALSE;
    
    if (FileName->value)
        g_free(FileName->value);
    g_free(FileName);
    FileName = (File_Name *)NULL;
    return TRUE;
}

    
/*
 * Frees the full list: GList *TagList.
 */
gboolean ET_Free_File_Tag_List (GList *FileTagList)
{
    GList *list;

    if (!FileTagList) return FALSE;

    list = g_list_last(FileTagList);
    while (list)
    {
        if ( (File_Tag *)list->data )
            ET_Free_File_Tag_Item( (File_Tag *)list->data );

        if (!list->prev) break;
        list = list->prev;
    }
    g_list_free(list);
    FileTagList = (GList *)NULL;
    return TRUE;
}


/*
 * Frees the list of 'other' field in a File_Tag item (contains attached gchar data).
 */
gboolean ET_Free_File_Tag_Item_Other_Field (File_Tag *FileTag)
{
    GList *other = FileTag->other;
    while (other)
    {
        if (other->data)
            g_free(other->data);
        other = other->next;
    }
    g_list_free(FileTag->other);
}


/*
 * Frees a File_Tag item.
 */
gboolean ET_Free_File_Tag_Item (File_Tag *FileTag)
{
    if (!FileTag) return FALSE;

    if (FileTag->title)       g_free(FileTag->title);
    if (FileTag->artist)      g_free(FileTag->artist);
    if (FileTag->album)       g_free(FileTag->album);
    if (FileTag->year)        g_free(FileTag->year);
    if (FileTag->track)       g_free(FileTag->track);
    if (FileTag->track_total) g_free(FileTag->track_total);
    if (FileTag->genre)       g_free(FileTag->genre);
    if (FileTag->comment)     g_free(FileTag->comment);
    // Free list of other fields
    ET_Free_File_Tag_Item_Other_Field(FileTag);

    g_free(FileTag);
    FileTag = (File_Tag *)NULL;
    return TRUE;
}


/*
 * History list contains only pointers, so no data to free except the history structure.
 */
gboolean ET_Free_History_File_List (void)
{
    GList *list;

    if (!ETHistoryFileList) return FALSE;

    ETHistoryFileList = g_list_first(ETHistoryFileList);
    list = ETHistoryFileList;
    while (list)
    {
        if ( (ET_History_File *)list->data )
            g_free( (ET_History_File *)list->data );
        list = list->next;
    }
    g_list_free(ETHistoryFileList);
    ETHistoryFileList = (GList *)NULL;
    return TRUE;
}


/*********************
 * Copying functions *
 *********************/

/*
 * Duplicate the 'other' list
 */
gboolean ET_Copy_File_Tag_Item_Other_Field (ET_File *ETFile, File_Tag *FileTag)
{
    File_Tag *FileTagCur;
    GList *list = NULL;

    FileTagCur = (File_Tag *)(ETFile->FileTag)->data;
    list = FileTagCur->other;
    while (list)
    {
        FileTag->other = g_list_append(FileTag->other,g_strdup((gchar *)list->data));
        list = list->next;
    }
    return TRUE;
}


/*
 * Copy data of the File_Tag structure (of ETFile) to the FileTag item.
 * Reallocate data if not null.
 */
gboolean ET_Copy_File_Tag_Item (ET_File *ETFile, File_Tag *FileTag)
{
    File_Tag *FileTagCur;
    
    if (!ETFile || !(File_Tag *)(ETFile->FileTag)->data || !FileTag) return FALSE;
    
    /* The data to duplicate to FileTag */
    FileTagCur = (File_Tag *)(ETFile->FileTag)->data;
    // Key for the item, may be overwritten
    FileTag->key = ET_Undo_Key_New();

    if (FileTagCur->title)
    {
        FileTag->title = g_strdup(FileTagCur->title);
    }else
    {
        if (FileTag->title)
            g_free(FileTag->title);
        FileTag->title = NULL;
    }
    
    if (FileTagCur->artist)
    {
        FileTag->artist = g_strdup(FileTagCur->artist);
    }else
    {
        if (FileTag->artist)
            g_free(FileTag->artist);
        FileTag->artist = NULL;
    }
    
    if (FileTagCur->album)
    {
        FileTag->album = g_strdup(FileTagCur->album);
    }else
    {
        if (FileTag->album)
            g_free(FileTag->album);
        FileTag->album = NULL;
    }
    
    if (FileTagCur->year)
    {
        FileTag->year = g_strdup(FileTagCur->year);
    }else
    {
        if (FileTag->year)
            g_free(FileTag->year);
        FileTag->year = NULL;
    }
    
    if (FileTagCur->track)
    {
        FileTag->track = g_strdup(FileTagCur->track);
    }else
    {
        if (FileTag->track)
            g_free(FileTag->track);
        FileTag->track = NULL;
    }
    
    if (FileTagCur->track_total)
    {
        FileTag->track_total = g_strdup(FileTagCur->track_total);
    }else
    {
        if (FileTag->track_total)
            g_free(FileTag->track_total);
        FileTag->track_total = NULL;
    }
    
    if (FileTagCur->genre)
    {
        FileTag->genre = g_strdup(FileTagCur->genre);
    }else
    {
        if (FileTag->genre)
            g_free(FileTag->genre);
        FileTag->genre = NULL;
    }
    
    if (FileTagCur->comment)
    {
        FileTag->comment = g_strdup(FileTagCur->comment);
    }else
    {
        if (FileTag->comment)
            g_free(FileTag->comment);
        FileTag->comment = NULL;
    }
    
    if (FileTagCur->other)
    {
        ET_Copy_File_Tag_Item_Other_Field(ETFile,FileTag);
    }else
    {
        ET_Free_File_Tag_Item_Other_Field (FileTag);
    }

    return TRUE;
}


/*
 * Set the value of a field of a FileName item (for ex, value of FileName->value)
 * Must be used only for the 'gchar *' components (Only used for the filename)
 */
gboolean ET_Set_Field_File_Name_Item (gint *FileNameField, gchar *value)
{
    return ET_Set_Field_File_Tag_Item(FileNameField,value);
}


/*
 * Set the value of a field of a FileTag item (for ex, value of FileTag->title)
 * Must be used only for the 'gchar *' components
 */
gboolean ET_Set_Field_File_Tag_Item (gint *FileTagField, gchar *value)
{
    if (!FileTagField) return FALSE;
    
    if ((gchar *)*FileTagField != NULL)
    {
        g_free((gchar *)*FileTagField);
        (gchar *)*FileTagField = NULL;
    }
    
    if (value != NULL)
        (gchar *)*FileTagField = g_strdup(value);

    return TRUE;
}


/************************
 * Displaying functions *
 ************************/

/*
 * Display informations of the file (Position + Header + Tag) to the user interface.
 */
void ET_Display_File_Data_To_UI (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    gchar *cur_filename;
    gchar *new_filename;
    gchar *msg;


    if (!ETFile) return;

    new_filename = ((File_Name *)((GList *)ETFile->FileNameNew)->data)->value;
    cur_filename = ((File_Name *)((GList *)ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;
    
    /* Display position in list + show/hide icon if file writable/read_only */
    ET_Display_File_And_List_Status_To_UI(cur_filename);
    
    /* Display filename (and his path) */
    ET_Display_Filename_To_UI(new_filename);
    
    /* Display tag data */
    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("ID3 Tag"));
            ET_Display_File_Tag_To_UI(ETFile);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("OGG Vorbis Tag"));
            ET_Display_File_Tag_To_UI(ETFile);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            gtk_frame_set_label(GTK_FRAME(TagFrame),_("Tag"));
            ET_Display_File_Tag_To_UI(ETFile); // To reinit screen
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,cur_filename);
            break;
    }

    /* Display file data, header data and file type */
    switch (ETFileDescription->FileType)
    {
        case MP3_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("MP3 File"));
            Mpeg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
        case MP2_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("MP2 File"));
            Mpeg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
#ifdef ENABLE_OGG
        case OGG_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("OGG File"));
            Ogg_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            break;
#endif
        case FLAC_FILE:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("FLAC File"));
            Flac_Header_Display_File_Info_To_UI(cur_filename,ETFile->ETFileInfo);
            // Default displaying
            //ET_Display_File_Info_To_UI(ETFile->ETFileInfo);
            break;
        case UNKNOWN_FILE:
        default:
            gtk_frame_set_label(GTK_FRAME(FileFrame),_("File"));
            ET_Display_File_Info_To_UI(ETFile->ETFileInfo);
            g_print("ETFileInfo: Undefined file type %d for file %s.\n",ETFileDescription->FileType,cur_filename);
            break;
    }
    
    msg = g_strdup_printf(_("File: '%s'"),cur_filename);
    Statusbar_Message(msg,FALSE);
    g_free(msg);
}


/*
 * Toggle visibility of the small status icon if filename is read-only or not found.
 * Show the position of the current file in the list, by using the index and list length.
 */
void ET_Display_File_And_List_Status_To_UI (gchar *filename)
{
    FILE *file;
    gchar *text;

    /* Show/hide 'AccessStatusIcon' */
    if ( (file=fopen(filename,"r+b"))!=NULL )
    {
        gtk_widget_hide(ReadOnlyStatusIconBox);
        gtk_widget_hide(BrokenStatusIconBox);
        fclose(file);
    }else
    {
        switch(errno)
        {
            case EACCES:    /* Permission denied */
            case EROFS:     /* Read-only file system */
                /* Read only file */
                gtk_widget_show_all(ReadOnlyStatusIconBox);
                gtk_widget_hide(BrokenStatusIconBox);
                break;
            case ENOENT:    /* No such file or directory */
            default:
                /* File not found */
                gtk_widget_show_all(BrokenStatusIconBox);
                gtk_widget_hide(ReadOnlyStatusIconBox);
        }
    }

    /* Show position of current file in list */
    text = g_strdup_printf("%d/%d:",ETFileList_Index,ETFileList_Length);
    gtk_label_set_text(GTK_LABEL(FileIndex),text);
    g_free(text);
}


void ET_Display_Filename_To_UI (gchar *filename)
{
    gchar *text, *pos;

    if (!filename) return;

    /*
     * Set filename into FileEntry
     */
    text = g_strdup(g_basename(filename));
    // Remove the extension
    if ((pos=strrchr(text,'.'))!=NULL) *pos = 0;
    gtk_entry_set_text(GTK_ENTRY(FileEntry),text);
    g_free(text);
    // Justify to the left text into FileEntry
    gtk_editable_set_position(GTK_EDITABLE(FileEntry),0);

    /*
     * Set the path to the file into BrowserEntry (dirbrowser)
     */
    text = g_dirname(filename);
    Browser_Entry_Set_Text(text);
    g_free(text);
}


/*
 * Display all tag infos (tags fields) into entry boxes of the user interface.
 * These data have the same structure for all files.
 */
gboolean ET_Display_File_Tag_To_UI (ET_File *ETFile)
{
    File_Tag *FileTag = NULL;


    if (!ETFile || !ETFile->FileTag)
    {
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),                  "");
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),                 "");
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),                  "");
        gtk_entry_set_text(GTK_ENTRY(YearEntry),                   "");
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),             "");
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),                "");
        //Tag_Area_Set_Sensitive(FALSE);
        return FALSE;
    }

    //Tag_Area_Set_Sensitive(TRUE); // Causes displaying problem when saving files

    FileTag = (File_Tag *)(ETFile->FileTag->data);

    /* Show title */
    if (FileTag && FileTag->title)
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),FileTag->title);
    else
        gtk_entry_set_text(GTK_ENTRY(TitleEntry),"");

    /* Show artist */
    if (FileTag && FileTag->artist)
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),FileTag->artist);
    else
        gtk_entry_set_text(GTK_ENTRY(ArtistEntry),"");

    /* Show album */
    if (FileTag && FileTag->album)
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),FileTag->album);
    else
        gtk_entry_set_text(GTK_ENTRY(AlbumEntry),"");

    /* Show year */
    if (FileTag && FileTag->year)
        gtk_entry_set_text(GTK_ENTRY(YearEntry),FileTag->year);
    else
        gtk_entry_set_text(GTK_ENTRY(YearEntry),"");

    /* Show track */
    if (FileTag && FileTag->track)
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),FileTag->track);
    else
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");

    /* Show number of tracks on the album */
    if (FileTag && FileTag->track_total)
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),FileTag->track_total);
    else
        gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),"");

    /* Show genre */
    if (FileTag && FileTag->genre)
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),FileTag->genre);
    else
        gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");

    /* Show comment */
    if (FileTag && FileTag->comment)
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),FileTag->comment);
    else
        gtk_entry_set_text(GTK_ENTRY(CommentEntry),"");

    return TRUE;
}


/*
 * "Default" way to display File Info to the user interface.
 */
gboolean ET_Display_File_Info_To_UI(ET_File_Info *ETFileInfo)
{
    gchar *text;
    gchar *time  = NULL;
    gchar *time1 = NULL;
    gchar *size  = NULL;
    gchar *size1 = NULL;

    /* MPEG, Layer versions */
    text = g_strdup_printf("%d, Layer %d",ETFileInfo->version,ETFileInfo->layer);
    gtk_label_set_text(GTK_LABEL(VersionValueLabel),text);
    g_free(text);

    /* Bitrate */
    text = g_strdup_printf(_("%d kb/s"),ETFileInfo->bitrate);
    gtk_label_set_text(GTK_LABEL(BitrateValueLabel),text);
    g_free(text);

    /* Samplerate */
    text = g_strdup_printf(_("%d Hz"),ETFileInfo->samplerate);
    gtk_label_set_text(GTK_LABEL(SampleRateValueLabel),text);
    g_free(text);

    /* Mode */
    text = g_strdup_printf("%d",ETFileInfo->mode);
    gtk_label_set_text(GTK_LABEL(ModeValueLabel),text);
    g_free(text);

    /* Size */
    size  = Convert_Size(ETFileInfo->size);
    size1 = Convert_Size(ETFileList_TotalSize);
    text  = g_strdup_printf("%s (%s)",size,size1);
    gtk_label_set_text(GTK_LABEL(SizeValueLabel),text);
    if (size)  g_free(size);
    if (size1) g_free(size1);
    g_free(text);

    /* Duration */
    time  = Convert_Duration(ETFileInfo->duration);
    time1 = Convert_Duration(ETFileList_TotalDuration);
    text  = g_strdup_printf("%s (%s)",time,time1);
    gtk_label_set_text(GTK_LABEL(DurationValueLabel),text);
    if (time)  g_free(time);
    if (time1) g_free(time1);
    g_free(text);

    return TRUE;
}




/********************
 * Saving functions *
 ********************/

/*
 * Save informations of the file, contained into the entries of the user interface, in the list.
 * An undo key is generated to be used for filename and tag if there are changed is the same time.
 * Filename and Tag.
 */
void ET_Save_File_Data_From_UI (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    File_Name *FileName;
    File_Tag  *FileTag;
    gulong     undo_key;
    gchar     *cur_filename;


    if (!ETFile) return;

    cur_filename = ((File_Name *)((GList *)ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;
    undo_key = ET_Undo_Key_New();

    /*
     * Save filename and generate undo for filename
     */
    FileName = g_malloc0(sizeof(File_Name));
    ET_Initialize_File_Name_Item(FileName);
    FileName->key = undo_key;
    ET_Save_File_Name_From_UI(ETFile,FileName); // Used for all files!

    /*
     * Save tag data and generate undo for tag
     */
    FileTag = g_malloc0(sizeof(File_Tag));
    ET_Initialize_File_Tag_Item(FileTag);
    FileTag->key = undo_key;

    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
#ifdef ENABLE_OGG
        case OGG_TAG:
#endif
            ET_Save_File_Tag_From_UI(FileTag);
            ET_Copy_File_Tag_Item_Other_Field(ETFile,FileTag);
            break;
        case UNKNOWN_TAG:
        default:
            g_print("FileTag: Undefined tag type %d for file %s.\n",ETFileDescription->TagType,cur_filename);
            break;
    }

    /*
     * Generate undo for the file and the main undo list.
     * If no changes detected, FileName and FileTag item are deleted.
     */
    ET_Manage_Changes_Of_File_Data(ETFile,FileName,FileTag);

    /* Refresh file into browser list */
    Browser_List_Refresh_File_In_List(ETFile);
}


/*
 * Save displayed filename into list if it had been changed. Generates also an history list for undo/redo.
 *  - ETFile : the current etfile that we want to save,
 *  - FileName : where are save 'temporary' the new filename.
 */
gboolean ET_Save_File_Name_From_UI (ET_File *ETFile, File_Name *FileName)
{
    gchar *filename_new = NULL;
    gchar *dirname = NULL;
    gchar *filename;


    if (!ETFile || !FileName)
        return FALSE;

    // Get the current path to the file
    dirname = g_dirname( ((File_Name *)ETFile->FileNameNew->data)->value );

    filename = gtk_entry_get_text_1(FileEntry);
    if (filename!=NULL && strlen(filename)>0)
    {
        // Regenerate the new filename (without path)
        filename_new = g_strconcat(filename,ETFile->ETFileDescription->Extension,NULL);
    }else
    {
        // Keep the 'last' filename (if it was deleted in the entry for ex)...
        filename_new = g_strdup(g_basename( ((File_Name *)ETFile->FileNameNew->data)->value) );
    }

    // Check if new filename seems to be correct
    if ( !filename_new || strlen(filename_new) <= strlen(ETFile->ETFileDescription->Extension) )
    {
        FileName->value = NULL;

        if (filename_new) g_free(filename_new);
        if (dirname)      g_free(dirname);
        return FALSE;
    }else
    {
        // Convert the illegal characters
        ET_File_Name_Convert_Character(filename_new);

        // Set the new file name
        FileName->value = g_strconcat(dirname,"/",filename_new,NULL);

        if (filename_new) g_free(filename_new);
        if (dirname)      g_free(dirname);
        return TRUE;
    }
}


/*
 * Do the same thing of ET_Save_File_Name_From_UI without getting the data from the UI.
 */
gboolean ET_Save_File_Name_Internal (ET_File *ETFile, File_Name *FileName)
{
    gchar *filename_new = NULL;
    gchar *dirname = NULL;
    gchar *filename;
    gchar *pos;


    if (!ETFile || !FileName)
        return FALSE;

    // Get the current path to the file
    dirname = g_dirname( ((File_Name *)ETFile->FileNameNew->data)->value );

    // Get the name of file (and rebuild it with extension with a 'correct' case)
    filename = g_strdup(g_basename( ((File_Name *)ETFile->FileNameNew->data)->value ));
    // Remove the extension
    if ((pos=strrchr(filename,'.'))!=NULL) *pos = 0;
    // Regenerate the new filename (without path)
    filename_new = g_strconcat(filename,ETFile->ETFileDescription->Extension,NULL);
    g_free(filename);

    // Check if new filename seems to be correct
    if (filename_new)
    {
        // Convert the illegal characters
        ET_File_Name_Convert_Character(filename_new);

        // Set the new file name
        FileName->value = g_strconcat(dirname,"/",filename_new,NULL);

        if (filename_new) g_free(filename_new);
        if (dirname)      g_free(dirname);
        return TRUE;
    }else
    {
        FileName->value = NULL;

        if (filename_new) g_free(filename_new);
        if (dirname)      g_free(dirname);
        return FALSE;
    }
}


/*
 * Load values, entered into entries of the UI, into a File_Tag structure which had been newly created.
 */
gboolean ET_Save_File_Tag_From_UI (File_Tag *FileTag)
{
    gchar *buffer = NULL;


    if (!FileTag)
        return FALSE;

    /* Title */
    buffer = gtk_entry_get_text_1(TitleEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->title = g_strdup(buffer);
    } else
    {
        FileTag->title = NULL;
    }


    /* Artist */
    buffer = gtk_entry_get_text_1(ArtistEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->artist = g_strdup(buffer);
    } else
    {
        FileTag->artist = NULL;
    }


    /* Album */
    buffer = gtk_entry_get_text_1(AlbumEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->album = g_strdup(buffer);
    } else
    {
        FileTag->album = NULL;
    }


    /* Year */
    buffer = gtk_entry_get_text_1(YearEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->year = g_strdup(buffer);
    } else
    {
        FileTag->year = NULL;
    }


    /* Track */
    buffer = gtk_entry_get_text_1(TrackEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0  )
    {
        if (NUMBER_TRACK_FORMATED)
            FileTag->track = g_strdup_printf("%.2d",atoi(buffer));
        else
            FileTag->track = g_strdup(buffer);
    } else
    {
        FileTag->track = NULL;
    }


    /* Track Total */
    buffer = gtk_entry_get_text_1(TrackTotalEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0  )
    {
        if (NUMBER_TRACK_FORMATED)
            FileTag->track_total = g_strdup_printf("%.2d",atoi(buffer));
        else
            FileTag->track_total = g_strdup(buffer);
    } else
    {
        FileTag->track_total = NULL;
    }


    /* Genre */
    buffer = gtk_entry_get_text_1(GenreEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->genre = g_strdup(buffer);
    } else
    {
        FileTag->genre = NULL;
    }


    /* Comment */
    buffer = gtk_entry_get_text_1(CommentEntry);
    Strip_String(buffer);

    if ( strlen(buffer) > 0 )
    {
        FileTag->comment = g_strdup(buffer);
    } else
    {
        FileTag->comment = NULL;
    }

    return TRUE;
}


/*
 * Do the same thing of ET_Save_File_Tag_From_UI without getting the data from the UI.
 */
gboolean ET_Save_File_Tag_Internal (ET_File *ETFile, File_Tag *FileTag)
{
    File_Tag *FileTagCur;


    if (!ETFile || !ETFile->FileTag || !FileTag)
        return FALSE;

    FileTagCur = (File_Tag *)ETFile->FileTag->data;

    /* Title */
    if ( FileTagCur->title && strlen(FileTagCur->title)>0 )
    {
        FileTag->title = g_strdup(FileTagCur->title);
    } else
    {
        FileTag->title = NULL;
    }
    Strip_String(FileTag->title);

    /* Artist */
    if ( FileTagCur->artist && strlen(FileTagCur->artist)>0 )
    {
        FileTag->artist = g_strdup(FileTagCur->artist);
    } else
    {
        FileTag->artist = NULL;
    }
    Strip_String(FileTag->artist);


    /* Album */
    if ( FileTagCur->album && strlen(FileTagCur->album)>0 )
    {
        FileTag->album = g_strdup(FileTagCur->album);
    } else
    {
        FileTag->album = NULL;
    }
    Strip_String(FileTag->album);


    /* Year */
    if ( FileTagCur->year && strlen(FileTagCur->year)>0 )
    {
        FileTag->year = g_strdup(FileTagCur->year);
    } else
    {
        FileTag->year = NULL;
    }
    Strip_String(FileTag->year);


    /* Track */
    if ( FileTagCur->track && strlen(FileTagCur->track)>0 )
    {
        gchar *tmp_str;

        if (NUMBER_TRACK_FORMATED)
            FileTag->track = g_strdup_printf("%.2d",atoi(FileTagCur->track));
        else
            FileTag->track = g_strdup(FileTagCur->track);
        // This field must contain only digits
        tmp_str = FileTag->track;
        while (isdigit(*tmp_str)) tmp_str++;
            *tmp_str = 0;
    } else
    {
        FileTag->track = NULL;
    }
    Strip_String(FileTag->track);


    /* Track Total */
    if ( FileTagCur->track_total && strlen(FileTagCur->track_total)>0 )
    {
        if (NUMBER_TRACK_FORMATED)
            FileTag->track_total = g_strdup_printf("%.2d",atoi(FileTagCur->track_total));
        else
            FileTag->track_total = g_strdup(FileTagCur->track_total);
    } else
    {
        FileTag->track_total = NULL;
    }
    Strip_String(FileTag->track_total);


    /* Genre */
    if ( FileTagCur->genre && strlen(FileTagCur->genre)>0 )
    {
        FileTag->genre = g_strdup(FileTagCur->genre);
    } else
    {
        FileTag->genre = NULL;
    }
    Strip_String(FileTag->genre);


    /* Comment */
    if ( FileTagCur->comment && strlen(FileTagCur->comment)>0 )
    {
        FileTag->comment = g_strdup(FileTagCur->comment);
    } else
    {
        FileTag->comment = NULL;
    }
    Strip_String(FileTag->comment);


    return TRUE;
}


/*
 * Save data contained into File_Tag structure to the file on hard disk.
 */
gboolean ET_Save_File_Tag_To_HD (ET_File *ETFile)
{
    ET_File_Description *ETFileDescription;
    gchar *cur_filename;
    gboolean state;
    struct stat statbuf;
    gboolean file_set_properties;


    if (!ETFile) return FALSE;

    cur_filename = ((File_Name *)(ETFile->FileNameCur)->data)->value;
    ETFileDescription = ETFile->ETFileDescription;

    // Save permissions of the file (cause they may change with files on NFS)
    if ( stat(cur_filename,&statbuf)!=-1 )
        file_set_properties = TRUE;
    else
        file_set_properties = FALSE;

    switch (ETFileDescription->TagType)
    {
        case ID3_TAG:
            state = Id3tag_Write_File_Tag(ETFile);
            break;
#ifdef ENABLE_OGG
        case OGG_TAG:
            state = Ogg_Tag_Write_File_Tag(ETFile);
            break;
#endif
        case UNKNOWN_TAG:
        default:
            g_print("Saving to HD: Undefined function for tag type '%d' (file %s).\n",
                ETFileDescription->TagType,cur_filename);
            state = FALSE;
            break;
    }

    // Update properties for the file
    if ( file_set_properties == TRUE )
    {
        chmod(cur_filename,statbuf.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO));
        chown(cur_filename,statbuf.st_uid,statbuf.st_gid);
    }

    if (state==TRUE)
    {
        ET_Mark_File_Tag_As_Saved(ETFile);
        return TRUE;
    }else
    {
        return FALSE;
    }
}

/*
 * Function used to update path of filenames into list after renaming a parent directory
 * (for ex: "/mp3/old_path/file.mp3" to "/mp3/new_path/file.mp3"
 */
void ET_Update_Directory_Name_Into_File_List(gchar* last_path, gchar *new_path)
{
    GList *filelist;
    ET_File *file;
    GList *filenamelist;
    gchar *filename;
    gchar *last_path_tmp;    

    if (!ETFileList || !last_path || !new_path || strlen(last_path)<1 || strlen(new_path)<1 )
        return;
    
    // Add '/' to end of path to avoid ambiguity between a directory and a filename...
    if (last_path[strlen(last_path)-1]=='/') last_path_tmp = g_strdup(last_path);
    else                                     last_path_tmp = g_strconcat(last_path,"/",NULL);

    filelist = g_list_first(ETFileList);
    while (filelist)
    {
        if ( (file=filelist->data) && (filenamelist=file->FileNameList) )
        {
            while (filenamelist)
            {
                if ( filenamelist->data && (filename=((File_Name *)filenamelist->data)->value) )
                {
                    // Replace path of filename
                    if ( strncmp(filename,last_path_tmp,strlen(last_path_tmp))==0 )
                    {
                        gchar *filename_tmp;

                        // Create the new filename
                        filename_tmp = g_strconcat(new_path,
                                                   (new_path[strlen(new_path)-1]=='/')?"":"/",
                                                   &filename[strlen(last_path_tmp)],NULL);
                        // Replace the file name
                        g_free(filename);
                        ((File_Name *)filenamelist->data)->value = g_strdup(filename_tmp);
                        g_free(filename_tmp);
                    }
                }
                filenamelist = g_list_next(filenamelist);
             }
        }
        filelist = g_list_next(filelist);
    }

    g_free(last_path_tmp);
}



/***********************
 * Undo/Redo functions *
 ***********************/

/*
 * Check if 'FileName' and 'FileTag' differ with those of 'ETFile'.
 * Manage undo feature for the ETFile and the main undo list.
 */
gboolean ET_Manage_Changes_Of_File_Data (ET_File *ETFile, File_Name *FileName, File_Tag *FileTag)
{
    gboolean undo_added = FALSE;
    
    if (!ETFile) return FALSE;
    
    /*
     * Detect changes of filename and generate the filename undo list
     */
    if (FileName)
    {
        if ( ETFile->FileNameNew && ET_Detect_Changes_Of_File_Name( (File_Name *)(ETFile->FileNameNew)->data,FileName )==TRUE )
        {
            ET_Add_File_Name_To_List(ETFile,FileName);
            undo_added |= TRUE;
        }else
        {
            ET_Free_File_Name_Item(FileName);
        }
    }

    /*
     * Detect changes in tag data and generate the tag undo list
     */
    if (FileTag)
    {
        if ( ETFile->FileTag && ET_Detect_Changes_Of_File_Tag( (File_Tag *)(ETFile->FileTag)->data,FileTag )==TRUE )
        {
            ET_Add_File_Tag_To_List(ETFile,FileTag);
            undo_added |= TRUE;
        }else
        {
            ET_Free_File_Tag_Item(FileTag);
        }
    }

    /*
     * Generate main undo (file history of modifications)
     */
    if (undo_added)
        ET_Add_File_To_History_List(ETFile);

    return TRUE; //undo_added;
}


/*
 * Compares two File_Name items and returns TRUE if there aren't the same.
 */
gboolean ET_Detect_Changes_Of_File_Name (File_Name *FileName1, File_Name *FileName2)
{
    if ( (!FileName1 && !FileName2)
      || (!FileName1->value && !FileName2->value) )
        return FALSE;

    if ( ( FileName1 && !FileName2)
      || (!FileName1 &&  FileName2)
      || ( FileName1->value && !FileName2->value)
      || (!FileName1->value &&  FileName2->value) )
        return TRUE;

    /* Filename changed ? */
    if ( strcmp(g_basename(FileName1->value),g_basename(FileName2->value))!=0 )
        return TRUE;
    else
        return FALSE; /* No changes */
}

/*
 * Compares two File_Tag items and returns TRUE if there aren't the same.
 * Notes:
 *  - if field is '' or NULL => will be removed
 */
gboolean ET_Detect_Changes_Of_File_Tag (File_Tag *FileTag1, File_Tag *FileTag2)
{
    if ( !FileTag1 && !FileTag2 )
        return FALSE;

    if ( ( FileTag1 && !FileTag2)
      || (!FileTag1 &&  FileTag2) )
        return TRUE;

    /* Title */
    if ( FileTag1->title && !FileTag2->title && strlen(FileTag1->title)>0 ) return TRUE;
    if (!FileTag1->title &&  FileTag2->title && strlen(FileTag2->title)>0 ) return TRUE;
    if ( FileTag1->title &&  FileTag2->title && strcmp(FileTag1->title,FileTag2->title)!=0 ) return TRUE;

    /* Artist */
    if ( FileTag1->artist && !FileTag2->artist && strlen(FileTag1->artist)>0 ) return TRUE;
    if (!FileTag1->artist &&  FileTag2->artist && strlen(FileTag2->artist)>0 ) return TRUE;
    if ( FileTag1->artist &&  FileTag2->artist && strcmp(FileTag1->artist,FileTag2->artist)!=0 ) return TRUE;

    /* Album */
    if ( FileTag1->album && !FileTag2->album && strlen(FileTag1->album)>0 ) return TRUE;
    if (!FileTag1->album &&  FileTag2->album && strlen(FileTag2->album)>0 ) return TRUE;
    if ( FileTag1->album &&  FileTag2->album && strcmp(FileTag1->album,FileTag2->album)!=0 ) return TRUE;

    /* Year */
    if ( FileTag1->year && !FileTag2->year && strlen(FileTag1->year)>0 ) return TRUE;
    if (!FileTag1->year &&  FileTag2->year && strlen(FileTag2->year)>0 ) return TRUE;
    if ( FileTag1->year &&  FileTag2->year && strcmp(FileTag1->year,FileTag2->year)!=0 ) return TRUE;

    /* Track */
    if ( FileTag1->track && !FileTag2->track && strlen(FileTag1->track)>0 ) return TRUE;
    if (!FileTag1->track &&  FileTag2->track && strlen(FileTag2->track)>0 ) return TRUE;
    if ( FileTag1->track &&  FileTag2->track && strcmp(FileTag1->track,FileTag2->track)!=0 ) return TRUE;

    /* Track Total */
    if ( FileTag1->track_total && !FileTag2->track_total && strlen(FileTag1->track_total)>0 ) return TRUE;
    if (!FileTag1->track_total &&  FileTag2->track_total && strlen(FileTag2->track_total)>0 ) return TRUE;
    if ( FileTag1->track_total &&  FileTag2->track_total && strcmp(FileTag1->track_total,FileTag2->track_total)!=0 ) return TRUE;

    /* Genre */
    if ( FileTag1->genre && !FileTag2->genre && strlen(FileTag1->genre)>0 ) return TRUE;
    if (!FileTag1->genre &&  FileTag2->genre && strlen(FileTag2->genre)>0 ) return TRUE;
    if ( FileTag1->genre &&  FileTag2->genre && strcmp(FileTag1->genre,FileTag2->genre)!=0 ) return TRUE;

    /* Comment */
    if ( FileTag1->comment && !FileTag2->comment && strlen(FileTag1->comment)>0 ) return TRUE;
    if (!FileTag1->comment &&  FileTag2->comment && strlen(FileTag2->comment)>0 ) return TRUE;
    if ( FileTag1->comment &&  FileTag2->comment && strcmp(FileTag1->comment,FileTag2->comment)!=0 ) return TRUE;

    return FALSE; /* No changes */
}


/*
 * Add a FileName item to the history list of ETFile
 */
gboolean ET_Add_File_Name_To_List (ET_File *ETFile, File_Name *FileName)
{
    GList *cut_list = NULL;

    if (!ETFile || !FileName)
        return FALSE;

    /* How it works : Cut the FileNameList list after the current item,
     * and appends it to the FileNameListBak list for saving the data.
     * Then appends the new item to the FileNameList list */
    if (ETFile->FileNameList)
    {
        cut_list = ETFile->FileNameNew->next; // Cut after the current item...
        ETFile->FileNameNew->next = NULL;
    }
    if (cut_list)
        cut_list->prev = NULL;

    /* Add the new item to the list */
    ETFile->FileNameList = g_list_append(ETFile->FileNameList,FileName);
    /* Set the current item to use */
    ETFile->FileNameNew  = g_list_last(ETFile->FileNameList);
    /* Backup list */
    /* FIX ME! Keep only the saved item */
    ETFile->FileNameListBak = g_list_concat(ETFile->FileNameListBak,cut_list);

    return TRUE;
}

/*
 * Add a FileTag item to the history list of ETFile
 */
gboolean ET_Add_File_Tag_To_List (ET_File *ETFile, File_Tag *FileTag)
{
    GList *cut_list = NULL;

    if (!ETFile || !FileTag)
        return FALSE;
    
    if (ETFile->FileTag)
    {
        cut_list = ETFile->FileTag->next; // Cut after the current item...
        ETFile->FileTag->next = NULL;
    }
    if (cut_list)
        cut_list->prev = NULL;

    /* Add the new item to the list */
    ETFile->FileTagList = g_list_append(ETFile->FileTagList,FileTag);
    /* Set the current item to use */
    ETFile->FileTag     = g_list_last(ETFile->FileTagList);
    /* Backup list */
    ETFile->FileTagListBak = g_list_concat(ETFile->FileTagListBak,cut_list);

    return TRUE;
}

/*
 * Add a ETFile item to the main undo list of files
 */
gboolean ET_Add_File_To_History_List (ET_File *ETFile)
{
    ET_History_File *ETHistoryFile;

    if (!ETFile) return FALSE;

    ETHistoryFile = g_malloc0(sizeof(ET_History_File));
    ETHistoryFile->ETFile = ETFile;

    /* The undo list must contains one item before the 'first undo' data */
    if (!ETHistoryFileList)
        ETHistoryFileList = g_list_append(ETHistoryFileList,g_malloc0(sizeof(ET_History_File)));

    /* Add the item to the list (cut end of list from the current element) */
    ETHistoryFileList = g_list_append(ETHistoryFileList,ETHistoryFile);
    ETHistoryFileList = g_list_last(ETHistoryFileList);

    return TRUE;
}


/*
 * Applies one undo to the ETFile data (returns previous data).
 * Returns TRUE if an undo had been applied.
 */
gboolean ET_Undo_File_Data (ET_File *ETFile)
{
    gboolean has_filename_undo_data = FALSE;
    gboolean has_filetag_undo_data  = FALSE;
    gulong   filename_key, filetag_key, undo_key;
    
    if (!ETFile)
        return FALSE;

    /* Find the valid key */
    if (ETFile->FileNameNew->prev && ETFile->FileNameNew->data)
        filename_key = ((File_Name *)ETFile->FileNameNew->data)->key;
    else
        filename_key = 0;
    if (ETFile->FileTag->prev && ETFile->FileTag->data)
        filetag_key = ((File_Tag *)ETFile->FileTag->data)->key;
    else
        filetag_key = 0;
    // The key to use
    undo_key = MAX(filename_key,filetag_key);

    /* Undo filename */
    if (ETFile->FileNameNew->prev && ETFile->FileNameNew->data
    && (undo_key==((File_Name *)ETFile->FileNameNew->data)->key))
    {
        ETFile->FileNameNew = ETFile->FileNameNew->prev;
        has_filename_undo_data = TRUE; // To indicate that an undo has been applied
    }

    /* Undo tag data */
    if (ETFile->FileTag->prev && ETFile->FileTag->data
    && (undo_key==((File_Tag *)ETFile->FileTag->data)->key))
    {
        ETFile->FileTag = ETFile->FileTag->prev;
        has_filetag_undo_data  = TRUE;
    }

    return has_filename_undo_data | has_filetag_undo_data;
}


/*
 * Returns TRUE if file contains undo data (filename or tag)
 */
gboolean ET_File_Data_Has_Undo_Data (ET_File *ETFile)
{
    gboolean has_filename_undo_data = FALSE;
    gboolean has_filetag_undo_data  = FALSE;

    if (!ETFile) return FALSE;

    if (ETFile->FileNameNew && ETFile->FileNameNew->prev) has_filename_undo_data = TRUE;
    if (ETFile->FileTag && ETFile->FileTag->prev)         has_filetag_undo_data  = TRUE;

    return has_filename_undo_data | has_filetag_undo_data;
}


/*
 * Applies one redo to the ETFile data. Returns TRUE if a redo had been applied.
 */
gboolean ET_Redo_File_Data (ET_File *ETFile)
{
    gboolean has_filename_redo_data = FALSE;
    gboolean has_filetag_redo_data  = FALSE;
    gulong   filename_key, filetag_key, undo_key;
    
    if (!ETFile)
        return FALSE;

    /* Find the valid key */
    if (ETFile->FileNameNew->next && ETFile->FileNameNew->next->data)
        filename_key = ((File_Name *)ETFile->FileNameNew->next->data)->key;
    else
        filename_key = (gulong)~0; // To have the max value for gulong
    if (ETFile->FileTag->next && ETFile->FileTag->next->data)
        filetag_key = ((File_Tag *)ETFile->FileTag->next->data)->key;
    else
        filetag_key = (gulong)~0; // To have the max value for gulong
    // The key to use
    undo_key = MIN(filename_key,filetag_key);

    /* Redo filename */
    if (ETFile->FileNameNew->next && ETFile->FileNameNew->next->data
    && (undo_key==((File_Name *)ETFile->FileNameNew->next->data)->key))
    {
        ETFile->FileNameNew = ETFile->FileNameNew->next;
        has_filename_redo_data = TRUE; // To indicate that a redo has been applied
    }

    /* Redo tag data */
    if (ETFile->FileTag->next && ETFile->FileTag->next->data
    && (undo_key==((File_Tag *)ETFile->FileTag->next->data)->key))
    {
        ETFile->FileTag = ETFile->FileTag->next;
        has_filetag_redo_data  = TRUE;
    }

    return has_filename_redo_data | has_filetag_redo_data;
}


/*
 * Returns TRUE if file contains redo data (filename or tag)
 */
gboolean ET_File_Data_Has_Redo_Data (ET_File *ETFile)
{
    gboolean has_filename_redo_data = FALSE;
    gboolean has_filetag_redo_data  = FALSE;

    if (!ETFile) return FALSE;

    if (ETFile->FileNameNew && ETFile->FileNameNew->next) has_filename_redo_data = TRUE;
    if (ETFile->FileTag && ETFile->FileTag->next)         has_filetag_redo_data  = TRUE;

    return has_filename_redo_data | has_filetag_redo_data;
}


/*
 * Execute one 'undo' in the main undo list
 */
gboolean ET_Undo_History_File_Data (void)
{
    ET_File *ETFile;
    ET_History_File *ETHistoryFile;
    
    if (!ETHistoryFileList || !ET_History_File_List_Has_Undo_Data()) return FALSE;

    ETHistoryFile = (ET_History_File *)ETHistoryFileList->data;
    ET_File_List_Nth_By_Key(ETHistoryFile->ETFile->ETFileKey);
    ETFile = (ET_File *)ETFileList->data;
    ET_Undo_File_Data(ETFile);
    
    if (ETHistoryFileList->prev)
        ETHistoryFileList = ETHistoryFileList->prev;
    return TRUE;
}


/*
 * Returns TRUE if undo file list contains undo data
 */
gboolean ET_History_File_List_Has_Undo_Data (void)
{
    if (ETHistoryFileList && ETHistoryFileList->prev)
        return TRUE;
    else
        return FALSE;
}


/*
 * Execute one 'redo' in the main undo list
 */
gboolean ET_Redo_History_File_Data (void)
{
    ET_File *ETFile;
    ET_History_File *ETHistoryFile;

    if (!ETHistoryFileList || !ET_History_File_List_Has_Redo_Data()) return FALSE;

    ETHistoryFile = (ET_History_File *)ETHistoryFileList->next->data;
    ET_File_List_Nth_By_Key(ETHistoryFile->ETFile->ETFileKey);
    ETFile = (ET_File *)ETFileList->data;
    ET_Redo_File_Data(ETFile);
 
    if (ETHistoryFileList->next)
        ETHistoryFileList = ETHistoryFileList->next;
    return TRUE;
}


/*
 * Returns TRUE if undo file list contains redo data
 */
gboolean ET_History_File_List_Has_Redo_Data (void)
{
    if (ETHistoryFileList && ETHistoryFileList->next)
        return TRUE;
    else
        return FALSE;
}




/**********************
 * Checking functions *
 **********************/


/*
 * Ckecks if the current files had been changed but not saved.
 * Returns TRUE if the file has been saved.
 * Returns FALSE if some changes haven't been saved.
 */
gboolean ET_Check_If_File_Is_Saved (ET_File *ETFile)
{
    File_Tag  *FileTag     = NULL;
    File_Name *FileNameNew = NULL;

    if (!ETFile) return TRUE;
    
    if (ETFile->FileTag)     FileTag = ETFile->FileTag->data;
    if (ETFile->FileNameNew) FileNameNew = ETFile->FileNameNew->data;
    
    // Check if the tag has been changed
    if ( FileTag && FileTag->saved != TRUE )
        return FALSE;

    // Check if name of file has been changed
    if ( FileNameNew && FileNameNew->saved != TRUE )
        return FALSE;

    // No changes
    return TRUE;
}


/*
 * Ckecks if some files, in the list, had been changed but not saved.
 * Returns TRUE if all files have been saved.
 * Returns FALSE if some changes haven't been saved.
 */
gboolean ET_Check_If_All_Files_Are_Saved (void)
{
    /* Check if some file haven't been saved, if didn't nochange=0 */
    if (!ETFileList)
    {
        return TRUE;
    }else
    {
        GList *tmplist = g_list_first(ETFileList);
        while (tmplist)
        {
            if ( ET_Check_If_File_Is_Saved((ET_File *)tmplist->data) == FALSE )
                return FALSE;

            if (!tmplist->next) break;
            tmplist = g_list_next(tmplist);
        }
        return TRUE;
    }
}




/*******************
 * Extra functions *
 *******************/

/*
 * Set to TRUE the value of 'FileTag->saved' for the File_Tag item passed in parameter.
 * And set ALL other values of the list to FALSE.
 */
void Set_Saved_Value_Of_File_Tag (File_Tag *FileTag, gboolean saved)
{
    if (FileTag) FileTag->saved = saved;
}
void ET_Mark_File_Tag_As_Saved (ET_File *ETFile)
{
    File_Tag *FileTag;
    GList *FileTagList;

    FileTag     = (File_Tag *)ETFile->FileTag->data;    // The current FileTag, to set to TRUE
    FileTagList = ((ET_File *)ETFileList->data)->FileTagList;
    g_list_foreach(FileTagList,(GFunc)Set_Saved_Value_Of_File_Tag,FALSE); // All other FileTag set to FALSE
    FileTag->saved = TRUE; // The current FileTag set to TRUE
}


/*
 * Set to TRUE the value of 'FileName->saved' for the File_Name item passed in parameter.
 * And set ALL other values of the list to FALSE.
 */
void Set_Saved_Value_Of_File_Name (File_Name *FileName, gboolean saved)
{
    if (FileName) FileName->saved = saved;
}
void ET_Mark_File_Name_As_Saved (ET_File *ETFile)
{
    File_Name *FileNameNew;
    GList *FileNameList;

    FileNameNew  = (File_Name *)ETFile->FileNameNew->data;    // The current FileName, to set to TRUE
    FileNameList = ((ET_File *)ETFileList->data)->FileNameList;
    g_list_foreach(FileNameList,(GFunc)Set_Saved_Value_Of_File_Tag,FALSE);
    FileNameNew->saved = TRUE;
}


/*
 * Currently, it's a way by default to fill file size into ET_File_Info structure
 */
gboolean ET_Read_File_Info (gchar *filename, ET_File_Info *ETFileInfo)
{
    FILE *file;

    if (!filename || !ETFileInfo)
        return FALSE;

    if ( (file=fopen(filename,"r"))==NULL )
    {
        g_print(_("ERROR while opening file: '%s' (%s).\n\a"),filename,g_strerror(errno));
        return FALSE;
    }
    fclose(file);

    ETFileInfo->version    = 0;
    ETFileInfo->bitrate    = 0;
    ETFileInfo->samplerate = 0;
    ETFileInfo->mode       = 0;
    ETFileInfo->size       = Get_File_Size(filename);
    ETFileInfo->duration   = 0;

    ETFileList_TotalSize     += ETFileInfo->size;
    ETFileList_TotalDuration += ETFileInfo->duration;

    return TRUE;
}


/*
 * Make a copy of the ETFIle item passed in parameter. The copy
 * is generated as if it was saved from the UI.
 */
/*ET_File * ET_Generate_File_Data_Internal (ET_File *ETFile)
{
    if (!ETFile) return NULL;

}*/
    
    
/*
 * Delete the corresponding file. Return TRUE if deleted.
 */
gboolean ET_Delete_File (ET_File *ETFile)
{
    gchar *cur_filename;
    
    if (!ETFile) return FALSE;

    // Filename of the file to delete
    cur_filename = ((File_Name *)(ETFile->FileNameCur)->data)->value;
    if (Delete_File(cur_filename)==TRUE)
    {
        // Now, we delete the file in the list
        GList *list = g_list_first(ETFileList);
        while(list)
        {
            if ( (ET_File *)list->data == ETFile )
            {
                ET_File_List_Remove(list);
                break;
            }
            list = list->next;
        }
        return TRUE;
    }else
    {
        return FALSE;
    }
}


/*
 * Used to replace the illegal characters in the filename (it doesn't contain the path).
 */
gboolean ET_File_Name_Convert_Character (gchar *filename)
{
    gchar *character;

    if (!filename)
        return FALSE;

    // Convert automatically '/' to '-'.
    while ( (character=strchr(filename,'/'))!=NULL )
        *character = '-';

    // Convert other illegal characters on FAT32/16 filesystems and ISO9660 and Joliet (CD-ROM filesystems)
    if (REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME)
    {
        while ( (character=strchr(filename,'\\'))!=NULL )
            *character = ',';
        while ( (character=strchr(filename,':'))!=NULL )
            *character = '-';
        while ( (character=strchr(filename,';'))!=NULL )
            *character = '-';
        while ( (character=strchr(filename,'*'))!=NULL )
            *character = '+';
        while ( (character=strchr(filename,'?'))!=NULL )
            *character = '_';
        while ( (character=strchr(filename,'\"'))!=NULL )
            *character = '\'';
        while ( (character=strchr(filename,'<'))!=NULL )
            *character = '(';
        while ( (character=strchr(filename,'>'))!=NULL )
            *character = ')';
        while ( (character=strchr(filename,'|'))!=NULL )
            *character = '-';
    }
    return TRUE;
}



/***********************
 * Debugging functions *
 ***********************/

/*
 * Functions for DEBUGGING ONLY
 * ET_Print_File_List => show list of filename
 * Parameters: file = __FILE__
 *             line = __LINE__
 */
void ET_Debug_Print_File_List (gchar *file, gint line, gchar *function)
{
    gint efl_item = 1;
    gint fnl_item = 1;
    gint ftl_item = 1;
    gint etfilelist_length;
    gint filenamelist_length;
    gint filetaglist_length;
    GList *etfilelist;
    GList *filenamelist;
    GList *filetaglist;


    g_print("\n#### File list from %s:%d - start ####\n",file,line);
    g_print("#### Function : %s ####\n",function);
    etfilelist = g_list_first(ETFileList);
    etfilelist_length = g_list_length(etfilelist);
    while (etfilelist)
    {
        g_print("\n####################\n");
        g_print("### item %3d/%3d ###\n",efl_item,etfilelist_length);
        g_print("####################\n");
        g_print("# file_cur : '%s'\n",((File_Name *)((ET_File *)etfilelist->data)->FileNameCur->data)->value);
        g_print("# file_new : '%s'\n",((File_Name *)((ET_File *)etfilelist->data)->FileNameNew->data)->value);

        filenamelist = g_list_first( ((ET_File *)etfilelist->data)->FileNameList );
        filenamelist_length = g_list_length(filenamelist);
        fnl_item = 1;
        while (filenamelist)
        {
            g_print("  ### sub item filename : %3d/%3d ### %s\n",fnl_item,filenamelist_length,
                                                      (filenamelist==((ET_File *)etfilelist->data)->FileNameNew)?"!! CURRENT !!":"");
            g_print("  # key     : '%lu'\n",((File_Name *)filenamelist->data)->key);
            g_print("  # filename: '%s'\n",((File_Name *)filenamelist->data)->value);

            if (!filenamelist->next) break;
            filenamelist = g_list_next(filenamelist);
            fnl_item++;
        }

        g_print("\n");

        filetaglist = g_list_first( ((ET_File *)etfilelist->data)->FileTagList );
        filetaglist_length = g_list_length(filetaglist);
        ftl_item = 1;
        while (filetaglist)
        {
            g_print("  ### sub item tag : %3d/%3d ### %s\n",ftl_item,filetaglist_length,
                                                      (filetaglist==((ET_File *)etfilelist->data)->FileTag)?"!! CURRENT !!":"");
            g_print("  # key     : '%lu'\n",((File_Tag *)filetaglist->data)->key);
            g_print("  # title   : '%s'\n",((File_Tag *)filetaglist->data)->title);
            g_print("  # artist  : '%s'\n",((File_Tag *)filetaglist->data)->artist);
            g_print("  # album   : '%s'\n",((File_Tag *)filetaglist->data)->album);
            g_print("  # year    : '%s'\n",((File_Tag *)filetaglist->data)->year);
            g_print("  # track   : '%s'\n",((File_Tag *)filetaglist->data)->track);
            g_print("  # t._total: '%s'\n",((File_Tag *)filetaglist->data)->track_total);
            g_print("  # genre   : '%s'\n",((File_Tag *)filetaglist->data)->genre);
            g_print("  # comment : '%s'\n",((File_Tag *)filetaglist->data)->comment);

            if (!filetaglist->next) break;
            filetaglist = g_list_next(filetaglist);
            ftl_item++;
        }

        if (!etfilelist->next) break;
        etfilelist = g_list_next(etfilelist);
        efl_item++;
    }
    g_print("#### Undo list from %s:%d - end   ####\n",file,line);
}


