/* cddb.c - 2000/09/15 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000-2002  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
// Patch OpenBSD from Jim Geovedi
#include <netinet/in.h>
#include <arpa/inet.h>
// End patch
#include <netdb.h>
#include <errno.h>

#include "cddb.h"
#include "easytag.h"
#include "et_core.h"
#include "browser.h"
#include "scan.h"
#include "misc.h"
#include "setting.h"
#include "id3_tag.h"
#include "setting.h"
#include "msgbox.h"
#include "i18n.h"

#define CDDB_ANSWER_LINE_SIZE 3072-1


/****************
 * Declarations *
 ****************/
GList *CddbAlbumList = NULL;

GtkWidget *CddbSearchStringEntry = NULL;
GtkWidget *CddbAlbumCList = NULL;
GtkWidget *CddbTrackCList = NULL;
GtkWidget *CddbApplyButton = NULL;
GtkWidget *CddbSearchButton = NULL;
GtkWidget *CddbSearchStringInResultEntry;
GtkWidget *CddbStatusBar;
guint      CddbStatusBarContext;

GtkWidget *CddbStopSearchButton;
GtkWidget *CddbSearchStringInResultNextButton;
GtkWidget *CddbSearchStringInResultPrevButton;

gboolean   CddbStopSearch = FALSE;


/**************
 * Prototypes *
 **************/
void Cddb_Destroy_Window   (void);
void Cddb_Window_Key_Press (GtkWidget *window, GdkEvent *event);
void Cddb_Show_Album_Info  (GtkCList *clist, gint row, gint column);
void Cddb_Run_Search       (void);

gboolean Cddb_Free_Album_List       (void);
gboolean Cddb_Free_Track_Album_List (GList *track_list);

gint Cddb_Open_Connection  (gchar *host, gint port);
void Cddb_Close_Connection (gint socket_id);
gint Cddb_Read_Line        (gint socket_id, gchar *cddb_out);
gint Cddb_Read_Http_Header (gint socket_id, gchar *cddb_out);
gint Cddb_Read_Cddb_Header (gint socket_id, gchar *cddb_out);

gboolean Cddb_Search_Album_List_From_String (void);
gboolean Cddb_Get_Album_Tracks_List_CB     (GtkCList *clist, gint row, gint column);
gboolean Cddb_Get_Album_Tracks_List        (GtkCList *clist, gint row, gint column);

void     Cddb_Load_Album_List              (void);
void     Cddb_Load_Track_Album_List        (GList *track_list);
gboolean Cddb_Set_Track_Infos_To_File_List (void);

void Cddb_Search_In_All_Fields_Check_Button_Toggled     (void);
void Cddb_Search_In_All_Categories_Check_Button_Toggled (void);
void Cddb_Set_To_All_Fields_Check_Button_Toggled        (void);
void Cddb_Stop_Search                                   (void);
void Cddb_Search_String_In_Result                       (GtkWidget *entry, GtkButton *button);

void   Cddb_Set_Apply_Button_Sensivity     (void);
void   Cddb_Set_Search_Button_Sensivity    (void);
void   Cddb_Show_Categories_Button_Toggled (void) ;
gchar *Cddb_Generate_Request_String_With_Fields_And_Categories_Options (void);
void   Cddb_Album_List_Set_Row_Visible     (GtkCList *clist, gint row);

void Cddb_Track_List_Row_Selected (GtkCList *clist, gint row, gint column);
GtkWidget *Create_Cddb_Track_Clist_Popup_Menu (GtkCList *clist);
void Cddb_Track_Clist_Select_All_Lines   (void);
void Cddb_Track_Clist_Unselect_All_Lines (void);
void Cddb_Track_Clist_Invert_Selection   (void);


/*************
 * Functions *
 *************/

void Init_CddbWindow (void)
{
    CddbWindow = (GtkWidget *)NULL;
}

/*
 * The window to connect to the cd data base.
 */
#include "../pixmaps/stop.xpm"
#include "../pixmaps/up.xpm"
#include "../pixmaps/down.xpm"
void Open_Cddb_Window (void)
{
    GtkWidget *MainVBox, *VBox, *vbox, *hbox;
    GtkWidget *Frame;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkWidget *ScrollWindow;
    GtkWidget *PopupMenu;
    GtkTooltips *Tips;
    GList *History_List;
    //gint mw_x, mw_y;
    gchar *CddbAlbumCList_Titles[] = { N_("Artist / Album"), N_("Category")};
    gchar *CddbTrackCList_Titles[] = { "#", N_("Track Name"), N_("Time")};


    if (CddbWindow != NULL) 
    {
        gdk_window_raise(CddbWindow->window);
        return;
    }
    
    Tips = gtk_tooltips_new_1();

    CddbWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(CddbWindow),_("CDDB Search"));
    //gtk_window_set_transient_for(GTK_WINDOW(CddbWindow),GTK_WINDOW(MainWindow));
    gtk_window_set_policy(GTK_WINDOW(CddbWindow),FALSE,TRUE,TRUE);
    if (SET_CDDB_WINDOW_SIZE)
        gtk_window_set_default_size(GTK_WINDOW(CddbWindow),CDDB_WINDOW_WIDTH,CDDB_WINDOW_HEIGHT);
    gtk_signal_connect(GTK_OBJECT(CddbWindow),"destroy",        (GtkSignalFunc)Cddb_Destroy_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbWindow),"delete_event",   (GtkSignalFunc)Cddb_Destroy_Window,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbWindow),"key_press_event",(GtkSignalFunc)Cddb_Window_Key_Press,NULL);

    MainVBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(CddbWindow),MainVBox);
    gtk_container_border_width(GTK_CONTAINER(MainVBox),1);

    Frame = gtk_frame_new(NULL);
    gtk_box_pack_start(GTK_BOX(MainVBox),Frame,TRUE,TRUE,0);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),2);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),2);


    /*
     * Words to search
     */
    hbox = gtk_hbox_new(FALSE,4);
    gtk_box_pack_start(GTK_BOX(VBox),hbox,FALSE,FALSE,0);

    Label = gtk_label_new(_("Words :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1.0,0.5);
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    CddbSearchStringEntry = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(CddbSearchStringEntry));
    gtk_combo_set_value_in_list(GTK_COMBO(CddbSearchStringEntry),FALSE,FALSE);
    gtk_combo_set_case_sensitive(GTK_COMBO(CddbSearchStringEntry),TRUE);
    gtk_widget_set_usize(GTK_WIDGET(CddbSearchStringEntry),220,-1);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSearchStringEntry,FALSE,TRUE,0);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(CddbSearchStringEntry)->entry,_("Enter the words to "
        "search (separated by a space or '+')"),NULL);
    // History List
    History_List = Load_Cddb_Search_String_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(CddbSearchStringEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(CddbSearchStringEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(CddbSearchStringEntry)->entry)),"activate",
        (GtkSignalFunc)Add_To_Combo_Box_History,GTK_OBJECT(CddbSearchStringEntry));
    gtk_signal_connect(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(CddbSearchStringEntry)->entry)),"activate",
        (GtkSignalFunc)Cddb_Search_Album_List_From_String,NULL);
    gtk_entry_set_text_1(CddbSearchStringEntry,"");

    CddbStopSearchButton = Create_Button_With_Icon_And_Label(stop_xpm,NULL);
    gtk_box_pack_start(GTK_BOX(hbox),CddbStopSearchButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(CddbStopSearchButton),GTK_RELIEF_NONE);
    gtk_widget_set_sensitive(GTK_WIDGET(CddbStopSearchButton),FALSE);
    gtk_signal_connect(GTK_OBJECT(CddbStopSearchButton),"clicked",(GtkSignalFunc)Cddb_Stop_Search,NULL);

    // Button to run the search
    CddbSearchButton = Create_Button_With_Pixmap(BUTTON_SEARCH);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSearchButton,FALSE,FALSE,0);
    GTK_WIDGET_SET_FLAGS(CddbSearchButton,GTK_CAN_DEFAULT);
    gtk_widget_grab_default(CddbSearchButton);
    gtk_signal_connect(GTK_OBJECT(CddbSearchButton),"clicked",(GtkSignalFunc)Cddb_Search_Album_List_From_String,NULL);
    gtk_signal_connect_object(GTK_OBJECT(CddbSearchButton),"clicked",(GtkSignalFunc)Add_To_Combo_Box_History,GTK_OBJECT(CddbSearchStringEntry));
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(CddbSearchStringEntry)->entry),"changed",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);

    // Button to quit
    Button = Create_Button_With_Pixmap(BUTTON_CLOSE);
    gtk_box_pack_end(GTK_BOX(hbox),Button,FALSE,FALSE,0);
    GTK_WIDGET_SET_FLAGS(Button,GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Cddb_Destroy_Window,NULL);


    /*
     * Search options
     */
    Frame = gtk_frame_new(_("Search In :"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,TRUE,0);

    Table = gtk_table_new(7,4,FALSE);
    gtk_container_add(GTK_CONTAINER(Frame),Table);
    gtk_table_set_row_spacings(GTK_TABLE(Table),1);
    gtk_table_set_col_spacings(GTK_TABLE(Table),1);

    CddbSearchInAllFields      = gtk_check_button_new_with_label(_("All Fields"));
    Separator                  = gtk_vseparator_new();
    CddbSearchInArtistField    = gtk_check_button_new_with_label(_("Artist"));
    CddbSearchInTitleField     = gtk_check_button_new_with_label(_("Title"));
    CddbSearchInTrackNameField = gtk_check_button_new_with_label(_("Track Name"));
    CddbSearchInOtherField     = gtk_check_button_new_with_label(_("Other"));
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInAllFields,     0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),Separator,                 1,2,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInArtistField,   2,3,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInTitleField,    3,4,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInTrackNameField,4,5,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInOtherField,    5,6,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInAllFields),     CDDB_SEARCH_IN_ALL_FIELDS);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInArtistField),   CDDB_SEARCH_IN_ARTIST_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInTitleField),    CDDB_SEARCH_IN_TITLE_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInTrackNameField),CDDB_SEARCH_IN_TRACK_NAME_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInOtherField),    CDDB_SEARCH_IN_OTHER_FIELD);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInAllFields),     "toggled",(GtkSignalFunc)Cddb_Search_In_All_Fields_Check_Button_Toggled,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInAllFields),     "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInArtistField),   "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInTitleField),    "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInTrackNameField),"toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInOtherField),    "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);

    CddbSeparatorH = gtk_hseparator_new();
    gtk_table_attach(GTK_TABLE(Table),CddbSeparatorH,0,7,1,2,GTK_FILL,GTK_FILL,0,0);

    CddbSearchInAllCategories      = gtk_check_button_new_with_label(_("All Categories"));
    CddbSeparatorV                 = gtk_vseparator_new();
    CddbSearchInBluesCategory      = gtk_check_button_new_with_label(_("Blues"));
    CddbSearchInClassicalCategory  = gtk_check_button_new_with_label(_("Classical"));
    CddbSearchInCountryCategory    = gtk_check_button_new_with_label(_("Country"));
    CddbSearchInFolkCategory       = gtk_check_button_new_with_label(_("Folk"));
    CddbSearchInJazzCategory       = gtk_check_button_new_with_label(_("Jazz"));
    CddbSearchInMiscCategory       = gtk_check_button_new_with_label(_("Misc"));
    CddbSearchInNewageCategory     = gtk_check_button_new_with_label(_("Newage"));
    CddbSearchInReggaeCategory     = gtk_check_button_new_with_label(_("Reggae"));
    CddbSearchInRockCategory       = gtk_check_button_new_with_label(_("Rock"));
    CddbSearchInSoundtrackCategory = gtk_check_button_new_with_label(_("Soundtrack"));
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInAllCategories,     0,1,2,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSeparatorV,                1,2,2,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInBluesCategory,     2,3,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInClassicalCategory, 3,4,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInCountryCategory,   4,5,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInFolkCategory,      5,6,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInJazzCategory,      6,7,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInMiscCategory,      2,3,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInNewageCategory,    3,4,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInReggaeCategory,    4,5,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInRockCategory,      5,6,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_table_attach(GTK_TABLE(Table),CddbSearchInSoundtrackCategory,6,7,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_label_set_line_wrap(GTK_LABEL(GTK_BIN(CddbSearchInAllCategories)->child),TRUE); // Wrap label of the check button.
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInAllCategories),     CDDB_SEARCH_IN_ALL_CATEGORIES);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInBluesCategory),     CDDB_SEARCH_IN_BLUES_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInClassicalCategory), CDDB_SEARCH_IN_CLASSICAL_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInCountryCategory),   CDDB_SEARCH_IN_COUNTRY_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInFolkCategory),      CDDB_SEARCH_IN_FOLK_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInJazzCategory),      CDDB_SEARCH_IN_JAZZ_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInMiscCategory),      CDDB_SEARCH_IN_MISC_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInNewageCategory),    CDDB_SEARCH_IN_NEWAGE_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInReggaeCategory),    CDDB_SEARCH_IN_REGGAE_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInRockCategory),      CDDB_SEARCH_IN_ROCK_CATEGORY);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSearchInSoundtrackCategory),CDDB_SEARCH_IN_SOUNDTRACK_CATEGORY);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInAllCategories),     "toggled",(GtkSignalFunc)Cddb_Search_In_All_Categories_Check_Button_Toggled,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInAllCategories),     "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInBluesCategory),     "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInClassicalCategory), "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInCountryCategory),   "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInFolkCategory),      "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInJazzCategory),      "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInMiscCategory),      "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInNewageCategory),    "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInReggaeCategory),    "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInRockCategory),      "toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSearchInSoundtrackCategory),"toggled",(GtkSignalFunc)Cddb_Set_Search_Button_Sensivity,NULL);
    gtk_tooltips_set_tip(Tips,CddbSearchInRockCategory,_("included : funk, soul, rap, pop, industrial, metal, etc."),NULL);
    gtk_tooltips_set_tip(Tips,CddbSearchInSoundtrackCategory,_("movies, shows"),NULL);
    gtk_tooltips_set_tip(Tips,CddbSearchInMiscCategory,_("others that do not fit in the above categories"),NULL);

    // Button to display/hide the categories
    CddbShowCategoriesButton = gtk_toggle_button_new_with_label(_(" Categories "));
    gtk_table_attach(GTK_TABLE(Table),CddbShowCategoriesButton,6,7,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbShowCategoriesButton),CDDB_SHOW_CATEGORIES);
    gtk_signal_connect(GTK_OBJECT(CddbShowCategoriesButton),"toggled",(GtkSignalFunc)Cddb_Show_Categories_Button_Toggled,NULL);


    /*
     * Results command
     */
    Frame = gtk_frame_new(_("Results :"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,TRUE,0);

    hbox = gtk_hbox_new(FALSE,4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox),2);
    gtk_container_add(GTK_CONTAINER(Frame),hbox);

    Label = gtk_label_new(_("Search :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1.0,0.5);
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    CddbSearchStringInResultEntry = gtk_combo_new();
    gtk_combo_disable_activate(GTK_COMBO(CddbSearchStringInResultEntry));
    gtk_combo_set_value_in_list(GTK_COMBO(CddbSearchStringInResultEntry),FALSE,FALSE);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSearchStringInResultEntry,FALSE,FALSE,0);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(CddbSearchStringInResultEntry)->entry)),"activate",
        (GtkSignalFunc)Cddb_Search_String_In_Result,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_tooltips_set_tip(Tips,GTK_COMBO(CddbSearchStringInResultEntry)->entry,_("Enter the words to "
        "search in the list below"),NULL);
    // History List
    History_List = Load_Cddb_Search_String_In_Result_List();
    if (History_List)
        gtk_combo_set_popdown_strings(GTK_COMBO(CddbSearchStringInResultEntry),History_List);
    gtk_object_set_data(GTK_OBJECT(CddbSearchStringInResultEntry),"History",History_List);
    gtk_signal_connect_object(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(CddbSearchStringInResultEntry)->entry)),"activate",
        (GtkSignalFunc)Add_To_Combo_Box_History,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_entry_set_text_1(CddbSearchStringInResultEntry,"");

    CddbSearchStringInResultNextButton = Create_Button_With_Icon_And_Label(down_xpm,NULL);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSearchStringInResultNextButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(CddbSearchStringInResultNextButton),GTK_RELIEF_NONE);
    gtk_signal_connect_object(GTK_OBJECT(CddbSearchStringInResultNextButton),"clicked",(GtkSignalFunc)Cddb_Search_String_In_Result,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_signal_connect_object(GTK_OBJECT(CddbSearchStringInResultNextButton),"clicked",(GtkSignalFunc)Add_To_Combo_Box_History,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_tooltips_set_tip(Tips,CddbSearchStringInResultNextButton,_("Search Next"),NULL);

    CddbSearchStringInResultPrevButton = Create_Button_With_Icon_And_Label(up_xpm,NULL);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSearchStringInResultPrevButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(CddbSearchStringInResultPrevButton),GTK_RELIEF_NONE);
    gtk_signal_connect_object(GTK_OBJECT(CddbSearchStringInResultPrevButton),"clicked",(GtkSignalFunc)Cddb_Search_String_In_Result,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_signal_connect_object(GTK_OBJECT(CddbSearchStringInResultPrevButton),"clicked",(GtkSignalFunc)Add_To_Combo_Box_History,GTK_OBJECT(CddbSearchStringInResultEntry));
    gtk_tooltips_set_tip(Tips,CddbSearchStringInResultPrevButton,_("Search Previous"),NULL);


    /*
     * Result of research
     */
    CddbWindowHPaned = gtk_hpaned_new();
    gtk_box_pack_start(GTK_BOX(VBox),CddbWindowHPaned,TRUE,TRUE,0);
    gtk_paned_set_handle_size(GTK_PANED(CddbWindowHPaned),8);
    gtk_paned_set_gutter_size(GTK_PANED(CddbWindowHPaned),6);                       
    if (SET_CDDB_PANE_HANDLE_POSITION)
        gtk_paned_set_position(GTK_PANED(CddbWindowHPaned),CDDB_PANE_HANDLE_POSITION);

    // List of albums
    ScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindow),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(ScrollWindow),-1,100);
    gtk_paned_pack1(GTK_PANED(CddbWindowHPaned),ScrollWindow,TRUE,FALSE);
    CddbAlbumCList = gtk_clist_new_with_titles(2,CddbAlbumCList_Titles);
    gtk_container_add(GTK_CONTAINER(ScrollWindow),CddbAlbumCList);
    gtk_clist_set_selection_mode(GTK_CLIST(CddbAlbumCList),GTK_SELECTION_SINGLE);
    gtk_clist_column_titles_passive(GTK_CLIST(CddbAlbumCList));
    gtk_signal_connect(GTK_OBJECT(CddbAlbumCList),"select_row",(GtkSignalFunc)Cddb_Album_List_Set_Row_Visible,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbAlbumCList),"select_row",(GtkSignalFunc)Cddb_Show_Album_Info,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbAlbumCList),"select_row",(GtkSignalFunc)Cddb_Get_Album_Tracks_List_CB,NULL);

    gtk_clist_set_column_auto_resize(GTK_CLIST(CddbAlbumCList),0,TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(CddbAlbumCList),1,TRUE);


    // List of tracks
    ScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindow),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(ScrollWindow),-1,100);
    gtk_paned_pack2(GTK_PANED(CddbWindowHPaned),ScrollWindow,TRUE,FALSE);
    CddbTrackCList = gtk_clist_new_with_titles(3,CddbTrackCList_Titles);
    gtk_container_add(GTK_CONTAINER(ScrollWindow),CddbTrackCList);
    gtk_clist_set_selection_mode(GTK_CLIST(CddbTrackCList),GTK_SELECTION_EXTENDED);
    gtk_clist_column_titles_passive(GTK_CLIST(CddbTrackCList));
    //gtk_clist_set_reorderable(GTK_CLIST(CddbTrackCList),FALSE);
    gtk_signal_connect(GTK_OBJECT(CddbTrackCList),"select_row",(GtkSignalFunc)Cddb_Track_List_Row_Selected,NULL);
    gtk_tooltips_set_tip(Tips,CddbTrackCList,_("Select lines to 'apply' to your files list. "
        "Note that all lines will be processed if no line is selected."),NULL);

    gtk_clist_set_column_auto_resize(GTK_CLIST(CddbTrackCList),0,TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(CddbTrackCList),1,TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(CddbTrackCList),2,TRUE);
    gtk_clist_set_column_justification(GTK_CLIST(CddbTrackCList),0,GTK_JUSTIFY_RIGHT);
    gtk_clist_set_column_justification(GTK_CLIST(CddbTrackCList),2,GTK_JUSTIFY_RIGHT);

    // Create Popup Menu on CddbTrackCList
    PopupMenu = Create_Cddb_Track_Clist_Popup_Menu(GTK_CLIST (CddbTrackCList));


    /*
     * Apply results to fields...
     */
    Frame = gtk_frame_new(_("Set Into :"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,TRUE,0);

    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);

    CddbSetToAllFields  = gtk_check_button_new_with_label(_("All"));
    Separator           = gtk_vseparator_new();
    CddbSetToFileName   = gtk_check_button_new_with_label(_("File Name"));
    CddbSetToTitle      = gtk_check_button_new_with_label(_("Title"));
    CddbSetToArtist     = gtk_check_button_new_with_label(_("Artist"));
    CddbSetToAlbum      = gtk_check_button_new_with_label(_("Album"));
    CddbSetToYear       = gtk_check_button_new_with_label(_("Year"));
    CddbSetToTrack      = gtk_check_button_new_with_label(_("Track #"));
    CddbSetToTrackTotal = gtk_check_button_new_with_label(_("# Tracks"));
    CddbSetToGenre      = gtk_check_button_new_with_label(_("Genre"));
    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToAllFields, FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),Separator,          FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToFileName,  FALSE,FALSE,2);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToTitle,     FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToArtist,    FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToAlbum,     FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToYear,      FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToTrack,     FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToTrackTotal,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),CddbSetToGenre,     FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToAllFields), CDDB_SET_TO_ALL_FIELDS);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToTitle),     CDDB_SET_TO_TITLE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToArtist),    CDDB_SET_TO_ARTIST);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToAlbum),     CDDB_SET_TO_ALBUM);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToYear),      CDDB_SET_TO_YEAR);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToTrack),     CDDB_SET_TO_TRACK);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToTrackTotal),CDDB_SET_TO_TRACK_TOTAL);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToGenre),     CDDB_SET_TO_GENRE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbSetToFileName),  CDDB_SET_TO_FILE_NAME);
    gtk_signal_connect(GTK_OBJECT(CddbSetToAllFields), "toggled",(GtkSignalFunc)Cddb_Set_To_All_Fields_Check_Button_Toggled,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToAllFields), "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToTitle),     "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToArtist),    "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToAlbum),     "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToYear),      "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToTrack),     "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToTrackTotal),"toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToGenre),     "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);
    gtk_signal_connect(GTK_OBJECT(CddbSetToFileName),  "toggled",(GtkSignalFunc)Cddb_Set_Apply_Button_Sensivity,NULL);

    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);

    // Check box to run the scanner
    CddbRunScanner = gtk_check_button_new_with_label(_("Run the current scanner for each file"));
    gtk_box_pack_start(GTK_BOX(hbox),CddbRunScanner,FALSE,TRUE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbRunScanner),CDDB_RUN_SCANNER);
    gtk_tooltips_set_tip(Tips,CddbRunScanner,_("When activating this option, after loading the "
        "fields, the current selected scanner will be ran (the scanner window must be opened)."),NULL);

    // Button to apply
    CddbApplyButton = Create_Button_With_Pixmap(BUTTON_APPLY);
    gtk_box_pack_end(GTK_BOX(hbox),CddbApplyButton,FALSE,FALSE,2);
    gtk_signal_connect(GTK_OBJECT(CddbApplyButton),"clicked",(GtkSignalFunc)Cddb_Set_Track_Infos_To_File_List,NULL);


    /*
     * Status bar
     */
    CddbStatusBar = gtk_statusbar_new();
    gtk_box_pack_start(GTK_BOX(MainVBox),CddbStatusBar,FALSE,TRUE,0);
    gtk_widget_set_usize(CddbStatusBar,300,-1);
    CddbStatusBarContext = gtk_statusbar_get_context_id(GTK_STATUSBAR(CddbStatusBar),"Messages");
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Ready to search..."));


    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(CddbSearchStringEntry)->entry),"changed",NULL);
    gtk_signal_emit_by_name(GTK_OBJECT(CddbSearchInAllFields),"toggled");
    gtk_signal_emit_by_name(GTK_OBJECT(CddbSearchInAllCategories),"toggled");
    gtk_signal_emit_by_name(GTK_OBJECT(CddbSetToAllFields),"toggled");
    CddbStopSearch = FALSE;
    gtk_widget_show_all(CddbWindow);
    gtk_widget_set_usize(GTK_WIDGET(CddbSearchInAllFields),CddbSearchInAllCategories->allocation.width,-1);
    gtk_signal_emit_by_name(GTK_OBJECT(CddbShowCategoriesButton),"toggled");

    //gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    //gtk_window_reposition(GTK_WINDOW(CddbWindow),
    //    mw_x + (MainWindow->allocation.width/2)  - (CddbWindow->allocation.width)/2,
    //    mw_y + (MainWindow->allocation.height/2) - (CddbWindow->allocation.height)/2);
}
void Cddb_Destroy_Window (void)
{
    CddbStopSearch = TRUE;
    if (CddbWindow)
    {
        GList *list;
        
        /* Save combobox history lists before exit */
        list = gtk_object_get_data(GTK_OBJECT(CddbSearchStringEntry),"History");
        Save_Cddb_Search_String_List(list);
        list = gtk_object_get_data(GTK_OBJECT(CddbSearchStringInResultEntry),"History");
        Save_Cddb_Search_String_In_Result_List(list);

        // FIX ME : This causes problem with memory !!
        Cddb_Free_Album_List();

        gtk_widget_destroy(CddbWindow);
        CddbWindow = NULL;
        CddbSearchStringEntry = NULL;
        CddbAlbumCList = NULL;
        CddbTrackCList = NULL;
        CddbApplyButton = NULL;
        CddbSearchButton = NULL;
    }
}
void Cddb_Window_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                Cddb_Destroy_Window();
                break;
        }
    }
}

void Cddb_Search_In_All_Fields_Check_Button_Toggled (void) 
{
    if (CddbSearchInAllFields)
    {
        gtk_widget_set_sensitive(CddbSearchInArtistField,   !GTK_TOGGLE_BUTTON(CddbSearchInAllFields)->active);
        gtk_widget_set_sensitive(CddbSearchInTitleField,    !GTK_TOGGLE_BUTTON(CddbSearchInAllFields)->active);
        gtk_widget_set_sensitive(CddbSearchInTrackNameField,!GTK_TOGGLE_BUTTON(CddbSearchInAllFields)->active);
        gtk_widget_set_sensitive(CddbSearchInOtherField,    !GTK_TOGGLE_BUTTON(CddbSearchInAllFields)->active);
    }
}
void Cddb_Show_Categories_Button_Toggled (void) 
{
    if (CddbShowCategoriesButton)
    {
        if (GTK_TOGGLE_BUTTON(CddbShowCategoriesButton)->active)
        {
            gtk_widget_show(CddbSeparatorH);
            gtk_widget_show(CddbSearchInAllCategories);
            gtk_widget_show(CddbSeparatorV);
            gtk_widget_show(CddbSearchInBluesCategory);
            gtk_widget_show(CddbSearchInClassicalCategory);
            gtk_widget_show(CddbSearchInCountryCategory);
            gtk_widget_show(CddbSearchInFolkCategory);
            gtk_widget_show(CddbSearchInJazzCategory);
            gtk_widget_show(CddbSearchInMiscCategory);
            gtk_widget_show(CddbSearchInNewageCategory);
            gtk_widget_show(CddbSearchInReggaeCategory);
            gtk_widget_show(CddbSearchInRockCategory);
            gtk_widget_show(CddbSearchInSoundtrackCategory);
        }else
        {
            gtk_widget_hide(CddbSeparatorH);
            gtk_widget_hide(CddbSearchInAllCategories);
            gtk_widget_hide(CddbSeparatorV);
            gtk_widget_hide(CddbSearchInBluesCategory);
            gtk_widget_hide(CddbSearchInClassicalCategory);
            gtk_widget_hide(CddbSearchInCountryCategory);
            gtk_widget_hide(CddbSearchInFolkCategory);
            gtk_widget_hide(CddbSearchInJazzCategory);
            gtk_widget_hide(CddbSearchInMiscCategory);
            gtk_widget_hide(CddbSearchInNewageCategory);
            gtk_widget_hide(CddbSearchInReggaeCategory);
            gtk_widget_hide(CddbSearchInRockCategory);
            gtk_widget_hide(CddbSearchInSoundtrackCategory);
        }
        // Force the window to be redrawed
        gtk_widget_queue_resize(CddbWindow);
    }
}
void Cddb_Search_In_All_Categories_Check_Button_Toggled (void) 
{
    if (CddbSearchInAllCategories)
    {
        gtk_widget_set_sensitive(CddbSearchInBluesCategory,     !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInClassicalCategory, !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInCountryCategory,   !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInFolkCategory,      !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInJazzCategory,      !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInMiscCategory,      !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInNewageCategory,    !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInReggaeCategory,    !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInRockCategory,      !GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
        gtk_widget_set_sensitive(CddbSearchInSoundtrackCategory,!GTK_TOGGLE_BUTTON(CddbSearchInAllCategories)->active);
    }
}
void Cddb_Set_To_All_Fields_Check_Button_Toggled (void) 
{
    if (CddbSetToAllFields)
    {
        gtk_widget_set_sensitive(CddbSetToTitle,     !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToArtist,    !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToAlbum,     !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToYear,      !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToTrack,     !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToTrackTotal,!GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToGenre,     !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
        gtk_widget_set_sensitive(CddbSetToFileName,  !GTK_TOGGLE_BUTTON(CddbSetToAllFields)->active);
    }
}
void Cddb_Set_Apply_Button_Sensivity (void)
{
    gboolean cddbsettoallfields, cddbsettotitle, cddbsettoartist, cddbsettoalbum,
             cddbsettoyear, cddbsettotrack, cddbsettotracktotal, cddbsettogenre, cddbsettofilename;

    // Tag fields
    cddbsettoallfields  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAllFields));
    cddbsettotitle      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTitle));
    cddbsettoartist     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToArtist));
    cddbsettoalbum      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAlbum));
    cddbsettoyear       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToYear));
    cddbsettotrack      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrack));
    cddbsettotracktotal = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrackTotal));
    cddbsettogenre      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToGenre));
    cddbsettofilename   = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToFileName));

    if ( CddbApplyButton && CddbTrackCList && GTK_CLIST(CddbTrackCList)->rows 
    && (cddbsettoallfields || cddbsettotitle || cddbsettoartist     || cddbsettoalbum || cddbsettoyear
        || cddbsettotrack  || cddbsettotracktotal || cddbsettogenre || cddbsettofilename) )
        gtk_widget_set_sensitive(GTK_WIDGET(CddbApplyButton),TRUE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(CddbApplyButton),FALSE);
}
void Cddb_Set_Search_Button_Sensivity (void)
{
    gboolean cddbinallfields, cddbinartistfield, cddbintitlefield, cddbintracknamefield, cddbinotherfield;
    gboolean cddbinallcategories, cddbinbluescategory, cddbinclassicalcategory, cddbincountrycategory,
             cddbinfolkcategory, cddbinjazzcategory, cddbinmisccategory, cddbinnewagecategory,
             cddbinreggaecategory, cddbinrockcategory, cddbinsoundtrackcategory;

    // Fields
    cddbinallfields      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllFields));
    cddbinartistfield    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInArtistField));
    cddbintitlefield     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTitleField));
    cddbintracknamefield = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTrackNameField));
    cddbinotherfield     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInOtherField));
    // Categories
    cddbinallcategories      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllCategories));
    cddbinbluescategory      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInBluesCategory));
    cddbinclassicalcategory  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInClassicalCategory));
    cddbincountrycategory    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInCountryCategory));
    cddbinfolkcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInFolkCategory));
    cddbinjazzcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInJazzCategory));
    cddbinmisccategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInMiscCategory));
    cddbinnewagecategory     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInNewageCategory));
    cddbinreggaecategory     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInReggaeCategory));
    cddbinrockcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInRockCategory));
    cddbinsoundtrackcategory = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInSoundtrackCategory));

    if ( CddbSearchButton && CddbSearchStringEntry && strlen(gtk_entry_get_text_1 (CddbSearchStringEntry))>0
    && (cddbinallfields     || cddbinartistfield   || cddbintitlefield        || cddbintracknamefield || cddbinotherfield)
    && (cddbinallcategories || cddbinbluescategory || cddbinclassicalcategory || cddbincountrycategory
        || cddbinfolkcategory   || cddbinjazzcategory || cddbinmisccategory || cddbinnewagecategory
        || cddbinreggaecategory || cddbinrockcategory || cddbinsoundtrackcategory) )
        gtk_widget_set_sensitive(GTK_WIDGET(CddbSearchButton),TRUE);
    else
        gtk_widget_set_sensitive(GTK_WIDGET(CddbSearchButton),FALSE);
}

void Cddb_Stop_Search (void)
{
    CddbStopSearch = TRUE;
}

void Cddb_Search_String_In_Result (GtkWidget *entry, GtkButton *button)
{
    gchar *string;
    gchar  buffer[256];
    gchar *text;
    gint   row, i;
    GList *selection;
    gboolean found = FALSE;

    if (!entry || !button)
        return;

    string = gtk_entry_get_text_1(entry);
    if (!string || strlen(string)==0)
        return;
    g_strdown(string);

    if (button != GTK_BUTTON(CddbSearchStringInResultPrevButton))
    {
        // Search in the album list (from top to bottom)
        if ( (selection=GTK_CLIST(CddbAlbumCList)->selection) != NULL )
        {
            row = (gint)selection->data + 1;
            gtk_clist_unselect_row(GTK_CLIST(CddbAlbumCList),row-1,0);
        }else
        {
            row = 0;
        }

        for (i=row;i<GTK_CLIST(CddbAlbumCList)->rows;i++)
        {
            gtk_clist_get_text(GTK_CLIST(CddbAlbumCList),i,0,&text);
            strncpy(buffer,text,256);
            g_strdown(buffer);
            if (buffer && strstr(buffer,string)!=NULL)
            {
                gtk_clist_select_row(GTK_CLIST(CddbAlbumCList),i,0);
                found = TRUE;
                break;
            }
        }
        if (found == FALSE)
            for (i=0;i<row;i++)
            {
                gtk_clist_get_text(GTK_CLIST(CddbAlbumCList),i,0,&text);
                strncpy(buffer,text,256);
                g_strdown(buffer);
                if (buffer && strstr(buffer,string)!=NULL)
                {
                    gtk_clist_select_row(GTK_CLIST(CddbAlbumCList),i,0);
                    break;
                }
            }
    }else
    {
        // Search in the album list (from bottom to top)
        if ( (selection=GTK_CLIST(CddbAlbumCList)->selection) != NULL )
        {
            row = (gint)selection->data - 1;
            gtk_clist_unselect_row(GTK_CLIST(CddbAlbumCList),row+1,0);
        }else
        {
            row = GTK_CLIST(CddbAlbumCList)->rows - 1;
        }

        for (i=row;i>=0;i--)
        {
            gtk_clist_get_text(GTK_CLIST(CddbAlbumCList),i,0,&text);
            strncpy(buffer,text,256);
            g_strdown(buffer);
            if (buffer && strstr(buffer,string)!=NULL)
            {
                gtk_clist_select_row(GTK_CLIST(CddbAlbumCList),i,0);
                found = TRUE;
                break;
            }
        }
        if (found == FALSE)
            for (i=GTK_CLIST(CddbAlbumCList)->rows-1;i>=row;i--)
            {
                gtk_clist_get_text(GTK_CLIST(CddbAlbumCList),i,0,&text);
                strncpy(buffer,text,256);
                g_strdown(buffer);
                if (buffer && strstr(buffer,string)!=NULL)
                {
                    gtk_clist_select_row(GTK_CLIST(CddbAlbumCList),i,0);
                    break;
                }
            }
    }
}


/*
 * Set a row visible in the clist (by scrolling the list) if the row is hidden
 */
void Cddb_Album_List_Set_Row_Visible (GtkCList *clist, gint row)
{
    GtkVisibility visible = gtk_clist_row_is_visible(clist,row);
    switch (visible)
    {
        case GTK_VISIBILITY_NONE:
        case GTK_VISIBILITY_PARTIAL:
            gtk_clist_moveto(clist,row,0,0.5,0.0);
            break;
        case GTK_VISIBILITY_FULL:
            break;
    }
}


/*
 * Show collected infos of the album in the status bar
 */
void Cddb_Show_Album_Info (GtkCList *clist, gint row, gint column)
{
    CddbAlbum *cddbalbum = NULL;
    gchar *msg;
    
    cddbalbum = gtk_clist_get_row_data(GTK_CLIST(clist),row);
    if (!cddbalbum) return;
    msg = g_strdup_printf(_("Album: '%s', artist: '%s', length: '%s', year: '%s', genre: '%s', ID: '%s'"),
        cddbalbum->album?cddbalbum->album:"",
        cddbalbum->artist?cddbalbum->artist:"",
        Convert_Duration((gulong)cddbalbum->duration),
        cddbalbum->year?cddbalbum->year:"",
        cddbalbum->genre?cddbalbum->genre:"",
        cddbalbum->id?cddbalbum->id:"");
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
    g_free(msg);
}


/*
 * Select the corresponding file into the main file list
 */
void Cddb_Track_List_Row_Selected (GtkCList *clist, gint row, gint column)
{
    Action_Select_Nth_File_By_Position(row+1);
}


/*
 * Cddb_Popup_Menu_Handler : displays the corresponding menu
 * Create_Browser_Tree_Popup_Menu: Create a popup menu for the tree browser
 * Create_Browser_List_Popup_Menu: Create a popup menu for the list of files of browser
 */
gboolean Cddb_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}
GtkWidget *Create_Cddb_Track_Clist_Popup_Menu (GtkCList *clist)
{
    GtkWidget *CddbPopupMenu;
    GtkWidget *MenuItem;


    CddbPopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(clist),"button_press_event",
        (GtkSignalFunc)Cddb_Popup_Menu_Handler,GTK_OBJECT(CddbPopupMenu));

    MenuItem = gtk_menu_item_new_with_label(_("Select all lines"));
    gtk_menu_append(GTK_MENU(CddbPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Cddb_Track_Clist_Select_All_Lines,NULL);

    MenuItem = gtk_menu_item_new_with_label(_("Unselect all lines"));
    gtk_menu_append(GTK_MENU(CddbPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Cddb_Track_Clist_Unselect_All_Lines,NULL);

    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(CddbPopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Invert selection"));
    gtk_menu_append(GTK_MENU(CddbPopupMenu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Cddb_Track_Clist_Invert_Selection,NULL);

    gtk_widget_show_all(CddbPopupMenu);
    return CddbPopupMenu;
}

void Cddb_Track_Clist_Select_All_Lines (void)
{
    // Must block the select signal to avoid selecting all files (one by one) in the main files list
    gtk_signal_handler_block_by_func(GTK_OBJECT(CddbTrackCList),GTK_SIGNAL_FUNC(Cddb_Track_List_Row_Selected),NULL);
    gtk_clist_select_all(GTK_CLIST(CddbTrackCList));
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(CddbTrackCList),GTK_SIGNAL_FUNC(Cddb_Track_List_Row_Selected),NULL);
}
void Cddb_Track_Clist_Unselect_All_Lines (void)
{
    gtk_clist_unselect_all(GTK_CLIST(CddbTrackCList));
}
void Cddb_Track_Clist_Invert_Selection (void)
{
    gint row;
    GList *list;
    
    // Must block the select signal to avoid selecting all files (one by one) in the main files list
    gtk_signal_handler_block_by_func(GTK_OBJECT(CddbTrackCList),GTK_SIGNAL_FUNC(Cddb_Track_List_Row_Selected),NULL);
    list = g_list_first(GTK_CLIST(CddbTrackCList)->row_list);
    for (row=0;row<GTK_CLIST(CddbTrackCList)->rows;row++)
    {
        if (GTK_CLIST_ROW(list)->state == GTK_STATE_SELECTED)
            gtk_clist_unselect_row(GTK_CLIST(CddbTrackCList),row,1);
        else
            gtk_clist_select_row(GTK_CLIST(CddbTrackCList),row,1);
        list = list->next;
    }
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(CddbTrackCList),GTK_SIGNAL_FUNC(Cddb_Track_List_Row_Selected),NULL);
}


/*
 * Open a connection to "server_name" and retun the socket_id
 * On error, returns 0.
 */
gint Cddb_Open_Connection (gchar *host, gint port)
{
    gint               socket_id = 0;
    struct hostent    *hostent;
    struct sockaddr_in sockaddr;
    gint               optval = 1;
    gchar *msg;


    if (!host || port <= 0)
        return 0;

    msg = g_strdup_printf(_("Resolving host '%s' ..."),host);
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
    g_free(msg);
    while (gtk_events_pending()) gtk_main_iteration();

    if ( (hostent=gethostbyname((const gchar*)host)) == NULL )
    {
        msg = g_strdup_printf(_("Can't resolve host '%s' (%s)!"),host,g_strerror(errno));
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        g_print("%s\n",msg); 
        g_free(msg);
        return 0;
    }

    bzero((void *)&sockaddr,sizeof(sockaddr)); // Initialize with zero
    memcpy(&sockaddr.sin_addr.s_addr,*(hostent->h_addr_list),sizeof(sockaddr.sin_addr.s_addr));
    sockaddr.sin_family = AF_INET;
    sockaddr.sin_port   = htons(port);

    // Create socket
    if( (socket_id = socket(AF_INET,SOCK_STREAM,0)) < 0 )
    {
        msg = g_strdup_printf(_("Can't create a new socket (%s)!"),g_strerror(errno));
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        g_print("%s\n",msg); 
        g_free(msg);
        return 0;
    }

    // FIX ME : must catch SIGPIPE?
    if ( setsockopt(socket_id,SOL_SOCKET,SO_KEEPALIVE,(gchar *)&optval,sizeof(optval)) < 0 )
    {
        g_print("Can't set option of the new created socket!\n"); 
    }

    // Open connection to the server
    msg = g_strdup_printf(_("Connecting to host '%s', port '%d' ..."),host,port);
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
    g_free(msg);
    while (gtk_events_pending()) gtk_main_iteration();
    if ( connect(socket_id,(struct sockaddr *)&sockaddr,sizeof(struct sockaddr_in)) < 0 )
    {
        msg = g_strdup_printf(_("Can't connect to host '%s' (%s)!"),host,g_strerror(errno));
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        g_print("%s\n",msg); 
        g_free(msg);
        return 0;
    }
    msg = g_strdup_printf(_("Connected to host '%s'"),host);
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
    g_free(msg);
    while (gtk_events_pending()) gtk_main_iteration();

    return socket_id;
}


/*
 * Close the connection correcponding to the socket_id
 */
void Cddb_Close_Connection (gint socket_id)
{
    shutdown(socket_id,SHUT_RDWR);
    close(socket_id);
    socket_id = 0;
    CddbStopSearch = FALSE;
}


/*
 * Read one line (of the connection) into cddb_out. And return the number of bytes read.
 * If bytes_read=0 => no more data.
 *
 * Server answser is formated like this :
 *
 * HTTP/1.1 200 OK\r\n                              }
 * Server: Apache/1.3.19 (Unix) PHP/4.0.4pl1\r\n    } "Header"
 * Connection: close\r\n                            }
 * \r\n
 * <html>\n                                         }
 * [...]                                            } "Body"
 */
gint Cddb_Read_Line (gint socket_id, gchar *cddb_out)
{
    gint bytes_returned = 0;
    gint bytes_read = 0;
    gint i;

    for (i=0;i<CDDB_ANSWER_LINE_SIZE;i++)
    {
        bytes_read = read(socket_id,&cddb_out[i],1);
        if (bytes_read < 0)
            return -1; // Error!

        bytes_returned += bytes_read;
        // Cases : end of the line, or no more data
        if (bytes_read<=0 || cddb_out[i]=='\r' || cddb_out[i]=='\n')
        {
            if (cddb_out[i]!='\r') // Else the next time we'll read the '\n' in the header
            {
                cddb_out[i] = 0;
                break;
            }else
            {
                cddb_out[i] = 0;
            }
        }
    }
    return bytes_returned;
}


/*
 * Read HTTP header data : from "HTTP/1.1 200 OK" to the blank line
 */

gint Cddb_Read_Http_Header (gint socket_id, gchar *cddb_out)
{
    gint bytes_returned = 0;
    gint bytes_read;
    
    if ( (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) < 0 )
        return -1; // Error!
    // First line must be "HTTP/1.1 200 OK"
    if ( strncmp("HTTP",cddb_out,4)!=0 || strstr(cddb_out,"200 OK")==NULL )
        return -1;
    // Read until end of the http header
    while ( (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) > 0 && strlen(cddb_out) > 0 )
        bytes_returned += bytes_read;

    return bytes_returned;
}


/*
 * Read CDDB header data when requesting a file (cmd=cddb+read+<album genre>+<discid>)
 * Must be read after the HTTP header.
 * Take one line like this : "210 rock 780dfe09 CD database entry follows (until terminating `.')"
 */

gint Cddb_Read_Cddb_Header (gint socket_id, gchar *cddb_out)
{
    gint bytes_read;
    
    if ( (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) < 0 )
        return -1; // Error!

    // Some request receive some strange data at the beginning (2 or 3 characters)... so we read one line more...
    if ( !cddb_out || strlen(cddb_out) < 10 )
        if ( (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) < 0 )
            return -1; // Error!
    
    // Read the line
    // 200 - exact match
    // 210 - multiple exact matches
    // 211 - inexact match
    if ( cddb_out[0] != '2' )
        return -1;

    return bytes_read;
}



/*
 * Free the CddbAlbumList
 */
gboolean Cddb_Free_Album_List (void)
{
    if (!CddbAlbumList) return FALSE;
    
    CddbAlbumList = g_list_last(CddbAlbumList);
    while (CddbAlbumList)
    {
        CddbAlbum *cddbalbum = CddbAlbumList->data;
        if (cddbalbum)
        {
            if (cddbalbum->artist_album) g_free(cddbalbum->artist_album);
            if (cddbalbum->category)     g_free(cddbalbum->category);
            if (cddbalbum->id)           g_free(cddbalbum->id);
            if (cddbalbum->track_list)   Cddb_Free_Track_Album_List(cddbalbum->track_list);
            if (cddbalbum->artist)       g_free(cddbalbum->artist);
            if (cddbalbum->album)        g_free(cddbalbum->album);
            if (cddbalbum->genre)        g_free(cddbalbum->genre);
            if (cddbalbum->year)         g_free(cddbalbum->year);
            g_free(cddbalbum);
            cddbalbum = (CddbAlbum *)NULL;
        }
        if (!CddbAlbumList->prev) break;
        CddbAlbumList = CddbAlbumList->prev;
    }
    g_list_free(CddbAlbumList);
    CddbAlbumList = (GList *)NULL;
    return TRUE;
}
gboolean Cddb_Free_Track_Album_List (GList *track_list)
{
    GList *CddbTrackAlbumList;

    if (!track_list) return FALSE;
    
    CddbTrackAlbumList = g_list_last(track_list);
    while (CddbTrackAlbumList)
    {
        CddbTrackAlbum *cddbtrackalbum = CddbTrackAlbumList->data;
        if (cddbtrackalbum)
        {
            if (cddbtrackalbum->track_name) g_free(cddbtrackalbum->track_name);
            g_free(cddbtrackalbum);
            cddbtrackalbum = (CddbTrackAlbum *)NULL;
        }
        if (!CddbTrackAlbumList->prev) break;
        CddbTrackAlbumList = CddbTrackAlbumList->prev;
    }
    g_list_free(CddbTrackAlbumList);
    CddbTrackAlbumList = (GList *)NULL;
    return TRUE;
}



/*
 * Load the CddbAlbumList into the corresponding CList
 */
void Cddb_Load_Album_List (void)
{
    if (CddbAlbumList && CddbAlbumCList)
    {
        GdkColor GREY = {0, 0xaa00, 0xaa00, 0xaa00};
        GList *cddbalbumlist = g_list_first(CddbAlbumList);
        GtkStyle *style = gtk_style_copy(gtk_widget_get_style(CddbAlbumCList));
        
        gtk_clist_clear(GTK_CLIST(CddbAlbumCList));
        while (cddbalbumlist)
        {
            gchar *row_text[2];
            gint   new_row;
            CddbAlbum *cddbalbum = cddbalbumlist->data;
    
            row_text[0] = cddbalbum->artist_album;
            row_text[1] = cddbalbum->category;
            
            // Load the row in the list
            new_row = gtk_clist_append(GTK_CLIST(CddbAlbumCList),row_text);
            gtk_clist_set_row_data(GTK_CLIST(CddbAlbumCList),new_row,cddbalbum);
            
            // For other versions of the same album
            if ( cddbalbum->other_version == TRUE )
            {
                style->fg[GTK_STATE_NORMAL] = GREY;
                gtk_clist_set_row_style(GTK_CLIST(CddbAlbumCList),new_row,style);
            }

            cddbalbumlist = cddbalbumlist->next;
        }
    }
}


/*
 * Load the CddbAlbumList into the corresponding CList
 */
void Cddb_Load_Track_Album_List (GList *track_list)
{
    if (track_list && CddbTrackCList)
    {
        GList *tracklist = g_list_first(track_list);

        gtk_clist_clear(GTK_CLIST(CddbTrackCList));
        while (tracklist)
        {
            gchar *row_text[3];
            gint   new_row;
            CddbTrackAlbum *cddbtrackalbum = tracklist->data;
    
            row_text[0] = g_strdup_printf("%d",cddbtrackalbum->track_number);
            row_text[1] = cddbtrackalbum->track_name;
            row_text[2] = Convert_Duration((gulong)cddbtrackalbum->duration);
            
            // Load the row in the list
            new_row = gtk_clist_append(GTK_CLIST(CddbTrackCList),row_text);
            gtk_clist_set_row_data(GTK_CLIST(CddbTrackCList),new_row,cddbtrackalbum);

            tracklist = tracklist->next;
            if (row_text[0]) g_free(row_text[0]);
            if (row_text[2]) g_free(row_text[2]);
        }
        Cddb_Set_Apply_Button_Sensivity();
    }
}


/*
 * Fields: artist, title, track, rest
 * Categories: blues, classical, country, data, folk, jazz, misc, newage, reggae, rock, soundtrack
 */
gchar *Cddb_Generate_Request_String_With_Fields_And_Categories_Options (void)
{
    gchar string[256];
    gboolean cddbinallfields, cddbinartistfield, cddbintitlefield, cddbintracknamefield, cddbinotherfield;
    gboolean cddbinallcategories, cddbinbluescategory, cddbinclassicalcategory, cddbincountrycategory,
             cddbinfolkcategory, cddbinjazzcategory, cddbinmisccategory, cddbinnewagecategory,
             cddbinreggaecategory, cddbinrockcategory, cddbinsoundtrackcategory;
    
    // Init
    string[0] = 0;

    // Fields
    cddbinallfields      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllFields));
    cddbinartistfield    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInArtistField));
    cddbintitlefield     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTitleField));
    cddbintracknamefield = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTrackNameField));
    cddbinotherfield     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInOtherField));

    if (cddbinallfields)      strncat(string,"&allfields=YES",14);
    else                      strncat(string,"&allfields=NO",13);

    if (cddbinartistfield)    strncat(string,"&fields=artist",14);
    if (cddbintitlefield)     strncat(string,"&fields=title",13);
    if (cddbintracknamefield) strncat(string,"&fields=track",13);
    if (cddbinotherfield)     strncat(string,"&fields=rest",12);
    

    // Categories (warning : there is one other CDDB catogories not used here ("data"))
    cddbinallcategories      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllCategories));
    cddbinbluescategory      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInBluesCategory));
    cddbinclassicalcategory  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInClassicalCategory));
    cddbincountrycategory    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInCountryCategory));
    cddbinfolkcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInFolkCategory));
    cddbinjazzcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInJazzCategory));
    cddbinmisccategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInMiscCategory));
    cddbinnewagecategory     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInNewageCategory));
    cddbinreggaecategory     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInReggaeCategory));
    cddbinrockcategory       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInRockCategory));
    cddbinsoundtrackcategory = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInSoundtrackCategory));

    strncat(string,"&allcats=NO",11);
    if (cddbinallcategories)
    {
        // All categories except "data"
        strncat(string,"&cats=blues&cats=classical&cats=country&cats=folk&cats=jazz"
                       "&cats=misc&cats=newage&cats=reggae&cats=rock&cats=soundtrack",119);
    }else
    {
        if (cddbinbluescategory)      strncat(string,"&cats=blues",11);
        if (cddbinclassicalcategory)  strncat(string,"&cats=classical",15);
        if (cddbincountrycategory)    strncat(string,"&cats=country",13);
        if (cddbinfolkcategory)       strncat(string,"&cats=folk",10);
        if (cddbinjazzcategory)       strncat(string,"&cats=jazz",10);
        if (cddbinmisccategory)       strncat(string,"&cats=misc",10);
        if (cddbinnewagecategory)     strncat(string,"&cats=newage",12);
        if (cddbinreggaecategory)     strncat(string,"&cats=reggae",12);
        if (cddbinrockcategory)       strncat(string,"&cats=rock",10);
        if (cddbinsoundtrackcategory) strncat(string,"&cats=soundtrack",16);
    }

    return g_strdup(string);
}




/*
 * Send request to the CD database to get the list of albums matching with a string
 */
gboolean Cddb_Search_Album_List_From_String (void)
{
    gint   socket_id;
    gchar *string = NULL;
    gchar *tmp, *tmp1;
    gchar *cddb_in;    // For the request to send
    gchar *cddb_out;   // Answer received
    gchar *cddb_out_tmp;
    gchar *msg;

    gchar *ptr_cat, *cat_str, *id_str, *art_alb_str;
    gchar *art_alb_tmp = NULL;
    gchar *end_str;
    gchar  buffer[CDDB_ANSWER_LINE_SIZE+1];
    gint   bytes_written;
    gint   bytes_read;
    gulong bytes_read_total = 0;


    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,"");

    // Get words to search...
    string = g_strdup(gtk_entry_get_text_1(CddbSearchStringEntry));
    if ( !string || strlen(string)<=0 )
        return FALSE;
    
    // Format the string of words
    Strip_String(string);
    // Remove the duplicated spaces
    while ((tmp=strstr(string,"  "))!=NULL) // Search 2 spaces
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
    // Convert spaces to '+'
    while( (tmp=strchr(string,' '))!=NULL )
        *tmp = '+';

    // Connection to the server
    if ( (socket_id=Cddb_Open_Connection(CDDB_USE_PROXY?CDDB_PROXY_NAME:"www.freedb.org",
                                         CDDB_USE_PROXY?CDDB_PROXY_PORT:80)) <= 0 )
    {
        g_free(string);
        return FALSE;
    }

    // Build request
    cddb_in = g_strdup_printf("GET http://www.freedb.org/freedb_search.php?"
                              "words=%s"
                              "%s"
                              "&grouping=none"
                              " HTTP/1.1\r\n"
                              "Host: %s:%d\r\n"
                              "User-Agent: %s %s\r\n"
                              "Connection: close\r\n"
                              "\r\n"
                              ,string
                              ,(tmp=Cddb_Generate_Request_String_With_Fields_And_Categories_Options())
                              ,CDDB_SERVER_NAME,CDDB_SERVER_PORT
                              ,APPNAME,VERSION
                              );
    g_free(string);
    g_free(tmp);

    // Send the request
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Sending request ..."));
    while (gtk_events_pending()) gtk_main_iteration();
    if ( (bytes_written=send(socket_id,cddb_in,strlen(cddb_in)+1,0)) < 0)
    {
        g_print(_("Can't send the request (%s)!\n"),g_strerror(errno)); 
        Cddb_Close_Connection(socket_id);
        g_free(cddb_in);
        return FALSE;
    }
    g_free(cddb_in);


    // Delete previous album list
    gtk_clist_clear(GTK_CLIST(CddbAlbumCList));
    gtk_clist_clear(GTK_CLIST(CddbTrackCList));
    Cddb_Free_Album_List();
    gtk_widget_set_sensitive(GTK_WIDGET(CddbStopSearchButton),TRUE);


    /*
     * Read the answer
     */
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Receiving data ..."));
    while (gtk_events_pending()) gtk_main_iteration();
    cddb_out = g_malloc0(CDDB_ANSWER_LINE_SIZE+1);
    // Parse server answer : Check returned code in the first line
    if ( !cddb_out || (bytes_read=Cddb_Read_Http_Header(socket_id,cddb_out)) <= 0 )
    {
        msg = g_strdup_printf(_("The server returned a wrong answer! (%s)"),cddb_out);
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        g_print("%s\n",msg);
        g_free(msg);
        return FALSE;
    }
    bytes_read_total = bytes_read;

    // Read other lines,and get list of matching albums
    // Composition of a line : 
    // <a href="http://www.freedb.org/freedb_search_fmt.php?cat=rock&id=8c0f0a0b">Bob Dylan / MTV Unplugged</a><br>
    cat_str     = g_strdup("http://www.freedb.org/freedb_search_fmt.php?cat=");
    id_str      = g_strdup("&id=");
    art_alb_str = g_strdup("\">");
    end_str     = g_strdup("</a>"); //"</a><br>");
    while ( !CddbStopSearch && (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) > 0 )
    {
        bytes_read_total += bytes_read;
        msg = g_strdup_printf(_("Receiving data (%s) ..."),Convert_Size_1(bytes_read_total));
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        while (gtk_events_pending()) gtk_main_iteration();
        g_free(msg);

        cddb_out_tmp = cddb_out;
        //g_print("%s\n",cddb_out);

        // We may have severals album in the same line (other version of the same album?)
        // Note : we test that the 'end' delimiter exists to avoid crashes
        while ( cddb_out != NULL && (ptr_cat=strstr(cddb_out_tmp,cat_str)) != NULL && strstr(cddb_out_tmp,end_str) != NULL )
        {
            gchar *ptr_font, *ptr_font1;
            gchar *ptr_id, *ptr_art_alb, *ptr_end;
            CddbAlbum *cddbalbum;

            cddbalbum = g_malloc0(sizeof(CddbAlbum));

            // Get album category
            cddb_out_tmp = ptr_cat + strlen(cat_str);
            strncpy(buffer,cddb_out_tmp,CDDB_ANSWER_LINE_SIZE);
            if ( (ptr_id=strstr(buffer,id_str)) != NULL )
                *ptr_id = 0;
            cddbalbum->category = g_strdup(buffer);

            // Get album ID
            cddb_out_tmp = strstr(cddb_out_tmp,id_str) + strlen(id_str);
            strncpy(buffer,cddb_out_tmp,CDDB_ANSWER_LINE_SIZE);
            if ( (ptr_art_alb=strstr(buffer,art_alb_str)) != NULL )
                *ptr_art_alb = 0;
            cddbalbum->id = g_strdup(buffer);

            // Get album and artist names.
            // Note : some names can be like this "<font size=-1>2</font>" (for other version of the same album)
            cddb_out_tmp = strstr(cddb_out_tmp,art_alb_str) + strlen(art_alb_str);
            strncpy(buffer,cddb_out_tmp,CDDB_ANSWER_LINE_SIZE);
            if ( (ptr_end=strstr(buffer,end_str)) != NULL )
                *ptr_end = 0;
            if ( (ptr_font=strstr(buffer,"</font>")) != NULL )
            {
                *ptr_font = 0;
                if ( (ptr_font1=strstr(buffer,">")) != NULL )
                {
                    cddbalbum->artist_album = g_strdup_printf("%s -> %s",ptr_font1+1,art_alb_tmp);
                    //cddbalbum->artist_album = g_strdup(ptr_font1+1);
                    cddbalbum->other_version = TRUE;
                }else
                {
                    cddbalbum->artist_album = g_strdup(buffer);
                }
            }else
            {
                cddbalbum->artist_album = g_strdup(buffer);
                art_alb_tmp = cddbalbum->artist_album;
            }

            // New position the search the next string
            cddb_out_tmp = strstr(cddb_out_tmp,end_str) + strlen(end_str);
        
            CddbAlbumList = g_list_append(CddbAlbumList,cddbalbum);
        }
    }
    g_free(cat_str); g_free(id_str); g_free(art_alb_str); g_free(end_str);
    gtk_widget_set_sensitive(GTK_WIDGET(CddbStopSearchButton),FALSE);

    // Close connection
    Cddb_Close_Connection(socket_id);

    msg = g_strdup_printf(_("Found %d matching album(s)"),g_list_length(CddbAlbumList));
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
    g_free(msg);

    // Load the albums found in the list
    Cddb_Load_Album_List();

    return TRUE;
}


/*
 * Callback when selecting a row in the Album CList.
 * We get the list of tracks of the selected album
 */
gboolean Cddb_Get_Album_Tracks_List_CB (GtkCList *clist, gint row, gint column)
{
    gint i;
    gint i_max = 5;

    /* As may be not opened the first time (The server returned a wrong answer!)
     * me try to reconnect severals times */
    for (i=1;i<=i_max;i++)
        if ( Cddb_Get_Album_Tracks_List(clist,row,column) == TRUE )
            break;
    if (i <= i_max) return TRUE;
    else            return FALSE;
}
gboolean Cddb_Get_Album_Tracks_List (GtkCList *clist, gint row, gint column)
{
    gint   socket_id;
    CddbAlbum *cddbalbum = NULL;
    GList *TrackOffsetList = NULL;
    gchar *cddb_in, *cddb_out, *msg;
    gint   bytes_written, bytes_read, bytes_read1, bytes_read_total;
    gboolean read_track_offset = FALSE;


    gtk_clist_clear(GTK_CLIST(CddbTrackCList));
    Cddb_Set_Apply_Button_Sensivity();
    cddbalbum = gtk_clist_get_row_data(GTK_CLIST(clist),row);
    if (!cddbalbum)
        return FALSE;

    // We have already the track list
    if (cddbalbum->track_list != NULL)
    {
        Cddb_Load_Track_Album_List(cddbalbum->track_list);
        return TRUE;
    }

    // Connection to the server    
    if ( (socket_id=Cddb_Open_Connection(CDDB_USE_PROXY?CDDB_PROXY_NAME:CDDB_SERVER_NAME,
                                         CDDB_USE_PROXY?CDDB_PROXY_PORT:CDDB_SERVER_PORT)) <= 0 )
        return FALSE;

    // http://www.freedb.org/~cddb/cddb.cgi?cmd...
	cddb_in = g_strdup_printf("GET http://%s%s?cmd=cddb+read+"
                              "%s+%s"
                              "&hello=noname+localhost+%s+%s"
                              "&proto=1 HTTP/1.1\r\n"
                              "Host: %s:%d\r\nConnection: close\r\n\r\n",
                              CDDB_SERVER_NAME,CDDB_SERVER_CGI_PATH,
                              cddbalbum->category,cddbalbum->id,
                              APPNAME,VERSION,
                              CDDB_SERVER_NAME,CDDB_SERVER_PORT
                              );

    // Send the request
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Sending request ..."));
    while (gtk_events_pending()) gtk_main_iteration();
    if ( (bytes_written=send(socket_id,cddb_in,strlen(cddb_in)+1,0)) < 0)
    {
        g_print(_("Can't send the request (%s)!\n"),g_strerror(errno)); 
        Cddb_Close_Connection(socket_id);
        return FALSE;
    }
    g_free(cddb_in);


    // Read the answer
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Receiving data ..."));
    while (gtk_events_pending()) gtk_main_iteration();
    cddb_out = g_malloc0(CDDB_ANSWER_LINE_SIZE+1);
    // Parse server answer : Check HTTP Header and CDDB Header
    if ( !cddb_out
         || (bytes_read =Cddb_Read_Http_Header(socket_id,cddb_out)) <= 0
         || (bytes_read1=Cddb_Read_Cddb_Header(socket_id,cddb_out)) <= 0 )
    {
        gchar *msg = g_strdup_printf(_("The server returned a wrong answer! (%s)"),cddb_out);
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        g_print("%s\n",msg);
        g_free(msg);
        return FALSE;
    }
    bytes_read_total = bytes_read + bytes_read1;

    while ( !CddbStopSearch && (bytes_read=Cddb_Read_Line(socket_id,cddb_out)) > 0 )
    {
        bytes_read_total += bytes_read;
        msg = g_strdup_printf(_("Receiving data (%s) ..."),Convert_Size_1(bytes_read_total));
        gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,msg);
        while (gtk_events_pending()) gtk_main_iteration();
        g_free(msg);

        //g_print("%s\n",cddb_out);
        if (!cddb_out) continue; // Empty line?

        // We are reading a track offset? (generates TrackOffsetList)
        if (read_track_offset)
        {
            if ( strtoul(cddb_out+1,NULL,10)>0 )
            {
                CddbTrackFrameOffset *cddbtrackframeoffset = g_malloc0(sizeof(CddbTrackFrameOffset));
                cddbtrackframeoffset->offset = strtoul(cddb_out+1,NULL,10);
                TrackOffsetList = g_list_append(TrackOffsetList,cddbtrackframeoffset);
                continue;
             }else
             {
                 read_track_offset = FALSE; // No more track offset
             }
        }
        
        if ( strncmp(cddb_out,"DTITLE=",7)==0 ) // "Artist / Album" names
        {
            gchar *alb_ptr = strstr(cddb_out," / ");
            if (alb_ptr && alb_ptr+3)
            {
                cddbalbum->album = g_strdup(alb_ptr+3);
                *alb_ptr = 0;
            }
            cddbalbum->artist = g_strdup(cddb_out+7);

        }else if ( strstr(cddb_out,"Track frame offsets")!=NULL ) // We read the Track frame offset
        {
            read_track_offset = TRUE; // The next reads are for the tracks offset

        }else if ( strstr(cddb_out,"Disc length: ")!=NULL ) // Length of album (in second)
        {
            cddbalbum->duration = atoi(strchr(cddb_out,':')+1);
            if (TrackOffsetList) // As it must be the last item, do nothing if no previous data
            {
                CddbTrackFrameOffset *cddbtrackframeoffset = g_malloc0(sizeof(CddbTrackFrameOffset));
                cddbtrackframeoffset->offset = cddbalbum->duration*75; // It's the last offset
                TrackOffsetList = g_list_append(TrackOffsetList,cddbtrackframeoffset);
            }

        }else if ( strncmp(cddb_out,"TTITLE",6)==0 ) // Track title
        {
            CddbTrackAlbum *cddbtrackalbum_last = NULL;

            CddbTrackAlbum *cddbtrackalbum = g_malloc0(sizeof(CddbTrackAlbum));
            cddbtrackalbum->cddbalbum = cddbalbum; // To find the CddbAlbum father quickly
            cddbtrackalbum->track_name = g_strdup(strchr(cddb_out,'=')+1);
            *strchr(cddb_out,'=') = 0;
            cddbtrackalbum->track_number = atoi(cddb_out+6)+1;

            // Note : titles too long take severals lines. For example :
            // TTITLE15=Bob Marley vs. Funkstar De Luxe Remix - Sun Is Shining (Radio De Lu
            // TTITLE15=xe Edit)
            // So to check it, we compare current track number with the previous one...
            if (cddbalbum->track_list)
                cddbtrackalbum_last = g_list_last(cddbalbum->track_list)->data;
            if (cddbtrackalbum_last && cddbtrackalbum_last->track_number == cddbtrackalbum->track_number)
            {
                gchar *track_name = g_strconcat(cddbtrackalbum_last->track_name,cddbtrackalbum->track_name,NULL);
                g_free(cddbtrackalbum_last->track_name);
                cddbtrackalbum_last->track_name = track_name;            

                // Frees useless allocated data previously
                g_free(cddbtrackalbum->track_name);
                g_free(cddbtrackalbum);
            }else
            {
                if (TrackOffsetList && TrackOffsetList->next)
                {
                    cddbtrackalbum->duration = ( ((CddbTrackFrameOffset *)TrackOffsetList->next->data)->offset - ((CddbTrackFrameOffset *)TrackOffsetList->data)->offset ) / 75; // Calculate time in seconds
                    TrackOffsetList = TrackOffsetList->next;
                }
                cddbalbum->track_list = g_list_append(cddbalbum->track_list,cddbtrackalbum);
            }

        }else if ( strncmp(cddb_out,"EXTD=",5)==0 ) // Extended album data
        {
            gchar *genre_ptr = strstr(cddb_out,"ID3G:");
            gchar *year_ptr  = strstr(cddb_out,"YEAR:");
            // May contains severals EXTD field it too long
            // EXTD=Techno
            // EXTD= YEAR: 1997 ID3G:  18
            // EXTD= ID3G:  17
            if (year_ptr)
                cddbalbum->year = g_strdup_printf("%d",atoi(year_ptr+5));
            if (genre_ptr)
                cddbalbum->genre = g_strdup(Id3tag_Genre_To_String(atoi(genre_ptr+5)));
        }
    }

    // Close connection
    Cddb_Close_Connection(socket_id);

    // Set forecolor of the selected row to red (light red for an other version of the album)
    // to show that we've already the track list for this album
    if ( cddbalbum->other_version == TRUE )
    {
        GdkColor LIGHT_RED = {0, 0xffff, 0x7f00, 0x7f00};
        GtkStyle *style = gtk_style_copy(gtk_widget_get_style(CddbAlbumCList));
        style->fg[GTK_STATE_NORMAL] = LIGHT_RED;
        gtk_clist_set_row_style(GTK_CLIST(CddbAlbumCList),row,style);
    }else
    {
        GdkColor RED = {0, 0xffff, 0x0000, 0x0000};
        GtkStyle *style = gtk_style_copy(gtk_widget_get_style(CddbAlbumCList));
        style->fg[GTK_STATE_NORMAL] = RED;
        gtk_clist_set_row_style(GTK_CLIST(CddbAlbumCList),row,style);
    }

    // Load the track list of the album
    gtk_statusbar_push(GTK_STATUSBAR(CddbStatusBar),CddbStatusBarContext,_("Loading album track list ..."));
    while (gtk_events_pending()) gtk_main_iteration();
    Cddb_Load_Track_Album_List(cddbalbum->track_list);
    Cddb_Show_Album_Info(clist,row,column);

    // Frees 'TrackOffsetList'
    TrackOffsetList = g_list_last(TrackOffsetList);
    while (TrackOffsetList)
    {
        if (TrackOffsetList->data)
            g_free(TrackOffsetList->data);
        if (!TrackOffsetList->prev) break;
        TrackOffsetList = TrackOffsetList->prev;
    }
    g_list_free(TrackOffsetList);
    TrackOffsetList = (GList *)NULL;

    return TRUE;
}




/*
 * Set CDDB data (from tracks list) into tags of the main file list
 */
gboolean Cddb_Set_Track_Infos_To_File_List (void)
{
    guint clist_row;
    guint list_length;
    GList *etfilelist;
    GList *list;
    gchar buffer[256];
    gboolean CddbTrackCList_Line_Selected;
    gboolean cddbsettoallfields, cddbsettotitle, cddbsettoartist, cddbsettoalbum, cddbsettoyear,
             cddbsettotrack, cddbsettotracktotal, cddbsettogenre, cddbsettofilename;


    if (!CddbTrackCList || !ETFileList) return FALSE;
    
    // Save the current displayed data
    ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);

    etfilelist = g_list_first(ETFileList);
    list_length = GTK_CLIST(CddbTrackCList)->rows;
    
    if (g_list_length(etfilelist) != list_length)
    {
        GtkWidget *msgbox;
        gchar *msg;
        gint button;

        msg = g_strdup_printf(_("The album from CDDB contains %d tracks against %d files in the "
            "list!\n\nDo you want to continue ?"),list_length,g_list_length(etfilelist));
        msgbox = msg_box_new (_("Write Tag from CDDB..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        button = msg_box_run(MSG_BOX(msgbox));

        if (button != BUTTON_YES)
            return FALSE;
    }

    cddbsettoallfields  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAllFields));
    cddbsettotitle      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTitle));
    cddbsettoartist     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToArtist));
    cddbsettoalbum      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAlbum));
    cddbsettoyear       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToYear));
    cddbsettotrack      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrack));
    cddbsettotracktotal = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrackTotal));
    cddbsettogenre      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToGenre));
    cddbsettofilename   = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToFileName));

    // Check if at least one line was selected. No line selected is equal to all lines selected.
    CddbTrackCList_Line_Selected = FALSE;
    list = g_list_first(GTK_CLIST(CddbTrackCList)->row_list);
    while (list)
    {
        if (GTK_CLIST_ROW(list)->state == GTK_STATE_SELECTED)
        {
            CddbTrackCList_Line_Selected = TRUE;
            break;
        }
        list = list->next;
    }

    etfilelist = g_list_first(ETFileList);
    for (clist_row=0;clist_row<GTK_CLIST(CddbTrackCList)->rows;clist_row++)
    {
        list = g_list_nth(GTK_CLIST(CddbTrackCList)->row_list,clist_row);
        if (CddbTrackCList_Line_Selected == FALSE || GTK_CLIST_ROW(list)->state == GTK_STATE_SELECTED)
        {
            CddbTrackAlbum *cddbtrackalbum = NULL;
            cddbtrackalbum = gtk_clist_get_row_data(GTK_CLIST(CddbTrackCList),clist_row);
            if (cddbtrackalbum && etfilelist)
            {
                ET_File   *etfile   = (ET_File *)etfilelist->data;
                File_Name *FileName = ET_File_Name_Item_New();
                File_Tag  *FileTag  = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(etfilelist->data,FileTag);

                if (cddbsettoallfields || cddbsettotitle)
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->title,cddbtrackalbum->track_name);

                if (cddbsettoallfields || cddbsettotrack)
                {
                    if (NUMBER_TRACK_FORMATED) snprintf(buffer,sizeof(buffer),"%.2d",cddbtrackalbum->track_number);
                    else                       snprintf(buffer,sizeof(buffer),"%d",  cddbtrackalbum->track_number);
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->track,buffer);
                }            

                if (cddbsettoallfields || cddbsettotracktotal)
                {
                    if (NUMBER_TRACK_FORMATED) snprintf(buffer,sizeof(buffer),"%.2d",list_length);
                    else                       snprintf(buffer,sizeof(buffer),"%d",  list_length);
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->track_total,buffer);
                }
            
                if ( (cddbsettoallfields || cddbsettoalbum) && cddbtrackalbum->cddbalbum->album)
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->album, cddbtrackalbum->cddbalbum->album);

                if ( (cddbsettoallfields || cddbsettoartist) && cddbtrackalbum->cddbalbum->artist)
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->artist,cddbtrackalbum->cddbalbum->artist);

                if ( (cddbsettoallfields || cddbsettogenre) && cddbtrackalbum->cddbalbum->genre)
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->genre, cddbtrackalbum->cddbalbum->genre);

                if ( (cddbsettoallfields || cddbsettoyear) && cddbtrackalbum->cddbalbum->year)
                    ET_Set_Field_File_Tag_Item((gint *)&FileTag->year,  cddbtrackalbum->cddbalbum->year);

                if ( (cddbsettoallfields || cddbsettofilename) )
                {
                    // Build the filename with the path
                    gchar *dirname, *filename_generated, *filename_new = NULL;
                    if (NUMBER_TRACK_FORMATED) snprintf(buffer,sizeof(buffer),"%.2d",cddbtrackalbum->track_number);
                    else                       snprintf(buffer,sizeof(buffer),"%d",  cddbtrackalbum->track_number);
                    filename_generated = g_strconcat(buffer," - ",cddbtrackalbum->track_name,NULL);
                    ET_File_Name_Convert_Character(filename_generated); // Replace invalid characters
                    dirname = g_dirname(((File_Name *)etfile->FileNameNew->data)->value);
                    if (dirname)
                    {
                        if ( strcmp(dirname,"/")==0 ) // Root directory?
                            filename_new = g_strconcat("/",filename_generated,etfile->ETFileDescription->Extension,NULL);
                        else
                            filename_new = g_strconcat(dirname,"/",filename_generated,etfile->ETFileDescription->Extension,NULL);
                        ET_Set_Field_File_Name_Item((gint *)&FileName->value,filename_new);
                    }
                    g_free(filename_generated);
                    if (filename_new) g_free(filename_new);
                }

                ET_Manage_Changes_Of_File_Data(etfile,FileName,FileTag);

                // Then run current scanner if asked...
                if (ScannerWindow && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbRunScanner)) )
                    Scan_Select_Mode_And_Run_Scanner(etfile);
            }
        }
        etfilelist = g_list_next(etfilelist);
    }
    
    Browser_List_Refresh_Whole_List();
    ET_Display_File_Data_To_UI((ET_File *)ETFileList->data);

    return TRUE;
}


