/* scan.c - 2000/06/16 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2000-2002  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "scan.h"
#include "easytag.h"
#include "prefs.h"
#include "setting.h"
#include "id3_tag.h"
#include "bar.h"
#include "misc.h"
#include "et_core.h"
#include "mylocale.h"


/****************
 * Declarations *
 ****************/
GtkWidget *DummyEntry = NULL;    /* Used to simulate a gtkentry widget for mask code '%i' */
GtkWidget *ScannerScanTagMaskCombo    = NULL;
GtkWidget *ScannerRenameFileMaskCombo = NULL;
GtkWidget *ScannerOptionMenu          = NULL;
GtkWidget *MaskStatusIconBox          = NULL;
GtkWidget *MaskStatusIconBox1         = NULL;
GtkWidget *MaskStatusIconBox2         = NULL;

GtkWidget *ScanTagFrame;
GtkWidget *RenameFileFrame;
GtkWidget *ProcessFieldsFrame;
GtkWidget *ScanTagPreviewLabel;
GtkWidget *RenameFilePreviewLabel;

GtkWidget *ProcessFileNameField;
GtkWidget *ProcessTitleField;
GtkWidget *ProcessArtistField;
GtkWidget *ProcessAlbumField;
GtkWidget *ProcessCommentField;
GtkWidget *ProcessFieldsConvertIntoSpace      = NULL;
GtkWidget *ProcessFieldsConvertSpace          = NULL;
GtkWidget *ProcessFieldsAllUppercase          = NULL;
GtkWidget *ProcessFieldsAllDowncase           = NULL;
GtkWidget *ProcessFieldsFirstLetterUppercase  = NULL;
GtkWidget *ProcessFieldsFirstLettersUppercase = NULL;
GtkWidget *ProcessFieldsRemoveSpace           = NULL;
GtkWidget *ProcessFieldsInsertSpace           = NULL;
GtkWidget *ProcessFieldsOnlyOneSpace          = NULL;

GtkWidget *LegendFrame    = NULL;
GtkWidget *LegendButton   = NULL;

GtkWidget *MaskEditorButton = NULL;
GtkWidget *MaskEditorFrame  = NULL;
GtkWidget *MaskEditorVBox;
GtkWidget *MaskEditorHBox;
GtkWidget *MaskEditorScrollWindow;
GtkWidget *MaskEditorClist;
GtkWidget *MaskEditorEntry;
GtkWidget *MaskEditorNewButton;
GtkWidget *MaskEditorCopyButton;
GtkWidget *MaskEditorAddButton;
GtkWidget *MaskEditorRemoveButton;
GtkWidget *MaskEditorUpButton;
GtkWidget *MaskEditorDownButton;
GtkWidget *MaskEditorSaveButton;

/* Lists */
GList *MasksList           = NULL;
GList *RenameFileMasksList = NULL;

/* Some predefined masks */
gchar *Scan_Masks [] = 
{
    "%a - %b/%n - %t",
    "%a_-_%b/%n_-_%t",
    "%a - %b/%n. %t",
    "%a_-_%b/%n._%t",
    "%a-%b/%n-%t",
    "%b/%n. %a - %t",
    "%b/%n._%a_-_%t",
    "%b/%n - %a - %t",
    "%b/%n_-_%a_-_%t",
    "%b/%n-%a-%t",
    "%a-%b/%n-%t",
    "%a/%b/%n. %t",
    "%a_-_%b-%n-%t-%y",
    "%a - %b/%n. %t(%c)",
    "%t",
    "Track%n",
    "Track%i %n"
};

gchar *Rename_File_Masks [] = 
{
    "%n - %a - %b - %t",
    "%n_-_%a_-_%b_-_%t",
    "%a - %b - %t",
    "%a_-_%b_-_%t",
    "%a - %b - %n - %t",
    "%a_-_%b_-_%n_-_%t",
    "%a - %t",
    "%a_-_%t",
    "%n. %a - %t",
    "%n._%a_-_%t",
    "%n - %t",
    "%n_-_%t",
    "%n. %t",
    "%n._%t",
    "Track %n"
};

gchar *Scanner_Option_Menu_Items [] = 
{
    N_("Fill Tag"),
    N_("Rename File"),
    N_("Process Fields")
};
/* Must have the same order than Scanner_Option_Menu_Items */
/*typedef enum
{
    SCANNER_FILL_TAG = 0,
    SCANNER_RENAME_FILE,
    SCANNER_PROCESS_FIELDS
} Scanner_Option_Menu_Items_Value;
*/
typedef enum
{
    LEADING_SEPARATOR = 0, /* characters before the first code */
    TRAILING_SEPARATOR,    /* characters after the last code */
    SEPARATOR,             /* item is a separator between two codes */
    FIELD,                 /* item contains text (not empty) of entry */
    EMPTY_FIELD            /* item when entry contains no text */
} Mask_Item_Type;

/*
 * Used into Rename File Scanner
 */
typedef struct _File_Mask_Item File_Mask_Item;
struct _File_Mask_Item
{
    Mask_Item_Type  type;
    gchar          *string;
};



/**************
 * Prototypes *
 **************/
void     Scan_Display_String_To_Ui (GtkWidget *wid, gchar *string);
void     ScannerWindow_Quit (void);
gboolean Scan_Check_Scan_Tag_Mask (gchar *mask);
gboolean Scan_Check_Rename_File_Mask (gchar *mask);
void     Scan_Check_Mask (GtkObject *wid_emit);
void     Scan_Toggle_Legend_Button (void);
void     Scan_Toggle_Mask_Editor_Button (void);
gchar   *Scan_Replace_String (gchar *string, gchar *last, gchar *new);
void     Scan_Option_Button (void);
void     ScannerWindow_Key_Press (GtkWidget *window, GdkEvent *event);

gchar     *Scan_Generate_New_Filename_From_Mask (gchar *mask);
GtkWidget *Scan_Return_Entry_From_Mask_Code (gchar code);
void       Scan_Rename_File_Generate_Preview (void);

void Process_Fields_Check_Button_Toggled (GtkObject *object, GList *list);
void Select_Fields_Invert_Selection (void);
void Select_Fields_Select_Unselect_All (void);

void Mask_Editor_Clist_Select_Row (GtkCList *clist,gint row,gint column,GdkEventButton *event,gpointer data);
void Mask_Editor_Clist_Unselect_Row (void);
void Mask_Editor_Clist_New (void);
void Mask_Editor_Clist_Copy (void);
void Mask_Editor_Clist_Add (void);
void Mask_Editor_Clist_Remove (void);
void Mask_Editor_Clist_Move_Up (void);
void Mask_Editor_Clist_Move_Down (void);
void Mask_Editor_Clist_Save_Button (void);
void Mask_Editor_Entry_Changed (void);
void Mask_Editor_Clist_Key_Press (GtkWidget *widget, GdkEvent *event);

void Mask_Editor_Create_Masks_List_From_Clist (void);
void Mask_Editor_Clist_Load_Mask_List (GList *list);

void Scanner_Option_Menu_Activate_Item (GtkWidget *widget, Scanner_Option_Menu_Items_Value item);



/*************
 *           * 
 * Functions *
 *           * 
 *************/

void Init_ScannerWindow (void)
{
    ScannerWindow     = (GtkWidget *)NULL;
    ScannerOptionMenu = (GtkWidget *)NULL;
    SWScanButton      = (GtkWidget *)NULL;
    SWScanAllButton   = (GtkWidget *)NULL;
}


/*******************
 * Scanner for Tag *
 *******************/
/*
 * Uses the filename and path to fill tag informations
 * Note: mask and source are read from the right to the left
 */
void Scan_Tag_With_Mask (ET_File *ETFile)
{
    GtkWidget *dest = NULL;
    gchar *source = NULL;
    gchar *mask = NULL;
    gchar *tmp;
    gchar buf[256];
    gchar separator[256];
    gchar string[1024];
    gint len, i, loop=0;
    gchar **mask_splitted;
    gchar **file_splitted;
    guint mask_splitted_number;
    guint file_splitted_number;
    guint mask_splitted_index;
    guint file_splitted_index;


    if (!ScannerWindow || !ScannerScanTagMaskCombo || !ETFile) return;

    mask   = g_strdup(gtk_entry_get_text_1(ScannerScanTagMaskCombo));
    source = g_strdup(((File_Name *)((GList *)ETFile->FileNameNew)->data)->value);
    if (!mask || !source) return;

    // Remove extension of file (if found)
    tmp = strrchr(source,'.');
    for (i=0; i<ET_FILE_DESCRIPTION_SIZE; i++)
    {
        if ( strcasecmp(tmp,ETFileDescription[i].Extension)==0 )
        {
            *tmp = 0; //strrchr(source,'.') = 0;
            break;
        }
    }
    if (i==ET_FILE_DESCRIPTION_SIZE)
        g_print("Tag scanner: strange..., the extension '%s' was not found into filename '%s'!\n",tmp,g_basename(source));


    // Replace characters into mask and filename before parsing
    if (CONVERT_UNDERSCORE_AND_P20_INTO_SPACE)
    {
        Scan_Convert_Underscore_Into_Space(mask);
        Scan_Convert_Underscore_Into_Space(source);
        Scan_Convert_P20_Into_Space(mask);
        Scan_Convert_P20_Into_Space(source);
    }
    if (CONVERT_SPACE_INTO_UNDERSCORE)
    {
        Scan_Convert_Space_Into_Undescore(mask);
        Scan_Convert_Space_Into_Undescore(source);
    }


    // Scanner mask
    mask_splitted = g_strsplit(mask,"/",0);
    // Get number of arguments into 'mask_splitted'
    for (mask_splitted_number=0;mask_splitted[mask_splitted_number];mask_splitted_number++);

    // File Path
    file_splitted = g_strsplit(source,"/",0);
    // Get number of arguments into 'file_splitted'
    for (file_splitted_number=0;file_splitted[file_splitted_number];file_splitted_number++);

    // Set the starting position of each tab
    if (mask_splitted_number <= file_splitted_number)
    {
        mask_splitted_index = 0;
        file_splitted_index = file_splitted_number - mask_splitted_number;
    }else
    {
        mask_splitted_index = mask_splitted_number - file_splitted_number;
        file_splitted_index = 0;
    }


    loop = 0;
    while ( mask_splitted[mask_splitted_index]!= NULL && file_splitted[file_splitted_index]!=NULL )
    {
        gchar *mask_seq = mask_splitted[mask_splitted_index];
        gchar *file_seq = file_splitted[file_splitted_index];
        
        //g_print(">%d> seq '%s' '%s'\n",loop,mask_seq,file_seq);
        while ( mask_seq && strlen(mask_seq)>0 )
        {

            /*
             * Determine (first) code and destinataire
             */
            if ( (tmp=strchr(mask_seq,'%')) == NULL || strlen(tmp) < 2 )
            {
                break;
            }
            
            // Get the target entry for this code
            dest = Scan_Return_Entry_From_Mask_Code(tmp[1]);

            /*
             * Delete text before the code
             */
            if ( (len = strlen(mask_seq) - strlen(tmp)) > 0 )
            {
                // Get this text in 'mask_seq'
                strncpy(buf,mask_seq,len);
                buf[len] = 0;
                // We remove it in 'mask_seq'
                mask_seq = mask_seq + len;
                // Find the same text at the begining of 'file_seq' ?
                if ( (strstr(file_seq,buf)) == file_seq )
                {
                    file_seq = file_seq + len; // We remove it
                }else
                {
                    g_print(_("Scan Error: can't find separator '%s' within '%s'\n"),buf,file_seq);
                }
            }

            // Remove the current code into 'mask_seq'
            mask_seq = mask_seq + 2;

            /*
             * Determine separator between two code or trailing text (after code)
             */
            if ( mask_seq && strlen(mask_seq)>0 )
            {
                if ( (tmp=strchr(mask_seq,'%')) == NULL || strlen(tmp) < 2 )
                {
                    // No more code found
                    strcpy(separator,mask_seq);
                    separator[strlen(mask_seq)] = 0;
                }else
                {
                    len = strlen(mask_seq) - strlen(tmp);
                    strncpy(separator,mask_seq,len);
                    separator[len] = 0;
                }

                // Remove the current separator in 'mask_seq'
                mask_seq = mask_seq + strlen(separator);

                // Try to find the separator in 'file_seq'
                if ( (tmp=strstr(file_seq,separator)) == NULL )
                {
                    g_print(_("Scan Error: can't find separator '%s' within '%s'\n"),separator,file_seq);
                    separator[0] = 0; // Needed to avoid error when calculting 'len' below
                }

                // Get the string affected to the code (or the corresponding entry field)
                len = strlen(file_seq) - (tmp!=NULL?strlen(tmp):0);
                strncpy(string,file_seq,len);
                string[len] = 0;

                // Remove the current separator in 'file_seq'
                file_seq = file_seq + strlen(string) + strlen(separator);

                // We display the text affected to the code
                Scan_Display_String_To_Ui(dest,string);
            }else
            {
                // We display the remaining text, affected to the code (no more data in 'mask_seq')
                Scan_Display_String_To_Ui(dest,file_seq);
                
            }
        }
        
        // Next sequences
        mask_splitted_index++;
        file_splitted_index++;
        loop++;
    }

    g_free(mask);
    g_free(source);
    g_strfreev(mask_splitted);
    g_strfreev(file_splitted);

    Statusbar_Message(_("Tag successfully scanned..."),TRUE);
    g_print(_("Tag successfully scanned...(%s)\n"),g_basename( ((File_Name *)ETFile->FileNameNew->data)->value ) );
}



/*
 * Set the string into the 'dest' entry widget.
 * We can set the string if the widget is empty or we force it (overwrite)
 */
void Scan_Display_String_To_Ui (GtkWidget *wid, gchar *string)
{
    if ( wid==NULL || wid==DummyEntry ) return;

    if (wid == TrackEntry)
    {
        /* For the Track Entry widget (combobox) */
        if (OVERWRITE_TAG_FIELD || strlen(gtk_entry_get_text_1(wid))==0)
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(wid)->entry),string);
    } else if (wid == GenreEntry)
    {
        /* For the Genre Entry widget (combobox) */
        if (OVERWRITE_TAG_FIELD || strlen(gtk_entry_get_text_1(wid))==0)
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(wid)->entry),Id3tag_Genre_To_String(atoi(string)));
    } else
    {
        /* For other entries widget (entry) */
        if (OVERWRITE_TAG_FIELD || strlen(gtk_entry_get_text_1(wid))==0)
            gtk_entry_set_text(GTK_ENTRY(wid),string);
    }
}


/*
 * Function to replace underscore '_' by a space
 */
void Scan_Convert_Underscore_Into_Space (gchar *string)
{
    gchar *tmp;

    while ((tmp=strchr(string,'_'))!=NULL)
        *tmp = ' ';
}

/*
 * Function to replace %20 by a space
 */
void Scan_Convert_P20_Into_Space (gchar *string)
{
    gchar *tmp, *tmp1;

    while ((tmp=strstr(string,"%20"))!=NULL)
    {
        tmp1 = tmp + 3;
        *(tmp++) = ' ';
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}

/*
 * Function to replace space by '_'
 */
void Scan_Convert_Space_Into_Undescore (gchar *string)
{
    gchar *tmp;

    while ((tmp=strchr(string,' '))!=NULL)
        *tmp = '_';
}



/**************************
 * Scanner To Rename File *
 **************************/
/*
 * Uses tag informations (displayed into tag entries) to rename file
 * Note: mask and source are read from the right to the left.
 * Note1: a mask code may be used severals times...
 */
void Scan_Rename_File_With_Mask (ET_File *ETFile)
{
    gchar *filename_new = NULL;
    gchar *mask = NULL;


    if (!ScannerWindow || !ScannerRenameFileMaskCombo || !ETFile) return;

    mask = g_strdup(gtk_entry_get_text_1(ScannerRenameFileMaskCombo));
    filename_new = Scan_Generate_New_Filename_From_Mask(mask);

    if (mask) g_free(mask);
    if (!filename_new)
        return;
    
    /* Replace characters */
    if (CONVERT_UNDERSCORE_AND_P20_INTO_SPACE)
    {
        Scan_Convert_Underscore_Into_Space(filename_new);
        Scan_Convert_P20_Into_Space(filename_new);
    }
    if (CONVERT_SPACE_INTO_UNDERSCORE)
        Scan_Convert_Space_Into_Undescore(filename_new);

    /* Set the new filename into the entry */
    gtk_entry_set_text(GTK_ENTRY(FileEntry),filename_new);
    g_free(filename_new);

    Statusbar_Message(_("New file name successfully scanned..."),TRUE);
    g_print(_("New file name successfully scanned...(%s)\n"),g_basename( ((File_Name *)ETFile->FileNameNew->data)->value ) );

    return;
}

gchar *Scan_Generate_New_Filename_From_Mask (gchar *mask)
{
    gchar *tmp;
    GtkWidget *source_entry = NULL;
    gchar *entry_text;
    gchar *filename_new = NULL;
    gchar *filename_tmp = NULL;
    GList *new_filename_list = NULL;
    File_Mask_Item *mask_item;
    File_Mask_Item *mask_item_prev;
    File_Mask_Item *mask_item_next;
    gint counter = 0;

    if (!mask) return NULL;

    /*
     * Parse the codes to generate a list (1rst item = 1rst code)
     */
    while ( mask!=NULL && (tmp=strrchr(mask,'%'))!=NULL && strlen(tmp)>1 )
    {
        // Mask contains some characters after the code ('%b__')
        if (strlen(tmp)>2)
        {
            mask_item = g_malloc0(sizeof(File_Mask_Item));
            if (counter) mask_item->type = SEPARATOR;
            else         mask_item->type = TRAILING_SEPARATOR;
            mask_item->string = g_strdup(tmp+2);
            new_filename_list = g_list_prepend(new_filename_list,mask_item);
        }
        // Now, parses the code
        source_entry = Scan_Return_Entry_From_Mask_Code(tmp[1]);
        if (source_entry)
            entry_text = gtk_entry_get_text_1(source_entry);
        else
            entry_text = NULL;
        mask_item = g_malloc0(sizeof(File_Mask_Item));
        if (source_entry && entry_text && strlen(entry_text)>0)
        {
            mask_item->type = FIELD;
            mask_item->string = g_strdup(entry_text);
        }else
        {
            mask_item->type = EMPTY_FIELD;
            mask_item->string = NULL;
        }
        new_filename_list = g_list_prepend(new_filename_list,mask_item);
        *tmp = '\0'; // Cut parsed data of mask
        counter++; // To indicate that we made at least one loop to identifiate 'separator' or 'trailing_separator'
    }
    
    // It may have some characters before the last remaining code ('__%a')
    if (mask!=NULL && strlen(mask)>0)
    {
        mask_item = g_malloc0(sizeof(File_Mask_Item));
        mask_item->type = LEADING_SEPARATOR;
        mask_item->string = g_strdup(mask);
        new_filename_list = g_list_prepend(new_filename_list,mask_item);
    }
    
    if (!new_filename_list) return NULL;


    /*
     * Build the new filename with items placed into the list
     * (read the list from the end to the beginning)
     */
    new_filename_list = g_list_last(new_filename_list);
    filename_new = g_strdup("");

    while (new_filename_list)
    {
        File_Mask_Item *mask_item = new_filename_list->data;

        if ( mask_item->type==TRAILING_SEPARATOR ) // Trailing characters of mask
        {
            // Doesn't write it if previous field is empty
            if (new_filename_list->prev && ((File_Mask_Item *)new_filename_list->prev->data)->type!=EMPTY_FIELD)
            {
                filename_tmp = filename_new;
                filename_new = g_strconcat(mask_item->string,filename_new,NULL);
                g_free(filename_tmp);
            }
        }else
        if ( mask_item->type==LEADING_SEPARATOR ) // Leading characters of mask
        {
            filename_tmp = filename_new;
            filename_new = g_strconcat(mask_item->string,filename_new,NULL);
            g_free(filename_tmp);
        }else
        if ( mask_item->type==EMPTY_FIELD )
        // We don't contatenate the field value (empty) and the previous separator to the filename
        // If field is the 'first', we don't concatenate it and the next separator too.
        {
            if (new_filename_list->prev) // If previous string is a separator, we don't use it
            {
                mask_item_prev = new_filename_list->prev->data;
                if ( mask_item_prev->type==SEPARATOR )
                    new_filename_list = new_filename_list->prev;
                if (new_filename_list->next && (mask_item_next=new_filename_list->next->data)
                &&  mask_item_prev->type==LEADING_SEPARATOR && mask_item_next->type==SEPARATOR)
                {
                    if ( filename_new && (strlen(filename_new)>=strlen(mask_item_next->string)) ) // To avoid crash if filename_new is 'empty'
                    {
                        filename_tmp = filename_new;
                        filename_new = g_strdup(filename_new+strlen(mask_item_next->string));
                        g_free(filename_tmp);
                     }
                }
            }else
            if (new_filename_list->next && (mask_item_next=new_filename_list->next->data)
            &&  mask_item_next->type==SEPARATOR) // We are at the 'beginning' of the mask
            {
                if ( filename_new && (strlen(filename_new)>=strlen(mask_item_next->string)) ) // To avoid crash if filename_new is 'empty'
                {
                    filename_tmp = filename_new;
                    filename_new = g_strdup(filename_new+strlen(mask_item_next->string));
                    g_free(filename_tmp);
                 }
            }

        }else // SEPARATOR, FIELD
        {
            filename_tmp = filename_new;
            filename_new = g_strconcat(mask_item->string,filename_new,NULL);
            g_free(filename_tmp);
        }
        
        if (!new_filename_list->prev) break;
        new_filename_list = new_filename_list->prev;
    }

    // Free the list
    new_filename_list = g_list_last(new_filename_list);
    while (new_filename_list)
    {
        if (new_filename_list->data)
        {
            if ( ((File_Mask_Item *)new_filename_list->data)->string )
                g_free(((File_Mask_Item *)new_filename_list->data)->string);
            g_free( (File_Mask_Item *)new_filename_list->data );
        }
        if (!new_filename_list->prev) break;
        new_filename_list = new_filename_list->prev;
    }
    g_list_free(new_filename_list);

    return filename_new;
}

void Scan_Rename_File_Generate_Preview (void)
{
    gchar *filename_new = NULL;
    gchar *mask = NULL;

    if (!ScannerWindow || !ScannerRenameFileMaskCombo || !RenameFilePreviewLabel) return;

    mask = g_strdup(gtk_entry_get_text_1(ScannerRenameFileMaskCombo));
    filename_new = Scan_Generate_New_Filename_From_Mask(mask);

    if (GTK_IS_LABEL(RenameFilePreviewLabel))
    {
        if (filename_new)
            gtk_label_set_text(GTK_LABEL(RenameFilePreviewLabel),filename_new);
        else
            gtk_label_set_text(GTK_LABEL(RenameFilePreviewLabel),"");
    }

    if (mask) g_free(mask);
    if (filename_new) g_free(filename_new);
}



/*****************************
 * Scanner To Process Fields *
 *****************************/
void Scan_Process_Fields (ET_File *ETFile)
{
    GList *list_entries = NULL;


    if (!ScannerWindow || !ETFile) return;

    /* Create a list of entries to process */
    if (GTK_TOGGLE_BUTTON(ProcessFileNameField)->active)
        list_entries = g_list_append(list_entries,FileEntry);
    if (GTK_TOGGLE_BUTTON(ProcessTitleField)->active)
        list_entries = g_list_append(list_entries,TitleEntry);
    if (GTK_TOGGLE_BUTTON(ProcessArtistField)->active)
        list_entries = g_list_append(list_entries,ArtistEntry);
    if (GTK_TOGGLE_BUTTON(ProcessAlbumField)->active)
        list_entries = g_list_append(list_entries,AlbumEntry);
    if (GTK_TOGGLE_BUTTON(ProcessCommentField)->active)
        list_entries = g_list_append(list_entries,CommentEntry);

    while (list_entries)
    {
        guint string_len = 2 * strlen(gtk_entry_get_text_1(list_entries->data));
        gchar *text = g_malloc(string_len+1);
        strncpy(text,gtk_entry_get_text_1(list_entries->data),string_len);
        text[string_len]='\0';

        if (GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active)
        {
            Scan_Convert_Underscore_Into_Space(text);
            Scan_Convert_P20_Into_Space(text);
        }

        if (GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active)
            Scan_Convert_Space_Into_Undescore(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase)->active)
            Scan_Process_Fields_All_Uppercase(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase)->active)
            Scan_Process_Fields_All_Downcase(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase)->active)
            Scan_Process_Fields_Letter_Uppercase(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase)->active)
            Scan_Process_Fields_First_Letters_Uppercase(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsRemoveSpace)->active)
            Scan_Process_Fields_Remove_Space(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsInsertSpace)->active)
            Scan_Process_Fields_Insert_Space(text);

        if (GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace)->active)
            Scan_Process_Fields_Keep_One_Space(text);
    
        /* Set the processed string to the entry */
        if (GTK_IS_COMBO(list_entries->data))
            gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(list_entries->data)->entry),text);
        else
            gtk_entry_set_text(GTK_ENTRY(list_entries->data),text);
        g_free(text);

        list_entries = g_list_next(list_entries);
    }
}

void Scan_Process_Fields_All_Uppercase (gchar *text)
{
    gint j;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<strlen(text); j++)
        if ( (text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='') )
            text[j] = text[j] + ascii_caps_interval;
}
void Scan_Process_Fields_All_Downcase (gchar *text)
{
    gint j;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<strlen(text); j++)
        if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='') )
            text[j] = text[j] - ascii_caps_interval;
}
void Scan_Process_Fields_Letter_Uppercase (gchar *text)
{
    gint j;
    gint set_to_upper_case = 1;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<strlen(text); j++)
    {
        if ( ((text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='')) && set_to_upper_case )
            text[j] = text[j] + ascii_caps_interval;
        else if ( ((text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='')) && !set_to_upper_case )
            text[j] = text[j] - ascii_caps_interval;
        set_to_upper_case = 0;
    }
}
void Scan_Process_Fields_First_Letters_Uppercase (gchar *text)
{
    gint j;
    gint set_to_upper_case = 1;    /* if 1 => capitalize */
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<strlen(text); j++)
    {
        /* It's an alphabetic character? (including '', '', ...) */
        if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='a' && text[j]<='z')
          || (text[j]>='' && text[j]<='') || (text[j]>='' && text[j]<='') )
        {
            if (set_to_upper_case)
            {
                if ( (text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='') )
                    text[j] = text[j] + ascii_caps_interval;
                /* The next character musn't be capitalized */
                set_to_upper_case = 0;
            }else
            {
                if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='') )
                    text[j] = text[j] - ascii_caps_interval;
            }
        }else
        {
            /* If character isn't alphabetic => unclock => next alphabetic character to uppercase
             * If a point found => probably an extension => doesn't uppercase next chararacter
             * If an apostrophe found (code 39) => ex: "...I'll..." => doesn't uppercase 
             * next chararacter */
            if (text[j]!='.' && text[j]!=39)
                set_to_upper_case = 1;
        }
    }
}
void Scan_Process_Fields_Remove_Space (gchar *text)
{
    gchar *tmp, *tmp1;

    while ((tmp=strchr(text,' '))!=NULL)
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}
void Scan_Process_Fields_Insert_Space (gchar *text)
{
    gint i,j;
    
    /* FIXME: try to use realloc */
    for (i=1; i<strlen(text); i++)    // i=1 to not consider first "uppercase" letter
    {
        if (isupper(text[i]))
        {
            for (j=strlen(text); j>=i; j--)
            {
                text[j+1] = text[j];
            }
            text[i] = ' ';
            i++;
        }
    }
}
void Scan_Process_Fields_Keep_One_Space (gchar *text)
{
    gchar *tmp, *tmp1;

    /* Remove multiple consecutive spaces
     * Note: if exists severals spaces => exist at least 2 spaces */
    while ((tmp=strstr(text,"  "))!=NULL)
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }

    /* Remove multiple consecutive underscores */
    while ((tmp=strstr(text,"__"))!=NULL)
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}


/*
 * Return the entry correxponding to the mask code
 */
GtkWidget *Scan_Return_Entry_From_Mask_Code (gchar code)
{
    /* Create the dummy entry */
    if (!DummyEntry)
        DummyEntry = gtk_entry_new();

    switch (code)
    {
        case 't':    /* Title */
            return TitleEntry;
        case 'a':    /* Artist */
            return ArtistEntry;
        case 'b':    /* Album */
            return AlbumEntry;
        case 'y':    /* Year */
            return YearEntry;
        case 'g':    /* Genre */
            return GenreEntry;
        case 'n':    /* Track */
            return TrackEntry;
        case 'l':    /* Track Total */
            return TrackTotalEntry;
        case 'c':    /* Comment */
            return CommentEntry;
        case 'i':    /* Ignored */
            return DummyEntry;
        default:
            g_print("Scanner: Invalid code '%%%c' found!\n",code);
            return NULL;
    }
}



/******************
 * Scanner Window *
 ******************/
#include "../pixmaps/scan.xpm"
#include "../pixmaps/scan_all.xpm"
#include "../pixmaps/help.xpm"
#include "../pixmaps/exit.xpm"
#include "../pixmaps/setting.xpm"
#include "../pixmaps/mask.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/up.xpm"
#include "../pixmaps/down.xpm"
#include "../pixmaps/new.xpm"
#include "../pixmaps/copy.xpm"
#include "../pixmaps/trash.xpm"
#include "../pixmaps/add.xpm"
#include "../pixmaps/blackwhite.xpm"
#include "../pixmaps/forbidden.xpm"

void Open_ScannerWindow (Scanner_Option_Menu_Items_Value scanner_type)
{
    GtkWidget *ScanVBox;
    GtkWidget *HBox1, *HBox2, *HBox4, *VBox, *hbox, *vbox;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkWidget *Icon;
    GtkWidget *EventBox;
    GtkTooltips *Tips;
    GtkWidget *Menu, *MenuItem;
    gint i;
    GList *pf_cb_group1 = NULL;
    GList *pf_cb_group2 = NULL;
    GList *pf_cb_group3 = NULL;


    /* Check if already opened */
    if (ScannerWindow)
    {
        //gdk_window_show(ScannerWindow->window);
        gdk_window_raise(ScannerWindow->window);
        if (ScannerOptionMenu)
        {
            gtk_option_menu_set_history(GTK_OPTION_MENU(ScannerOptionMenu),scanner_type);
            Scanner_Option_Menu_Activate_Item(NULL,scanner_type);
        }
        return;
    }

    if (scanner_type < 0)
        scanner_type = SCANNER_FILL_TAG;

    /* The window */
    ScannerWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    /* Config */
    gtk_container_border_width(GTK_CONTAINER(ScannerWindow),5);
    gtk_window_set_policy(GTK_WINDOW(ScannerWindow),FALSE,FALSE,TRUE);
    if (SCANNER_WINDOW_ON_TOP)
        gtk_window_set_transient_for(GTK_WINDOW(ScannerWindow),GTK_WINDOW(MainWindow));

    /* The init position is define below, cause the scanner window must be showed before
     * to be able to move it. */

    /* Title */
    gtk_window_set_title(GTK_WINDOW(ScannerWindow),_("Tag and File Name scan"));

    /* Signals connection */
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"destroy",(GtkSignalFunc)ScannerWindow_Quit,NULL);
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"delete_event",(GtkSignalFunc)ScannerWindow_Quit,NULL);
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"key_press_event",(GtkSignalFunc)ScannerWindow_Key_Press,NULL);

    /* The tooltips */
    Tips = gtk_tooltips_new_1();

    /* The main vbox */
    ScanVBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(ScannerWindow),ScanVBox);


    /*
     * The hbox for mode buttons + buttons + what to scan
     */
    HBox1 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(ScanVBox),HBox1,FALSE,FALSE,0);

    /* Option Menu */
    Label = gtk_label_new(_("Scanner :"));
    gtk_box_pack_start(GTK_BOX(HBox1),Label,FALSE,FALSE,0);
    ScannerOptionMenu = gtk_option_menu_new();
    gtk_box_pack_start(GTK_BOX(HBox1),ScannerOptionMenu,TRUE,TRUE,2);
    gtk_widget_set_usize(ScannerOptionMenu,120,-1);
    /* The menu */
    Menu = gtk_menu_new();
    /* Option for Tag */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_FILL_TAG]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_FILL_TAG));
    /* Option for FileName */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_RENAME_FILE]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_RENAME_FILE));
    /* Option for ProcessFields */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_PROCESS_FIELDS]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_PROCESS_FIELDS));

    gtk_option_menu_set_menu(GTK_OPTION_MENU(ScannerOptionMenu),Menu);
    // Selection of the item made at the end of the function
    gtk_option_menu_set_history(GTK_OPTION_MENU(ScannerOptionMenu),scanner_type);
    gtk_tooltips_set_tip(Tips,ScannerOptionMenu,_("Select the type of scanner to use"),NULL);


    /* 'Scan file' button */
    SWScanButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(scan_xpm);
    gtk_container_add(GTK_CONTAINER(SWScanButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),SWScanButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(SWScanButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,SWScanButton,_("Scan this file"),NULL);
    gtk_signal_connect(GTK_OBJECT(SWScanButton),"clicked",(GtkSignalFunc)Action_Scan_File,NULL);

    /* 'Scan all files' button */
    SWScanAllButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(scan_all_xpm);
    gtk_container_add(GTK_CONTAINER(SWScanAllButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),SWScanAllButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(SWScanAllButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,SWScanAllButton,_("Scan all files"),NULL);
    gtk_signal_connect(GTK_OBJECT(SWScanAllButton),"clicked",(GtkSignalFunc)Action_Scan_All_Files,NULL);

    /* Separator line */
    Separator = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(HBox1),Separator,FALSE,FALSE,2);

    /* Options button */
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(setting_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Scanner Options"),NULL);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Scan_Option_Button,NULL);

    /* Mask Editor button */
    MaskEditorButton = gtk_toggle_button_new();
    Icon = Create_Pixmap_Icon(mask_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),MaskEditorButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorButton,_("Show / Hide Masks Editor"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorButton),"toggled",
        (GtkSignalFunc)Scan_Toggle_Mask_Editor_Button,NULL);

    /* Legend button */
    LegendButton = gtk_toggle_button_new();
    Icon = Create_Pixmap_Icon(help_xpm);
    gtk_container_add(GTK_CONTAINER(LegendButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),LegendButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(LegendButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,LegendButton,_("Show / Hide Legend"),NULL);
    gtk_signal_connect(GTK_OBJECT(LegendButton),"toggled",(GtkSignalFunc)Scan_Toggle_Legend_Button,NULL);

    /* Close button */
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(exit_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Close this window"),NULL);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)ScannerWindow_Quit,NULL);

    /*
     * Frame for Scan Tag
     */
    ScanTagFrame = gtk_frame_new (_("Scan Tag"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),ScanTagFrame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(ScanTagFrame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    gtk_widget_show(vbox);

    /* The combo box + Status icon */
    HBox2 = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),HBox2,TRUE,TRUE,0);
    /* The combo box to select the mask to apply */
    ScannerScanTagMaskCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(HBox2),ScannerScanTagMaskCombo,TRUE,TRUE,2);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ScannerScanTagMaskCombo)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(ScannerScanTagMaskCombo),TRUE);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(ScannerScanTagMaskCombo)->entry,_("Select or type in a mask "
        "using codes (see Legend) to parse file name and path. Used to fill in tag fields."),NULL);
    /* Signals */
    /* Signal connection to check if mask is correct into the main mask entry */
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerScanTagMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Mask),GTK_OBJECT(ScannerScanTagMaskCombo));

    /* Load masks into the combobox from a file */
    MasksList = Load_Scan_Tag_Masks_List();

    /* No mask read => load default masks */
    if (MasksList==NULL)
    {
        g_print(_("Loading default 'Fill Tag' masks...\n"));
        for (i=0;i<(sizeof(Scan_Masks)/sizeof(Scan_Masks[0]));i++)
            MasksList = g_list_append(MasksList,Scan_Masks[i]);
    }
    gtk_combo_set_popdown_strings(GTK_COMBO(ScannerScanTagMaskCombo),MasksList);

    /* Mask status icon */
    MaskStatusIconBox = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(HBox2),MaskStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox,_("Invalid Scanner Mask"),NULL);



    /*
     * Frame for Rename File
     */
    RenameFileFrame = gtk_frame_new (_("Scan File Name"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),RenameFileFrame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(RenameFileFrame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    gtk_widget_show(vbox);

    /* The combo box + Status icon */
    HBox4 = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),HBox4,TRUE,TRUE,0);
    //gtk_container_add(GTK_CONTAINER(RenameFileFrame),HBox4);
    /* The combo box to select the mask to apply */
    ScannerRenameFileMaskCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(HBox4),ScannerRenameFileMaskCombo,TRUE,TRUE,2);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(ScannerRenameFileMaskCombo),TRUE);
    gtk_container_border_width(GTK_CONTAINER(GTK_BIN(RenameFileFrame)->child),4);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(ScannerRenameFileMaskCombo)->entry,_("Select or type in "
        "a mask using codes (see Legend) to parse tag fields. Used to rename the file."),NULL);
    /* Signal to check if mask is correct into the main mask entry */
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Mask),GTK_OBJECT(ScannerRenameFileMaskCombo));
    /* Signal to generate preview */
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Rename_File_Generate_Preview),NULL);

    /* Load masks into the combobox */
    RenameFileMasksList = Load_Rename_File_Masks_List();
    /* Load default masks */
    if (RenameFileMasksList==NULL)
    {
        g_print(_("Loading default 'Rename File' masks...\n"));
        for (i=0;i<(sizeof(Rename_File_Masks)/sizeof(Rename_File_Masks[0]));i++)
            RenameFileMasksList = g_list_append(RenameFileMasksList,Rename_File_Masks[i]);
    }
    gtk_combo_set_popdown_strings(GTK_COMBO(ScannerRenameFileMaskCombo),RenameFileMasksList);

    /* Mask status icon */
    MaskStatusIconBox1 = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(HBox4),MaskStatusIconBox1,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox1,_("Invalid Scanner Mask"),NULL);

    /* Preview label */
    RenameFilePreviewLabel = gtk_label_new(_("Rename file preview..."));
    gtk_label_set_line_wrap(GTK_LABEL(RenameFilePreviewLabel),TRUE);
    gtk_widget_show(RenameFilePreviewLabel);
    gtk_box_pack_start(GTK_BOX(vbox),RenameFilePreviewLabel,TRUE,TRUE,0);



    /*
     * Frame for Processing Fields
     */
    ProcessFieldsFrame = gtk_frame_new (_("Process Fields"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),ProcessFieldsFrame,FALSE,FALSE,0);

    VBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(ProcessFieldsFrame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);
    gtk_widget_show(VBox);

    /* Group: select entry fields to process */
    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),hbox,FALSE,FALSE,2);
    EventBox = gtk_event_box_new();
    Label = gtk_label_new(_("Select fields:"));
    gtk_box_pack_start(GTK_BOX(hbox),EventBox,FALSE,FALSE,2);
    gtk_container_add(GTK_CONTAINER(EventBox),Label);
    gtk_tooltips_set_tip(Tips,EventBox,_("The buttons on the right represent the fields which can "
        "be processed. Select those who interest you."),NULL);
    ProcessFileNameField = gtk_toggle_button_new_with_label(_("F"));    
    gtk_tooltips_set_tip(Tips,ProcessFileNameField,         _("Process file name field"),NULL);
    ProcessTitleField = gtk_toggle_button_new_with_label(   _("T"));    
    gtk_tooltips_set_tip(Tips,ProcessTitleField,            _("Process title field"),NULL);
    ProcessArtistField = gtk_toggle_button_new_with_label(  _("Ar"));    
    gtk_tooltips_set_tip(Tips,ProcessArtistField,           _("Process file artist field"),NULL);
    ProcessAlbumField = gtk_toggle_button_new_with_label(   _("Al"));    
    gtk_tooltips_set_tip(Tips,ProcessAlbumField,            _("Process album field"),NULL);
    ProcessCommentField = gtk_toggle_button_new_with_label( _("C"));    
    gtk_tooltips_set_tip(Tips,ProcessCommentField,          _("Process comment field"),NULL);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessFileNameField,TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessTitleField,   TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessArtistField,  TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessAlbumField,   TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessCommentField, TRUE,TRUE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),PROCESS_FILENAME_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),   PROCESS_TITLE_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),  PROCESS_ARTIST_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),   PROCESS_ALBUM_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField), PROCESS_COMMENT_FIELD);
    /* The small buttons */
    vbox = gtk_vbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
    Button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",GTK_SIGNAL_FUNC(Select_Fields_Invert_Selection),NULL);
    gtk_box_pack_start(GTK_BOX(vbox),Button,FALSE,FALSE,0);
    gtk_widget_set_usize(Button,10,10);
    Icon = Create_Pixmap_Icon(blackwhite_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_tooltips_set_tip(Tips,Button,_("Invert Selection"),NULL);
    Button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",GTK_SIGNAL_FUNC(Select_Fields_Select_Unselect_All),NULL);
    gtk_box_pack_start(GTK_BOX(vbox),Button,FALSE,FALSE,0);
    gtk_widget_set_usize(Button,10,10);
    gtk_tooltips_set_tip(Tips,Button,_("Select/Unselect All."),NULL);
     
    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: character conversion */
    ProcessFieldsConvertIntoSpace = gtk_check_button_new_with_label(_("Convert '_' and '%20' to ' '"));
    ProcessFieldsConvertSpace     = gtk_check_button_new_with_label(_("Convert ' ' to '_'"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsConvertIntoSpace,    FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsConvertSpace,    FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group1 = g_list_append (pf_cb_group1,ProcessFieldsConvertIntoSpace);
    pf_cb_group1 = g_list_append (pf_cb_group1,ProcessFieldsConvertSpace);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvertIntoSpace),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group1);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvertSpace),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group1);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace),PF_CONVERT_INTO_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace),PF_CONVERT_SPACE);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsConvertIntoSpace,
        _("The underscore character or the string '%20' are replaced by one space. "
          "Example, before: 'Text%20In%20An_Entry', after: 'Text In An Entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsConvertSpace,
        _("The space character is replaced by one underscore character. "
          "Example, before: 'Text In An Entry', after: 'Text_In_An_Entry'."),NULL);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: capitalize, ... */
    ProcessFieldsAllUppercase = gtk_check_button_new_with_label         (_("All uppercase"));
    ProcessFieldsAllDowncase  = gtk_check_button_new_with_label         (_("All downcase"));
    ProcessFieldsFirstLetterUppercase  = gtk_check_button_new_with_label(_("First letter uppercase"));
    ProcessFieldsFirstLettersUppercase = gtk_check_button_new_with_label(_("First letter uppercase of each word"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsAllUppercase,         FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsAllDowncase,          FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsFirstLetterUppercase, FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsFirstLettersUppercase,FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsAllUppercase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsAllDowncase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsFirstLetterUppercase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsFirstLettersUppercase);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsAllUppercase),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsAllDowncase),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsFirstLetterUppercase),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsFirstLettersUppercase),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group2);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase),PF_CONVERT_ALL_UPPERCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase),PF_CONVERT_ALL_DOWNCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase),PF_CONVERT_FIRST_LETTER_UPPERCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase),PF_CONVERT_FIRST_LETTERS_UPPERCASE);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsAllUppercase,
        _("Convert all words in all fields to upper case. "
          "Example, before: 'Text IN AN entry', after: 'TEXT IN AN ENTRY'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsAllDowncase,
        _("Convert all words in all fields to lower case. "
          "Example, before: 'TEXT IN an entry', after: 'text in an entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsFirstLetterUppercase,
        _("Convert the initial of the first word in all fields to upper case. "
          "Example, before: 'text IN An ENTRY', after: 'Text in an entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsFirstLettersUppercase,
        _("Convert the initial of each word in all fields to upper case. "
          "Example, before: 'Text in an ENTRY', after: 'Text In An Entry'."),NULL);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: insert/remove spaces */
    ProcessFieldsRemoveSpace = gtk_check_button_new_with_label(_("Remove spaces"));
    ProcessFieldsInsertSpace = gtk_check_button_new_with_label(_("Insert a space before an uppercase letter"));
    ProcessFieldsOnlyOneSpace = gtk_check_button_new_with_label(_("Remove duplicates of space or underscore"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsRemoveSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsInsertSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsOnlyOneSpace,FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsRemoveSpace);
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsInsertSpace);
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsOnlyOneSpace);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsRemoveSpace),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group3);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsInsertSpace),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group3);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsOnlyOneSpace),"toggled",
        Process_Fields_Check_Button_Toggled,pf_cb_group3);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsRemoveSpace),PF_REMOVE_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsInsertSpace),PF_INSERT_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace),PF_ONLY_ONE_SPACE);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsRemoveSpace,
        _("All spaces between words are removed. "
          "Example, before: 'Text In An Entry', after: 'TextInAnEntry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsInsertSpace,
        _("A space is inserted before each upper case letter. "
          "Example, before: 'TextInAnEntry', after: 'Text In An Entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsOnlyOneSpace,
        _("Duplicated spaces or underscores are removed. "
          "Example, before: 'Text__In__An   Entry', after: 'Text_In_An Entry'."),NULL);



    /*
     * Frame to display codes legend
     */
    LegendFrame = gtk_frame_new (_("Legend"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),LegendFrame,FALSE,FALSE,0);
    /* Legend labels */
    Table = gtk_table_new(3,3,FALSE);
    gtk_container_add(GTK_CONTAINER(LegendFrame),Table);
    gtk_container_border_width(GTK_CONTAINER(Table),4);
    Label = gtk_label_new(_("%t : title"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%a : artist"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%b : album"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%y : year"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%g : genre"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%n : track"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%l : number of tracks"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%c : comment"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%i : ignored"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);


    /*
     * Masks Editor
     */
    MaskEditorFrame = gtk_frame_new (_("Mask Editor"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),MaskEditorFrame,FALSE,FALSE,0);
    MaskEditorHBox = gtk_hbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(MaskEditorFrame),MaskEditorHBox);
    gtk_container_border_width(GTK_CONTAINER(MaskEditorHBox),4);

    /* The editor part */
    MaskEditorVBox = gtk_vbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(MaskEditorHBox),MaskEditorVBox,TRUE,TRUE,0);
    MaskEditorScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorScrollWindow,TRUE,TRUE,0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(MaskEditorScrollWindow),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(MaskEditorScrollWindow),-1,101);

    /* The Clist */
    MaskEditorClist = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(MaskEditorScrollWindow),MaskEditorClist);
    gtk_clist_set_reorderable(GTK_CLIST(MaskEditorClist),TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(MaskEditorClist),0,TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(MaskEditorClist),GTK_SELECTION_EXTENDED);
    gtk_signal_connect_after(GTK_OBJECT(MaskEditorClist),"select-row",
        (GtkSignalFunc)Mask_Editor_Clist_Select_Row,NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorClist),"key-press-event",
        (GtkSignalFunc)Mask_Editor_Clist_Key_Press, NULL);

    /* The entry */
    hbox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),hbox,FALSE,FALSE,0);
    MaskEditorEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox),MaskEditorEntry,TRUE,TRUE,2);
    gtk_signal_connect(GTK_OBJECT(MaskEditorEntry),"changed",
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
    /* Signal connection to check if mask is correct into the mask entry of mask editor */
    gtk_signal_connect_object(GTK_OBJECT(MaskEditorEntry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Mask),GTK_OBJECT(MaskEditorEntry));
    /* Mask status icon */
    MaskStatusIconBox2 = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(hbox),MaskStatusIconBox2,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox2,_("Invalid Scanner Mask"),NULL);

    /* The buttons part */
    MaskEditorVBox = gtk_vbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(MaskEditorHBox),MaskEditorVBox,FALSE,FALSE,0);

    /* New mask button */
    MaskEditorNewButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(new_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorNewButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorNewButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorNewButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorNewButton,_("Create New Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorNewButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_New,NULL);

    /* Move up mask button */
    MaskEditorUpButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(up_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorUpButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorUpButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorUpButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorUpButton,_("Move Up this Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorUpButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Move_Up,NULL);

    /* Move down mask button */
    MaskEditorDownButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(down_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorDownButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorDownButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorDownButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorDownButton,_("Move Down this Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorDownButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Move_Down,NULL);

    /* Copy mask button */
    MaskEditorCopyButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(copy_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorCopyButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorCopyButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorCopyButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorCopyButton,_("Duplicate Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorCopyButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Copy,NULL);

    /* Add mask button */
    MaskEditorAddButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(add_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorAddButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorAddButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorAddButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorAddButton,_("Add Default Masks"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorAddButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Add,NULL);

    /* Remove mask button */
    MaskEditorRemoveButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(trash_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorRemoveButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorRemoveButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorRemoveButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorRemoveButton,_("Remove Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorRemoveButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Remove,NULL);

    /* Save mask button */
    MaskEditorSaveButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(save_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorSaveButton),Icon);
    gtk_box_pack_end(GTK_BOX(MaskEditorVBox),MaskEditorSaveButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorSaveButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorSaveButton,_("Save Masks"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorSaveButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Save_Button,NULL);

    /* Load masks list into clist */
    Mask_Editor_Clist_Load_Mask_List(MasksList);

    gtk_widget_show(ScanVBox);
    gtk_widget_show_all(HBox1);
    gtk_widget_show_all(HBox2);
    gtk_widget_show_all(HBox4);
    gtk_widget_show(ScannerWindow);

    /* Init position of the scanner window */
    Scan_Set_Scanner_Window_Init_Position();

    /* To initialize the mask status icon and visibility */
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(ScannerScanTagMaskCombo)->entry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(LegendButton),"toggled");        /* To hide legend frame */
    gtk_signal_emit_by_name(GTK_OBJECT(MaskEditorButton),"toggled");    /* To hide mask editor frame */

    // Activate the current menu in the option menu
    Scanner_Option_Menu_Activate_Item(NULL,scanner_type);
}
void ScannerWindow_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                ScannerWindow_Quit();
                break;
        }
    }
}



/*
 * Select the scanner to run
 */
void Scan_Select_Mode_And_Run_Scanner (void)
{
    if (!ScannerWindow && !ETFileList) return;

    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        /* Set the default text to comment for tag scanner */
        if (SET_DEFAULT_COMMENT && (OVERWRITE_TAG_FIELD || strlen(gtk_entry_get_text_1(CommentEntry))==0 ) )
            gtk_entry_set_text(GTK_ENTRY(CommentEntry),DEFAULT_COMMENT);

        /* Run Scanner Type: Scan Tag */
        Scan_Tag_With_Mask((ET_File *)ETFileList->data);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        /* Run Scanner Type: Rename File */
        Scan_Rename_File_With_Mask((ET_File *)ETFileList->data);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_PROCESS_FIELDS))
    {
        /* Run Scanner Type: Process Fields */
        Scan_Process_Fields((ET_File *)ETFileList->data);
    }
}


/* Callback from Open_ScannerWindow */
void ScannerWindow_Quit (void)
{
    if (ScannerWindow)
    {
        Scan_Apply_Changes();

        gtk_widget_destroy(ScannerWindow);
        ScannerWindow     = (GtkWidget *)NULL;
        ScannerOptionMenu = (GtkWidget *)NULL;
        SWScanButton      = (GtkWidget *)NULL;
        SWScanAllButton   = (GtkWidget *)NULL;

        // To avoid crashs after tests
        ScannerScanTagMaskCombo    = (GtkWidget *)NULL;
        MaskStatusIconBox          = (GtkWidget *)NULL;
        ScannerRenameFileMaskCombo = (GtkWidget *)NULL;
        MaskStatusIconBox1         = (GtkWidget *)NULL;
        MaskEditorEntry            = (GtkWidget *)NULL;
        MaskStatusIconBox2         = (GtkWidget *)NULL;
        LegendFrame                = (GtkWidget *)NULL;
        ProcessFieldsConvertIntoSpace = (GtkWidget *)NULL;
        ProcessFieldsConvertSpace     = (GtkWidget *)NULL;
    }
}


/* Callback from Option button */
void Scan_Option_Button (void)
{
    Open_OptionsWindow();
    gtk_notebook_set_page(GTK_NOTEBOOK(OptionsNoteBook),OptionsNoteBook_Scanner_Page_Num);
}



/*
 * Check if mask is valid. Return 1 if valid, 0 else.
 */
gboolean Scan_Check_Scan_Tag_Mask (gchar *mask)
{
    gchar *tmp = NULL;
    gint loop = 0;


    if (!mask || strlen(mask)<1)
        goto Bad_Mask;

    while (mask)
    {
        if ( (tmp=strrchr(mask,'%'))==NULL )
        {
            if (loop==0)
                /* There is no code the first time => not accepted */
                goto Bad_Mask;
            else
                /* There is no more code => accepted */
                goto Good_Mask;
        }
        if (strlen(tmp)>1 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* Code is correct */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }

        /* Check the following code and separator */
        if ( (tmp=strrchr(mask,'%'))==NULL )
            /* There is no more code => accepted */
            goto Good_Mask;

        if (strlen(tmp)>2 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* There is a separator and code is correct */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }
        loop++;
    }

    Bad_Mask:
        if (mask) g_free(mask);
        return 0;

    Good_Mask:
        if (mask) g_free(mask);
        return 1;
}
gboolean Scan_Check_Rename_File_Mask (gchar *mask)
{
    gchar *tmp = NULL;

    if (!mask || strlen(mask)<1)
        goto Bad_Mask;

    if ( strchr(mask,'/')!=NULL ) // Renaming directory is not yet available
        goto Bad_Mask;

    while (mask)
    {
        if ( (tmp=strrchr(mask,'%'))==NULL )
        {
            /* There is no more code. */
            /* No code in mask is accepted. */
            goto Good_Mask;
        }
        if (strlen(tmp)>1 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* The code is valid. */
            /* No separator is accepted. */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }
    }

    Bad_Mask:
        if (mask) g_free(mask);
        return 0;

    Good_Mask:
        if (mask) g_free(mask);
        return 1;
}


/*
 * Check if the entered mask is valid, else display the mask status icon
 */
void Scan_Check_Mask (GtkObject *wid_emit)
{
    gchar *mask = NULL;
    GtkWidget *target = NULL;
    gboolean hide_it;


    /* Select the target to display/hide if mask is bad/good
     * NOTE: targets must be set to NULL when ScannerWindow is destroyed */
    if (wid_emit == GTK_OBJECT(ScannerScanTagMaskCombo))
    {
        mask = g_strdup(gtk_entry_get_text_1(ScannerScanTagMaskCombo));
        target = MaskStatusIconBox;
    }else if (wid_emit == GTK_OBJECT(ScannerRenameFileMaskCombo))
    {
        mask = g_strdup(gtk_entry_get_text_1(ScannerRenameFileMaskCombo));
        target = MaskStatusIconBox1;
    }else if (wid_emit == GTK_OBJECT(MaskEditorEntry))
    {
        mask = g_strdup(gtk_entry_get_text_1(MaskEditorEntry));
        target = MaskStatusIconBox2;
    }
    else
        return;


    /* Select and get result of check scanner */
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        hide_it = Scan_Check_Scan_Tag_Mask(mask);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        hide_it = Scan_Check_Rename_File_Mask(mask);
    }else
        hide_it = 0;


    /* Display/hide the target */
    if (hide_it)
    {
        if (ScannerWindow && target)
            gtk_widget_hide(target);
    } else
    {
        if (ScannerWindow && target)
            gtk_widget_show(target);
    }
}


void Scan_Toggle_Legend_Button (void)
{
    if (!LegendButton && !LegendFrame) return;

    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(LegendButton)) )
        gtk_widget_show_all(LegendFrame);
    else
        gtk_widget_hide(LegendFrame);
}


void Scan_Toggle_Mask_Editor_Button (void)
{
    if (!MaskEditorButton && !MaskEditorFrame) return;

    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(MaskEditorButton)) )
    {
        gtk_widget_show_all(MaskEditorFrame);
        /* Update status of the icon box cause prev instruction show it for all cases */
        Scan_Check_Mask(GTK_OBJECT(MaskEditorEntry));
    }else
    {
        gtk_widget_hide(MaskEditorFrame);
    }
}



/*
 * Manage/Toggle check buttons into 'Process Fields' frame
 */
void Process_Fields_Check_Button_Toggled (GtkObject *object, GList *list)
{
    gint i = 0;
    
    if ( GTK_TOGGLE_BUTTON(object)->active )
    {
        while (list)
        {
            if ( list->data!=NULL && GTK_OBJECT(list->data)!=object )
                gtk_toggle_button_set_active((GtkToggleButton *)list->data,FALSE);
            i++;
            if (!list->next) break;
            list = list->next;
        }
    }
}


/*
 * Small buttons of Process Fields scanner
 */
void Select_Fields_Invert_Selection (void)
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),
                                !GTK_TOGGLE_BUTTON(ProcessFileNameField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),
                                !GTK_TOGGLE_BUTTON(ProcessTitleField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),
                                !GTK_TOGGLE_BUTTON(ProcessArtistField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),
                                !GTK_TOGGLE_BUTTON(ProcessAlbumField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField),
                                !GTK_TOGGLE_BUTTON(ProcessCommentField)->active);
}
void Select_Fields_Select_Unselect_All (void)
{
    static gboolean state = 1;
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField),state);
    state = !state;
}



/*
 *  Load a masks list into the scan editor clist
 */
void Mask_Editor_Clist_Load_Mask_List (GList *list)
{
    GList *tmplist;

    /* Clear the clist */
    gtk_clist_clear(GTK_CLIST(MaskEditorClist));

    tmplist = g_list_first(list);
    while (tmplist)
    {
        gchar *tmp;
        tmp = (gchar*)tmplist->data;
        gtk_clist_append(GTK_CLIST(MaskEditorClist),&tmp);
        tmplist = g_list_next(tmplist);
    }
    /* Select the first row */
    gtk_clist_select_row(GTK_CLIST(MaskEditorClist),0,0);
}


/*
 * Callbacks from Mask Editor buttons
 */
void Mask_Editor_Clist_Select_Row (GtkCList *clist,gint row,gint column,GdkEventButton *event,gpointer data)
{
    GList *SelectedRow = NULL;
    gchar *text = NULL;

    /* We must block the function, else the previous selected row will be modified */
    gtk_signal_handler_block_by_func(GTK_OBJECT(MaskEditorEntry),
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);

    /* Get the text of the last selected row */
    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    SelectedRow = g_list_last(SelectedRow);
    if (SelectedRow)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),(gint)SelectedRow->data,0,&text);
        if (text)
            gtk_entry_set_text(GTK_ENTRY(MaskEditorEntry),text);
    }

    gtk_signal_handler_unblock_by_func(GTK_OBJECT(MaskEditorEntry),
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
}

void Mask_Editor_Clist_Unselect_Row (void)
{
/*    gtk_signal_handler_block_by_func(GTK_OBJECT(MaskEditorEntry),
 *        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
 */
    gtk_entry_set_text(GTK_ENTRY(MaskEditorEntry),"");

/*    gtk_signal_handler_unblock_by_func(GTK_OBJECT(MaskEditorEntry),
 *        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
 */
}

gint Sort_List_Compare(gconstpointer a,gconstpointer b)
{
    if (a==NULL || b==NULL) return 0; /* Be carefull: a or b may have the value 0 */

    /* Sort from the bigger to the lower... */
    if ( (gint)a < (gint)b ) return 1;
    if ( (gint)a > (gint)b ) return -1;
    return 0;
}

void Mask_Editor_Clist_New (void)
{
    gchar *text = _("New_mask");

    gtk_clist_insert(GTK_CLIST(MaskEditorClist),0,&text);
}

void Mask_Editor_Clist_Copy (void)
{
    GList *SelectedRow = NULL;
    GList *SelectedRow_Sorted = NULL;
    gchar *text = NULL;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Copy: No row selected!\n"));
        return;
    }

    /* Duplicate and sort (numbers of the rows) the list.
     * The list is sorted to insert news masks into the same order of the clist */
    SelectedRow_Sorted = g_list_copy(SelectedRow);
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    SelectedRow_Sorted = g_list_sort(SelectedRow_Sorted,Sort_List_Compare);

    /* Replace number of rows into list by text */
    while(SelectedRow_Sorted)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),(gint)SelectedRow_Sorted->data,0,&text);
        SelectedRow_Sorted->data = g_strdup(text);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }

    /* Insert the new row(s) */
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    while(SelectedRow_Sorted)
    {
        gchar *tmp = g_strdup(SelectedRow_Sorted->data);
        gtk_clist_insert(GTK_CLIST(MaskEditorClist),0,&tmp);
        g_free(tmp);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }

    /* Free data */
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    while(SelectedRow_Sorted)
    {
        g_free(SelectedRow_Sorted->data);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }
    g_list_free(SelectedRow_Sorted);
}

void Mask_Editor_Clist_Add (void)
{
    gint i;
    gchar *text = NULL;


    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        for (i=0;i<(sizeof(Scan_Masks)/sizeof(Scan_Masks[0]));i++)
        {
            text = g_strdup(Scan_Masks[i]);
            gtk_clist_append(GTK_CLIST(MaskEditorClist),&text);
            g_free(text);
        }
    }

    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        for (i=0;i<(sizeof(Rename_File_Masks)/sizeof(Rename_File_Masks[0]));i++)
        {
            text = g_strdup(Rename_File_Masks[i]);
            gtk_clist_append(GTK_CLIST(MaskEditorClist),&text);
            g_free(text);
        }
    }
}

void Mask_Editor_Clist_Remove (void)
{
    GList *SelectedRow = NULL;
    GList *SelectedRow_Sorted = NULL;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Remove: No row selected!\n"));
        return;
    }

    /* Duplicate and sort (numbers of the rows) the list */
    SelectedRow_Sorted = g_list_copy(SelectedRow);
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    SelectedRow_Sorted = g_list_sort(SelectedRow_Sorted,Sort_List_Compare);

    while(SelectedRow_Sorted)
    {
        gtk_clist_remove(GTK_CLIST(MaskEditorClist),(gint)SelectedRow_Sorted->data);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }
    if ( (gint)SelectedRow_Sorted->data < ((GtkCList *)MaskEditorClist)->rows )
        gtk_clist_select_row(GTK_CLIST(MaskEditorClist),(gint)SelectedRow_Sorted->data,0);
    else
        gtk_clist_select_row(GTK_CLIST(MaskEditorClist),((GtkCList *)MaskEditorClist)->rows-1,0);
    g_list_free(SelectedRow_Sorted);
}

void Mask_Editor_Clist_Move_Up (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Move Up: No row selected!\n"));
        return;
    }

    while(SelectedRow)
    {
        row = (gint)SelectedRow->data;
        gtk_clist_row_move(GTK_CLIST(MaskEditorClist),row,row-1);
        if (!SelectedRow->next) break;
        SelectedRow = SelectedRow->next;
    }
}

void Mask_Editor_Clist_Move_Down (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Move Down: No row selected!\n"));
        return;
    }

    while(SelectedRow)
    {
        row = (gint)SelectedRow->data;
        gtk_clist_row_move(GTK_CLIST(MaskEditorClist),row,row+1);
        if (!SelectedRow->next) break;
        SelectedRow = SelectedRow->next;
    }
}

void Mask_Editor_Clist_Save_Button (void)
{
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        Mask_Editor_Create_Masks_List_From_Clist();
        Save_Scan_Tag_Masks_List(MasksList);
        /* Attach the new list to the combo box */
        if (!MasksList)
            MasksList = g_list_append(MasksList,"");
        gtk_combo_set_popdown_strings(GTK_COMBO(ScannerScanTagMaskCombo),MasksList);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        Mask_Editor_Create_Masks_List_From_Clist();
        Save_Rename_File_Masks_List(RenameFileMasksList);
        /* Attach the new list to the combo box */
        if (!RenameFileMasksList)
            RenameFileMasksList = g_list_append(RenameFileMasksList,"");
        gtk_combo_set_popdown_strings(GTK_COMBO(ScannerRenameFileMaskCombo),RenameFileMasksList);
    }
}


/*
 * Read rows of clist to recompose the masks list
 */
void Mask_Editor_Create_Masks_List_From_Clist (void)
{
/*    GList *tmplist; */
    gchar *text = NULL;
    gchar *text1 = NULL;
    gint row;
    gint row1;

    /* Free data of the list */
/*    MasksList = g_list_first(MasksList);
    tmplist = MasksList;
    g_print("len1:%d\n"),g_list_lenght(MasksList));
    while (tmplist)
    {
        g_free(tmplist->data);
        tmplist = g_list_next(tmplist);
    }
*/

    /* Suppress blank and duplicate row(s) */
    for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
        if (text!=NULL && strlen(text)==0)
        {
            gtk_clist_remove(GTK_CLIST(MaskEditorClist),row--);
            continue;
        }
        for (row1=row+1;row1<((GtkCList *)MaskEditorClist)->rows;row1++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row1,0,&text1);
            if (text1!=NULL && strcmp(text,text1)==0)
                gtk_clist_remove(GTK_CLIST(MaskEditorClist),row1--);

        }
    }


    /* Recreate list */
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        MasksList = NULL;
        for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
            if (text!=NULL)
                MasksList = g_list_append(MasksList,g_strdup(text));
        }
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        RenameFileMasksList = NULL;
        for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
            if (text!=NULL)
                RenameFileMasksList = g_list_append(RenameFileMasksList,g_strdup(text));
        }
    }
}


void Mask_Editor_Entry_Changed (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("No row selected!\n"));
        return;
    }
    row = (gint)SelectedRow->data;
    gtk_clist_set_text(GTK_CLIST(MaskEditorClist),row,0,gtk_entry_get_text_1(MaskEditorEntry));
}


/*
 * Actions when the a key is pressed into the masks editor clist
 */
void Mask_Editor_Clist_Key_Press (GtkWidget *widget, GdkEvent *event)
{
    if (event && event->type == GDK_KEY_PRESS)
    {
        GdkEventKey *kevent = (GdkEventKey *)event;

        switch(kevent->keyval)
        {
            case GDK_Delete:
                Mask_Editor_Clist_Remove();
                break;
/*            case GDK_Up:
                Mask_Editor_Clist_Move_Up();
                break;
            case GDK_Down:
                Mask_Editor_Clist_Move_Down();
                break;
*/        }
    }
}




/*
 * For the configuration file...
 */
void Scan_Apply_Changes (void)
{
    /* Then this variables must be set to NULL into function 'ScannerWindow_Quit' */

    if (ScannerWindow)
    {
        /* Group: select entries to process */
        PROCESS_FILENAME_FIELD = GTK_TOGGLE_BUTTON(ProcessFileNameField)->active;
        PROCESS_TITLE_FIELD    = GTK_TOGGLE_BUTTON(ProcessTitleField)->active;
        PROCESS_ARTIST_FIELD   = GTK_TOGGLE_BUTTON(ProcessArtistField)->active;
        PROCESS_ALBUM_FIELD    = GTK_TOGGLE_BUTTON(ProcessAlbumField)->active;
        PROCESS_COMMENT_FIELD  = GTK_TOGGLE_BUTTON(ProcessCommentField)->active;
        /* Group: convert one character */
        PF_CONVERT_INTO_SPACE = GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active;
        PF_CONVERT_SPACE      = GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active;
        /* Group: capitalize */
        PF_CONVERT_ALL_UPPERCASE = GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase)->active;
        PF_CONVERT_ALL_DOWNCASE  = GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase)->active;
        PF_CONVERT_FIRST_LETTER_UPPERCASE  = GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase)->active;
        PF_CONVERT_FIRST_LETTERS_UPPERCASE = GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase)->active;
        /* Group: remove/insert space */
        PF_REMOVE_SPACE   = GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active;
        PF_INSERT_SPACE   = GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active;
        PF_ONLY_ONE_SPACE = GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace)->active;
    }
}


/*
 * Send a message to the statubar to display status of scanner
 */
void Scan_Display_Info (void)
{
    if (!ScannerWindow)
        Statusbar_Message(_("The Scanner window isn't opened."),TRUE);
    else
        Statusbar_Message(_("Press the Scan button again to run the scanner."),TRUE);
}


/*
 * Returns 1 if item match with the current active option menu item, else returns 0.
 */
gboolean Scanner_Option_Menu_Get_Active_Menu_Item (Scanner_Option_Menu_Items_Value item)
{
    GtkWidget *child;
    gchar *text;

    if (ScannerOptionMenu && GTK_BIN(ScannerOptionMenu)->child)
    {
        child = GTK_BIN(ScannerOptionMenu)->child;
        if (GTK_IS_LABEL(child))
            gtk_label_get(GTK_LABEL(child), &text);
        else
            return 0;
    }else
    {
        return 0;
    }

    if ( item >= 0 && item < (sizeof(Scanner_Option_Menu_Items)/sizeof(Scanner_Option_Menu_Items[0])) )
    {
        /* Do not forget gettext codes to enable the comparison */
        if (strcmp(text,_(Scanner_Option_Menu_Items[item]))==0)
            return 1;
        else
            return 0;
    }else
        return 0;
}


/*
 * Function when you select an item of the option menu
 */
void Scanner_Option_Menu_Activate_Item (GtkWidget *widget, Scanner_Option_Menu_Items_Value item)
{
    switch (item)
    {
        case SCANNER_FILL_TAG:
            gtk_widget_show(MaskEditorButton);
            gtk_widget_show(LegendButton);
            gtk_widget_show(ScanTagFrame);
            gtk_widget_hide(RenameFileFrame);
            gtk_widget_hide(ProcessFieldsFrame);
            Mask_Editor_Clist_Load_Mask_List(MasksList);
            break;
        case SCANNER_RENAME_FILE:
            gtk_widget_show(MaskEditorButton);
            gtk_widget_show(LegendButton);
            gtk_widget_hide(ScanTagFrame);
            gtk_widget_show(RenameFileFrame);
            gtk_widget_hide(ProcessFieldsFrame);
            Mask_Editor_Clist_Load_Mask_List(RenameFileMasksList);
            Scan_Rename_File_Generate_Preview();
            break;
        case SCANNER_PROCESS_FIELDS:
            gtk_widget_hide(MaskEditorButton);
            gtk_widget_hide(LegendButton);
            gtk_widget_hide(ScanTagFrame);
            gtk_widget_hide(RenameFileFrame);
            gtk_widget_show_all(ProcessFieldsFrame);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(MaskEditorButton),FALSE);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(LegendButton),FALSE);
            Mask_Editor_Clist_Load_Mask_List(NULL);
            break;
        
    }
}


/*
 * Init the position of the scanner window
 */
void Scan_Set_Scanner_Window_Init_Position (void)
{
    gint x,y;
    gint width,height;
    gint MIN_X, MAX_X,MIN_Y,MAX_Y;


    if (!MainWindow && !ScannerWindow) return;

    if (SET_SCANNER_WINDOW_POSITION)
    {
        gtk_widget_realize(ScannerWindow);
        gdk_window_get_position(MainWindow->window,&x,&y);
        /* Calculate init position */
        x = x + SCANNER_WINDOW_X;
        y = y + SCANNER_WINDOW_Y;

        /* Check if init position doesn't exceed screen boundary */
        gdk_window_get_size(ScannerWindow->window,&width,&height);
        /* Boundaries */
        MIN_X = 0;
        MAX_X = gdk_screen_width() - width;
        MIN_Y = 20; /* "~20" for WM border size */
        MAX_Y = gdk_screen_height() - height;
        if (x<=MIN_X) x = MIN_X;
        if (x>=MAX_X) x = MAX_X;
        if (y<=MIN_Y) y = MIN_Y;
        if (y>=MAX_Y) y = MAX_Y;
        /* Now, move the window */
        gdk_window_move(ScannerWindow->window,x,y);
    }
}





/*
 * For DEBUGGING: Display data (text) of a masks list
 */
void Scan_Display_Data_List (GList *list)
{
    GList *list_save;
    guint item = 0;

    if (!list) g_print("list is NULL\n");
    g_print("\n");

    list_save = list;
    list = g_list_first(list);
    while (list)
    {
        g_print("(%2d) text:'%s'\n",item,(gchar *)list->data);
        list = g_list_next(list);
        item++;
    }
    list = list_save;
}
void Scan_Display_Data_List_Masks_List (void)
{
    Scan_Display_Data_List(MasksList);
}

