/* ogg_tag.c - 2001/11/08 */
/*
 *  EasyTAG - Tag editor for MP3 and OGG files
 *  Copyright (C) 2001-2002  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h> // For definition of ENABLE_OGG

#ifdef ENABLE_OGG

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#include <unistd.h>

#include "easytag.h"
#include "ogg_tag.h"
#include "vcedit.h"
#include "et_core.h"
#include "misc.h"
#include "setting.h"
#include "charset.h"
#include "mylocale.h"


/***************
 * Declaration *
 ***************/

#define MULTIFIELD_SEPARATOR " - "

/* OGG fields names :
 *  - TITLE        : Track name
 *  - VERSION      : The version field may be used to differentiate multiple version of the same track title in a single collection. (e.g. remix info)
 *  - ALBUM        : The collection name to which this track belongs
 *  - TRACKNUMBER  : The track number of this piece if part of a specific larger collection or album
 *  - ARTIST       : Track performer
 *  - ORGANIZATION : Name of the organization producing the track (i.e. the 'record label')
 *  - DESCRIPTION  : A short text description of the contents
 *  - GENRE        : A short text indication of music genre
 *  - DATE         : Date the track was recorded
 *  - LOCATION     : Location where track was recorded
 *  - COPYRIGHT    : Copyright information
 *  - ISRC         : ISRC number for the track; see the ISRC intro page for more information on ISRC numbers.
 *
 * Field names should not be 'internationalized'; this is a concession to simplicity
 * not an attempt to exclude the majority of the world that doesn't speak English.
 * Field *contents*, however, are represented in UTF-8 to allow easy representation
 * of any language.
 */



/**************
 * Prototypes *
 **************/
gboolean Ogg_Tag_Write_File (FILE *file_in, gchar *filename_in, vcedit_state *state);


/*************
 * Functions *
 *************/

/*
 * Read tag data into an ogg file.
 * Note:
 *  - if field is found but contains no info (strlen(str)==0), we don't read it
 */
gboolean Ogg_Tag_Read_File_Tag (gchar *filename, File_Tag *FileTag)
{
    FILE *file;
    vcedit_state *state;
    vorbis_comment *vc;
    gchar *string = NULL;
    gchar *string1 = NULL;
    guint field_num;


    if (!filename || !FileTag)
        return FALSE;

    ogg_error_msg = NULL;

    if ( (file=fopen(filename,"rb")) == NULL )
    {
        g_print(_("ERROR while opening file: '%s' (%s).\n\a"),filename,g_strerror(errno));
        return FALSE;
    }

    state = vcedit_new_state();    // Allocate memory for 'state'
    if ( vcedit_open(state,file) < 0 )
    {
        g_print(_("ERROR: Failed to open file: '%s' as vorbis (%s).\n"),filename,vcedit_error(state));
        ogg_error_msg = vcedit_error(state);
        fclose(file);
        return FALSE;
    }

    /* Get data from tag */
    vc = vcedit_comments(state);
    //{gint i; for (i=0;i<vc->comments;i++) g_print("%s -> ogg vc< '%s'\n",g_basename(filename),vc->user_comments[i]);}

    /*********
     * Title *
     *********/
    field_num = 0;
    while ( (string = vorbis_comment_query(vc,"title",field_num++)) != NULL )
    {
        if ( strlen(string)>0 )
        {
            string = convert_from_utf8(string);
            //Strip_String(string);
            if (FileTag->title==NULL)
                FileTag->title = g_strdup(string);
            else
                FileTag->title = g_strconcat(FileTag->title,MULTIFIELD_SEPARATOR,string,NULL);
        }
        if (string) g_free(string);
    }

    /**********
     * Artist *
     **********/
    field_num = 0;
    while ( (string = vorbis_comment_query(vc,"artist",field_num++)) != NULL )
    {
        if ( strlen(string)>0 )
        {
            string = convert_from_utf8(string);
            //Strip_String(string);
            if (FileTag->artist==NULL)
                FileTag->artist = g_strdup(string);
            else
                FileTag->artist = g_strconcat(FileTag->artist,MULTIFIELD_SEPARATOR,string,NULL);
        }
        if (string) g_free(string);
    }

    /*********
     * Album *
     *********/
    field_num = 0;
    while ( (string = vorbis_comment_query(vc,"album",field_num++)) != NULL )
    {
        if ( strlen(string)>0 )
        {
            string = convert_from_utf8(string);
            //Strip_String(string);
            if (FileTag->album==NULL)
                FileTag->album = g_strdup(string);
            else
                FileTag->album = g_strconcat(FileTag->album,MULTIFIELD_SEPARATOR,string,NULL);
        }
        if (string) g_free(string);
    }

    /********
     * Year *
     ********/
    if ( (string = vorbis_comment_query(vc,"date",0)) != NULL && strlen(string)>0 )
    {
        string = convert_from_utf8(string);
        //Strip_String(string);
        FileTag->year = g_strdup(string);
        g_free(string);
    }

    /*************************
     * Track and Total Track *
     *************************/
    if ( (string = vorbis_comment_query(vc,"tracknumber",0)) != NULL && strlen(string)>0 )
    {
        string = convert_from_utf8(string);
        //Strip_String(string);
        string1 = strchr(string,'/');
        if (NUMBER_TRACK_FORMATED)
        {
            if (string1)
            {
                FileTag->track_total = g_strdup_printf("%.2d",atoi(string1+1));
                *string1 = '\0';
            }
            FileTag->track = g_strdup_printf("%.2d",atoi(string));
        }else
        {
            if (string1)
            {
                FileTag->track_total = g_strdup(string1+1);
                *string1 = '\0';
            }
            FileTag->track = g_strdup(string);
        }
        g_free(string);
    }

    /*********
     * Genre *
     *********/
    field_num = 0;
    while ( (string = vorbis_comment_query(vc,"genre",field_num++)) != NULL )
    {
        if ( strlen(string)>0 )
        {
            string = convert_from_utf8(string);
            //Strip_String(string);
            if (FileTag->genre==NULL)
                FileTag->genre = g_strdup(string);
            else
                FileTag->genre = g_strconcat(FileTag->genre,MULTIFIELD_SEPARATOR,string,NULL);
        }
        if (string) g_free(string);
    }

    /***********
     * Comment *
     ***********/
    field_num = 0;
    string1 = NULL; // Cause it may be not updated into the 'while' condition
    while ( ((string  = vorbis_comment_query(vc,"comment",field_num)) != NULL )   // Winamp format
         || ((string1 = vorbis_comment_query(vc,"",       field_num)) != NULL ) ) // Xmms format
    {
        if ( string && strlen(string)>0 ) // Contains comment to Winamp format and we prefer this format (field name defined)
        {
            string = convert_from_utf8(string);
            //Strip_String(string);
            if (FileTag->comment==NULL)
                FileTag->comment = g_strdup(string);
            else
                FileTag->comment = g_strconcat(FileTag->comment,MULTIFIELD_SEPARATOR,string,NULL);
        }else if ( string1 && strlen(string1)>0 ) // Contains comment to Xmms format only
        {
            string1 = convert_from_utf8(string1);
           // Strip_String(string1);
            if (FileTag->comment==NULL)
                FileTag->comment = g_strdup(string1);
            else
                FileTag->comment = g_strconcat(FileTag->comment,MULTIFIELD_SEPARATOR,string1,NULL);
        }
        if (string)  g_free(string);
        if (string1) g_free(string1);
        string1 = NULL;
        field_num++;
    }


    vcedit_clear(state);
    fclose(file);

    return TRUE;
}



gboolean Ogg_Tag_Write_File_Tag (ET_File *ETFile)
{
    File_Tag *FileTag;
    gchar *filename_in;
    FILE *file_in;
    vcedit_state *state;
    vorbis_comment *vc;
    gchar *string, *string1;


    if (!ETFile || !ETFile->FileTag)
        return FALSE;

    FileTag     = (File_Tag *)ETFile->FileTag->data;
    filename_in = ((File_Name *)ETFile->FileNameCur->data)->value;
    ogg_error_msg = NULL;

    /* Test to know if we can write into the file */
    if ( (file_in=fopen(filename_in,"rb"))==NULL )
    {
        g_print(_("ERROR while opening file: '%s' (%s).\n\a"),filename_in,g_strerror(errno));
        return FALSE;
    }

    state = vcedit_new_state();    // Allocate memory for 'state'
    if ( vcedit_open(state,file_in) < 0 )
    {
        g_print(_("ERROR: Failed to open file: '%s' as vorbis (%s).\n"),filename_in,vcedit_error(state));
        ogg_error_msg = vcedit_error(state);
        fclose(file_in);
        return FALSE;
    }

    /* Get data from tag */
    vc = vcedit_comments(state);
    vorbis_comment_clear(vc);
    vorbis_comment_init(vc);

    /*********
     * Title *
     *********/
    if ( FileTag->title )
    {
        string  = g_strconcat("title=",FileTag->title,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /**********
     * Artist *
     **********/
    if ( FileTag->artist )
    {
        string  = g_strconcat("artist=",FileTag->artist,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /*********
     * Album *
     *********/
    if ( FileTag->album )
    {
        string  = g_strconcat("album=",FileTag->album,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /********
     * Year *
     ********/
    if ( FileTag->year )
    {
        string  = g_strconcat("date=",FileTag->year,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /*************************
     * Track and Total Track *
     *************************/
    if ( FileTag->track )
    {
        if ( FileTag->track_total && strlen(FileTag->track_total)>0 )
            string = g_strconcat("tracknumber=",FileTag->track,"/",FileTag->track_total,NULL);
        else
            string = g_strconcat("tracknumber=",FileTag->track,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /*********
     * Genre *
     *********/
    if ( FileTag->genre )
    {
        string  = g_strconcat("genre=",FileTag->genre,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);
    }

    /***********
     * Comment *
     ***********/
    // We write the comment using the "both" format
    if ( FileTag->comment )
    {
        // Format used in winamp plugin
        string = g_strconcat("comment=",FileTag->comment,NULL);
        string1 = convert_to_utf8(string);
        vorbis_comment_add(vc,string1);
        g_free(string);
        g_free(string1);

        if (OGG_TAG_WRITE_XMMS_COMMENT)
        {
            // Format used into xmms-1.2.5
            string  = g_strconcat("=",FileTag->comment,NULL);
            string1 = convert_to_utf8(string);
            vorbis_comment_add(vc,string1);
            g_free(string);
            g_free(string1);
        }
    }

    //{gint i; for (i = 0; i < vc->comments; i++) g_print("%s -> ogg vc< '%s'\n",g_basename(filename_in),vc->user_comments[i]);}

    /* Write tag */
    if ( Ogg_Tag_Write_File(file_in,filename_in,state) == FALSE )
    {
        g_print(_("ERROR: Failed to write comments to file '%s' (%s).\n"),filename_in,vcedit_error(state));
        ogg_error_msg = vcedit_error(state);
        return FALSE;
    }else
    {
        g_print(_("Written tag of '%s'\n"),g_basename(filename_in));
    }

    vcedit_clear(state);

    return TRUE;
}



/* This function is based on work of :
 *  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2001  Hvard Kvlen
 *  Copyright (C) 2001, Jorn Baayen <jorn@nl.linux.org>
 */
gboolean Ogg_Tag_Write_File (FILE *file_in, gchar *filename_in, vcedit_state *state)
{
    gchar *filename_tmp;
    gint   file_mkstemp;
    FILE  *file_out;
    gboolean return_code = TRUE;


    filename_tmp = g_strdup_printf("%s.XXXXXX",filename_in);

    if ((file_mkstemp = mkstemp(filename_tmp)) < 0)
    {
        g_free(filename_tmp);
        fclose(file_in);
        return FALSE;
    }

    if ( (file_out=fdopen(file_mkstemp,"wb")) == NULL )
    {
        close(file_mkstemp);
        remove(filename_tmp);
        g_free(filename_tmp);
        fclose(file_in);
        return FALSE;
    }

    if(vcedit_write(state,file_out) < 0)
        return_code = FALSE;

    fclose(file_in);

    if (fclose(file_out) != 0)
        return_code = FALSE;

    if ( (return_code==FALSE) || (rename(filename_tmp,filename_in)<0) )
    {
        remove(filename_tmp);
        return_code = FALSE;
    }

    g_free(filename_tmp);

    return return_code;
}





/***************
 * Header info *
 ***************/

gboolean Ogg_Header_Read_File_Info (gchar *filename, ET_File_Info *ETFileInfo)
{
    FILE *file;
    OggVorbis_File vf;
    vorbis_info *vi;
    gint encoder_version = 0;
    gint channels = 0;
    glong rate = 0;
    glong bitrate_upper = 0;
    glong bitrate_nominal = 0;
    glong bitrate_lower = 0;
    gdouble duration = 0;
    gulong filesize;


    if (!filename || !ETFileInfo)
        return FALSE;

    if ( (file=fopen(filename,"r"))==NULL )
    {
        g_print(_("ERROR while opening file: '%s' (%s).\n\a"),filename,g_strerror(errno));
        return FALSE;
    }

    if (ov_open(file,&vf,NULL,0) == 0)
    {
        if ( (vi=ov_info(&vf,0)) != NULL )
        {
            encoder_version = vi->version;         // Vorbis encoder version used to create this bitstream.
            channels        = vi->channels;        // Number of channels in bitstream.
            rate            = vi->rate;            // (Hz) Sampling rate of the bitstream.
            bitrate_upper   = vi->bitrate_upper;   // (b/s) Specifies the upper limit in a VBR bitstream.
            bitrate_nominal = vi->bitrate_nominal; // (b/s) Specifies the average bitrate for a VBR bitstream.
            //bitrate_nominal = ov_bitrate(&vf,-1);  // (b/s) Specifies the average bitrate for the specified logical bitstream.
            bitrate_lower   = vi->bitrate_nominal; // (b/s) Specifies the lower limit in a VBR bitstream.

            duration        = ov_time_total(&vf,-1); // (s) Total time.
        }
        ov_clear(&vf); // This close also the file
    }else
    {
        fclose(file);
    }

    filesize = Get_File_Size(filename);

    ETFileInfo->version    = encoder_version;
    ETFileInfo->bitrate    = bitrate_nominal/1000;
    ETFileInfo->samplerate = rate;
    ETFileInfo->mode       = channels;
    ETFileInfo->size       = filesize;
    ETFileInfo->duration   = duration;

    ETFileList_TotalSize     += filesize;
    ETFileList_TotalDuration += duration;

    return TRUE;
}


gboolean Ogg_Header_Display_File_Info_To_UI (gchar *filename, ET_File_Info *ETFileInfo)
{
    gchar *text;
    gchar *time = NULL;
    gchar *time1 = NULL;
    gchar *size = NULL;
    gchar *size1 = NULL;

    /* Encoder version */
    gtk_label_set_text(GTK_LABEL(VersionLabel),_("Encoder:"));
    text = g_strdup_printf("%d",ETFileInfo->version);
    gtk_label_set_text(GTK_LABEL(VersionValueLabel),text);
    g_free(text);

    /* Bitrate */
    text = g_strdup_printf(_("%d kb/s"),ETFileInfo->bitrate);
    gtk_label_set_text(GTK_LABEL(BitrateValueLabel),text);
    g_free(text);

    /* Samplerate */
    text = g_strdup_printf(_("%d Hz"),ETFileInfo->samplerate);
    gtk_label_set_text(GTK_LABEL(SampleRateValueLabel),text);
    g_free(text);

    /* Mode */
    gtk_label_set_text(GTK_LABEL(ModeLabel),_("Channels:"));
    text = g_strdup_printf("%d",ETFileInfo->mode);
    gtk_label_set_text(GTK_LABEL(ModeValueLabel),text);
    g_free(text);

    /* Size */
    size  = Convert_Size(ETFileInfo->size);
    size1 = Convert_Size(ETFileList_TotalSize);
    text  = g_strdup_printf("%s (%s)",size,size1);
    gtk_label_set_text(GTK_LABEL(SizeValueLabel),text);
    if (size)  g_free(size);
    if (size1) g_free(size1);
    g_free(text);

    /* Duration */
    time  = Convert_Duration(ETFileInfo->duration);
    time1 = Convert_Duration(ETFileList_TotalDuration);
    text  = g_strdup_printf("%s (%s)",time,time1);
    gtk_label_set_text(GTK_LABEL(DurationValueLabel),text);
    if (time)  g_free(time);
    if (time1) g_free(time1);
    g_free(text);

    return TRUE;
}

#endif /* ENABLE_OGG */

