/*
 * dsocks.c
 *
 * Copyright (c) 2003 Dug Song <dugsong@monkey.org>
 *
 * $Id: dsocks.c,v 1.7 2005/02/25 00:53:20 dugsong Exp $
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <dlfcn.h>
#include <err.h>
#include <errno.h>
#include <netdb.h>
#include <pwd.h>
#include <resolv.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "atomicio.h"
#include "dsocks.h"

typedef int (*connect_fn)(int, const struct sockaddr *, socklen_t);
typedef int (*getaddrinfo_fn)(const char *, const char *,
                              const struct addrinfo *, struct addrinfo **);
typedef struct hostent *(*gethostbyname_fn)(const char *name);

struct sockaddr_in _dsocks_sin, _dsocks_ns;
static char	   _dsocks_user[MAXLOGNAME + 1];
static connect_fn  _dsocks_connect, _sys_connect;
static getaddrinfo_fn _sys_getaddrinfo;
static gethostbyname_fn _sys_gethostbyname;
static int	 (*_sys_res_init)(void);

#define IS_LOOPBACK(sa)	(((struct sockaddr_in *)sa)->sin_addr.s_addr == \
			 htonl(INADDR_LOOPBACK))

static int
_dsocks4_connect(int fd, const struct sockaddr *sa, socklen_t slen)
{
	struct sockaddr_in *sin = (struct sockaddr_in *)sa;
	struct dsocks4_hdr *ds4;
	char buf[DSOCKS4_HDR_LEN + MAXLOGNAME + 1];
	int len;

	ds4 = (struct dsocks4_hdr *)buf;
	ds4->vn = DSOCKS4_VN_REQUEST;
	ds4->cd = DSOCKS4_CD_CONNECT;
	ds4->dport = sin->sin_port;
	ds4->dst = sin->sin_addr.s_addr;
	len = DSOCKS4_HDR_LEN;
	len += strlcpy(buf + len, _dsocks_user, MAXLOGNAME + 1) + 1;
	
	if (atomicio(write, fd, buf, len) != len) {
		warn("(dsocks4) error sending request");
	} else if (atomicio(read, fd, buf, DSOCKS4_HDR_LEN) !=
	    DSOCKS4_HDR_LEN) {
		warn("(dsocks4) error reading reply");
	} else if (ds4->vn != DSOCKS4_VN_REPLY) {
		warnx("(dsocks4) invalid reply");
	} else if (ds4->cd != DSOCKS4_CD_OK) {
		warnx("(dsocks4) proxy connection refused");
	} else
		return (0);
	
	return (-1);
}

static int
_dsocks5_error(int rep)
{
	if (rep == DSOCKS5_REP_NOTALLOWED) {
		errno = ECONNRESET;
	} else if (rep == DSOCKS5_REP_NETUNREACH) {
		errno = ENETUNREACH;
	} else if (rep == DSOCKS5_REP_HOSTUNREACH) {
		errno = EHOSTUNREACH;
	} else if (rep == DSOCKS5_REP_CONNREFUSED) {
		errno = ECONNREFUSED;
	} else if (DSOCKS5_REP_TTLEXPIRED) {
		errno = ETIMEDOUT;
	} else if (rep != DSOCKS5_REP_SUCCESS) {
		errno = ECONNABORTED;
	} else
		return (0);
	
	return (1);
}

static int
_dsocks5_connect(int fd, const struct sockaddr *sa, socklen_t slen)
{
	struct sockaddr_in *sin = (struct sockaddr_in *)sa;
	struct dsocks5_auth auth[1];
	struct dsocks5_msg msg[1];
	int len;

	auth->ver = 5;
	auth->nmeths = 1;
	auth->method = DSOCKS5_METHOD_NOAUTH;
	
	msg->ver = 5;
	msg->cmd = DSOCKS5_CMD_CONNECT;
	msg->rsv = 0;
	msg->atyp = DSOCKS5_ATYP_IPV4;
	msg->dst = sin->sin_addr.s_addr;
	msg->dport = sin->sin_port;
	len = DSOCKS5_MSG_LEN;

	if (atomicio(write, fd, (char *)auth, 3) != 3) {
		warn("(dsocks5) error sending auth request");
	} else if (atomicio(read, fd, (char *)auth, 2) != 2) {
		warn("(dsocks5) error reading auth reply");
	} else if (auth->ver != 5) {
		warnx("(dsocks5) invalid auth reply: version = %d", auth->ver);
	} else if (auth->nmeths != DSOCKS5_METHOD_NOAUTH) {
		warnx("(dsocks5) authentication required");
	} else if (atomicio(write, fd, (char *)msg, len) != len) {
		warn("(dsocks5) error sending connect request");
	} else if (atomicio(read, fd, (char *)msg, len) != len) {
		warn("(dsocks5) error reading connect reply");
	} else if (!_dsocks5_error(msg->cmd))
		return (0);
	
	return (-1);
}
	
int
connect(int fd, const struct sockaddr *sa, socklen_t len)
{
	struct sockaddr_in sin;
	
	if (sa->sa_family != AF_INET || IS_LOOPBACK(sa))
		return ((*_sys_connect)(fd, sa, len));
	
	if (htons(((struct sockaddr_in *)sa)->sin_port) != 53) {
		/* Tunnel non-DNS connection. */
		memcpy(&sin, sa, sizeof(sin));
	} else if (_dsocks_ns.sin_addr.s_addr != 0) {
		if (IS_LOOPBACK(&_dsocks_ns)) {
			/* Redirect to loopback nameserver. */
			return ((*_sys_connect)(fd,
				    (struct sockaddr *)&_dsocks_ns,
				    sizeof(_dsocks_ns)));
		} else	/* Tunnel DNS request. */
			memcpy(&sin, &_dsocks_ns, sizeof(sin));
	} else	/* "Leak" DNS request. */
		return ((*_sys_connect)(fd, sa, len));
	
	if ((*_sys_connect)(fd, (struct sockaddr *)&_dsocks_sin,
		sizeof(_dsocks_sin)) == 0) {
		return ((*_dsocks_connect)(fd, (struct sockaddr *)&sin,
			    sizeof(sin)));
	}
	warn("(dsocks) connect to proxy %s:%d",
	    inet_ntoa(_dsocks_sin.sin_addr), ntohs(_dsocks_sin.sin_port));
	
	return (-1);
}

static int
_sin_aton(char *str, struct sockaddr_in *sin, int default_port)
{
	uint32_t ip;
	uint16_t port;
	char *p;
	int ret = -1;
	
	if ((p = str = strdup(str)) != NULL) {
		ip = inet_addr(strsep(&p, ":"));
		port = p ? atoi(p) : default_port;
		if (ip != -1 && port != 0) {
			sin->sin_family = AF_INET;
			sin->sin_len = sizeof(sin);
			sin->sin_addr.s_addr = ip;
			sin->sin_port = htons(port);
			ret = 0;
		}
		free(str);
	}
	return (ret);
}

/* Redirect name lookups to local Tor proxy. */
struct tor_req {
	u_int8_t	v;
	u_int8_t	cmd;
	u_int16_t	port;
	u_int32_t	ip;
};

static u_int32_t
_tor_resolve(const char *name)
{
	struct tor_req *req;
	char *p, buf[512];
	u_int32_t ip = 0;
	int fd;
	
	req = (struct tor_req *)buf;
	req->v = 4;
	req->cmd = 0xF0;	/* resolve */
	req->port = 0;
	req->ip = htonl(1);	/* 0.0.0.1 */
	p = buf + sizeof(*req);
	*p++ = '\0';
	strcpy(p, name);
	p += strlen(name) + 1;
	
	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) != -1 &&
	    (*_sys_connect)(fd, (struct sockaddr *)&_dsocks_sin,
		sizeof(_dsocks_sin)) != -1) {
		if (send(fd, buf, p - buf, 0) >= 0) {
			if (recv(fd, buf, sizeof(buf), 0) == 8) {
				if (req->cmd == 90)
					ip = req->ip;
			}
		}
		close(fd);
	}
	return (ip);
}

struct hostent *
gethostbyname(const char *name)
{
	static char buf[BUFSIZ];
	struct hostent *h;
	u_int32_t *ip;

	if (getenv(DSOCKS_ENV_TOR)) {
		warnx("using tor gethostbyname");
		memset(buf, 0, sizeof(buf));
		h = (struct hostent *)buf;
		ip = (u_int32_t *)(h + 1);
		h->h_name = (char *)(ip + 2);
		strcpy(h->h_name, name);
		h->h_aliases = NULL;
		h->h_addrtype = AF_INET;
		h->h_length = 4;
		h->h_addr_list = (char **)&ip;
		
		if ((*ip = _tor_resolve(name)) != 0)
			return (h);
	}
	return (_sys_gethostbyname(name));
}

int
getaddrinfo(const char *hostname, const char *servname,
    const struct addrinfo *hints, struct addrinfo **res)
{
	struct in_addr addr;
	
	if (getenv(DSOCKS_ENV_TOR) && hostname) {
		warnx("using tor getaddrinfo");
		if ((addr.s_addr = _tor_resolve(hostname)) != 0) 
			hostname = inet_ntoa(addr);
	}
	return (_sys_getaddrinfo(hostname, servname, hints, res));
}

void
_dsocks_init(void)
{
	struct passwd *pw;
	char *env;
	void *libc;

	_dsocks_connect = _dsocks4_connect;
	
	if ((env = getenv(DSOCKS_ENV_VERSION)) != NULL) {
		int v = atoi(env);
		if (v == 5) {
			_dsocks_connect = _dsocks5_connect;
		} else if (v != 4)
			errx(1, "(dsocks) unsupported version %s", env);
	}
	if ((env = getenv(DSOCKS_ENV_PROXY)) != NULL) {
		if (_sin_aton(env, &_dsocks_sin, 1080) < 0)
			errx(1, "(dsocks) invalid proxy: %s", env);
	} else
		_sin_aton("127.0.0.1", &_dsocks_sin, 1080);
	
	if ((env = getenv(DSOCKS_ENV_NAMESERVER)) != NULL) {
		if (_sin_aton(env, &_dsocks_ns, 53) < 0)
			errx(1, "(dsocks) invalid nameserver: %s", env);
	}
	if ((pw = getpwuid(getuid())) != NULL) {
		strlcpy(_dsocks_user, pw->pw_name, sizeof(_dsocks_user));
	} else
		errx(1, "(dsocks) who are you?");

	if (!(libc = dlopen(DSOCKS_PATH_LIBC, DL_LAZY)))
		err(1, "(dsocks) couldn't dlopen %s", DSOCKS_PATH_LIBC);
	else if (!(_sys_connect = dlsym(libc, DSOCKS_SYM_CONNECT)))
		err(1, "(dsocks) couldn't dlsym '%s'", DSOCKS_SYM_CONNECT);
	else if (!(_sys_gethostbyname = dlsym(libc, DSOCKS_SYM_GETHOSTBYNAME)))
		err(1, "(dsocks) couldn't dlsym '%s'", DSOCKS_SYM_GETHOSTBYNAME);
	else if (!(_sys_getaddrinfo = dlsym(libc, DSOCKS_SYM_GETADDRINFO)))
		err(1, "(dsocks) couldn't dlsym '%s'", DSOCKS_SYM_GETADDRINFO);
	else if (!(_sys_res_init = dlsym(libc, DSOCKS_SYM_RES_INIT)))
		err(1, "(dsocks) couldn't dlsym '%s'", DSOCKS_SYM_RES_INIT);
}

int
res_init(void)
{
	int ret = (*_sys_res_init)();

	/* XXX - if nameserver set, use TCP for DNS lookups */
	if (_dsocks_ns.sin_addr.s_addr != 0)
		_res.options |= RES_USEVC;
	
	return (ret);
}
