/**
 * Contains functions to read /etc/resolv.conf style configuration files
 *
 * (c) NLnet Labs, 2004
 * 
 * See the file LICENSE for the license
 *
 */
    
#include "common.h"

struct t_hostlist_host *
hostlist_host_create (const char *ip_addr)
{
	struct t_hostlist_host *host;
	host = xmalloc(sizeof(struct t_hostlist_host));
	if (host == NULL) {
		error("Error allocating data in hostlist_host_create");
		return NULL;
	}
	host->ip_addr = xstrdup(ip_addr);
	host->names = zone_list_create();
	return host;
}

int
hostlist_host_add_name (struct t_hostlist_host *host, const char *hostname)
{
	assert(host != NULL);
	add_zone_list(host->names, hostname);
	return RET_SUC;
}

void
hostlist_host_destroy(struct t_hostlist_host *host)
{
	assert(host != NULL);
	xfree(host->ip_addr);
	zone_list_destroy(host->names);
	xfree(host);
}

struct t_hostlist *
hostlist_create(void)
{
	struct t_hostlist *hostlist = xmalloc(sizeof(struct t_hostlist));
	hostlist->size = 0;
	return hostlist;
}

int
hostlist_add_host(struct t_hostlist *hostlist, struct t_hostlist_host *host)
{
	assert(hostlist != NULL);
	assert(host != NULL);
	if (hostlist->size < 1000) {
		hostlist->hosts[hostlist->size] = host;
		hostlist->size++;
		return RET_SUC;
	} else {
		error("Maximum number of hosts (%d) reached", 1000);
		return RET_FAIL;
	}

}

void
hostlist_destroy(struct t_hostlist *hostlist)
{
	int i;
	assert(hostlist != NULL);
	for (i=0; i<hostlist->size; i++) {
		hostlist_host_destroy(hostlist->hosts[i]);
	}
	xfree(hostlist);
}

const char *
hostlist_get_ipaddr(struct t_hostlist *hostlist, const char *hostname)
{
	int i,j;
	char *hostlistname;
	assert(hostlist != NULL);
	for (i=0; i<hostlist->size; i++) {
		for (j=0; j<hostlist->hosts[i]->names->size; j++) {
			hostlistname = hostlist->hosts[i]->names->zones[j];
			if (strcmp(hostname, hostlistname) == 0) {
				return hostlist->hosts[i]->ip_addr;
			}
		}
	}
	return NULL;
}

void
hostlist_print (struct t_hostlist *hostlist)
{
	int i,j;
	assert(hostlist != NULL);
	for (i=0; i<hostlist->size; i++) {
		printf("%s: ", hostlist->hosts[i]->ip_addr);
		
		for (j=0; j<hostlist->hosts[i]->names->size; j++) {
			printf("%s ",hostlist->hosts[i]->names->zones[j]);
		}
		printf("\n");
	}
}

/* *
 * macro borrowed from libc
 * see http://sources.redhat.com/glibc/
 */
#define MATCH(line, name) \
        (!strncmp(line, name, sizeof(name) - 1) && \
        (line[sizeof(name) - 1] == ' ' || \
         line[sizeof(name) - 1] == '\t'))

#define FILEBUFSIZE 255

struct t_hostlist *
read_hosts_file(const char *filename)
{
	struct t_hostlist *hostlist;
	FILE *fp;
	char buf[BUFSIZE];
	char *str;
	size_t offset;
	size_t spacepos;
	
	hostlist = hostlist_create();

	fp = fopen(filename, "r");
	
	if (fp != NULL) {
		while (fgets(buf, FILEBUFSIZE, fp) != NULL) {
			if (*buf == ';' || *buf == '#') {
				continue;
			}
			offset = 0;
			
			spacepos = offset;
			
			while(buf[spacepos] != ' ') {
				spacepos++;
			}
			
			str = xmalloc(spacepos+1);
			strncpy(str, &buf[offset], spacepos);
			str[spacepos] = '\0';
			
			if (is_ip_address(str)) {
				struct t_hostlist_host *host = hostlist_host_create(str);
				while(spacepos > offset) {
					spacepos++;
					offset = spacepos;
					
					while(buf[spacepos] != ' ' && buf[spacepos] != '\n' && spacepos < strlen(buf)) {
						spacepos++;
					}
					
					if(spacepos > offset) {
						xfree(str);
						str = xmalloc(spacepos-offset+1);
						strncpy(str, &buf[offset], spacepos-offset);
						str[spacepos-offset] = '\0';
						hostlist_host_add_name(host, str);
					}
				}
				hostlist_add_host(hostlist, host);
			}
			xfree(str);
		}
	}
	return hostlist;
}

/**
 * Reads the given file for nameserver address(es)
 */
struct t_rr *
read_resolv_conf_ns(const char *filename)
{
	FILE *fp;
	char buf[FILEBUFSIZE];
	struct t_rr *ns = NULL;
	char *ip;
	size_t start, end;

	fp = fopen(filename, "r");
	if (fp != NULL) {
		while (fgets(buf, FILEBUFSIZE - 1, fp) != NULL) {
			if (*buf == ';') {
				continue;
			}
			if (MATCH(buf, "nameserver")) {
				
				start = 10;
				while (buf[start] == ' ' || buf[start] == '\t') { 
					/* skip whitespace */
					start++;
				}
				end = start;
				while (buf[end] != '\n' && buf[end] != '\0') {
					end++;
				}

				if (end > start) {
					ip = xmalloc(end - start + 1);
					strncpy(ip, &buf[start], end - start);
					
					/* remove trailing characters */
					ip[end - start] = '\0';
					/* TODO IP6 */
					if (ns == NULL) {
						ns = ns_create(NULL, (uint8_t *) ip, TYPE_A);
					} else {
						ns = ns_add_ip(ns, (uint8_t *) ip, TYPE_A);
					}
					xfree(ip);
				}
			}
		}
		(void) fclose(fp);
	}
	
	if (rr_size(ns) == 0) {
		if (have_drill_opt && drill_opt->transport == 6) {
			warning("No nameservers found in %s, assuming ::1", filename);
			/* avoid warnin with -Wwrite-strings */
			/* ip = "::1"; */
			ns = ns_create(NULL, (uint8_t *) "::1", TYPE_A);
		} else {
			warning("No nameservers found in %s, assuming 127.0.0.1", filename);
			ns = ns_create(NULL, (uint8_t *) "127.0.01" , TYPE_A);
		}
	}
	return ns;
}

/**
 * Reads the given file for search entries
 */
struct zone_list *
read_resolv_conf_search(const char *filename)
{
	FILE *fp;
	char buf[FILEBUFSIZE];
	struct zone_list *sz;
	char *str;
	size_t start, end;
	
	sz = zone_list_create();

	fp = fopen(filename, "r");
	if (fp != NULL) {
		while (fgets(buf, FILEBUFSIZE, fp) != NULL) {
			if (*buf == ';') {
				continue;
			}
			if (MATCH(buf, "domain") || MATCH(buf, "search")) {
				start = 6;
				while (buf[start] == ' ' || buf[start] == '\t') { 
					/* skip whitespace */
					start++;
				}
				end = start;
				while (buf[end] != '\n' && buf[end] != '\0') {
					end++;
				}
				
				if (end > start) {
					str = xmalloc(end - start + 2);
					memset(str, 0, end-start+2);
					strncpy(str, &buf[start], end-start);
					str[end-start] = '\0';
					verbose("found search zone: '%s'", str);
					add_zone_list(sz, str);
					xfree(str);
				}
			}
		}
		(void) fclose(fp);
	} else {
		error("Reading file: %s\n", filename);
		perror("failed to open");
		exit(0);
	}
	
	return sz;
}


/**
 * reads the given file for 'trusted-key' entries
 */
struct t_rr *
read_resolv_conf_dnskey(const char *filename)
{
	FILE *fp;
	char buf[FILEBUFSIZE];
	struct t_rr *dnskey = NULL;
	char *str;
	size_t start, end;
	
	fp = fopen(filename, "r");
	if (fp != NULL) {
		while (fgets(buf, FILEBUFSIZE, fp) != NULL) {
			if (*buf == ';') {
				continue;
			}
			if (MATCH(buf, "trusted-key")) {
				start = 11;
				while (buf[start] == ' ' || buf[start] == '\t') { 
					/* skip whitespace */
					start++;
				}
				end = start;
				while (buf[end] != '\n' && buf[end] != '\0') {
					end++;
				}
				
				if (end > start) {
					str = xmalloc(end - start);

					if (dnskey == NULL) {
						dnskey = pubkey_fromstring(&buf[start]);
					} else {
						(void) rr_add_rr(dnskey, pubkey_fromstring(&buf[start]));
					}
					xfree(str);
				}
			}
		}
		(void) fclose(fp);
	}
	return dnskey;
}

/**
 * Reads the given file for secure_zone entries
 */
struct zone_list *
read_resolv_conf_must_be_secure(const char *filename)
{
	FILE *fp;
	char buf[FILEBUFSIZE];
	struct zone_list *sz;
	char *str;
	size_t start, end;
	
	sz = zone_list_create();

	fp = fopen(filename, "r");
	if (fp != NULL) {
		while (fgets(buf, FILEBUFSIZE, fp) != NULL) {
			if (*buf == ';') {
				continue;
			}
			if (MATCH(buf, "must-be-secure")) {
				start = 14;
				while (buf[start] == ' ' || buf[start] == '\t') { 
					/* skip whitespace */
					start++;
				}
				end = start;
				while (buf[end] != '\n' && buf[end] != '\0') {
					end++;
				}
				
				if (end > start) {
					str = xmalloc(end - start);
					memset(str, 0, end-start);
					strncpy(str, &buf[start], end-start);
					str[end-start] = '\0';
					verbose("found secure zone: '%s'", str);
					add_zone_list(sz, str);
				}
			}
		}
		(void) fclose(fp);
	}
	
	return sz;
}
