/* 
 * librec.c
 *
 * (c) NLnet Labs, 2004
 * 
 * See the file LICENSE for the license
 *
 */

#include "common.h"
#include "libres.h"

struct t_rr *nameservers = NULL;
struct t_rr *trusted_keys = NULL;
struct zone_list *sz = NULL;
struct zone_list *searchlist = NULL;

void
print_addrinfo(const struct addrinfo *ai)
{
	printf("Addri:");
	if (ai) {
		printf("\tflags: %d\n", ai->ai_flags);
		printf("\tfamily: %d\n", ai->ai_family);
		printf("\tsocktype: %d\n", ai->ai_socktype);
		printf("\tprotocol: %d\n", ai->ai_protocol);
		printf("\taddrlen: %d\n", (int) ai->ai_addrlen);
		if (ai->ai_addr != NULL) {
			printf("\tAddr type: %d\n", (int) ai->ai_addr->sa_family);
			if (ai->ai_addr->sa_family == AF_INET) {
				struct sockaddr_in *addr = (struct sockaddr_in *) ai->ai_addr;
				
				printf("\taddress: %s\n", bin2ipv4((uint8_t *)&(addr->sin_addr)));
				printf("\tport: %d\n", (int) ntohs(addr->sin_port));
			} else if (ai->ai_addr->sa_family == AF_INET6) {
				struct sockaddr_in6 *addr = (struct sockaddr_in6 *) ai->ai_addr;
				
				printf("\taddress: %s\n", bin2ipv6((uint8_t *)&(addr->sin6_addr)));
				printf("\tport: %d\n", (int) ntohs(addr->sin6_port));
			} else {
				printf("Family unknown, not printing\n");
			}
		} else {
			printf("\tNo addr given\n");
		}
		
		printf("\tName: %s\n", ai->ai_canonname);
		
		printf("\tnext: %p\n", ai->ai_next);
		if (ai->ai_next) {
			print_addrinfo(ai->ai_next);
		}	
	} else {
		printf(" (nil)\n");
	}
}

int
drill_init(void)
{
	/* if sresolv.conf specifies nameserver(s) use those
	 * instead of the ones in /etc/resolv.conf
	 */
	struct t_rr *secure_override_nameserver = NULL;
	
	if (nameservers == NULL) {
		/* init TODO centralize, add sz etc and make consistent */
		nameservers = read_resolv_conf_ns("/etc/resolv.conf");
		trusted_keys = read_resolv_conf_dnskey("/etc/sresolv.conf");
		sz = read_resolv_conf_must_be_secure("/etc/sresolv.conf");

		/*secure_override_nameserver = read_resolv_conf_ns("/etc/sresolv.conf");*/
		if (secure_override_nameserver) {
			nameservers = secure_override_nameserver;
		}
	}
	
	searchlist = read_resolv_conf_search("/etc/resolv.conf");
	
	drill_opt_create();

	return RET_SUC;
}

/*
 * drill equivalents of standard DNS functions
 * if type=0 -> any AF type
 */
struct hostent *
drill_gethostbyname(const char *name, int type)
{
	struct hostent *hostent = NULL;
	struct t_rr *rrs;
	struct t_rr *rrnext;
	int i;
	int protocol = PROTO_ANY;
	
	verbose("gethostbyname(\"%s\") called\n", name);
	hostent = xmalloc(sizeof(hostent));
	
	if (is_ip_address(name)) {
		/* copy address and inaddr equiv */
		hostent->h_name = xmalloc(strlen(name)+1);
		strncpy(hostent->h_name, name, strlen(name));

		if (is_ipv6_addr(name)) {
			hostent->h_addrtype = AF_INET6;
			hostent->h_length = 16;
			hostent->h_addr_list = xmalloc(2);
			hostent->h_addr_list[0] = xmalloc(16);
			
			ipv62bin(name, (uint8_t *) hostent->h_addr_list[0]);
			
			hostent->h_addr_list[1] = NULL;
		} else {
			hostent->h_addrtype = AF_INET;
			hostent->h_length = 4;
			hostent->h_addr_list = xmalloc(2);
			hostent->h_addr_list[0] = xmalloc(4);
			
			ipv42bin(name, (uint8_t *) hostent->h_addr_list[0]);
			
			hostent->h_addr_list[1] = NULL;
		}
	} 
	
	drill_init();
	
	/* TODO: check result? */
	resolve2rr((char *)name, nameservers, trusted_keys, searchlist, sz, protocol,
				type, &rrs);

	if (rrs == NULL) {
		return NULL;
	}

	hostent->h_name = rdata2str(rrs->name);

	hostent->h_aliases = xmalloc(rr_size(rrs));

	i=0;
	rrnext = rrs->next;
	while(rrnext != NULL) {
		if (rrnext->type == TYPE_CNAME) {
			hostent->h_aliases[i] = rdata2str(rrnext->name);
			rrnext = rrnext->next;
		}
		i++;
	}

	hostent->h_aliases[i] = NULL;

	/* TODO: not only use first */
	if (rrs->type == TYPE_A) {
			hostent->h_addrtype = AF_INET;
			hostent->h_length = 4;
			hostent->h_addr_list = xmalloc(2);
			hostent->h_addr_list[0] = xmalloc(4);
			
			ipv42bin(rdata2str(rrs->rdata[0]), (uint8_t *) hostent->h_addr_list[0]);
			
			hostent->h_addr_list[1] = NULL;
	} else if (rrs->type == TYPE_AAAA) {
			hostent->h_addrtype = AF_INET6;
			hostent->h_length = 16;
			hostent->h_addr_list = xmalloc(2);
			hostent->h_addr_list[0] = xmalloc(16);
			
			ipv62bin(rdata2str(rrs->rdata[0]), (uint8_t *) hostent->h_addr_list[0]);
			
			hostent->h_addr_list[1] = NULL;
	
	} else {
		error("type not A or AAAA");
	}
	return hostent;
}

int
rr2addrinfo (struct t_rr *rrs, int protocol, int port, const struct addrinfo *hints, struct addrinfo **res)
{
	struct addrinfo *result = xmalloc(sizeof(struct addrinfo));
	char *adrstr;

	if (hints) {
		result->ai_flags = hints->ai_flags;
	} else {
		/* values spied from normal getaddrinfo
		 * hmmz they don't work on OpenBSD
		 * result->ai_flags = AI_V4MAPPED | AI_ADDRCONFIG;
		 */
		result->ai_flags = 0;
	}
	
	if (rrs->type == TYPE_AAAA) {
		result->ai_family = AF_INET6;
		
		if (hints) {
			result->ai_socktype = hints->ai_socktype;
		} else {
			/* TODO: determine socket type */
			result->ai_socktype = 1;
		}

		/* TODO: how to determine protocol? */
		result->ai_protocol = protocol;
		
		result->ai_addrlen = (socklen_t) sizeof(struct sockaddr_in6);

#ifndef S_SPLINT_S
		struct sockaddr_in6 *addr = xmalloc(sizeof(struct sockaddr_in6));
#endif
		addr->sin6_port = htons((uint16_t) port);
		addr->sin6_family = AF_INET6;

		adrstr = rdata2str(rrs->rdata[0]);

		ipv62bin(adrstr, (uint8_t *)&(addr->sin6_addr));

		result->ai_addr = (struct sockaddr *) addr;

		xfree(adrstr);
		
		/* TODO: next */
		result->ai_next = NULL;
	} else if (rrs->type == TYPE_A) {
		result->ai_flags = 0;
		result->ai_family = AF_INET;
		
		if (hints) {
			result->ai_socktype = hints->ai_socktype;
		} else {
			/* TODO: determine socket type */
			result->ai_socktype = 1;
		}
		
		/* TODO: how to determine protocol? */
		result->ai_protocol = protocol;
		
		result->ai_addrlen = (socklen_t) sizeof(struct sockaddr_in);
#ifndef S_SPLINT_S
		struct sockaddr_in *addr = xmalloc(sizeof(struct sockaddr_in));
#endif
		addr->sin_port = (uint16_t) htons((uint16_t) port);
		addr->sin_family = AF_INET;

		adrstr = rdata2str(rrs->rdata[0]);

		ipv42bin(adrstr, (uint8_t *)&(addr->sin_addr));

		//result->ai_addr=xmalloc(sizeof(struct sockaddr *));
		result->ai_addr = (struct sockaddr *) addr;
		
		xfree(adrstr);
		
		/* TODO: next */
		result->ai_next = NULL;
	} else if (rrs->type == TYPE_CNAME) {

		result->ai_flags = 0;
		result->ai_family = AF_INET;
		
		if (hints) {
			result->ai_socktype = hints->ai_socktype;
		} else {
			/* TODO: determine socket type */
			result->ai_socktype = 1;
		}
		
		/* TODO: how to determine protocol? */
		result->ai_protocol = protocol;
		
		result->ai_addrlen = 0;
		
		result->ai_addr = NULL;
		
		/* TODO: next */
		result->ai_next = NULL;
	}
	
	/* TODO when multi?
	   if (rrs->next != NULL) {
		rrs2addrinfo(rrs->next, protocol, port, hints, &(result->ai_next));	
	   }
	*/
	
	if (hints) {
		if (hints->ai_flags & AI_CANONNAME) {
			/* i think every implementation puts something else here... */
			result->ai_canonname = rdata2str(rrs->name);
		} else {
			result->ai_canonname = NULL;
		}
	} else {
			result->ai_canonname = NULL;
	}
	
	*res = result;

	return RET_SUC;
}

int
drill_getaddrinfo(const char *node, const char *service, const struct addrinfo *hints, struct addrinfo **res)
{
	struct t_rr *node_rr;
	struct addrinfo *result, *orig_result, *last_result;
	int port = 0, protocol = 0;
	int lookup_result = 0;

	char *proto = NULL;
	int type = 0;
	
	/* Check if socket type is ok */
	/* TODO: more checks? */
	if (hints) {
		/* waarom al die exits? */
		if (hints->ai_socktype == 0 && service != NULL) {
			*res = NULL;
			printf("aou\n");
			exit(0);
			return -1;
			return EAI_SERVICE; /* SERVICE not supported for i_socktype'.  */
		}
		
		if (hints->ai_protocol == IPPROTO_UDP) {
			proto = "udp";
			printf("protocol udp\n");
			exit(0);
		} else if (hints->ai_protocol == IPPROTO_TCP) {
			proto = "tcp";
			printf("protocol tcp\n");
			exit(0);
		}

		if (hints->ai_family == PF_INET) {
			type = AF_INET;
		} else if (hints->ai_family == PF_INET6) {
			type = AF_INET6;
		}
		
		
		/* TODO: check AI_NUMERICHOST flag */
		/* TODO: check AI_CANONNAME? flag */
		/* TODO: check AI_PASSIVE flag */
	}
	
	if (getenv("LIBRES_VERBOSE")) {
		verbose("PROTOCOL is %d for service %s", protocol, service);
	}
	
	/* TODO: find port in /etc/services? */
	if (service != NULL) {
		struct servent *serv;
		
		serv = getservbyport((int) htons((uint16_t) atoi(service)), NULL);
		
		if (serv == NULL) {
			serv = getservbyname(service, NULL);
		}

		if (serv ==NULL) {
			port = atoi(service);
			if (getenv("LIBRES_VERBOSE")) {
				warning("no service given, default to 0");
			}
			protocol = 0;
		} else {
			port = (int) ntohs((uint16_t) serv->s_port);
#ifndef S_SPLINT_S
			struct protoent *protoent;
#endif
			protoent = getprotobyname(serv->s_proto);
			if (protoent != NULL) {
				protocol = protoent->p_proto;
				if (getenv("LIBRES_VERBOSE")) {
					verbose("protocol set to %d by getprotobyname", protocol);
				}
			} else {
				if (getenv("LIBRES_VERBOSE")) {
					warning("protocol not found, default to 0");
				}
				protocol = 0;
			}
		}


	}

	if (getenv("LIBRES_VERBOSE")) {
		verbose("PROTOCOL is %d\n", protocol);
	}

	if (port < 0) {
		*res = NULL;
		return -8;
	}

	drill_init();
	
	if (getenv("LIBRES_VERBOSE")) {
		drill_opt->verbose = 2;
	}

	/* TODO: check result? */
	lookup_result = resolve2rr((char *)node, nameservers, trusted_keys, searchlist, sz, PROTO_ANY,
				type, &node_rr);
	
	if (getenv("LIBRES_VERBOSE")) {
		verbose("RESULT: %d\n", lookup_result);

		printf("All rrs:\n");
		print_rr(node_rr, FOLLOW);
		printf("end all rrs\n");
	}
	
	orig_result = NULL;
	last_result = NULL;

	while (node_rr) {
		if (node_rr->type == TYPE_A || node_rr->type == TYPE_AAAA) {
			if (protocol == 0 || protocol == IPPROTO_UDP) {
				if (getenv("LIBRES_VERBOSE")) {
					verbose("Adding udp: ");
					print_rr(node_rr, NO_FOLLOW);
				}
				rr2addrinfo (node_rr, IPPROTO_UDP, port, hints, &result);
				if (last_result) {
					last_result->ai_next = result;
				} else {
					orig_result = result;
				}
				last_result = result;
				result = NULL;
			}
			if (protocol == 0 || protocol == IPPROTO_TCP) {
				if (getenv("LIBRES_VERBOSE")) {
					verbose("Adding tcp: %d ", protocol);
					print_rr(node_rr, NO_FOLLOW);
				}
				rr2addrinfo (node_rr, IPPROTO_TCP, port, hints, &result);
				if (last_result) {
					last_result->ai_next = result;
				} else {
					orig_result = result;
				}
				last_result = result;
				result = NULL;
			}
		}
		node_rr = node_rr->next;
	}
	*res = orig_result;

	return lookup_result;
}

extern struct hostent *
gethostbyname(const char *name)
{
	struct hostent *result;

	verbose("gethostbyname called");

	result = drill_gethostbyname(name, 0);
	
	return result;
}
extern struct hostent *
gethostbyname2(const char *name, int type)
{
	verbose("gethostbyname2 called (type %d)", type);

	return drill_gethostbyname(name, type);
}


extern int getaddrinfo(const char *node, const char *service, const struct addrinfo *hints, struct addrinfo **res)
{
	int result;
	
	if (getenv("LIBRES_SHOW_HINTS") || getenv("LIBRES_SHOW")) {
		if (hints == NULL) {
			verbose("No hints?!?");
		} else {
			printf("Hints:\n");
			print_addrinfo((struct addrinfo *)hints);
			printf("end of hints\n\n");
		}
	}
	result = drill_getaddrinfo(node, service, hints, res);
	
	if (getenv("LIBRES_SHOW") || getenv("LIBRES_SHOW_RESULT")) {
		printf("result: %d (%s)\n", result, gai_strerror(result));
		print_addrinfo(*res);
		printf("end of result\n\n");
	}

	if (getenv("LIBRES_STOP")) {
		exit(0);
	}
	return result;
}    

int main(int argc, char **argv) {
	warning("Libdrill should not be executred directly\n");
	return 1;
}
