/* chaselogic.c
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 */

#include "common.h"
#include <assert.h>

/**
 * Tries to find the zone of the given owner name, i.e. strips the first label
 *
 * Don't forget to free the result somewhere :)
 */
char *
guess_zone(char *name, int len)
{
	char *result = NULL;
	int i, pos = -1;
	for(i = 0; i < len; i++) {
		if (name[i] == '.') {
			if (i > 0 && i < len && name[i -1] != '\\') {
				pos = i+1;
				break;
			}
		}
	}
	
	/* normally we don't want to copy the found ., but we do if we
	 * encounter the root zone 
	 */
	if (pos > -1 && pos < len) { result = xmalloc((size_t)len - pos);
		strncpy(result, name + pos, (size_t)len - pos); result[len - pos] = '\0';
	} else if (pos > -1 && pos == len) {
		result = xmalloc(2);
		result[0] = '.';
		result[1] = '\0';
	}
	
	return result;
}

/* TODO eigenlijk wil je een do_chase op een rr+rrsig... (isdazo?) */
/**
 * Performs a signature chase on the given rr
 */
int
do_chase_rr(struct t_rr *data, struct t_dpacket *packet, struct t_rr
		*nameserver, struct t_rr *trusted_keys, int protocol)
{
	int result;
	char *name = rdata2str(data->name);
	
	result = do_chase_name(name, data->type, nameserver, trusted_keys, protocol);
	
	xfree(name);
	
	return result;
	
	
}

/**
 * Performs a query + sigchase on the given owner name
 */
int
do_chase_name(char *name, uint16_t type, struct t_rr *nameserver, struct t_rr
		*trusted_keys, int protocol)
{
	struct t_rr *query;
	struct t_dpacket *answer;
	struct t_rr *rrset = NULL;
	struct t_rr *sig;
	struct t_rdata *name_rdata;
	struct t_rdata *key_owner;
	struct t_rr *key;
	struct t_rr *goodkeys = NULL;
	struct t_rr *goodsigs = NULL;
	struct t_rr *origsig, *origkey;
	struct t_rr *badsigs = NULL;
	struct t_rr *tk = NULL;

	char *tmpstr;
	char *cname_target = NULL;
	char *zone;
	int result;

	if (drill_opt->debug) {
		tmpstr = typebyint(type, ztypes);
		verbose("chasing: [%s] [%s]\n\t", name, tmpstr);
		xfree(tmpstr);
	}
	
	name_rdata = rdata_create((uint8_t*)name, strlen(name));
	query = rr_create(name_rdata,
		type, DEF_TTL, SEC_QUESTION);
	
	answer = do_query_rr(query, nameserver, protocol);
	rr_destroy(query, FOLLOW);

	if (!answer) {
		tmpstr = typebyint(type, ztypes);
		error("%s %s not found\n", name, tmpstr);
		xfree(tmpstr);
		rdata_destroy(name_rdata);
		/* gvd answer is NULL here */
		/* dpacket_destroy(answer); */
		return RET_FAIL;
	}
	
	if (answer->flags.rcode + 1 != RCODE_NOERROR) {
		error("Received error packet: %d (%s)\n",
			answer->flags.rcode + 1, 
			namebyint(answer->flags.rcode + 1, rcodes));
		print_packet(answer);
		rdata_destroy(name_rdata);
		dpacket_destroy(answer);
		return RET_FAIL;
	}
	
	rrset = dpacket_get_rrset(name_rdata, type, answer, SEC_ANSWER);

	if (!rrset) {
		/* answer might be a cname */
		rrset = dpacket_get_rrset(name_rdata, TYPE_CNAME, answer, SEC_ANSWER);
		if (!rrset) {
			error("Unable to find %s record for %s\n", namebyint(type,
				ztypes), name);
			rdata_destroy(name_rdata);
			dpacket_destroy(answer);
			return RET_FAIL;
		} else {
			cname_target = rdata2str(rrset->rdata[0]);
		} 
	}

	/* have sig? */
	sig = dpacket_get_rrsig(rrset, answer);
	
	if (sig == NULL) {
		/* go get it */
		query = rr_create(name_rdata, TYPE_RRSIG, DEF_TTL, SEC_QUESTION);
		//rr_destroy(rrset, FOLLOW);
		//rrset = dpacket_get_rrset(name_rdata, TYPE_RRSIG, answer, SEC_ANSWER);
		dpacket_destroy(answer);
		answer = do_query_rr(query, nameserver, protocol);
		rr_destroy(query, FOLLOW);
		sig = dpacket_get_rrsig(rrset, answer);
	}
	
	rdata_destroy(name_rdata);
	
	if (sig == NULL) {
		error("No signature for the data (%s, %s) could be found\n", name, namebyint(type, ztypes));
		dpacket_destroy(answer);
		if (cname_target) {
			xfree(cname_target);
		}
		return RET_FAIL;
	}

	/* find signing dnskey */
	key_owner = rdata_clone(sig->rdata[7]);

	query = rr_create(key_owner, TYPE_DNSKEY, DEF_TTL, SEC_QUESTION);
	if (answer) {
		dpacket_destroy(answer);
	}
	
	answer = do_query_rr(query, nameserver, protocol);
	rr_destroy(query, FOLLOW);
	key = dpacket_get_rrset(key_owner, TYPE_DNSKEY, answer, SEC_ANSWER);
	if (answer)
		dpacket_destroy(answer);

	if (key == NULL) {
		error("Unable to find dnskey\n");
		if (cname_target) {
			xfree(cname_target);
		}
		return RET_FAIL;
	}

	origsig = sig;
	origkey = key;	
	/* remember which keys and signatures were ok */
	while (key != NULL) {
		while (sig != NULL) {
			if (have_drill_opt && drill_opt->verbose) {
				mesg("Verifying sig:");
				prettyprint_rr(sig, NO_FOLLOW, 0, 0);
				mesg("with key:");
				prettyprint_rr(key, NO_FOLLOW, 0, 0);
			}
			
			if (verify_rrsig(rrset, sig, key) == RET_SUC) {
				if (goodkeys == NULL) {
					goodkeys = rr_clone(key, NO_FOLLOW);
				} else {
					(void)rr_add_rr(goodkeys, rr_clone(key, NO_FOLLOW));
				}
				if (goodsigs == NULL) {
					goodsigs = rr_clone(sig, NO_FOLLOW);
				} else {
					(void)rr_add_rr(goodsigs, rr_clone(sig, NO_FOLLOW));
				}
				break;
			} else {
				if (have_drill_opt && drill_opt->verbose) {
					mesg("Bad signature");
				}
			}
			sig = sig->next;
		}
		key = key->next;
		sig = origsig;

	}

	/* print which sigs did not work out */
	sig = origsig;
	if (goodsigs) {
		while (sig != NULL) {
			if (!rr_get_rr(goodsigs, sig)) {
				if (badsigs == NULL) {
					badsigs = rr_clone(sig, NO_FOLLOW);
				} else {
					(void)rr_add_rr(badsigs, rr_clone(sig, NO_FOLLOW));
				}
			}
			sig = sig->next;
		}
	} else {
		badsigs = rr_clone(sig, FOLLOW);
	}
	
	if (rr_size(badsigs) > 0) {
		mesg("Bad signatures:");
		prettyprint_rr(badsigs, FOLLOW, 1, 30);
	}

	/* maybe one of the good keys is trusted */
	tk = keyset_contains_key(trusted_keys, goodkeys);

	if (goodkeys)
		rr_destroy(goodkeys, FOLLOW);
	if (goodsigs)
		rr_destroy(goodsigs, FOLLOW);
	if (origsig)
		rr_destroy(origsig, FOLLOW);
	if (origkey)
		rr_destroy(origkey, FOLLOW);
		
	if (tk) {
		origkey = tk;
		while(tk) {
			tk = tk->next;
		}
		if (cname_target) {
			xfree(cname_target);
		}
		rr_destroy(rrset, FOLLOW);
		rr_destroy(origkey, FOLLOW);
		rdata_destroy(key_owner);
		return RET_SUC;
	}
	/* if CNAME, chase name, otherwise chase zone upwards */
	if (cname_target) {
		result = do_chase_name(cname_target, type, nameserver, trusted_keys, protocol);
		xfree(cname_target);
	} else {
		/* find DS for this key
		 *
		 * key has probably got DS in upper zone, let's assume there are no
		 * weird things in here and we can safely chop off one label 
		 * zone = guess_zone(key_owner->data, key_owner->length);
		 */
		 /* TODO: this probably only works for 1 key still */
		zone = rdata2str(key_owner);
		if (zone == NULL) {
			error("Unable to determine zone\n");
		}

		/* lather, rinse, repeat */
		result = do_chase_name(zone, TYPE_DS, nameserver, trusted_keys,
				protocol);
		
		xfree(zone);
	}

	rdata_destroy(key_owner);
	
	if (result == RET_FAIL) {
		mesg("No secure delegation paths to a trusted entry point found");
	}

	if (rrset) {
		rr_destroy(rrset, FOLLOW);
	}
	return result;
}
