/* AIPlayer.c   Code for dopewars computer players                      */
/* Copyright (C)  1998-2000  Ben Webb                                   */
/*                Email: ben@bellatrix.pcl.ox.ac.uk                     */
/*                WWW: http://bellatrix.pcl.ox.ac.uk/~ben/dopewars/     */

/* This program is free software; you can redistribute it and/or        */
/* modify it under the terms of the GNU General Public License          */
/* as published by the Free Software Foundation; either version 2       */
/* of the License, or (at your option) any later version.               */

/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */

/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston,               */
/*                   MA  02111-1307, USA.                               */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include "dopeos.h"
#include "dopewars.h"
#include "message.h"
#include "clientside.h"
#include "AIPlayer.h"

#if NETWORKING
#define NUMNAMES      8
#define MINSAFECASH   300
#define MINSAFEHEALTH 40

/* Reserve some space for picking up new guns */
#define SPACERESERVE  10

/* Locations of the loan shark, bank, gun shop and pub      */
/* Note: these are not the same as the global variables     */
/*       LoanSharkLoc, BankLoc, GunShopLoc and RoughPubLoc, */
/*       which are set locally. The remote server could     */
/*       have different locations set, and the AI must work */
/*       out where these locations are for itself.          */
int RealLoanShark,RealBank,RealGunShop,RealPub;

void AIPlayerLoop() {
/* Main loop for AI players. Connects to server, plays game, */
/* and then disconnects.                                     */
   char *pt,text[BUFLEN];
   struct PLAYER *AIPlay;
   fd_set readfs;
   AIPlay=AddPlayer(0,&FirstClient);
   if (!AIPlay) {
      printf("Error: Cannot allocate memory. AI Player terminating \
abnormally.\n");
      return;
   }
   printf("AI Player started; attempting to contact server ");
   printf("at %s:%d...\n",ServerName,Port);
   pt=SetupNetwork();
   if (pt) {
      printf("Error: Could not connect to dopewars server\n(%s)\n",pt);
      printf("AI Player terminating abnormally.\n");
      return;
   }
   AISetName(AIPlay);
   printf("Connection established\n");

   /* Forget where the "special" locations are */
   RealLoanShark=RealBank=RealGunShop=RealPub=-1;

   while (1) {
      FD_ZERO(&readfs);
      FD_SET(ClientSock,&readfs);
      if (bselect(ClientSock+1,&readfs,NULL,NULL,NULL)==-1) {
         if (errno==EINTR) continue;
         printf("Error in select\n"); exit(1);
      }
      if (FD_ISSET(ClientSock,&readfs)) {
         pt=bgets(text,BUFLEN,ClientSock);
         chomp(text);
         if (!pt) {
            printf("Connection to server lost!\n");
            ShutdownNetwork();
            break;
         } else {
            if (HandleAIMessage(text,AIPlay)) {
               ShutdownNetwork();
               break;
            }
         }
      }
   }
   sleep(1);
   printf("AI Player terminated OK.\n");
}

void AISetName(struct PLAYER *AIPlay) {
/* Chooses a random name for the AI player, and informs the server */
   char *AINames[NUMNAMES] = {
      "Chip", "Dopey", "Al", "Dan", "Bob", "Fred", "Bert", "Jim"
   };
   sprintf(AIPlay->Name,"AI) %s",AINames[brandom(0,NUMNAMES)]);
   SendClientMessage(NULL,C_NONE,C_NAME,NULL,AIPlay->Name);
   printf("Using name %s\n",AIPlay->Name);
}

int HandleAIMessage(char *Message,struct PLAYER *AIPlay) {
/* Performs appropriate processing on an incoming network message */
/* "Message" for AI player "AIPlay". Returns 1 if the game should */
/* be ended as a result, 0 otherwise.                             */
   char Data[BUFLEN],Version[20],Code,AICode,WasFighting,*pt,wrd[BUFLEN];
   struct PLAYER *From,*To,*tmp;
   struct timeval tv;
   if (ProcessMessage(Message,&From,&AICode,&Code,&To,Data,FirstClient)==-1) {
      printf("Bad network message. Oops.\n"); return 0;
   }
   if (!HandleGenericClientMessage(From,AICode,Code,To,Data)) switch(Code) {
      case C_ENDLIST:
         printf("Players in this game:-\n");
         for (tmp=FirstClient;tmp;tmp=tmp->Next) {
            printf("    %s\n",tmp->Name);
         }
         break;
      case C_INIT:
         strcpy(Version,VERSION);
         pt=Data;
         pt=ExtractWord(wrd,BUFLEN,pt);
         if (strcmp(Version,wrd)!=0) {
            printf("This server is version %s, while your client is \
version %s.\nBe warned that different versions may not be fully compatible!\n\
Refer to the website at http://bellatrix.pcl.ox.ac.uk/~dopewars/\n\
for the latest version.",wrd,Version);
         }
         pt=ExtractWord(wrd,BUFLEN,pt);
         Location=ResizeStruct(Location,&NumLocation,atoi(wrd),
                               sizeof(struct LOCATION));
         pt=ExtractWord(wrd,BUFLEN,pt);
         Gun=ResizeStruct(Gun,&NumGun,atoi(wrd),sizeof(struct GUN));
         pt=ExtractWord(wrd,BUFLEN,pt);
         Drug=ResizeStruct(Drug,&NumDrug,atoi(wrd),sizeof(struct DRUG));
         for (tmp=FirstClient;tmp;tmp=tmp->Next) UpdatePlayer(tmp);
         pt=ExtractWord(Names.Bitch,sizeof(Names.Bitch),pt);
         pt=ExtractWord(Names.Bitches,sizeof(Names.Bitches),pt);
         pt=ExtractWord(Names.Gun,sizeof(Names.Gun),pt);
         pt=ExtractWord(Names.Guns,sizeof(Names.Guns),pt);
         pt=ExtractWord(Names.Drug,sizeof(Names.Drug),pt);
         pt=ExtractWord(Names.Drugs,sizeof(Names.Drugs),pt);
         pt=ExtractWord(Names.Month,sizeof(Names.Month),pt);
         pt=ExtractWord(Names.Year,sizeof(Names.Year),pt);
         break;
      case C_NEWNAME:
         AISetName(AIPlay);
         break;
      case C_FIGHTPRINT:
         PrintAIMessage(Data);
         if (From!=&Noone) {
            AIPlay->Flags |= FIGHTING+CANSHOOT;
         }
         if (TotalGunsCarried(AIPlay)>0 && AIPlay->Health>MINSAFEHEALTH) {
            SendClientMessage(AIPlay,C_NONE,C_FIGHTACT,NULL,"F");
         } else {
            AIJet(AIPlay);
         }
         break;
      case C_PRINTMESSAGE:
         PrintAIMessage(Data);
         break;
      case C_MSG:
         printf("%s: %s\n",From->Name,Data);
         break;
      case C_MSGTO:
         printf("%s->%s: %s\n",From->Name,To->Name,Data);
         break;
      case C_JOIN:
         printf("%s joins the game.\n",Data);
         tmp=AddPlayer(0,&FirstClient);
         strcpy(tmp->Name,Data);
         break;
      case C_LEAVE:
         if (From!=&Noone) {
            printf("%s has left the game.\n",From->Name);
            RemovePlayer(From,&FirstClient);
         }
         break;
      case C_SUBWAYFLASH:
         /* Use bselect rather than sleep, as this is portable to Win32 */
         tv.tv_sec=AITurnPause;
         tv.tv_usec=0;
         bselect(0,NULL,NULL,NULL,&tv);
         printf("Jetting to %s with %s cash",
                Location[(int)AIPlay->IsAt].Name,
                FormatPrice(AIPlay->Cash,Data));
         printf(" and %s debt\n",FormatPrice(AIPlay->Debt,Data));
         if (brandom(0,100)<10) AISendRandomMessage(AIPlay);
         break;
      case C_UPDATE:
         WasFighting=FALSE;
         if (From==&Noone) {
            if (AIPlay->Flags & FIGHTING) WasFighting=TRUE;
            ReceivePlayerData(Data,To);
         } else {
            ReceivePlayerData(Data,From); /* spy reports */
         }
         if (!(AIPlay->Flags & FIGHTING) && WasFighting) {
            AIDealDrugs(AIPlay);
            AIJet(AIPlay);
         }
         if (AIPlay->Health==0) {
            printf("AI Player killed. Terminating normally.\n");
            return 1;
         }
         break;
      case C_DRUGHERE:
         ReceiveDrugsHere(Data,To);
         AIDealDrugs(AIPlay);
         AIJet(AIPlay);
         break;
      case C_GUNSHOP:
         AIGunShop(AIPlay);
         break;
      case C_LOANSHARK:
         AIPayLoan(AIPlay);
         break;
      case C_QUESTION:
         AIHandleQuestion(Data,AICode,AIPlay,From);
         break;
      case C_HISCORE: case C_STARTHISCORE:
         break;
      case C_ENDHISCORE:
         printf("Game time is up. Leaving game.\n");
         return 1;
      case C_PUSH:
         printf("AI Player pushed from the server.\n");
         return 1;
      case C_QUIT:
         printf("The server has terminated.\n");
         return 1;
      default:
         printf("%s^%c^%s%s\n",From->Name,Code,To->Name,Data);
         break;
   }
   return 0;
}

void PrintAIMessage(char *Text) {
/* Prints a message received via a printmessage or question */
/* network message, stored in "Text"                        */
   int i;
   char SomeText=0;
   for (i=0;i<strlen(Text);i++) {
      if (Text[i]=='^') {
         if (SomeText) putchar('\n'); 
      } else {
         putchar(Text[i]);
         SomeText=1;
      }
   }
   putchar('\n');
}

void AIDealDrugs(struct PLAYER *AIPlay) {
/* Buy and sell drugs for AI player "AIPlay" */
   price_t *Profit,MaxProfit;
   char text[BUFLEN];
   int i,LastHighest,Highest,Num,MinProfit;
   Profit = (price_t *)malloc(sizeof(price_t)*NumDrug);
   for (i=0;i<NumDrug;i++) {
      Profit[i]=AIPlay->Drugs[i].Price-(Drug[i].MaxPrice+Drug[i].MinPrice)/2;
   }
   MinProfit=0;
   for (i=0;i<NumDrug;i++) if (Profit[i]<MinProfit) MinProfit=Profit[i];
   MinProfit--;
   for (i=0;i<NumDrug;i++) if (Profit[i]<0) Profit[i]=MinProfit-Profit[i];
   LastHighest=-1;
   while (1) {
      MaxProfit=MinProfit;
      Highest=-1;
      for (i=0;i<NumDrug;i++) {
         if (Profit[i]>MaxProfit && i!=LastHighest && 
             (LastHighest==-1 || Profit[LastHighest]>Profit[i])) {
            Highest=i;
            MaxProfit=Profit[i];
         }
      }
      LastHighest=Highest;
      if (Highest==-1) break;
      Num=AIPlay->Drugs[Highest].Carried;
      if (MaxProfit>0 && Num>0) {
         printf("Selling %d %s at %s\n",Num,Drug[Highest].Name,
                FormatPrice(AIPlay->Drugs[Highest].Price,text));
         AIPlay->CoatSize+=Num;
         AIPlay->Cash+=Num*AIPlay->Drugs[Highest].Price;
         sprintf(text,"drug^%d^%d",Highest,-Num);
         SendClientMessage(AIPlay,C_NONE,C_BUYOBJECT,NULL,text);
      }
      if (AIPlay->Drugs[Highest].Price != 0 &&
          AIPlay->CoatSize>SPACERESERVE) {
         Num=AIPlay->Cash/AIPlay->Drugs[Highest].Price;
         if (Num>AIPlay->CoatSize-SPACERESERVE) {
            Num=AIPlay->CoatSize-SPACERESERVE;
         }
         if (MaxProfit<0 && Num>0) {
            printf("Buying %d %s at %s\n",Num,Drug[Highest].Name,
                   FormatPrice(AIPlay->Drugs[Highest].Price,text));
            sprintf(text,"drug^%d^%d",Highest,Num);
            AIPlay->CoatSize-=Num;
            AIPlay->Cash-=Num*AIPlay->Drugs[Highest].Price;
            SendClientMessage(AIPlay,C_NONE,C_BUYOBJECT,NULL,text);
         }
      }
   }
   free(Profit);
}

void AIGunShop(struct PLAYER *AIPlay) {
/* Handles a visit to the gun shop by AI player "AIPlay" */
   int i;
   int Bought;
   char text[BUFLEN];
   do {
      Bought=0;
      for (i=0;i<NumGun;i++) {
         if (TotalGunsCarried(AIPlay)<AIPlay->Bitches.Carried+2 &&
             Gun[i].Space<=AIPlay->CoatSize &&
             Gun[i].Price<=AIPlay->Cash-MINSAFECASH) {
            AIPlay->Cash-=Gun[i].Price;
            AIPlay->CoatSize-=Gun[i].Space;
            AIPlay->Guns[i].Carried++;
            Bought++;
            printf("Buying a %s for %s at the gun shop\n",Gun[i].Name,
                   FormatPrice(Gun[i].Price,text));
            sprintf(text,"gun^%d^1",i);
            SendClientMessage(AIPlay,C_NONE,C_BUYOBJECT,NULL,text);
         }
      }
   } while (Bought);
   SendClientMessage(AIPlay,C_NONE,C_DONE,NULL,NULL);
}

void AIJet(struct PLAYER *AIPlay) {
/* Decides on a new game location for AI player "AIPlay" and jets there */
   int NewLocation;
   char text[40];
   if (!AIPlay) return;
   NewLocation=AIPlay->IsAt;
   if (RealLoanShark>=0 && AIPlay->Cash > (price_t)((float)AIPlay->Debt*1.2)) {
      NewLocation=RealLoanShark;
   } else if (RealPub>=0 && brandom(0,100)<30 && AIPlay->Cash>MINSAFECASH*10) {
      NewLocation=RealPub;
   } else if (RealGunShop>=0 && brandom(0,100)<70 &&
              TotalGunsCarried(AIPlay)<AIPlay->Bitches.Carried+2 &&
              AIPlay->Cash>MINSAFECASH*5) {
      NewLocation=RealGunShop;
   }
   while (NewLocation==AIPlay->IsAt) NewLocation=brandom(0,NumLocation);
   sprintf(text,"%d",NewLocation);
   SendClientMessage(AIPlay,C_NONE,C_REQUESTJET,NULL,text);
}

void AIPayLoan(struct PLAYER *AIPlay) {
/* Pays off the loan of AI player "AIPlay" if this doesn't leave */
/* the player with insufficient funds for further dealing        */
   char text[BUFLEN];
   if (AIPlay->Cash-AIPlay->Debt >= MINSAFECASH) {
      pricetostr(AIPlay->Debt,text,BUFLEN);
      SendClientMessage(AIPlay,C_NONE,C_PAYLOAN,NULL,text);
      printf("Debt of %s paid off to loan shark\n",
             FormatPrice(AIPlay->Debt,text));
   }
   SendClientMessage(AIPlay,C_NONE,C_DONE,NULL,NULL);
}

void AISendAnswer(struct PLAYER *From,struct PLAYER *To,char *answer) {
/* Sends the answer "answer" from AI player "From" to the server,        */
/* claiming to be for player "To". Also prints the answer on the screen. */
   SendClientMessage(From,C_NONE,C_ANSWER,To,answer); puts(answer);
}

void AIHandleQuestion(char *Data,char AICode,struct PLAYER *AIPlay,
     struct PLAYER *From) {
/* Works out a sensible response to the question coded in "Data" and with */
/* computer-readable code "AICode", claiming to be from "From" and for AI */
/* player "AIPlay", and sends it                                          */
   char *Prompt,allowed[BUFLEN];
   if (From==&Noone) From=NULL;
   Prompt=Data;
   Prompt=ExtractWord(allowed,BUFLEN,Prompt);
   PrintAIMessage(Prompt);
   switch (AICode) {
      case C_ASKLOAN:
         if (RealLoanShark==-1) {
            printf("Loan shark located at %s\n",
                   Location[(int)AIPlay->IsAt].Name);
         }
         RealLoanShark=AIPlay->IsAt;
         AISendAnswer(AIPlay,From,"Y");
         break;
      case C_ASKGUNSHOP:
         if (RealGunShop==-1) {
            printf("Gun shop located at %s\n",
                   Location[(int)AIPlay->IsAt].Name);
         }
         RealGunShop=AIPlay->IsAt;
         AISendAnswer(AIPlay,From,"Y");
         break;
      case C_ASKPUB:
         if (RealPub==-1) {
            printf("Pub located at %s\n",Location[(int)AIPlay->IsAt].Name);
         }
         RealPub=AIPlay->IsAt;
         AISendAnswer(AIPlay,From,"Y");
         break;
      case C_ASKBITCH: case C_ASKRUN: case C_ASKGUN:
         AISendAnswer(AIPlay,From,"Y");
         break;
      case C_ASKRUNFIGHT:
         AISendAnswer(AIPlay,From,AIPlay->Health<MINSAFEHEALTH ? "R" : "F");
         break;
      case C_ASKBANK:
         if (RealBank==-1) {
            printf("Bank located at %s\n",Location[(int)AIPlay->IsAt].Name);
         }
         RealBank=AIPlay->IsAt;
         AISendAnswer(AIPlay,From,"N");
         break;
      case C_MEETPLAYER:
         if (TotalGunsCarried(AIPlay)>0) AISendAnswer(AIPlay,From,"A");
         else {
            AISendAnswer(AIPlay,From,"E");
            AIJet(AIPlay);
         }
         break;
      case C_ASKSEW:
         AISendAnswer(AIPlay,From,AIPlay->Health<MINSAFEHEALTH ? "Y" : "N");
         break;
      default:
         AISendAnswer(AIPlay,From,"N");
         break;
   }
}

void AISendRandomMessage(struct PLAYER *AIPlay) {
/* Sends a random message to all other dopewars players */
   char *RandomInsult[5]= {
      "Call yourselves drug dealers?",
      "A trained monkey could do better...",
      "Think you\'re hard enough to deal with the likes of me?",
      "Zzzzz... are you dealing in candy or what?",
      "Reckon I'll just have to shoot you for your own good."
   };
   SendClientMessage(AIPlay,C_NONE,C_MSG,NULL,RandomInsult[brandom(0,5)]);
}

#else /* NETWORKING */

void AIPlayerLoop() {
      puts("This binary has been compiled without networking support, and \
thus cannot act");
      puts("as an AI player. Recompile passing --enable-networking to the \
configure script.");
}

#endif /* NETWORKING */
