/* message.c  Message-handling routines for dopewars                    */
/* Copyright (C)  1998-2000  Ben Webb                                   */
/*                Email: ben@bellatrix.pcl.ox.ac.uk                     */
/*                WWW: http://bellatrix.pcl.ox.ac.uk/~ben/dopewars/     */

/* This program is free software; you can redistribute it and/or        */
/* modify it under the terms of the GNU General Public License          */
/* as published by the Free Software Foundation; either version 2       */
/* of the License, or (at your option) any later version.               */

/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */

/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston,               */
/*                   MA  02111-1307, USA.                               */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <stdlib.h>
#include "dopeos.h"
#include "dopewars.h"
#include "serverside.h"
#include "clientside.h"
#include "message.h"

/* dopewars is built around a client-server model. Each client handles the
   user interface, but all the important calculation and processing is
   handled by the server. All communication is conducted via. TCP by means
   of plain text newline-delimited messages.

   Message structure:-
       From^To^ACData

   From,To: Player names identifying the sender and intended recipient of
            the message. Either field may be blank, although the server will
            usually reject incoming messages if they are not properly
            identified with a correct "From" field.
   A:       One-letter code; used by AI players to identify the message subtype
                                       (check AIPlayer.h)
   C:       One-letter code to identify the message type (check message.h)
   Data:    Message-dependent information

   For example, a common message is the "printmessage" message (message code 
   C is C_PRINTMESSAGE), which simply instructs the client to display "Data". 
   Any ^ characters within Data are replaced by newlines on output. So in order 
   for the server to instruct player "Fred" to display "Hello world" it would 
   send the message:-
       ^Fred^AAHello world
   Note that the server has left the From field blank, and has specified the
   AI code 'A' - defined in AIPlayer.h as C_NONE (i.e. an "unimportant" 
   message) as well as the main code 'A', defined as C_PRINTMESSAGE in
   message.h

   When the network is down, a server is simulated locally. Client to server
   messages are simply passed directly to the server message handling routine
   in serverside.c, while server to client messages are queued in MessageList
   and read by the do_game loop within dopewars.c                        */

MessageList *FirstMessage=NULL;

void SendClientMessage(struct PLAYER *From,char AICode,char Code,
                       struct PLAYER *To,char *Data) {
/* Send a message from client player "From" with computer code "AICode",  */
/* human-readable code "Code" and data "Data". The message is sent to the */
/* server, identifying itself as for "To". From, To, or Data may be NULL. */
/*   FILE *outfile=NULL;*/
   int fd=0;
   char text[BUFLEN];
   struct PLAYER *ServerFrom;
   sprintf(text,"%s^%s^%c%c%s",From ? From->Name : "",
                              To ? To->Name : "",AICode,Code,Data ? Data : "");

#if NETWORKING
   if (!Network) {
#endif
      if (From) ServerFrom=GetPlayerByName(From->Name,FirstServer);
      else ServerFrom=NULL;
      HandleServerMessage(text,ServerFrom);
#if NETWORKING
   } else {
      strcat(text,"\n");
      if (Client) fd=ClientSock;
      else if (Server && To) fd=To->fd;
      if (fd==0) return;
      if (SendSocket(fd,text,strlen(text))!=strlen(text)) {
         fprintf(stderr,"Write error in SendClientMessage\n");
      }
   }
#endif /* NETWORKING */
}

void SendPrintMessage(struct PLAYER *From,char AICode,
                      struct PLAYER *To,char *Data) {
/* Shorthand for the server sending a "printmessage"; instructs the */
/* client "To" to display "Data"                                    */
   SendServerMessage(From,AICode,C_PRINTMESSAGE,To,Data);
}

void SendQuestion(struct PLAYER *From,char AICode,
                  struct PLAYER *To,char *Data) {
/* Shorthand for the server sending a "question"; instructs the client  */
/* "To" to display the second word of Data and accept any letter within */
/* the first word of Data as suitable reply                             */
   SendServerMessage(From,AICode,C_QUESTION,To,Data);
}

void SendServerMessage(struct PLAYER *From,char AICode,char Code,
                       struct PLAYER *To,char *Data) {
/* Sends a message from the server to client player "To" with computer    */
/* code "AICode", human-readable code "Code" and data "Data". The message */
/* will claim to be from or on behalf of player "From"                    */
   char text[BUFLEN];
   if (!Network) {
      sprintf(text,"%s^%s^%c%c%s",From ? From->Name : "",
                              To ? To->Name : "",AICode,Code,Data ? Data : "");
      AddWaitingClientMessage(To,text);
   } else SendClientMessage(From,AICode,Code,To,Data);
}

#if NETWORKING
char *bgets(char *buf,int maxlen,int fd) {
/* Drop-in substitute for fgets; reads up to maxlen bytes into buf from */
/* file descriptor fd. Used for non-blocking read from TCP sockets.     */
   ssize_t len;
   int i;
   for (i=0;i<maxlen;i++) {
      len=ReadSocket(fd,buf+i,1);
      if (len==0) return NULL;
      if (buf[i]=='\n') {
         buf[i+1]=0;
         return buf;
      }
   }
   buf[maxlen]=0;
   return buf;
}
#endif /* NETWORKING */

void chomp(char *str) {
/* Removes a terminating newline from "str", if one is present. */
   int len=strlen(str);
   if (str[len-1]=='\n') str[len-1]=0;
}

void BroadcastToClients(char AICode,char Code,char *Data,
                        struct PLAYER *From,struct PLAYER *Except) {
/* Sends the message made up of AICode,Code and Data to all players except */
/* "Except" (if non-NULL). It will be sent by the server, and on behalf of */
/* player "From"                                                           */
   struct PLAYER *tmp;
   for (tmp=FirstServer;tmp;tmp=tmp->Next) if (tmp!=Except) {
      SendServerMessage(From,AICode,Code,tmp,Data);
   }
}

void SendInventory(struct PLAYER *From,char AICode,char Code,
                   struct PLAYER *To,Inventory *Guns,Inventory *Drugs) {
/* Encodes an Inventory structure into a string, and sends it as the data */
/* with a server message constructed from the other arguments.            */ 
   int i;
   char text[20],buf[BUFLEN];
   buf[0]=0;
   for (i=0;i<NumGun;i++) {
      sprintf(text,"%d:",Guns ? Guns[i].Carried : 0);
      strcat(buf,text);
   }
   for (i=0;i<NumDrug;i++) {
      sprintf(text,"%d:",Drugs ? Drugs[i].Carried : 0);
      strcat(buf,text);
   }
   SendClientMessage(From,AICode,Code,To,buf);
}

void ReceiveInventory(char *Data,Inventory *Guns,Inventory *Drugs) {
/* Decodes a string representation (in "Data") to its original Inventory */
/* contents, and stores it in "Guns" and "Drugs" if non-NULL             */
   int i;
   char *pt,wrd[BUFLEN];
   pt=Data;
   for (i=0;i<NumGun;i++) {
      pt=ExtractWord(wrd,BUFLEN,pt);
      if (Guns) Guns[i].Carried=atoi(wrd);
   }
   for (i=0;i<NumDrug;i++) {
      pt=ExtractWord(wrd,BUFLEN,pt);
      if (Drugs) Drugs[i].Carried=atoi(wrd);
   }
}

void SendPlayerData(struct PLAYER *To) {
/* Sends all pertinent data about player "To" from the server to player "To" */
   SendSpyReport(NULL,To);
}

void SendSpyReport(struct PLAYER *To,struct PLAYER *SpiedOn) {
/* Sends pertinent data about player "SpiedOn" from the server to player "To" */
   char text[BUFLEN],buf[BUFLEN];
   int i;
   text[0]=0;
   strcat(text,pricetostr(SpiedOn->Cash,buf,BUFLEN));
   strcat(text,"^");
   strcat(text,pricetostr(SpiedOn->Debt,buf,BUFLEN));
   strcat(text,"^");
   strcat(text,pricetostr(SpiedOn->Bank,buf,BUFLEN));

   sprintf(buf,"^%d^%d^%d^%d^%d^",SpiedOn->Health,SpiedOn->CoatSize,
           SpiedOn->IsAt,SpiedOn->Turn,SpiedOn->Flags);
   strcat(text,buf);
   for (i=0;i<NumGun;i++) {
      sprintf(buf,"%d^",SpiedOn->Guns[i].Carried);
      strcat(text,buf);
   }
   for (i=0;i<NumDrug;i++) {
      sprintf(buf,"%d^",SpiedOn->Drugs[i].Carried);
      strcat(text,buf);
   }
   sprintf(buf,"%d",SpiedOn->Bitches.Carried); strcat(text,buf);
   if (To) SendServerMessage(SpiedOn,C_NONE,C_UPDATE,To,text);
   else SendServerMessage(NULL,C_NONE,C_UPDATE,SpiedOn,text);
}

void ReceivePlayerData(char *text,struct PLAYER *From) {
/* Decode player data from the string "text" into player "From" */
   char *cp,wrd[BUFLEN];
   int i;
   cp=text;
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Cash=atol(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Debt=atol(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Bank=atol(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Health=atoi(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->CoatSize=atoi(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->IsAt=atoi(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Turn=atoi(wrd);
   cp=ExtractWord(wrd,BUFLEN,cp);
   From->Flags=atoi(wrd);
   for (i=0;i<NumGun;i++) {
      cp=ExtractWord(wrd,BUFLEN,cp);
      From->Guns[i].Carried=atoi(wrd);
   }
   for (i=0;i<NumDrug;i++) {
      cp=ExtractWord(wrd,BUFLEN,cp);
      From->Drugs[i].Carried=atoi(wrd);
   }
   From->Bitches.Carried=atoi(cp);
}

char *ExtractWord(char *NewWord,int MaxLen,char *text) {
/* Extracts the first "word" - a sequence of characters delimited by ^   */
/* characters - from the string pointed to by "text". The word is placed */
/* in "NewWord" if found, and a pointer is returned to the next word. If */
/* no word is found, NULL is returned. No more than MaxLen-1 characters  */
/* are written into "NewWord" (plus a null terminator); a word longer    */
/* than this length is simply truncated.                                 */
/* WARNING: "text" is modified.                                          */
   return ExtractWordDelim(NewWord,MaxLen,text,'^');
}

char *ExtractWordDelim(char *NewWord,int MaxLen,char *text,char Delim) {
/* Extracts the first "word" - a sequence of characters delimited by the */
/* delimiter "Delim" - from the string pointed to by "text". The word is */
/* placed in "NewWord" if found, and a pointer is returned to the next   */
/* word. If no word is found, NULL is returned. "NewWord" is truncated   */
/* to MaxLen-1 characters if necessary. WARNING: "text" is modified.     */
   char *ind;
   if (!text) return NULL;
   ind=index(text,Delim);
   NewWord[0]=0;
   if (ind) {
      *ind=0;
      strncpy(NewWord,text,MaxLen-1);
      NewWord[MaxLen-1]='\0';
      return ind+1;
   }
   return text;
}

#if NETWORKING
char *SetupNetwork() {
/* Sets up the connection from the client to the server. If the connection */
/* is successful, Network and Client are set to TRUE, and ClientSock is a  */
/* file descriptor for the newly-opened socket. NULL is returned. If the   */
/* connection fails, a pointer to an error message is returned.            */
   struct sockaddr_in ClientAddr;
   struct hostent *he;
   static char NoHost[]="Could not find host";
   static char NoSocket[]="Could not create network socket";
   static char NoConnect[]="Connection refused or no server present";

   Network=Client=Server=FALSE;

   if ((he=gethostbyname(ServerName))==NULL) {
      return NoHost;
   }
   ClientSock=socket(AF_INET,SOCK_STREAM,0);
   if (ClientSock==-1) {
      return NoSocket;
   }

   ClientAddr.sin_family=AF_INET;
   ClientAddr.sin_port=htons(Port);
   ClientAddr.sin_addr=*((struct in_addr *)he->h_addr);
   memset(ClientAddr.sin_zero,0,sizeof(ClientAddr.sin_zero));

   if (connect(ClientSock,(struct sockaddr *)&ClientAddr,
       sizeof(struct sockaddr))==-1) {
      close(ClientSock);
      return NoConnect;
   }
/*   out=fdopen(ClientSock,"w"); */
   Client=TRUE; Network=TRUE;
   return NULL;
}
#endif /* NETWORKING */

void SwitchToSinglePlayer(struct PLAYER *Play) {
/* Called when the client is pushed off the server, or the server  */
/* terminates. Using the client information, starts a local server */
/* to reproduce the current game situation as best as possible so  */
/* that the game can be continued in single player mode            */
   struct PLAYER *NewPlayer;
   if (!Network || !Client || !FirstClient) return;
   if (Play!=FirstClient) {
      clear_bottom();
      mvaddstr(20,0,"Oops! FirstClient should be player!");
      nice_wait();
      exit(1);
   }
   while (FirstClient->Next) RemovePlayer(FirstClient->Next,&FirstClient);
/*   fclose(out);*/
   close(ClientSock);
   CleanUpServer();
   Network=Server=Client=FALSE;
   print_status(Play,TRUE); refresh();
   NewPlayer=AddPlayer(0,&FirstServer);
   CopyPlayer(NewPlayer,Play);
   NewPlayer->Flags=0;
   NewPlayer->EventNum=E_ARRIVE;
   SendEvent(NewPlayer);
}

void ShutdownNetwork() {
/* Closes down the client side of the network connection. Clears the list */
/* of client players, and closes the network socket.                      */
   while (FirstClient) RemovePlayer(FirstClient,&FirstClient);
#if NETWORKING
   if (Client) {
      close(ClientSock);
   }
#endif /* NETWORKING */
   Client=Network=Server=FALSE;
}

int ProcessMessage(char *Msg,struct PLAYER **From,char *AICode,char *Code,
                   struct PLAYER **To,char *Data,struct PLAYER *First) {
/* Given a "raw" message in "Msg" and a pointer to the start of the linked   */
/* list of known players in "First", sets the other arguments to the message */
/* fields. Returns 0 on success, -1 on failure.                              */
   char *pt,wrd[BUFLEN];
   struct PLAYER *tmp;
   pt=Msg;
   pt=ExtractWord(wrd,BUFLEN,pt);
   if (!pt || (tmp=GetPlayerByName(wrd,First))==NULL) return -1;
   *From=tmp;
   pt=ExtractWord(wrd,BUFLEN,pt);
   if (!pt || (tmp=GetPlayerByName(wrd,First))==NULL) return -1;
   *To=tmp;
   if (!pt || strlen(pt)<2) {
      Data[0]=0; *AICode=C_NONE; *Code=C_PRINTMESSAGE;
   } else {
      strcpy(Data,pt+2);
      *AICode=pt[0];
      *Code=pt[1];
   }
   return 0;
}

void AddWaitingClientMessage(struct PLAYER *To,char *Text) {
/* Adds a message in "Text" and for player "To" the local message queue; */
/* only required when the network is down and a local server is running. */
   MessageList *mp,*newmess;
   newmess = (MessageList *)malloc(sizeof(MessageList));
   if (!newmess) { 
      fprintf(stderr,"AddWaitingClientMessage: Out of memory!\n"); 
      return; 
   }
   newmess->Next=NULL;
   strcpy(newmess->Message,Text);
   if (FirstMessage) {
      mp=FirstMessage; while (mp->Next) mp=mp->Next;
      mp->Next=newmess; 
   } else {
      FirstMessage=newmess;
   }
/*puts("call done");*/
}

char CheckWaitingClientMessage(struct PLAYER *To,char *Text) {
/* Returns TRUE if a message is waiting in the queue for player "To";     */
/* in this case the message is removed from the list and returned in Text */
   MessageList *mp;
   if (FirstMessage) {
      mp=FirstMessage;
      FirstMessage=mp->Next;
      strcpy(Text,mp->Message);
      free(mp); 
      return TRUE; 
   } else {
      Text[0]=0;
      return FALSE;
   } 
}

void ClearWaitingClientMessages() {
/* Clears the local message queue */
   MessageList *mp,*mpt;
   mp=FirstMessage;
   while (mp) {
      mpt=mp;
      mp=mp->Next;
      free(mpt);
   }
   FirstMessage=NULL;
}

void ReceiveDrugsHere(char *text,struct PLAYER *To) {
/* Decodes the message data "text" into a list of drug prices for */
/* player "To"                                                    */
   char *cp,wrd[BUFLEN];
   int i;

   To->EventNum=E_ARRIVE;
   cp=text;
   for (i=0;i<NumDrug;i++) {
      cp=ExtractWord(wrd,BUFLEN,cp);
      To->Drugs[i].Price=atoi(wrd);
   }
}
