/*
**  Copyright (c) 2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef lint
static char dkim_policy_c_id[] = "@(#)$Id: dkim-policy.c,v 1.5 2007/05/21 22:19:33 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netdb.h>
#include <resolv.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <assert.h>
#include <ctype.h>

/* libsm includes */
#include <sm/string.h>

/* libar includes */
#if USE_ARLIB
# include <ar.h>
#endif /* USE_ARLIB */

/* libdkim includes */
#include "dkim.h"
#include "dkim-types.h"
#include "dkim-policy.h"
#ifdef _FFR_QUERY_CACHE
# include "dkim-cache.h"
#endif /* _FFR_QUERY_CACHE */

/* prototypes */
extern void dkim_error __P((DKIM *, const char *, ...));

/* limits, macros, etc. */
#define	BUFRSZ			1025

/* local definitions needed for DNS queries */
#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN		256
#endif /* ! MAXHOSTNAMELEN */
#define MAXPACKET		8192
#if defined(__RES) && (__RES >= 19940415)
# define RES_UNC_T		char *
#else /* __RES && __RES >= 19940415 */
# define RES_UNC_T		unsigned char *
#endif /* __RES && __RES >= 19940415 */

/*
**  DKIM_GET_POLICY_FILE -- acquire a domain's policy record using a local file
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer into which to write policy
**  	buflen -- number of bytes available at "buf"
**
**  Return value:
**  	1 -- policy retrieved, stored in buffer
**  	0 -- no policy found
**  	-1 -- failure
*/

int
dkim_get_policy_file(DKIM *dkim, unsigned char *buf, size_t buflen)
{
	bool found;
	int first;
	int n;
	int nlabels = 0;
	int c;
	int len;
	char *path;
	char *ctx;
	char *p;
	FILE *f;
	unsigned char *label[MAXLABELS];
	unsigned char inbuf[BUFRSZ + 1];
	char qname[MAXHOSTNAMELEN + 1];
	unsigned char domain[MAXHOSTNAMELEN + 1];

	assert(dkim != NULL);
	assert(dkim->dkim_domain != NULL);
	assert(dkim->dkim_querymethod == DKIM_QUERY_FILE);

	memset(label, '\0', sizeof label);
	path = dkim->dkim_libhandle->dkiml_queryinfo;

	f = fopen(path, "r");
	if (f == NULL)
	{
		dkim_error(dkim, "%s: fopen(): %s", path,
		           strerror(errno));
		return -1;
	}

	/* make an array of labels */
	sm_strlcpy(domain, dkim->dkim_domain, sizeof domain);
	for (p = strtok_r(domain, ".", &ctx);
	     p != NULL;
	     p = strtok_r(NULL, ".", &ctx))
	{
		for (n = 1; n < MAXLABELS; n++)
			label[n - 1] = label[n];
		label[MAXLABELS - 1] = p;
		nlabels++;
	}

	first = MAXLABELS - nlabels;

	/* query upwards */
	found = FALSE;
	for (n = 0; n < MAXPOLICYDEPTH && n < nlabels; n++)
	{
		/* set up the key for the lookup */
		memset(qname, '\0', sizeof qname);
#ifdef _FFR_ALLMAN_SSP_02
		if (dkim->dkim_userpolicy)
		{
			snprintf(qname, sizeof qname, "%s.%s", dkim->dkim_user,
			         DKIM_DNSUSERPOLICYNAME);
		}
#else /* _FFR_ALLMAN_SSP_02 */
		snprintf(qname, sizeof qname, "%s.%s", DKIM_DNSPOLICYNAME,
		         DKIM_DNSKEYNAME);
#endif /* _FFR_ALLMAN_SSP_02 */

		for (c = first + n; c < MAXLABELS; c++)
		{
			if (qname[0] != '\0')
				sm_strlcat(qname, ".", sizeof qname);
			sm_strlcat(qname, label[c], sizeof qname);
		}

		len = strlen(qname);

		memset(inbuf, '\0', sizeof inbuf);
		rewind(f);

		while (!found && fgets(inbuf, sizeof inbuf - 1, f) != NULL)
		{
			for (p = buf; *p != '\0'; p++)
			{
				if (*p == '\n' || *p == '#')
				{
					*p = '\0';
					break;
				}
			}

			/* is this a match? */
			if (strncasecmp(inbuf, qname, n) == 0 &&
			    isascii(inbuf[n]) && isspace(inbuf[n]))
			{
				found = TRUE;

				/* move past spaces */
				for (p = &inbuf[n] + 1;
				     isascii(*p) && isspace(*p);
				     p++)
					continue;

				sm_strlcpy(buf, p, buflen);

				break;
			}
		}

		if (ferror(f))
		{
			dkim_error(dkim, "%s: fgets(): %s", path,
			           strerror(errno));
			fclose(f);
			return -1;
		}

		if (found)
			break;
	}

	fclose(f);

	return (found ? 1 : 0);
}

/*
**  DKIM_GET_POLICY_DNS -- acquire a domain's policy record using DNS queries
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer into which to write policy
**  	buflen -- number of bytes available at "buf"
**
**  Return value:
**  	1 -- policy retrieved, stored in buffer
**  	0 -- no policy found
**  	-1 -- failure
*/

int
dkim_get_policy_dns(DKIM *dkim, unsigned char *buf, size_t buflen)
{
	int qdcount;
	int ancount;
	int status;
	int n;
	int nlabels = 0;
	int first;
	int c;
	int type = -1;
	int class = -1;
#ifdef _FFR_QUERY_CACHE
	int ttl;
#endif /* _FFR_QUERY_CACHE */
	bool found;
	size_t anslen;
#if USE_ARLIB
	AR_LIB ar;
	AR_QUERY q;
	int arerror;
#endif /* USE_ARLIB */
	DKIM_LIB *lib;
	char *ctx;
	unsigned char *label[MAXLABELS];
	unsigned char *p;
	unsigned char *cp;
	unsigned char *eom;
	unsigned char ansbuf[MAXPACKET];
	char qname[MAXHOSTNAMELEN + 1];
	unsigned char domain[MAXHOSTNAMELEN + 1];
#if USE_ARLIB
	struct timeval timeout;
#endif /* USE_ARLIB */
	HEADER hdr;

	assert(buf != NULL);
	assert(dkim != NULL);
	assert(dkim->dkim_domain != NULL);
	assert(dkim->dkim_querymethod == DKIM_QUERY_DNS);

#if USE_ARLIB
	ar = dkim->dkim_libhandle->dkiml_arlib;
	assert(ar != NULL);
#endif /* USE_ARLIB */

	lib = dkim->dkim_libhandle;

	memset(label, '\0', sizeof label);

	/* make an array of labels */
	sm_strlcpy(domain, dkim->dkim_domain, sizeof domain);
	for (p = strtok_r(domain, ".", &ctx);
	     p != NULL;
	     p = strtok_r(NULL, ".", &ctx))
	{
		for (n = 1; n < MAXLABELS; n++)
			label[n - 1] = label[n];
		label[MAXLABELS - 1] = p;
		nlabels++;
	}

	first = MAXLABELS - nlabels;

	/* query upwards */
	found = FALSE;
	for (n = 0; n < MAXPOLICYDEPTH && n < nlabels; n++)
	{
		/* send the NS query */
		memset(qname, '\0', sizeof qname);
#ifdef _FFR_ALLMAN_SSP_02
		if (dkim->dkim_userpolicy)
		{
			snprintf(qname, sizeof qname, "%s.%s", dkim->dkim_user,
			         DKIM_DNSUSERPOLICYNAME);
		}
#else /* _FFR_ALLMAN_SSP_02 */
		snprintf(qname, sizeof qname, "%s.%s", DKIM_DNSPOLICYNAME,
		         DKIM_DNSKEYNAME);
#endif /* _FFR_ALLMAN_SSP_02 */

		for (c = first + n; c < MAXLABELS; c++)
		{
			if (qname[0] != '\0')
				sm_strlcat(qname, ".", sizeof qname);
			sm_strlcat(qname, label[c], sizeof qname);
		}

#ifdef _FFR_QUERY_CACHE
		if (dkim->dkim_libhandle->dkiml_cache != NULL)
		{
			int err = 0;
			size_t blen = buflen;

			dkim->dkim_cache_queries++;

			status = dkim_cache_query(dkim->dkim_libhandle->dkiml_cache,
			                          qname, 0, buf, &blen, &err);

			if (status == 0)
			{
				dkim->dkim_cache_hits++;
				return (status == DKIM_STAT_OK ? 0 : -1);
			}
			/* XXX -- do something with errors here */
		}
#endif /* _FFR_QUERY_CACHE */

#if USE_ARLIB
		timeout.tv_sec = dkim->dkim_timeout;
		timeout.tv_usec = 0;

# ifdef _FFR_ALLMAN_SSP_02
		q = ar_addquery(ar, qname, C_IN, T_DKIMP, MAXCNAMEDEPTH,
		                ansbuf, sizeof ansbuf, &arerror,
		                dkim->dkim_timeout == 0 ? NULL : &timeout);
# else /* _FFR_ALLMAN_SSP_02 */
		q = ar_addquery(ar, qname, C_IN, T_TXT, MAXCNAMEDEPTH, ansbuf,
		                sizeof ansbuf, &arerror,
		                dkim->dkim_timeout == 0 ? NULL : &timeout);
# endif /* _FFR_ALLMAN_SSP_02 */
		if (q == NULL)
		{
			dkim_error(dkim, "ar_addquery() failed for `%s'",
			           qname);
			return -1;
		}

		if (lib->dkiml_dns_callback == NULL)
		{
			status = ar_waitreply(ar, q, NULL, NULL);
		}
		else
		{
			for (;;)
			{
				timeout.tv_sec = lib->dkiml_callback_int;
				timeout.tv_usec = 0;

				status = ar_waitreply(ar, q, NULL, &timeout);

				if (status != AR_STAT_NOREPLY)
					break;

				lib->dkiml_dns_callback(dkim->dkim_user_context);
			}
		}

		(void) ar_cancelquery(ar, q);
#else /* USE_ARLIB */

# ifdef _FFR_ALLMAN_SSP_02
		status = res_query(qname, C_IN, T_DKIMP, ansbuf,
		                   sizeof ansbuf);
# else /* _FFR_ALLMAN_SSP_02 */
		status = res_query(qname, C_IN, T_TXT, ansbuf, sizeof ansbuf);
# endif /* _FFR_ALLMAN_SSP_02 */

#endif /* USE_ARLIB */

#if USE_ARLIB
		if (status == AR_STAT_ERROR || status == AR_STAT_EXPIRED)
		{
			dkim_error(dkim, "ar_waitreply(): `%s' query %s",
			           qname,
			           status == AR_STAT_ERROR ? "error"
			                                   : "expired");
			return -1;
		}

#else /* USE_ARLIB */
		/*
		**  A -1 return from res_query could mean a bunch of things,
		**  not just NXDOMAIN.  You can use h_errno to determine what
		**  -1 means.  This is poorly documented.
		*/

		if (status == -1)
		{
			switch (h_errno)
			{
			  case HOST_NOT_FOUND:
			  case NO_DATA:
				return 0;

			  case TRY_AGAIN:
			  case NO_RECOVERY:
			  default:
				dkim_error(dkim, "res_query(): `%s' %s",
				           qname, hstrerror(h_errno));
				return -1;
			}
		}
#endif /* USE_ARLIB */

		/* set up pointers */
		anslen = sizeof ansbuf;
		memcpy(&hdr, ansbuf, sizeof hdr);
		cp = (u_char *) &ansbuf + HFIXEDSZ;
		eom = (u_char *) &ansbuf + anslen;

		/* skip over the name at the front of the answer */
		for (qdcount = ntohs((unsigned short) hdr.qdcount);
		     qdcount > 0;
		     qdcount--)
		{
			/* copy it first */
			(void) dn_expand((unsigned char *) &ansbuf, eom, cp,
			                 qname, sizeof qname);

			if ((n = dn_skipname(cp, eom)) < 0)
			{
				dkim_error(dkim, "`%s' reply corrupt", qname);
				return -1;
			}

			cp += n;

			/* extract the type and class */
			if (cp + INT16SZ + INT16SZ > eom)
			{
				dkim_error(dkim, "`%s' reply corrupt", qname);
				return -1;
			}

			GETSHORT(type, cp);
			GETSHORT(class, cp);
		}

		if (type != T_TXT || class != C_IN)
		{
			dkim_error(dkim, "`%s' unexpected reply class/type",
			            qname);
			return -1;
		}

		/* if NXDOMAIN, return DKIM_STAT_CANTVRFY */
		if (hdr.rcode == NXDOMAIN)
			return 0;

		/* get the answer count */
		ancount = ntohs((unsigned short) hdr.ancount);
		if (ancount == 0)
			return 0;

		/* got something useable! */
		found = TRUE;
		break;
	}

	/* no policy found; apply defaults */
	if (!found)
		return 0;

	/* if truncated, we can't do it */
	if (hdr.tc)
	{
		dkim_error(dkim, "reply for `%s' truncated", qname);
		return -1;
	}

	/* grab the label, even though we know what we asked... */
	if ((n = dn_expand((unsigned char *) &ansbuf, eom, cp,
	                   (RES_UNC_T) qname, sizeof qname)) < 0)
	{
		dkim_error(dkim, "`%s' reply corrupt", qname);
		return -1;
	}
	/* ...and move past it */
	cp += n;

	/* extract the type and class */
	if (cp + INT16SZ + INT16SZ > eom)
	{
		dkim_error(dkim, "`%s' reply corrupt", qname);
		return -1;
	}
	GETSHORT(type, cp);
	GETSHORT(class, cp);

	/* reject anything that's not valid (stupid wildcards) */
	if (type != T_TXT || class != C_IN)
	{
		dkim_error(dkim, "`%s' unexpected reply class/type", qname);
		return -1;
	}

#ifdef _FFR_QUERY_CACHE
	GETLONG(ttl, cp);
#else /* _FFR_QUERY_CACHE */
	/* skip the TTL */
	cp += INT32SZ;
#endif /* _FFR_QUERY_CACHE */

	/* get payload length */
	if (cp + INT16SZ > eom)
	{
		dkim_error(dkim, "`%s' reply corrupt", qname);
		return -1;
	}
	GETSHORT(n, cp);

	/* XXX -- maybe deal with a partial reply rather than require it all */
	if (cp + n > eom)
	{
		dkim_error(dkim, "`%s' reply corrupt", qname);
		return -1;
	}

	if (n > BUFRSZ)
	{
		dkim_error(dkim, "`%s' reply corrupt", qname);
		return -1;
	}

	/* extract the payload */
	memset(buf, '\0', sizeof buf);
	p = buf;
	while (n > 0)
	{
		c = *cp++;
		n--;
		while (c > 0)
		{
			*p++ = *cp++;
			c--;
			n--;
		}
	}

#ifdef _FFR_QUERY_CACHE
	if (dkim->dkim_libhandle->dkiml_cache != NULL)
	{
		int err = 0;

		status = dkim_cache_insert(dkim->dkim_libhandle->dkiml_cache,
		                           qname, buf, ttl, &err);
		/* XXX -- do something with errors here */
	}
#endif /* _FFR_QUERY_CACHE */

	return 1;
}
