/*
**  Copyright (c) 2005, 2006 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef lint
static char dkim_c_id[] = "@(#)$Id: dkim.c,v 1.176 2006/09/15 21:23:36 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netdb.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <unistd.h>
#include <resolv.h>
#include <regex.h>

/* libsm includes */
#include <sm/string.h>

/* libar includes */
#if USE_ARLIB
# include <ar.h>
#endif /* USE_ARLIB */

/* OpenSSL includes */
#include <openssl/pem.h>
#include <openssl/rsa.h>
#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/sha.h>

/* libdkim includes */
#include "dkim.h"
#include "dkim-types.h"
#include "dkim-tables.h"
#include "dkim-keys.h"
#include "util.h"
#include "base64.h"

/* prototypes */
static void *dkim_malloc(DKIM_LIB *, void *, size_t);
static void dkim_mfree(DKIM_LIB *, void *, void *);

/* macros */
#define	DKIM_MALLOC(x,y)	dkim_malloc((x)->dkim_libhandle, \
				            (x)->dkim_closure, y)
#define	DKIM_FREE(x,y)		dkim_mfree((x)->dkim_libhandle, \
				           (x)->dkim_closure, y)

#define	DKIM_STATE_INIT		0
#define	DKIM_STATE_HEADER	1
#define	DKIM_STATE_EOH		2
#define	DKIM_STATE_BODY		3
#define	DKIM_STATE_EOM		4

#define	DKIM_MODE_UNKNOWN	(-1)
#define	DKIM_MODE_SIGN		0
#define	DKIM_MODE_VERIFY	1

#define	DKIM_SIGTYPE_UNKNOWN	(-1)
#define	DKIM_SIGTYPE_RSA	0

#define	DKIM_HDR_SIGNED		0x01

#define	BUFRSZ			1024
#define	CRLF			"\r\n"
#define	SP			" "
#define	DEFTIMEOUT		10

/* local definitions needed for DNS queries */
#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN		256
#endif /* ! MAXHOSTNAMELEN */
#define MAXPACKET		8192
#if defined(__RES) && (__RES >= 19940415)
# define RES_UNC_T		char *
#else /* __RES && __RES >= 19940415 */
# define RES_UNC_T		unsigned char *
#endif /* __RES && __RES >= 19940415 */

/* list of headers which may contain the sender */
const u_char *default_senderhdrs[] =
{
	"resent-sender",
	"resent-from",
	"sender",
	"from",
	NULL
};

/* default/minimum list of headers to sign */
const u_char *default_signhdrs[] =
{
	"resent-sender",
	"resent-from",
	"sender",
	"from",
	"date",
	"subject",
	"content-*",
	NULL
};

/* ========================= PRIVATE SECTION ========================= */

/*
**  DKIM_MALLOC -- allocate memory
**
**  Parameters:
**  	libhandle -- DKIM library context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	nbytes -- number of bytes desired
**
**  Return value:
**  	Pointer to allocated memory, or NULL on failure.
*/

static void *
dkim_malloc(DKIM_LIB *libhandle, void *closure, size_t nbytes)
{
	assert(libhandle != NULL);

	if (libhandle->dkiml_malloc == NULL)
		return malloc(nbytes);
	else
		return libhandle->dkiml_malloc(closure, nbytes);
}

/*
**  DKIM_MFREE -- release memory
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	ptr -- pointer to memory to be freed
**
**  Return value:
**  	None.
*/

static void
dkim_mfree(DKIM_LIB *libhandle, void *closure, void *ptr)
{
	assert(libhandle != NULL);

	if (libhandle->dkiml_free == NULL)
		free(ptr);
	else
		libhandle->dkiml_free(closure, ptr);
}

/*
**  DKIM_STRDUP -- duplicate a string
**
**  Parameters:
**  	dkim -- DKIM handle
**  	str -- string to clone
**  	len -- bytes to copy (0 == copy to NULL byte)
**
**  Return value:
**  	Pointer to a new copy of "str" allocated from within the appropriate
**  	context, or NULL on failure.
*/

static unsigned char *
dkim_strdup(DKIM *dkim, const unsigned char *str, size_t len)
{
	unsigned char *new;

	assert(dkim != NULL);
	assert(str != NULL);

	if (len == 0)
		len = strlen(str);
	new = DKIM_MALLOC(dkim, len + 1);
	if (new != NULL)
		sm_strlcpy(new, str, len + 1);
	return new;
}

/*
**  DKIM_SELECTHDRS -- choose headers to be included in canonicalization
**
**  Parameters:
**  	DKIM -- DKIM context in which this is performed
**  	hdrlist -- string containing headers that should be marked, separated
**  	           by the ":" character
**  	ptrs -- array of header pointers (modified)
**  	nptr -- number of pointers available at "ptrs"
**
**  Return value:
**  	Count of headers added to "ptrs", or -1 on error.
**
**  Notes:
**  	Selects headers to be passed to canonicalization and the order in
**  	which this is done.  "ptrs" is populated by pointers to headers
**  	in the order in which they should be fed to canonicalization.
**
**  	If any of the returned pointers is NULL, then a header named by
**  	"hdrlist" was not found.
*/

static int
dkim_selecthdrs(DKIM *dkim, u_char *hdrlist, struct dkim_header **ptrs,
                int nptrs)
{
	u_char *bar;
	u_char *colon;
	char *ctx;
	int c;
	int n;
	int m;
	size_t len;
	struct dkim_header *hdr;
	struct dkim_header *lhdrs[MAXHDRCNT];
	u_char *hdrs[MAXHDRCNT];

	assert(dkim != NULL);
	assert(ptrs != NULL);
	assert(nptrs != 0);

	/* if there are no headers named, use them all */
	if (hdrlist == NULL)
	{
		n = 0;

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (n >= nptrs)
				break;
			ptrs[n] = hdr;
			n++;
		}

		return n;
	}

	dkim->dkim_hdrlist = dkim_strdup(dkim, hdrlist, 0);
	if (dkim->dkim_hdrlist == NULL)
		return -1;

	/* mark all headers as not used */
	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		hdr->hdr_flags &= ~DKIM_HDR_SIGNED;

	n = 0;

	/* make a split-out copy of hdrlist */
	memset(hdrs, '\0', sizeof hdrs);
	for (bar = strtok_r(dkim->dkim_hdrlist, ":", &ctx);
	     bar != NULL;
	     bar = strtok_r(NULL, ":", &ctx))
	{
		if (n >= MAXHDRCNT)
			return -1;

		hdrs[n] = bar;
		n++;
	}

	/* bounds check */
	if (n >= nptrs)
		return -1;

	/* for each named header, find the last unused one and use it up */
	for (c = 0; c < n; c++)
	{
		lhdrs[c] = NULL;

		len = MIN(MAXHEADER, strlen(hdrs[c]));

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (hdr->hdr_flags & DKIM_HDR_SIGNED)
				continue;

			colon = strchr(hdr->hdr_text, ':');
			if (colon != NULL)
			{
				len = colon - hdr->hdr_text;

				if (strncasecmp(hdr->hdr_text, hdrs[c],
				                len) == 0)
					lhdrs[c] = hdr;
			}
			else
			{
				if (strcasecmp(hdr->hdr_text, hdrs[c]) == 0)
					lhdrs[c] = hdr;
			}
		}

		if (lhdrs[c] != NULL)
			lhdrs[c]->hdr_flags |= DKIM_HDR_SIGNED;
	}

	/* copy to the caller's buffers */
	m = 0;
	for (c = 0; c < n; c++)
	{
		if (lhdrs[c] != NULL)
		{
			ptrs[m] = lhdrs[c];
			m++;
		}
	}

	return m;
}

/*
**  DKIM_SET_FIRST -- return first set in a context
**
**  Parameters:
**  	dkim -- DKIM context
**  	type -- type to find, or DKIM_SETTYPE_ANY
**
**  Return value:
**  	Pointer to the first DKIM_SET in the context, or NULL if none.
*/

static DKIM_SET *
dkim_set_first(DKIM *dkim, dkim_set_t type)
{
	DKIM_SET *set;

	assert(dkim != NULL);

	if (type == DKIM_SETTYPE_ANY)
		return dkim->dkim_sethead;

	for (set = dkim->dkim_sethead; set != NULL; set = set->set_next)
	{
		if (set->set_type == type)
			return set;
	}

	return NULL;
}

/*
**  DKIM_SET_NEXT -- return next set in a context
**
**  Parameters:
**  	set -- last set reported (i.e. starting point for this search)
**  	type -- type to find, or DKIM_SETTYPE_ANY
**
**  Return value:
**  	Pointer to the next DKIM_SET in the context, or NULL if none.
*/

static DKIM_SET *
dkim_set_next(DKIM_SET *cur, dkim_set_t type)
{
	DKIM_SET *set;

	assert(cur != NULL);

	if (type == DKIM_SETTYPE_ANY)
		return cur->set_next;

	for (set = cur->set_next; set != NULL; set = set->set_next)
	{
		if (set->set_type == type)
			return set;
	}

	return NULL;
}

/*
**  DKIM_PARAM_GET -- get a parameter from a set
**
**  Parameters:
**  	set -- set to search
**  	param -- parameter to find
**
**  Return value:
**  	Pointer to the parameter requested, or NULL if it's not in the set.
*/

static u_char *
dkim_param_get(DKIM_SET *set, u_char *param)
{
	DKIM_PLIST *plist;

	assert(set != NULL);
	assert(param != NULL);

	for (plist = set->set_plist; plist != NULL; plist = plist->plist_next)
	{
		if (strcasecmp(plist->plist_param, param) == 0)
			return plist->plist_value;
	}

	return NULL;
}

/*
**  DKIM_ADD_PLIST -- add an entry to a parameter-value set
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	set -- set to modify
**   	param -- parameter
**  	value -- value
**  	force -- override existing value, if any
**
**  Return value:
**  	0 on success, -1 on failure.
**
**  Notes:
**  	Data is not copied; a reference to it is stored.
*/

static int
dkim_add_plist(DKIM *dkim, DKIM_SET *set, u_char *param, u_char *value,
               bool force)
{
	DKIM_PLIST *plist;

	assert(dkim != NULL);
	assert(set != NULL);
	assert(param != NULL);
	assert(value != NULL);

	/* see if we have one already */
	for (plist = set->set_plist; plist != NULL; plist = plist->plist_next)
	{
		if (strcasecmp(plist->plist_param, param) == 0)
			break;
	}

	/* nope; make one and connect it */
	if (plist == NULL)
	{
		plist = (DKIM_PLIST *) DKIM_MALLOC(dkim, sizeof(DKIM_PLIST));
		if (plist == NULL)
			return -1;
		force = TRUE;
		plist->plist_next = set->set_plist;
		set->set_plist = plist;
		plist->plist_param = param;
	}

	/* set the value if "force" was set (or this was a new entry) */
	if (force)
		plist->plist_value = value;

	return 0;
}

/*
**  DKIM_PROCESS_SET -- process a parameter set, i.e. a string of the form
**                      param=value[; param=value]*
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	type -- a DKIM_SETTYPE constant
**  	str -- string to be scanned
**  	len -- number of bytes available at "str"
**
**  Return value:
**  	A DKIM_STAT constant.
*/

static DKIM_STAT
dkim_process_set(DKIM *dkim, dkim_set_t type, u_char *str, size_t len)
{
	bool spaced;
	int state;
	int status;
	u_char *p;
	u_char *param;
	u_char *value;
	u_char *hcopy;
	DKIM_SET *set;

	assert(dkim != NULL);
	assert(str != NULL);

	param = NULL;
	value = NULL;
	state = 0;
	spaced = FALSE;

	hcopy = (u_char *) DKIM_MALLOC(dkim, len + 1);
	if (hcopy == NULL)
		return DKIM_STAT_INTERNAL;
	sm_strlcpy(hcopy, str, len + 1);

	set = (DKIM_SET *) DKIM_MALLOC(dkim, sizeof(DKIM_SET));
	if (set == NULL)
	{
		DKIM_FREE(dkim, hcopy);
		return DKIM_STAT_INTERNAL;
	}

	set->set_type = type;

	if (dkim->dkim_sethead == NULL)
		dkim->dkim_sethead = set;
	else
		dkim->dkim_settail->set_next = set;

	dkim->dkim_settail = set;

	set->set_next = NULL;
	set->set_plist = NULL;
	set->set_data = hcopy;
	set->set_done = FALSE;

	for (p = hcopy; *p != '\0'; p++)
	{
		if (!isascii(*p))
			return DKIM_STAT_SYNTAX;

		switch (state)
		{
		  case 0:				/* before param */
			if (isspace(*p))
			{
				continue;
			}
			else if (isalnum(*p))
			{
				param = p;
				state = 1;
			}
			else
			{
				return DKIM_STAT_SYNTAX;
			}
			break;

		  case 1:				/* in param */
			if (isspace(*p))
			{
				spaced = TRUE;
			}
			else if (*p == '=')
			{
				*p = '\0';
				state = 2;
				spaced = FALSE;
			}
			else if (*p == ';' || spaced)
			{
				return DKIM_STAT_SYNTAX;
			}
			break;

		  case 2:				/* before value */
			if (isspace(*p))
			{
				continue;
			}
			else if (*p == ';')		/* empty value */
			{
				*p = '\0';
				value = p;

				/* collapse the parameter */
				dkim_collapse(param);

				/* create the DKIM_PLIST entry */
				status = dkim_add_plist(dkim, set, param,
				                        value, TRUE);
				if (status == -1)
					return DKIM_STAT_INTERNAL;

				/* reset */
				param = NULL;
				value = NULL;
				state = 0;
			}
			else
			{
				value = p;
				state = 3;
			}
			break;

		  case 3:				/* in value */
			if (*p == ';')
			{
				*p = '\0';

				/* collapse the parameter and value */
				dkim_collapse(param);
				dkim_collapse(value);

				/* create the DKIM_PLIST entry */
				status = dkim_add_plist(dkim, set, param,
				                        value, TRUE);
				if (status == -1)
					return DKIM_STAT_INTERNAL;

				/* reset */
				param = NULL;
				value = NULL;
				state = 0;
			}
			break;

		  default:				/* shouldn't happen */
			assert(0);
		}
	}

	switch (state)
	{
	  case 0:					/* before param */
	  case 3:					/* in value */
		/* parse the data found, if any */
		if (value != NULL)
		{
			/* collapse the parameter and value */
			dkim_collapse(param);
			dkim_collapse(value);

			/* create the DKIM_PLIST entry */
			status = dkim_add_plist(dkim, set, param, value, TRUE);
			if (status == -1)
				return DKIM_STAT_INTERNAL;
		}
		break;

	  case 1:					/* after param */
	  case 2:					/* before value */
		return DKIM_STAT_SYNTAX;

	  default:					/* shouldn't happen */
		assert(0);
	}

	/* load up defaults, assert requirements */
	switch (set->set_type)
	{
	  case DKIM_SETTYPE_SIGNATURE:
		/* make sure required stuff is here */
		if (dkim_param_get(set, "s") == NULL ||
		    dkim_param_get(set, "h") == NULL ||
		    dkim_param_get(set, "d") == NULL ||
		    dkim_param_get(set, "b") == NULL ||
		    dkim_param_get(set, "a") == NULL)
			return DKIM_STAT_SYNTAX;

		/* default for "c" */
		status = dkim_add_plist(dkim, set, "c", "simple/simple", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		/* default for "q" */
		status = dkim_add_plist(dkim, set, "q", "dns", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		/* default for "i" */
		if (dkim_param_get(set, "i") == NULL)
		{
			char addr[MAXADDRESS + 1];

			value = dkim_param_get(set, "d");
			snprintf(addr, sizeof addr, "@%s", value);
			dkim->dkim_signer = dkim_strdup(dkim, addr, 0);
			status = dkim_add_plist(dkim, set, "i",
			                        dkim->dkim_signer, FALSE);
			if (status == -1)
				return DKIM_STAT_INTERNAL;
		}

		break;


	  case DKIM_SETTYPE_POLICY:
		status = dkim_add_plist(dkim, set, "o", "~", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		break;

	  case DKIM_SETTYPE_KEY:
		status = dkim_add_plist(dkim, set, "g", "*", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "k", "rsa", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "s", "*", FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		status = dkim_add_plist(dkim, set, "v", DKIM_VERSION, FALSE);
		if (status == -1)
			return DKIM_STAT_INTERNAL;

		break;
			

	  default:
		assert(0);
	}

	set->set_done = TRUE;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GET_HEADER -- find a header in a queue of headers
**
**  Parameters:
**  	dkim -- DKIM handle
**  	name -- name of the header to find
**  	inst -- instance to find (0 == first/any)
**
**  Return value:
**  	Pointer to a (struct dkim_header), or NULL if not found.
*/

struct dkim_header *
dkim_get_header(DKIM *dkim, u_char *name, int inst)
{
	u_char *colon;
	struct dkim_header *hdr;

	assert(dkim != NULL);
	assert(name != NULL);

	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
	{
		colon = strchr(hdr->hdr_text, ':');
		if (colon == NULL && strcasecmp(hdr->hdr_text, name) == 0)
		{
			if (inst == 0)
				return hdr;
			else
				inst--;
		}
		else
		{
			size_t len;

			len = colon - hdr->hdr_text;
			if (strncasecmp(hdr->hdr_text, name, len) == 0)
			{
				if (inst == 0)
					return hdr;
				else
					inst--;
			}
		}
	}

	return NULL;
}

/*
**  DKIM_KEY_SMTP -- return TRUE iff a parameter set defines an SMTP key
**
**  Parameters:
**  	set -- set to be checked
**
**  Return value:
**  	TRUE iff "set" contains an "s" parameter whose value is either
**  	"email" or "*".
*/

static bool
dkim_key_smtp(DKIM_SET *set)
{
	u_char *val;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_KEY);

	val = dkim_param_get(set, "s");

	if (val != NULL &&
	    (strcmp(val, "*") == 0 || strcasecmp(val, "email") == 0))
		return TRUE;
	else
		return FALSE;
}

/*
**  DKIM_KEY_GRANOK -- return TRUE iff the granularity of the key is
**                     appropriate to the signature being evaluated
**
**  Parameters:
**  	dkim -- DKIM handle
**  	gran -- granularity string from the retrieved key
**
**  Return value:
**  	TRUE iff the value of the granularity is a match for the signer.
*/

static bool
dkim_key_granok(DKIM *dkim, u_char *gran)
{
	int status;
	DKIM_SET *set;
	char *at;
	char *end;
	char *p;
	char *q;
	char restr[MAXADDRESS + 1];
	char user[MAXADDRESS + 1];
	regex_t re;

	assert(dkim != NULL);
	assert(gran != NULL);

	/* if it's empty, it matches nothing */
	if (gran[0] == '\0')
		return FALSE;

	/* if it's just "*", it matches everything */
	if (gran[0] == '*' && gran[1] == '\0')
		return TRUE;

	/* ensure we're evaluating against a signature data set */
	set = dkim->dkim_sigset;
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	/* get the value of the "i" parameter */
	p = dkim_param_get(set, "i");
	assert(p != NULL);
	sm_strlcpy(user, p, sizeof user);

	/* validate the "i" parameter */
	at = strchr(user, '@');
	assert(at != NULL);

	/* if it's not wildcarded, enforce an exact match */
	if (strchr(gran, '*') == NULL)
		return (strncmp(gran, user, at - user) == 0);

	/* evaluate the wildcard */
	end = restr + sizeof restr;
	memset(restr, '\0', sizeof restr);
	for (p = gran, q = restr; q < end - 1; p++)
	{
		if (isascii(*p) && ispunct(*p))
		{
			if (*p == '*')
			{
				*q++ = '.';
				*q++ = '*';
			}
			else
			{
				*q++ = '\\';
				*q++ = *p;
			}
		}
		else
		{
			*q++ = *p;
		}
	}

	if (q >= end - 1)
		return FALSE;

	status = regcomp(&re, user, 0);
	if (status != 0)
		return FALSE;

	status = regexec(&re, user, 0, NULL, 0);
	(void) regfree(&re);

	return (status == 0 ? TRUE : FALSE);
}

/*
**  DKIM_KEY_HASHOK -- return TRUE iff a particular hash is in the approved
**                     list of hashes for a given key
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hashlist -- colon-separated approved hash list
**
**  Return value:
**  	TRUE iff a particular hash is in the approved list of hashes.
*/

static bool
dkim_key_hashok(DKIM *dkim, u_char *hashlist)
{
	int hashalg;
	u_char *x, *y;
	u_char tmp[BUFRSZ + 1];

	assert(dkim != NULL);

	if (hashlist == NULL)
		return TRUE;

	x = NULL;
	memset(tmp, '\0', sizeof tmp);

	y = hashlist;
	for (;;)
	{
		if (*y == ':' || *y == '\0')
		{
			if (x != NULL)
			{
				sm_strlcpy(tmp, x, sizeof tmp);
				tmp[y - x] = '\0';
				hashalg = dkim_name_to_code(hashes, tmp);
				if (hashalg == dkim->dkim_hashtype)
					return TRUE;
			}

			x = NULL;
		}
		else if (x == NULL)
		{
			x = y;
		}

		if (*y == '\0')
			return FALSE;
		y++;
	}

	/* NOTREACHED */
}

/*
**  DKIM_KEY_HASHESOK -- return TRUE iff this key supports at least one
**                       hash method we know about (or doesn't specify)
**
**  Parameters:
**  	hashlist -- colon-separated list of hashes (or NULL)
**
**  Return value:
**  	TRUE iff this key supports at least one hash method we know about
**  	(or doesn't specify)
*/

static bool
dkim_key_hashesok(u_char *hashlist)
{
	u_char *x, *y;
	u_char tmp[BUFRSZ + 1];

	if (hashlist == NULL)
		return TRUE;

	x = NULL;
	memset(tmp, '\0', sizeof tmp);

	y = hashlist;
	for (;;)
	{
		if (*y == ':' || *y == '\0')
		{
			if (x != NULL)
			{
				sm_strlcpy(tmp, x, sizeof tmp);
				tmp[y - x] = '\0';
				if (dkim_name_to_code(hashes, tmp) != -1)
					return TRUE;
			}

			x = NULL;
		}
		else if (x == NULL)
		{
			x = y;
		}

		if (*y == '\0')
			return FALSE;
		y++;
	}

	/* NOTREACHED */
}

/*
**  DKIM_SIG_SIGNEROK -- return TRUE iff the signer is specified in a signed
**                       sender header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	set -- signature set to be checked
**  	hdrs -- names of sender headers
**
**  Return value:
**  	TRUE iff the value of the "i" parameter appears in a signed sender
**  	header.
*/

static bool
dkim_sig_signerok(DKIM *dkim, DKIM_SET *set, u_char **hdrs)
{
	int status;
	int c;
	int clen;
	struct dkim_header *cur;
	u_char *colon;
	char *i;
	char *user;
	char *domain;
	char buf[MAXADDRESS + 1];
	char addr[MAXADDRESS + 1];

	assert(dkim != NULL);
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	i = dkim_param_get(set, "i");

	assert(i != NULL);

	/* for each header in the "sender header" list */
	for (c = 0; hdrs[c] != NULL; c++)
	{
		/* for each header in the message */
		for (cur = dkim->dkim_hhead; cur != NULL; cur = cur->hdr_next)
		{
			/* skip unsigned headers */
			if ((cur->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			/* determine header name size */
			colon = strchr(cur->hdr_text, ':');
			if (colon == NULL)
				clen = strlen(cur->hdr_text);
			else
				clen = colon - cur->hdr_text;

			/* if this is a sender header */
			if (strncasecmp(hdrs[c], cur->hdr_text, clen) == 0)
			{
				if (colon == NULL)
					colon = cur->hdr_text;
				else
					colon += 1;

				sm_strlcpy(buf, colon, sizeof buf);

				status = rfc2822_mailbox_split(buf, &user,
				                               &domain);
				if (status != 0)
					continue;

				snprintf(addr, sizeof addr, "%s@%s",
				         user, domain);

				/* see if the sender matches "i" */
				if (dkim_addrcmp(addr, i) == 0)
					return TRUE;
			}
		}
	}

	return FALSE;
}

/*
**  DKIM_SIG_DOMAINOK -- return TRUE iff a signature appears to have valid
**                       domain correlation; that is, "i" must be the same
**                       domain as or a subdomain of "d"
**
**  Parameters:
**  	dkim -- DKIM handle
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff the "i" parameter and the "d" parameter match up.
*/

static bool
dkim_sig_domainok(DKIM *dkim, DKIM_SET *set)
{
	char *at;
	char *dot;
	char *i;
	char *d;

	assert(dkim != NULL);
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	i = dkim_param_get(set, "i");
	d = dkim_param_get(set, "d");

	assert(i != NULL);
	assert(d != NULL);

	at = strchr(i, '@');
	if (at == NULL)
		return FALSE;

	if (strcasecmp(at + 1, d) == 0)
		return TRUE;

	for (dot = strchr(at, '.'); dot != NULL; dot = strchr(dot + 1, '.'))
	{
		if (strcasecmp(dot + 1, d) == 0)
		{
			dkim->dkim_subdomain = TRUE;
			return TRUE;
		}
	}

	return FALSE;
}

/*
**  DKIM_SIG_EXPIRED -- return TRUE iff a signature appears to have expired
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff "set" contains an "x=" parameter which indicates a time
**  	which has passed.
**
**  Notes:
**  	Syntax is not checked here.  It should be checked elsewhere.
*/

static bool
dkim_sig_expired(DKIM_SET *set)
{
	time_t expire;
	time_t now;
	u_char *val;
	char *end;

	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	val = dkim_param_get(set, "x");
	if (val == NULL)
		return FALSE;

	expire = strtoul(val, &end, 10);

	(void) time(&now);

	return (now >= expire);
}

/*
**  DKIM_SIG_VERSIONOK -- return TRUE iff a signature appears to have a version
**                        we can accept
**
**  Parameters:
**  	set -- signature set to be checked
**
**  Return value:
**  	TRUE iff "set" appears to be based on a version of DKIM that is
**  	supported by this API.
*/

static bool
dkim_sig_versionok(DKIM_SET *set)
{
	assert(set != NULL);
	assert(set->set_type == DKIM_SETTYPE_SIGNATURE);

	return (dkim_param_get(set, "v") == NULL);
}

/*
**  DKIM_TMPFILE -- open a temporary file
**
**  Parameters:
**  	dkim -- DKIM handle
**  	fp -- descriptor (returned)
**  	keep -- if FALSE, unlink() the file once created
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_tmpfile(DKIM *dkim, int *fp, bool keep)
{
	int fd;
	char path[MAXPATHLEN + 1];

	assert(dkim != NULL);
	assert(fp != NULL);

	if (dkim->dkim_id != NULL)
	{
		snprintf(path, MAXPATHLEN, "%s/dkim.%s.XXXXXX",
		         dkim->dkim_libhandle->dkiml_tmpdir, dkim->dkim_id);
	}
	else
	{
		snprintf(path, MAXPATHLEN, "%s/dkim.XXXXXX",
		         dkim->dkim_libhandle->dkiml_tmpdir);
	}

	fd = mkstemp(path);
	if (fd == -1)
		return DKIM_STAT_NORESOURCE;

	*fp = fd;

	if (!keep)
		(void) unlink(path);

	return DKIM_STAT_OK;
}

/*
**  DKIM_CANONINIT -- initialize canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**  	htype -- type of the hash to be performed
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_canoninit(DKIM *dkim, int htype)
{
	bool tmp = FALSE;
	bool keep = FALSE;

	assert(dkim != NULL);
#ifdef SHA256_DIGEST_LENGTH
	assert(htype == DKIM_HASHTYPE_SHA1 || htype == DKIM_HASHTYPE_SHA256);
#else /* SHA256_DIGEST_LENGTH */
	assert(htype == DKIM_HASHTYPE_SHA1);
#endif /* SHA256_DIGEST_LENGTH */
	assert(dkim->dkim_hash == NULL);

	tmp = (dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_TMPFILES) != 0;
	keep = (dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_KEEPFILES) != 0;

	switch (htype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1_h;
		struct dkim_sha1 *sha1_b;

		sha1_h = (struct dkim_sha1 *) DKIM_MALLOC(dkim,
		                                          sizeof(struct dkim_sha1));
		if (sha1_h == NULL)
			return DKIM_STAT_NORESOURCE;

		sha1_b = (struct dkim_sha1 *) DKIM_MALLOC(dkim,
		                                          sizeof(struct dkim_sha1));
		if (sha1_b == NULL)
		{
			DKIM_FREE(dkim, sha1_h);
			return DKIM_STAT_NORESOURCE;
		}

		memset(sha1_h, '\0', sizeof(struct dkim_sha1));
		memset(sha1_b, '\0', sizeof(struct dkim_sha1));
		SHA1_Init(&sha1_h->sha1_ctx);
		SHA1_Init(&sha1_b->sha1_ctx);

		dkim->dkim_hash = sha1_h;
		dkim->dkim_bhash = sha1_b;

		if (tmp)
		{
			DKIM_STAT status;

			status = dkim_tmpfile(dkim, &sha1_h->sha1_tmpfd, keep);
			if (status != DKIM_STAT_OK)
			{
				DKIM_FREE(dkim, sha1_h);
				DKIM_FREE(dkim, sha1_b);
				return status;
			}

			status = dkim_tmpfile(dkim, &sha1_b->sha1_tmpfd, keep);
			if (status != DKIM_STAT_OK)
			{
				close(sha1_h->sha1_tmpfd);
				DKIM_FREE(dkim, sha1_h);
				DKIM_FREE(dkim, sha1_b);
				return status;
			}

			sha1_h->sha1_tmpbio = BIO_new_fd(sha1_h->sha1_tmpfd, 1);
			sha1_b->sha1_tmpbio = BIO_new_fd(sha1_b->sha1_tmpfd, 1);
		}

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256_h;
		struct dkim_sha256 *sha256_b;

		sha256_h = (struct dkim_sha256 *) DKIM_MALLOC(dkim,
		                                              sizeof(struct dkim_sha256));
		if (sha256_h == NULL)
			return DKIM_STAT_NORESOURCE;

		sha256_b = (struct dkim_sha256 *) DKIM_MALLOC(dkim,
		                                              sizeof(struct dkim_sha256));
		if (sha256_b == NULL)
		{
			DKIM_FREE(dkim, sha256_h);
			return DKIM_STAT_NORESOURCE;
		}

		memset(sha256_h, '\0', sizeof(struct dkim_sha256));
		memset(sha256_b, '\0', sizeof(struct dkim_sha256));
		SHA256_Init(&sha256_h->sha256_ctx);
		SHA256_Init(&sha256_b->sha256_ctx);

		dkim->dkim_hash = sha256_h;
		dkim->dkim_bhash = sha256_b;

		if (tmp)
		{
			DKIM_STAT status;

			status = dkim_tmpfile(dkim, &sha256_h->sha256_tmpfd,
			                      keep);
			if (status != DKIM_STAT_OK)
			{
				DKIM_FREE(dkim, sha256_h);
				DKIM_FREE(dkim, sha256_b);
				return status;
			}

			status = dkim_tmpfile(dkim, &sha256_b->sha256_tmpfd,
			                      keep);
			if (status != DKIM_STAT_OK)
			{
				close(sha256_h->sha256_tmpfd);
				DKIM_FREE(dkim, sha256_h);
				DKIM_FREE(dkim, sha256_b);
				return status;
			}

			sha256_h->sha256_tmpbio = BIO_new_fd(sha256_h->sha256_tmpfd,
			                                     1);
			sha256_b->sha256_tmpbio = BIO_new_fd(sha256_b->sha256_tmpfd,
			                                     1);
		}

		break;
	  }
#endif /* SHA256_DIGEST_LENGTH */
	}

	dkim->dkim_hashtype = htype;

	return DKIM_STAT_OK;
}

/*
**  DKIM_CANONWRITE -- write data to canonicalization stream(s)
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer containing canonicalized data
**  	buflen -- number of bytes to consume
**
**  Return value:
**  	None.
*/

static void
dkim_canonwrite(DKIM *dkim, u_char *buf, size_t buflen)
{
	u_long wlen;

	assert(dkim != NULL);

	if (buf == NULL || buflen == 0)
		return;

	assert(dkim->dkim_hash != NULL);

	if (dkim->dkim_writesep &&
	    dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			SHA1_Update(&sha1->sha1_ctx, CRLF, 2);

			if (sha1->sha1_tmpbio != NULL)
				BIO_write(sha1->sha1_tmpbio, CRLF, 2);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			SHA256_Update(&sha256->sha256_ctx, CRLF, 2);

			if (sha256->sha256_tmpbio != NULL)
				BIO_write(sha256->sha256_tmpbio, CRLF, 2);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}
	}

	dkim->dkim_writesep = FALSE;

	if (dkim->dkim_state == DKIM_STATE_BODY)
	{
		wlen = (u_long) MIN((u_long) buflen,
		                    (u_long) dkim->dkim_signlen);
		if (wlen == 0)
		{
			dkim->dkim_partial = TRUE;
			return;
		}
	}
	else
	{
		wlen = (u_long) buflen;
	}

	switch (dkim->dkim_hashtype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1;

		if (dkim->dkim_state >= DKIM_STATE_BODY &&
		    !dkim->dkim_bodydone &&
		    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
		else
			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

		SHA1_Update(&sha1->sha1_ctx, buf, wlen);

		if (sha1->sha1_tmpbio != NULL)
			BIO_write(sha1->sha1_tmpbio, buf, wlen);

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256;

		if (dkim->dkim_state >= DKIM_STATE_BODY &&
		    !dkim->dkim_bodydone &&
		    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
		else
			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

		SHA256_Update(&sha256->sha256_ctx, buf, wlen);

		if (sha256->sha256_tmpbio != NULL)
			BIO_write(sha256->sha256_tmpbio, buf, wlen);

		break;
	  }
#endif /*SHA256_DIGEST_LENGTH */
	}

	if (dkim->dkim_state == DKIM_STATE_BODY)
	{
		dkim->dkim_bodylen += buflen;
		dkim->dkim_signlen -= wlen;
		dkim->dkim_canonlen += wlen;
	}
}

/*
**  DKIM_CANONHEADER -- canonicalize a header and write it
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdr -- header handle
**  	crlf -- write a CRLF at the end?
**
**  Return value:
**  	A DKIM_STAT constant.
*/

static DKIM_STAT
dkim_canonheader(DKIM *dkim, struct dkim_header *hdr, bool crlf)
{
	bool space;
	bool first;
	bool colon;
	int n;
	u_char *p;
	char tmp[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(hdr != NULL);

	memset(tmp, '\0', sizeof tmp);
	n = 0;

	switch (dkim->dkim_hdrcanonalg)
	{
	  case DKIM_CANON_SIMPLE:
		if (sm_strlcat(tmp, hdr->hdr_text, sizeof tmp) >= sizeof tmp ||
		    (crlf && sm_strlcat(tmp, CRLF, sizeof tmp) >= sizeof tmp))
				return DKIM_STAT_INTERNAL;
		break;

	  case DKIM_CANON_NOWSP:
		for (p = hdr->hdr_text; *p != '\0'; p++)
		{
			if (isascii(*p) && isspace(*p))
				continue;

			tmp[n++] = *p;
			if (n == sizeof tmp)
				return DKIM_STAT_INTERNAL;
		}
		dkim_lowerhdr(tmp);
		if (crlf && sm_strlcat(tmp, CRLF, sizeof tmp) >= sizeof tmp)
			return DKIM_STAT_INTERNAL;
		break;

	  case DKIM_CANON_RELAXED:
		space = FALSE;
		first = TRUE;
		colon = FALSE;
		for (p = hdr->hdr_text; *p != '\0'; p++)
		{
			if (isascii(*p) && isspace(*p) && space && colon)
				continue;

			if (isascii(*p) && isspace(*p) && !space)
			{
				space = TRUE;
				continue;
			}

			if ((!isascii(*p) || !isspace(*p)) && space)
			{
				if (first)
				{
					first = FALSE;
				}
				else
				{
					tmp[n++] = ' ';
					if (n == sizeof tmp)
						return DKIM_STAT_INTERNAL;
				}
				space = FALSE;
			}

			tmp[n++] = *p;
			if (n == sizeof tmp)
				return DKIM_STAT_INTERNAL;

			if (*p == ':')
				colon = TRUE;
		}

		dkim_lowerhdr(tmp);

		if (crlf && sm_strlcat(tmp, CRLF, sizeof tmp) >= sizeof tmp)
			return DKIM_STAT_INTERNAL;
		break;
	}

	dkim_canonwrite(dkim, tmp, strlen(tmp));

	return DKIM_STAT_OK;
}

/*
**  DKIM_CANONFINAL -- close canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**  	body -- TRUE iff the one to be closed is the body
**
**  Return value:
**  	None.
*/

static void
dkim_canonfinal(DKIM *dkim, bool body)
{
	assert(dkim != NULL);
	assert(dkim->dkim_hash != NULL);

	switch (dkim->dkim_hashtype)
	{
	  case DKIM_HASHTYPE_SHA1:
	  {
		struct dkim_sha1 *sha1;

		if (dkim->dkim_bhash != NULL && body)
		{
			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
			SHA1_Final(sha1->sha1_out, &sha1->sha1_ctx);
		}
		else
		{
			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;
			SHA1_Final(sha1->sha1_out, &sha1->sha1_ctx);
		}

		break;
	  }

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_HASHTYPE_SHA256:
	  {
		struct dkim_sha256 *sha256;

		if (dkim->dkim_bhash != NULL && body)
		{
			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
			SHA256_Final(sha256->sha256_out, &sha256->sha256_ctx);
		}
		else
		{
			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;
			SHA256_Final(sha256->sha256_out, &sha256->sha256_ctx);
		}

		break;
	  }
#endif /* SHA256_DIGEST_LENGTH */

	  default:
		assert(0);
	}
}

/*
**  DKIM_FLUSH_BLANKS -- use accumulated blank lines in canonicalization
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	None.
*/

static void
dkim_flush_blanks(DKIM *dkim)
{
	int c;

	assert(dkim != NULL);

	for (c = 0; c < dkim->dkim_blanks; c++)
		dkim_canonwrite(dkim, CRLF, 2);
	dkim->dkim_blanks = 0;
}

/*
**  DKIM_GENSIGHDR -- generate a signature header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- where to write
**  	buflen -- bytes available at "buf"
**  	delim -- delimiter
**
**  Return value:
**  	Number of bytes written to "buf".
*/

static size_t
dkim_gensighdr(DKIM *dkim, u_char *buf, size_t buflen, char *delim)
{
	bool firsthdr;
	u_char *colon;
	struct dkim_header *hdr;
	u_char tmp[MAXHEADER + 1];
	u_char tmphdr[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(buf != NULL);
	assert(delim != NULL);

	/*
	**  We need to generate a DKIM-Signature: header template
	**  and include it in the canonicalization.
	*/

	/* basic required stuff */
	snprintf(tmphdr, sizeof tmphdr,
	         "a=%s;%sc=%s/%s;%sd=%s;%ss=%s;%st=%lu",
	         dkim_code_to_name(algorithms, dkim->dkim_signalg), delim,
	         dkim_code_to_name(canonicalizations, dkim->dkim_hdrcanonalg),
	         dkim_code_to_name(canonicalizations,
	                           dkim->dkim_bodycanonalg), delim,
	         dkim->dkim_domain, delim,
	         dkim->dkim_selector, delim,
	         dkim->dkim_timestamp);

	if (dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01 &&
	    dkim->dkim_bhash != NULL)
	{
		int status;
		size_t hashlen;
		unsigned char *hash;
		char b64hash[MAXHEADER + 1];

		memset(b64hash, '\0', sizeof b64hash);

		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
			hash = (void *) &sha1->sha1_out;
			hashlen = SHA_DIGEST_LENGTH;

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
			hash = (void *) &sha256->sha256_out;
			hashlen = SHA256_DIGEST_LENGTH;

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}

		status = dkim_base64_encode(hash, hashlen,
		                            b64hash, sizeof b64hash);

		snprintf(tmp, sizeof tmp, ";%sbh=%s", delim,
		         b64hash);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	/* l= */
	if (dkim->dkim_partial)
	{
		snprintf(tmp, sizeof tmp, ";%sl=%lu", delim,
		         (u_long) dkim->dkim_canonlen);
		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	/* h= */
	firsthdr = TRUE;
	for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
	{
		if ((hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
			continue;

		colon = strchr(hdr->hdr_text, ':');
		if (colon == NULL)
		{
			sm_strlcpy(tmp, hdr->hdr_text, sizeof tmp);
		}
		else
		{
			size_t len;

			len = MIN(MAXHEADER, colon - hdr->hdr_text);
			memset(tmp, '\0', sizeof tmp);
			strncpy(tmp, hdr->hdr_text, len);
		}

		if (!firsthdr)
		{
			sm_strlcat(tmphdr, ":", sizeof tmphdr);
		}
		else
		{
			sm_strlcat(tmphdr, ";", sizeof tmphdr);
			sm_strlcat(tmphdr, delim, sizeof tmphdr);
			sm_strlcat(tmphdr, "h=", sizeof tmphdr);
		}

		firsthdr = FALSE;

		sm_strlcat(tmphdr, tmp, sizeof tmphdr);
	}

	/* and finally, an empty b= */
	sm_strlcat(tmphdr, ";", sizeof tmphdr);
	sm_strlcat(tmphdr, delim, sizeof tmphdr);
	sm_strlcat(tmphdr, "b=", sizeof tmphdr);

	sm_strlcpy(buf, tmphdr, buflen);

	return strlen(buf);
}

/*
**  DKIM_GETSENDER -- determine sender (actually just multi-search)
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdrs -- list of header names to find
**  	signedhdrs -- if TRUE, only check signed headers
**
**  Return value:
**  	Pointer to the first such header found, or NULL if none.
*/

static struct dkim_header *
dkim_getsender(DKIM *dkim, u_char **hdrs, bool signedhdrs)
{
	int c;
	size_t clen;
	u_char *colon;
	struct dkim_header *cur;

	assert(dkim != NULL);
	assert(hdrs != NULL);

	for (c = 0; hdrs[c] != NULL; c++)
	{
		for (cur = dkim->dkim_hhead; cur != NULL; cur = cur->hdr_next)
		{
			if (signedhdrs &&
			    (cur->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			colon = strchr(cur->hdr_text, ':');
			if (colon == NULL)
				clen = strlen(cur->hdr_text);
			else
				clen = colon - cur->hdr_text;

			if (strncasecmp(hdrs[c], cur->hdr_text, clen) == 0)
				return cur;
		}
	}

	return NULL;
}

/*
**  DKIM_GET_POLICY -- acquire a domain's policy record
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	0 on policy retrieved (or none found), -1 on failure.
*/

static int
dkim_get_policy(DKIM *dkim)
{
	int qdcount;
	int ancount;
	int status;
	int n;
	int c;
	int type = -1;
	int class = -1;
	bool found;
	size_t anslen;
#if USE_ARLIB
	AR_LIB ar;
	AR_QUERY q;
	int arerror;
#endif /* USE_ARLIB */
	char *ctx;
	unsigned char *label[MAXPOLICYDEPTH];
	unsigned char *p;
	unsigned char *cp;
	unsigned char *eom;
	unsigned char ansbuf[MAXPACKET];
	unsigned char buf[BUFRSZ + 1];
	char qname[MAXHOSTNAMELEN + 1];
	unsigned char domain[MAXHOSTNAMELEN + 1];
#if USE_ARLIB
	struct timeval timeout;
#endif /* USE_ARLIB */
	HEADER hdr;

	assert(dkim != NULL);
	assert(dkim->dkim_domain != NULL);
	assert(dkim->dkim_querymethod == DKIM_QUERY_DNS);

#if USE_ARLIB
	ar = dkim->dkim_libhandle->dkiml_arlib;
	assert(ar != NULL);
#endif /* USE_ARLIB */

	/* make an array of labels */
	for (n = 0; n < MAXPOLICYDEPTH; n++)
		label[n] = NULL;
	sm_strlcpy(domain, dkim->dkim_domain, sizeof domain);
	for (p = strtok_r(domain, ".", &ctx);
	     p != NULL;
	     p = strtok_r(NULL, ".", &ctx))
	{
		for (n = 1; n < MAXPOLICYDEPTH; n++)
			label[n - 1] = label[n];
		label[MAXPOLICYDEPTH - 1] = p;
	}

	/* query upwards */
	found = FALSE;
	for (n = 0; n < MAXPOLICYDEPTH; n++)
	{
		/* send the NS query */
		memset(qname, '\0', sizeof qname);
#ifdef _FFR_ALLMAN_SSP_02
		if (dkim->dkim_userpolicy)
		{
			snprintf(qname, sizeof qname, "%s.%s", dkim->dkim_user,
			         DKIM_DNSUSERPOLICYNAME);
		}
#else /* _FFR_ALLMAN_SSP_02 */
		snprintf(qname, sizeof qname, "%s.%s", DKIM_DNSPOLICYNAME,
		         DKIM_DNSKEYNAME);
#endif /* _FFR_ALLMAN_SSP_02 */

		for (c = n; c < MAXPOLICYDEPTH; c++)
		{
			if (label[c] == NULL)
				continue;
			if (qname[0] != '\0')
				sm_strlcat(qname, ".", sizeof qname);
			sm_strlcat(qname, label[c], sizeof qname);
		}

#if USE_ARLIB
		timeout.tv_sec = dkim->dkim_timeout;
		timeout.tv_usec = 0;

#ifdef _FFR_ALLMAN_SSP_02
		q = ar_addquery(ar, qname, C_IN, T_DKIMP, MAXCNAMEDEPTH,
		                ansbuf, sizeof ansbuf, &arerror,
		                dkim->dkim_timeout == 0 ? NULL : &timeout);
#else /* _FFR_ALLMAN_SSP_02 */
		q = ar_addquery(ar, qname, C_IN, T_TXT, MAXCNAMEDEPTH, ansbuf,
		                sizeof ansbuf, &arerror,
		                dkim->dkim_timeout == 0 ? NULL : &timeout);
#endif /* _FFR_ALLMAN_SSP_02 */
		if (q == NULL)
			return -1;

# if _FFR_SET_DNS_CALLBACK
		if (dkim->dkim_dns_callback == NULL)
		{
			status = ar_waitreply(ar, q, NULL, NULL);
		}
		else
		{
			for (;;)
			{
				timeout.tv_sec = dkim->dkim_callback_int;
				timeout.tv_usec = 0;

				status = ar_waitreply(ar, q, NULL, &timeout);

				if (status != AR_STAT_NOREPLY)
					break;

				dkim->dkim_dns_callback(dkim->dkim_user_context);
			}
		}
# else /* _FFR_SET_DNS_CALLBACK */
		status = ar_waitreply(ar, q, NULL, NULL);
# endif /* _FFR_SET_DNS_CALLBACK */

		(void) ar_cancelquery(ar, q);
#else /* USE_ARLIB */

# ifdef _FFR_ALLMAN_SSP_02
		status = res_query(qname, C_IN, T_DKIMP, ansbuf,
		                   sizeof ansbuf);
# else /* _FFR_ALLMAN_SSP_02 */
		status = res_query(qname, C_IN, T_TXT, ansbuf, sizeof ansbuf);
# endif /* _FFR_ALLMAN_SSP_02 */

#endif /* USE_ARLIB */

#if USE_ARLIB
		if (status == AR_STAT_ERROR || status == AR_STAT_EXPIRED)
			return -1;
#else /* USE_ARLIB */
		/*
		**  A -1 return from res_query could mean a bunch of things,
		**  not just NXDOMAIN.  You can use h_errno to determine what
		**  -1 means.  This is poorly documented.
		*/

		if (status == -1)
		{
			switch (h_errno)
			{
			  case HOST_NOT_FOUND:
			  case NO_DATA:
				continue;

			  case TRY_AGAIN:
			  case NO_RECOVERY:
			  default:
				return -1;
			}
		}
#endif /* USE_ARLIB */

		/* set up pointers */
		anslen = sizeof ansbuf;
		memcpy(&hdr, ansbuf, sizeof hdr);
		cp = (u_char *) &ansbuf + HFIXEDSZ;
		eom = (u_char *) &ansbuf + anslen;

		/* skip over the name at the front of the answer */
		for (qdcount = ntohs((unsigned short) hdr.qdcount);
		     qdcount > 0;
		     qdcount--)
		{
			/* copy it first */
			(void) dn_expand((unsigned char *) &ansbuf, eom, cp,
			                 qname, sizeof qname);

			if ((n = dn_skipname(cp, eom)) < 0)
				return -1;
			cp += n;

			/* extract the type and class */
			if (cp + INT16SZ + INT16SZ > eom)
				return -1;

			GETSHORT(type, cp);
			GETSHORT(class, cp);
		}

		if (type != T_TXT || class != C_IN)
			return -1;

		/* if NXDOMAIN, return DKIM_STAT_CANTVRFY */
		if (hdr.rcode == NXDOMAIN)
			continue;

		/* get the answer count */
		ancount = ntohs((unsigned short) hdr.ancount);
		if (ancount == 0)
			continue;

		/* got something useable! */
		found = TRUE;
		break;
	}

	/* no policy found; apply defaults */
	if (!found)
	{
		status = dkim_process_set(dkim, DKIM_SETTYPE_POLICY,
		                          DKIM_POLICY_DEFAULTTXT,
		                          strlen(DKIM_POLICY_DEFAULTTXT));

		return (status == DKIM_STAT_OK ? 0 : -1);
	}

	/* if truncated, we can't do it */
	if (hdr.tc)
		return -1;

	/* grab the label, even though we know what we asked... */
	if ((n = dn_expand((unsigned char *) &ansbuf, eom, cp,
	                   (RES_UNC_T) qname, sizeof qname)) < 0)
		return -1;
	/* ...and move past it */
	cp += n;

	/* extract the type and class */
	if (cp + INT16SZ + INT16SZ > eom)
		return -1;
	GETSHORT(type, cp);
	GETSHORT(class, cp);

	/* reject anything that's not valid (stupid wildcards) */
	if (type != T_TXT || class != C_IN)
		return -1;

	/* skip the TTL */
	cp += INT32SZ;

	/* get payload length */
	if (cp + INT16SZ > eom)
		return -1;
	GETSHORT(n, cp);

	/* XXX -- maybe deal with a partial reply rather than require it all */
	if (cp + n > eom)
		return -1;

	if (n > BUFRSZ)
		return -1;

	/* extract the payload */
	memset(buf, '\0', sizeof buf);
	p = buf;
	while (n > 0)
	{
		c = *cp++;
		n--;
		while (c > 0)
		{
			*p++ = *cp++;
			c--;
			n--;
		}
	}

	/* decode the payload */
	status = dkim_process_set(dkim, DKIM_SETTYPE_POLICY, buf, strlen(buf));

#ifdef _FFR_ALLMAN_SSP_02
	/* if this is a site policy and requires user re-querying, do so */
	if (!dkim->dkim_userpolicy)
	{
		DKIM_SET *s;

		s = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
		assert(s != NULL);

		p = dkim_param_get(s, "u");
		if (p != NULL && strcasecmp(p, "yes") == 0)
		{
			dkim->dkim_userpolicy = TRUE;
			return dkim_get_policy(dkim);
		}
	}
#endif /* _FFR_ALLMAN_SSP_02 */

	return (status == DKIM_STAT_OK ? 0 : -1);
}

/*
**  DKIM_POLICY_FLAG -- see if a policy flag is set
**
**  Parameters:
**  	dkim -- DKIM handle
**  	flag -- flag to check
**
**  Return value:
**  	1 -- flag set
**  	0 -- flag not set
**  	-1 -- error
*/

static int
dkim_policy_flag(DKIM *dkim, int flag)
{
	struct dkim_set *set;
	u_char *param;

	assert(dkim != NULL);
	assert(flag != '\0');

	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
		return -1;

	param = dkim_param_get(set, "t");
	if (param != NULL)
		return (strchr(param, flag) == NULL) ? 0 : 1;
	else
		return 0;
}

/*
**  DKIM_GET_KEY -- acquire a public key used for verification
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

static DKIM_STAT
dkim_get_key(DKIM *dkim)
{
	int status;
	struct dkim_set *set;
	unsigned char *p;
	unsigned char buf[BUFRSZ + 1];

	assert(dkim != NULL);
	assert(dkim->dkim_selector != NULL);
	assert(dkim->dkim_domain != NULL);

	/* use appropriate get method */
	switch (dkim->dkim_querymethod)
	{
	  case DKIM_QUERY_DNS:
		status = (int) dkim_get_key_dns(dkim, buf, sizeof buf);
		if (status != (int) DKIM_STAT_OK)
			return (DKIM_STAT) status;
		break;

	  default:
		assert(0);
	}

	/* decode the payload */
	status = dkim_process_set(dkim, DKIM_SETTYPE_KEY, buf, strlen(buf));
	if (status != DKIM_STAT_OK)
		return status;

	/* verify key version first */
	set = dkim_set_first(dkim, DKIM_SETTYPE_KEY);
	assert(set != NULL);
	p = dkim_param_get(set, "v");
	if (p == NULL || strcmp(p, DKIM_VERSION) != 0)
		return DKIM_STAT_SYNTAX;

	/* then make sure the hash type is something we can handle */
	p = dkim_param_get(set, "h");
	if (!dkim_key_hashesok(p))
		return DKIM_STAT_SYNTAX;

	/* then key type */
	p = dkim_param_get(set, "k");
	if (p == NULL || dkim_name_to_code(keytypes, p) == -1)
		return DKIM_STAT_SYNTAX;

	/* decode the key */
	dkim->dkim_b64key = dkim_param_get(set, "p");
	if (dkim->dkim_b64key == NULL)
		return DKIM_STAT_SYNTAX;
	else if (dkim->dkim_b64key[0] == '\0')
		return DKIM_STAT_REVOKED;
	dkim->dkim_b64keylen = strlen(dkim->dkim_b64key);

	dkim->dkim_key = DKIM_MALLOC(dkim, dkim->dkim_b64keylen);
	if (dkim->dkim_key == NULL)
		return DKIM_STAT_NORESOURCE;

	status = dkim_base64_decode(dkim->dkim_b64key, dkim->dkim_key,
	                            dkim->dkim_b64keylen);
	if (status < 0)
		return DKIM_STAT_SYNTAX;

	dkim->dkim_keylen = status;

	return DKIM_STAT_OK;
}

/*
**  DKIM_NEW -- allocate a new message context
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- transaction ID string
**  	memclosure -- memory closure
**  	hdrcanon_alg -- canonicalization algorithm to use for headers
**  	bodycanon_alg -- canonicalization algorithm to use for headers
**  	sign_alg -- signature algorithm to use
**  	statp -- status (returned)
**
**  Return value:
**  	A new DKIM handle, or NULL on failure.
*/

static DKIM *
dkim_new(DKIM_LIB *libhandle, const char *id, void *memclosure,
         dkim_canon_t hdrcanon_alg, dkim_canon_t bodycanon_alg,
         dkim_alg_t sign_alg, DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);

	/* allocate the handle */
	new = (DKIM *) dkim_malloc(libhandle, memclosure,
	                           sizeof(struct dkim));
	if (new == NULL)
	{
		*statp = DKIM_STAT_NORESOURCE;
		return NULL;
	}

	/* populate defaults */
	memset(new, '\0', sizeof(struct dkim));
	new->dkim_id = id;
	new->dkim_signalg = (sign_alg == -1 ? DKIM_SIGN_DEFAULT
	                                    : sign_alg);
	new->dkim_hdrcanonalg = (hdrcanon_alg == -1 ? DKIM_CANON_DEFAULT
	                                            : hdrcanon_alg);
	new->dkim_bodycanonalg = (bodycanon_alg == -1 ? DKIM_CANON_DEFAULT
	                                              : bodycanon_alg);
	new->dkim_querymethod = DKIM_QUERY_DEFAULT;
	new->dkim_mode = DKIM_MODE_UNKNOWN;
	new->dkim_state = DKIM_STATE_INIT;
	new->dkim_closure = memclosure;
	new->dkim_libhandle = libhandle;
	new->dkim_tmpdir = libhandle->dkiml_tmpdir;
	new->dkim_timeout = libhandle->dkiml_timeout;
	new->dkim_hashtype = DKIM_HASHTYPE_UNKNOWN;
	new->dkim_version = DKIM_VERSION_UNKNOWN;
	new->dkim_blankline = TRUE;

	*statp = DKIM_STAT_OK;
	return new;
}

/* ========================= PUBLIC SECTION ========================== */

/*
**  DKIM_INIT -- initialize a DKIM library context
**
**  Parameters:
**  	caller_mallocf -- caller-provided memory allocation function
**  	caller_freef -- caller-provided memory release function
**
**  Return value:
**  	A new DKIM_LIB handle suitable for use with other DKIM functions, or
**  	NULL on failure.
**
**  Side effects:
**  	Crop circles near Birmingham.
*/

DKIM_LIB *
dkim_init(void *(*caller_mallocf)(void *closure, size_t nbytes),
          void (*caller_freef)(void *closure, void *p))
{
	u_char *td;
	DKIM_LIB *libhandle;

	/* initialize OpenSSL algorithms */
	OpenSSL_add_all_algorithms();

	/* copy the parameters */
	libhandle = (DKIM_LIB *) malloc(sizeof(struct dkim_lib));
	if (libhandle == NULL)
		return NULL;

	td = getenv("DKIM_TMPDIR");
	if (td == NULL || td[0] == '\0')
		td = DEFTMPDIR;

	libhandle->dkiml_signre = FALSE;
	libhandle->dkiml_malloc = caller_mallocf;
	libhandle->dkiml_free = caller_freef;
	sm_strlcpy(libhandle->dkiml_tmpdir, td, 
	           sizeof libhandle->dkiml_tmpdir);
	libhandle->dkiml_flags = DKIM_LIBFLAGS_DEFAULT;
	libhandle->dkiml_timeout = DEFTIMEOUT;
	libhandle->dkiml_senderhdrs = (u_char **) default_senderhdrs;
	libhandle->dkiml_version = DKIM_VERSION_IETF_BASE_02;

	/* initialize the resolver */
#if USE_ARLIB
	libhandle->dkiml_arlib = ar_init(NULL, NULL, NULL, 0);
	if (libhandle->dkiml_arlib == NULL)
	{
		free(libhandle);
		return NULL;
	}
#else /* USE_ARLIB */
	(void) res_init();
#endif /* USE_ARLIB */

	return libhandle;
}

/*
**  DKIM_OPTIONS -- get or set a library option
**
**  Parameters:
**  	lib -- DKIM library handle
**  	op -- operation to perform
**  	opt -- option to get/set
**  	ptr -- pointer to its old/new value
**  	len -- memory available at "ptr"
**
**  Return value:
**  	A DKIM_STAT constant.
**
**  XXX -- the various pieces of the switch section in here are all very
**         similar; some common-factoring is in order
*/

DKIM_STAT
dkim_options(DKIM_LIB *lib, int op, dkim_opts_t opt, void *ptr, size_t len)
{
	assert(lib != NULL);
	assert(op == DKIM_OP_SETOPT || op == DKIM_OP_GETOPT);
	assert(ptr != NULL);
	assert(len != 0);

	switch (opt)
	{
	  case DKIM_OPTS_TMPDIR:
		if (op == DKIM_OP_GETOPT)
		{
			sm_strlcpy((u_char *) ptr,
			           lib->dkiml_tmpdir, len);
		}
		else
		{
			sm_strlcpy(lib->dkiml_tmpdir, (u_char *) ptr,
			           sizeof lib->dkiml_tmpdir);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_FLAGS:
		if (len != sizeof lib->dkiml_flags)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_flags, len);
		}
		else
		{
			memcpy(&lib->dkiml_flags, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_VERSION:
		if (len != sizeof lib->dkiml_version)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_version, len);
		}
		else
		{
			u_int test;

			memcpy(&test, ptr, sizeof test);
			if (test != DKIM_VERSION_ALLMAN_BASE_00 &&
			    test != DKIM_VERSION_IETF_BASE_00 &&
			    test != DKIM_VERSION_IETF_BASE_01 &&
			    test != DKIM_VERSION_IETF_BASE_02)
				return DKIM_STAT_INVALID;

			memcpy(&lib->dkiml_version, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_TIMEOUT:
		if (len != sizeof lib->dkiml_timeout)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_timeout, len);
		}
		else
		{
			memcpy(&lib->dkiml_timeout, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SENDERHDRS:
		if (len != sizeof lib->dkiml_senderhdrs)
			return DKIM_STAT_INVALID;

		if (op == DKIM_OP_GETOPT)
		{
			memcpy(ptr, &lib->dkiml_senderhdrs, len);
		}
		else
		{
			memcpy(&lib->dkiml_senderhdrs, ptr, len);
		}
		return DKIM_STAT_OK;

	  case DKIM_OPTS_SIGNHDRS:
		if (len != sizeof(char **))
		{
			return DKIM_STAT_INVALID;
		}
		else if (ptr == NULL)
		{
			if (lib->dkiml_signre)
			{
				(void) regfree(&lib->dkiml_hdrre);
				lib->dkiml_signre = FALSE;
			}
		}
		else
		{
			int status;
			u_char **hdrs;
			char buf[BUFRSZ + 1];

			if (lib->dkiml_signre)
			{
				(void) regfree(&lib->dkiml_hdrre);
				lib->dkiml_signre = FALSE;
			}

			memset(buf, '\0', sizeof buf);

			hdrs = (u_char **) ptr;

			(void) sm_strlcpy(buf, "^(", sizeof buf);

			if (!dkim_hdrlist(buf, sizeof buf,
			                  (u_char **) default_signhdrs, TRUE))
				return DKIM_STAT_INVALID;
			if (!dkim_hdrlist(buf, sizeof buf, hdrs, FALSE))
				return DKIM_STAT_INVALID;

			if (sm_strlcat(buf, ")$", sizeof buf) >= sizeof buf)
				return DKIM_STAT_INVALID;

			status = regcomp(&lib->dkiml_hdrre, buf,
			                 (REG_EXTENDED|REG_ICASE));
			if (status != 0)
				return DKIM_STAT_INTERNAL;

			lib->dkiml_signre = TRUE;
		}
		return DKIM_STAT_OK;

	  default:
		return DKIM_STAT_INVALID;
	}

	/* to silence -Wall */
	return DKIM_STAT_INTERNAL;
}

#define	CLOBBER(x)	if ((x) != NULL) \
			{ \
				dkim_mfree(dkim->dkim_libhandle, dkim->dkim_closure, (x)); \
				(x) = NULL; \
			}

#define BIO_CLOBBER(x)	if ((x) != NULL) \
			{ \
				BIO_free((x)); \
				(x) = NULL; \
			}

#define RSA_CLOBBER(x)	if ((x) != NULL) \
			{ \
				RSA_free((x)); \
				(x) = NULL; \
			}

#define	EVP_CLOBBER(x)	if ((x) != NULL) \
			{ \
				EVP_PKEY_free((x)); \
				(x) = NULL; \
			}

/*
**  DKIM_FREE -- destroy a DKIM handle
**
**  Parameters:
**  	dkim -- DKIM handle to destroy
**
**  Return value:
**  	A DKIM_STAT constant.
*/

DKIM_STAT
dkim_free(DKIM *dkim)
{
	assert(dkim != NULL);

	if (dkim->dkim_hash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			BIO_CLOBBER(sha1->sha1_tmpbio);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			BIO_CLOBBER(sha256->sha256_tmpbio);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  case DKIM_HASHTYPE_UNKNOWN:
			break;

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_hash);
	}

	if (dkim->dkim_bhash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;

			BIO_CLOBBER(sha1->sha1_tmpbio);

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;

			BIO_CLOBBER(sha256->sha256_tmpbio);

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  case DKIM_HASHTYPE_UNKNOWN:
			break;

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_bhash);
	}

	if (dkim->dkim_signature != NULL)
	{
		switch (dkim->dkim_signalg)
		{
		  case DKIM_SIGN_RSASHA1:
#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_SIGN_RSASHA256:
#endif /* SHA256_DIGEST_LENGTH */
		  {
			struct dkim_rsa *rsa;

			rsa = (struct dkim_rsa *) dkim->dkim_signature;

			EVP_CLOBBER(rsa->rsa_pkey);
			RSA_CLOBBER(rsa->rsa_rsa);
			CLOBBER(rsa->rsa_rsaout);

			break;
		  }

		  default:
			assert(0);
		}

		CLOBBER(dkim->dkim_signature);
	}

	/* blast the headers */
	if (dkim->dkim_hhead != NULL)
	{
		struct dkim_header *next;
		struct dkim_header *hdr;

		for (hdr = dkim->dkim_hhead; hdr != NULL; )
		{
			next = hdr->hdr_next;

			CLOBBER(hdr->hdr_text);
			CLOBBER(hdr);

			hdr = next;
		}
	}

	/* blast the data sets */
	if (dkim->dkim_sethead != NULL)
	{
		DKIM_SET *set;
		DKIM_SET *next;

		for (set = dkim->dkim_sethead; set != NULL; )
		{
			next = set->set_next;

			if (set->set_plist != NULL)
			{
				DKIM_PLIST *plist;
				DKIM_PLIST *pnext;

				for (plist = set->set_plist;
				     plist != NULL; )
				{
					pnext = plist->plist_next;

					CLOBBER(plist);

					plist = pnext;
				}
			}

			CLOBBER(set->set_data);
			CLOBBER(set);

			set = next;
		}
	}

	CLOBBER(dkim->dkim_b64sig);
	CLOBBER(dkim->dkim_hdrlist);
	CLOBBER(dkim->dkim_selector);
	CLOBBER(dkim->dkim_domain);
	CLOBBER(dkim->dkim_user);
	CLOBBER(dkim->dkim_key);
	CLOBBER(dkim->dkim_sender);
	CLOBBER(dkim->dkim_signer);
	CLOBBER(dkim->dkim_sig);

	dkim_mfree(dkim->dkim_libhandle, dkim->dkim_closure, dkim);

	return DKIM_STAT_OK;
}

/*
**  DKIM_SIGN -- allocate a handle for use in a signature operation
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- identification string (e.g. job ID) for logging
**  	memclosure -- memory closure for allocations (or NULL)
**  	secretkey -- secret key (PEM format)
**  	selector -- selector to be used when generating the signature header
**  	domain -- domain for which this message is being signed
**  	hdrcanonalg -- canonicalization algorithm to use for headers
**  	bodycanonalg -- canonicalization algorithm to use for body
**  	signalg -- signing algorithm to use
**  	length -- how many bytes of the body to sign (-1 for all)
**  	statp -- status (returned)
**
**  Return value:
**  	A new signing handle, or NULL.
*/

DKIM *
dkim_sign(DKIM_LIB *libhandle, const char *id, void *memclosure,
          const dkim_sigkey_t *secretkey, const char *selector,
          const char *domain, dkim_canon_t hdrcanonalg,
	  dkim_canon_t bodycanonalg, dkim_alg_t signalg,
          off_t length, DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);
	assert(secretkey != NULL);
	assert(selector != NULL);
	assert(domain != NULL);
	assert(hdrcanonalg == DKIM_CANON_SIMPLE ||
	       hdrcanonalg == DKIM_CANON_NOWSP ||
	       hdrcanonalg == DKIM_CANON_RELAXED);
	assert(bodycanonalg == DKIM_CANON_SIMPLE ||
	       bodycanonalg == DKIM_CANON_NOWSP ||
	       bodycanonalg == DKIM_CANON_RELAXED);
#ifdef SHA256_DIGEST_LENGTH
	assert(signalg == DKIM_SIGN_RSASHA1 || signalg == DKIM_SIGN_RSASHA256);
#else /* SHA256_DIGEST_LENGTH */
	assert(signalg == DKIM_SIGN_RSASHA1);
#endif /* SHA256_DIGEST_LENGTH */
	assert(statp != NULL);

	/* version-specific stuff */
	if (libhandle->dkiml_version != DKIM_VERSION_ALLMAN_BASE_00)
	{
		/* allman-base-00 was the only one that had "nowsp" */
		if (hdrcanonalg == DKIM_CANON_NOWSP ||
		    bodycanonalg == DKIM_CANON_NOWSP)
		{
			*statp = DKIM_STAT_INVALID;
			return NULL;
		}
	}
	if (libhandle->dkiml_version == DKIM_VERSION_ALLMAN_BASE_00)
	{
		/* "relaxed" is only available after allman-base-00 */
		if (hdrcanonalg == DKIM_CANON_RELAXED ||
		    bodycanonalg == DKIM_CANON_RELAXED)
		{
			*statp = DKIM_STAT_INVALID;
			return NULL;
		}

		/* allman-base-00 didn't have split canonicalizations */
		if (hdrcanonalg != bodycanonalg)
		{
			*statp = DKIM_STAT_INVALID;
			return NULL;
		}
	}

	new = dkim_new(libhandle, id, memclosure, hdrcanonalg, bodycanonalg,
	               signalg, statp);

	if (new != NULL)
	{
		new->dkim_mode = DKIM_MODE_SIGN;

		new->dkim_keylen = strlen((const char *) secretkey);
		new->dkim_key = (unsigned char *) DKIM_MALLOC(new,
		                                              new->dkim_keylen + 1);

		if (new->dkim_key == NULL)
		{
			*statp = DKIM_STAT_NORESOURCE;
			dkim_free(new);
			return NULL;
		}

		memcpy(new->dkim_key, (char *) secretkey,
		       new->dkim_keylen + 1);
	}

	new->dkim_selector = dkim_strdup(new, selector, 0);
	new->dkim_domain = dkim_strdup(new, domain, 0);
	if (length == (off_t) -1)
		new->dkim_signlen = ULONG_MAX;
	else
		new->dkim_signlen = length;

	return new;
}

/*
**  DKIM_VERIFY -- allocate a handle for use in a verify operation
**
**  Parameters:
**  	libhandle -- DKIM_LIB handle
**  	id -- identification string (e.g. job ID) for logging
**  	memclosure -- memory closure for allocations (or NULL)
**  	statp -- status (returned)
**
**  Return value:
**  	A new signing handle, or NULL.
*/

DKIM *
dkim_verify(DKIM_LIB *libhandle, const char *id, void *memclosure,
            DKIM_STAT *statp)
{
	DKIM *new;

	assert(libhandle != NULL);
	assert(statp != NULL);

	new = dkim_new(libhandle, id, memclosure, DKIM_CANON_UNKNOWN,
	               DKIM_CANON_UNKNOWN, DKIM_SIGN_UNKNOWN, statp);

	if (new != NULL)
		new->dkim_mode = DKIM_MODE_VERIFY;

	return new;
}

/*
**  DKIM_HEADER -- process a header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hdr -- header text
**  	len -- bytes available at "hdr"
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_header(DKIM *dkim, u_char *hdr, size_t len)
{
	size_t nlen;
	u_char *colon;
	struct dkim_header *h;

	assert(dkim != NULL);
	assert(hdr != NULL);
	assert(len != 0);

	if (dkim->dkim_state > DKIM_STATE_HEADER)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_HEADER;

	h = DKIM_MALLOC(dkim, sizeof(struct dkim_header));

	if (h == NULL)
		return DKIM_STAT_NORESOURCE;

	h->hdr_text = dkim_strdup(dkim, hdr, len);
	h->hdr_flags = 0;
	h->hdr_next = NULL;

	if (dkim->dkim_hhead == NULL)
	{
		dkim->dkim_hhead = h;
		dkim->dkim_htail = h;
	}
	else
	{
		dkim->dkim_htail->hdr_next = h;
		dkim->dkim_htail = h;
	}

	colon = strchr(hdr, ':');
	if (colon != NULL)
	{
		nlen = colon - hdr;
		if (strncasecmp(hdr, DKIM_SIGNHEADER, nlen) == 0)
		{
			DKIM_STAT status;

			status = dkim_process_set(dkim, DKIM_SETTYPE_SIGNATURE,
			                          colon + 1, len - nlen);

			if (status != DKIM_STAT_OK)
				return status;
		}
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_EOH -- declare end-of-headers
**
**  Parameters:
**  	dkim -- DKIM handle
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_eoh(DKIM *dkim)
{
	DKIM_STAT status;
	int htype = DKIM_HASHTYPE_UNKNOWN;
	int nhdrs;
	int c;
	struct dkim_header *hdr;
	DKIM_SET *set;
	u_char *hdrlist;
	struct dkim_header *hdrset[MAXHDRCNT];

	assert(dkim != NULL);

	if (dkim->dkim_state >= DKIM_STATE_EOH)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_EOH;

	/* select headers */
	memset(hdrset, '\0', sizeof hdrset);
	if (dkim->dkim_mode == DKIM_MODE_VERIFY)
	{
		u_char *param;

		/* pick first signature that looks useable */
		for (set = dkim_set_first(dkim, DKIM_SETTYPE_SIGNATURE);
		     set != NULL;
		     set = dkim_set_next(set, DKIM_SETTYPE_SIGNATURE))
		{
			/*
			**  XXX -- move all of this to dkim_eom() to support
			**         the idea of trying each signature in order
			**         hoping for one that passes?
			*/

			hdrlist = dkim_param_get(set, "h");
			nhdrs = dkim_selecthdrs(dkim, hdrlist, hdrset,
			                        MAXHDRCNT);
			if (nhdrs == -1)
				return DKIM_STAT_INTERNAL;

			if (dkim_sig_versionok(set) &&
			    dkim_sig_domainok(dkim, set) &&
			    dkim_sig_signerok(dkim, set,
			                      dkim->dkim_libhandle->dkiml_senderhdrs) &&
			    !dkim_sig_expired(set))
				break;
		}

		if (set == NULL)
			return DKIM_STAT_NOSIG;

		param = dkim_param_get(set, "l");
		if (param == NULL)
		{
			dkim->dkim_signlen = ULONG_MAX;
		}
		else
		{
			char *q;

			dkim->dkim_signlen = strtoul(param, &q, 10);
			if (*q != '\0')
				return DKIM_STAT_SYNTAX;
		}

		param = dkim_param_get(set, "c");
		if (param != NULL)
		{
			u_char *p;
			u_char tmp[BUFRSZ + 1];

			sm_strlcpy(tmp, param, sizeof tmp);
			p = strchr(tmp, '/');
			if (p != NULL)
				*p = '\0';

			dkim->dkim_hdrcanonalg = dkim_name_to_code(canonicalizations,
			                                           tmp);
			if (dkim->dkim_hdrcanonalg == -1)
				return DKIM_STAT_SYNTAX;

			if (p == NULL)
			{
				if (dkim->dkim_hdrcanonalg == DKIM_CANON_NOWSP)
					dkim->dkim_bodycanonalg = DKIM_CANON_NOWSP;
				else
					dkim->dkim_bodycanonalg = DKIM_CANON_SIMPLE;
			}
			else
			{
				dkim->dkim_bodycanonalg = dkim_name_to_code(canonicalizations,
				                                            p + 1);
				if (dkim->dkim_bodycanonalg == -1)
					return DKIM_STAT_SYNTAX;
			}
		}

		param = dkim_param_get(set, "a");
		if (param != NULL)
		{
			dkim->dkim_signalg = dkim_name_to_code(algorithms,
			                                       param);
			if (dkim->dkim_signalg == -1)
				return DKIM_STAT_SYNTAX;
		}

		param = dkim_param_get(set, "q");
		if (param != NULL)
		{
			u_char *opts;
			u_char tmp[BUFRSZ + 1];

			opts = strchr(param, ':');
			if (opts != NULL)
			{
				sm_strlcpy(tmp, param, sizeof tmp);
				opts = strchr(tmp, ':');
				*opts = '\0';
				opts++;
				param = tmp;
			}

			dkim->dkim_querymethod = dkim_name_to_code(querytypes,
			                                           param);
			if (dkim->dkim_querymethod == -1)
				return DKIM_STAT_SYNTAX;

			if (dkim->dkim_querymethod == DKIM_QUERY_DNS &&
			    opts != NULL)
				return DKIM_STAT_SYNTAX;
		}

		param = dkim_param_get(set, "d");
		if (param == NULL)
			return DKIM_STAT_SYNTAX;
		dkim->dkim_domain = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_domain == NULL)
			return DKIM_STAT_NORESOURCE;

		param = dkim_param_get(set, "s");
		if (param == NULL)
			return DKIM_STAT_SYNTAX;
		dkim->dkim_selector = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_selector == NULL)
			return DKIM_STAT_NORESOURCE;

		param = dkim_param_get(set, "b");
		if (param == NULL)
			return DKIM_STAT_SYNTAX;
		dkim->dkim_b64sig = dkim_strdup(dkim, param, 0);
		if (dkim->dkim_b64sig == NULL)
			return DKIM_STAT_NORESOURCE;
		dkim->dkim_b64siglen = strlen(dkim->dkim_b64sig);

		dkim->dkim_sig = DKIM_MALLOC(dkim, dkim->dkim_b64siglen);
		if (dkim->dkim_sig == NULL)
			return DKIM_STAT_NORESOURCE;

		status = dkim_base64_decode(dkim->dkim_b64sig,
		                            dkim->dkim_sig,
		                            dkim->dkim_b64siglen);
		if (status < 0)
			return DKIM_STAT_INTERNAL;
		else
			dkim->dkim_siglen = status;

		/* this is the one we will use */
		dkim->dkim_sigset = set;

		/* determine version */
		if (dkim->dkim_version == DKIM_VERSION_UNKNOWN &&
		    dkim->dkim_bodycanonalg == DKIM_CANON_NOWSP)
			dkim->dkim_version = DKIM_VERSION_ALLMAN_BASE_00;

		if (dkim->dkim_version == DKIM_VERSION_UNKNOWN &&
		    dkim_param_get(set, "bh") == NULL)
			dkim->dkim_version = DKIM_VERSION_IETF_BASE_00;

		if (dkim->dkim_version == DKIM_VERSION_UNKNOWN)
			dkim->dkim_version = DKIM_VERSION_DEFAULT;
	}
	else
	{
		u_int signcnt = 0;
		size_t len = 0;
		u_char *colon;
		u_char *p;
		u_char *q;

		dkim->dkim_version = dkim->dkim_libhandle->dkiml_version;

		/* tag headers to be signed */
		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (!dkim->dkim_libhandle->dkiml_signre)
			{
				hdr->hdr_flags |= DKIM_HDR_SIGNED;
				signcnt++;
				continue;
			}

			colon = strchr(hdr->hdr_text, ':');

			if (colon != NULL)
				*colon = '\0';
			status = regexec(&dkim->dkim_libhandle->dkiml_hdrre,
			                 hdr->hdr_text, 0, NULL, 0);
			if (colon != NULL)
				*colon = ':';

			if (status == 0)
			{
				hdr->hdr_flags |= DKIM_HDR_SIGNED;
				signcnt++;
			}
			else
			{
				assert(status == REG_NOMATCH);
			}
		}

		/* figure out how much space we need */
		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (signcnt > 0 &&
			    (hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			colon = strchr(hdr->hdr_text, ':');

			if (len > 0)
				len++;				/* colon */

			if (colon == NULL)
			{
				len += strlen(hdr->hdr_text);
			}
			else
			{
				len += colon - hdr->hdr_text;
			}
		}

		/* now build a header list */
		dkim->dkim_hdrlist = DKIM_MALLOC(dkim, len + 1);
		if (dkim->dkim_hdrlist == NULL)
			return DKIM_STAT_INTERNAL;
		memset(dkim->dkim_hdrlist, '\0', len + 1);
		hdrlist = dkim->dkim_hdrlist;
		q = hdrlist;

		for (hdr = dkim->dkim_hhead; hdr != NULL; hdr = hdr->hdr_next)
		{
			if (signcnt > 0 &&
			    (hdr->hdr_flags & DKIM_HDR_SIGNED) == 0)
				continue;

			if (q != hdrlist)
			{
				*q = ':';
				q++;
			}

			for (p = hdr->hdr_text; *p != '\0' && *p != ':'; p++)
			{
				*q = *p;
				q++;
			}
		}

		/* and finally, select the headers based on the DKIM spec */
		nhdrs = dkim_selecthdrs(dkim, hdrlist, hdrset, MAXHDRCNT);
		if (nhdrs == -1)
			return DKIM_STAT_INTERNAL;

		if (nhdrs >= MAXHDRCNT)
			return DKIM_STAT_INTERNAL;
	}

	/* start up canonicalization */
	switch (dkim->dkim_signalg)
	{
	  case DKIM_SIGN_RSASHA1:
		htype = DKIM_HASHTYPE_SHA1;
		break;

#ifdef SHA256_DIGEST_LENGTH
	  case DKIM_SIGN_RSASHA256:
		htype = DKIM_HASHTYPE_SHA256;
		break;
#endif /* SHA256_DIGEST_LENGTH */
	}

	status = dkim_canoninit(dkim, htype);
	if (status != DKIM_STAT_OK)
		return status;

	/* canonicalize headers */
	for (c = 0; c < nhdrs; c++)
	{
		status = dkim_canonheader(dkim, hdrset[c], TRUE);
		if (status != DKIM_STAT_OK)
			return status;
	}
		
	dkim->dkim_writesep = TRUE;

	return DKIM_STAT_OK;
}

/*
**  DKIM_BODY -- pass a body chunk in for processing
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- body chunk
**  	buflen -- number of bytes at "buf"
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_body(DKIM *dkim, u_char *buf, size_t buflen)
{
	size_t wlen;
	size_t lidx;
	u_char *p;
	u_char *eob;
	u_char *wrote;
	unsigned char lbuf[BUFRSZ];

	assert(dkim != NULL);
	assert(buf != NULL);

	if (dkim->dkim_state > DKIM_STATE_BODY)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_BODY;

	wrote = buf;
	wlen = 0;
	lidx = 0;
	eob = buf + buflen - 1;

	/* canonicalize */
	switch (dkim->dkim_bodycanonalg)
	{
	  case DKIM_CANON_SIMPLE:
	  case DKIM_CANON_RELAXED:
		for (p = buf; p <= eob; p++)
		{
			if (*p == '\n')
			{
				if (dkim->dkim_lastchar == '\r')
				{
					if (dkim->dkim_blankline)
					{
						dkim->dkim_blanks++;
					}
					else if (wlen == 1)
					{
						dkim_canonwrite(dkim, CRLF, 2);
					}
					else
					{
						dkim_canonwrite(dkim, wrote,
						                wlen + 1);
					}

					wrote = p + 1;
					wlen = 0;
					dkim->dkim_blankline = TRUE;
				}
			}
			else if (dkim->dkim_bodycanonalg == DKIM_CANON_RELAXED)
			{
				/* non-space after a space */
				if (dkim_islwsp(dkim->dkim_lastchar) &&
				    !dkim_islwsp(*p) && *p != '\r')
				{
					dkim_flush_blanks(dkim);
					dkim_canonwrite(dkim, SP, 1);
					wrote = p;
					wlen = 1;
					dkim->dkim_blankline = FALSE;
				}

				/* space after a non-space */
				else if (!dkim_islwsp(dkim->dkim_lastchar) &&
				         dkim_islwsp(*p))
				{
					dkim_canonwrite(dkim, wrote, wlen);
					wlen = 0;
					wrote = p + 1;
				}

				/* space after a space */
				else if (dkim_islwsp(dkim->dkim_lastchar) &&
				         dkim_islwsp(*p))
				{
					/* XXX -- fix logic */
					;
				}

				/* non-space after a non-space */
				else
				{
					wlen++;
					if (dkim->dkim_blankline &&
					    (wlen > 1 || *p != '\r'))
					{
						dkim_flush_blanks(dkim);
						dkim->dkim_blankline = FALSE;
					}
				}
			}
			else
			{
				if (*p != '\r')
				{
					dkim_flush_blanks(dkim);
					dkim->dkim_blankline = FALSE;
				}
				wlen++;
			}

			dkim->dkim_lastchar = *p;
		}

		dkim_canonwrite(dkim, wrote, wlen);

		break;

	  case DKIM_CANON_NOWSP:
		for (p = buf; p <= eob; p++)
		{
			if (!dkim_iswsp(*p))
			{
				if (lidx == sizeof lbuf)
				{
					dkim_canonwrite(dkim, lbuf, lidx);
					lidx = 0;
				}

				lbuf[lidx] = *p;
				lidx++;
			}

			dkim->dkim_lastchar = *p;
		}

		if (lidx > 0)
			dkim_canonwrite(dkim, lbuf, lidx);

		break;

	  default:
		assert(0);
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_EOM -- declare end-of-body; conduct verification or prepare to sign
**
**  Parameters:
**  	dkim -- DKIM handle
**  	testkey -- TRUE iff the a matching key was found but is marked as a
**  	           test key (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
**
**  XXX -- consider removing "testkey", and having the caller use
**         dkim_policy() retrieve flags like that one instead
*/

DKIM_STAT
dkim_eom(DKIM *dkim, bool *testkey)
{
	DKIM_STAT ret;

	assert(dkim != NULL);

	if (dkim->dkim_state >= DKIM_STATE_EOM)
		return DKIM_STAT_INVALID;
	dkim->dkim_state = DKIM_STATE_EOM;

	if (dkim->dkim_mode == DKIM_MODE_SIGN)
	{
		size_t n;
		DKIM_STAT status;
		u_char tmp[MAXHEADER + 1];
		struct dkim_header hdr;

		(void) time(&dkim->dkim_timestamp);

		n = sm_strlcpy(tmp, DKIM_SIGNHEADER ": ", sizeof tmp);

		dkim_canonfinal(dkim, TRUE);
		dkim->dkim_bodydone = TRUE;

		status = dkim_getsighdr(dkim, tmp + n, sizeof tmp - n,
		                        DKIM_HDRMARGIN,
		                        strlen(DKIM_SIGNHEADER) + 2);
		if (status != DKIM_STAT_OK)
			return status;

		if (strlen(tmp) == MAXHEADER)
			return DKIM_STAT_NORESOURCE;

		hdr.hdr_text = tmp;
		hdr.hdr_flags = 0;
		hdr.hdr_next = NULL;

		/* canonicalize */
		if (dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
			dkim_canonwrite(dkim, CRLF, 2);
		dkim_canonheader(dkim, &hdr, FALSE);

		/* sign with l= if requested */
		if ((dkim->dkim_libhandle->dkiml_flags & DKIM_LIBFLAGS_SIGNLEN) != 0)
			dkim->dkim_partial = TRUE;
	}
	else
	{
		int n;
		u_char last = '\0';
		u_char in = '\0';
		u_char *p;
		u_char *start;
		struct dkim_header *hdr;
		struct dkim_header tmphdr;
		u_char tmp[MAXHEADER + 1];

		/*
		**  We need to copy the DKIM-Signature: header being verified,
		**  minus the contents of the "b=" part, and include it in the
		**  canonicalization.
		*/

		dkim_canonfinal(dkim, TRUE);
		dkim->dkim_bodydone = TRUE;

		hdr = dkim_get_header(dkim, DKIM_SIGNHEADER,
		                      dkim->dkim_signum);
		if (hdr == NULL)
			return DKIM_STAT_NOSIG;

		start = hdr->hdr_text;
		memset(tmp, '\0', sizeof tmp);
		n = 0;
		for (p = hdr->hdr_text; *p != '\0'; p++)
		{
			if (*p == ';')
				in = '\0';

			if (in == 'b')
			{
				last = *p;
				continue;
			}

			if (in == '\0' && *p == '=')
				in = last;

			tmp[n++] = *p;
			last = *p;

			if (n == MAXHEADER)
				return DKIM_STAT_NORESOURCE;
		}

		/* canonicalize */
		tmphdr.hdr_text = tmp;
		tmphdr.hdr_flags = 0;
		tmphdr.hdr_next = NULL;

		if (dkim->dkim_version < DKIM_VERSION_IETF_BASE_01)
			dkim_canonwrite(dkim, CRLF, 2);
		if (dkim->dkim_hdrcanonalg == DKIM_CANON_RELAXED ||
		    dkim->dkim_hdrcanonalg == DKIM_CANON_NOWSP)
			dkim_lowerhdr(tmp);
		dkim_canonheader(dkim, &tmphdr, FALSE);
	}

	/* finalize canonicalization */
	dkim_canonfinal(dkim, FALSE);

	/* conduct verification, or compute signature */
	if (dkim->dkim_mode == DKIM_MODE_SIGN)
	{
		size_t diglen;
		size_t siglen;
		size_t l;
		int status;
		u_char *digest;
		u_char *signature;
		BIO *key;

		key = BIO_new_mem_buf(dkim->dkim_key, dkim->dkim_keylen);
		if (key == NULL)
			return DKIM_STAT_NORESOURCE;

		/* get the digest */
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

			diglen = SHA_DIGEST_LENGTH;
			digest = sha1->sha1_out;

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

			diglen = SHA256_DIGEST_LENGTH;
			digest = sha256->sha256_out;

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */

		  default:
			assert(0);
		}

		/* compute and store the signature */
		switch (dkim->dkim_signalg)
		{
		  case DKIM_SIGN_RSASHA1:
#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_SIGN_RSASHA256:
#endif /* SHA256_DIGEST_LENGTH */
		  {
			int nid;
			struct dkim_rsa *rsa;

#ifdef SHA256_DIGEST_LENGTH
			assert(dkim->dkim_hashtype == DKIM_HASHTYPE_SHA1 ||
			       dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256);
#else /* SHA256_DIGEST_LENGTH */
			assert(dkim->dkim_hashtype == DKIM_HASHTYPE_SHA1);
#endif /* SHA256_DIGEST_LENGTH */

			rsa = DKIM_MALLOC(dkim, sizeof(struct dkim_rsa));
			if (rsa == NULL)
				return DKIM_STAT_NORESOURCE;
			memset(rsa, '\0', sizeof(struct dkim_rsa));

			dkim->dkim_signature = (void *) rsa;
			dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;

			rsa->rsa_pkey = PEM_read_bio_PrivateKey(key, NULL,
			                                        NULL, NULL);

			if (rsa->rsa_pkey == NULL)
			{
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			rsa->rsa_rsa = EVP_PKEY_get1_RSA(rsa->rsa_pkey);
			if (rsa->rsa_rsa == NULL)
			{
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			rsa->rsa_keysize = RSA_size(rsa->rsa_rsa);
			rsa->rsa_pad = RSA_PKCS1_PADDING;
			rsa->rsa_rsaout = DKIM_MALLOC(dkim, rsa->rsa_keysize);
			if (rsa->rsa_rsaout == NULL)
			{
				BIO_free(key);
				return DKIM_STAT_NORESOURCE;
			}

			nid = NID_sha1;

#ifdef SHA256_DIGEST_LENGTH
			if (dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256)
				nid = NID_sha256;
#endif /* SHA256_DIGEST_LENGTH */

			status = RSA_sign(nid, digest, diglen,
		                          rsa->rsa_rsaout, &l, rsa->rsa_rsa);
			if (status == 0 || l == 0)
			{
				BIO_free(key);
				return DKIM_STAT_INTERNAL;
			}

			rsa->rsa_rsaoutlen = l;

			signature = rsa->rsa_rsaout;
			siglen = rsa->rsa_rsaoutlen;

			break;
		  }

		  default:
			assert(0);
		}

		/* base64-encode the signature */
		dkim->dkim_b64siglen = siglen * 3 + 5;
		dkim->dkim_b64siglen += (dkim->dkim_b64siglen / 60);
		dkim->dkim_b64sig = DKIM_MALLOC(dkim, dkim->dkim_b64siglen);
		if (dkim->dkim_b64sig == NULL)
		{
			BIO_free(key);
			return DKIM_STAT_NORESOURCE;
		}
		memset(dkim->dkim_b64sig, '\0', dkim->dkim_b64siglen);

		status = dkim_base64_encode(signature, siglen,
		                            dkim->dkim_b64sig,
		                            dkim->dkim_b64siglen);

		BIO_free(key);

		if (status == -1)
		{
			BIO_free(key);
			return DKIM_STAT_NORESOURCE;
		}
	}
	else						/* verifying */
	{
		bool msgsigned = FALSE;
		bool verified = FALSE;
		dkim_policy_t policy;
		bool test = FALSE;
		u_char *domain;
		struct dkim_set *sigset;
		struct dkim_header *sender;

		sigset = dkim->dkim_sigset;

		domain = dkim_param_get(sigset, "d");
		if (domain == NULL)			/* unsigned */
		{
			int status;
			u_char *user;
			u_char *domain;
			u_char *colon;

			sender = dkim_getsender(dkim,
			                        dkim->dkim_libhandle->dkiml_senderhdrs,
			                        FALSE);
			if (sender == NULL)
				return DKIM_STAT_SYNTAX;
			dkim->dkim_senderhdr = sender;

			colon = strchr(sender->hdr_text, ':');
			if (colon == NULL)
				return DKIM_STAT_SYNTAX;

			dkim->dkim_sender = dkim_strdup(dkim, colon + 1, 0);
			if (dkim->dkim_sender == NULL)
				return DKIM_STAT_NORESOURCE;

			/* determine domain */
			status = rfc2822_mailbox_split(dkim->dkim_sender,
			                               (char **) &user,
			                               (char **) &domain);
			if (status != 0)
				return DKIM_STAT_SYNTAX;

			if (dkim->dkim_domain == NULL)
			{
				dkim->dkim_domain = dkim_strdup(dkim,
				                                domain, 0);
				if (dkim->dkim_domain == NULL)
					return DKIM_STAT_NORESOURCE;
			}
			dkim->dkim_user = dkim_strdup(dkim, user, 0);
			if (dkim->dkim_user == NULL)
				return DKIM_STAT_NORESOURCE;
		}
		else					/* signed */
		{
			int status;
			u_int keyalg;
			size_t bhashlen = 0;
			size_t hashlen = 0;
			BIO *key = NULL;
			u_char *keytype;
			u_char *hashtypes;
			u_char *keygran;
			u_char *test;
			u_char *bhash;
			void *hashout = NULL;
			void *bhashout = NULL;
			struct dkim_set *keyset;

			msgsigned = TRUE;

			/* get key, using selector */
			status = dkim_get_key(dkim);
			if (status != DKIM_STAT_OK)
				return status;

			sender = dkim_getsender(dkim,
			                        dkim->dkim_libhandle->dkiml_senderhdrs,
			                        TRUE);
			if (sender == NULL)
				return DKIM_STAT_SYNTAX;
			dkim->dkim_senderhdr = sender;

			/* confirm the key and signature match */
			for (keyset = dkim_set_first(dkim, DKIM_SETTYPE_KEY);
			     keyset != NULL;
			     keyset = dkim_set_next(keyset, DKIM_SETTYPE_KEY))
			{
				if (dkim_key_smtp(keyset))
					break;
			}
			if (keyset == NULL)
				return DKIM_STAT_INTERNAL;

			/* key type is OK? */
			keytype = dkim_param_get(keyset, "k");
			if (keytype == NULL)
				return DKIM_STAT_CANTVRFY;
			keyalg = dkim_name_to_code(keytypes, keytype);
			if (keyalg == -1)
				return DKIM_STAT_CANTVRFY;
#ifdef SHA256_DIGEST_LENGTH
			if (keyalg == DKIM_KEYTYPE_RSA &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA1 &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA256)
#else /* SHA256_DIGEST_LENGTH */
			if (keyalg == DKIM_KEYTYPE_RSA &&
			    dkim->dkim_signalg != DKIM_SIGN_RSASHA1)
#endif /* SHA256_DIGEST_LENGTH */
				return DKIM_STAT_CANTVRFY;

			/* hash type is OK? */
			hashtypes = dkim_param_get(keyset, "h");
			if (!dkim_key_hashok(dkim, hashtypes))
				return DKIM_STAT_CANTVRFY;

			/* key granularity is OK? */
			keygran = dkim_param_get(keyset, "g");
			if (keygran != NULL && !dkim_key_granok(dkim, keygran))
				return DKIM_STAT_CANTVRFY;

			test = dkim_param_get(keyset, "t");
			if (testkey != NULL)
			{
				if (test != NULL && test[0] == 'y')
					*testkey = TRUE;
				else
					*testkey = FALSE;
			}

			/* load the public key */
			key = BIO_new_mem_buf(dkim->dkim_key,
			                      dkim->dkim_keylen);
			if (key == NULL)
				return DKIM_STAT_NORESOURCE;

			switch (dkim->dkim_signalg)
			{
			  case DKIM_SIGN_RSASHA1:
				dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;
				dkim->dkim_hashtype = DKIM_HASHTYPE_SHA1;
				break;

#ifdef SHA256_DIGEST_LENGTH
			  case DKIM_SIGN_RSASHA256:
				dkim->dkim_sigtype = DKIM_SIGTYPE_RSA;
				dkim->dkim_hashtype = DKIM_HASHTYPE_SHA256;
				break;
#endif /* SHA256_DIGEST_LENGTH */

			  default:
				assert(0);
			}

			switch (dkim->dkim_hashtype)
			{
			  case DKIM_HASHTYPE_SHA1:
			  {
				struct dkim_sha1 *sha1;

				sha1 = (struct dkim_sha1 *) dkim->dkim_hash;

				hashout = sha1->sha1_out;
				hashlen = SHA_DIGEST_LENGTH;

				if (dkim->dkim_bhash != NULL)
				{
					sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;

					bhashout = sha1->sha1_out;
					bhashlen = SHA_DIGEST_LENGTH;
				}

				break;
			  }

#ifdef SHA256_DIGEST_LENGTH
			  case DKIM_HASHTYPE_SHA256:
			  {
				struct dkim_sha256 *sha256;

				sha256 = (struct dkim_sha256 *) dkim->dkim_hash;

				hashout = sha256->sha256_out;
				hashlen = SHA256_DIGEST_LENGTH;

				if (dkim->dkim_bhash != NULL)
				{
					sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;

					bhashout = sha256->sha256_out;
					bhashlen = SHA256_DIGEST_LENGTH;
				}

				break;
			  }
#endif /* SHA256_DIGEST_LENGTH */

			  default:
				assert(0);
			}

			/* body hash, if present, matches this body? */
			bhash = dkim_param_get(sigset, "bh");
			if (bhash != NULL && bhashout != NULL)
			{
				char b64hash[MAXHEADER + 1];

				memset(b64hash, '\0', sizeof b64hash);
				status = dkim_base64_encode(bhashout, bhashlen,
				                            b64hash,
				                            sizeof b64hash);

				if (strcmp(b64hash, bhash) != 0)
					return DKIM_STAT_BADSIG;
			}

			switch (dkim->dkim_sigtype)
			{
			  case DKIM_SIGTYPE_RSA:
			  {
				int nid;
				struct dkim_rsa *rsa;

				rsa = DKIM_MALLOC(dkim,
				                  sizeof(struct dkim_rsa));
				if (rsa == NULL)
					return DKIM_STAT_NORESOURCE;
				memset(rsa, '\0', sizeof(struct dkim_rsa));

				dkim->dkim_signature = rsa;

				rsa->rsa_pkey = d2i_PUBKEY_bio(key, NULL);
				if (rsa->rsa_pkey == NULL)
				{
					BIO_free(key);
					return DKIM_STAT_NORESOURCE;
				}

				/* set up the RSA object */
				rsa->rsa_rsa = EVP_PKEY_get1_RSA(rsa->rsa_pkey);
				if (rsa->rsa_rsa == NULL)
				{
					BIO_free(key);
					return DKIM_STAT_NORESOURCE;
				}

				rsa->rsa_keysize = RSA_size(rsa->rsa_rsa);
				rsa->rsa_pad = RSA_PKCS1_PADDING;

				rsa->rsa_rsain = dkim->dkim_sig;
				rsa->rsa_rsainlen = dkim->dkim_siglen;

				dkim->dkim_keybits = 8 * rsa->rsa_keysize;

				verified = TRUE;

				nid = NID_sha1;

#ifdef SHA256_DIGEST_LENGTH
				if (dkim->dkim_hashtype == DKIM_HASHTYPE_SHA256)
					nid = NID_sha256;
#endif /* SHA256_DIGEST_LENGTH */

				status = RSA_verify(nid, hashout, hashlen,
				                    rsa->rsa_rsain,
				                    rsa->rsa_rsainlen,
				                    rsa->rsa_rsa);
				if (status == 0)
					verified = FALSE;

				break;
			  }

			  default:
				assert(0);
			}

			BIO_free(key);
		}

		/* retrieve policy */
		if (dkim_get_policy(dkim) != 0)
			return DKIM_STAT_CANTVRFY;

		/*
		**  Determine final result.
		*/

		ret = DKIM_STAT_OK;
		policy = dkim_policy(dkim, &test);
		if (testkey != NULL)
			*testkey = test;

		/*
		**  If the "s" policy flag is set and the signature claims
		**  the "i=" domain is a subdomain of the "d=" domain,
		**  the signature is not acceptable.
		*/

		if (dkim_policy_flag(dkim, 's') && dkim->dkim_subdomain)
			return DKIM_STAT_CANTVRFY;

		/* react to the published policy */
		switch (policy)
		{
#ifdef _FFR_ALLMAN_SSP_02

		  case DKIM_POLICY_UNKNOWN:
			break;

		  case DKIM_POLICY_ALL:
			if (!msgsigned)
				ret = DKIM_STAT_NOSIG;
			break;

		  case DKIM_POLICY_STRICT:
			if (!verified)
				ret = DKIM_STAT_BADSIG;
			break;

#else /* _FFR_ALLMAN_SSP_02 */

		  case DKIM_POLICY_SIGNSOME:
			break;

		  case DKIM_POLICY_REQUIRED:
			if (!msgsigned)
				ret = DKIM_STAT_NOSIG;
			break;

		  case DKIM_POLICY_AUTHORITY:
			/* XXX -- verify signer domain and sender domain match */
			break;

		  case DKIM_POLICY_NEVER:
			if (msgsigned)
				ret = DKIM_STAT_BADSIG;
			break;

		  case DKIM_POLICY_USER:
			ret = DKIM_STAT_NOTIMPLEMENT;
			break;

#endif /* _FFR_ALLMAN_SSP_02 */

		  default:
			ret = DKIM_STAT_SYNTAX;
			break;
		}

		if (msgsigned && !verified)
			ret = DKIM_STAT_BADSIG;

		return ret;
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGHDR -- for signing operations, retrieve the complete signature
**                    header
**
**  Parameters:
**  	dkim -- DKIM handle
**  	buf -- buffer for the signature
**  	buflen -- bytes available at "buf"
**  	margin -- attempt to cut at specified line width; 0 == don't bother
**  	initial -- initial line width
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsighdr(DKIM *dkim, u_char *buf, size_t buflen, size_t margin,
               size_t initial)
{
	size_t len;
	char *ctx;
	char *pv;
	u_char hdr[MAXHEADER + 1];

	assert(dkim != NULL);
	assert(buf != NULL);
	assert(buflen != 0);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_SIGN)
		return DKIM_STAT_INVALID;

#define	DELIMITER	"\001"

	/* compute and extract the signature header */
	memset(hdr, '\0', sizeof hdr);
	len = dkim_gensighdr(dkim, hdr, sizeof hdr, DELIMITER);
	if (len == MAXHEADER)
		return DKIM_STAT_NORESOURCE;

	if (dkim->dkim_b64sig != NULL)
	{
		len = sm_strlcat(hdr, dkim->dkim_b64sig, sizeof hdr);
		if (len >= sizeof hdr || len >= buflen)
			return DKIM_STAT_NORESOURCE;
	}

	if (margin == 0)
	{
		bool first = TRUE;

		for (pv = strtok_r(hdr, DELIMITER, &ctx);
		     pv != NULL;
		     pv = strtok_r(NULL, DELIMITER, &ctx))
		{
			if (!first)
				sm_strlcat(buf, " ", buflen);

			sm_strlcat(buf, pv, buflen);

			first = FALSE;
		}
	}
	else
	{
		bool first = TRUE;
		char *p;
		char *q;
		char *end;
		size_t len;
		char which[MAXTAGNAME + 1];

		len = initial;
		end = which + MAXTAGNAME;

		for (pv = strtok_r(hdr, DELIMITER, &ctx);
		     pv != NULL;
		     pv = strtok_r(NULL, DELIMITER, &ctx))
		{
			for (p = pv, q = which; *p != '=' && q <= end; p++, q++)
			{
				*q = *p;
				*(q + 1) = '\0';
			}

			if (len == 0 || first)
			{
				sm_strlcat(buf, pv, buflen);
				len += strlen(pv);
				first = FALSE;
			}
			else if (len + strlen(pv) > margin)
			{
				bool first = TRUE;

				if (strcmp(which, "h") == 0)
				{			/* break at colons */
					char *hdr;
					char *ctx2;

					sm_strlcat(buf, " ", buflen);
					len += 1;

					for (hdr = strtok_r(pv, ":", &ctx2);
					     hdr != NULL;
					     hdr = strtok_r(NULL, ":", &ctx2))
					{
						if (first)
						{
							sm_strlcat(buf, pv,
							           buflen);
							len += strlen(pv);
							first = FALSE;
						}
						else if (len + strlen(hdr) + 1 > margin)
						{
							sm_strlcat(buf, ":",
							           buflen);
							len += 1;
							sm_strlcat(buf,
							           "\r\n\t ",
							           buflen);
							len = 9;
							sm_strlcat(buf, hdr,
							           buflen);
							len += strlen(hdr);
						}
						else
						{
							sm_strlcat(buf, ":",
							           buflen);
							len += 1;
							sm_strlcat(buf, hdr,
							           buflen);
							len += strlen(hdr);
						}
					}

				}
				else if (strcmp(which, "b") == 0 ||
				         strcmp(which, "bh") == 0)
				{			/* break at margins */
					bool more;
					int idx;
					char *x;

					sm_strlcat(buf, " b=", buflen);
					len += 2;

					idx = strlen(buf);

					for (x = pv + 2;
					     *x != '\0' && idx < buflen - 2;
					     x++)
					{
						more = (*(x + 1) != '\0');

						buf[idx] = *x;
						idx++;
						len++;

						if (len >= margin && more)
						{
							sm_strlcat(buf,
							           "\r\n\t",
							           buflen);
							len = 8;
							idx += 3;
						}
					}
				}
				else
				{			/* break at delimiter */
					sm_strlcat(buf, "\r\n\t", buflen);
					len = 8;
					sm_strlcat(buf, pv, buflen);
					len += strlen(pv);
				}
			}
			else
			{
				if (!first)
					sm_strlcat(buf, " ", buflen);
				first = FALSE;
				len += 1;
				sm_strlcat(buf, pv, buflen);
				len += strlen(pv);
			}
		}
	}

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETIDENTITY -- get the identity on a signature; can be used for
**  	                signing or verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	hname -- where to write the name of the header or property which
**   	         contained the identity
**  	hnamelen -- number of bytes available at "hname"
**  	vname -- where to write the value of the header or property which
**   	         contained the identity
**  	vnamelen -- number of bytes available at "vname"
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getidentity(DKIM *dkim, u_char *hname, size_t hnamelen,
                 u_char *hval, size_t hvallen)
{
	int status;
	size_t clen;
	u_char *colon;

	assert(dkim != NULL);

	if (dkim->dkim_state < DKIM_STATE_EOH)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_senderhdr == NULL)
		return DKIM_STAT_SYNTAX;

	colon = strchr(dkim->dkim_senderhdr->hdr_text, ':');

	if (dkim->dkim_sender == NULL)
	{
		u_char *user;
		u_char *domain;

		dkim->dkim_sender = dkim_strdup(dkim, colon + 1, 0);
		if (dkim->dkim_sender == NULL)
			return DKIM_STAT_NORESOURCE;

		/* determine domain */
		status = rfc2822_mailbox_split(dkim->dkim_sender,
		                               (char **) &user,
		                               (char **) &domain);
		if (status != 0)
			return DKIM_STAT_SYNTAX;

		if (dkim->dkim_domain == NULL)
		{
			dkim->dkim_domain = dkim_strdup(dkim, domain, 0);
			if (dkim->dkim_domain == NULL)
				return DKIM_STAT_NORESOURCE;
		}
		dkim->dkim_user = dkim_strdup(dkim, user, 0);
		if (dkim->dkim_user == NULL)
			return DKIM_STAT_NORESOURCE;
	}

	clen = colon - dkim->dkim_senderhdr->hdr_text;
	sm_strlcpy(hname, dkim->dkim_senderhdr->hdr_text, hnamelen);
	hname[clen] = '\0';

	snprintf(hval, hvallen, "%s@%s", dkim->dkim_user, dkim->dkim_domain);

	return DKIM_STAT_OK;
}

/*
**  DKIM_REPORTINFO -- return the address to which failure reports are sent
**
**  Parameters:
**  	dkim -- DKIM handle
**  	fd -- descriptor for canonicalized message (returned)
**  	bfd -- descriptor for canonicalized body (returned)
**  	buf -- buffer for the address
**  	buflen -- bytes available at "buf"
**
**  Return value:
**  	A DKIM_STAT_* constant.
**
**  Notes:
**  	DKIM_STAT_CANTVRFY returned by this function means no policy record
**  	was found or "r" was not part of the policy record found.
**
**  	If the IETF "base-01" draft is being applied, the headers and
**  	body are hashed separately.  Thus, there are two descriptors of
**  	interest.  In that case, the body comes back in "bfd".  Otherwise,
**  	"bfd" will be unaltered.
*/

DKIM_STAT
dkim_reportinfo(DKIM *dkim, int *fd, int *bfd, u_char *buf, size_t buflen)
{
	DKIM_SET *set;

	assert(dkim != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_hash != NULL)
	{
		switch (dkim->dkim_hashtype)
		{
		  case DKIM_HASHTYPE_SHA1:
		  {
			struct dkim_sha1 *sha1;

			sha1 = (struct dkim_sha1 *) dkim->dkim_hash;
			if (fd != NULL)
				*fd = sha1->sha1_tmpfd;

			if (bfd != NULL &&
			    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			{
				sha1 = (struct dkim_sha1 *) dkim->dkim_bhash;
				*bfd = sha1->sha1_tmpfd;
			}

			break;
		  }

#ifdef SHA256_DIGEST_LENGTH
		  case DKIM_HASHTYPE_SHA256:
		  {
			struct dkim_sha256 *sha256;

			sha256 = (struct dkim_sha256 *) dkim->dkim_hash;
			if (fd != NULL)
				*fd = sha256->sha256_tmpfd;

			if (bfd != NULL &&
			    dkim->dkim_version >= DKIM_VERSION_IETF_BASE_01)
			{
				sha256 = (struct dkim_sha256 *) dkim->dkim_bhash;
				*bfd = sha256->sha256_tmpfd;
			}

			break;
		  }
#endif /* SHA256_DIGEST_LENGTH */
		}
	}

	/* extract the information */
	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
		return DKIM_STAT_CANTVRFY;

	dkim->dkim_reportaddr = dkim_param_get(set, "r");
	if (dkim->dkim_reportaddr == NULL)
		return DKIM_STAT_CANTVRFY;

	memset(buf, '\0', buflen);

	sm_strlcpy(buf, dkim->dkim_reportaddr, buflen);

	return DKIM_STAT_OK;
}

/*
**  DKIM_POLICY -- parse policy associated with the sender's domain
**
**  Parameters:
**  	dkim -- DKIM handle
**  	test -- test flag (returned)
**
**  Return value:
**  	A DKIM_POLICY_* value, or -1 on error.
*/

dkim_policy_t
dkim_policy(DKIM *dkim, bool *test)
{
	struct dkim_set *set;
#ifdef _FFR_ALLMAN_SSP_02
	struct dkim_set *uset;
#endif /* _FFR_ALLMAN_SSP_02 */
	u_char *param;

	assert(dkim != NULL);
	assert(test != NULL);

	set = dkim_set_first(dkim, DKIM_SETTYPE_POLICY);
	if (set == NULL)
		return -1;

#ifdef _FFR_ALLMAN_SSP_02
	uset = dkim_set_next(set, DKIM_SETTYPE_POLICY);
	if (uset != NULL)
		set = uset;
#endif /* _FFR_ALLMAN_SSP_02 */

	param = dkim_param_get(set, "t");
	if (param != NULL)
	{
		if (strchr(param, 'y') != NULL)
			*test = TRUE;
		else
			*test = FALSE;
	}
	else
	{
		*test = FALSE;
	}

#ifdef _FFR_ALLMAN_SSP_02
	param = dkim_param_get(set, "p");
	if (param == NULL)
		return -1;

	return (dkim_policy_t) dkim_name_to_code(policies, param);
#else /* _FFR_ALLMAN_SSP_02 */
	param = dkim_param_get(set, "o");
	if (param == NULL)
		return -1;

	return (dkim_policy_t) dkim_char_to_code(policies, param[0]);
#endif /* _FFR_ALLMAN_SSP_02 */
}

/*
**  DKIM_GETCANONLEN -- return canonicalized and total body lengths
**
**  Parameters:
**  	dkim -- DKIM handle
**  	msglen -- total body length (returned)
**  	canonlen -- total canonicalized length (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getcanonlen(DKIM *dkim, off_t *msglen, off_t *canonlen)
{
	assert(dkim != NULL);

	if (msglen != NULL)
		*msglen = dkim->dkim_bodylen;

	if (canonlen != NULL)
		*canonlen = dkim->dkim_canonlen;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETKEYSIZE -- retrieve key size (in bits) when verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	bits -- number of bits in the key (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getkeysize(DKIM *dkim, unsigned int *bits)
{
	assert(dkim != NULL);
	assert(bits != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	*bits = dkim->dkim_keybits;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGNALG -- retrieve signature algorithm when verifying
**
**  Parameters:
**  	dkim -- DKIM handle
**  	alg -- signature algorithm used (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsignalg(DKIM *dkim, dkim_alg_t *alg)
{
	assert(dkim != NULL);
	assert(alg != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	*alg = dkim->dkim_signalg;

	return DKIM_STAT_OK;
}

/*
**  DKIM_GETSIGNTIME -- retrieve signature timestamp
**
**  Parameters:
**  	dkim -- DKIM handle
**  	when -- signature timestamp (returned)
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

DKIM_STAT
dkim_getsigntime(DKIM *dkim, time_t *when)
{
	time_t tmp;
	char *end;
	char *timestr;

	assert(dkim != NULL);
	assert(when != NULL);

	if (dkim->dkim_state != DKIM_STATE_EOM ||
	    dkim->dkim_mode != DKIM_MODE_VERIFY)
		return DKIM_STAT_INVALID;

	if (dkim->dkim_sigset == NULL)
		return DKIM_STAT_INTERNAL;

	timestr = dkim_param_get(dkim->dkim_sigset, "t");
	if (timestr == NULL)
		return DKIM_STAT_SYNTAX;

	tmp = strtoul(timestr, &end, 10);
	if (end != '\0')
		return DKIM_STAT_SYNTAX;

	*when = tmp;

	return DKIM_STAT_OK;
}

#if _FFR_SET_DNS_CALLBACK
/*
**  DKIM_SET_DNS_CALLBACK -- set the DNS wait callback
**
**  Parameters:
**  	dkim -- DKIM handle
**  	func -- function to call; should take an opaque context pointer
**  	ctx -- opaque context handle
**  	interval -- how often to call back
**
**  Return value:
**  	None.
*/

DKIM_STAT
dkim_set_dns_callback(DKIM *dkim, void (*func)(void *context), void *ctx,
                      unsigned int interval)
{
	assert(dkim != NULL);

# if USE_ARLIB
	if (func != NULL && interval == 0)
		return DKIM_STAT_INVALID;

	dkim->dkim_dns_callback = func;
	dkim->dkim_user_context = ctx;
	dkim->dkim_callback_int = interval;

	return DKIM_STAT_OK;
# else /* USE_ARLIB */
	return DKIM_STAT_INVALID;
# endif /* USE_ARLIB */
}
#endif /* _FFR_SET_DNS_CALLBACK */
