/*
 * File: cookies.c
 *
 * Copyright 2001 Lars Clausen   <lrclause@cs.uiuc.edu>
 *                Jrgen Viksell <jorgen.viksell@telia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/* Handling of cookies takes place here.
 * This implementation aims to follow RFC 2965:
 * http://www.cis.ohio-state.edu/cs/Services/rfc/rfc-text/rfc2965.txt
 */

#ifndef DISABLE_COOKIES

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>       /* for time() and time_t */
#include <ctype.h>

#include "IO/Url.h"
#include "misc.h"
#include "list.h"
#include "cookies.h"

#define DEBUG_LEVEL 8
#include "debug.h"

/* The maximum length of a line in the cookie file */
#define LINE_MAXLEN 4096

typedef enum {
   COOKIE_ACCEPT,
   COOKIE_ACCEPT_SESSION,
   COOKIE_DENY
} CookieControlAction;

typedef struct {
   CookieControlAction action;
   char *domain;
} CookieControl;

typedef struct {
   char *name;
   char *value;
   char *domain;
   char *path;
   time_t expires_at;
   guint version;
   char *comment;
   char *comment_url;
   gboolean secure;
   gboolean session_only;
   GList *ports;
} CookieData_t;

/* Hashtable indexed by domain, each value is a set of cookies for
 * that domain. */
static GHashTable *cookies;

/* Variables for access control */
static CookieControl *ccontrol = NULL;
static int num_ccontrol = 0;
static int num_ccontrol_max = 1;
static CookieControlAction default_action = COOKIE_DENY;

static gboolean disabled;
static FILE *file_stream;

static FILE *Cookies_fopen(const char *file, gchar *init_str);
static CookieControlAction Cookies_control_check(const DilloUrl *url);
static CookieControlAction Cookies_control_check_domain(const char *domain);
static void Cookie_control_init();
static void Cookies_parse_ports(CookieData_t *cookie, const char *port_str);
static char *Cookies_build_ports_str(CookieData_t *cookie);

/*
 * Return a file pointer. If the file doesn't exist, try to create it,
 * with the optional 'init_str' as its content.
 * This function is either successful or exits Dillo.
 */
static FILE *Cookies_fopen(const char *filename, gchar *init_str)
{
   FILE *F_in;
   int fd;

   if ((F_in = fopen(filename, "r+")) == NULL) {
      /* Create the file */
      fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
      if (fd != -1) {
         if (init_str)
            write(fd, init_str, strlen(init_str));
         close(fd);

         DEBUG_MSG(10, "Cookies: Created file: %s\n", filename);
         F_in = Cookies_fopen(filename, NULL);
      } else {
         DEBUG_MSG(10, "Cookies: Could not create file: %s!\n", filename);
      }
   }

   return F_in;
}

static void Cookies_free_cookie(CookieData_t *cookie)
{
   if (cookie) {
      g_free(cookie->name);
      g_free(cookie->value);
      g_free(cookie->domain);
      g_free(cookie->path);
      g_free(cookie->comment);
      g_free(cookie->comment_url);
      g_list_free(cookie->ports);
      g_free(cookie);
   }
}

void a_Cookies_init()
{
   CookieData_t *cookie;
   GList *domain_cookies;
   char *filename;
   char line[LINE_MAXLEN];

   Cookie_control_init();

   /* Get a stream for the cookies file */
   filename = a_Misc_prepend_user_home(".dillo/cookies");
   file_stream = Cookies_fopen(filename, NULL);
   g_free(filename);

   if (!file_stream) {
      /* Can't open the file, so we disable cookies */
      disabled = TRUE;
      return;
   }

   /* Try to get a lock from the file descriptor */
   disabled = (lockf(fileno(file_stream), F_TLOCK, 0) == -1);
   if (disabled) {
      DEBUG_MSG(10, "The cookies file has a file lock:\n"
                    "  disabling cookies in this dillo!\n");
      fclose(file_stream);
      return;
   }

   cookies = g_hash_table_new(g_str_hash, g_str_equal);

   /* Get all lines in the file */
   while (!feof(file_stream)) {
      line[0] = '\0';
      fgets(line, LINE_MAXLEN, file_stream);

      /* Remove leading and trailing whitespaces */
      g_strstrip(line);

      if (line[0] != '\0') {
         /* Would use g_strsplit, but it doesn't give empty trailing pieces.
          */
         /* Split the row into pieces using a tab as the delimiter.
          * pieces[0] The version this cookie was set as (0 / 1)
          * pieces[1] The domain name
          * pieces[2] A comma separated list of accepted ports
          * pieces[3] The path
          * pieces[4] Is the cookie unsecure or secure (0 / 1)
          * pieces[5] Timestamp of expire date
          * pieces[6] Name of the cookie
          * pieces[7] Value of the cookie
          * pieces[8] Comment
          * pieces[9] Comment url
          */
         CookieControlAction action;
         char *piece;
         char *line_marker = line;

         cookie = g_new0(CookieData_t, 1);

         cookie->session_only = FALSE;
         piece = d_strsep(&line_marker, "\t");
         if (piece != NULL)
            cookie->version = strtol(piece, NULL, 10);
         cookie->domain  = g_strdup(d_strsep(&line_marker, "\t"));
         Cookies_parse_ports(cookie, d_strsep(&line_marker, "\t"));
         cookie->path = g_strdup(d_strsep(&line_marker, "\t"));
         piece = d_strsep(&line_marker, "\t");
         if (piece != NULL && piece[0] == '1')
            cookie->secure = TRUE;
         piece = d_strsep(&line_marker, "\t");
         if (piece != NULL)
            cookie->expires_at = (time_t) strtol(piece, NULL, 10);
         cookie->name = g_strdup(d_strsep(&line_marker, "\t"));
         cookie->value = g_strdup(d_strsep(&line_marker, "\t"));
         cookie->comment = g_strdup(d_strsep(&line_marker, "\t"));
         cookie->comment_url = g_strdup(d_strsep(&line_marker, "\t"));

         action = Cookies_control_check_domain(cookie->domain);
         if (action == COOKIE_DENY) {
            Cookies_free_cookie(cookie);
            continue;
         } else if (action == COOKIE_ACCEPT_SESSION) {
            cookie->session_only = TRUE;
         }

         /* Save cookie in memory */
         if (cookie->expires_at > time(NULL)) {
            domain_cookies = (GList *) g_hash_table_lookup(cookies,
                                                           cookie->domain);

            if (domain_cookies == NULL) {
               domain_cookies = g_list_append(domain_cookies, cookie);
               g_hash_table_insert(cookies, g_strdup(cookie->domain),
                                   domain_cookies);
            } else {
               domain_cookies = g_list_append(domain_cookies, cookie);
            }
         } else {
            /* Has expired */
            Cookies_free_cookie(cookie);
         }
      }
   }
}

/*
 * Save the cookies and remove them from the hash table
 */
static gboolean Cookies_freeall_cb(gpointer key,
                                   gpointer value,
                                   gpointer data)
{
   CookieData_t *cookie;
   GList *domain_cookies;
   char *ports_str;

   for (domain_cookies = g_list_first((GList *) value);
        domain_cookies != NULL;
        domain_cookies = g_list_next(domain_cookies)) {
      cookie = (CookieData_t *) domain_cookies->data;

      if (cookie != NULL) {
         if (!cookie->session_only) {
            ports_str = Cookies_build_ports_str(cookie);
            fprintf(file_stream, "%d\t%s\t%s\t%s\t%d\t%ld\t%s\t%s\t%s\t%s\n",
                    cookie->version,
                    cookie->domain,
                    ports_str,
                    cookie->path,
                    cookie->secure ? 1 : 0,
                    (long)cookie->expires_at,
                    cookie->name,
                    cookie->value,
                    (cookie->comment) ? cookie->comment : "",
                    (cookie->comment_url) ? cookie->comment_url : "");
            g_free(ports_str);
         }
         Cookies_free_cookie(cookie);
      }
   }
   g_list_free((GList *) value);
   g_free(key);

   /* Return TRUE to tell GLIB to free this key from the hash table */
   return TRUE;
}

/*
 * Flush cookies to disk and free all the memory allocated.
 */
void a_Cookies_freeall()
{
   int fd;

   if (disabled)
      return;

   rewind(file_stream);
   fd = fileno(file_stream);
   ftruncate(fd, 0);

   g_hash_table_foreach_remove(cookies, Cookies_freeall_cb, NULL);

   lockf(fd, F_ULOCK, 0);
   fclose(file_stream);
}

/*
 * ?
 */
static char *Cookies_try_header(char **string, char *header, gboolean compat)
{
   char *marker;
   char *result;
   char *value_start;

   if (*string == NULL)
      return NULL;

   if (g_strncasecmp(*string, header, strlen(header))) {
      return NULL;
   }
   value_start = *string + strlen(header);

   marker = strchr(value_start, '=');
   if (marker == NULL) {
      *string = NULL;
      return g_strdup(""); /* Some attributes don't use '=' */
   }
   marker++;
   g_strstrip(marker);

   value_start = marker;

   marker = strchr(value_start, ';');
   if (marker == NULL) {
      if (!compat) {
         marker = strchr(value_start, ',');
         if (marker == NULL) {
            result = g_strdup(value_start);
            *string = NULL;
            return result;
         }
      } else {
         /* Netscape Expires combatability -- commas in string (doh!) */
         result = g_strdup(value_start);
         *string = NULL;
         return result;
      }
   }
   result = g_strndup(value_start, marker - value_start);
   *string = marker;

   return result;
}

static char *months[] =
{ "",
  "Jan", "Feb", "Mar",
  "Apr", "May", "Jun",
  "Jul", "Aug", "Sep",
  "Oct", "Nov", "Dec"
};

/*
 * Take a months name and return a number between 1-12.
 * E.g. 'April' -> 4
 */
static int Cookies_get_month(const char *month_name)
{
   int i;

   for (i = 1; i <= 12; i++) {
      if (!g_strncasecmp(months[i], month_name, 3))
         return i;
   }
   return 0;
}

/*
 * Return a local timestamp from a GMT date string
 * Accept: RFC-1123 | RFC-850 | ANSI asctime
 * (return 0 on malformed date string syntax)
 */
static time_t Cookies_create_timestamp(const char *expires)
{
   gchar *E_msg = "Cookies: Expire date is malformed! Ignoring cookie...\n";
   time_t ret;
   int day, month, year, hour, minutes, seconds;
   gchar *cp;

   if (!(cp = strchr(expires, ',')) && strlen(expires) == 24) {
      /* Looks like ANSI asctime format... */
      cp = (gchar *)expires;
      day = strtol(cp + 8, NULL, 10);       /* day */
      month = Cookies_get_month(cp + 4);    /* month */
      year = strtol(cp + 20, NULL, 10);     /* year */
      hour = strtol(cp + 11, NULL, 10);     /* hour */
      minutes = strtol(cp + 14, NULL, 10);  /* minutes */
      seconds = strtol(cp + 17, NULL, 10);  /* seconds */

   } else if (cp && ((cp - expires == 3 && strlen(cp) == 26) ||
                     (cp - expires > 5  && strlen(cp) == 24))) {
      /* RFC-1123 | RFC-850 format */
      day = strtol(cp + 2, NULL, 10);
      month = Cookies_get_month(cp + 5);
      year = strtol(cp + 9, &cp, 10);
      /* todo: tricky, because two digits for year IS ambiguous! */
      year += (year < 70) ? 2000 : ((year < 100) ? 1900 : 0);
      hour = strtol(cp + 1, NULL, 10);
      minutes = strtol(cp + 4, NULL, 10);
      seconds = strtol(cp + 7, NULL, 10);

   } else {
      DEBUG_MSG(5, E_msg);
      return (time_t) 0;
   }

   /* Error checks  --this may be overkill */
   if (!(day > 0 && day < 32 && month > 0 && month && year > 1970 &&
         hour >= 0 && hour < 24 && minutes >= 0 && minutes < 60 &&
         seconds >= 0 && seconds < 60)) {
      DEBUG_MSG(5, E_msg);
      return (time_t) 0;
   }
   
   /* Calculate local timestamp.
    * [stolen from Lynx... (http://lynx.browser.org)] */
   month -= 3;
   if (month < 0) {
      month += 12;
      year--;
   }

   day += (year - 1968) * 1461 / 4;
   day += ((((month * 153) + 2) / 5) - 672);
   ret = (time_t)((day * 60 * 60 * 24) +
                  (hour * 60 * 60) +
                  (minutes * 60) +
                  seconds);

   DEBUG_MSG(4, "\tExpires in %ld seconds, at %s",
             (long)ret - time(NULL),
             ctime(&ret));

   return ret;
}

/*
 * Parse a string containing a list of port numbers.
 */
static void Cookies_parse_ports(CookieData_t *cookie, const char *port_str)
{
   if (port_str[0] == '\0') {
      /* There was no list, so only the calling urls port should
       * be allowed. We set the first one to -1 and continues in
       * a_Cookies_set */
      cookie->ports = g_list_append(cookie->ports, GINT_TO_POINTER(-1));
   } else if (port_str[0] == '"' && port_str[1] != '"') {
      char **tokens, **i;
      int port;

      tokens = g_strsplit(port_str + 1, ",", -1);
      for (i = tokens; *i; ++i) {
         port = strtol(*i, NULL, 10);
         if (port > 0) {
            cookie->ports = g_list_append(cookie->ports,
                                          GINT_TO_POINTER(port));
         }
      }
      g_strfreev(tokens);
   }
}

/*
 * Build a string of the ports in 'cookie'.
 */
static char *Cookies_build_ports_str(CookieData_t *cookie)
{
   GString *gstr;
   GList *list;
   char *ret;
   int port;

   gstr = g_string_new("\"");
   for (list = g_list_first(cookie->ports); list; list = g_list_next(list)) {
      port = GPOINTER_TO_INT(list->data);
      g_string_sprintfa(gstr, "%d,", port);
   }

   /* Remove any trailing comma */
   if (gstr->len > 1)
      g_string_erase(gstr, gstr->len - 1, 1);

   g_string_append(gstr, "\"");

   ret = gstr->str;
   g_string_free(gstr, FALSE);

   return ret;
}

/*
 * Parse optional fields
 * (domain, path, expires, max-age, comment,
 *  commenturl, secure, version, discard)
 */
static void Cookies_parse_optional(CookieData_t *cookie, char *cookie_string)
{
   char *value_string;
   gboolean discard, max_age;

   max_age = FALSE;
   discard = FALSE;

   while (cookie_string) {
      if (*cookie_string == ',' || *cookie_string == '\0')
         return;
      cookie_string++;
      g_strstrip(cookie_string);
      if (*cookie_string == '\0')
         return;

      value_string = Cookies_try_header(&cookie_string, "Path", FALSE);
      if (value_string) {
         cookie->path = value_string;
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "Domain", FALSE);
      if (value_string) {
         cookie->domain = value_string;
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "Discard", FALSE);
      if (value_string) {
         /* The server wants us to toss this cookie away at exit */
         cookie->session_only = TRUE;
         discard = TRUE;
         g_free(value_string);
         continue;
      }

      if (!discard) {
         value_string = Cookies_try_header(&cookie_string, "Max-Age", FALSE);
         if (value_string) {
            cookie->expires_at = time(NULL) + strtol(value_string, NULL, 10);
            cookie->session_only = FALSE;
            max_age = TRUE;
            g_free(value_string);
            continue;
         }

         if (!max_age) {
            value_string = Cookies_try_header(&cookie_string, "Expires", TRUE);
            if (value_string) {
               cookie->expires_at = Cookies_create_timestamp(value_string);
               cookie->session_only = FALSE;
               g_free(value_string);
               continue;
            }
         }
      }

      value_string = Cookies_try_header(&cookie_string, "Port", TRUE);
      if (value_string) {
         Cookies_parse_ports(cookie, value_string);
         g_free(value_string);
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "Comment", FALSE);
      if (value_string) {
         cookie->comment = value_string;
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "CommentURL", FALSE);
      if (value_string) {
         cookie->comment_url = value_string;
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "Version", FALSE);
      if (value_string) {
         cookie->version = strtol(value_string, NULL, 10);
         g_free(value_string);
         continue;
      }

      value_string = Cookies_try_header(&cookie_string, "Secure", FALSE);
      if (value_string) {
         cookie->secure = TRUE;
         DEBUG_MSG(4, "Secure cookie requested.\n");
         g_free(value_string);
      }
   }
}

/*
 * Remove the domain from the hash table and free its key.
 */
static void Cookies_free_domain(const char *domain)
{
   gpointer orig_key;

   if (g_hash_table_lookup_extended(cookies, domain, &orig_key, NULL)) {
      g_hash_table_remove(cookies, domain);
      g_free(orig_key);
   }
}

/*
 * Parse the supplied string and return a newly allocated cookie.
 */
static CookieData_t *Cookies_parse_string(char *cookie_string)
{
   CookieData_t *cookie = g_new0(CookieData_t, 1);
   char *marker;

   DEBUG_MSG(4, "Parsing cookie string `%s'\n", cookie_string);

   cookie->session_only = TRUE;

   /* Parse the name and value */
   marker = strchr(cookie_string, '=');
   if (marker == NULL) {
      /* Malformed cookie */
      Cookies_free_cookie(cookie);
      return NULL;
   }
   cookie->name = g_strndup(cookie_string, marker-cookie_string);
   marker++;
   if (*marker == '"') {
      /* Quoted string */
      char *value_start = ++marker;
      marker = strchr(value_start, '\"');
      if (marker == NULL) {
         Cookies_free_cookie(cookie);
         return NULL;
      }
      cookie->value = g_strndup(cookie_string, marker-value_start);
      marker++;
   } else {
      char *value_start = marker;
      marker = strchr(value_start, ';');
      if (marker == NULL) {
         /* No path etc, maybe ended with ',' */
         marker = strchr(value_start, ',');
         if (marker == NULL) {
            /* No more stuff, value is entire rest */
            marker = value_start + strlen(value_start);
         }
      }
      cookie->value = g_strndup(value_start, marker-value_start);
   }

   /* Parse optional fields
      (domain, path, expires, max-age, comment, secure, version) */
   Cookies_parse_optional(cookie, marker);

   return cookie;
}

/*
 * Compare cookies (return 1 if equal, 0 otherwise)
 */
static gboolean Cookies_equals(CookieData_t *cookie1,
                               CookieData_t *cookie2)
{
   return ((!strcmp(cookie1->name, cookie2->name)) &&
           (!(cookie1->domain && cookie2->domain &&
              strcmp(cookie1->domain, cookie2->domain))) &&
           (!(cookie1->path && cookie2->path &&
              strcmp(cookie1->path, cookie2->path))));
}

/*
 * Validate cookies domain against some security checks.
 */
static gboolean Cookies_validate_domain(CookieData_t *cookie,
                                        const DilloUrl *set_url)
{
   const char *host;
   int dots, diff, i;
   gboolean is_ip;

   /* If the server never set a domain, or set one without a leading
    * dot (which isn't allowed), we use the calling URL's hostname. */
   if (cookie->domain == NULL || cookie->domain[0] != '.') {
      cookie->domain = g_strdup(URL_HOST(set_url));
      return TRUE;
   }

   /* Count the number of dots and also find out if it is an IP-address */
   is_ip = TRUE;
   for (i = 0, dots = 0; cookie->domain[i] != '\0'; i++) {
      if (cookie->domain[i] == '.')
         dots++;
      else if (!isdigit(cookie->domain[i]))
         is_ip = FALSE;
   }

   /* A valid domain must have at least two dots in it */
   /* NOTE: this breaks cookies on localhost... */
   if (dots < 2) {
      return FALSE;
   }

   /* Now see if the url matches the domain */
   host = URL_HOST(set_url);
   diff = strlen(host) - i;
   if (diff > 0) {
      if (g_strcasecmp(host + diff, cookie->domain))
         return FALSE;

      if (!is_ip) {
         /* "x.y.test.com" is not allowed to set cookies for ".test.com";
          *  only an url of the form "y.test.com" would be. */
         while ( diff-- )
            if (host[diff] == '.')
               return FALSE;
      }
   }

   return TRUE;
}

/*
 * Strip of the filename from a full path
 */
static char *Cookies_strip_path(const char *path)
{
   char *ret;
   int len;

   if (path) {
      len = strlen(path);

      while(len && path[len] != '/')
         len--;
      ret = g_strndup(path, len + 1);
   } else {
      ret = g_strdup("/");
   }

   return ret;
}

/*
 * Set the value corresponding to the cookie string
 */
void a_Cookies_set(GList *cookie_strings, const DilloUrl *set_url)
{
   CookieControlAction action;

   if (disabled)
      return;

   action = Cookies_control_check(set_url);
   if (action == COOKIE_DENY) {
      DEBUG_MSG(5, "Cookies: denied SET for %s\n", URL_HOST_(set_url));
      return;
   }

   while (cookie_strings != NULL ) {
      char *cookie_string = (char *)cookie_strings->data;
      CookieData_t *cookie = Cookies_parse_string(cookie_string);
      GList *domain_cookies;

      if (cookie == NULL) {
         DEBUG_MSG(8, "Malformed cookie field %s\n", cookie_string);
         return;
      }

      if (action == COOKIE_ACCEPT_SESSION)
         cookie->session_only = TRUE;

      if (cookie->ports) {
         GList *first = g_list_first(cookie->ports);

         if (GPOINTER_TO_INT(first->data) == -1) {
            /* Now set it to the port we are accessing */
            cookie->ports = g_list_insert(cookie->ports,
                                          GINT_TO_POINTER(URL_PORT(set_url)),
                                          0);
         }
      }

      if (Cookies_validate_domain(cookie, set_url)) {
         if (cookie->path == NULL) {
            cookie->path = Cookies_strip_path(URL_PATH_(set_url));
         }
         domain_cookies = (GList *)g_hash_table_lookup(cookies,
                                                       cookie->domain);

         if (domain_cookies != NULL) {
            GList *list_temp = g_list_first(domain_cookies);
            CookieData_t *old_cookie;

            for (; list_temp != NULL; list_temp = g_list_next(list_temp)) {
               old_cookie = list_temp->data;

               if (Cookies_equals(cookie, old_cookie)) {
                  DEBUG_MSG(4, "Found old cookie %s, replacing with new %s\n",
                            old_cookie->name, cookie->name);
                  domain_cookies = g_list_remove(domain_cookies, old_cookie);

                  if (domain_cookies == NULL) {
                     Cookies_free_domain(old_cookie->domain);
                  } else {
                     g_hash_table_insert(cookies, old_cookie->domain,
                                         g_list_first(domain_cookies));
                  }

                  Cookies_free_cookie(old_cookie);
                  break;
               }
            }
         }
         /* If this is non-expiring, add it */
         if (g_list_length(domain_cookies) < 20 &&
             (cookie->session_only || cookie->expires_at > time(NULL))) {

            DEBUG_MSG(5, "Adding cookie '%s' for %s\n",
                      cookie->name, cookie->domain);
            domain_cookies = g_list_append(domain_cookies, cookie);

            if (!g_hash_table_lookup(cookies, cookie->domain))
               g_hash_table_insert(cookies, g_strdup(cookie->domain),
                                   domain_cookies);
         } else {
            DEBUG_MSG(5, "Removing cookie '%s'\n", cookie->name);
            Cookies_free_cookie(cookie);
         }
      } else {
         DEBUG_MSG(5, "Rejecting cookie for %s from %s:\n",
                   cookie->domain, URL_STR_(set_url));
         Cookies_free_cookie(cookie);
      }

      cookie_strings = g_list_next(cookie_strings);
   }
}

/*
 * Used by g_list_insert_sorted() to sort the cookies by most specific path
 */
static gint Cookies_compare(gconstpointer a, gconstpointer b)
{
   return strcmp(((CookieData_t *) b)->path, ((CookieData_t *) a)->path);
}

/*
 * Return a string that contains all relevant cookies as headers.
 */
char *a_Cookies_get(const DilloUrl *request_url)
{
   char *domain_string, *q, *str, *path;
   CookieControlAction action;
   CookieData_t *cookie;
   GList *matching_cookies = NULL;
   GList *domain_cookie;
   gboolean is_ssl;
   gboolean cookies_removed;
   GString *cookie_gstring;

   if (disabled)
      return g_strdup("");

   action = Cookies_control_check(request_url);
   if (action == COOKIE_DENY) {
      DEBUG_MSG(5, "Cookies: denied GET for %s\n", URL_HOST_(request_url));
      return g_strdup("");
   }

   path = Cookies_strip_path(URL_PATH_(request_url));

   /* Check if the protocol is secure or not */
   is_ssl = (!g_strcasecmp(URL_SCHEME(request_url), "https"));

   for (domain_string = (char *) URL_HOST(request_url);
        domain_string != NULL && *domain_string;
        domain_string = strchr(domain_string+1, '.')) {
      domain_cookie = g_hash_table_lookup(cookies, domain_string);
      cookies_removed = FALSE;

      for (domain_cookie = g_list_first(domain_cookie); domain_cookie != NULL;
           domain_cookie = g_list_next(domain_cookie)) {
         cookie = (CookieData_t*)domain_cookie->data;

         /* Insecure cookies matches both secure and insecure urls, secure
            cookies matches only secure urls */
         if (!cookie->secure || (cookie->secure == is_ssl)) {
            /* Check that the cookie path is a subpath of the current path */
            if (!(strncmp(cookie->path, path, strlen(cookie->path)))) {
               if (!cookie->session_only && cookie->expires_at < time(NULL)) {
                  /* Expired cookies die here.  RIP */
                  DEBUG_MSG(4, "Cookie %s expiring\n", cookie->name);
                  domain_cookie = g_list_remove(domain_cookie, cookie);
                  Cookies_free_cookie(cookie);
                  cookies_removed = TRUE;
               } else {
                  /* Check if the port of the request URL matches any
                   * of those set in the cookie */
                  if (cookie->ports) {
                     GList *list;
                     int port = URL_PORT(request_url);

                     for (list = g_list_first(cookie->ports); list;
                          list = g_list_next(list)) {
                        if (GPOINTER_TO_INT(list->data) == port) {
                           matching_cookies =
                              g_list_insert_sorted(matching_cookies,
                                                   cookie,
                                                   Cookies_compare);
                           break;
                        }
                     }
                  } else {
                     matching_cookies = g_list_insert_sorted(matching_cookies,
                                                             cookie,
                                                             Cookies_compare);
                  }
               }
            }
         }
      }
      if (cookies_removed) {
         if (domain_cookie == NULL) {
            Cookies_free_domain(domain_string);
         } else {
            g_hash_table_insert(cookies, domain_string,
                                g_list_first(domain_cookie));
         }
      }
   }

   /* Found the cookies, now make the string */
   cookie_gstring = g_string_new("");
   if (matching_cookies != NULL) {
      CookieData_t *first_cookie = (CookieData_t*)matching_cookies->data;

      g_string_sprintfa(cookie_gstring, "Cookie: ");

      if (first_cookie->version != 0) {
         g_string_sprintfa(cookie_gstring, "\"%d\"; ",
                           first_cookie->version);
      }

      for (; matching_cookies != NULL;
           matching_cookies = g_list_next(matching_cookies)) {
         cookie = (CookieData_t *) matching_cookies->data;
         q = (cookie->version == 0 ? "" : "\"");

         g_string_sprintfa(cookie_gstring, "%s=%s%s%s",
                           cookie->name, q, cookie->value, q);
         if (cookie->path)
            g_string_sprintfa(cookie_gstring, "; $Path=%s%s%s",
                              q, cookie->path, q);
         if (cookie->domain)
            g_string_sprintfa(cookie_gstring, "; $Domain=%s%s%s",
                              q, cookie->domain, q);

         if (cookie->ports) {
            char *ports_str = Cookies_build_ports_str(cookie);
            g_string_sprintfa(cookie_gstring, "; $Port=%s", ports_str);
            g_free(ports_str);
         }

         if (g_list_next(matching_cookies) != NULL)
            g_string_append(cookie_gstring, "; ");
         else
            g_string_append(cookie_gstring, "\r\n");
      }
      DEBUG_MSG(4, "Final cookie string: %s", cookie_gstring->str);
   }

   g_free(path);
   str = cookie_gstring->str;
   g_string_free(cookie_gstring, FALSE);
   return str;
}

/* -------------------------------------------------------------
 *                    Access control routines
 * ------------------------------------------------------------- */


/*
 * Get the cookie control rules
 */
static void Cookie_control_init()
{
   CookieControl cc;
   FILE *stream;
   char *filename;
   char line[LINE_MAXLEN];
   char domain[LINE_MAXLEN];
   char rule[LINE_MAXLEN];
   int i, j;

   /* Get a file pointer */
   filename = a_Misc_prepend_user_home(".dillo/cookiesrc");
   stream = Cookies_fopen(filename, "DEFAULT DENY\n");
   g_free(filename);

   if (!stream)
      return;

   /* Get all lines in the file */
   while (!feof(stream)) {
      line[0] = '\0';
      fgets(line, LINE_MAXLEN, stream);

      /* Remove leading and trailing whitespaces */
      g_strstrip(line);

      if (line[0] != '\0' && line[0] != '#') {
         i = 0;
         j = 0;

         /* Get the domain */
         while (!isspace(line[i]))
            domain[j++] = line[i++];
         domain[j] = '\0';

         /* Skip past whitespaces */
         i++;
         while (isspace(line[i]))
            i++;

         /* Get the rule */
         j = 0;
         while (line[i] != '\0' && !isspace(line[i]))
            rule[j++] = line[i++];
         rule[j] = '\0';

         if (g_strcasecmp(rule, "ACCEPT") == 0)
            cc.action = COOKIE_ACCEPT;
         else if (g_strcasecmp(rule, "ACCEPT_SESSION") == 0)
            cc.action = COOKIE_ACCEPT_SESSION;
         else if (g_strcasecmp(rule, "DENY") == 0)
            cc.action = COOKIE_DENY;
         else {
            g_print("Cookies: rule '%s' for domain '%s' is not recognised.\n",
                    rule, domain);
            continue;
         }

         cc.domain = g_strdup(domain);
         if (g_strcasecmp(cc.domain, "DEFAULT") == 0) {
            /* Set the default action */
            default_action = cc.action;
            g_free(cc.domain);
         } else {
            a_List_add(ccontrol, num_ccontrol, num_ccontrol_max);
            ccontrol[num_ccontrol++] = cc;
         }
      }
   }

   fclose(stream);
}

/*
 * Check the rules for an appropriate action for this domain
 */
static CookieControlAction Cookies_control_check_domain(const char *domain)
{
   int i, diff;

   for (i = 0; i < num_ccontrol; i++) {
      if (ccontrol[i].domain[0] == '.') {
         diff = strlen(domain) - strlen(ccontrol[i].domain);
         if (diff >= 0) {
            if (g_strcasecmp(domain + diff, ccontrol[i].domain) != 0)
               continue;
         } else {
            continue;
         }
      } else {
         if (g_strcasecmp(domain, ccontrol[i].domain) != 0)
            continue;
      }

      /* If we got here we have a match */
      return( ccontrol[i].action );
   }

   return default_action;
}

/*
 * Same as the above except it takes an URL
 */
static CookieControlAction Cookies_control_check(const DilloUrl *url)
{
   return Cookies_control_check_domain(URL_HOST(url));
}

#endif /* !DISABLE_COOKIES */
