/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * gdl.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/socket.h>
#include <ctype.h>
#include <netinet/in.h>
#include <glib.h>

#include "gdl.h"
#include "display.h"
#include "network.h"
#include "action.h"
#include "var.h"
#include "user_manage.h"
#include "uaddr.h"
#include "macro.h"

static GPtrArray *gdl_array=NULL;
G_LOCK_DEFINE_STATIC(gdl_array);

static void check_if_dl_complete(GDL_ENTRY *gdle);

/************************************************************/
/* scan gdl_array to find a GDL_ENTRY with the given gdl_id */
/***************************************************************/
/* output: -1=not found else it is the index in the array      */
/* NOTE: this function MUST be called when gdl_array is locked */
/***************************************************************/
static int gdl_index(unsigned int gdl_id)
{
	if(gdl_array!=NULL)
	{
		int i;
		for(i=0;i<gdl_array->len;i++)
		{
			GDL_ENTRY *gdle;

			gdle=g_ptr_array_index(gdl_array,i);
			if((gdle!=NULL)&&(gdle->gdl_id==gdl_id))
				return i;
		}
	}
	return -1;
}

/****************************************************************************/
/* scan links of the given gdle to find a GDL_DL_ENTRY matching information */
/****************************************************************************/
/* output: -1=not found else it is the index in the array      */
/* NOTE: this function MUST be called when gdl_array is locked */
/***************************************************************/
static int gdldle_index(GDL_ENTRY *gdle, char *nickname, char *remote_fname)
{
	if(gdle->gdl_links!=NULL)
	{
		int i;

		for(i=0;i<gdle->gdl_links->len;i++)
		{
			GDL_DL_ENTRY *gdldl;

			gdldl=g_ptr_array_index(gdle->gdl_links,i);
			if((gdldl!=NULL)&&
				(!strcmp(gdldl->nickname->str,nickname))&&
				(!strcmp(gdldl->remote_fname->str,remote_fname)))
				return i;
		}
	}
	return -1;
}

typedef struct {unsigned long b; unsigned long int e;} S_RNG;

static int com_s_rng(const void *a, const void *b)
{
	int u;

	u=(((S_RNG*)a)->b) - (((S_RNG*)b)->b);
	if(u!=0)
		return u;
	return (((S_RNG*)a)->e) - (((S_RNG*)b)->e);
}

/*******************************************************************/
/* group all already downloaded segments into a more usable format */
/*******************************************************************/
GArray *ranges_to_planar(GDL_ENTRY *gdle, int non_running_only)
{
	GArray *nw;
	int i;
	S_RNG tm;

	nw=g_array_new(FALSE,FALSE,sizeof(S_RNG));

	/* put already download ranges inside the array */
	if(gdle->dld_ranges!=NULL)
	{
		for(i=0;i<gdle->dld_ranges->len;i++)
		{
			RANGE_ENTRY *re;

			re=&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,i));
			tm.b=re->range[0];
			tm.e=re->range[1];
			nw=g_array_append_val(nw,tm);
		}
	}

	/* and add part of the download in progress */
	if(non_running_only==0)
	{
		if(gdle->gdl_links!=NULL)
		{
			for(i=0;i<gdle->gdl_links->len;i++)
			{
				GDL_DL_ENTRY *dle;
				dle=g_ptr_array_index(gdle->gdl_links,i);
				if(dle->is_started==1)
				{
					tm.b=dle->range[0];
					tm.e=dle->range[0]+dle->cur_dled;	/* NOTE: here, we can have empty segment but it is necessary */
					nw=g_array_append_val(nw,tm);
				}
			}
		}
	}

	if(nw->len>1)
	{
		qsort(nw->data,nw->len,sizeof(S_RNG),com_s_rng);

		/* now, we can group consecutive segments */
		i=0;
		while(i<(nw->len-1))
		{
			S_RNG *p1,*p2;

			p1=&(g_array_index(nw,S_RNG,i));
			p2=&(g_array_index(nw,S_RNG,i+1));

			if(p1->e>=p2->b)		/* end of the first segment is (at least) the beginning of the second one => group them */
			{							/* if we can really have a full control, we must use an exact test */
				p1->e=p2->e;
				nw=g_array_remove_index(nw,i+1);
			}
			else
				i++;
		}
	}
	return nw;
}

/* convert the list of downloaded segment into a list of free segment */
static GArray *dl_planar_to_free_planar(GArray *dl_ar, GDL_ENTRY *gdle)
{
	S_RNG nw;
	S_RNG *p1,*p2;
	int i;

	if(dl_ar->len==0)
	{
		/* special case when nothing is availabled */
		nw.b=0;
		nw.e=gdle->total_size;
		dl_ar=g_array_append_val(dl_ar,nw);
		return dl_ar;
	}

	/* be sure the first interval starts with 0 */
	p1=&(g_array_index(dl_ar,S_RNG,0));
	if(p1->b!=0)
	{
		nw.b=0;
		nw.e=0;
		dl_ar=g_array_insert_val(dl_ar,0,nw);
	}

	/* be sure the last interval ends with the file size */
	p1=&(g_array_index(dl_ar,S_RNG,dl_ar->len-1));
	if(p1->e!=gdle->total_size)
	{
		nw.b=gdle->total_size;
		nw.e=gdle->total_size;
		dl_ar=g_array_append_val(dl_ar,nw);
	}

	/* now, it is easy. we remove the begin of the first interval, the end of the last */
	/* the begin of each interval is its old ends and the end of each interval is the old begin of the next one */
	for(i=0;i<dl_ar->len-1;i++)
	{
		p1=&(g_array_index(dl_ar,S_RNG,i));
		p2=&(g_array_index(dl_ar,S_RNG,i+1));

		p1->b=p1->e;
		p1->e=p2->b;
	}
	/* and we remove the last one */
	dl_ar=g_array_remove_index_fast(dl_ar,dl_ar->len-1);
	return dl_ar;
}

/*****************************************************************************/
/* adjust the list of free segments to be sure no segment goes above max_pos */
/*****************************************************************************/
static GArray *crop_free_planar_by_size(GArray *dl_ar, unsigned long int max_pos)
{
	int i;
	S_RNG *p1;

	i=0;
	while(i<dl_ar->len)
	{
		p1=&(g_array_index(dl_ar,S_RNG,i));
		if(p1->b>max_pos)
		{
			dl_ar=g_array_remove_index_fast(dl_ar,i);	/* we can use fast here because all segments above this one */
																	/* start after this one so, they will be deleted */
		}
		else if(p1->e>max_pos)
		{
			p1->e=max_pos;			/* truncate the given segment */
			i++;
		}
		else
			i++;
		
	}

	return dl_ar;
}

/*****************************************************************************/
/* check if the given position is the current position of a running transfer */
/*****************************************************************************/
/* output: 1=yes, 0=no */
/***********************/
static int is_an_end_range(GDL_ENTRY *gdle, unsigned long int upper_bound)
{
	int j;

	for(j=0;j<gdle->gdl_links->len;j++)
	{
		GDL_DL_ENTRY *gdldle;

		gdldle=g_ptr_array_index(gdle->gdl_links,j);

		if(gdldle->is_started==1)
		{
			if((gdldle->range[0]+gdldle->cur_dled)==upper_bound)
			{
				return 1;
			}
		}
	}
	return 0;
}

/**********************************************************************************************/
/* move free segments of dl_ar into run_dl_ar if they appear after a running transfer of gdle */
/**********************************************************************************************/
static GArray *split_free_planar_if_running(GArray *dl_ar, GArray **run_dl_ar, GDL_ENTRY *gdle)
{
	int i;

	*run_dl_ar=g_array_new(FALSE,FALSE,sizeof(S_RNG));

	i=0;
	while(i<dl_ar->len)
	{
		S_RNG *nw;

		nw=&(g_array_index(dl_ar,S_RNG,i));

		if(is_an_end_range(gdle,nw->b))
		{
			/* the beginning of this free bloc is at end of a running download */
			(*run_dl_ar)=g_array_append_val((*run_dl_ar),*nw);
			dl_ar=g_array_remove_index(dl_ar,i);				/* don't forget to always keep the bloc in order */
		}
		else
			i++;
	}

	return dl_ar;
}

/***************************************************/
/* allocate a download range to the given DL_ENTRY */
/***************************************************/
/* output: 0=no range allocated */
/********************************/
static int allocate_range(GDL_DL_ENTRY *gdldle, GDL_ENTRY *gdle)
{
	struct stat st;
	GArray *dl_ar;
	GArray *run_dl_ar;
	int i;
	unsigned long int max_size;
	int cur_idx;
	S_RNG *cur;

	/* this function performs 2 tasks */
	/* find a free download range */
	dl_ar=ranges_to_planar(gdle,0);
	if(dl_ar==NULL)
		return 0;			/* unable to compute ranges */

	dl_ar=dl_planar_to_free_planar(dl_ar,gdle);
	dl_ar=crop_free_planar_by_size(dl_ar,gdldle->remote_fsize);
	dl_ar=split_free_planar_if_running(dl_ar,&run_dl_ar,gdle);
	
	/* now, dl_ar and run_dl_ar are the list of all free blocks */
	/* run_dl_ar are blocks having size shrinking due to running downloads */
	/* dl_ar are non shrinking blocs */

	/* algo: if dl_ar is not empty, take the biggest free bloc inside it */
	/*       if dl_ar is empty, take the biggest free bloc inside run_dl_ar, */
	/*         the upper limit this bloc is the upper limite of the range */
	/*         the lower limit of the range is the middle of the bloc */
	if(dl_ar->len!=0)
	{	/* great, there is free non shrinking blocs */
		cur_idx=0;
		cur=&(g_array_index(dl_ar,S_RNG,cur_idx));
		max_size=cur->e-cur->b;

		/* find the biggest bloc */
		for(i=1;i<dl_ar->len;i++)
		{
			cur=&(g_array_index(dl_ar,S_RNG,i));

			if((cur->e-cur->b)>max_size)
			{
				max_size=cur->e-cur->b;
				cur_idx=i;
			}
		}

		/* and assign the range */
		cur=&(g_array_index(dl_ar,S_RNG,cur_idx));
		gdldle->range[0]=cur->b;
		gdldle->range[1]=cur->e;
	}
	else
	{
		if(run_dl_ar->len==0)
		{
			/* no free non shrinking and non shrinking blocs available, abort here */
			g_array_free(dl_ar,TRUE);
			g_array_free(run_dl_ar,TRUE);
			return 0;
		}

		cur_idx=0;
		cur=&(g_array_index(run_dl_ar,S_RNG,cur_idx));
		max_size=cur->e-cur->b;

		/* find the biggest bloc */
		for(i=1;i<run_dl_ar->len;i++)
		{
			cur=&(g_array_index(run_dl_ar,S_RNG,i));

			if((cur->e-cur->b)>max_size)
			{
				max_size=cur->e-cur->b;
				cur_idx=i;
			}
		}

		/* ok, we have the biggest bloc */
		cur=&(g_array_index(run_dl_ar,S_RNG,cur_idx));
		gdldle->range[0]=cur->b+(max_size/2);		/* start in the middle of the bloc */
		gdldle->range[1]=cur->e;						/* and end at its end */
	}

	g_array_free(dl_ar,TRUE);
	g_array_free(run_dl_ar,TRUE);

	gdldle->cur_dled=0;					/* reset the number of downloaded bytes */

	/* initialize temp_local_fname according to the download range */
	/* the localname is initialized to "GDL/localname/range[0]-range[1]" */
	gdldle->temp_local_fname=g_string_new("GDL");
	if(stat(gdldle->temp_local_fname->str,&st)==-1)
	{
		/* on error, abort or try to create the directory */
		if((errno!=ENOENT)||(mkdir(gdldle->temp_local_fname->str,0777)==-1))
		{	
			g_string_free(gdldle->temp_local_fname,TRUE);
			gdldle->temp_local_fname=NULL;
			return 0;
		}
	}
	gdldle->temp_local_fname=g_string_append_c(gdldle->temp_local_fname,'/');
	gdldle->temp_local_fname=g_string_append(gdldle->temp_local_fname,gdle->local_fname->str);
	if(stat(gdldle->temp_local_fname->str,&st)==-1)
	{
		/* on error, abort or try to create the directory */
		if((errno!=ENOENT)||(mkdir(gdldle->temp_local_fname->str,0777)==-1))
		{	
			g_string_free(gdldle->temp_local_fname,TRUE);
			gdldle->temp_local_fname=NULL;
			return 0;
		}
	}

	g_string_sprintfa(gdldle->temp_local_fname,"/%08lX-%08lX",gdldle->range[0],gdldle->range[1]);
	return 1;	/* range allocated */
}

/*****************************/
/* try to start its download */
/*****************************/
static void start_xdl(GDL_ENTRY *gdle, GDL_DL_ENTRY *gdldle)
{
	GString *nw;

	gdldle->last_start_time=time(NULL);	/* always update time to avoid repeted try in short time */

	if(!user_in_list(hub_user_list,gdldle->nickname->str))		/* user on the hub */
	{
		if((with_ddl==0)||(!check_uaddr_entry_by_name(gdldle->nickname->str)))	/* or DDL enable and user address is known */
			return;												/* is required else, we leave */
	}

	gdldle->is_running=1;
	gdldle->is_started=0;

	/* make a download request */
	nw=g_string_new("");
	g_string_sprintf(nw,"/XDL|%u|%s|",gdle->gdl_id,gdldle->nickname->str);
	add_new_sim_input(0,nw->str);
	g_string_free(nw,TRUE);
}

/**************************************************************************************************/
/* check if the given nickname (ptr->nickname->str) has gdldle with is_running=1 and is_started=0 */
/**************************************************************************************************/
/* output: 1=yes, 0=no */
/***********************/
static int has_running_not_started_xdl(GDL_ENTRY *gdle, GDL_DL_ENTRY *gdldle)
{
	char *nick=gdldle->nickname->str;
	int i;

	for(i=0;i<gdle->gdl_links->len;i++)
	{
		GDL_DL_ENTRY *ptr;

		ptr=g_ptr_array_index(gdle->gdl_links,i);
		if(! ((ptr->is_running==1)&&(ptr->is_started==0)) )
			continue;
		if(!strcmp(nick,ptr->nickname->str))
			return 1;
	}
	return 0;
}

/*********************************************************************/
/* scan the given ENTRY and try to start downloads on inactive links */
/*********************************************************************/
static void handle_one_gdl(GDL_ENTRY *gdle)
{
	int i;
	time_t cur_time;

	if(gdle->gdl_links==NULL)
		return;				/* nothing can be done */

	cur_time=time(NULL);

	/* try to start downloads on inactive links */
	for(i=0;i<gdle->gdl_links->len;i++)
	{
		GDL_DL_ENTRY *ptr;

		ptr=g_ptr_array_index(gdle->gdl_links,i);

		/* as soon as this link is up, it cannot be changed */
		if(ptr->is_running)
		{
			/* these few lines fix a bug somewhere else in the code */
			/* sometimes, tasks are set to waiting but in fact, the real task (/XDL has been destroyed) */
			if((ptr->is_started==0)&&((ptr->last_start_time+600)<cur_time))
			{
				/* this XDL command has been sent more than 10 minutes and still not running ? BUG BUG BUG */
				ptr->is_started=0;
				ptr->is_running=0;
			}
			else
				continue;
		}

		if((ptr->last_start_time+300)>cur_time)		/* wait at least 5 minutes after an error */
			continue;

		if(!has_running_not_started_xdl(gdle,ptr))	/* to avoid a race and forever waiting transfer (which in fact no more exist) */
																	/* we initiate only one connection at a time to the same user */
			start_xdl(gdle,ptr);
	}
}

/*************************************************************/
/* append the given range_entry at the end of the given file */
/*********************************************************************************/
/* input: f=FILE * to save data. The position in file is *cur_pos                */
/*        *cur_pos= current save position in f                                   */
/*       RANGE_ENTRY= segment to append (only part of the segment can be useful) */
/* output: return value !=0: it is the errno of the command making an error      */
/*         return value==0, *cur_pos is the new position in file f and this one  */
/*         contains the data.                                                    */
/*********************************************************************************/
static int append_this_range_entry_to_file(FILE *f, unsigned long *cur_pos, RANGE_ENTRY *re)
{
	FILE *add;
	unsigned long to_append;
	int a;
	char buf[8192];

	add=fopen(re->temp_local_fname->str,"rb");
	if(add==NULL)
	{
		return errno;
	}

	if(*cur_pos!=re->range[0])
	{
		/* we wont use the whole part, we have to skip the beginning */
		if(fseek(add,(*cur_pos)-re->range[0],SEEK_SET))
		{
			a=errno;
			fclose(add);
			return a;
		}
	}

	/* copy the remaining part of the block at the end of the file */
	a=0;
	to_append=re->range[1]-(*cur_pos);
	while(to_append!=0)
	{
		int want,have;
		
		/* read a segment slice */
		want=MIN(sizeof(buf),to_append);
		have=fread(buf,1,want,add);

		if(have==-1)
		{
			a=errno;
			break;
		}
		if(have!=want)
		{
			a=ENODATA;			/* set the error message to "no data available */
			break;
		}

		/* and write it at the end of the file */
		want=fwrite(buf,1,have,f);
		if(want!=have)
		{
			a=ferror(f);
			break;
		}

		(*cur_pos)+=have;
		to_append-=have;
	}

	fclose(add);
	return a;
}

/*******************************************************************************************************/
/* scan the array of RANGE_ENTRY and return the index of an entry where cur_pos is between range value */
/*******************************************************************************************************/
/* output: index or -1 */
/***********************/
static int get_range_entry_for_position(GArray *dld_ranges, unsigned long cur_pos)
{
	int i;
	RANGE_ENTRY *nw;

	for(i=0;i<dld_ranges->len;i++)
	{
		nw=&(g_array_index(dld_ranges,RANGE_ENTRY,i));

		if((nw->range[0]<=cur_pos)&&(cur_pos<nw->range[1]))	/* be careful, the upper bound is excluded while the lower bound is included */
			return i;
	}
	return -1;
}

/*********************************************/
/* gather part of the file into one big file */
/*********************************************/
/* output:0=success, !=0=error */
/*******************************/
static int gather_gdl_part(GDL_ENTRY *gdle)
{
	FILE *f;
	unsigned long cur_pos=0;
	int idx;
	int a;

	f=fopen(gdle->local_fname->str,"wb");
	if(f==NULL)
	{
		a=errno;
		disp_msg(ERR_MSG,"gather_gdl_part","unable to create",gdle->local_fname->str,strerror(a),NULL);
		return 1;		/* error */
	}

	cur_pos=0;
	while(cur_pos!=gdle->total_size)
	{
		idx=get_range_entry_for_position(gdle->dld_ranges,cur_pos);
		if(idx==-1)
		{
			disp_msg(ERR_MSG,"gather_gdl_part","Internal error, a completed GDL contains missing parts",gdle->local_fname->str,
									"position","|lu",(unsigned long)cur_pos,NULL);
			fclose(f);
			unlink(gdle->local_fname->str);
			return 1;		/* error */
		}

		if((a=append_this_range_entry_to_file(f,&cur_pos,&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,idx))))!=0)
		{
			disp_msg(ERR_MSG,"gather_gdl_part","Fail to append",(g_array_index(gdle->dld_ranges,RANGE_ENTRY,idx)).temp_local_fname->str,"at the end of",
									gdle->local_fname->str,"Reason:",strerror(a),NULL);
			fclose(f);
			unlink(gdle->local_fname->str);
			return 1;		/* error */
		}
	}
	fclose(f);

	{
		FILE *f;
		GString *o;
		G_LOCK_DEFINE_STATIC(done_log);

		o=g_string_new("");
		g_string_sprintf(o,"%s.done",local_dctc_sock_path->str);

		G_LOCK(done_log);

		f=fopen(o->str,"ab");
		if(f!=NULL)
		{
			fprintf(f,"(multi-source)|XDL|%s|%lu\n",
							gdle->local_fname->str,gdle->total_size);
			fclose(f);
		}
		G_UNLOCK(done_log);
		g_string_free(o,TRUE);
	}

	if(when_done)
	{
		GString *dir_part;
		char *t;
		int offset;

		/* source: gdle->local_fname->str */

		dir_part=g_string_new(gdle->local_fname->str);
		t=strrchr(dir_part->str,'/');
		if(t!=NULL)
		{
			offset=t-dir_part->str+1;	  /* position just after the / */
		}
		else
		{
			offset=0;
		}
		dir_part=g_string_truncate(dir_part,offset);
		dir_part=g_string_append(dir_part,"done");

		/* create the destination directory */
		mkdir(dir_part->str,0777);

		dir_part=g_string_assign(dir_part,gdle->local_fname->str);
		dir_part=g_string_insert(dir_part,offset,"done/");
		/* move the file */
		if(rename(gdle->local_fname->str,dir_part->str))
		{
			int err=errno;

			disp_msg(ERR_MSG,NULL,"Fail to rename",gdle->local_fname->str,"into",dir_part->str,"because:",strerror(err),NULL);
		}
		g_string_free(dir_part,TRUE);
	}

	return 0;
}

/******************************/
/* delete the given directory */
/******************************/
static void wipe_temp_directory(GString *str)
{
	DIR *dir;
	struct dirent *obj;
	struct stat st;
	GString *tmp;

	dir=opendir(str->str);
	if(dir==NULL)
		return;

	tmp=g_string_new(NULL);

	/* delete all files of the given directory */
	while((obj=readdir(dir))!=NULL)
	{
		tmp=g_string_assign(tmp,str->str);
		g_string_sprintfa(tmp,"/%s",obj->d_name);

		if(stat(tmp->str,&st)==0)
		{
			if(S_ISREG(st.st_mode))
			{
				unlink(tmp->str);
			}
		}
	}
	g_string_free(tmp,TRUE);
	closedir(dir);
	rmdir(str->str);
}

/*********************************/
/* gather all files parts of GDL */
/*********************************/
static void inline gather_part_and_end_gdl(GDL_ENTRY *gdle)
{
	if(!gather_gdl_part(gdle))
	{	/* on gathering success, the GDL is destroyed */
		GString *str;

		str=g_string_new("GDL");
		str=g_string_append_c(str,'/');
		str=g_string_append(str,gdle->local_fname->str);

		do_gdl_end(gdle->gdl_id,1);		/* enable override because the is_completed==1 */
		wipe_temp_directory(str);
		g_string_free(str,TRUE);
	}
}

/*************************************************************************/
/* build and send queries according to autoscan_delay and search pattern */
/***************************************************************************/
/* this function is called only when it is possible to send query to a hub */
/***************************************************************************/
static void do_gdl_autoscan(GDL_ENTRY *gdle)
{
	int i;
	time_t min_time,cur_time;

	if((gdle==NULL)||(gdle->autoscan==NULL)||(gdle->autoscan->len==0))
		return;

	cur_time=time(NULL);
	min_time=cur_time-auto_scan_delay;

	for(i=0;i<gdle->autoscan->len;i++)
	{
		GDL_AS_ENTRY *gae;

		gae=&(g_array_index(gdle->autoscan,GDL_AS_ENTRY,i));
		if((gae->last_scan<min_time)&&(gae->search_pattern!=NULL))
		{
			GString *str;
			str=g_string_new("");

			g_string_sprintf(str,"$Search %s:%hu T?F?%lu?%s|",														/* the size is always important, it is at least */
												host_ip,gdle->autoscan_sock_port,
												gdle->total_size-1,			/* it is not possible to ask for an exact size (N) */
																					/* to reduce bandwidth waste, we ask a size of at least N-1 bytes */
												gae->search_pattern);		/* it is a datatype?pattern in fact */
			
			add_new_sim_input(0,str->str);	/* yes, it is possible to sim_input DC command directly, not only dctc keyboard command, great isn't it ? :) */
			g_string_free(str,TRUE);
			gae->last_scan=cur_time;
		}
	}
}

/*******************************************************************************/
/* process incoming autoscan search result and update GDL download source list */
/*******************************************************************************/
/* this function is basically the same as the $SR one */
/******************************************************/
static void	receive_and_decode_autoscan_result(GDL_ENTRY *gdle)
{
	char buf[8192];
	int ret;
	struct sockaddr_in gdl_srch_port_raddr;
	int gdl_srch_port_raddr_len;

	gdl_srch_port_raddr_len=sizeof(gdl_srch_port_raddr);
	ret=recvfrom(gdle->autoscan_sockfd,buf,sizeof(buf),MSG_NOSIGNAL,(void*)&gdl_srch_port_raddr,&gdl_srch_port_raddr_len);

	if(ret!=-1)
	{
		buf[ret]='\0';
		if(!strncmp(buf,"$SR ",4))			/* only process $SR message */
		{
			/* buf format: "$SR nick filename\5filesize slotratio\5hubname (hubIP)|" */
			char *nick;
			char *fname;
			char *fsize;
			unsigned long file_size;

			nick=buf+3;
			SKIP_SPACE(nick);
			
			if(*nick=='\0')
				return;

			fname=strchr(nick,' ');
			if(fname==NULL)
				return;

			*fname++='\0';
			SKIP_SPACE(fname);
			if(*fname=='\0')
				return;

			fsize=strchr(fname,5);		/* search the \5 */
			if(fsize==NULL)
				return;
			*fsize++='\0';

			if(*fsize=='\0')
				return;

			file_size=strtoul(fsize,NULL,10);

			/* well, now, nick, fname and file_size are ready */
			if(file_size==gdle->total_size)
			{
				/* everything is ok, we can add the source */
				/* NOTE: we cannot call do_gdl_add because gdl_array is already locked */
				GString *new_src;

				new_src=g_string_new("");
				g_string_sprintf(new_src,"/GDLADD %u|%s|%s|%lu",gdle->gdl_id,nick,fname,file_size);
				add_new_sim_input(0,new_src->str);
				g_string_free(new_src,TRUE);
			}
		}
	}
}

/**************************************************/
/* scans gdl_array every second and perform tasks */
/**************************************************/
static void gdl_thread(void *dummy)
{
	int i;
	fd_set rd;
	int max_fd=-1;

	while(1)
	{
		restart_loop:

		FD_ZERO(&rd);
		G_LOCK(gdl_array);

		i=0;
		while(i<gdl_array->len)
		{
			GDL_ENTRY *gdle;
			gdle=g_ptr_array_index(gdl_array,i);

			if(gdle==NULL)
			{
				g_ptr_array_remove_index_fast(gdl_array,i);
			}
			else
			{
				if(gdle->is_completed==0)
				{
					handle_one_gdl(g_ptr_array_index(gdl_array,i));
					i++;
				}
				else
				{
					if(gdle->is_completed==1)
					{
						/* due to the fact this gdle->is_completed==1, no action */
						/* will be performed by anyone, including its destruction */
						/* so we can release the lock and perform the end of the computation */
						G_UNLOCK(gdl_array);
						gather_part_and_end_gdl(gdle);
						goto restart_loop;
					}
					i++;
				}
			}

			if((main_sck!=-1)&&(cnx_in_progress==0)&&(!behind_fw))
			{	/* if we are connected to the hub, we must handle GDL autoscan */
				do_gdl_autoscan(gdle);
				
				if(gdle->autoscan_sockfd!=-1)
				{
					FD_SET(gdle->autoscan_sockfd,&rd);
					max_fd=MAX(max_fd,gdle->autoscan_sockfd);
				}
			}
		}

		G_UNLOCK(gdl_array);

		if((max_fd==-1)||(behind_fw))		/* no search socket or behind a firewall */
			sleep(1);
		else
		{
			struct timeval tv;
			int ln;

			/* wait at most 1 second */
			tv.tv_sec=1;
			tv.tv_usec=0;

			ln=select(max_fd+1,&rd,NULL,NULL,&tv);
			if(ln>0)
			{
				G_LOCK(gdl_array);
				i=0;
				while(i<gdl_array->len)
				{
					GDL_ENTRY *gdle;
					gdle=g_ptr_array_index(gdl_array,i);
		
					if((gdle!=NULL)&&(gdle->autoscan_sockfd!=-1)&&(FD_ISSET(gdle->autoscan_sockfd,&rd)))
					{
						/* this gdle have received something. */
						receive_and_decode_autoscan_result(gdle);
					}
					i++;
				}
				G_UNLOCK(gdl_array);
			}
		}
	}
	pthread_exit(NULL);
}

/**********************************************************/
/* create the GDL thread. On error, program is terminated */
/**********************************************************/
void start_gdl_thread(void)
{
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	if(gdl_array==NULL)
	{
		gdl_array=g_ptr_array_new();
	}

	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);

	if(pthread_create(&thread_id,&thread_attr, (void*)gdl_thread,NULL)!=0)
	{
		disp_msg(ERR_MSG,"start_gdl_thread","pthread_create failed",NULL);
		exit(1);
	}
}

/* -------------------------------------------------------------------------- */
/**************************************************************/
/* the following functions are called using keyboard commands */
/**************************************************************/

static void create_dl_dir_and_lock(char *fname,GDL_ENTRY *gdle)
{
	GString *str;
	str=g_string_new("GDL");
	mkdir(str->str,0777);			/* just to avoid problem, create GDL/ */
	str=g_string_append_c(str,'/');
	str=g_string_append(str,fname);
	mkdir(str->str,0777);			/* create GDL/filename */
	str=g_string_append(str,"/.lock");
	gdle->lock_fd=open(str->str,O_RDWR|O_CREAT|O_TRUNC,0777);
	if(lockf(gdle->lock_fd,F_TLOCK,1))	/* enable try lock to avoid handing client */
	{
		disp_msg(ERR_MSG,"create_dl_dir_and_lock","lockf failed",strerror(errno),"WARNING: error ignored",NULL);
	}

	g_string_free(str,TRUE);
}

/*********************************************************************/
/* check if the given filename does not still exist in GDL directory */
/* if it exist, try to lock the file .lock. If it does not exist,    */
/* return 0 (=not exist), if it exists and lock fails, return 1      */
/* (=exist) and *is_running=1 (someone else runs it). Else *is_running*/
/* =0 (nobody else runs it)                                          */
/*********************************************************************/
static int this_gdl_still_exist(char *local_filename,int *is_running)
{
	int ret=0;
	GString *str;
	struct stat st;
	int fd;

	str=g_string_new("GDL");
	mkdir(str->str,0777);			/* just to avoid problem */
	str=g_string_append_c(str,'/');
	str=g_string_append(str,local_filename);

	if(stat(str->str,&st)==0)
	{
		if(S_ISDIR(st.st_mode))
		{
			str=g_string_append(str,"/.lock");
			
			fd=open(str->str,O_RDWR);
			if(fd==-1)
			{	/* lock file does not exist */
				if(errno==EISDIR)
				{
					disp_msg(ERR_MSG,"this_gdl_still_exists",str->str,"exists but is not a file as expected",NULL);
					*is_running=1;
					ret=1;
				}

				/* so, there is nothing, assume it is not a GDL */
			}
			else
			{	/* lock file exists */
				ret=1;		/* the lock exists */

				/* test the lock */
				if(lockf(fd,F_TEST,1))
				{	
					*is_running=1;	
				}
				else
					*is_running=0;		/* able to lock -> nobody uses it */
				close(fd);
			}
		}
		else
		{
			disp_msg(ERR_MSG,"this_gdl_still_exists",str->str,"exists but is not a directory as expected",NULL);
			*is_running=1;
			ret=1;
		}
	}
	g_string_free(str,TRUE);
	return ret;
}

/******************************************/
/* update the .cmd file of the given gdle */
/******************************************/
static void update_gdl_cmd_of_this_gdle(GDL_ENTRY *gdle)
{
	FILE *f;
	GString *str;
	int i;

	str=g_string_new("GDL/");
	str=g_string_append(str,gdle->local_fname->str);
	str=g_string_append(str,"/.cmd");

	f=fopen(str->str,"wb");
	if(f==NULL)
	{
		disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","unable to create",str->str,NULL);
		g_string_free(str,TRUE);
		return;
	}
	
	/* create the GDL name entry */
	g_string_sprintf(str,"G|%u|%s|%lu\n",gdle->gdl_id,gdle->local_fname->str,gdle->total_size);
	if(fwrite(str->str,1,str->len,f)!=str->len)
	{
		disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","write fail",str->str,NULL);
	}
	else
	{
		/* put the GDL source entries */
		for(i=0;i<gdle->gdl_links->len;i++)
		{
			GDL_DL_ENTRY *ptr;
			ptr=g_ptr_array_index(gdle->gdl_links,i);
			g_string_sprintf(str,"S|%s|%s|%lu\n",ptr->nickname->str,ptr->remote_fname->str,ptr->remote_fsize);
			if(fwrite(str->str,1,str->len,f)!=str->len)
			{
				disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","write fail",str->str,NULL);
				goto abrt;
			}

			/* if this entry is active, we also add its file to the list */
			if((ptr->is_running)&&(ptr->is_started)&&(ptr->temp_local_fname!=NULL))
			{
				g_string_sprintf(str,"A|%s|%lu|-\n",ptr->temp_local_fname->str,ptr->range[0]);		/* always 3 | per line */
				if(fwrite(str->str,1,str->len,f)!=str->len)
				{
					disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","write fail",str->str,NULL);
					goto abrt;
				}
			}
		}

		/* put the ranges */
		for(i=0;i<gdle->dld_ranges->len;i++)
		{
			RANGE_ENTRY *ptr;
			ptr=&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,i));

			g_string_sprintf(str,"R|%s|%lu|%lu\n",ptr->temp_local_fname->str,ptr->range[0],ptr->range[1]);
			if(fwrite(str->str,1,str->len,f)!=str->len)
			{
				disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","write fail",str->str,NULL);
				goto abrt;
			}
		}

		/* put the autoscan pattern */
		for(i=0;i<gdle->autoscan->len;i++)
		{
			GDL_AS_ENTRY *ptr;

			ptr=&(g_array_index(gdle->autoscan,GDL_AS_ENTRY,i));
			g_string_sprintf(str,"F|%s|-|-\n",ptr->search_pattern);		/* always 3 | per line */
			if(fwrite(str->str,1,str->len,f)!=str->len)
			{
				disp_msg(ERR_MSG,"update_gdl_cmd_of_this_gdle","write fail",str->str,NULL);
				goto abrt;
			}
		}
	}

	abrt:
	fclose(f);
	g_string_free(str,TRUE);
	return;
}

/*******************************************/
/* create a new GDL_ENTRY inside gdl_array */
/*******************************************/
/* output: 0= success, !=0: error */
/**********************************/
int do_gdl_new(unsigned int gdl_id, char *local_filename, unsigned long int total_size)
{
	int ret=1;	/* default: error */
	int idx;
	int is_running;

	if(this_gdl_still_exist(local_filename,&is_running))
	{
		if(is_running)
			disp_msg(ERR_MSG,"do_gdl_new","a GDL with the same name still runs on another client",NULL);
		else
			disp_msg(ERR_MSG,"do_gdl_new","a GDL with the same name still exists but does not run on another client",
									"Use /GDLATTACH to attach it to this client",NULL);
		return ret;
	}

	G_LOCK(gdl_array);

	/* first, check if the given index is not already used */
	idx=gdl_index(gdl_id);
	if(idx==-1)
	{
		/* ok, it does not exist */
		GDL_ENTRY *nw;

		nw=malloc(sizeof(GDL_ENTRY));
		if(nw!=NULL)
		{
			nw->gdl_id=gdl_id;
			nw->start_time=time(NULL);
			nw->is_completed=0;
			nw->local_fname=g_string_new(local_filename);
			nw->total_size=total_size;
			nw->gdl_links=g_ptr_array_new();
			nw->dld_ranges=g_array_new(FALSE,FALSE,sizeof(RANGE_ENTRY));
			nw->autoscan_sockfd=-1;
			nw->autoscan=g_array_new(FALSE,FALSE,sizeof(GDL_AS_ENTRY));
			create_dl_dir_and_lock(nw->local_fname->str,nw);
			g_ptr_array_add(gdl_array,nw);
			ret=0;
			update_gdl_cmd_of_this_gdle(nw);
			nw->dl_offset=0;			/* no data are in dld_ranges */
		}
	}
	else
	{
		disp_msg(ERR_MSG,"do_gdl_new","a GDL with the same number still exists",NULL);
	}
	
	G_UNLOCK(gdl_array);
	return ret;
}

/*****************************************************/
/* load the .cmd file into the given empty GDL entry */
/*****************************************************/
/* output: 0=ok, !=0=error */
/***************************/
static int load_gdl_cmd(GDL_ENTRY *gdle, char *filename)
{
	char buf[10240];
	FILE *f;
	char *t;
	gchar **fields=NULL;
	GString *str;

	str=g_string_new("GDL/");
	str=g_string_append(str,filename);
	str=g_string_append(str,"/.cmd");

	f=fopen(str->str,"rb");
	g_string_free(str,TRUE);
	if(f==NULL)
	{
		disp_msg(ERR_MSG,"load_gdl_cmd","Unable to open",filename,NULL);
		return 1;		/* error */
	}

	if(fgets(buf,sizeof(buf),f)==NULL)
	{
		abrt:
		if(fields!=NULL)
		{
			disp_msg(ERR_MSG,"load_gdl_cmd","Erroneous file",filename,fields[0],fields[1],fields[2],fields[3],NULL);
			g_strfreev(fields);
		}
		else
			disp_msg(ERR_MSG,"load_gdl_cmd","Erroneous file",filename,NULL);

		fclose(f);
		return 1;		/* error */
	}

	if((t=strchr(buf,'\n'))==NULL)
		goto abrt;
	*t='\0';

	/* read the G line which describes the main GDL information */
	fields=g_strsplit(buf,"|",0);
	if((fields[0]==NULL)||(fields[1]==NULL)||(fields[2]==NULL)||(fields[3]==NULL)||(fields[4]!=NULL))
		goto abrt;

	if(strcmp(fields[0],"G"))
		goto abrt;
	
	gdle->local_fname=g_string_new(fields[2]);
	gdle->total_size=strtoul(fields[3],NULL,10);
	g_strfreev(fields);

	while(fgets(buf,sizeof(buf),f)!=NULL)
	{
		fields=g_strsplit(buf,"|",0);
		if((fields[0]==NULL)||(fields[1]==NULL)||(fields[2]==NULL)||(fields[3]==NULL)||(fields[4]!=NULL))
		{
			disp_msg(ERR_MSG,"load_gdl_cmd","Erroneous line (ignored)",filename,fields[0],fields[1],fields[2],fields[3],NULL);
			g_strfreev(fields);
			continue;
		}

		if(!strcmp(fields[0],"S"))
		{
			GDL_DL_ENTRY *nw;

			nw=malloc(sizeof(GDL_DL_ENTRY));
			if(nw==NULL)
				goto abrt;

			nw->nickname=g_string_new(fields[1]);
			nw->remote_fname=g_string_new(fields[2]);
			nw->remote_fsize=strtoul(fields[3],NULL,10);
			nw->temp_local_fname=NULL;
			nw->last_start_time=0;
			nw->is_running=0;
			nw->is_started=0;

			g_ptr_array_add(gdle->gdl_links,nw);
		}
		else if(!strcmp(fields[0],"R"))
		{
			RANGE_ENTRY re;

			re.temp_local_fname=g_string_new(fields[1]);
			re.range[0]=strtoul(fields[2],NULL,10);
			re.range[1]=strtoul(fields[3],NULL,10);
			gdle->dld_ranges=g_array_append_val(gdle->dld_ranges,re);

			gdle->dl_offset+=(re.range[1]-re.range[0]);
		}
		else if(!strcmp(fields[0],"A"))
		{
			struct stat st;

			if((stat(fields[1],&st)==0) &&
				(S_ISREG(st.st_mode)) &&
				(st.st_size!=0) )
			{
				RANGE_ENTRY re;

				re.temp_local_fname=g_string_new(fields[1]);
				re.range[0]=strtoul(fields[2],NULL,10);
				re.range[1]=re.range[0]+st.st_size;
				gdle->dld_ranges=g_array_append_val(gdle->dld_ranges,re);
				gdle->dl_offset+=(re.range[1]-re.range[0]);
			}
		}
		else if(!strcmp(fields[0],"F"))
		{
			GDL_AS_ENTRY gae;

			gae.last_scan=0;
			gae.gae_id=rand();
			gae.search_pattern=g_strdup(fields[1]);
			gdle->autoscan=g_array_append_val(gdle->autoscan,gae);
		}
		else
				goto abrt;
		g_strfreev(fields);
	}
	fclose(f);
	return 0;
}

/**********************************************************************/
/* try to attach an unused GDL named "filename" to the current client */
/**********************************************************************/
void do_gdl_attach(char *filename)
{
	int is_running;

	G_LOCK(gdl_array);
	if(this_gdl_still_exist(filename,&is_running)==0)
	{
		disp_msg(ERR_MSG,"do_gdl_attach","no GDL has this name",filename,NULL);
	}
	else
	{
		if(is_running)
		{
			disp_msg(ERR_MSG,"do_gdl_attach","This GDL cannot be attached, it is still in use",filename,NULL);
		}
		else
		{
			int i;
			unsigned int gdl_id;
			GDL_ENTRY *nw;

			/* find an used id */
			gdl_id=time(NULL);
			while(gdl_index(gdl_id)!=-1)
			{
				gdl_id=rand();
			}

			/* ok, it does not exist */

			nw=malloc(sizeof(GDL_ENTRY));
			if(nw!=NULL)
			{
				nw->gdl_id=gdl_id;
				nw->start_time=time(NULL);
				nw->is_completed=0;
				nw->local_fname=NULL;
				nw->total_size=0;
				nw->gdl_links=g_ptr_array_new();
				nw->dld_ranges=g_array_new(FALSE,FALSE,sizeof(RANGE_ENTRY));
				nw->autoscan_sockfd=-1;
				nw->autoscan=g_array_new(FALSE,FALSE,sizeof(GDL_AS_ENTRY));
				nw->dl_offset=0;
				create_dl_dir_and_lock(filename,nw);

				if(load_gdl_cmd(nw,filename))
				{	/* on error, free allocated memory */
					if(nw->local_fname)
						g_string_free(nw->local_fname,TRUE);
					if(nw->lock_fd!=-1)
					{
						lockf(nw->lock_fd,F_ULOCK,1);
						close(nw->lock_fd);
					}

					/* free gdl_links */
					for(i=0;i<nw->gdl_links->len;i++)
					{
						GDL_DL_ENTRY *dle;

						dle=g_ptr_array_index(nw->gdl_links,i);
						if(dle!=NULL)
						{
							if(dle->nickname)
								g_string_free(dle->nickname,TRUE);
							if(dle->remote_fname)
								g_string_free(dle->remote_fname,TRUE);
							if(dle->temp_local_fname)
								g_string_free(dle->temp_local_fname,TRUE);
						}
					}
					g_ptr_array_free(nw->gdl_links,TRUE);

					/* free range */
					for(i=0;i<nw->dld_ranges->len;i++)
					{
						RANGE_ENTRY *re;

						re=&(g_array_index(nw->dld_ranges,RANGE_ENTRY,i));
						if((re!=NULL)&&(re->temp_local_fname!=NULL))
							g_string_free(re->temp_local_fname,TRUE);
					}
					g_array_free(nw->dld_ranges,TRUE);

					/* free autoscan entries */
					for(i=0;i<nw->autoscan->len;i++)
					{
						GDL_AS_ENTRY *re;

						re=&(g_array_index(nw->autoscan,GDL_AS_ENTRY,i));
						if((re!=NULL)&&(re->search_pattern!=NULL))
							g_free(re->search_pattern);
					}
					g_array_free(nw->autoscan,TRUE);
					free(nw);
				}
				else
				{	/* on success, end function */
					g_ptr_array_add(gdl_array,nw);
					update_gdl_cmd_of_this_gdle(nw);		/* reupdate the cmd */
					check_if_dl_complete(nw);
				}
			}
		}	
	}

	G_UNLOCK(gdl_array);
}

/***************************************************************************************************/
/* free data used by the content of the given RANGE_ENTRY. The RANGE_ENTRY itself is not destroyed */
/***************************************************************************************************/
static void free_range_entry(RANGE_ENTRY *re, int without_unlink)
{
	if(re->temp_local_fname!=NULL)
	{
		if(!without_unlink)
			unlink(re->temp_local_fname->str);
		g_string_free(re->temp_local_fname,TRUE);
		re->temp_local_fname=NULL;
	}
}

/************************/
/* end a running thread */
/************************/
static void terminate_xfer_gdldle(GDL_DL_ENTRY *gdldle)
{
	if(gdldle->is_started==1)
	{
		GString *nw;

		nw=g_string_new("");
		g_string_sprintf(nw,"/KILL %lu",gdldle->thread_id);
		add_new_sim_input(0,nw->str);
		g_string_free(nw,TRUE);
		gdldle->is_started=0;
	}
}

/****************************************************************************/
/* remove the nth DL_ENTRY on the given GDL_ENTRY and free allocated memory */
/* if a file download is in progress, the file is deleted.                  */
/****************************************************************************/
static void unlink_and_free_dl_entry(GDL_ENTRY *gdle, int gdl_links_index,int without_unlink)
{
	GDL_DL_ENTRY *gdldle;

	gdldle=g_ptr_array_remove_index_fast(gdle->gdl_links,gdl_links_index);
	if(gdldle!=NULL)
	{
		if(gdldle->nickname)
			g_string_free(gdldle->nickname,TRUE);
		if(gdldle->remote_fname)
			g_string_free(gdldle->remote_fname,TRUE);
		if(gdldle->temp_local_fname)
		{
			if(!without_unlink)
				unlink(gdldle->temp_local_fname->str);
			g_string_free(gdldle->temp_local_fname,TRUE);
		}
		free(gdldle);
	}
}


/**********************************************************/
/* detach a running GDL of this client                    */
/* there is only a minor difference between do_gdl_detach */
/* and do_gdl_end, only files are not destroyed           */
/**********************************************************/
int do_gdl_detach(unsigned int gdl_id)
{
	GDL_ENTRY *gdle;
	int idx;

	G_LOCK(gdl_array);
	idx=gdl_index(gdl_id);
	if(idx==-1)
	{
		G_UNLOCK(gdl_array);
		return -1;
	}

	gdle=g_ptr_array_index(gdl_array,idx);
	if(gdle->is_completed==1)
	{
		G_UNLOCK(gdl_array);
		disp_msg(ERR_MSG,"do_gdl_detach","This GDL is over, no further modification is possible",NULL);
		return -1;
	}

	gdle=g_ptr_array_remove_index_fast(gdl_array,idx);
	G_UNLOCK(gdl_array);

	/* destroy all gdldle but without unlink files */
	if(gdle->gdl_links!=NULL)
	{
		while(gdle->gdl_links->len!=0)
		{
			GDL_DL_ENTRY *nw;
			nw=g_ptr_array_index(gdle->gdl_links,0);

			terminate_xfer_gdldle(nw);									/* abort transfert */
		
			unlink_and_free_dl_entry(gdle,0,1);						/* and destroy this entry but don't free data */
		}
		g_ptr_array_free(gdle->gdl_links,TRUE);
	}

	/* free all ranges */
	if(gdle->dld_ranges!=NULL)
	{
		while(gdle->dld_ranges->len!=0)
		{
			RANGE_ENTRY *nw;
			nw=&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,0));
			free_range_entry(nw,1);										/* without unlink files */
			gdle->dld_ranges=g_array_remove_index_fast(gdle->dld_ranges,0);
		}
		g_array_free(gdle->dld_ranges,TRUE);
	}

	/* close autoscan socket fd */
	if(gdle->autoscan_sockfd!=-1)
	{
		shutdown(gdle->autoscan_sockfd,2);
		close(gdle->autoscan_sockfd);
	}

	/* free all autoscan entries */
	if(gdle->autoscan!=NULL)
	{
		int i;

		for(i=0;i<gdle->autoscan->len;i++)
		{
			gchar *sp;

			sp=g_array_index(gdle->autoscan,GDL_AS_ENTRY,i).search_pattern;

			if(sp!=NULL)
				free(sp);
		}

		g_array_free(gdle->autoscan,TRUE);
	}

	if(gdle->local_fname)
		g_string_free(gdle->local_fname,TRUE);

	if(gdle->lock_fd!=-1)
	{	/* release .lock file */
		lockf(gdle->lock_fd,F_ULOCK,1);
		close(gdle->lock_fd);
	}
		
	free(gdle);
}

/*********************************************/
/* add a new GDL_DL_ENTRY inside a GDL_ENTRY */
/*********************************************/
/* output: 0= success, !=0: error */
/**********************************/
int do_gdl_add(unsigned int gdl_id, char *nickname, char *remote_fname,unsigned long int remote_fsize)
{
	int ret=1;	/* default: error */
	int idx;

	G_LOCK(gdl_array);

	/* first, check if the given index is not already used */
	idx=gdl_index(gdl_id);
	if(idx!=-1)
	{	
		GDL_ENTRY *gdle;

		gdle=g_ptr_array_index(gdl_array,idx);
		if(gdle->is_completed==0)
		{
			idx=gdldle_index(gdle,nickname,remote_fname);
			if(idx==-1)
			{
				GDL_DL_ENTRY *nw;
	
				nw=malloc(sizeof(GDL_DL_ENTRY));
				if(nw!=NULL)
				{
					nw->nickname=g_string_new(nickname);
					nw->remote_fname=g_string_new(remote_fname);
					nw->remote_fsize=remote_fsize;
					nw->temp_local_fname=NULL;
					nw->last_start_time=0;
					nw->is_running=0;
					nw->is_started=0;

					g_ptr_array_add(gdle->gdl_links,nw);
					ret=0;
					update_gdl_cmd_of_this_gdle(gdle);
				}
			}
			else
			{	/* a gdldle still exists, just update its size */
				GDL_DL_ENTRY *nw;
				nw=g_ptr_array_index(gdle->gdl_links,idx);
				nw->remote_fsize=remote_fsize;
				ret=0;
				update_gdl_cmd_of_this_gdle(gdle);
			}
		}
		else
		{
			disp_msg(ERR_MSG,"do_gdl_add","This GDL is over, no further modification is possible",NULL);
		}
	}
	else
	{
		disp_msg(ERR_MSG,"do_gdl_add","No existing GDLs have this ID",NULL);
	}
	
	G_UNLOCK(gdl_array);

	return ret;
}

/********************************************/
/* delete a GDL_DL_ENTRY inside a GDL_ENTRY */
/********************************************/
/* output: 0= success, !=0: error */
/**********************************/
int do_gdl_del(unsigned int gdl_id, char *nickname, char *remote_fname)
{
	int ret=1;	/* default: error */
	int idx;

	G_LOCK(gdl_array);

	/* first, check if the given index exists */
	idx=gdl_index(gdl_id);
	if(idx!=-1)
	{	
		GDL_ENTRY *gdle;
		gdle=g_ptr_array_index(gdl_array,idx);

		if(gdle->is_completed==0)
		{
			idx=gdldle_index(gdle,nickname,remote_fname);
			if(idx!=-1)
			{	/* a gdldle still exists, just update its size */
				GDL_DL_ENTRY *nw;
				nw=g_ptr_array_index(gdle->gdl_links,idx);
	
				terminate_xfer_gdldle(nw);									/* abort transfert */
				
				unlink_and_free_dl_entry(gdle,idx,0);						/* and destroy this entry */
				ret=0;
				update_gdl_cmd_of_this_gdle(gdle);
			}
		}
		else
		{
			disp_msg(ERR_MSG,"do_gdl_del","This GDL is over, no further modification is possible",NULL);
		}
	}
	else
	{
		disp_msg(ERR_MSG,"do_gdl_del","No existing GDL has this ID",NULL);
	}
	
	G_UNLOCK(gdl_array);

	return ret;
}

/***********************************/
/* delete a GDL_ENTRY of gdl_array */
/********************************************************************************/
/* input: override must always be ==0. The only case where it is !=0 is when    */
/*        the GDL thread call this function after successfully gather all parts */
/*        of the downloaded files.                                              */
/********************************************************************************/
/* output: 0= success, !=0: error */
/**********************************/
int do_gdl_end(unsigned int gdl_id,int override)
{
	GDL_ENTRY *gdle;
	int idx;

	G_LOCK(gdl_array);
	idx=gdl_index(gdl_id);
	if(idx==-1)
	{
		G_UNLOCK(gdl_array);
		return -1;
	}

	/* remove the entry of the gdl_array and release the lock */
	/* NOTE: it is only possible to remove an entry if it is not completed */
	/*       or completed but using the override flag (GDL thread only)    */
	if(override==0)
	{
		gdle=g_ptr_array_index(gdl_array,idx);
		if(gdle->is_completed==1)
		{
			G_UNLOCK(gdl_array);
			disp_msg(ERR_MSG,"do_gdl_end","This GDL is over, no further modification is possible",NULL);
			return -1;
		}
	}
	gdle=g_ptr_array_remove_index_fast(gdl_array,idx);
	G_UNLOCK(gdl_array);

	/* destroy all gdldle */
	if(gdle->gdl_links!=NULL)
	{
		while(gdle->gdl_links->len!=0)
		{
			GDL_DL_ENTRY *nw;
			nw=g_ptr_array_index(gdle->gdl_links,0);

			terminate_xfer_gdldle(nw);									/* abort transfert */
		
			unlink_and_free_dl_entry(gdle,0,0);						/* and destroy this entry */
		}
		g_ptr_array_free(gdle->gdl_links,TRUE);
	}

	/* free all ranges */
	if(gdle->dld_ranges!=NULL)
	{
		while(gdle->dld_ranges->len!=0)
		{
			RANGE_ENTRY *nw;
			nw=&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,0));
			free_range_entry(nw,0);
			gdle->dld_ranges=g_array_remove_index_fast(gdle->dld_ranges,0);
		}
		g_array_free(gdle->dld_ranges,TRUE);
	}

	/* close autoscan socket fd */
	if(gdle->autoscan_sockfd!=-1)
	{
		shutdown(gdle->autoscan_sockfd,2);
		close(gdle->autoscan_sockfd);
	}

	/* free all autoscan entries */
	if(gdle->autoscan!=NULL)
	{
		int i;

		for(i=0;i<gdle->autoscan->len;i++)
		{
			gchar *sp;

			sp=g_array_index(gdle->autoscan,GDL_AS_ENTRY,i).search_pattern;

			if(sp!=NULL)
				free(sp);
		}

		g_array_free(gdle->autoscan,TRUE);
	}

	if(gdle->local_fname)
		g_string_free(gdle->local_fname,TRUE);

	if(gdle->lock_fd!=-1)
	{	/* release .lock file */
		lockf(gdle->lock_fd,F_ULOCK,1);
		close(gdle->lock_fd);
	}
		
	/* FIXME: GDL/fname is not destroyed in this function */
	free(gdle);

	return 0;
}

/***********************************************/
/* add a new autoscan pattern to the given GDL */
/***********************************************/
void do_gdl_as_add(unsigned int gdl_id, int filetype, char *pattern)
{
	int idx;

	G_LOCK(gdl_array);

	/* first, check if the given index is not already used */
	idx=gdl_index(gdl_id);
	if(idx!=-1)
	{	
		GDL_ENTRY *gdle;
		GDL_AS_ENTRY gae;
		GString *str;

		gdle=g_ptr_array_index(gdl_array,idx);
		
		str=g_string_new("");
		g_string_sprintf(str,"%d?%s",filetype,pattern);

		gae.last_scan=0;
		gae.gae_id=rand();		/* not safe but should be enough to avoid conflict */
		gae.search_pattern=g_strdup(str->str);

		g_string_free(str,TRUE);
		gdle->autoscan=g_array_append_val(gdle->autoscan,gae);

		update_gdl_cmd_of_this_gdle(gdle);		/* update the cmd */

		if(gdle->autoscan_sockfd==-1)
		{
			gdle->autoscan_sockfd=_x_udp(0);		/* create an UDP socket and found an unused port */
			if(gdle->autoscan_sockfd==-1)
			{
				disp_msg(ERR_MSG,"do_gdl_as_add","WARNING: unable to create UDP socket for autoscan of this GDL","|lu",(unsigned long)(gdle->gdl_id),NULL);
			}
			else
			{
				/* search query requires local port number to receive search result */
				struct sockaddr_in lcl;
				int len_lcl=sizeof(lcl);

				set_non_bloquant_sock(gdle->autoscan_sockfd);

				if(getsockname(gdle->autoscan_sockfd,(void*)&lcl,&len_lcl)!=0)
				{
					disp_msg(ERR_MSG,"do_gdl_as_add","WARNING: unable to create UDP socket for autoscan of this GDL","|lu",(unsigned long)(gdle->gdl_id),"error on getsockname",strerror(errno),NULL);
					close(gdle->autoscan_sockfd);
					gdle->autoscan_sockfd=-1;
				}
				else
				{
					gdle->autoscan_sock_port=ntohs(lcl.sin_port);
				}
			}
		}
	}
	else
	{
		disp_msg(ERR_MSG,"do_gdl_as_add","No existing GDLs have this ID",NULL);
	}
	
	G_UNLOCK(gdl_array);
	return;
}

/***********************************************/
/* remove an autoscan pattern to the given GDL */
/***********************************************/
void do_gdl_as_del(unsigned int gdl_id, unsigned long gae_id)
{
	int idx;

	G_LOCK(gdl_array);

	/* first, check if the given index is not already used */
	idx=gdl_index(gdl_id);
	if(idx!=-1)
	{	
		GDL_ENTRY *gdle;
		GDL_AS_ENTRY *gae;
		int i;
		int fnd=0;

		gdle=g_ptr_array_index(gdl_array,idx);
		
		for(i=0;i<gdle->autoscan->len;i++)
		{
			gae=&(g_array_index(gdle->autoscan,GDL_AS_ENTRY,i));
			if(gae->gae_id==gae_id)
			{
				fnd=1;

				if(gae->search_pattern!=NULL)
					g_free(gae->search_pattern);
				gdle->autoscan=g_array_remove_index_fast(gdle->autoscan,i);
				break;
			}
		}

		update_gdl_cmd_of_this_gdle(gdle);		/* update the cmd */

		if(!fnd)
		{
			disp_msg(ERR_MSG,"do_gdl_as_del","No autoscan of this GDL has this ID",NULL);
		}
	}
	else
	{
		disp_msg(ERR_MSG,"do_gdl_as_del","No existing GDLs have this ID",NULL);
	}
	
	G_UNLOCK(gdl_array);
	return;
}

/* -------------------------------------------------------------------------- */
/*************************************************************************/
/* the following functions are used by the standard download function to */
/* return status of transfert.                                           */
/*************************************************************************/

/**************************************/
/* check if the given GDL is complete */
/**************************************/
static void check_if_dl_complete(GDL_ENTRY *gdle)
{
	GArray *dl_ar;
	S_RNG *one;

	if(gdle==NULL)
		return;

	dl_ar=ranges_to_planar(gdle,1);
	if(dl_ar==NULL)
		return;			/* we may have an error later here because complete GDL are not gathered */

	if(dl_ar->len!=1)
	{
		g_array_free(dl_ar,TRUE);
		return;
	}

	/* we must have one bloc begining at 0 and ending at the end of the wanted size */
	one=&(g_array_index(dl_ar,S_RNG,0));
	if((one->b!=0)||(one->e!=gdle->total_size))
	{
		g_array_free(dl_ar,TRUE);
		return;
	}
	
	/* Yes ... we have everything, it is time to mix this produce one big file */
	gdle->is_completed=1;		/* NOTE: we can't gather file parts here because it can take a long time */
										/* and gdl_array is locked so all GDL xfers are waiting */

	printf("gdl_post_download_update done\n");
}


/**************************************************************************/
/* this function is called when a queued GDL download has a download slot */
/**************************************************************************/
/* output: 0= a range has been assigned */
/*        !=0 no range available        */
/******************************************************************/
/* if a range is assigned, *str is contains the $Get line to send */
/* and *lfile contains the name of the local filename             */
/******************************************************************/
int do_gdl_start(unsigned int gdl_id, char *nickname, pthread_t thread_id, GString **str,GString **lfile)
{
	int ret=-1;
	int i;
	int j;
	GDL_ENTRY *gdle=NULL;
	GDL_DL_ENTRY *gdldle;

	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
		for(i=0;i<gdl_array->len;i++)
		{
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle->gdl_id==gdl_id)
			{
				if(gdle->gdl_links!=NULL)
				{
					for(j=0;j<gdle->gdl_links->len;j++)
					{
						gdldle=g_ptr_array_index(gdle->gdl_links,j);

						disp_msg(DEBUG_MSG,"do_gdl_start","|lu",gdl_id,"|d",j,"|d",(int)(gdldle->is_started),NULL);
						if((gdldle->is_started==0)&&(gdldle->is_running==1))	/* we can only start a not yet started function */
																								/* we first try to find a xfer in the Trying status */
						{
							if(!strcmp(gdldle->nickname->str,nickname))
							{
								if(allocate_range(gdldle,gdle))
								{
									gdldle->is_started=1;
									gdldle->thread_id=thread_id;
									ret=0;

									*str=g_string_new("");
									g_string_sprintf(*str,"$Get %s$%lu",gdldle->remote_fname->str,gdldle->range[0]+1);
																						/* +1 because the 1st byte has the offset 1 instead of 0*/
									*lfile=g_string_new(gdldle->temp_local_fname->str);
									update_gdl_cmd_of_this_gdle(gdle);		/* reupdate the cmd */
									break;
								}
								else
								{
									gdldle->is_started=0;
									gdldle->is_running=0;
									gdldle->last_start_time=time(NULL)+360;	/* wait a lot longer */
								}
							}
						}
					}
				}
				break;
			}
		}

		if(ret==-1)
		{
			/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
			for(i=0;i<gdl_array->len;i++)
			{
				gdle=g_ptr_array_index(gdl_array,i);
				if(gdle->gdl_id==gdl_id)
				{
					if(gdle->gdl_links!=NULL)
					{
						for(j=0;j<gdle->gdl_links->len;j++)
						{
							gdldle=g_ptr_array_index(gdle->gdl_links,j);
	
							disp_msg(DEBUG_MSG,"do_gdl_start","|lu",gdl_id,"|d",j,"|d",(int)(gdldle->is_started),NULL);
							if(gdldle->is_started==0)		/* we can only start a not yet started function */
																	/* now, we can take even waiting xfer */
							{
								if(!strcmp(gdldle->nickname->str,nickname))
								{
									if(allocate_range(gdldle,gdle))
									{
										gdldle->is_running=1;			/* mark this transfer as started */
										gdldle->is_started=1;
										gdldle->thread_id=thread_id;
										ret=0;
		
										*str=g_string_new("");
										g_string_sprintf(*str,"$Get %s$%lu",gdldle->remote_fname->str,gdldle->range[0]+1);
																							/* +1 because the 1st byte has the offset 1 instead of 0*/
										*lfile=g_string_new(gdldle->temp_local_fname->str);
										update_gdl_cmd_of_this_gdle(gdle);		/* reupdate the cmd */
										break;
									}
									else
									{
										gdldle->is_started=0;
										gdldle->is_running=0;
										gdldle->last_start_time=time(NULL)+360;	/* wait a lot longuer */
									}
								}
							}
						}
					}
					break;
				}
			}
		}
	}
	G_UNLOCK(gdl_array);
	if(ret==0)
		disp_msg(DEBUG_MSG,"do_gdl_start","|lu",gdl_id,nickname,"|lu",(unsigned long)thread_id,(*str)->str,(*lfile)->str,NULL);
	else
	{
		disp_msg(DEBUG_MSG,"do_gdl_start","|lu",gdl_id,nickname,"|lu",(unsigned long)thread_id,"fail to find something to do",NULL);

		/* maybe everything is over */
		check_if_dl_complete(gdle);
	}

	return ret;
}

/******************************************************************************/
/* when a download ends, the GDL_ENTRY must be update using GDL_DL_ENTRY data */
/* to take into account downloaded data.                                      */
/******************************************************************************/
/* NOTE: this function must be called when gdldle->is_started==1 */
/*****************************************************************/
static void	gdl_post_download_update(GDL_ENTRY *gdle, GDL_DL_ENTRY *gdldle)
{
	RANGE_ENTRY nw;
	int i;

	if(gdldle->cur_dled==0)
		return;						/* nothing downloaded */

	nw.range[0]=gdldle->range[0];
	nw.range[1]=gdldle->range[0]+gdldle->cur_dled;
	nw.temp_local_fname=g_string_new(gdldle->temp_local_fname->str);
	
	if(gdle->dld_ranges->len==0)
	{
		gdle->dld_ranges=g_array_append_val(gdle->dld_ranges,nw);
	}
	else
	{	/* insert inside dld_ranges but keep it ordered */
		int inserted=0;

		for(i=0;i<gdle->dld_ranges->len;i++)
		{
			RANGE_ENTRY *p1;
			p1=&(g_array_index(gdle->dld_ranges,RANGE_ENTRY,i));

			if(nw.range[0]<p1->range[0])
			{
				gdle->dld_ranges=g_array_insert_val(gdle->dld_ranges,i,nw);
				inserted=1;
				break;
			}
		}

		if(!inserted)
		{
			gdle->dld_ranges=g_array_append_val(gdle->dld_ranges,nw);
		}
	}
	
	/* maybe we have the whole file now ? */
	check_if_dl_complete(gdle);
}

/********************************************************************/
/* this function is called when a GDL download fails                */
/* this works when a request has been made but has never started    */
/* this can occurs either if another xfer is still in progress with */
/* the given user or either because it never replies to request     */
/********************************************************************/
void do_gdl_abort(unsigned int gdl_id, char *nickname)
{
	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		int i;
		GDL_ENTRY *gdle;

		/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
		for(i=0;i<gdl_array->len;i++)
		{
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle->gdl_id==gdl_id)
			{
				if(gdle->gdl_links!=NULL)
				{
					int j;

					for(j=0;j<gdle->gdl_links->len;j++)
					{
						GDL_DL_ENTRY *gdldle;

						gdldle=g_ptr_array_index(gdle->gdl_links,j);

						if(gdldle->is_started==0)		/* we can only start a not yet started function */
						{
							if(!strcmp(gdldle->nickname->str,nickname))
							{
								/* ok, we have found GDL_DL_ENTRY */
								gdldle->is_running=0;
								gdldle->last_start_time=time(NULL);	/* to avoid immediat retry, adjust the start time */

								break;
							}
						}
					}
				}
				break;
			}
		}
	}
	G_UNLOCK(gdl_array);
}


/******************************************************************************/
/* this function is called when a GDL download fails but after a do_gdl_start */
/******************************************************************************/
/* if is_fatal is set, this GDL_DL_ENTRY is discarded */
/******************************************************/
void do_gdl_fail(unsigned int gdl_id, char *nickname, char *local_fname, int is_fatal)
{
	int fnd=0;

	disp_msg(DEBUG_MSG,"do_gdl_fail","|lu",gdl_id,nickname,local_fname,NULL);

	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		int i;
		GDL_ENTRY *gdle;

		/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
		for(i=0;i<gdl_array->len;i++)
		{
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle->gdl_id==gdl_id)
			{
				if(gdle->gdl_links!=NULL)
				{
					int j;

					for(j=0;j<gdle->gdl_links->len;j++)
					{
						GDL_DL_ENTRY *gdldle;

						gdldle=g_ptr_array_index(gdle->gdl_links,j);

						if(gdldle->is_started)
						{
							if( (!strcmp(gdldle->nickname->str,nickname)) &&
								 (!strcmp(gdldle->temp_local_fname->str,local_fname)) )
							{
								/* ok, we have found GDL_DL_ENTRY */
								fnd=1;

								/* the download has at least successfully started, some data may have been downloaded */
								gdl_post_download_update(gdle,gdldle);
								gdldle->is_started=0;
								g_string_free(gdldle->temp_local_fname,TRUE);
								gdldle->temp_local_fname=NULL;

								gdldle->is_running=0;
								gdldle->last_start_time=time(NULL);	/* to avoid immediat retry, adjust the start time */

								if(is_fatal)
									unlink_and_free_dl_entry(gdle,j,0);
								break;
							}
						}
					}
				}
				break;
			}
		}
	}
	G_UNLOCK(gdl_array);
	if(!fnd)
	{
		disp_msg(ERR_MSG,"do_gdl_fail","|lu",gdl_id,nickname,local_fname,NULL);
	}
}

/*****************************************************************/
/* this function is called when a GDL download successfully ends */
/* if the remote client is standard, there is no way to start another */
/* segment, this entry must released because the thread will die */
/*****************************************************************/
void do_gdl_success(unsigned int gdl_id, pthread_t ptid, int with_end)
{
	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		int i;
		GDL_ENTRY *gdle;

		/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
		for(i=0;i<gdl_array->len;i++)
		{
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle->gdl_id==gdl_id)
			{
				if(gdle->gdl_links!=NULL)
				{
					int j;

					for(j=0;j<gdle->gdl_links->len;j++)
					{
						GDL_DL_ENTRY *gdldle;

						gdldle=g_ptr_array_index(gdle->gdl_links,j);

						if((gdldle->is_started==1)&&(gdldle->thread_id==ptid))
						{
							/* ok, we have found GDL_DL_ENTRY */

							/* the download has at least successfully started, some data may have been downloaded */
							gdl_post_download_update(gdle,gdldle);
							g_string_free(gdldle->temp_local_fname,TRUE);
							gdldle->temp_local_fname=NULL;
							gdldle->is_started=0;
							if(with_end)
							{
								gdldle->is_running=0;
								gdldle->last_start_time=time(NULL);	/* to avoid immediat retry, adjust the start time */
							}
							break;
						}
					}
				}
				break;
			}
		}
	}
	G_UNLOCK(gdl_array);
}

/***********************************************************************************/
/* update the number of bytes downloaded since the beginning of the DL_ENTRY range */
/* at the same time, compute the number of bytes we can download (max: 8192 bytes) */
/* when nothing can be downloaded, return 0                                        */
/***********************************************************************************/
unsigned int gdl_get_amount(unsigned int gdl_id, pthread_t ptid, unsigned long total_pos)
{
	unsigned int ret=0;
	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		int i;
		GDL_ENTRY *gdle;

		/* search and update the GDL_DL_ENTRY of a GDL_ENTRY */
		for(i=0;i<gdl_array->len;i++)
		{
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle->gdl_id==gdl_id)
			{
				if(gdle->gdl_links!=NULL)
				{
					int j;

					for(j=0;j<gdle->gdl_links->len;j++)
					{
						GDL_DL_ENTRY *gdldle;

						gdldle=g_ptr_array_index(gdle->gdl_links,j);

						if((gdldle->is_started==1)&&(gdldle->thread_id==ptid))
						{
							GArray *dl_ar;

							/* ok, we have found GDL_DL_ENTRY */
							gdldle->cur_dled=total_pos;

							/* compute the number of bytes we still can download */
							dl_ar=ranges_to_planar(gdle,0);
							if(dl_ar!=NULL)
							{
								int k;
								unsigned long cur_pos=gdldle->range[0]+total_pos;

								k=0;
								while(k<dl_ar->len)
								{
									S_RNG *cur;
									cur=&(g_array_index(dl_ar,S_RNG,k));
									if(cur->e==cur_pos)
									{
										/* we have found our entry */
										/* 2 cases: k==(dl_ar->len-1) [we are the last segment or not] */
										if(k==dl_ar->len-1)
										{	/* we are the last segment */
											/* the biggest downloadable size is: gdldle->remote_fsize-cur_pos */
											ret=gdldle->remote_fsize-cur_pos;
										}
										else
										{
											/* not the last segment, so the biggest size is the number of bytes until the beginning of the next segment */
											S_RNG *nxt;
											nxt=&(g_array_index(dl_ar,S_RNG,k+1));	/* next segment */

											/* we take the block size but we also take into account our own filesize */
											ret=MIN(nxt->b,gdldle->remote_fsize)-cur_pos;
										}

										/* the biggest size is 8KB */
										if(ret>8192)
											ret=8192;
										break;
									}
									k++;
								}
								/* if we don't found ourselves, this means our block has reached the next one, so nothing can be download */
								g_array_free(dl_ar,TRUE);
							}
							break;
						}
					}
				}
				break;
			}
		}
	}
	G_UNLOCK(gdl_array);
	return ret;
}

/******************************/
/* display the GDL quick list */
/******************************/
void do_gdl_qlst(int sck)
{
	int i;

	disp_msg(GDL_QLST_BEGIN,"",NULL);
	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		for(i=0;i<gdl_array->len;i++)
		{
			GDL_ENTRY *gdle;
			
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle!=NULL)
			{
				disp_msg(GDL_QLST_ENTRY,"","|lu",(unsigned long)(gdle->gdl_id),
													gdle->local_fname->str,
													"|lu",(unsigned long)(gdle->total_size),
													NULL);
			}
		}
	}
	G_UNLOCK(gdl_array);
	disp_msg(GDL_QLST_END,"",NULL);
}

static GString *build_gdl_links_str(GPtrArray *gdl_dl_entries,unsigned long *dld_bytes)
{
	GString *nw;
	int i;

	nw=g_string_new("");
	if(gdl_dl_entries!=NULL)
	{
		for(i=0;i<gdl_dl_entries->len;i++)
		{
			GDL_DL_ENTRY *ptr;

			ptr=g_ptr_array_index(gdl_dl_entries,i);
			if(ptr!=NULL)
			{
				g_string_sprintfa(nw,"%s$%s$%lu$",ptr->nickname->str,ptr->remote_fname->str,ptr->remote_fsize);

				if(ptr->is_running==0)
				{
					nw=g_string_append_c(nw,'W');		/* WAITING */
				}
				else
				{
					if(ptr->is_started==0)
					{
						nw=g_string_append_c(nw,'T');		/* TRYING */
					}
					else
					{
						g_string_sprintfa(nw,"R%lu[%lu;%lu]",ptr->cur_dled,ptr->range[0],ptr->range[1]);
						(*dld_bytes)+=ptr->cur_dled;
					}
				}
			}
			nw=g_string_append_c(nw,'|');		/* SEPARATOR */
		}
	}
	return nw;
}

static GString *build_gdl_range_str(GArray *dld_ranges, unsigned long *dld_bytes)
{
	GString *nw;
	int i;

	nw=g_string_new("");
	if(dld_ranges!=NULL)
	{
		for(i=0;i<dld_ranges->len;i++)
		{
			RANGE_ENTRY *ptr;

			ptr=&(g_array_index(dld_ranges,RANGE_ENTRY,i));
			if(ptr!=NULL)
			{
				g_string_sprintfa(nw,"%s$[%lu;%lu]",ptr->temp_local_fname->str,ptr->range[0],ptr->range[1]);
				(*dld_bytes)+=(ptr->range[1]-ptr->range[0]);
			}
			nw=g_string_append_c(nw,'|');		/* SEPARATOR */
		}
	}
	return nw;
}

static GString *build_gdl_autoscan_str(GArray *autoscan)
{
	GString *nw;
	int i;

	nw=g_string_new("");
	if(autoscan!=NULL)
	{
		for(i=0;i<autoscan->len;i++)
		{
			GDL_AS_ENTRY *ptr;

			ptr=&(g_array_index(autoscan,GDL_AS_ENTRY,i));
			if(ptr!=NULL)
			{
				g_string_sprintfa(nw,"%lu$%s",ptr->gae_id,ptr->search_pattern);
			}
			nw=g_string_append_c(nw,'|');		/* SEPARATOR */
		}
	}
	return nw;
}

/*****************************/
/* display the GDL long list */
/*****************************/
void do_gdl_lst(int sck)
{
	int i;

	disp_msg(GDL_LST_BEGIN,"",NULL);
	G_LOCK(gdl_array);
	if(gdl_array!=NULL)
	{
		for(i=0;i<gdl_array->len;i++)
		{
			GDL_ENTRY *gdle;
			unsigned long dld_size=0;
			
			gdle=g_ptr_array_index(gdl_array,i);
			if(gdle!=NULL)
			{
				GString *links_str,*range_str, *ascan_str;
				links_str=build_gdl_links_str(gdle->gdl_links,&dld_size);
				range_str=build_gdl_range_str(gdle->dld_ranges,&dld_size);
				ascan_str=build_gdl_autoscan_str(gdle->autoscan);

				disp_msg(GDL_LST_ENTRY,"","|lu",(unsigned long)(gdle->gdl_id),
													gdle->local_fname->str,
													"|lu",(unsigned long)(gdle->total_size),
													"|lu",(unsigned long)(gdle->dl_offset),		/* byte here before attachement */
													"|lu",(unsigned long)(dld_size),					/* amount of bytes still here */
													"|lu",(unsigned long)(gdle->start_time),		/* creation/attachement time */
													links_str->str,
													range_str->str,
													ascan_str->str,
													NULL);

				g_string_free(links_str,TRUE);
				g_string_free(range_str,TRUE);
				g_string_free(ascan_str,TRUE);
			}
		}
	}
	G_UNLOCK(gdl_array);
	disp_msg(GDL_LST_END,"",NULL);
}

