#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <string.h>

/* Added for printer(), using isprint() */
#include <ctype.h>

#include <errno.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "dctc_process.h"
#include "main.h"
#include "misc_gtk.h"
#include "do_connect.h"
#include "init_fnc.h"
#include "str_array.h"
#include "tos_key.h"
#include "timed_out_string.h"
#include "gui_define.h"
#include "search.xpm"

GPtrArray *op_array=NULL;

/******************************************************************************/
/* count the number of entry inside the given gchar * array (NULL terminated) */
/******************************************************************************/
int gchar_array_len(gchar **array)
{
	int nb=0;
	if (array==NULL)
		return 0;

	while(array[nb]!=NULL)
		nb++;

	return nb;
}


/*************************************************************/
/* close the connection to the current DCTC client if exists */
/*************************************************************/
void close_current_dctc(void)
{
	if(current_dctc==NULL)
		return;

	close_and_free_dctc_com(current_dctc);
	current_dctc=NULL;
}

/*********************************************************/
/* check if the given nickname appears in the user_clist */
/*********************************************************/
/* output: -1=no else position in the clist */
/********************************************/
static int user_here(char *nickname)
{
	GtkWidget *w;
	GtkCList *clst;
	int i;

	w=get_widget_by_widget_name("user_clist");
	if(w==NULL)
		return -1;

	clst=GTK_CLIST(w);

	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);

		if((t!=NULL)&&(!strcmp(t,nickname)))
		{
			return i;
		}
	}
	return -1;
}


/***************************************************/
/* delete the array containing the nick of all ops */
/***************************************************/
void clear_op_array(void)
{
	str_array_destroy(op_array);
	op_array=NULL;
}

/********************************/
/* print a line to stdout       */
/* filter out unprintable chars */
/* adds a newline to the end    */
/********************************/
static void printer(const GString *s)
{
	char *p = s->str;

	/* printf("DEBUG: printer()\n"); */

	while ( p[0] != '\0' ) {
		if ( isprint(p[0]) )
			printf("%c", p[0]);
		else
			printf(" ");
		
		p++;
	}
	printf("\n");

	/* printf("DEBUG: printer() END\n"); */
}

/************************/
/* get a line from DCTC */
/************************/
static GString *get_dctc_line(void)
{
	GString *s;
	int n;
	char c;

	if(current_dctc==NULL)
		return NULL;

	s=g_string_sized_new(512);

	do
	{
		n=read(current_dctc->dctc_fd,&c,1);
		if(n!=1)
		{
			/* Using printer() instead
			   printf("%s\n",s->str); 
			*/
			printer(s);

			g_string_free(s,TRUE);
			close_current_dctc();
			return NULL;
		}

		s=g_string_append_c(s,c);
	}while(c!='\n');

	return s;
}

static void dummy_fnc(const GString *s)
{
	/* Using printer() instead
	    printf("--> %s\n",s->str);
	 */

	GString *s2 = g_string_new("");
	g_string_sprintf(s2, "--> %s", s->str);
	printer(s2);    
	g_string_free(s2,TRUE);
}

/*******************************************************************/
/* this function adds the nickname to the user_clist or updates it */
/*******************************************************************/
static void add_or_update_user(char *cnx_type, char *nickname, char *size, char *email, char *description, unsigned int flag)
{
	GtkWidget *w;
	GtkCList *clst;
	int fnd=-1;
	int i;
	int row_num;

	w=get_widget_by_widget_name("user_clist");
	if(w==NULL)
		return;

	clst=GTK_CLIST(w);

	gtk_clist_freeze(clst);
	
	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);

		if((t!=NULL)&&(!strcmp(t,nickname)))
		{
			fnd=i;
			break;
		}
	}

	if(fnd==-1)
	{
		gchar *nw[5];
		nw[0]=cnx_type;
		nw[1]=nickname;
		nw[2]=size;
		nw[3]=email;
		nw[4]=description;

		row_num=gtk_clist_append(clst,nw);
	}
	else
	{
		gtk_clist_set_text(clst,fnd,0,cnx_type);
		gtk_clist_set_text(clst,fnd,1,nickname);
		gtk_clist_set_text(clst,fnd,2,size);
		gtk_clist_set_text(clst,fnd,3,email);
		gtk_clist_set_text(clst,fnd,4,description);
		row_num=fnd;
	}

	if(flag&2)
	{
		/* user is away */
		gtk_clist_set_background(clst,row_num,&light_grey);
	}
	else
	{
		/* user is here */
		gtk_clist_set_background(clst,row_num,&white);
	}

	if(flag&8)
	{
		/* fast xfer (>100KB/s) */
               
		/*Using printer() instead
		  printf("%s is fast\n",nickname);
		GString *s2 = g_string_new("");
		g_string_sprintf(s2, "%s is fast", nickname);
		printer(s2);    
		g_string_free(s2,TRUE);
		*/
		gtk_clist_set_cell_style(clst,row_num,0,fast_style);
	}
	gtk_clist_sort(clst);
	gtk_clist_thaw(clst);
}

static void colorize_queued_xfer_list(const char *nickname, GdkColor *bg_color)
{
	GtkWidget *w;
	GtkCList *clst;
	int i;

	w=get_widget_by_widget_name("queue_clist");
	if(w==NULL)
		return;

	clst=GTK_CLIST(w);

	gtk_clist_freeze(clst);
	
	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);

		if((t!=NULL)&&(!strcmp(t,nickname)))
		{
			gtk_clist_set_background(clst,i,bg_color);
		}
	}

	gtk_clist_thaw(clst);
}

typedef struct
{
	GdkColor *bg_color;
	const char *nickname;
} NICK_COLOR;

static void colorize_gdl_ctree_entry(GtkCTree *ctree, GtkCTreeNode *node, gpointer data)
{
	char *stored_nick;

	if(GTK_CTREE_ROW(node)->level!=2)
		return;

	if(gtk_ctree_node_get_pixtext(ctree,node,0,&stored_nick,NULL,NULL,NULL))
	{
		NICK_COLOR *nc=data;
		if((stored_nick!=NULL)&&(!strcmp(stored_nick,nc->nickname)))
		{
			gtk_ctree_node_set_background (ctree,node,nc->bg_color);
		}
	}
}


static void colorize_gdl_ctree_list(const char *nickname, GdkColor *bg_color)
{
	GtkWidget *w;
	NICK_COLOR nc;

	w=get_widget_by_widget_name("gdl_ctree");
	if(w==NULL)
		return;

	nc.bg_color=bg_color;
	nc.nickname=nickname;
	gtk_ctree_pre_recursive_to_depth(GTK_CTREE(w),NULL,2,colorize_gdl_ctree_entry,&nc);
}


/****************************************************************/
/* the following function processes "USER ]" and "USER+]" lines */
/****************************************************************/
static void user__fnc(const GString *s)
{
	char *t;
	char *nick;

	/* "USER ]" and "USER+]" work in the same manner */
	/* we check if the given user exists. If not, he is added to "user_clist" */
	/* s format: USER ] "xxxxx"nickname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t='\0';

	add_or_update_user("",nick,"","","",0);

	if(!strncmp(s->str,"USER ]",6))
	{
		GString *ui;

		ui=g_string_new("/UINFO ");
		g_string_sprintfa(ui,"%s\n",nick);
		send_data_to_dctc(ui->str);
		g_string_free(ui,TRUE);
	}

	colorize_queued_xfer_list(nick,&green);
	colorize_gdl_ctree_list(nick,&green);
}

/*****************************************************************/
/* remove the user having the given nickname from the user_clist */
/*****************************************************************/
static void remove_user(const char *nickname)
{
	GtkWidget *w;
	GtkCList *clst;
	int i;

	w=get_widget_by_widget_name("user_clist");
	if(w==NULL)
		return;

	clst=GTK_CLIST(w);

	gtk_clist_freeze(clst);
	
	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);

		if((t!=NULL)&&(!strcmp(t,nickname)))
		{
			gtk_clist_remove(clst,i);
			break;
		}
	}

	gtk_clist_thaw(clst);

	colorize_queued_xfer_list(nickname,&white);
	colorize_gdl_ctree_list(nickname,&white);
}

static void update_users_info(void)
{
	GtkWidget *w;
	GtkCList *clst;
	int i;
	char buf[5120];
	double amount=0;

	w=get_widget_by_widget_name("user_clist");
	if(w==NULL)
		return;

	clst=GTK_CLIST(w);

	for(i=0;i<clst->rows;i++)
	{
		char *t;
		double val;
		char unit;
		
		/* get size */
		gtk_clist_get_text(clst,i,2,&t);

		if(sscanf(t,"%lf%c",&val,&unit)==2)
		{
			switch(unit)
			{
				case 'G':	val*=1024.0*1024.0*1024.0;
								break;
				case 'M':	val*=1024.0*1024.0;
								break;
				case 'K':	val*=1024.0;
								break;
			}
			amount+=val;
		}
	}

	if( (amount/(1024.0*1024.0*1024.0)) > 1024.0)
		sprintf(buf,"%d users (%.2lfTB)",clst->rows,amount/(1024.0*1024.0*1024.0*1024.0));
	else
		sprintf(buf,"%d users (%.2lfGB)",clst->rows,amount/(1024.0*1024.0*1024.0));
	w=get_widget_by_widget_name("users_info_label");
	if(w==NULL)
		return;
	gtk_label_set(GTK_LABEL(w),buf);
}

/***************************************************/
/* the following function processes "USER-]" lines */
/***************************************************/
static void user_minus_fnc(const GString *s)
{
	char *t;
	char *nick;

	/* "USER ]" and "USER+]" work in the same manner */
	/* we check if the given user exists. If not, he is added to "user_clist" */
	/* s format: USER ] "xxxxx"nickname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* remove this user from the op array (if exist) */
	op_array=str_array_del(op_array,nick);

	remove_user(nick);
	update_users_info();
}

/***************************************************/
/* the following function processes "OP   ]" lines */
/***************************************************/
static void op_fnc(const GString *s)
{
	char *t;
	char *nick;

	/* s format: OP   ] "xxxxx"nickname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* add this operator to the op array */
	op_array=str_array_add(op_array,nick);

	{
		GtkWidget *w;
		GtkCList *clst;
		int i;

		w=get_widget_by_widget_name("user_clist");
		if(w==NULL)
			return;

		clst=GTK_CLIST(w);

		gtk_clist_freeze(clst);
	
		for(i=0;i<clst->rows;i++)
		{
			char *t;
		
			gtk_clist_get_text(clst,i,1,&t);

			if((t!=NULL)&&(!strcmp(t,nick)))
			{
				gtk_clist_set_foreground(clst,i,&light_red);
				break;
			}
		}
	
		gtk_clist_thaw(clst);
	}
}

/***************************************************/
/* the following function processes "UINFO]" lines */
/***************************************************/
static void uinfo_fnc(const GString *s)
{
	char *t;
	char *nick;
	char *size;
	char *cnx;
	char *mail;
	char *description;
	GString *fsize;
	double vsize;
	unsigned int flag;

	/* s format: UINFO] "xxxxx"nickname size cnx [(flag)misc] 'mail' description| */
	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,' ');
	if(t==NULL)
		return;
	*t++='\0';

	size=t;
	t=strchr(size,' ');
	if(t==NULL)
		return;
	*t++='\0';

	cnx=t;
	t=strchr(cnx,' ');
	if(t==NULL)
		return;
	*t++='\0';

	flag=strtoul(t+2,NULL,10);	/* +2: we skip the [ and the ( */

	t=strchr(t,'\'');
	if(t==NULL)
		return;
	t++;
	mail=t;
	t=strchr(mail,'\'');
	if(t==NULL)
		return;
	*t++='\0';

	t++;
	description=t;
	t=strchr(description,'|');
	if(t==NULL)
		return;
	*t='\0';

	vsize=strtod(size,NULL);
	fsize=g_string_new("");
	
	if(vsize>(1024.0*1024.0*1024.0))
		g_string_sprintf(fsize,"%.2fGB",vsize/(1024.0*1024.0*1024.0));
	else if(vsize>(1024.0*1024.0))
		g_string_sprintf(fsize,"%.2fMB",vsize/(1024.0*1024.0));
	else if(vsize>(1024.0))
		g_string_sprintf(fsize,"%.2fKB",vsize/(1024.0));
	else
		g_string_sprintf(fsize,"%.2fB",vsize);
	add_or_update_user(cnx,nick,fsize->str,mail,description, flag);

	g_string_free(fsize,TRUE);

	update_users_info();
}

/***************************************************/
/* the following function processes "CHAT ]" lines */
/***************************************************/
static void chat_fnc(const GString *s)
{
	char *t;
	char *msg;

	/* s format: CHAT ] "xxxxx"message| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	msg=t;
	t=strchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace \r by \n */
	t=msg;
	while(*t!='\0')
	{
		if(*t=='\r')
		{
			if(t[1]=='\r')
				*t++=' ';

			*t='\n';
		}
		t++;
	}

	{
		GtkWidget *w;
		char *eonick;

		w=get_widget_by_widget_name("chat_output");
		if(w==NULL)
			return;

		eonick=strstr(msg,"> ");
		gtk_text_freeze(GTK_TEXT(w));
		if((msg[0]!='<')||(eonick==NULL))
		{
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,msg,-1);
		}
		else
		{
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,&green,msg,eonick+1-msg);
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,eonick+1,-1);
		}
		gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,"\n",-1);
		gtk_text_thaw(GTK_TEXT(w));

		/* keep the window at the bottom place */
		{
			GtkAdjustment *v;
			float t;
			v=GTK_TEXT(w)->vadj;

			t=v->upper-v->page_size;
			gtk_adjustment_set_value(v,t>=0.0?t:0.0);
		}
	}
}

/***************************************************/
/* the following function processes "HUBNM]" lines */
/***************************************************/
static void hubnm_fnc(const GString *s)
{
	char *t;
	char *msg;

	/* s format: HUBNM] "xxxxx"hubname| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	msg=t;
	t=strchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	{
		GtkWidget *w;

		w=get_widget_by_widget_name("xfer_notebook");
		if(w==NULL)
		{
			printf("no widget app1\n");
			return;
		}
		w=gtk_widget_get_toplevel(w);
		/* for an unknown reason, get_widget_by_widget_name doesn't work for "app1" */

		gtk_window_set_title(GTK_WINDOW(w),msg);
	}

	/* display a message in the chat to notify hub change */
	{
		GString *str;

		str=g_string_new("CHAT ] \"\">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\r"
										    "entering hub ");
		g_string_sprintfa(str,  "%s\r<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|",msg);

		chat_fnc(str);
		g_string_free(str,TRUE);
	}
}

/***************************************************/
/* the following function processes "ERR  ]" lines */
/***************************************************/
static void err_fnc(const GString *s)
{
	char *t;
	char *fnc;
	char *msg;
	GString *out;

	/* s format: ERR  ] "xxxxx"errmsg| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	fnc=t;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	msg=t;
	t=strrchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace | by \n */
	t=msg;
	while(*t!='\0')
	{
		if(*t=='|')
			*t='\n';
		t++;
	}

	if(strstr(msg,"use /QUIT to quit")==NULL)
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name("disperrorcheckbutton")))==TRUE)
		{	/* don't display the "use /QUIT to quit" message */
			out=g_string_new("");
			g_string_sprintfa(out,"\n%s\n",msg);

			gnome_app_error(GNOME_APP(main_window),out->str);
			g_string_free(out,TRUE);
		}

		{
			GtkWidget *w;
			struct tm tm;
			time_t tt;
			char bbb[512];

			w=get_widget_by_widget_name("error_messages_text");
			if(w==NULL)
				return;

			tt=time(NULL);
			localtime_r(&tt,&tm);
			strftime(bbb,sizeof(bbb),"%c: ",&tm);

			gtk_text_freeze(GTK_TEXT(w));
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,bbb,-1);
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,msg,-1);
			gtk_text_insert(GTK_TEXT(w),NULL,NULL,NULL,"\n",-1);
			gtk_text_thaw(GTK_TEXT(w));

			/* keep the window at the bottom place */
			{
				GtkAdjustment *v;
				float t;
				v=GTK_TEXT(w)->vadj;
	
				t=v->upper-v->page_size;
				gtk_adjustment_set_value(v,t>=0.0?t:0.0);
			}
		}	
	}
}

/***************************************************/
/* the following function processes "SREST]" lines */
/***************************************************/
static void srest_fnc(const GString *s)
{
	char *t;
	char *nick;
	char *fname;
	char *size;
	char *dl_ratio;
	char *hubname;
	char *hubip;
	char *localfname;
	GString *hub;

	/* s format: SREST] "xxxxx"nickname|file|size|ratio|hubname|hubip| */
	/* (result of SEARCH and MSEARCH) */
	/* or */
	/* s format: SREST] "xxxxx"nickname|file|size|ratio|hubname|hubip|localfname| */
	/* (result of CSEARCH) */


	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';

	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t++='\0';

	size=t;
	t=strchr(size,'|');
	if(t==NULL)
		return;
	*t++='\0';

	dl_ratio=t;
	t=strchr(dl_ratio,'|');
	if(t==NULL)
		return;
	*t++='\0';

	hubname=t;
	t=strchr(hubname,'|');
	if(t==NULL)
		return;
	*t++='\0';

	hubip=t;
	t=strchr(hubip,'|');
	if(t==NULL)
		return;
	*t++='\0';

	localfname=t;
	t=strchr(localfname,'|');
	if(t==NULL)
		localfname="";
	else
		*t++='\0';

	hub=g_string_new("");
	
	g_string_sprintf(hub,"%s (%s)",hubname,hubip);
	{
		GtkWidget *w;
		GtkCList *clst;

		w=get_widget_by_widget_name("find_result");
		if(w!=NULL)
		{

			clst=GTK_CLIST(w);

			gtk_clist_freeze(clst);
	
			{
				int num_row;
				gchar *nw[6];
				nw[0]=nick;
				nw[1]=fname;
				nw[2]=size;
				nw[3]=dl_ratio;
				nw[4]=hub->str;
				nw[5]=localfname;

				num_row=gtk_clist_append(clst,nw);

				/* put operator result in red */
				if(str_array_is_inside(op_array,nw[0]))
				{
					gtk_clist_set_foreground(clst,num_row,&light_red);
				}
			}
			gtk_clist_sort(clst);
			gtk_clist_thaw(clst);
		}
	}
	g_string_free(hub,TRUE);
}

#if 0
/***************************************************/
/* the following function processes "$UL+ ]" lines */
/***************************************************/
static void add_ul_xfer_fnc(const GString *s)
{
	char *nw[4];
	GtkWidget *w;

	char *t;
	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nw[0]=t;
	t=strchr(nw[0],'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	nw[3]=t;
	t=strchr(nw[3],'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("upload_clist");
	if(w==NULL)
		return;

	nw[1]="";
	nw[2]="";
	gtk_clist_append(GTK_CLIST(w),nw);
}

/***************************************************/
/* the following function processes "$DL+ ]" lines */
/***************************************************/
static void add_dl_xfer_fnc(const GString *s)
{
	char *nw[5];
	GtkWidget *w;

	char *t;
	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nw[0]=t;
	t=strchr(nw[0],'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	nw[4]=t;
	t=strchr(nw[4],'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	nw[1]="trying";
	nw[2]="";
	nw[3]="";
	gtk_clist_append(GTK_CLIST(w),nw);
}

static int find_xfer_row(GtkWidget *w,char *nick, int col_fname, char *fname)
{
	GtkCList *clst=GTK_CLIST(w);
	int i;
	
	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,0,&t);
		if(!strcmp(t,nick))
		{
			gtk_clist_get_text(clst,i,col_fname,&t);
			if(!strcmp(t,fname))
				return i;
		}
	}
	
	return -1;
}

/***************************************************/
/* the following function processes "$DL~ ]" lines */
/***************************************************/
static void remove_dl_xfer_on_error_fnc(const GString *s)
{
	GtkWidget *w;

	char *t;
	char *nick;
	char *fname;
	int i;

	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	i=find_xfer_row(w,nick,4,fname);
	if(i!=-1)
	{
		gtk_clist_remove(GTK_CLIST(w),i);
	}
}

/***************************************************/
/* the following function processes "$DL- ]" lines */
/***************************************************/
static void remove_dl_xfer_on_success_fnc(const GString *s)
{
	GtkWidget *w;

	char *t;
	char *nick;
	char *fname;
	int i;

	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	i=find_xfer_row(w,nick,4,fname);
	if(i!=-1)
	{
		GtkWidget *w1;

		w1=get_widget_by_widget_name("done_clist");
		if(w1!=NULL)
		{
			char *nw[5];
			int j;

			for(j=1;j<6;j++)
			{
				gtk_clist_get_text(GTK_CLIST(w),i,j,&nw[j-1]);
			}
			gtk_clist_append(GTK_CLIST(w1),nw);
		}
		gtk_clist_remove(GTK_CLIST(w),i);
	}
}

/***************************************************/
/* the following function processes "$UL~ ]" lines */
/***************************************************/
static void remove_ul_xfer_on_error_fnc(const GString *s)
{
	GtkWidget *w;

	char *t;
	char *nick;
	char *fname;
	int i;

	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("upload_clist");
	if(w==NULL)
		return;

	i=find_xfer_row(w,nick,3,fname);
	if(i!=-1)
	{
		gtk_clist_remove(GTK_CLIST(w),i);
	}
}

static void dl_start_fnc(const GString *s)
{
	char *ptr[7];
	char *t;
	char *res;
	GtkWidget *w;

	/* s format: ] "xxxxx"nickname|remotefname|localfname|startpos|file_size|start_time|remoteip| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	ptr[0]=strtok_r(t,"|",&res);
	for(i=1;i<7;i++)
	{
		if(ptr[i-1]==NULL)
			return;
		ptr[i]=strtok_r(NULL,"|",&res);
	}
	if(ptr[6]==NULL)
		return;

	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	i=find_xfer_row(w,ptr[0],4,ptr[2]);
	if(i!=-1)
	{
		gtk_clist_set_text(GTK_CLIST(w),i,1,ptr[4]);
		gtk_clist_set_text(GTK_CLIST(w),i,3,ptr[1]);
	}
}

static void remove_xfer_from_queue_fnc(const GString *s)
{
	char *t;
	/* s format: ] "xxxxx"command */
	/* where command is either a /DL or /LS */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	if(!strncmp("/DL ",t,4))
	{
		char sep[2];
		char *array[3];
		sep[0]=*t++;
		sep[1]='\0';

		
	}
}
#else

static void update_lists(const GString *s)
{
	send_data_to_dctc("/XFER\n/GDLLST\n");
}

/****************************************************************/
/* free the given GPtrArray of gchar ** (freed with g_strfreev) */
/****************************************************************/
static void free_temp_array(GPtrArray **temp_array)
{
	int i;

	if(*temp_array==NULL)
		return;

	for(i=0;i<(*temp_array)->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index((*temp_array),i);
		if(k)
			g_strfreev(k);
	}
	g_ptr_array_free((*temp_array),TRUE);
	*temp_array=NULL;
}


static GPtrArray *xferr_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of XFERR) */
static GPtrArray *xferq_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of XFERQ) */
static GPtrArray *cmdkb_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of CMDKB) */

/******************************/
/* beginning of the xfer list */
/******************************/
static void begin_xfer_list_fnc(const GString *s)
{
	free_temp_array(&xferr_temp_array);
	xferr_temp_array=g_ptr_array_new();

	free_temp_array(&xferq_temp_array);
	xferq_temp_array=g_ptr_array_new();

	free_temp_array(&cmdkb_temp_array);
	cmdkb_temp_array=g_ptr_array_new();
}

/****************************************/
/* add an xferr entry to its temp array */
/****************************************/
static void add_xferr_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL)&&(fields[2]!=NULL))
	{
		g_ptr_array_add(xferr_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/****************************************/
/* add an xferq entry to its temp array */
/****************************************/
static void add_xferq_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL))
	{
		g_ptr_array_add(xferq_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/****************************************/
/* add an cmdkb entry to its temp array */
/****************************************/
static void add_cmdkb_fnc(const GString *in)
{
	gchar **fields;
	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	if((fields[0]!=NULL)&&(fields[1]!=NULL)&&(fields[2]!=NULL))
	{
		g_ptr_array_add(cmdkb_temp_array,fields);
	}
	else
		g_strfreev(fields);
}

/*====================================================================================================*/
/*= XFERR clist update =*/
/*======================*/
/***************************************************************/
/* update the given row of the given clist using ptr parameter */
/***************************************************************/
static void update_xferr_dl_row(GtkCList *clst,int num_row, gchar **entry)
{
	if(!strncmp(entry[2],"LS/",3))
	{
		gtk_clist_set_text(clst,num_row,2,"");
		gtk_clist_set_text(clst,num_row,3,"");
		gtk_clist_set_text(clst,num_row,4,"File list");
		gtk_clist_set_text(clst,num_row,5,"");
	}
	else if(!strncmp(entry[2],"DL/",3))
	{
		/* entry[2]+3== the remote filename (followed by a $ and a digit). */
		/* entry[3]=local filename */
		/* entry[4]=download start position */
		/* entry[5]=size of the complete file */
		/* entry[6]=download start time (obtained using time()). */
		/* entry[7]="the remote host name (the IP or the FQDN)":"the remote host port" */
		char *t;

		t=strrchr(entry[2]+3,'$');	/* remote start position of the remote filename */
		if(t!=NULL)
		{
			*t='\0';
		}

		gtk_clist_set_text(clst,num_row,2,entry[5]);
		gtk_clist_set_text(clst,num_row,3,"");
		gtk_clist_set_text(clst,num_row,4,entry[2]+3);
		gtk_clist_set_text(clst,num_row,5,entry[3]);

		update_dl_clist_size(clst,num_row,entry[5],get_var("dl_path"));
	}
}

/***************************************************************/
/* update the given row of the given clist using ptr parameter */
/***************************************************************/
static void update_xferr_ul_row(GtkCList *clst,int num_row, gchar **entry)
{
	int set=0;
	unsigned long val1;
	unsigned long *ptr_status;
	int size_status;

	gtk_clist_set_text(clst,num_row,4,entry[2]+3);

	/* try to restore the current transfer status */
	val1=strtoul(entry[0],NULL,10);
	if(get_tos_entry(ULST_TOSKEY,(char*)&val1,sizeof(unsigned long),0, (char**)&ptr_status, &size_status))
	{
		if(size_status==(5*sizeof(unsigned long)))
		{
			char stt[512];
				
			sprintf(stt,"%lu",ptr_status[2]);
			gtk_clist_set_text(clst,num_row,2,stt);
			sprintf(stt,"%lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 
														  100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
			gtk_clist_set_text(clst,num_row,3,stt);
			set=1;
		}
		if(ptr_status)
			free(ptr_status);
	}
	
	if(!set)
	{
		gtk_clist_set_text(clst,num_row,2,"");
		gtk_clist_set_text(clst,num_row,3,"");
	}
}

static void create_xferr_row(GtkCList *dl_clst, GtkCList *up_clst, gchar **entry)
{
	char *nw[6];
	int num_row;

	/* entry[0]= id */
	/* entry[1]= nickname */
	/* entry[2]= the command */
	/* entry[3...]= end with null array of command arguments */
	
	nw[0]=entry[0];
	nw[1]=entry[1];

	if(!strncmp(entry[2],"LS/",3))
	{
		nw[2]="";
		nw[3]="";
		nw[4]="File list";
		nw[5]="";
		num_row=gtk_clist_append(dl_clst,nw);

		/* put operator xfer in red */
		if(str_array_is_inside(op_array,nw[1]))
		{
			gtk_clist_set_foreground(dl_clst,num_row,&light_red);
		}
	}
	else if(!strncmp(entry[2],"UL/",3))
	{
		unsigned long val1;
		unsigned long *ptr_status;
		int size_status;

		nw[2]="";
		nw[3]="";
		nw[4]=entry[2]+3;
		num_row=gtk_clist_append(up_clst,nw);
		/* try to restore the current transfer status */
		val1=strtoul(nw[0],NULL,10);
		if(get_tos_entry(ULST_TOSKEY,(char*)&val1,sizeof(unsigned long),0, (char**)&ptr_status, &size_status))
		{
			if(size_status==(5*sizeof(unsigned long)))
			{
				char stt[512];
					
				sprintf(stt,"%lu",ptr_status[2]);
				gtk_clist_set_text(up_clst,num_row,2,stt);
				sprintf(stt,"%lu (%.2f%%)",ptr_status[1]+ptr_status[3] /* => current global position */, 
														  100*((float)(ptr_status[1]+ptr_status[3]))/(float)(ptr_status[2]));
				gtk_clist_set_text(up_clst,num_row,3,stt);
			}
			if(ptr_status)
				free(ptr_status);
		}

		/* put operator xfer in red */
		if(str_array_is_inside(op_array,nw[1]))
		{
			gtk_clist_set_foreground(up_clst,num_row,&light_red);
		}
	}
	else if(!strncmp(entry[2],"DL/",3))
	{
		/* entry[2]+3== the remote filename (followed by a $ and a digit). */
		/* entry[3]=local filename */
		/* entry[4]=download start position */
		/* entry[5]=size of the complete file */
		/* entry[6]=download start time (obtained using time()). */
		/* entry[7]="the remote host name (the IP or the FQDN)":"the remote host port" */
		char *t;
		int rw;
		unsigned long start_pos=strtoul(entry[4],NULL,10);
		time_t start_time=strtoul(entry[6],NULL,10);

		nw[2]=entry[5];		/* file full size */
		nw[3]="";				/* no speed */
		nw[4]=entry[2]+3;		/* remote filename */
		nw[5]=entry[3];		/* local filename */

		t=strrchr(nw[4],'$');
		if(t!=NULL)
		{
			*t='\0';
		}

		rw=gtk_clist_append(dl_clst,nw);

		/* put operator xfer in red */
		if(str_array_is_inside(op_array,nw[1]))
		{
			gtk_clist_set_foreground(dl_clst,rw,&light_red);
		}

		{
			DL_XTRA *xt;

			xt=malloc(sizeof(DL_XTRA));
			if(xt==NULL)
			{
				gtk_clist_set_row_data(dl_clst,rw,NULL);
			}
			else
			{
				xt->start_time=start_time;
				xt->start_pos=start_pos;
				gtk_clist_set_row_data_full(dl_clst,rw,xt,free);
			}
		}
		update_dl_clist_size(dl_clst,rw,nw[2],get_var("dl_path"));
	}
}

/*******************************************************************************/
/* scan download and upload clist, remove all running xfer which no more exist */
/* and update the matching one                                                 */
/*******************************************************************************/
static void clean_running_ul_and_dl_clist(GtkCList *dl_clst, GtkCList *up_clst)
{
	int i;
	int j;
	char *xfer_id;

	/* first, the download clist */
	for(i=dl_clst->rows-1;i>=0;i--)
	{
		if(gtk_clist_get_text(dl_clst,i,0,&xfer_id))
		{
			if(strlen(xfer_id))			/* xferq entries has an empty string as ID */
			{
				int found=0;
				for(j=0;j<xferr_temp_array->len;j++)
				{
					gchar **ptr;
	
					ptr=g_ptr_array_index(xferr_temp_array,j);
					if((ptr!=NULL)&&(ptr[0][0]!='\001'))
					{
						if(!strcmp(xfer_id,ptr[0]))
						{
							update_xferr_dl_row(dl_clst,i,ptr);
							ptr[0][0]='\001';
							found=1;
							break;
						}
					}
				}

				if(!found)
				{
					gtk_clist_remove(dl_clst,i);
				}
			}
		}
	}

	/* and the upload clist */
	for(i=up_clst->rows-1;i>=0;i--)
	{
		if(gtk_clist_get_text(up_clst,i,0,&xfer_id))
		{
			int found=0;

			for(j=0;j<xferr_temp_array->len;j++)
			{
				gchar **ptr;

				ptr=g_ptr_array_index(xferr_temp_array,j);
				if((ptr!=NULL)&&(ptr[0][0]!='\001'))
				{
					if(!strcmp(xfer_id,ptr[0]))
					{
						update_xferr_ul_row(up_clst,i,ptr);
						ptr[0][0]='\001';
						found=1;
						break;
					}
				}
			}

			if(!found)
			{
				gtk_clist_remove(up_clst,i);
			}
		}
	}
}

static void end_xfer_update_xferr(void)
{
	GtkWidget *dl_clst, *ul_clst;
	int i;

	dl_clst=get_widget_by_widget_name("download_clist");
	if(dl_clst==NULL)
		return;

	ul_clst=get_widget_by_widget_name("upload_clist");
	if(ul_clst==NULL)
		return;

	gtk_clist_freeze(GTK_CLIST(dl_clst));
	gtk_clist_freeze(GTK_CLIST(ul_clst));

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_running_ul_and_dl_clist(GTK_CLIST(dl_clst),GTK_CLIST(ul_clst));

	/* and add all missing entries */
	for(i=0;i<xferr_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(xferr_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				create_xferr_row(GTK_CLIST(dl_clst),GTK_CLIST(ul_clst),ptr);
			}
		}
	}
	
	gtk_clist_thaw(GTK_CLIST(dl_clst));
	gtk_clist_thaw(GTK_CLIST(ul_clst));

	/* free temp array buffer */
	free_temp_array(&xferr_temp_array);
}

/*====================================================================================================*/
/*= XFERQ clist update =*/
/*======================*/

static void update_xferq_row(GtkCList *clst,int num_row,gchar **ptr)
{
	if(!strncmp(ptr[1],"LS/",3))
	{
		gtk_clist_set_text(clst,num_row,3,"");
		gtk_clist_set_text(clst,num_row,4,"File list");
		gtk_clist_set_text(clst,num_row,5,"");
	}
	else if(!strncmp(ptr[1],"DL/",3))
	{
		/* ptr[2]=remote filename */
		/* ptr[3]=local filename */
		gtk_clist_set_text(clst,num_row,3,"");
		gtk_clist_set_text(clst,num_row,4,ptr[3]);		/* FIXME: inverted values ? */
		gtk_clist_set_text(clst,num_row,5,ptr[2]);
	}
	else
	{
		gtk_clist_set_text(clst,num_row,3,"?");
		gtk_clist_set_text(clst,num_row,4,"?");
		gtk_clist_set_text(clst,num_row,5,"?");
	}
}


static void clean_queued_dl_clist(GtkCList *clst)
{
	int i;
	int j;
	char *xfer_id;
	char *nickname;

	/* first, the download clist */
	for(i=clst->rows-1;i>=0;i--)
	{
		if(gtk_clist_get_text(clst,i,0,&xfer_id))
		{
			if(strlen(xfer_id)==0)			/* xferq entries has an empty string as ID */
			{
				if(gtk_clist_get_text(clst,i,1,&nickname))
				{
					int found=0;
					for(j=0;j<xferq_temp_array->len;j++)
					{
						gchar **ptr;
	
						ptr=g_ptr_array_index(xferq_temp_array,j);
						if((ptr!=NULL)&&(ptr[0][0]!='\001'))
						{
							if(!strcmp(nickname,ptr[0]))
							{
								update_xferq_row(clst,i,ptr);
								ptr[0][0]='\001';
								found=1;
								break;
							}
						}
					}

					if(!found)
					{
						gtk_clist_remove(clst,i);
					}
				}
			}
		}
	}
}

static void create_xferq_row(GtkCList *clst,gchar **ptr)
{
	char *nw[6];
	int num_row;

	nw[0]="";
	nw[1]=ptr[0];
	nw[2]="trying";
	nw[3]="";

	if(!strncmp(ptr[1],"LS/",3))
	{
		nw[4]="File list";
		nw[5]="";
	}
	else if(!strncmp(ptr[1],"DL/",3))
	{
		/* ptr[2]=remote filename */
		/* ptr[3]=local filename */
		
		nw[4]=ptr[3];	/* FIXME: inverted values ? */
		nw[5]=ptr[2];
	}
	else
	{
		nw[3]="?";
		nw[4]="?";
		nw[5]="?";
	}

	num_row=gtk_clist_append(clst,nw);

	/* put operator xfer in red */
	if(str_array_is_inside(op_array,nw[1]))
	{
		gtk_clist_set_foreground(clst,num_row,&light_red);
	}
}

static void end_xfer_update_xferq(void)
{
	GtkWidget *dl_clst;	/* queued xfers only appear in download clist */
	int i;

	dl_clst=get_widget_by_widget_name("download_clist");
	if(dl_clst==NULL)
		return;

	gtk_clist_freeze(GTK_CLIST(dl_clst));

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_queued_dl_clist(GTK_CLIST(dl_clst));

	/* and add all missing entries */
	for(i=0;i<xferq_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(xferq_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				create_xferq_row(GTK_CLIST(dl_clst),ptr);
			}
		}
	}
	
	gtk_clist_thaw(GTK_CLIST(dl_clst));

	/* free temp array buffer */
	free_temp_array(&xferq_temp_array);
}

/*====================================================================================================*/
/*= CMDKB clist update =*/
/*======================*/

/********************************************************************************************/
/* scan the queue_clist and remove all entries which are not inside cmdkb_temp_array        */
/* at the same time, mark ('\001') each cmdkb_temp_array entry which are found in the clist */
/********************************************************************************************/
static void clean_queue_clist(GtkCList *clst)
{
	int i;
	int j;
	char *cmdkb_id;

	for(i=clst->rows-1;i>=0;i--)
	{
		if(gtk_clist_get_text(clst,i,0,&cmdkb_id))
		{
			int found=0;

			for(j=0;j<cmdkb_temp_array->len;j++)
			{
				gchar **ptr;

				ptr=g_ptr_array_index(cmdkb_temp_array,j);
				if((ptr!=NULL)&&(ptr[0][0]!='\001'))
				{
					if(!strcmp(cmdkb_id,ptr[0]))
					{
						ptr[0][0]='\001';
						found=1;
						break;
					}
				}
			}

			if(!found)
			{
				gtk_clist_remove(clst,i);
			}
		}
	}
}

static void end_xfer_update_cmdkb(void)
{
	char *nw[5];
	GtkWidget *w;
	int i;
	int num_row;

	w=get_widget_by_widget_name("queue_clist");
	if(w==NULL)
		return;

	gtk_clist_freeze(GTK_CLIST(w));

	/* remove all entries of the queue_clist which are no more inside cmdkb_temp_array */
	clean_queue_clist(GTK_CLIST(w));

	/* and add all missing entries */
	for(i=0;i<cmdkb_temp_array->len;i++)
	{
		gchar **ptr;

		ptr=g_ptr_array_index(cmdkb_temp_array,i);
		if(ptr!=NULL)
		{
			if(ptr[0][0]!='\001')
			{	/* unseen entry, time to add it */
				if(!strncmp(ptr[2],"/LS ",4))		/* is the waiting command a /LS ? */
				{
					char *nw[5];

					nw[0]=ptr[0];			/* the waiting ID */
					nw[1]=ptr[2]+4;		/* the nickname */
					nw[2]="File list";
					nw[3]="";
					nw[4]="";
					num_row=gtk_clist_append(GTK_CLIST(w),nw);

					if(user_here(nw[1])!=-1)
						gtk_clist_set_background(GTK_CLIST(w),num_row,&green);

					/* put operator xfer in red */
					if(str_array_is_inside(op_array,nw[1]))
					{
						gtk_clist_set_foreground(GTK_CLIST(w),num_row,&light_red);
					}
				}
				else if(!strncmp(ptr[2],"/DL ",4))
				{
					gchar **tmp_split=NULL;
					int is_set=0;

					if(ptr[2][4]=='\0')		/* the used separator is a | parameters are inside ptr */
					{
						if((ptr[3]!=NULL)&&(ptr[4]!=NULL)&&(ptr[5]!=NULL)&&(ptr[6]!=NULL))	/* nickname,localname,remotename,filesize */
						{
							nw[0]=ptr[0];		/* the waiting ID */
							nw[1]=ptr[3];		/* the nickname */
							nw[2]=ptr[5];		/* the remote filename */
							nw[3]=ptr[6];		/* the size */
							nw[4]=ptr[4];		/* the localname */
							is_set=1;
						}
					}
					else
					{								/* the used parameter is not a | */
													/* thus, it is not yet splited */
						char sep[2];
						sep[0]=ptr[2][4];
						sep[1]='\0';

						tmp_split=g_strsplit(ptr[2],sep,0);
						if((ptr[1]!=NULL)&&(ptr[2]!=NULL)&&(ptr[3]!=NULL)&&(ptr[4]!=NULL))	/* nickname,localname,remotename,filesize */
						{
							nw[0]=ptr[0];		/* the waiting ID */
							nw[1]=ptr[1];		/* the nickname */
							nw[2]=ptr[3];		/* the remote filename */
							nw[3]=ptr[4];		/* the size */
							nw[4]=ptr[2];		/* the localname */
							is_set=1;
						}
					}

					if(is_set)
					{
						num_row=gtk_clist_append(GTK_CLIST(w),nw);
	
						if(user_here(nw[1])!=-1)
							gtk_clist_set_background(GTK_CLIST(w),num_row,&green);
	
						/* put operator xfer in red */
						if(str_array_is_inside(op_array,nw[1]))
						{
							gtk_clist_set_foreground(GTK_CLIST(w),num_row,&light_red);
						}
					}
	
					if(tmp_split!=NULL)
						g_strfreev(tmp_split);
				}
			}
		}
	}
	
	gtk_clist_thaw(GTK_CLIST(w));

	/* free temp array buffer */
	free_temp_array(&cmdkb_temp_array);
}

/*====================================================================================================*/
/****************************************************************/
/* all temp arrays are filled, now, it is time to do the update */
/****************************************************************/
static void end_xfer_list_fnc(const GString *s)
{
	/* update xferr list */
	end_xfer_update_xferr();

	/* update xferq list, easiest than previous one */
	end_xfer_update_xferq();

	/* and the easiest of all, the cmdkb array */
	end_xfer_update_cmdkb();
}

static void update_ul_list(const GString *in)
{
	unsigned long values[5];

	char *t;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	
	values[0]=strtoul(t,&t,10);	/* thread id */
	if((t==NULL)||(*t!=':'))
		return;
	t++;
	
	values[1]=strtoul(t,&t,10);	/* upload start position */
	if((t==NULL)||(*t!='/'))
		return;
	t++;
	
	values[2]=strtoul(t,&t,10);	/* file length */
	if((t==NULL)||(*t!='/'))
		return;
	t++;
	
	values[3]=strtoul(t,&t,10);	/* sent data */
	if((t==NULL)||(*t!='/'))
		return;
	t++;

	values[4]=strtoul(t,&t,10);	/* total to send */
	if((t==NULL)||(*t!='|'))
		return;
	
	/* keep a copy of the status so we can refresh the display at anytime */
	add_tos_entry_v1_uniq(ULST_TOSKEY,300,(char*)&values[0],sizeof(unsigned long), (char*)&values[0],5*sizeof(unsigned long));

#if 0
	printf("%lu %lu %lu %lu %lu\n",values[0],values[1],values[2],values[3],values[4]);
#endif
	{
		GtkCList *clst;
		GtkWidget *w;
		int i;

		/* cmd is "UL/ficname" */
		w=get_widget_by_widget_name("upload_clist");
		if(w==NULL)
			return;

		clst=GTK_CLIST(w);

		/* search and update the row on the clist */
		for(i=0;i<clst->rows;i++)
		{
			char *txt;

			gtk_clist_get_text(clst,i,0,&txt);
			if(txt!=NULL)
			{
				if(values[0]==strtoul(txt,NULL,10))
				{
					char stt[512];
					gtk_clist_freeze(clst);
					
					sprintf(stt,"%lu",values[2]);
					gtk_clist_set_text(clst,i,2,stt);
					sprintf(stt,"%lu (%.2f%%)",values[1]+values[3] /* => current global position */, 
														  100*((float)(values[1]+values[3]))/(float)(values[2]));
					gtk_clist_set_text(clst,i,3,stt);
					gtk_clist_thaw(clst);
					break;
				}
			}
		}
	}
}

#if 0
static void add_xferq_fnc(const GString *in)
{
	char *nw[6];
	GtkWidget *w;
	char *t;
	char *s;
	int num_row;

	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	nw[0]="";
	nw[1]=t;
	nw[2]="trying";
	nw[3]="";
	
	t=strchr(nw[1],'|');
	if(t==NULL)
		return;
	*t++='\0';

	if(!strncmp(t,"LS/",3))
	{
		nw[4]="File list";
		nw[5]="";
	}
	else if(!strncmp(t,"DL/",3))
	{
		char sep[2];

		sep[0]=t[3];
		sep[1]='\0';

		t+=4;
		nw[5]=strtok_r(t,sep,&s);
		if(nw[5]==NULL)
			return;
		nw[4]=strtok_r(NULL,sep,&s);
		if(nw[4]==NULL)
			return;
	}
	else
	{
		nw[3]="?";
		nw[4]="?";
		nw[5]="?";
	}

	num_row=gtk_clist_append(GTK_CLIST(w),nw);

	/* put operator xfer in red */
	if(str_array_is_inside(op_array,nw[1]))
	{
		gtk_clist_set_foreground(GTK_CLIST(w),num_row,&light_red);
	}
}
#endif

#if 0
/*********************************************************/
/* check if the given nickname appears in the user_clist */
/*********************************************************/
/* output: -1=no else position in the clist */
/********************************************/
static int user_here(char *nickname)
{
	GtkWidget *w;
	GtkCList *clst;
	int i;

	w=get_widget_by_widget_name("user_clist");
	if(w==NULL)
		return -1;

	clst=GTK_CLIST(w);

	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);

		if((t!=NULL)&&(!strcmp(t,nickname)))
		{
			return i;
		}
	}
	return -1;
}

static void add_cmdkb_fnc(const GString *in)
{
	char *nw[5];
	GtkWidget *w;
	char *t;
	char *id;
	int i;

	w=get_widget_by_widget_name("queue_clist");
	if(w==NULL)
		return;

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	id=t+1;
	t=strchr(id,'|');
	if(t==NULL)
		return;
	*t++='\0';
	t=strchr(t,'|');
	if(t==NULL)
		return;
	t++;
	gtk_clist_freeze(GTK_CLIST(w));

	if(!strncmp(t,"/LS ",4))
	{
		char *eot;

		eot=strchr(t+4,'|');
		if(eot!=NULL)
			*eot='\0';

		nw[0]=id;
		nw[1]=t+4;
		nw[2]="File list";
		nw[3]="";
		nw[4]="";
		i=gtk_clist_append(GTK_CLIST(w),nw);

		if(user_here(nw[1])!=-1)
			gtk_clist_set_background(GTK_CLIST(w),i,&green);

		/* put operator xfer in red */
		if(str_array_is_inside(op_array,nw[1]))
		{
			gtk_clist_set_foreground(GTK_CLIST(w),i,&light_red);
		}
	}
	else if(!strncmp(t,"/DL ",4))
	{
		char sep[2];

		sep[0]=t[4];
		sep[1]='\0';

		nw[0]=id;
		nw[1]=t+5;
		t=strchr(nw[1],sep[0]);
		if(t==NULL)
			return;
		*t++='\0';
		nw[4]=t;
		t=strchr(nw[4],sep[0]);
		if(t==NULL)
			return;
		*t++='\0';
		nw[2]=t;
		t=strchr(nw[2],sep[0]);
		if(t==NULL)
			return;
		*t++='\0';
		nw[3]="";

		i=gtk_clist_append(GTK_CLIST(w),nw);

		if(user_here(nw[1])!=-1)
			gtk_clist_set_background(GTK_CLIST(w),i,&green);

		/* put operator xfer in red */
		if(str_array_is_inside(op_array,nw[1]))
		{
			gtk_clist_set_foreground(GTK_CLIST(w),i,&light_red);
		}
	}
	gtk_clist_thaw(GTK_CLIST(w));
}
#endif

static int find_xfer_row(GtkWidget *w,char *nick, int col_fname, char *fname)
{
	GtkCList *clst=GTK_CLIST(w);
	int i;
	
	for(i=0;i<clst->rows;i++)
	{
		char *t;
		
		gtk_clist_get_text(clst,i,1,&t);
		if(!strcmp(t,nick))
		{
			gtk_clist_get_text(clst,i,col_fname,&t);
			if(!strcmp(t,fname))
				return i;
		}
	}
	
	return -1;
}

/***************************************************/
/* the following function processes "$DL- ]" lines */
/***************************************************/
static void remove_dl_xfer_on_success_fnc(const GString *s)
{
	GtkWidget *w;

	char *t;
	char *nick;
	char *fname;
	int i;

	/* s format: ] "xxxxx"nickname|localfile| */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	fname=t;
	t=strchr(fname,'|');
	if(t==NULL)
		return;
	*t='\0';
	
	w=get_widget_by_widget_name("download_clist");
	if(w==NULL)
		return;

	i=find_xfer_row(w,nick,5,fname);
	if(i!=-1)
	{
		GtkWidget *w1;

		w1=get_widget_by_widget_name("done_clist");
		if(w1!=NULL)
		{
			char *nw[5];
			int j;

			for(j=1;j<6;j++)
			{
				gtk_clist_get_text(GTK_CLIST(w),i,j,&nw[j-1]);
			}
			gtk_clist_append(GTK_CLIST(w1),nw);
		}
		gtk_clist_remove(GTK_CLIST(w),i);
	}
	update_lists(NULL);
}

#endif

/***************************************************/
/* the following function processes "VAR  ]" lines */
/***************************************************/
static void var_fnc(const GString *s)
{
	char *t;
	char *vname;
	char *vval;

	/* s format: ] "xxxxx"var_name|var_value| */
	/* var value may contain | */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	vname=t;
	t=strchr(vname,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	vval=t;
	t=strrchr(vval,'|');
	if(t==NULL)
		return;
	*t='\0';

	add_var(vname,vval);
	fix_pref_window();
}
	
/*******************************************************************/
/* return the GTK_TEXT widget of the chat having the LABEL == nick */
/* if nick doesn't have a private chat, the function creates one.  */
/*******************************************************************/
GtkWidget *get_pchat_text_widget_from_nick(char *nick)
{
	GtkWidget *rhcw;
	int empty=-1;
	int i;

	for(i=0;i<9;i++)
	{
		rhcw=get_widget_by_widget_name(lbl_chat[i]);
		if(rhcw)
		{
			char *t;

			gtk_label_get(GTK_LABEL(rhcw),&t);
			if(t!=NULL)
			{
				if(!strcmp(t,nick))
				{
					/* we have the entry */
					return get_widget_by_widget_name(chat_text[i]);
				}
				if((empty==-1)&&(!strcmp(t,"empty")))
				{
					empty=i;
				}
			}
		}
	}

	if(empty!=-1)
	{
		rhcw=get_widget_by_widget_name(lbl_chat[empty]);
		gtk_label_set(GTK_LABEL(rhcw),nick);
		return get_widget_by_widget_name(chat_text[empty]);
	}
	return NULL;
}

/***************************************************/
/* the following function processes "PRIV ]" lines */
/***************************************************/
static void pchat_fnc(const GString *s)
{
	char *t;
	char *nick;
	char *msg;
	GtkWidget *chat_text;

	/* s format: ] "xxxxx"var_name|var_value| */
	/* var value may contain | */

	t=strchr(s->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;
	nick=t;
	t=strchr(nick,'|');
	if(t==NULL)
		return;
	*t++='\0';
	
	msg=t;
	t=strrchr(msg,'|');
	if(t==NULL)
		return;
	*t='\0';

	/* replace \r by \n */
	t=msg;
	while(*t!='\0')
	{
		if(*t=='\r')
		{
			if(t[1]=='\r')
			{
				*t++=' ';
			}

			*t='\n';
		}
		t++;
	}

	chat_text=get_pchat_text_widget_from_nick(nick);
	if(chat_text!=NULL)
	{
		char *eonick;

		eonick=strstr(msg,"> ");
		gtk_text_freeze(GTK_TEXT(chat_text));
		if((msg[0]!='<')||(eonick==NULL))
		{
			gtk_text_insert(GTK_TEXT(chat_text),NULL,NULL,NULL,msg,-1);
		}
		else
		{
			gtk_text_insert(GTK_TEXT(chat_text),NULL,NULL,&green,msg,eonick+1-msg);
			gtk_text_insert(GTK_TEXT(chat_text),NULL,NULL,NULL,eonick+1,-1);
		}
		gtk_text_insert(GTK_TEXT(chat_text),NULL,NULL,NULL,"\n",-1);
		gtk_text_thaw(GTK_TEXT(chat_text));
		/* keep the window at the bottom place */
		{
			GtkAdjustment *v;
			float t;
			v=GTK_TEXT(chat_text)->vadj;

			t=v->upper-v->page_size;
			gtk_adjustment_set_value(v,t>=0.0?t:0.0);
		}

	}
	else
	{
		GString *out;

		out=g_string_new("");

		g_string_sprintfa(out,"WARNING: all private chats are busy, close unused one.\n"
									 "From %s\n%s\n",nick,msg);

		gnome_app_error(GNOME_APP(main_window),out->str);
		g_string_free(out,TRUE);
	}
}

static void flst_fnc(const GString *in)
{
	char *nw[3];
	GtkWidget *w;
	char *t;

	w=get_widget_by_widget_name("user_file_list_clist");
	if(w==NULL)
		return;

	/* s format: ] "xxxxx"nick|size|filename| */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	nw[0]=t+1;
	t=strchr(nw[0],'|');
	if(t==NULL)
		return;
	*t++='\0';

	nw[2]=t;
	t=strchr(nw[2],'|');
	if(t==NULL)
		return;
	*t++='\0';

	nw[1]=t;
	t=strchr(nw[1],'|');
	if(t==NULL)
		return;
	*t++='\0';

	gtk_clist_append(GTK_CLIST(w),nw);
}

/* this function is called when the user has validated its password */
/* string is a g_malloc'd string which should be freed, or NULL if the user cancelled. */
static void password_entered(gchar *string, gpointer data)
{
	if(string==NULL)
	{
		/* user has cancelled */
		send_data_to_dctc("/FORCEQUIT\n");
	}
	else
	{
		GString *cmd;

		cmd=g_string_new("/PASSWD ");
		g_string_sprintfa(cmd,"%s\n",string);
		send_data_to_dctc(cmd->str);
		g_string_free(cmd,TRUE);
		g_free(string);
	}
}

/************************************************/
/* open password dialog to enter a new password */
/************************************************/
void enter_passwd_fnc(const GString *in)
{
	/* see gnome_request_dialog for information */
	gnome_app_request_password(GNOME_APP(main_window),_("Your nickname is password protected on this hub\nPlease enter your password."),
																	  password_entered,NULL);
}

/*************************************/
/* decode and display a progress bar */
/*************************************/
void prog_bar_fnc(const GString *in)
{
	char *t;
	gchar **splt;
	/* s format: ] "xxxxx"barname|progress|string|[string|...] */

	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;

	splt=g_strsplit(t+1,"|",0);
	if(splt!=NULL)
	{
		if((splt[0]!=NULL)&&(splt[1]!=NULL)&&(splt[2]!=NULL))
		{	/* at least 3 fields */
			float v;
			int pb_len;
			char *pb=NULL;
			GtkWidget *nw[3];		/* a progress bar is 2 widgets: the window containing the progressbar, the progressbar itself, the label containing the text */

			v=atof(splt[1]);
			if(v!=100.0)
			{	/* it is not a bar destruction */
				if(get_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0,&pb,&pb_len))
				{
					/* the bar still exists */
					if(pb!=NULL)
					{
						/* and seems to be valid */
						nw[0]=((GtkWidget **)pb)[0];
						nw[1]=((GtkWidget **)pb)[1];
						nw[2]=((GtkWidget **)pb)[2];
						goto set_field;
					}
				}
				else
				{
					/* the bar does not exist */
					{
						nw[0]=gtk_window_new(GTK_WINDOW_DIALOG);
						gtk_container_border_width (GTK_CONTAINER (nw[0]), 5);
						{
							GtkWidget *vbox;
							vbox=gtk_vbox_new(FALSE,0);
							gtk_container_add(GTK_CONTAINER(nw[0]),vbox);
							gtk_widget_show(vbox);
							{
								nw[2]=gtk_label_new(NULL);
								gtk_box_pack_start(GTK_BOX(vbox),nw[2],TRUE,TRUE,0);
								gtk_widget_show(nw[2]);
							}
							{
								nw[1]=gtk_progress_bar_new();
								gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(nw[1]),GTK_PROGRESS_CONTINUOUS);
								gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(nw[1]),GTK_PROGRESS_LEFT_TO_RIGHT);
								gtk_box_pack_start(GTK_BOX(vbox),nw[1],TRUE,TRUE,0);
								gtk_widget_show(nw[1]);
							}
						}
						gtk_widget_show(nw[0]);
	
						add_tos_entry(PGBAR_TOSKEY,1000000000,splt[0],strlen(splt[0]),(void*)&nw,sizeof(nw));
					}

					set_field:
					{
						GString *grp;
						int i;

						/* concat all strings */
						grp=g_string_new(splt[2]);
						i=3;
						while(splt[i]!=NULL)
						{
							grp=g_string_append_c(grp,'\n');
							grp=g_string_append(grp,splt[i]);
							i++;
						}

						gtk_progress_set_percentage(GTK_PROGRESS(nw[1]), v/100.0);		/* the value should be between 0 and 1 and we have it between 0 and 100 */
						gtk_label_set_text(GTK_LABEL(nw[2]),grp->str);
						g_string_free(grp,TRUE);
					}
				}

				if(pb!=NULL)
					free(pb);
			}
			else
			{
				/* it is a bar destruction */
				if(get_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0,&pb,&pb_len))
				{
					if(pb!=NULL)
					{
						gtk_widget_destroy(((GtkWidget **)pb)[0]);
					}
					if(pb!=NULL)
						free(pb);
					delete_this_tos_entry(PGBAR_TOSKEY,splt[0],strlen(splt[0]),0);
				}
			}
		}
		g_strfreev(splt);
	}
}

static GPtrArray *glst_temp_array=NULL;		/* it is an array of gchar ** (each has been obtain using g_strsplit on one line of GLSTC) */

/************************************************/
/* start the beginning of the GDL ctree refresh */
/************************************************/
void glst_begin_fnc(const GString *in)
{
	free_temp_array(&glst_temp_array);
	glst_temp_array=g_ptr_array_new();
}

static const char *ftypestr2num(char *m1)
{
	int i;

	static struct
	{
		const char *pattern;
		int pattern_length;
		const char *str;
	} str[]={ 
	 		{"[any]",sizeof("[any]")-1,"1"}, 
			{"[audio]",sizeof("[audio]")-1,"2"}, 
			{"[compressed]",sizeof("[compressed]")-1,"3"}, 
			{"[document]",sizeof("[document]")-1,"4"}, 
			{"[exe]",sizeof("[exe]")-1,"5"}, 
			{"[picture]",sizeof("[picture]")-1,"6"}, 
			{"[video]",sizeof("[video]")-1,"7"}, 
			{"[folder]",sizeof("[folder]")-1,"8"}, 
			{NULL,0,NULL}};

	i=0;
	while(str[i].pattern!=NULL)
	{
		if(!strncmp(m1,str[i].pattern,str[i].pattern_length))
			return str[i].str;
		i++;
	}
	return "";
}

/*******************************************************************************************************************/
/* for each source/range entry of this GDL, we check if it appear in the given glst_temp_array (data)              */
/* on success, we update it and mark the string of data as used (1st byte:'\001'). On error the entry is destroyed */
/*******************************************************************************************************************/
void second_level_glst_end_fnc(GtkCTree *ctree, GtkCTreeNode *cnode, gpointer data)
{
	gchar *type_entry;
	gchar *fname;
	gchar *size;
	GdkPixmap *pix;
	gchar **strv=((gchar**)data)+3;		/* skip the GDL ID, the local filename and the local filesize */
	int i;
	gchar *temp_str;

	if(GTK_CTREE_ROW(cnode)->level!=2)
		return;

   if((gtk_ctree_node_get_pixtext(ctree,cnode,0,&type_entry,NULL,&pix,NULL)) &&
		(gtk_ctree_node_get_text(ctree,cnode,1,&fname)) &&
		(gtk_ctree_node_get_text(ctree,cnode,2,&size)) )
   {
		if(pix!=NULL)
		{
			char *t;
			/* only the autoscan search pattern has an icon */
			/* this ctree entry is a GDL autoscan */
			i=0;

			t=strchr(fname,' ');
			if(t==NULL)
				temp_str=g_strdup(fname);
			else
				temp_str=g_strconcat(ftypestr2num(fname),"?",t+1,NULL);	/* create a string "type?pattern" */

			while(strv[i]!=NULL)
			{
				if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
				{
					i++;							/* now, i is the index of the first range string */

					while(strv[i]!=NULL)
					{
						if(strlen(strv[i])==0)		/* we have found the empty string between range and autoscan */
						{
							i++;
							while(strv[i]!=NULL)
							{
								if(strlen(strv[i])==0)		/* we have found the empty string at the end of the autoscan */
									break;

								if(strv[i][0]!='\001')		/* string unused ? */
								{
									/* now, we have to check if we found the wanted string */
									if(!strcmp(strv[i],temp_str))
									{
										strv[i][0]='\001';	/* mark the string as used */
										g_free(temp_str);					/* we have found it, it is over */
										return;
									}
								}
								i++;
							}
							break;
						}
						i++;
					}
					break;
				}
				i++;
			}

		}
		else if(strlen(type_entry)==0)
		{
			/* this ctree entry is a GDL range */
			/* build the temporary comparaison string */
			temp_str=g_strconcat(fname,"$",size,NULL);	/* create a string "filename$range" */

			i=0;
			while(strv[i]!=NULL)
			{
				if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
				{
					i++;							/* now, i is the index of the first range string */

					while(strv[i]!=NULL)
					{
						if(strlen(strv[i])==0)		/* we have found the empty string between range and autoscan */
							break;

						if(strv[i][0]!='\001')		/* string unused ? */
						{
							/* now, we have to check if we found the wanted string */
							if(!strcmp(strv[i],temp_str))
							{
								strv[i][0]='\001';	/* mark the string as used */
								g_free(temp_str);					/* we have found it, it is over */
								return;
							}
						}
						i++;
					}
					break;
				}
				i++;
			}
		}
		else
		{
			/* this ctree entry is GDL source */
			int temp_str_len;
			char *status_temp;

			temp_str=g_strconcat(type_entry,"$",fname,"$",NULL);	/* create a string "nickname$filename$" */
			temp_str_len=strlen(temp_str);
			i=0;
			while(strv[i]!=NULL)
			{
				if(strlen(strv[i])==0)		/* we have found the empty string between sources and range in the GDLSTC message */
					break;

				if(strv[i][0]!='\001')		/* string unused ? */
				{
					/* now, we have to check if we found the wanted string */
					if(!strncmp(strv[i],temp_str,temp_str_len))
					{
						/* we have found the string, check if we must update the row */
						status_temp=strchr(strv[i]+temp_str_len,'$');
						if(status_temp!=NULL)
						{
							*status_temp++='\0';
							/* strv[i]+temp_str_len is the remote file size */
							/* status_temp is the xfer status */
							if(strcmp(strv[i]+temp_str_len,size))
							{
								gtk_ctree_node_set_text(ctree,cnode,2,strv[i]+temp_str_len);
							}

							gtk_ctree_node_set_text(ctree,cnode,3,
												((status_temp[0]=='W')?_("Waiting"): ((status_temp[0]=='T')?_("Trying"):status_temp)));
	
							strv[i][0]='\001';	/* mark the string as used */
							g_free(temp_str);					/* we have found it, it is over */
							return;
						}
					}
				}
				i++;
			}
		}

		/* the entry does not exist anymore */
		gtk_ctree_remove_node(ctree,cnode);
		g_free(temp_str);
	}
}

static const char *ftype2str(int file_type)
{
#define NB_TYPE_ENT 8
	static const char *ftype_str[]={ "any", "audio", "compressed", "document", "exe", "picture", "video", "folder", NULL};

	file_type--;

	if((file_type<0)||(file_type>=NB_TYPE_ENT))
		return "";

	if(ftype_str[file_type]==NULL)
		return "";
	return ftype_str[file_type];
}

/*******************************************************************************************/
/* for each unused string of the given glst_temp_array (data), we create all ctree entries */
/*******************************************************************************************/
static void second_level_add_newly_created_entries(GtkCTree *ctree, GtkCTreeNode *cnode, gpointer data)
{
	gchar **strv=((gchar **)data)+3;		/* skip the GDL ID, the local filename and the local filesize */
	int i;
	char *ent_array[4];		/* 4 columns in the list */
	
	i=0;
	/* scan source parts of the string */
	while((strv[i]!=NULL)&&(strlen(strv[i])))
	{
		if(strv[i][0]!='\001')		/* string unused ? */
		{
			gchar **f4;
			GtkCTreeNode *new_node;

			f4=g_strsplit(strv[i],"$",3);
			if((f4[0]!=NULL)&&(f4[1]!=NULL)&&(f4[2]!=NULL)&&(f4[3]!=NULL))
			{
				ent_array[0]=f4[0];
				ent_array[1]=f4[1];
				ent_array[2]=f4[2];
				if(f4[3][0]=='W')
					ent_array[3]=_("Waiting");
				else if(f4[3][0]=='T')
					ent_array[3]=_("Trying");
				else
					ent_array[3]=f4[3];
				new_node=gtk_ctree_insert_node(ctree,cnode,NULL, ent_array, 5,NULL,NULL,NULL,NULL,TRUE,TRUE);

				if(user_here(ent_array[0])!=-1)
					gtk_ctree_node_set_background (ctree,new_node,&green);

				/* put operator xfer in red */
				if(str_array_is_inside(op_array,ent_array[0]))
				{
					gtk_ctree_node_set_foreground (ctree,new_node,&light_red);
				}
			}
			g_strfreev(f4);
		}
		i++;
	}

	if(strv[i]!=NULL)
	{
		/* we have found the empty string between sources and range in the GDLSTC message */
		i++;							/* now, i is the index of the first range string */
		while((strv[i]!=NULL)&&(strlen(strv[i])))
		{
			if(strv[i][0]!='\001')		/* string unused ? */
			{
				gchar **f2;

				f2=g_strsplit(strv[i],"$",1);
				if((f2[0]!=NULL)&&(f2[1]!=NULL))
				{
					ent_array[0]="";
					ent_array[1]=f2[0];
					ent_array[2]=f2[1];
					ent_array[3]="";
					gtk_ctree_insert_node(ctree,cnode,NULL, ent_array, 5,NULL,NULL,NULL,NULL,TRUE,TRUE);
				}
				g_strfreev(f2);
			}
			i++;
		}

		if(strv[i]!=NULL)
		{
			/* we have found the empty string between range and autoscan */
			i++;							/* now, i is the index of the first autoscan string */
			while((strv[i]!=NULL)&&(strlen(strv[i])))
			{
				if(strv[i][0]!='\001')		/* string unused ? */
				{
					gchar **f2;

					f2=g_strsplit(strv[i],"$",1);
					if((f2[0]!=NULL)&&(f2[1]!=NULL))
					{
						static GdkPixmap *as_pixmap=NULL;
						static GdkBitmap *as_mask=NULL;
						GtkCTreeNode *nw_node;
						GString *long_desc;
						int ftype;
						char *t;

						long_desc=g_string_new(f2[1]);
						ftype=atoi(long_desc->str);
						t=strchr(long_desc->str,'?');
						if(t!=NULL)
						{
							long_desc=g_string_erase(long_desc,0,1+t-long_desc->str);
						}
						long_desc=g_string_prepend(long_desc,"] ");
						long_desc=g_string_prepend(long_desc,ftype2str(ftype));
						long_desc=g_string_prepend_c(long_desc,'[');

						ent_array[0]=f2[0];
						ent_array[1]=long_desc->str;
						ent_array[2]="";
						ent_array[3]="";

						if(as_pixmap==NULL)
						{
							as_pixmap=gdk_pixmap_create_from_xpm_d(main_window->window,&as_mask,NULL,search_xpm);
						}
						nw_node=gtk_ctree_insert_node(ctree,cnode,NULL, ent_array, 5,NULL,NULL,NULL,NULL,TRUE,TRUE);
						gtk_ctree_node_set_pixtext(ctree,nw_node,0,f2[0],0,as_pixmap,as_mask);
						g_string_free(long_desc,TRUE);
					}
					g_strfreev(f2);
				}
				i++;
			}
		}
	}
}

/**********************************************************************************/
/* for each GDL of the ctree, we do this:                                         */
/* 1) check if it still exists. If not, delete it                                 */
/* 2) if it exists, we scan its sources and ranges to remove no more existing one */
/* 3) if it exists, we add newly inserted ranges and sources                      */
/**********************************************************************************/
static void first_level_glst_end_fnc(GtkCTree *ctree, GtkCTreeNode *cnode, gpointer data)
{
	gchar *gdl_id_txt;
	int i;

	/* the first column of a ctree is always a pixtext */
	if(gtk_ctree_node_get_pixtext(ctree,cnode,0,&gdl_id_txt,NULL,NULL,NULL))
	{
		/* now, search for this id in the glst_temp_array */
		for(i=0;i<glst_temp_array->len;i++)
		{
			gchar **ptr;

			ptr=g_ptr_array_index(glst_temp_array,i);
			if(ptr!=NULL)
			{

				if(!strcmp(gdl_id_txt,ptr[0]))		/* compare this temp entry to the gdl_id we search. On success, we start level2 scan */
				{
					GString *spd_str;
					unsigned long cur_size=strtoul(ptr[4],NULL,10);
					unsigned long ttl_size=strtoul(ptr[2],NULL,10);
					time_t duration;
					double spd;

					spd_str=g_string_new("");
					g_string_sprintf(spd_str,"%15lu (%.2f%%) ",cur_size,100.0*(double)cur_size/(double)ttl_size);
					
					cur_size-=strtoul(ptr[3],NULL,10);		/* number of bytes downloaded */
					duration=time(NULL)-strtoul(ptr[5],NULL,10);
					
					spd=(double)cur_size/(double)duration;
					if(spd<1024.0)
					{
						g_string_sprintfa(spd_str,"%.2lfB/s",spd);
					}
					else if(spd<(1024.0*1024.0))
					{
						g_string_sprintfa(spd_str,"%.2lfKB/s",spd/1024.0);
					}

					gtk_ctree_node_set_text(ctree,cnode,3,spd_str->str);
					g_string_free(spd_str,TRUE);

					gtk_ctree_post_recursive_to_depth(ctree,cnode,2,second_level_glst_end_fnc,ptr);
					
					second_level_add_newly_created_entries(ctree,cnode,ptr);
					g_strfreev(ptr);	/* and free memory */
					g_ptr_array_index(glst_temp_array,i)=NULL;
					return;
				}
			}
		}
		
		/* the ctree entry no more exists, delete it */
		gtk_ctree_remove_node(ctree,cnode);
	}
}

/*****************************/
/* end the GDL ctree refresh */
/*****************************/
void glst_end_fnc(const GString *in)
{
	GtkWidget *w;
	int i;

	if(glst_temp_array==NULL)
		return;

	w=get_widget_by_widget_name("gdl_ctree");
	if(w==NULL)
		return;

	gtk_clist_freeze(GTK_CLIST(w));

	/* now, we will scans the ctree to remove all sources entries which no more exists and add newly added sources */
	/* at the same time, we remove GDL entries which no more exist */
	gtk_ctree_post_recursive_to_depth(GTK_CTREE(w),NULL,1,first_level_glst_end_fnc,NULL);
	
	/* the only thing to do know is to add newly create GDL with their sources */
	for(i=0;i<glst_temp_array->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index(glst_temp_array,i);
		if(k)
		{
			/* at least 5 fields */
			GtkCTreeNode *new_gdl;
			char *ent_array[4];		/* 4 columns in the list */

			/* remove an entry with the same gdl_id */

			/* add a new entry in the root list */
			ent_array[0]=k[0];
			ent_array[1]=k[1];
			ent_array[2]=k[2];
			ent_array[3]="";
			new_gdl=gtk_ctree_insert_node(GTK_CTREE(w),NULL,NULL,
													ent_array,5,NULL,NULL,NULL,NULL,FALSE,TRUE);
			/* and put its info inside it */
			second_level_add_newly_created_entries(GTK_CTREE(w),new_gdl,k);

			g_strfreev(k);
			g_ptr_array_index(glst_temp_array,i)=NULL;
			
		}
	}
	gtk_ctree_sort_recursive (GTK_CTREE(w), NULL);

	gtk_clist_thaw(GTK_CLIST(w));

	/* and free unnecessary data of the glst_temp_array */
	/* this loop should be unuseful because the temp_array should already by empty */
	for(i=0;i<glst_temp_array->len;i++)
	{
		gchar **k;

		k=g_ptr_array_index(glst_temp_array,i);
		if(k)
			g_strfreev(k);
	}
	g_ptr_array_free(glst_temp_array,TRUE);
	glst_temp_array=NULL;

	return;
}

/***********************************************************************************/
/* search inside the root of the given ctree and delete a node having the given id */
/***********************************************************************************/
void found_and_delete_gdl_ctree_entry(GtkCTree *ctree, GtkCTreeNode *cnode, gpointer data)
{
	gchar *txt;

	/* the first column of a ctree is always a pixtext */
	if(gtk_ctree_node_get_pixtext(ctree,cnode,0,&txt,NULL,NULL,NULL))
	{
		if((txt!=NULL)&&(!strcmp(txt,data)))
		{
			gtk_ctree_remove_node(ctree,cnode);
		}
	}
}

/*********************************/
/* temporary store one GDL entry */
/*********************************/
void glst_content_fnc(const GString *in)
{
	char *t;
	gchar **fields;

	/* s format: ] "xxxxx"gdlID|localfname|fsize|offset|downloaded|start_time|runningxfer|donexfer|autoscan */
	t=strchr(in->str,'"');
	if(t==NULL)
		return;
	t=strchr(t+1,'"');
	if(t==NULL)
		return;
	t++;

	fields=g_strsplit(t,"|",0);
	g_ptr_array_add(glst_temp_array,fields);
}

static struct 
{
	const char *msg_type;
	void (*fnc)(const GString *);
} msg_fnc[]={
                              {"ERR  ]",err_fnc},			/* done, not checked */
                              {"DEBUG]",dummy_fnc},
                              {"INFO ]",dummy_fnc},
                              {"USER ]",user__fnc},		/* done, checked */
                              {"OP   ]",op_fnc},
                              {"ADMIN]",op_fnc},
                              {"USER+]",user__fnc},		/* done, checked */
                              {"USER-]",user_minus_fnc},	/* done, checked */
                              {"CHAT ]",chat_fnc},			/* done, checked */
                              {"UINFO]",uinfo_fnc},		/* done, checked (partial) */
                              {"HUBNM]",hubnm_fnc},		/* done, checked */
                              {"SREST]",srest_fnc},		/* done, checked */
                              {"$USER]",dummy_fnc},

                              {"$UL+ ]",update_lists},		/* done, not checked */
                              {"$UL- ]",update_lists},		/* same function on error and success */
                              {"$UL# ]",update_lists},	/* done, not checked */
                              {"$DL+ ]",update_lists},		/* done, not checked */
                              {"$DL- ]",remove_dl_xfer_on_success_fnc},	/* done, not checked */
                              {"$DL# ]",update_lists},	/* done, not checked */
                              {"$LS+ ]",update_lists},		/* in progress */
                              {"$LS- ]",update_lists},
                              {"$LS# ]",update_lists},
										{"$UL= ]",update_lists},
										{"$DL= ]",update_lists},
										{"$LS= ]",update_lists},
										{"ASTRT]",update_lists},
										{"ASTOP]",update_lists},
                              {"$DL~ ]",update_lists},
										{"$LS~ ]",update_lists},
										{"RFRSH]",update_lists},

                              {"XFERR]",add_xferr_fnc},
                              {"XFERQ]",add_xferq_fnc},
                              {"CMDKB]",add_cmdkb_fnc},
										{"BXFER]",begin_xfer_list_fnc},
										{"EXFER]",end_xfer_list_fnc},

										{"$ULST]",update_ul_list},
                              {"HUB- ]",dummy_fnc},
                              {"PRIV ]",pchat_fnc},
                              {"HUBGO]",dummy_fnc},
                              {"HUB+ ]",dummy_fnc},
                              {"FLST ]",flst_fnc},
                              {"FLSTE]",dummy_fnc},
                              {"VAR  ]",var_fnc},
                              {"PASWD]",enter_passwd_fnc},
                              {"PRGBR]",prog_bar_fnc},
										{"GLSTB]",glst_begin_fnc},
										{"GLSTC]",glst_content_fnc},
										{"GLSTE]",glst_end_fnc},
										{NULL,NULL}
				};

/**********************************************************************************/
/* this function is called when something comes from DCTC client                  */
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_READ */
/**********************************************************************************/
void process_data_from_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	GString *input;
	int i;

	input=get_dctc_line();
	if(input==NULL)
		return;

#if 0
	printf("%s\n",input->str);
#endif
	i=0;
	while(msg_fnc[i].msg_type!=NULL)
	{
		if(!strncmp(input->str,msg_fnc[i].msg_type,6))
		{
			(msg_fnc[i].fnc)(input);
			break;
		}
		i++;
	}

	g_string_free(input,TRUE);
	return;
}

/************************************************************************************/
/* same function as previous one except it tries to send queued data to DCTC client */
/************************************************************************************/
/* data is always NULL, source == current_dctc_fd and condition == GTK_INPUT_WRITE */
/***********************************************************************************/
void process_data_to_dctc(gpointer data, gint source, GdkInputCondition condition)
{
	if(current_dctc!=NULL)
	{	/* the previous test should never fail */

		/* as long as there is something queued and it is possible to send data */
		/* on the socket, we send */
		while(current_dctc->write_q->len!=0)
		{
			char *str;
			int l;
			int ret;
		
			str=g_ptr_array_index(current_dctc->write_q,0);
			l=strlen(str);

			ret=send(current_dctc->dctc_fd,str,l,MSG_DONTWAIT);
			if(ret==l)
			{
				/* successfully sent => go to next one */
				g_ptr_array_remove_index(current_dctc->write_q,0);
				free(str);
			}
			else
			{
				/* oh oh, an error occurs */
				if(errno!=EAGAIN)
				{
					/* and it is a fatal one */
					close_current_dctc();
					/* we can no longer do anything because the structure has been destroyed */
					gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with DCTC terminated."));
					return;
				}
				break;
			}
		}

		if(current_dctc->write_q->len==0)
		{
			gdk_input_remove(current_dctc->tag_write);
			current_dctc->tag_write=-1;
		}
	}
	else
	{
		fprintf(stderr,"process_data_to_dctc: invalid handler.\n");
	}
}

/****************************************/
/* send data to the current DCTC client */
/* the function handles all errors      */
/****************************************/
void send_data_to_dctc(char *str)
{
	char *tm;

	if(current_dctc==NULL)
		return;

	/* if the wait_q is empty, we try to send the string without queueing */
	/* else the string is queued and gdk_input handler is enabled */
	if(current_dctc->write_q->len==0)
	{
		int l;
		int ret;

		l=strlen(str);

	#if 0
		ret=send(current_dctc->dctc_fd,str,l,0);
	#else
		ret=send(current_dctc->dctc_fd,str,l,MSG_DONTWAIT);
	#endif
		if(ret==l)
			return;		/* everything ok, we end here */

		/* oh oh, an error occurs */
		if(errno!=EAGAIN)
		{
			close_current_dctc();
			gnome_app_error(GNOME_APP(main_window),_("Error while sending data to DCTC.\nConnection with DCTC terminated."));
		}
	}

	/* if a queue still exists or the atomic send fails, the string is queued */
	tm=strdup(str);
	if(tm==NULL)
	{
		close_current_dctc();
		gnome_app_error(GNOME_APP(main_window),_("Out of memory.\nConnection with DCTC terminated."));
		return;
	}

	g_ptr_array_add(current_dctc->write_q,tm);

	if(current_dctc->tag_write==-1)
	{
		current_dctc->tag_write=gdk_input_add(current_dctc->dctc_fd,GDK_INPUT_WRITE,process_data_to_dctc,NULL);
	}
}

/*************************************/
/* kill the given entry of the ctree */
/*********************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
void kill_this_gdl_entry(gpointer entry, gpointer ct)
{
	GtkCTree *ctree;

	if((ct==NULL)||(entry==NULL))
	{
		printf("kill_this_gdl_entry: NULL\n");
		return;
	}

	ctree=GTK_CTREE(ct);

	switch(GTK_CTREE_ROW(entry)->level)
	{
		case 1:	/* GDL ID level */
					{
						char *gdl_id_txt;

						gdl_id_txt=GTK_CTREE_ROW(entry)->row.cell[0].u.text;		/* extract the first column of the given ctree_row */
																									/* FIXME: someone knows a better way ? */
						if(gdl_id_txt!=NULL)
						{
							GString *str;

							str=g_string_new("/GDLEND ");
							str=g_string_append(str,gdl_id_txt);
							str=g_string_append_c(str,'\n');
							send_data_to_dctc(str->str);
							g_string_free(str,TRUE);
						}
						else
						{
							printf("no data on column 0\n");
						}
					}
					break;

		case 2:	/* range or download source */
					if(GTK_CTREE_ROW(entry)->row.cell[0].u.pt.pixmap!=NULL)	/* FIXME: someone knows a better way ? */
					{
						/* only the autoscan search pattern has an icon */
						/* this ctree entry is a GDL autoscan */
						char *gdl_id_txt;

						gdl_id_txt=GTK_CTREE_ROW(GTK_CTREE_ROW(entry)->parent)->row.cell[0].u.text;
																									/* extract the first column of parent the given ctree_row */
																									/* FIXME: someone knows a better way ? */
						if(gdl_id_txt!=NULL)
						{
							char *gae_id;
							gae_id=GTK_CTREE_ROW(entry)->row.cell[0].u.text;		/* extract the first column of the given ctree_row */
																								/* FIXME: someone knows a better way ? */
							if(gae_id!=NULL)
							{
								GString *str;

								str=g_string_new("");
								g_string_sprintf(str,"/GDLAS- %s|%s\n",gdl_id_txt,gae_id);
								send_data_to_dctc(str->str);
								g_string_free(str,TRUE);
							}
						}
					}
					else
					{
						char *gdl_id_txt;

						gdl_id_txt=GTK_CTREE_ROW(GTK_CTREE_ROW(entry)->parent)->row.cell[0].u.text;
																									/* extract the first column of parent the given ctree_row */
																									/* FIXME: someone knows a better way ? */
						if(gdl_id_txt!=NULL)
						{
							char *nick;
							nick=GTK_CTREE_ROW(entry)->row.cell[0].u.text;		/* extract the first column of the given ctree_row */
																								/* FIXME: someone knows a better way ? */
							if(nick!=NULL)
							{
								if(strlen(nick)!=0)
								{	/* it is a transfer line. A done xfer line always has an empty string here */
									char *rfname;
									rfname=GTK_CTREE_ROW(entry)->row.cell[1].u.text;		/* extract the 2nde column of the given ctree_row */
																											/* FIXME: someone knows a better way ? */
									if(rfname!=NULL)
									{
										GString *str;

										str=g_string_new("");
										g_string_sprintf(str,"/GDLDEL %s|%s|%s\n",gdl_id_txt,nick,rfname);
										send_data_to_dctc(str->str);
										g_string_free(str,TRUE);
									}
									else
									{
										printf("no data on column 1 (2)\n");
									}
								}
							}
							else
							{
								printf("no data on column 0 (2)\n");
							}
						}
						else
						{
							printf("no data on column 0 of parent\n");
						}
					}
					break;
	}

#if 0
	if(GTK_CTREE_ROW(entry)->row.state==GTK_STATE_SELECTED)
	{
		printf("selected\n");
	}
	else
	{
		printf("not selected\n");
	}
	printf("leaf: %d  level: %d\n",GTK_CTREE_ROW(entry)->is_leaf,GTK_CTREE_ROW(entry)->level);
#endif
}

/*****************************/
/* kill selected GDL entries */
/*****************************/
void kill_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkCTree *ctree;
	w=get_widget_by_widget_name("gdl_ctree");
	if(w==NULL)
		return;

	ctree=GTK_CTREE(w);

	g_list_foreach(GTK_CLIST(ctree)->selection,kill_this_gdl_entry,ctree);

	update_lists(NULL);
}

/***************************************/
/* detach the given entry of the ctree */
/*********************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
void detach_this_gdl_entry(gpointer entry, gpointer ct)
{
	GtkCTree *ctree;

	if((ct==NULL)||(entry==NULL))
	{
		printf("detach_this_gdl_entry: NULL\n");
		return;
	}

	ctree=GTK_CTREE(ct);

	if((GTK_CTREE_ROW(entry)->level)==1)
	{
		/* GDL ID level */
		{
			char *gdl_id_txt;

			gdl_id_txt=GTK_CTREE_ROW(entry)->row.cell[0].u.text;		/* extract the first column of the given ctree_row */
																									/* FIXME: someone knows a better way ? */
			if(gdl_id_txt!=NULL)
			{
				GString *str;

				str=g_string_new("/GDLDETACH ");
				str=g_string_append(str,gdl_id_txt);
				str=g_string_append_c(str,'\n');
				send_data_to_dctc(str->str);
				g_string_free(str,TRUE);
			}
			else
			{
				printf("no data on column 0\n");
			}
		}
	}
}

/*******************************/
/* detach selected GDL entries */
/*******************************/
void detach_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkCTree *ctree;
	w=get_widget_by_widget_name("gdl_ctree");
	if(w==NULL)
		return;

	ctree=GTK_CTREE(w);

	g_list_foreach(GTK_CLIST(ctree)->selection,detach_this_gdl_entry,ctree);

	update_lists(NULL);
}

static int as_created=0;

/*************************************************************************/
/* create autoscan configuration window for the given entry of the ctree */
/*************************************************************************/
/* entry is a GtkCTreeNode, ct is a GtkCTree */
/*********************************************/
static void create_autoscan_window_for_this_gdl_entry(gpointer entry, gpointer ct)
{
	GtkCTree *ctree;

	if(as_created==1)
		return;

	if((ct==NULL)||(entry==NULL))
	{
		return;
	}

	ctree=GTK_CTREE(ct);

	if((GTK_CTREE_ROW(entry)->level)==1)
	{
		/* GDL ID level */
		{
			char *gdl_id_txt;
			char *gdl_name_txt;

			gdl_id_txt=GTK_CTREE_ROW(entry)->row.cell[0].u.text;		/* extract the first column of the given ctree_row */
																									/* FIXME: someone knows a better way ? */
			gdl_name_txt=GTK_CTREE_ROW(entry)->row.cell[1].u.text;		/* extract the 2nd column of the given ctree_row */
																									/* FIXME: someone knows a better way ? */
			if((gdl_id_txt!=NULL)&&(gdl_name_txt!=NULL))
			{
				GtkWidget *w;

				w=get_widget_by_widget_name("as_gid_label");
				if(w!=NULL)
				{
					gtk_label_set(GTK_LABEL(w),gdl_id_txt);
				}

				w=get_widget_by_widget_name("as_fname_label");
				if(w!=NULL)
				{
					gtk_label_set(GTK_LABEL(w),gdl_name_txt);
				}

				/* switch to the find config tab */
   			w=get_widget_by_widget_name("main_notebook");
   			gtk_notebook_set_page(GTK_NOTEBOOK(w),FIND_CONFIG_TAB);

				w=get_widget_by_widget_name("gdl_as_pattern_entry");
				if(w!=NULL)
				{
					gtk_editable_set_position(GTK_EDITABLE(w),0);
					gtk_editable_delete_text(GTK_EDITABLE(w),0,-1);
					gtk_widget_grab_focus(w);
				}

			}
			else
			{
				printf("no data on column 0\n");
			}
		}
	}
}


/*****************************************************************/
/* create autoscan configuration window for selected GDL entries */
/*****************************************************************/
void create_autoscan_window_for_selected_gdl_entry(void)
{
	GtkWidget *w;
	GtkCTree *ctree;
	w=get_widget_by_widget_name("gdl_ctree");
	if(w==NULL)
		return;

	ctree=GTK_CTREE(w);

	as_created=0;
	g_list_foreach(GTK_CLIST(ctree)->selection,create_autoscan_window_for_this_gdl_entry,ctree);

	update_lists(NULL);
}


/**********************************************************************/
/* kill upload or download task using their ID (and /KILL or /KILLKB) */
/**********************************************************************/
void kill_selected_entry(char *cmd,char *clist_name)
{
	GtkWidget *w;
	GtkCList *clist;
	GtkCListRow *clist_row;

	int row;
	char buf[512];

	w=get_widget_by_widget_name(clist_name);
	if(w==NULL)
		return;
	clist=GTK_CLIST(w);

   for(row=0;row<clist->rows;row++)
   {
      clist_row=g_list_nth(clist->row_list,row)->data;

      if(clist_row->state==GTK_STATE_SELECTED)
      {
			char *t;

			gtk_clist_get_text(clist,row,0,&t);

			sprintf(buf,"%s %s\n",cmd,t);
			send_data_to_dctc(buf);
		}
	}

	update_lists(NULL);
}

/************************************/
/* view file list of selected users */
/************************************/
void view_file_selected_entry(char *cmd,char *clist_name)
{
	GtkWidget *w;
	GtkCList *clist;
	GtkCListRow *clist_row;

	int row;
	char buf[512];

	w=get_widget_by_widget_name(clist_name);
	if(w==NULL)
		return;
	clist=GTK_CLIST(w);

   for(row=0;row<clist->rows;row++)
   {
      clist_row=g_list_nth(clist->row_list,row)->data;

      if(clist_row->state==GTK_STATE_SELECTED)
      {
			char *t;

			gtk_clist_get_text(clist,row,1,&t);

			sprintf(buf,"%s %s\n",cmd,t);
			send_data_to_dctc(buf);
		}
	}
}

/***********************************************************************************/
/* when the multihub search is enabled, this function is called after 10 seconds   */
/* if more than 15 results are available, nothing occurs, else a /MSRCH is started */
/***********************************************************************************/
gint start_msearch(gpointer data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name("find_result");
	if(w!=NULL)
	{
		if(GTK_CLIST(w)->rows<15)
		{
			GString *str;
			int i;

			str=g_string_new("");
			for(i=0;i<2;i++)
			{
				if(last_search[i]!=NULL)
				{
					str=g_string_assign(str,last_search[i]->str);
					str=g_string_insert_c(str,1,'M');	/* transform the /SRCH into /MSRCH */
					send_data_to_dctc(str->str);
				}
			}
			g_string_free(str,TRUE);
		}
	}
	last_search_tag=-1;
	return FALSE;		/* don't timeout again */
}



