/* darkstat 3
 * copyright (c) 2001-2006 Emil Mikulic.
 *
 * hosts_db.c: database of hosts, ports, protocols.
 *
 * You may use, modify and redistribute this file under the terms of the
 * GNU General Public License version 2. (see COPYING.GPL)
 */

#include "darkstat.h"
#include "conv.h"
#include "decode.h"
#include "dns.h"
#include "err.h"
#include "hosts_db.h"
#include "html.h"
#include "ncache.h"

#include <arpa/inet.h> /* inet_aton() */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h> /* memset(), strcmp() */

/* FIXME: specify somewhere more sane/tunable */
#define MAX_ENTRIES 30 /* in an HTML table rendered from a hashtable */

/*
 * Hosts table reduction - when the number of hosts hits NUM_HOSTS_HIGH, we
 * reduce the table to NUM_HOSTS_LOW hosts.
 */
#define NUM_HOSTS_HIGH 1000
#define NUM_HOSTS_LOW  500

typedef uint32_t (hash_func_t)(const struct hashtable *, const void *);
typedef const void * (key_func_t)(const struct bucket *);
typedef int (find_func_t)(const struct bucket *, const void *);
typedef struct bucket * (make_func_t)(const void *);
typedef void (format_cols_func_t)(struct str *);
typedef void (format_row_func_t)(struct str *, const struct bucket *);

struct hashtable {
   uint8_t bits;     /* size of hashtable in bits */
   uint32_t size, mask;
   uint32_t count;   /* items in table */
   uint32_t coeff;   /* coefficient for Fibonacci hashing */
   struct bucket **table;

   struct {
      uint64_t inserts, searches, deletions, rehashes;
   } stats;

   hash_func_t *hash_func;
   /* returns hash value of given key (passed as void*) */

   key_func_t *key_func;
   /* returns pointer to key of bucket (to pass to hash_func) */

   find_func_t *find_func;
   /* returns true if given bucket matches key (passed as void*) */

   make_func_t *make_func;
   /* returns bucket containing new record with key (passed as void*) */

   format_cols_func_t *format_cols_func;
   /* append table columns to str */

   format_row_func_t *format_row_func;
   /* format record and append to str */
};

#define HOST_BITS 1  /* initial size of hosts table */
#define PORT_BITS 1  /* initial size of ports tables */
#define PROTO_BITS 1 /* initial size of proto table */

/* We only use one hosts_db hashtable and this is it. */
static struct hashtable *hosts_db = NULL;

/* phi^-1 (reciprocal of golden ratio) = (sqrt(5) - 1) / 2 */
static const double phi_1 =
   0.61803398874989490252573887119069695472717285156250;

/* Co-prime of u, using phi^-1 */
inline static uint32_t
coprime(const uint32_t u)
{
   return ( (uint32_t)( (double)(u) * phi_1 ) | 1U );
}

/*
 * This is the "recommended" IPv4 hash function, as seen in FreeBSD's
 * src/sys/netinet/tcp_hostcache.c 1.1
 */
inline static uint32_t
ipv4_hash(const uint32_t ip)
{
   return ( (ip) ^ ((ip) >> 7) ^ ((ip) >> 17) );
}

/* ---------------------------------------------------------------------------
 * hash_func collection
 */
#define CASTKEY(type) (*((const type *)key))

static uint32_t
hash_func_host(const struct hashtable *h _unused_, const void *key)
{
   return (ipv4_hash(CASTKEY(in_addr_t)));
}

static uint32_t
hash_func_short(const struct hashtable *h, const void *key)
{
   return (CASTKEY(uint16_t) * h->coeff);
}

/* ---------------------------------------------------------------------------
 * key_func collection
 */

static const void *
key_func_host(const struct bucket *b)
{
   return &(b->u.host.ip);
}

static const void *
key_func_port_tcp(const struct bucket *b)
{
   return &(b->u.port_tcp.port);
}

static const void *
key_func_port_udp(const struct bucket *b)
{
   return &(b->u.port_udp.port);
}

static const void *
key_func_ip_proto(const struct bucket *b)
{
   return &(b->u.ip_proto.proto);
}

/* ---------------------------------------------------------------------------
 * find_func collection
 */

static int
find_func_host(const struct bucket *b, const void *key)
{
   return (b->u.host.ip == CASTKEY(in_addr_t));
}

static int
find_func_port_tcp(const struct bucket *b, const void *key)
{
   return (b->u.port_tcp.port == CASTKEY(uint16_t));
}

static int
find_func_port_udp(const struct bucket *b, const void *key)
{
   return (b->u.port_udp.port == CASTKEY(uint16_t));
}

static int
find_func_ip_proto(const struct bucket *b, const void *key)
{
   return (b->u.ip_proto.proto == CASTKEY(uint16_t));
}

/* ---------------------------------------------------------------------------
 * make_func collection
 */

#define MAKE_BUCKET(name_bucket, name_content, type) struct { \
   struct bucket *next; \
   uint64_t in, out, total; \
   union { struct type t; } u; } _custom_bucket; \
   struct bucket *name_bucket = xmalloc(sizeof(_custom_bucket)); \
   struct type *name_content = &(name_bucket->u.type); \
   name_bucket->next = NULL; \
   name_bucket->in = name_bucket->out = name_bucket->total = 0;

static struct bucket *
make_func_host(const void *key)
{
   MAKE_BUCKET(b, h, host);
   h->ip = CASTKEY(in_addr_t);
   h->dns = NULL;
   h->ports_tcp = NULL;
   h->ports_udp = NULL;
   h->ip_protos = NULL;
   return (b);
}

static struct bucket *
make_func_port_tcp(const void *key)
{
   MAKE_BUCKET(b, p, port_tcp);
   p->port = CASTKEY(uint16_t);
   p->syn = 0;
   return (b);
}

static struct bucket *
make_func_port_udp(const void *key)
{
   MAKE_BUCKET(b, p, port_udp);
   p->port = CASTKEY(uint16_t);
   return (b);
}

static struct bucket *
make_func_ip_proto(const void *key)
{
   MAKE_BUCKET(b, p, ip_proto);
   p->proto = CASTKEY(uint16_t);
   return (b);
}

/* ---------------------------------------------------------------------------
 * format_func collection (ordered by struct)
 */

static void
format_cols_host(struct str *buf)
{
   str_append(buf,
      "<table border=\"1\">\n"
      "<tr>\n"
      " <th>IP</th>\n"
      " <th>Hostname</th>\n"
      " <th>In</th>\n"
      " <th>Out</th>\n"
      " <th>Total</th>\n"
      "</tr>\n"
   );
}

static void
format_row_host(struct str *buf, const struct bucket *b)
{
   const char *ip = ip_to_str( b->u.host.ip );

   str_appendf(buf,
      "<tr>\n"
      " <td><a href=\"%s/\">%s</a></td>\n"
      " <td>%s</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      "</tr>\n",
      ip, ip, (b->u.host.dns == NULL)?"":b->u.host.dns,
      b->in, b->out, b->total
   );

   /* Only resolve hosts "on demand" */
   if (b->u.host.dns == NULL)
      dns_queue(b->u.host.ip);
}

static void
format_cols_port_tcp(struct str *buf)
{
   str_append(buf,
      "<table border=\"1\">\n"
      "<tr>\n"
      " <td>Port</td>\n"
      " <td>Service</td>\n"
      " <td>In</td>\n"
      " <td>Out</td>\n"
      " <td>Total</td>\n"
      " <td>SYNs</td>\n"
      "</tr>\n"
   );
}

static void
format_row_port_tcp(struct str *buf, const struct bucket *b)
{
   const struct port_tcp *p = &(b->u.port_tcp);

   str_appendf(buf,
      "<tr>\n"
      " <td>%u</td>\n"
      " <td>%s</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      "</tr>\n",
      p->port, getservtcp(p->port), b->in, b->out, b->total, p->syn
   );
}

static void
format_cols_port_udp(struct str *buf)
{
   str_append(buf,
      "<table border=\"1\">\n"
      "<tr>\n"
      " <td>Port</td>\n"
      " <td>Service</td>\n"
      " <td>In</td>\n"
      " <td>Out</td>\n"
      " <td>Total</td>\n"
      "</tr>\n"
   );
}

static void
format_row_port_udp(struct str *buf, const struct bucket *b)
{
   const struct port_udp *p = &(b->u.port_udp);

   str_appendf(buf,
      "<tr>\n"
      " <td>%u</td>\n"
      " <td>%s</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      "</tr>\n",
      p->port, getservudp(p->port), b->in, b->out, b->total
   );
}

static void
format_cols_ip_proto(struct str *buf)
{
   str_append(buf,
      "<table border=\"1\">\n"
      "<tr>\n"
      " <td>#</td>\n"
      " <td>Protocol</td>\n"
      " <td>In</td>\n"
      " <td>Out</td>\n"
      " <td>Total</td>\n"
      "</tr>\n"
   );
}

static void
format_row_ip_proto(struct str *buf, const struct bucket *b)
{
   const struct ip_proto *p = &(b->u.ip_proto);

   str_appendf(buf,
      "<tr>\n"
      " <td>%u</td>\n"
      " <td>%s</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      " <td>%'qu</td>\n"
      "</tr>\n",
      p->proto, getproto(p->proto),
      b->in, b->out, b->total
   );
}

/* ---------------------------------------------------------------------------
 * Initialise a hashtable.
 */
static struct hashtable *
hashtable_make(const uint8_t bits,
   hash_func_t *hash_func,
   key_func_t *key_func,
   find_func_t *find_func,
   make_func_t *make_func,
   format_cols_func_t *format_cols_func,
   format_row_func_t *format_row_func)
{
   struct hashtable *hash;
   assert(bits > 0);

   hash = xmalloc(sizeof(*hash));
   hash->bits = bits;
   hash->size = 1U << bits;
   hash->mask = hash->size - 1;
   hash->coeff = coprime(hash->size);
   hash->hash_func = hash_func;
   hash->key_func = key_func;
   hash->find_func = find_func;
   hash->make_func = make_func;
   hash->format_cols_func = format_cols_func;
   hash->format_row_func = format_row_func;
   hash->count = 0;
   hash->table = xcalloc(hash->size, sizeof(*hash->table));
   memset(&(hash->stats), 0, sizeof(hash->stats));
   return (hash);
}

/* ---------------------------------------------------------------------------
 * Initialise global hosts_db.
 */
void
hosts_db_init(void)
{
   assert(hosts_db == NULL);
   hosts_db = hashtable_make(HOST_BITS,
      hash_func_host, key_func_host, find_func_host, make_func_host,
      format_cols_host, format_row_host);
}

static void
hashtable_rehash(struct hashtable *h, const uint8_t bits)
{
   struct bucket **old_table, **new_table;
   uint32_t i, old_size;
   assert(h != NULL);
   assert(bits > 0);

   h->stats.rehashes++;
   old_size = h->size;
   old_table = h->table;

   h->bits = bits;
   h->size = 1U << bits;
   h->mask = h->size - 1;
   h->coeff = coprime(h->size);
   new_table = xcalloc(h->size, sizeof(*new_table));

   for (i=0; i<old_size; i++) {
      struct bucket *next, *b = old_table[i];
      while (b != NULL) {
         uint32_t pos = h->hash_func(h, h->key_func(b)) & h->mask;
         next = b->next;
         b->next = new_table[pos];
         new_table[pos] = b;
         b = next;
      }
   }

   free(h->table);
   h->table = new_table;
   verbosef("hashtable rehashed to %d slots (was using %d of %d)",
      h->size, h->count, old_size);
}

static void
hashtable_insert(struct hashtable *h, struct bucket *b)
{
   uint32_t pos;
   assert(h != NULL);
   assert(b != NULL);
   assert(b->next == NULL);

   /* Rehash on 80% occupancy */
   if ((h->count > h->size) ||
       ((h->size - h->count) < h->size / 5))
      hashtable_rehash(h, h->bits+1);

   pos = h->hash_func(h, h->key_func(b)) & h->mask;
   if (h->table[pos] == NULL)
      h->table[pos] = b;
   else {
      /* Insert at top of chain. */
      b->next = h->table[pos];
      h->table[pos] = b;
   }
   h->count++;
   h->stats.inserts++;
}

/* Return bucket matching key, or NULL if no such entry. */
static struct bucket *
hashtable_search(struct hashtable *h, const void *key)
{
   uint32_t pos;
   struct bucket *b;

   h->stats.searches++;
   pos = h->hash_func(h, key) & h->mask;
   b = h->table[pos];
   while (b != NULL) {
      if (h->find_func(b, key))
         return (b);
      else
         b = b->next;
   }
   return (NULL);
}

/* Search for a key.  If it's not there, make and insert a bucket for it. */
static struct bucket *
hashtable_find_or_insert(struct hashtable *h, const void *key)
{
   struct bucket *b = hashtable_search(h, key);

   if (b == NULL) {
      /* Not found so insert. */
      b = h->make_func(key);
      hashtable_insert(h, b);
   }
   return (b);
}

/*
 * Frees the hashtable and the buckets.  The contents are assumed to be
 * "simple" -- i.e. no "destructor" action is required beyond simply freeing
 * the bucket.
 */
static void
hashtable_free_simple(struct hashtable *h)
{
   uint32_t i;

   if (h == NULL)
      return;
   for (i=0; i<h->size; i++) {
      struct bucket *tmp, *b = h->table[i];
      while (b != NULL) {
         tmp = b;
         b = b->next;
         free(tmp);
      }
   }
   free(h->table);
   free(h);
}

static void
host_free(struct host *h)
{
   if (h->dns != NULL)
      free(h->dns);
   hashtable_free_simple(h->ports_tcp);
   hashtable_free_simple(h->ports_udp);
   hashtable_free_simple(h->ip_protos);
}

/* ---------------------------------------------------------------------------
 * Return existing host or insert a new one.
 */
struct bucket *
host_get(const in_addr_t ip)
{
   return (hashtable_find_or_insert(hosts_db, &ip));
}

/* ---------------------------------------------------------------------------
 * Find host, returns NULL if not in DB.
 */
struct bucket *
host_find(const in_addr_t ip)
{
   return (hashtable_search(hosts_db, &ip));
}

/* ---------------------------------------------------------------------------
 * Find host, returns NULL if not in DB.
 */
static struct bucket *
host_search(const char *ipstr)
{
   struct in_addr addr;

   if (inet_aton(ipstr, &addr) != 1)
      return (NULL); /* invalid addr */
   return (hashtable_search(hosts_db, &(addr.s_addr)));
}

/* ---------------------------------------------------------------------------
 * Reduce hosts_db if necessary.
 */
void
hosts_db_reduce(void)
{
   uint32_t i, pos, rmd;
   const struct bucket **table;
   uint64_t cutoff;

   if (hosts_db->count < NUM_HOSTS_HIGH)
      return;

   /* Fill table with pointers to buckets in hashtable. */
   table = xmalloc(sizeof(*table) * hosts_db->count);
   for (pos=0, i=0; i<hosts_db->size; i++) {
      struct bucket *b = hosts_db->table[i];
      while (b != NULL) {
         table[pos++] = b;
         b = b->next;
      }
   }
   assert(pos == hosts_db->count);
   qsort_buckets(table, hosts_db->count, 0, NUM_HOSTS_LOW, TOTAL);
   cutoff = table[NUM_HOSTS_LOW - 1]->total;
   free(table);

   /* Remove all hosts with total < cutoff. */
   rmd = 0;
   for (i=0; i<hosts_db->size; i++) {
      struct bucket *last = NULL, *next, *b = hosts_db->table[i];
      while (b != NULL) {
         next = b->next;
         if (b->total < cutoff) {
            /* Remove this one. */
            host_free(&(b->u.host));
            free(b);
            if (last == NULL)
               hosts_db->table[i] = next;
            else
               last->next = next;
            rmd++;
            hosts_db->count--;
         } else {
            last = b;
         }
         b = next;
      }
   }
   verbosef("hosts_db_reduce: from %u, removed %u hosts",
      hosts_db->count+rmd, rmd);
   hashtable_rehash(hosts_db, hosts_db->bits); /* needed? */
}

/* ---------------------------------------------------------------------------
 * Deallocate hosts_db.
 */
void hosts_db_free(void)
{
   uint32_t i;

   assert(hosts_db != NULL);
   for (i=0; i<hosts_db->size; i++) {
      struct bucket *tmp, *b = hosts_db->table[i];
      while (b != NULL) {
         host_free(&(b->u.host));
         tmp = b;
         b = b->next;
         free(tmp);
      }
   }
   free(hosts_db->table);
   free(hosts_db);
   hosts_db = NULL;
}

/* ---------------------------------------------------------------------------
 * Find or create a port_tcp inside a host.
 */
struct bucket *
host_get_port_tcp(struct bucket *host, const uint16_t port)
{
   struct host *h = &host->u.host;
   assert(h != NULL);
   if (h->ports_tcp == NULL)
      h->ports_tcp = hashtable_make(PORT_BITS,
         hash_func_short, key_func_port_tcp,
         find_func_port_tcp, make_func_port_tcp,
         format_cols_port_tcp, format_row_port_tcp);
   return (hashtable_find_or_insert(h->ports_tcp, &port));
}

/* ---------------------------------------------------------------------------
 * Find or create a port_udp inside a host.
 */
struct bucket *
host_get_port_udp(struct bucket *host, const uint16_t port)
{
   struct host *h = &host->u.host;
   assert(h != NULL);
   if (h->ports_udp == NULL)
      h->ports_udp = hashtable_make(PORT_BITS,
         hash_func_short, key_func_port_udp,
         find_func_port_udp, make_func_port_udp,
         format_cols_port_udp, format_row_port_udp);
   return (hashtable_find_or_insert(h->ports_udp, &port));
}

/* ---------------------------------------------------------------------------
 * Find or create an ip_proto inside a host.
 */
struct bucket *
host_get_ip_proto(struct bucket *host, const uint16_t proto)
{
   struct host *h = &host->u.host;
   assert(h != NULL);
   if (h->ip_protos == NULL)
      h->ip_protos = hashtable_make(PROTO_BITS,
         hash_func_short, key_func_ip_proto,
         find_func_ip_proto, make_func_ip_proto,
         format_cols_ip_proto, format_row_ip_proto);
   return (hashtable_find_or_insert(h->ip_protos, &proto));
}

static struct str *html_hosts_main(void);
static struct str *html_hosts_detail(const char *ip);
static struct str *html_hashtable_stats(struct hashtable *h);
static struct str *html_tcp_stats(const char *ip);
static struct str *html_udp_stats(const char *ip);
static struct str *html_proto_stats(const char *ip);

/* ---------------------------------------------------------------------------
 * Web interface: delegate the /hosts/ space.
 */
struct str *
html_hosts(const char *uri)
{
   int i, num_elems;
   char **elem = split('/', uri, &num_elems);
   struct str *buf = NULL;

   assert(num_elems >= 1);
   assert(strcmp(elem[0], "hosts") == 0);

   if (num_elems == 1)
      /* /hosts/ */
      buf = html_hosts_main();
   else if ((num_elems == 2) && (strcmp(elem[1], "stats") == 0))
      /* /hosts/stats/ */
      buf = html_hashtable_stats(hosts_db);
   else if (num_elems == 2)
      /* /hosts/<IP of host>/ */
      buf = html_hosts_detail(elem[1]);
   else if ((num_elems == 3) && (strcmp(elem[2], "tcpstats") == 0))
      /* /hosts/<IP of host>/tcpstats/ */
      buf = html_tcp_stats(elem[1]);
   else if ((num_elems == 3) && (strcmp(elem[2], "udpstats") == 0))
      /* /hosts/<IP of host>/udpstats/ */
      buf = html_udp_stats(elem[1]);
   else if ((num_elems == 3) && (strcmp(elem[2], "protostats") == 0))
      /* /hosts/<IP of host>/protostats/ */
      buf = html_proto_stats(elem[1]);

   for (i=0; i<num_elems; i++)
      free(elem[i]);
   free(elem);
   return (buf); /* FIXME: a NULL here becomes 404 Not Found, we might want
   other codes to be possible */
}

/* ---------------------------------------------------------------------------
 * Format hashtable into HTML.
 */
static void
format_table(struct str *buf, struct hashtable *ht)
{
   const struct bucket **table;
   uint32_t i, pos, top;

   if ((ht == NULL) || (ht->count == 0)) {
      str_append(buf, "<p>The table is empty.</p>\n");
      return;
   }

   /* Fill table with pointers to buckets in hashtable. */
   table = xmalloc(sizeof(*table) * ht->count);
   for (pos=0, i=0; i<ht->size; i++) {
      struct bucket *b = ht->table[i];
      while (b != NULL) {
         table[pos++] = b;
         b = b->next;
      }
   }
   assert(pos == ht->count);

   top = min(ht->count, MAX_ENTRIES);
   str_appendf(buf, "(top %u of %u)<br/>", top, ht->count); /* FIXME: ugly */
   qsort_buckets(table, ht->count, 0, top, TOTAL
      /* FIXME: offset(left), count(right), direction */ );
   ht->format_cols_func(buf);
   for (i=0; i<top; i++)
      ht->format_row_func(buf, table[i]);
   free(table);
   str_append(buf, "</table>\n");
}

/* ---------------------------------------------------------------------------
 * Web interface: sorted table of hosts.
 */
static struct str *
html_hosts_main(void)
{
   struct str *buf;
   assert(hosts_db != NULL);

   buf = str_make();
   str_append(buf, html_header_1);
   str_append(buf, "<title>darkstat3: Hosts</title>\n");
   str_append(buf, html_header_2);
   str_append(buf, "<h2 class=\"pageheader\">Hosts</h2>\n");
   format_table(buf, hosts_db);
   str_append(buf, "<a href=\"stats/\">statistics</a><br/>\n");
   str_append(buf, html_footer);
   return (buf);
}

/* ---------------------------------------------------------------------------
 * Web interface: detailed view of a single host.
 */
static struct str *
html_hosts_detail(const char *ip)
{
   struct bucket *h;
   struct str *buf;

   h = host_search(ip);
   if (h == NULL)
      return (NULL); /* no such host */

   /* Overview. */
   buf = str_make();
   /* FIXME: dns name */
   str_appendf(buf,
      "%s"
      " <title>%s</title>\n"
      "%s"
      "<h2>%s</h2>\n"
      "<p>\n"
      " <b>In:</b> %'qu<br>\n"
      " <b>Out:</b> %'qu<br>\n"
      " <b>Total:</b> %'qu<br>\n"
      "</p>\n"
      ,
      html_header_1, ip, html_header_2, ip, h->in, h->out, h->total
   );

   str_append(buf, "<h3>TCP ports</h3>\n");
   format_table(buf, h->u.host.ports_tcp);
   str_append(buf, "<a href=\"tcpstats/\">statistics</a><br/>\n");

   str_append(buf, "<h3>UDP ports</h3>\n");
   format_table(buf, h->u.host.ports_udp);
   str_append(buf, "<a href=\"udpstats/\">statistics</a><br/>\n");

   str_append(buf, "<h3>IP protocols</h3>\n");
   format_table(buf, h->u.host.ip_protos);
   str_append(buf, "<a href=\"protostats/\">statistics</a><br/>\n");

   str_append(buf, html_footer);
   return (buf);
}

/* ---------------------------------------------------------------------------
 * Web interface: statistics of a given hashtable.
 */
static struct str *
html_hashtable_stats(struct hashtable *h)
{
   struct str *buf;
   uint32_t i, max_len = 0, num_empty = 0, ovr_buckets = 0, ovr_slots = 0;

   buf = str_make();
   str_appendf(buf,
      "%s"
      " <title>Hashtable histogram</title>\n"
      " <style type=\"text/css\">\n"
      "  thead td { font-weight:bold; background:#E0E0E0; }\n"
      "  tbody td { text-align:right; background:#F0F0F0; }\n"
      " </style>\n"
      "%s"
      "<h2>Hashtable statistics</h2>\n",
      html_header_1, html_header_2
   );

   if (h == NULL)
      str_append(buf, "<p>Table is empty.</p>\n");
   else {
      /* Traverse table. */
      for (i=0; i<h->size; i++)
      {
         struct bucket *b = h->table[i];
         uint32_t curr_len = 0;

         if (b == NULL)
            num_empty++;
         else
            for (b = h->table[i]; b != NULL; b = b->next)
               curr_len++;

         if (curr_len > 1)
         {
            ovr_slots++;
            ovr_buckets += curr_len - 1;
            max_len = max(max_len, curr_len);
         }
      }

      str_appendf(buf,
         "<p>\n"
         "<b>Slots:</b> %'u<br>\n"
         "<b>Records:</b> %'u<br>\n"
         "<b>Slots empty:</b> %'u<br>\n"
         "<b>Slots overflowed:</b> %'u<br>\n"
         "<b>Overflow buckets:</b> %'u<br>\n"
         "<b>Longest chain:</b> %'u<br>\n"
         "<b>Usage:</b> %u%%<br>\n"
         "<br>\n"
         "<b>Inserts:</b> %'qu<br>\n"
         "<b>Searches:</b> %'qu<br>\n"
         "<b>Deletions:</b> %'qu<br>\n"
         "<b>Rehashes:</b> %'qu<br>\n"
         "</p>\n"
         ,
         h->size, h->count, num_empty, ovr_slots, ovr_buckets, max_len,
         100 * h->count / h->size,
         h->stats.inserts, h->stats.searches,
         h->stats.deletions, h->stats.rehashes);
   }

   str_append(buf, html_footer);
   return (buf);
}

static struct str *
html_tcp_stats(const char *ip)
{
   struct bucket *h = host_search(ip);

   if (h == NULL)
      return (NULL); /* no such host */
   return html_hashtable_stats(h->u.host.ports_tcp);
}

static struct str *
html_udp_stats(const char *ip)
{
   struct bucket *h = host_search(ip);

   if (h == NULL)
      return (NULL); /* no such host */
   return html_hashtable_stats(h->u.host.ports_udp);
}

static struct str *
html_proto_stats(const char *ip)
{
   struct bucket *h = host_search(ip);

   if (h == NULL)
      return (NULL); /* no such host */
   return html_hashtable_stats(h->u.host.ip_protos);
}

/* vim:set ts=3 sw=3 tw=78 expandtab: */
