/* darkstat 3
 * copyright (c) 2001-2006 Emil Mikulic.
 *
 * dns.c: synchronous DNS in a child process.
 *
 * You may use, modify and redistribute this file under the terms of the
 * GNU General Public License version 2. (see COPYING.GPL)
 */

#include "darkstat.h"
#include "conv.h"
#include "decode.h"
#include "dns.h"
#include "err.h"
#include "hosts_db.h"
#include "queue.h"
#include "tree.h"

#include <sys/socket.h>
#include <sys/wait.h>
#include <netdb.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static void dns_main(void); /* this is what the child process runs */

#define CHILD 0 /* child process uses this socket */
#define PARENT 1
static int sock[2];
static pid_t pid = -1;

void
dns_init(void)
{
   if (socketpair(AF_UNIX, SOCK_STREAM, 0, sock) == -1)
      err(1, "socketpair");

   pid = fork();
   if (pid == -1)
      err(1, "fork");

   if (pid == 0) {
      /* We are the child. */
      close(sock[PARENT]);
      dns_main();
      verbosef("fell out of dns_main()");
      exit(0);
   } else {
      /* We are the parent. */
      close(sock[CHILD]);
      verbosef("DNS parent has child with PID %d", pid);
   }
}

void
dns_stop(void)
{
   if (pid == -1) {
      verbosef("tried to dns_stop() without child process");
      return;
   }
   close(sock[PARENT]);
   if (kill(pid, SIGINT) == -1)
      err(1, "kill");
   verbosef("dns_stop() waiting for child");
   if (waitpid(pid, NULL, 0) == -1)
      err(1, "waitpid");
   verbosef("dns_stop() done waiting for child");
}

struct tree_rec {
   RB_ENTRY(tree_rec) ptree;
   in_addr_t ip;
};

static int
tree_cmp(struct tree_rec *a, struct tree_rec *b)
{
   if (a->ip < b->ip) return (-1); else
   if (a->ip > b->ip) return (+1); else
   return (0);
}

static RB_HEAD(tree_t, tree_rec) ip_tree = RB_INITIALIZER(&tree_rec);
RB_GENERATE(tree_t, tree_rec, ptree, tree_cmp)

void
dns_queue(const in_addr_t ip)
{
   struct tree_rec *rec;

   rec = xmalloc(sizeof(*rec));
   rec->ip = ip;
   if (RB_INSERT(tree_t, &ip_tree, rec) != NULL) {
      /* already queued */
      free(rec);
      return;
   }
   /* FIXME: will this block? */
   write(sock[PARENT], &ip, sizeof(ip));
}

static int pending(const int fd);
static void xread(const int d, void *buf, const size_t nbytes);

/*
 * Returns non-zero if result waiting, stores IP and name into given pointers
 * (name buffer is allocated by dns_poll)
 */
static int
dns_get_result(in_addr_t *ip, char **name)
{
   int len;

   if (!pending(sock[PARENT]))
      return (0);

   xread(sock[PARENT], ip, sizeof(*ip));
   xread(sock[PARENT], &len, sizeof(len));
   if (len == -1)
      return (0); /* resolution failed */

   *name = xmalloc(len+1);
   xread(sock[PARENT], *name, len);
   (*name)[len] = '\0';
   return (1);
}

void
dns_poll(void)
{
   in_addr_t ip;
   char *name;

   while (dns_get_result(&ip, &name)) {
      /* push into hosts_db */
      struct bucket *b = host_find(ip);
      if (b == NULL) {
         verbosef("resolved %s to %s but it's not in the DB!",
            ip_to_str(ip), name);
         return;
      }
      if (b->u.host.dns != NULL) {
         verbosef("resolved %s to %s but it's already in the DB!",
            ip_to_str(ip), name);
         return;
      }
      b->u.host.dns = name;
   }
}

/* ------------------------------------------------------------------------ */

struct qitem {
   STAILQ_ENTRY(qitem) entries;
   in_addr_t ip;
};

STAILQ_HEAD(qhead, qitem) queue = STAILQ_HEAD_INITIALIZER(queue);

static void
enqueue(const in_addr_t ip)
{
   struct qitem *i;

   i = xmalloc(sizeof(*i));
   i->ip = ip;
   STAILQ_INSERT_TAIL(&queue, i, entries);
   verbosef("DNS: enqueued %s", ip_to_str(ip));
}

/* Return non-zero and populate <ip> pointer if queue isn't empty. */
static int
dequeue(in_addr_t *ip)
{
   struct qitem *i;

   i = STAILQ_FIRST(&queue);
   if (i == NULL)
      return (0);
   STAILQ_REMOVE_HEAD(&queue, entries);
   *ip = i->ip;
   free(i);
   return 1;
}

static void
xread(const int d, void *buf, const size_t nbytes)
{
   ssize_t ret = read(d, buf, nbytes);

   if (ret == 0)
      errx(1, "read: end of file");
   if (ret == -1)
      err(1, "read");
   if (ret != (ssize_t)nbytes)
      err(1, "read %d bytes instead of all %d bytes", ret, nbytes);
}

static void
xwrite(const int d, const void *buf, const size_t nbytes)
{
   ssize_t ret = write(d, buf, nbytes);

   if (ret == -1)
      err(1, "write");
   if (ret != (ssize_t)nbytes)
      err(1, "wrote %d bytes instead of all %d bytes", ret, nbytes);
}

/* Returns non-zero if there is data to be read from <fd>. */
static int
pending(const int fd)
{
   fd_set rd;
   struct timeval timeout = { 0, 0 };

   FD_ZERO(&rd);
   FD_SET(fd, &rd);
   if (select(fd+1, &rd, NULL, NULL, &timeout) == -1)
      err(1, "select");
   return (FD_ISSET(fd, &rd));
}

/* Sleep on an FD until there is data to be read from it. */
static void
sleep_on(const int fd)
{
   fd_set rd;

   FD_ZERO(&rd);
   FD_SET(fd, &rd);
   if (select(fd+1, &rd, NULL, NULL, NULL) == -1)
      err(1, "select");
}

static void
dns_main(void)
{
   in_addr_t ip;
   ssize_t numread;

#ifdef HAVE_SETPROCTITLE
   setproctitle("DNS child");
#endif
   verbosef("DNS child is alive");
   for (;;) {
      if (STAILQ_EMPTY(&queue))
         sleep_on(sock[CHILD]);

      while (pending(sock[CHILD])) {
         /* FIXME: can this block? */
         numread = read(sock[CHILD], &ip, sizeof(ip));
         if (numread == 0)
            exit(0); /* end of file, nothing more to do here. */
         if (numread == -1)
            err(1, "DNS: read failed");
         if (numread != sizeof(ip))
            err(1, "DNS: read expecting %d bytes, got %d", sizeof(ip), numread);
         enqueue(ip);
      }

      /* Process queue. */
      if (dequeue(&ip)) {
         struct hostent *he;
         int len;

         he = gethostbyaddr((char *)&ip, sizeof(ip), AF_INET);
         if (he == NULL) {
            verbosef("DNS: gethostbyaddr(%s) failed: %s",
               ip_to_str(ip), hstrerror(h_errno));
            len = -1;
         } else
            len = strlen(he->h_name);

         /* Return result to parent. */
         xwrite(sock[CHILD], &ip, sizeof(ip));
         xwrite(sock[CHILD], &len, sizeof(len));
         if (len != -1)
            xwrite(sock[CHILD], he->h_name, len);
      }
   }
}

/* vim:set ts=3 sw=3 tw=78 expandtab: */
