/* darkstat 3
 * copyright (c) 2001-2006 Emil Mikulic.
 *
 * conv.c: convenience functions.
 *
 * Permission to use, copy, modify, and distribute this file for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "conv.h"

#include <assert.h>
#include <ctype.h>
#include "err.h"
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* malloc() that exits on failure. */
void *
xmalloc(const size_t size)
{
   void *ptr = malloc(size);

   if (ptr == NULL)
      errx(1, "malloc(): out of memory");
   return (ptr);
}

/* calloc() that exits on failure. */
void *
xcalloc(const size_t num, const size_t size)
{
   void *ptr = calloc(num, size);

   if (ptr == NULL)
      errx(1, "calloc(): out of memory");
   return (ptr);
}

/* realloc() that exits on failure. */
void *
xrealloc(void *original, const size_t size)
{
    void *ptr = realloc(original, size);

    if (ptr == NULL)
      errx(1, "realloc(): out of memory");
    return (ptr);
}

/* strdup() that exits on failure. */
char *
xstrdup(const char *s)
{
   char *tmp = strdup(s);

   if (tmp == NULL)
      errx(1, "strdup(): out of memory");
   return (tmp);
}

/* ---------------------------------------------------------------------------
 * Split string out of src with range [left:right-1]
 */
char *
split_string(const char *src, const size_t left, const size_t right)
{
    char *dest;
    assert(left <= right);
    assert(left < strlen(src));   /* [left means must be smaller */
    assert(right <= strlen(src)); /* right) means can be equal or smaller */

    dest = xmalloc(right - left + 1);
    memcpy(dest, src+left, right-left);
    dest[right-left] = '\0';
    return (dest);
}

/* ---------------------------------------------------------------------------
 * Uppercasify all characters in a string of given length.
 */
void
strntoupper(char *str, const size_t length)
{
    size_t i;

    for (i=0; i<length; i++)
        str[i] = toupper(str[i]);
}

/* ---------------------------------------------------------------------------
 * Returns non-zero if haystack starts with needle.
 */
int
str_starts_with(const char *haystack, const char *needle)
{
   int i = 0;

   while (needle[i] != '\0') {
      if ((haystack[i] == '\0') || (haystack[i] != needle[i]))
         return (0);
      i++;
   }
   return (1);
}

/* split - splits a string by a delimiter character into an array of
 * string chunks.
 *
 * The chunks and the array are dynamically allocated using xmalloc() so
 * it will errx() if it runs out of memory.
 *
 *    int num_chunks;
 *    char **chunks = split('.', "..one...two....", &num_chunks);
 *
 *    num_chunks = 2, chunks = { "one", "two", NULL }
 */
char **
split(const char delimiter, const char *str, int *num_chunks)
{
   int num = 0;
   char **chunks = NULL;
   size_t left, right = 0;

   #define PUSH(c) do { num++;  chunks = (char**) xrealloc(chunks, \
      sizeof(*chunks) * num);  chunks[num-1] = c; } while(0)

   for(;;) {
      /* find first non-delimiter */
      for (left = right; str[left] == delimiter; left++)
            ;

      if (str[left] == '\0')
         break; /* ran out of string */

      /* find first delimiter or end of string */
      for (right=left+1;
         str[right] != delimiter && str[right] != '\0';
         right++)
            ;

      /* split chunk out */
      PUSH( split_string(str, left, right) );

      if (str[right] == '\0')
         break; /* ran out of string */
      else
         right++;
   }

   /* return */
   PUSH(NULL);
   if (num_chunks != NULL)
      *num_chunks = num-1; /* NULL doesn't count */
   return (chunks);
   #undef PUSH
}

/* vim:set ts=3 sw=3 tw=78 expandtab: */
