{-# OPTIONS -fffi #-}
module External (
    copyFileOrUrl, copyFilesOrUrls,
    fetchFilePS, gzFetchFilePS,
    sendEmail, sendEmailDoc, resendEmail,
    signString, verifyPS,
    execPipeIgnoreError,
    getTermNColors,
    pipeDoc_SSH_IgnoreError,
    maybeURLCmd,
    Cachable(Cachable, Uncachable, MaxAge)
  ) where

import List ( intersperse )
import Monad ( liftM, when )
import System ( ExitCode(..), system, getEnv )
import IO ( hPutStr, hPutStrLn, hClose, hFlush, try, stdout )
import System.IO.Unsafe ( unsafePerformIO )
import Char ( toUpper )
import Foreign.C ( CString, withCString )
import Foreign.Ptr ( nullPtr )
#ifdef HAVE_CURSES
import Foreign.C ( CChar )
import Foreign.Ptr ( Ptr )
import Foreign.Marshal.Alloc (allocaBytes)
import Autoconf ( use_color )
#endif
import Workaround ( createLink )

import DarcsFlags ( DarcsFlag( Quiet, SignAs, Sign, SignSSL,
                               Verbose, Verify, VerifySSL ) )
import DarcsUtils ( withCurrentDirectory )
import FastPackedString ( PackedString, readFilePS, gzReadFilePS, writeFilePS,
                          hPutPS, unpackPS, linesPS, unlinesPS,
                          lengthPS, takePS, dropPS, packString,
                          nullPS, nilPS, concatPS
                        )
import Lock ( withTemp, withOpenTemp, readDocBinFile,
              canonFilename, writeBinFile, writeDocBinFile
            )
import CommandLine ( parseCmd, addUrlencoded )
import Autoconf ( have_libcurl, have_sendmail, have_mapi, sendmail_path, darcs_version )
import Curl ( copyUrl )
import Curl ( Cachable(..) )
import Exec ( exec )
import DarcsURL ( is_file, is_url, is_ssh )
import DarcsUtils ( catchall )
import Printer ( Doc, hPutDoc, hPutDocLn, ($$), (<+>), renderPS,
                 text, empty, packedString, vcat, renderString )
#include "impossible.h"


fetchFilePS :: String -> Cachable -> IO PackedString
fetchFilePS fou _ | is_file fou = readFilePS fou
fetchFilePS fou cache = withTemp $ \t -> do copyFileOrUrl fou t cache
                                            readFilePS t

gzFetchFilePS :: String -> Cachable -> IO PackedString
gzFetchFilePS fou _ | is_file fou = gzReadFilePS fou
gzFetchFilePS fou cache = withTemp $ \t-> do copyFileOrUrl fou t cache
                                             gzReadFilePS t


copyFileOrUrl :: FilePath -> FilePath -> Cachable -> IO ()
copyFileOrUrl fou out _     | is_file fou = copyLocal fou out
copyFileOrUrl fou out cache | is_url  fou = copyRemote fou out cache
copyFileOrUrl fou out _     | is_ssh  fou = copySSH fou out
copyFileOrUrl fou _   _     = fail $ "unknown transport protocol: " ++ fou

copyLocal  :: String -> FilePath -> IO ()
copyLocal fou out =
    createLink fou out `catchall` do c <- readFilePS fou
                                     writeFilePS out c

maybeURLCmd :: String -> String -> IO(Maybe(String))
maybeURLCmd what url =
  do let prot = map toUpper $ takeWhile (/= ':') url
     liftM Just (getEnv ("DARCS_" ++ what ++ "_" ++ prot))
             `catch` \_ -> return Nothing

copyRemote :: String -> FilePath -> Cachable -> IO ()
copyRemote u v cache =
    do maybeget <- maybeURLCmd "GET" u
       case maybeget of
         Nothing -> copyRemoteNormal u v cache
         Just get ->
           do let cmd = head $ words get
                  args = tail $ words get
              r <- exec cmd (args++[u]) "/dev/null" v
              when (r /= ExitSuccess) $
                  fail $ "(" ++ get ++ ") failed to fetch: " ++ u

copyRemoteNormal :: String -> FilePath -> Cachable -> IO ()
copyRemoteNormal u v cache = if have_libcurl
                       then Curl.copyUrl u v cache
                       else copyRemoteCmd u v

copySSH :: String -> FilePath -> IO ()
copySSH u f = do p <- try $ getEnv "SSH_PORT" -- or DARCS_SSH_PORT ?
                 scp_command <- getEnv "DARCS_SCP" `catch`
                                \_ -> return "scp"
                 let port = either (const []) (\x->["-P",x]) p
                     scp = head $ words scp_command
                     scp_args = tail $ words scp_command
                 r <- exec scp (scp_args++port++[escape_dollar u,f])
                      "/dev/null" "/dev/null"
                 when (r /= ExitSuccess) $
                      fail $ "(scp) failed to fetch: " ++ u
    where {- '$' in filenames is troublesome for scp, for some reason.. -}
          escape_dollar :: String -> String
          escape_dollar = concatMap tr
           where tr '$' = "\\$"
                 tr c = [c]


copyFilesOrUrls :: [DarcsFlag]->FilePath->[String]->FilePath->Cachable->IO ()
copyFilesOrUrls opts dou ns out _ | is_file dou = copyLocals opts dou ns out
copyFilesOrUrls opts dou ns out c | is_url  dou = copyRemotes opts dou ns out c
copyFilesOrUrls _ dou ns out _    | is_ssh  dou = copySSHs dou ns out
copyFilesOrUrls _ dou _  _   _    = fail $ "unknown transport protocol: "++dou


copyLocals :: [DarcsFlag] -> String -> [String] -> FilePath -> IO ()
copyLocals opts u ns d =
    doWithPatches opts (\n -> copyLocal (u++"/"++n) (d++"/"++n)) ns

copyRemotes :: [DarcsFlag] -> String -> [String] -> FilePath -> Cachable -> IO()
copyRemotes opts u ns d cache =
    do maybeget <- maybeURLCmd "GET" u
       maybemget <- maybeURLCmd "MGET" u
       case (maybeget, maybemget) of
         (Nothing, _) -> copyRemotesNormal opts u ns d cache
         (Just _, Nothing) -> doWithPatches opts (\n -> copyRemote (u++"/"++n) (d++"/"++n) cache) ns
         (Just _, Just mget) -> mgetRemotes mget u ns d

stringToInt :: String -> Int -> Int
stringToInt num def = case reads num of [(x,"")] -> x
                                        _ -> def

mgetRemotes :: String -> String -> [String] -> FilePath -> IO()
mgetRemotes _ _ [] _ = return ()
mgetRemotes mget u ns d = do
    mgetmax <- getEnv "DARCS_MGETMAX" `catch` \_ -> return ""
    let (nsnow, nslater) = splitAt (stringToInt mgetmax 200) ns
        cmd = head $ words mget
        args = tail $ words mget
        urls = map (\n -> u++"/"++n) nsnow
    withCurrentDirectory d $ do
        r <- exec cmd (args++urls) "/dev/null" "/dev/null"
        when (r /= ExitSuccess) $
            fail $ unlines $
                ["(" ++ mget ++ ") failed to fetch files.",
                     "source directory: " ++ d,
                     "source files:"] ++ (upto 5 nsnow) ++ 
                     ["still to go:"] ++ (upto 5 nslater)
    mgetRemotes mget u nslater d
    where
    upto :: Integer -> [String] -> [String]
    upto _ [] = []
    upto 0 l = [ "(" ++ (show (length l)) ++ " more)" ]
    upto n (h : t) = h : (upto (n - 1) t)

copyRemotesNormal :: [DarcsFlag] -> String -> [String] -> FilePath -> Cachable -> IO()
copyRemotesNormal opts u ns d cache =
    if have_libcurl
    then doWithPatches opts (\n -> copyRemote (u++"/"++n) (d++"/"++n) cache) ns
    else wgetRemotes u ns d

-- Argh, this means darcs get will fail if we don't have libcurl and don't
-- have wget.  :(
wgetRemotes :: String -> [String] -> FilePath -> IO ()
wgetRemotes u ns d = do wget_command <- getEnv "DARCS_WGET" `catch`
                                               \_ -> return "wget"
                        let wget = head $ words wget_command
                            wget_args = tail $ words wget_command
                            input = unlines $ map (\n -> u++"/"++n) ns
                        withCurrentDirectory d $ withOpenTemp $ \(th,tn) ->
                            do hPutStr th input
                               hClose th
                               r <- exec wget (wget_args++["-i",tn])
                                    "/dev/null" "/dev/null"
                               when (r /= ExitSuccess) $
                                    fail $ unlines $
                                             ["(wget) failed to fetch files.",
                                              "source directory: " ++ d,
                                              "source files:"] ++ ns

copySSHs :: String -> [String] -> FilePath -> IO ()
copySSHs u ns d = do sftp_command <- getEnv "DARCS_SFTP" `catch`
                                     \_ -> return "sftp"
                     let sftp = head $ words sftp_command
                         sftp_args = tail $ words sftp_command
                         path = drop 1 $ dropWhile (/= ':') u
                         host = (takeWhile (/= ':') u)++":"
                         cd = "cd "++path++"\n"
                         input = cd++(unlines $ map ("get "++) ns)
                     withCurrentDirectory d $ withOpenTemp $ \(th,tn) ->
                         do hPutStr th input
                            hClose th
                            r <- exec sftp (sftp_args++["-b",tn,host])
                                 "/dev/null" "/dev/null"
                            when (r /= ExitSuccess) $
                                 fail $ unlines $
                                          ["(sftp) failed to fetch files.",
                                           "source directory: " ++ d,
                                           "source files:"] ++ ns


copyRemoteCmd :: String -> FilePath -> IO ()
copyRemoteCmd s tmp = do
    let cmd = get_ext_cmd
    r <- stupidexec (cmd tmp s) "/dev/null" "/dev/null"
    when (r /= ExitSuccess) $
         fail $ "failed to fetch: " ++ s ++" " ++ show r
    where stupidexec (c:args) inf outf = exec c args inf outf
          stupidexec [] _ _ = bug "stupidexec without a command"

doWithPatches :: [DarcsFlag] -> (String -> IO ()) -> [String] -> IO ()
doWithPatches opts f patches =
    if Quiet `elem` opts
    then sequence_ $ map f patches
    else doWithCount patches 1
        where verbose = Verbose `elem` opts
              total   = show $ length patches

              doWithCount :: [String] -> Int -> IO ()
              doWithCount (p:ps) index = do
                  putStr $ if verbose then "Copying patch "++(show index)++" of "++total++": "++p++"\n"
                           else "\rCopying patch "++(show index)++
                                " of "++total++"..."
                  hFlush stdout
                  f p
                  doWithCount ps (index + 1)
              doWithCount [] index = do
                  putStr $ if verbose then ""
                           else "\rCopying patch "++(show $ index - 1)++
                                " of "++total++"... done!\n"
                  hFlush stdout

{-# NOINLINE get_ext_cmd #-}
get_ext_cmd :: String -> String -> [String]
-- Only need to find the command once..
get_ext_cmd = unsafePerformIO get_ext_cmd'

-- Would be better to read possible command lines from config-file..
get_ext_cmd' :: IO (String -> String -> [String])
get_ext_cmd' = try_cmd cmds
    where cmds = [("wget", (("--version",0),
                          -- use libcurl for proper cache control
                          \t s -> ["wget", "-q",
                                   "--header=Pragma: no-cache",
                                   "--header=Cache-Control: no-cache",
                                   "-O",t,s])),
                  ("curl", (("--version",2),
                            \t s -> ["curl", "-s", "-L",
                                     "-H", "Pragma: no-cache",
                                     "-H", "Cache-Control: no-cache",
                                     "-o",t,s]))]
          try_cmd [] = fail $ "I need one of: " ++ cs
              where cs = concat $ intersperse ", " (map fst cmds)
          try_cmd ((c,(ok_check,f)):cs) = do
            True <- can_execute ok_check c
            return f
           `catch` (\_ -> try_cmd cs)


pipeDoc_SSH_IgnoreError :: [String] -> Doc -> IO Doc
pipeDoc_SSH_IgnoreError args input =
    do p <- try $ getEnv "SSH_PORT" -- or DARCS_SSH_PORT ?
       ssh_command <- getEnv "DARCS_SSH" `catch`
                      \_ -> return "ssh"
       let port = either (const []) (\x->["-p",x]) p
           ssh = head $ words ssh_command
           ssh_args = tail $ words ssh_command
       execPipeIgnoreError ssh (ssh_args++port++args) input


sendEmail :: String -> String -> String -> String -> String -> String -> IO ()
sendEmail _ "" _ "" _ _ = return ()
sendEmail f "" s cc scmd body = sendEmail f cc s "" scmd body
sendEmail f t s cc scmd body =
  if have_sendmail || scmd /= "" then do
    withOpenTemp $ \(h,fn) -> do
     hPutStr h $
        "To: "      ++ t ++ "\n" ++
        "From: "    ++ f ++ "\n" ++
        "Subject: " ++ s ++ "\n" ++
        formated_cc ++
        "X-Mail-Originator: Darcs Version Control System\n" ++
        "X-Darcs-Version: " ++ darcs_version ++ "\n" ++
        body
     hClose h
     let ftable = [('t',t),('c',cc),('t',fn),('f',f),('s',s),('b',body)]
     r <- execSendmail ftable scmd fn
     when (r /= ExitSuccess) $ fail ("failed to send mail to: " ++ t)
  else if have_mapi then do
     r <- withCString t $ \tp ->
           withCString f $ \fp ->
            withCString cc $ \ccp ->
             withCString s $ \sp ->
              withOpenTemp $ \(h,fn) -> do
               hPutStr h body
               hClose h
               writeBinFile "mailed_patch" body
               cfn <- canonFilename fn
               withCString cfn $ \pcfn ->
                c_send_email fp tp ccp sp nullPtr pcfn
     when (r /= 0) $ fail ("failed to send mail to: " ++ t)
  else fail $ "no mail facility (sendmail or mapi) located at configure time!"
  where formated_cc = if cc == ""
                      then ""
                      else "Cc: "++cc++"\n"

sendEmailDoc :: String -> String -> String -> String -> String
             -> Doc -> Doc -> Doc -> IO ()
sendEmailDoc _ "" _ "" _ _ _ _ = return ()
sendEmailDoc f "" s cc scmd content bundle body =
  sendEmailDoc f cc s "" scmd content bundle body
sendEmailDoc f t s cc scmd content bundle body =
  if have_sendmail || scmd /= "" then do
    withOpenTemp $ \(h,fn) -> do
     hPutDocLn h $
           text "To:"      <+> text t
        $$ text "From:"    <+> text f
        $$ text "Subject:" <+> text s
        $$ formated_cc
        $$ text "X-Mail-Originator: Darcs Version Control System"
        $$ text ("X-Darcs-Version: " ++ darcs_version)
        $$ body
     hClose h
     withOpenTemp $ \(hat,at) -> do
       hPutDocLn hat $ bundle
       hClose hat
       let ftable = [('t',t),('c',cc),('f',f),('s',s),
                     ('b',renderString content), ('a', at)]
       r <- execSendmail ftable scmd fn
       when (r /= ExitSuccess) $ fail ("failed to send mail to: " ++ t)
  else if have_mapi then do
     r <- withCString t $ \tp ->
           withCString f $ \fp ->
            withCString cc $ \ccp ->
             withCString s $ \sp ->
              withOpenTemp $ \(h,fn) -> do
               hPutDoc h body
               hClose h
               writeDocBinFile "mailed_patch" body
               cfn <- canonFilename fn
               withCString cfn $ \pcfn ->
                c_send_email fp tp ccp sp nullPtr pcfn
     when (r /= 0) $ fail ("failed to send mail to: " ++ t)
  else fail $ "no mail facility (sendmail or mapi) located at configure time!"
  where formated_cc = if cc == ""
                      then empty
                      else text "Cc:" <+> text cc

resendEmail :: String -> String -> PackedString -> IO ()
resendEmail "" _ _ = return ()
resendEmail t scmd body =
  case (have_sendmail || scmd /= "", have_mapi) of
   (True, _) -> do
    withOpenTemp $ \(h,fn) -> do
     hPutStrLn h $ "To: "++ t
     hPutStrLn h $ find_from (linesPS body)
     hPutStrLn h $ find_subject (linesPS body)
     hPutDocLn h $ fixit $ linesPS body
     hClose h
     let ftable = [('t',t)]
     r <-  execSendmail ftable scmd fn
     when (r /= ExitSuccess) $ fail ("failed to send mail to: " ++ t)
   (_, True) -> fail "Don't know how to resend email with MAPI"
   _ -> fail $ "no mail facility (sendmail or mapi) located at configure time (use the sendmail-command option)!"
  where br = packString "\r"
        darcsurl = packString "DarcsURL:"
        content = packString "Content-"
        from_start = packString "From:"
        subject_start = packString "Subject:"
        fixit (l:ls)
         | nullPS l = packedString nilPS $$ vcat (map packedString ls)
         | l == br = packedString nilPS $$ vcat (map packedString ls)
         | takePS 9 l == darcsurl || takePS 8 l == content
            = packedString l $$ fixit ls
         | otherwise = fixit ls
        fixit [] = empty
        find_from (l:ls) | takePS 5 l == from_start = unpackPS l
                         | otherwise = find_from ls
        find_from [] = "From: unknown"
        find_subject (l:ls) | takePS 8 l == subject_start = unpackPS l
                            | otherwise = find_subject ls
        find_subject [] = "Subject: (no subject)"

execSendmail :: [(Char,String)] -> String -> String -> IO ExitCode
execSendmail ftable scmd fn =
  if scmd == "" then
     exec sendmail_path ["-i", "-t"] fn "/dev/null"
  else case parseCmd (addUrlencoded ftable) scmd of
         Right (arg0:opts, wantstdin) ->
           do let stdin = if wantstdin then fn else "/dev/null"
              exec arg0 opts stdin "/dev/nulll"
         Left e -> fail $ ("failed to send mail, invalid sendmail-command: "++(show e))
         _ -> fail $ ("failed to send mail, invalid sendmail-command")

#ifdef HAVE_MAPI
foreign import ccall "win32/send_email.h send_email" c_send_email
#else
c_send_email
#endif
             :: CString -> {- sender -}
                CString -> {- recipient -}
                CString -> {- cc -}
                CString -> {- subject -}
                CString -> {- body -}
                CString -> {- path -}
                IO Int
#ifndef HAVE_MAPI
c_send_email = impossible
#endif

execPSPipe :: String -> [String] -> PackedString -> IO PackedString
execPSPipe c args ps = liftM renderPS
                     $ execDocPipe c args
                     $ packedString ps

execDocPipe :: String -> [String] -> Doc -> IO Doc
execDocPipe c args instr =
    withOpenTemp $ \(th,tn) -> do
      hPutDoc th instr
      hClose th
      withTemp $ \on -> do
        rval <- exec c args tn on
        case rval of
          ExitSuccess -> readDocBinFile on
          ExitFailure ec -> fail $ "External program '"++c++
                            "' failed with exit code "++ show ec

-- The following is needed for diff, which returns non-zero whenever
-- the files differ.
execPipeIgnoreError :: String -> [String] -> Doc -> IO Doc
execPipeIgnoreError c args instr =
    withOpenTemp $ \(th,tn) -> do
      hPutDoc th instr
      hClose th
      withTemp $ \on -> do exec c args tn on
                           readDocBinFile on

signString :: [DarcsFlag] -> Doc -> IO Doc
signString [] d = return d
signString (Sign:_) d = signPGP [] d
signString (SignAs keyid:_) d = signPGP ["--local-user", keyid] d
signString (SignSSL idf:_) d = signSSL idf d
signString (_:os) d = signString os d

signPGP :: [String] -> Doc -> IO Doc
signPGP args t = execDocPipe "gpg" ("--clearsign":args) t

signSSL :: String -> Doc -> IO Doc
signSSL idfile t =
    withTemp $ \cert -> do
    opensslPS ["req", "-new", "-key", idfile,
               "-outform", "PEM", "-days", "365"]
                (packString "\n\n\n\n\n\n\n\n\n\n\n")
                >>= opensslPS ["x509", "-req", "-extensions",
                               "v3_ca", "-signkey", idfile,
                               "-outform", "PEM", "-days", "365"]
                >>= opensslPS ["x509", "-outform", "PEM"]
                >>= writeFilePS cert
    opensslDoc ["smime", "-sign", "-signer", cert,
                "-inkey", idfile, "-noattr", "-text"] t
    where opensslDoc = execDocPipe "openssl"
          opensslPS = execPSPipe "openssl"


verifyPS :: [DarcsFlag] -> PackedString -> IO (Maybe PackedString)
verifyPS [] ps = return $ Just ps
verifyPS (Verify pks:_) ps = verifyGPG pks ps
verifyPS (VerifySSL auks:_) ps = verifySSL auks ps
verifyPS (_:os) ps = verifyPS os ps

verifyGPG :: FilePath -> PackedString -> IO (Maybe PackedString)
verifyGPG goodkeys s =
    withOpenTemp $ \(th,tn) -> do
      hPutPS th s
      hClose th
      rval <- exec "gpg"  ["--batch","--no-default-keyring",
                           "--keyring",goodkeys, "--verify"] tn "/dev/null"
      case rval of
          ExitSuccess -> return $ Just gpg_fixed_s
          _ -> return Nothing
      where gpg_fixed_s = unlinesPS $ map fix_line $ tail $
                          dropWhile (/= packString "-----BEGIN PGP SIGNED MESSAGE-----") $ linesPS s
            fix_line x | lengthPS x < 3 = x
                       | takePS 3 x == packString "- -" = dropPS 2 x
                       | otherwise = x

verifySSL :: FilePath -> PackedString -> IO (Maybe PackedString)
verifySSL goodkeys s = do
    certdata <- opensslPS ["smime", "-pk7out"] s
                >>= opensslPS ["pkcs7", "-print_certs"]
    cruddy_pk <- opensslPS ["x509", "-pubkey"] certdata
    let key_used = concatPS $ tail $
                   takeWhile (/= packString"-----END PUBLIC KEY-----")
                           $ linesPS cruddy_pk
        in do allowed_keys <- linesPS `liftM` readFilePS goodkeys
              if not $ key_used `elem` allowed_keys
                then return Nothing -- Not an allowed key!
                else withTemp $ \cert ->
                     withTemp $ \on ->
                     withOpenTemp $ \(th,tn) -> do
                     hPutPS th s
                     hClose th
                     writeFilePS cert certdata
                     rval <- exec "openssl" ["smime", "-verify", "-CAfile",
                                             cert, "-certfile", cert] tn on
                     case rval of
                       ExitSuccess -> Just `liftM` readFilePS on
                       _ -> return Nothing
    where opensslPS = execPSPipe "openssl"

can_execute :: (String,Int) -> String -> IO Bool
can_execute (arg,expected_return_value) exe = do
 withTemp $ \junk -> do
  ec <- system (unwords [exe,arg,">",junk])
  case ec of
    ExitSuccess | expected_return_value == 0 -> return True
    ExitFailure r | r == expected_return_value -> return True
    _ -> return False


{-
  - This function returns number of colours supported by current terminal
  - or -1 if colour output not supported or error occured.
  - Terminal type determined by TERM env. variable.
  -}
getTermNColors :: IO Int

#ifdef HAVE_CURSES

foreign import ccall "term.h tgetnum" c_tgetnum :: CString -> IO Int
foreign import ccall "term.h tgetent" c_tgetent :: Ptr CChar -> CString -> IO Int

termioBufSize :: Int
termioBufSize = 4096

getTermNColors = if not use_color
                 then return (-1)
                 else do term <- getEnv "TERM"
                         allocaBytes termioBufSize (getTermNColorsImpl term)
                      `catch` \_ -> return (-1)

getTermNColorsImpl :: String -> Ptr CChar -> IO Int
getTermNColorsImpl term buf = do rc <- withCString term $
                                       \termp -> c_tgetent buf termp
                                 if (rc /= 1) then return (-1)  else withCString "Co" $ \capap -> c_tgetnum capap

#else

getTermNColors = return (-1)

#endif

