%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\documentclass{report}
\usepackage{color}

\usepackage{verbatim}
\newenvironment{code}{\comment}{\endcomment}
% \newenvironment{code}{\color{blue}\verbatim}{\endverbatim}

\begin{document}

\begin{code}
module Main (main) where

import System
import IO
import Monad ( when )

import Patch ( Patch )
import SlurpDirectory
import DarcsCommands
import TheCommands
import External

darcs_version = "0.9.12"
\end{code}

% Definition of title page:
\title{
    Darcs \haskell{darcs_version}\\
{\Large \it David's advanced revision control system}
}
\author{
    David Roundy
}

\maketitle

\tableofcontents

\chapter{Introduction}

Darcs is a revision control system, along the lines of CVS or arch.  That
means that it keeps track of various revisions and branches of your
project, allows for changes to propogate from one branch to another.  Darcs
is intended to be an ``advanced'' revision control system.  Darcs has two
particularly distinctive features which differ from other revision control
systems: 1) each copy of the source is a fully functional branch, and 2)
underlying darcs is a consistent and powerful theory of patches.

\paragraph{Every source tree a branch}
The primary simplifying notion of darcs is that \emph{every} copy of your
source code is a full repository. This is dramatically different from CVS,
in which the normal usage is for there to be one central repository from
which source code will be checked out. It is closer to the notion of arch,
since the `normal' use of arch is for each developer to create his own
repository. However, darcs makes it even easier, since simply checking out
the code is all it takes to create a new repository. This has several
advantages, since you can harness the full power of darcs in any scratch
copy of your code, without committing your possibly destabilizing changes to
a central repository.

\paragraph{Theory of patches}
The development of a simplified theory of patches is what originally
motivated me to create darcs. This patch formalism means that darcs patches
have a set of properties, which make possible manipulations that couldn't be
done in other revision control systems. First, every patch is invertible.
Secondly, sequential patches (i.e. patches that are created in sequence, one
after the other) can be reordered, although this reordering can fail, which
means the second patch is dependent on the first. Thirdly, patches which are
in parallel (i.e. both patches were created by modifying identical trees)
can be merged, and the result of a set of merges is independent of the order
in which the merges are performed. This last property is critical to darcs'
philosophy, as it means that a particular version of a source tree is fully
defined by the list of patches that are in it. i.e. there is no issue
regarding the order in which merges are performed. For a more thorough
discussion of darcs' theory of patches, see Appendix~\ref{Patch}.

\paragraph{A simple advanced tool}
Besides being ``advanced'' as discussed above, darcs is actually also quite
simple. Versioning tools can be seen as three layers. At the foundation is
the ability to manipulate changes. On top of that must be placed some kind
of database system to keep track of the changes. Finally, at the very top is
some some sort of distribution system for getting changes from one place to
another.

Really, only the first of these three layers is of particular interest to
me, so the other two are done as simply as possible.  At the database
layer, darcs just has an ordered list of patches along with the patches
themselves, each stored as an individual file.  Darcs' distribution system
is strongly inspired by that of arch.  Like arch, darcs uses a dumb server,
typically apache or just a local or network file system.  Unlike arch,
darcs currently has no write ability to a remote file system.  This means
that darcs currently only supports for ``pulling'' of patches from a remote
repository to a local one, but not ``pushing'' of patches.  While this does
simplify matters by eliminating issues of user permissions, it isn't really
adequate, as it doesn't address the needs of users who lack a server with a
permanent net connection to host their repositories.  I do have plans for
supporting a push mechanism (which, by the way will be accelerated if I
hear there is a demand for such a thing, as I personally have no use for
it).

\paragraph{Keeping track of changes rather than versions}

In the last paragraph, I explained revision control systems in terms of
three layers.  One can also look at them as having two distinct uses.  One
is to provide a history of previous versions.  The other is to keep track
of changes that are made to the repository, and to allow these changes to
be merged and moved from one repository to another.  These two uses are
distinct, and almost orthogonal, in the sense that a tool can support one
of the two uses optimally while providing no support for the other.  Darcs
is not intended to maintain a history of versions, although it is possible
to kludge together such a revision history, either by making each new patch
depend on all previous patches, or by tagging regularly.  In a sense, this
is what the tag feature is for, but the intention is that tagging will be
used only to mark particularly notable versions (e.g. released versions, or
perhaps versions that pass a time consuming test suite).

As I understand them (and I certainly may be wrong), previous revision
control systems originated with their purpose being to keep track of a
history of versions, with the ability to merge changes being added as it
was seen that this would be desirable.  But the fundamental object remained
the versions themselves.

In such a system, a patch (I am using patch here to mean an encapsulated
set of changes) is uniquely determined by two trees.  Merging changes that
are in two trees consists of finding a common parent tree, computing the
diffs of each tree with their parent, and then cleverly combining those two
diffs and applying the combined diff to the parent tree, possibly at some
point in the process allowing human intervention, to allow for fixing up
problems in the merge such as conflicts.

Finding this parent tree poses problems.  This is where DAGs (Directed
Acyclic Graphs) come in.  A DAG is a convenient way to assure that two
trees (tree here meaning a source tree) have just one closest parent.  This
means that need keep track of the relationships of the trees, a most
imposing task, as the number of trees becomes large!  It also may put an
artificial constraint on the users, by not allowing them to create cyclic
loops of relationships between their versions.  Since I don't particularly
understand these DAGs (or, for that matter, other revision control
systems), I'll leave it here and just trust those who have gone before me
that they are difficult.

In the world of darcs, the source tree is \emph{not} the fundamental
object, but rather the patch is the fundamental object.  Rather than a
patch being defined in terms of the difference between two trees, a tree is
defined as the result of applying a given set of patches to an empty tree.
Moreover, these patches may be reordered (unless there are dependencies
between the patches involved) without changing the tree.  Thus there is no
need to find a common parent when performing a merge.  Or, if you like,
their common parent is defined by the set of common patches, and may not
correspond to any version in the version history (if we kept track of a
history).

One useful consequence of darcs' patch-oriented philosophy is that since a
patch need not be uniquely defined by a pair of trees (old and new), we can
have several ways of representing the same change, which differ only in how
they commute and what the result of merging them is.  Of course, creating
such a patch will require some sort of user input.  This is a Good Thing,
since the user \emph{creating} the patch should be the one forced to think
about what they really want to change, rather than the user merging the
patch.  An example of this is the token replace patch (See
Section~\ref{token_replace}).  This feature make it possible to create a
patch, for example, which changes every instance of the variable
``stupidly\_named\_var'' with ``better\_var\_name'', while leaving
``other\_stupidly\_named\_var'' untouched.  When this patch is merged with
any other patch involving the ``stupidly\_named\_var'', that instance will
also be modified to ``better\_var\_name''.  This is in contrast to a more
conventional merging method which would not only fail to change new
instances of the variable, but would also involves conflicts when merging
with any patch that modifies lines containing the variable.  By more using
additional information about the programmer's intent, darcs is thus able to
make the process of changing a variable name the trivial task that it
really is, which is really just a trivial search and replace, modulo
tokenizing the code appropriately.

The patch formalism discussed in Appendix~\ref{Patch} is what makes darcs'
approach possible.  In order for a tree to consist of a set of patches,
there must be a deterministic merge of any set patches, regardless of the
order in which they must be merged.  This requires that one be able to
reorder patches.  While I don't know that the patches are required to be
invertible as well, my implementation certainly requires inveribility.  In
particular, invertibility is required to make use of
Theorem~\ref{merge_thm}, which is used extensively in the manipulation of
merges.

To summarize, I believe that darcs has a solid theoretical foundation
leagues beyond what anyone else has developed.  On the down side, darcs is
currently slow, possibly still buggy---certainly still buggy in its
entirety, but I mean to say that its core may still be buggy.  Moreover it
is lacking an abundance of features.  However, I believe that the theory
behind darcs will be the foundation of the next generation of revision
control system.

\input{building_darcs.tex}

\chapter{Getting started}

This chapter will lead you through an example use of darcs, which hopefully
will allow you to get started using darcs with your project.

\section{Creating your repository}

Creating your repository in the first place just involves telling darcs to
create the special directory (called {\tt \_darcs}) in your project tree,
which will hold the revision information.  This is done my simply calling
from the root directory of your project:
\begin{verbatim}
% cd my_project/
% darcs inittree
\end{verbatim}
This creates the {\tt \_darcs} directory and populates it with whatever
files and directories are needed to describe an empty project.  You now
need to tell darcs what files and directories in your project should be
under revision control.  You do this using the command {\tt darcs add}:
\begin{verbatim}
% darcs add *.c Makefile.am configure.in
\end{verbatim}
When you have added all your files (or at least, think you have), you will
want to record your changes.  ``Recording'' always includes adding a note
as to why the change was made, or what it does.  In this case, we'll just
note that this is the initial version.
\begin{verbatim}
% darcs record --all
What is the patch name? Initial revision.
\end{verbatim}
Note that since we didn't specify a patch name on the command line we were
prompted for one.  If the environment variable `EMAIL' isn't set, you will
also be prompted for your email address.  Each patch that is recorded is
given a unique identifier consisting of the patch name, its creator's email
address, and the date when it was created.

\section{Making changes}

Now that we have created our repository, make a change to one or more of
your files.  After making the modification run:
\begin{verbatim}
% darcs whatsnew
\end{verbatim}
This should show you the modifications that you just made, in the darcs
patch format.  If you prefer to see your changes in a different format,
read Section~\ref{whatsnew}, which describs the whats-new command in
detail.

Let's say you have now made a change to your project.  The next thing to do
is to record a patch.  Recording a patch consists of grouping together a
set of related changes, and giving them a name.  It also tags the patch
with the date it was recorded and your email address.

To record a patch simply type:
\begin{verbatim}
% darcs record
\end{verbatim}
darcs will then prompt you with all the changes that you have made that
have not yet been recorded, asking you which ones you want to include in
the new patch.  Finally, darcs will ask you for a name for the patch.

You can now rerun whatsnew, and see that indeed the changes you have
recorded are no longer marked as new.

\section{Making your repository visible to others}
How do you let the world know about these wonderful changes?  Obviously,
they must be able to see your repository.  Currently the easiest way to do
this is typically by http using any web server.  The recommended way to do
this (using apache in a UNIX environment) is to create a directory called
{\tt /var/www/repos}, and then put a symlink to your repo there.

As long as you're running a web browser and making your repo available to
the world, you may as well make it easy for people to see what changes
you've made.  You can do this by running \verb!make installserver!, which
installs the program {\tt darcs\_cgi} at {\tt /usr/lib/cgi-bin/darcs}.  You
also will need to create a cache directory named
\verb!/var/cache/darcs_cgi!, and make sure the owner of that directory is
the same user that your web browser runs as its cgi scripts as.  For me,
this is www-data.  Now your friends and enemies should be able to easily
browse your repos by pointing their web browsers at {\tt
  http://your.server.org/cgi-bin/darcs}.  You can read more about this
interface in Chapter~\ref{web_interface}.

\section{Getting changes made to another repository}
Ok, so I can now browse your repository using their web browsers\ldots so
what? How do I get your changes into \emph{my} repository, where they can
do some good? It couldn't be easier.  I just {\tt cd} into my repository,
and there type:
\begin{verbatim}
% darcs pull http://your.server.org/repos/yourproject
\end{verbatim}
Darcs will check to see if you have recorded any changes that aren't in my
current repository.  If so, it'll prompt me for each one, to see which ones
I want to add to my repository.  Note that you may see a different series
of prompts depending your answers, since sometimes one patch depends on
another, so if you answer yes to the first one, you won't be prompted for
the second if the first depends on it.

Of course, maybe I don't even have a copy of your repository.  In that case
I'd want to do a
\begin{verbatim}
% darcs get --verbose http://your.server.org/repos/yourproject
\end{verbatim}
which gets the whole repo.

\section{Creating a pushable repository}

The darcs patcher (for a few more details see Chapter~\ref{darcs-patcher})
is a program that allows you to set up a repository to which you can
remotely ``push'' patches.  Although this isn't necesary to use darcs, it
can be useful if you want to give more than one person write access to a
repository, or if you do much of your work using a computer on which it is
not convenient to run a web server.  The patcher receives patches via
cryptographically signed email.  For each repository you create on a
machine a user is created which determines the email address for that
repository.  Alas, the patcher is considerably harder to set up than darcs
itself, which is why this section of the manual is here.

\subsection{Installing necesary programs}

To install the patcher program, you can either install the ``darcs-server''
package (via debian or whatever), or run ``make installserver''.  This
hopefully will put the programs in the right places.

To use a pushable repository, you also must install the following programs:
sudo, gnupg, a mailer configured to receive mail (e.g.  exim, sendmail or
postfix), and a web server (usually apache).  If you want to be able to
browse your repository via the web you must also configure your web server
to run cgi scripts and make sure the darcs cgi script was properly
installed.

\subsection{Setting up a repository}

To create a repository, as root run the `\verb!darcs-createrepo!'.  You
will be prompted for the email address of the repository and the location
of an existing copy of the repository.  If your desired email is
``myproject@my.url'', this will create a user named ``myproject'' with a
home directory of \verb!/var/lib/darcs/repos/myproject!. FIXME:  I have no
idea if the darcs-createrepo program will even run on any system other than
debian.  Success reports would be appreciated (or of course bug reports if
it fails).

The ``myproject'' user will be configured to run the darcs patcher on any
emails it receives.  However, the patcher will bounce any emails which
aren't signed by a key in the
\verb!/var/lib/darcs/repos/myproject/allowed_keys! gpg keyring (which is
empty).  To give yourself access to this repository you will need to create
a gpg key.  If you don't know about public key cryptography, take a look at
the gnupg manual.

\subsection{Granting access to a repository}

You create your gpg key by running (as your normal user):
\begin{verbatim}
% gpg --gen-key
\end{verbatim}
You will be prompted for your name and email address, among other options.
To add your public key to the allowed keys keyring.  Of course, you can
skip this step if you already have a gpg key you wish to use.

You now need to export the public key so we can tell the patcher about it.
You can do this will the following comman (again as your normal user):
\begin{verbatim}
% gpg --export "email@address" > /tmp/exported_key
\end{verbatim}
And now we can add your key to the \verb!allowed_keys!:
\begin{verbatim}
(as root)> gpg --keyring /var/lib/darcs/repos/myproject/allowed_keys \
               --no-default-keyring --import /tmp/exported_key
\end{verbatim}
You can repeat this process any number of times to authorize multiple users
to push to the repository.

You should now be able to push a patch to the repository by running as your
normal user, in a working copy of the repository:
\begin{verbatim}
% darcs push --sign http://your.computer/repos/myproject
\end{verbatim}
You may want to add ``push sign'' to the file \verb!_darcs/prefs/defaults!
so that you won't need to type \verb!--sign! every time you want to
push...

\chapter{Darcs commands}

\input{DarcsCommands.lhs}

\begin{code}
main = do
  initCurl
  argv <- getArgs;
  if length argv < 1
   then do
       progname <- getProgName
       putStr $ usage the_commands
       exitWith $ ExitFailure 2
   else return () -- null operation
\end{code}
Calling darcs with just ``{\tt--help}'' as an argument gives a brief
summary of what commands are available.
\begin{code}
  when (length argv == 1 && argv!!0 == "--help") $
       do putStr $ usage the_commands
          exitWith $ ExitSuccess
\end{code}
Calling darcs with the flag ``{\tt--version}'' tells you the version of
darcs you are using.
\begin{code}
  when (length argv == 1 && argv!!0 == "--version") $
       do putStr $ darcs_version ++ "\n"
          exitWith $ ExitSuccess
\end{code}
Similarly calling darcs with only ``{\tt--commands}'' gives a simple list
of available commands.  This latter arrangement is primarily intended for
the use of command-line autocompletion facilities, as are available in
bash.
\begin{code}
  when (length argv == 1 && argv!!0 == "--commands") $
       do is_valid <- sequence $ map command_prereq the_commands
          putStr $ unlines $ map (command_name . fst) $
                   filter (snd) $ zip the_commands is_valid
          putStr "--help\n"
          putStr "--version\n"
          exitWith $ ExitSuccess
  run_the_command (head argv) (tail argv) the_commands
\end{code}



\begin{code}
run_the_command :: String -> [String] -> [DarcsCommand] -> IO ()
run_the_command cmd args [] = do
    putStr "Unrecognized command!\n\n"
    putStr $ usage the_commands
    exitWith $ ExitFailure 2
run_the_command cmd args (c:cs)
    | cmd == command_name c = do
        is_good <- command_prereq c
        if is_good
           then run_command c args
           else do putStr $ "Can't run "++cmd++" right now...\n"
                   run_command c ["--help"]
                   exitWith $ ExitFailure 1
    | otherwise = run_the_command cmd args cs
\end{code}

\input{InitTree.lhs}

\input{Add.lhs}

\input{Remove.lhs}

\input{WhatsNew.lhs}

\input{Check.lhs}

\input{Record.lhs}

\input{Get.lhs}

\input{Pull.lhs}

\input{Push.lhs}

\input{Apply.lhs}

\input{Unrecord.lhs}

\input{Revert.lhs}

\input{Dist.lhs}

\input{Mv.lhs}

\input{Replace.lhs}

\input{Tag.lhs}

\input{DiffCommand.lhs}

\input{SetPref.lhs}

%%%%%%%%%% Chapter on the web interface %%%%%%%%%%%%

\input{darcs_cgi.lhs}

\input{darcs-patcher.lhs}

\appendix

\input{Patch.lhs}

\input{Repository.lhs}

\input{gpl.tex}

\end{document}

