%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{The Darcs Patcher}
\label{darcs-patcher}

\begin{code}
module Main (main) where

import System ( exitWith, ExitCode(..), system )
import IO ( hGetContents, stdin, hPutStr, hClose )
import Directory ( setCurrentDirectory, removeFile, doesFileExist )
import Monad ( liftM )
import Lock ( withTemp, withOpenTemp )
\end{code}

The \verb!darcs-patcher! E-mail-based server allows you to (easily?) set up
a centralized darcs repository to which multiple users can commit patches
without giving those users accounts on the server machine.
\verb!darcs-patcher! authentication is performed using gnupg signatures.
Users use \verb!darcs push --sign! to email a signed patch to the
repository.  \verb!darcs-patcher! verifies that the patch is signed by an
authorized user.  It then runs the \verb!darcs_test!, if available, to make
sure the patch doesn't break anything (in which case it would reject it),
and applies the patch to the repo.

\begin{code}
main = withOpenTemp $ \ (patchh,patchname) ->
  withTemp $ \ outputname -> do
  my_repodir <- return "repo"
  setCurrentDirectory my_repodir
  mail <- hGetContents stdin
  hPutStr patchh $ strip_out_patch mail
  hClose patchh
  retval <- system $
            "darcs apply --no-resolve-conflicts " ++
            "--verify ~/allowed_keys " ++ patchname ++
            " &> " ++ outputname
  if retval == ExitSuccess
    then exitWith ExitSuccess
    else do putStr "I had a problem applying the patch."
            putStr "Here is the error text:\n"
            ers <- readFile outputname
            putStr ers
            exitWith $ ExitFailure 1
\end{code}

To use \verb!darcs-patcher!, you first need to creat a user for its use
(one user per repository, I'm afraid).  In this user's home directory,
create a repository named ``repo''.  You have to be a bit careful when
creating this repo since if you just run a darcs get on a local repository
owned by a different user, darcs will use hard links for the patch files,
which may not work since you won't have the right to change them later.
This is a bug in darcs get (FIXME).

You next need to set up a gnupg keyring named \verb!allowed_keys! in the
\verb!.gnupg/! directory containing the gnupg public keys of all users
authorized to commit to the repository.  Finally, you create a
\verb!.forward! file containing the single line
``\verb!|/usr/sbin/darcs-patcher!'' (adjusted for where you actually
install \verb!darcs-patcher!), which tells your mailer to pipe any mail
that user receives through \verb!darcs-patcher!.  Note also that darcs
itself must reside in the system path.

\begin{code}
strip_out_patch :: String -> String
strip_out_patch s =
    (unlines $ dropWhile (/= "-----BEGIN PGP SIGNED MESSAGE-----") $
     takeWhile (/="-----END PGP SIGNATURE-----") $ lines s)
    ++ "-----END PGP SIGNATURE-----\n"
\end{code}

\input{darcs-createrepo.lhs}
