%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs changes}
\begin{code}
module Changes ( changes ) where

import IO ( putStr )
import RegexString ( mkRegex, matchRegex )
import List ( (\\) )

import DarcsCommands
import DarcsArguments
import Repository
import PatchInfo
\end{code}

\options{changes}
\begin{code}
changes_description = "Gives a human-readable list of changes between versions.
"
\end{code}
\haskell{changes_help}
\begin{code}
changes_help = "Changes gives a human-readable list of changes between versions
suitable for use as a changelog.
"

changes = DarcsCommand {command_name = "changes",
                        command_help = changes_help,
                        command_description = changes_description,
                        command_extra_args = 0,
                        command_get_arg_possibilities = return [],
                        command_command = changes_cmd,
                        command_prereq = am_in_repo,
                        command_argdefaults = nodefaults,
                        command_darcsoptions = [tagname, patchname_option]}
\end{code}


\begin{code}
changes_cmd opts args = do
  patches <- read_repo "."
  putStr $ changelog $ get_changes_info opts patches
\end{code}

\begin{code}
chose_two :: [DarcsFlag] -> Bool
chose_two (TagName _:fs) = chose_one fs
chose_two (PatchName _:fs) = chose_one fs
chose_two (_:as) = chose_two as
chose_two [] = False
chose_one :: [DarcsFlag] -> Bool
chose_one (TagName _:_) = True
chose_one (PatchName _:_) = True
chose_one (_:as) = chose_one as
chose_one [] = False

pick_second_patches :: [DarcsFlag] -> PatchSet -> PatchSet
pick_second_patches (TagName _:ps) = pick_patches ps
pick_second_patches (PatchName _:ps) = pick_patches ps
pick_second_patches (_:fs) = pick_second_patches fs

pick_patches :: [DarcsFlag] -> PatchSet -> PatchSet
pick_patches (TagName t:_) = pictag (mymatch t)
pick_patches (PatchName pn:_) = picpatch (mymatch pn)
pick_patches (_:fs) = pick_patches fs
pick_patches [] = \ ps -> ps

pictag :: (String -> Bool) -> PatchSet -> PatchSet
pictag t (((pi,mp):ps):r)
    | take 4 (just_name pi) == "TAG " && t (drop 4 (just_name pi))
        = ((pi,mp):ps):r
    | otherwise = pictag t (ps:r)
pictag t ([]:r) = pictag t r
pictag t [] = error "Couldn't find matching tag!\n"

picpatch :: (String -> Bool) -> PatchSet -> PatchSet
picpatch t (((pi,mp):ps):r) =
    if t (just_name pi) then ((pi,mp):ps):r
                        else picpatch t (ps:r)
picpatch t ([]:r) = picpatch t r
picpatch t [] = error "Couldn't find matching patch!\n"

mymatch ('^':r) s = matchRegex (mkRegex ('^':r)) s /= Nothing
mymatch r s = matchRegex (mkRegex (".*"++r)) s /= Nothing
\end{code}

\begin{code}
get_changes_info :: [DarcsFlag] -> PatchSet -> [PatchInfo]
get_changes_info opts ps =
  if not $ chose_one opts
  then map fst $ concat ps
  else
    case map fst $ concat $ pick_patches opts ps of
    pi1s ->
        case (if chose_two opts
              then map fst $ concat $ pick_second_patches opts ps
              else map fst $ concat ps) of
        pi2s -> pi2s \\ pi1s

changelog :: [PatchInfo] -> String
changelog pis = unlines $ map human_friendly pis
\end{code}
