/* $Id: readconf.c,v 1.6 1997/05/03 11:57:22 lexa Exp $ */

/* (C) Alex Tutubalin, 1996, lexa@lexa.ru       */




#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <netdb.h>
#include <syslog.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/errno.h>
		
#include "policy.h"
#include "readconf.h"

char tabledir[MAXPATH]="";
static int cur_string = 0;
static struct conf config = {NULL,NULL,NULL,NULL,NULL};


#define strcmpi strcasecmp

static int 
configComplain(char *fmt, ...)
{
  va_list argptr;
#ifndef NO_VSNPRINTF
  char buffer[80];
#else
   /* large buffer - Solaris don't have vsnprintf, so we
      must reserve more space to aviod buffer overflow.
      This is DANGEROUS - inccorect line in configfile
      can result to stack overflow and unpredictable behaviour
      of program
   */
#define LOGBUFSIZE 1024
   char buffer[LOGBUFSIZE];
#endif
  
  int cnt;

  va_start(argptr, fmt);
#ifndef NO_VSNPRINTF
  cnt = vsnprintf(buffer,79, fmt, argptr);
#else
  cnt = vsprintf(buffer,fmt,argptr);
#endif
  va_end(argptr);
  syslog(LOG_ERR,"ReadConfig: line %d : %s\n",cur_string,buffer);
  return(cnt);
}



/* List Operations */

#define AddToList(lvar,confvar,structname)		\
{							\
  lvar=confvar;						\
  if(lvar == NULL)					\
    {							\
      confvar = lvar=malloc(sizeof(structname));	\
      lvar->chain = NULL;				\
    }							\
  else							\
    {							\
      while(lvar->chain)				\
	lvar=lvar->chain;				\
      lvar->chain=malloc(sizeof(structname));		\
      lvar=lvar->chain;					\
      lvar->chain=NULL;					\
    }							\
}

#define FreeLIST(type,chvar) 			\
{						\
  type *var1,*var2;				\
  var1=chvar;					\
  while(var1) {					\
    var2=var1->chain;				\
    free(var1);					\
    var1=var2;					\
  }						\
}




static int
addEncoding(tok_array arr) 
{
  encoding *encptr;

  AddToList(encptr,config.enc,encoding);

  strcpy(encptr->name,arr[0]);
  memset(encptr->tablefrom,0,TABLESIZE);
  memset(encptr->tableto,0,TABLESIZE);
  encptr->changecharset=0;
  encptr->servercharset = encptr->clientcharset = NULL;

  if(!strcmpi(arr[1],NOTABLE)) 
    encptr->tablefromfile[0]='\0';
  else 
     sprintf(encptr->tablefromfile,"%s%s",tabledir,arr[1]);
  
  if(!strcmpi(arr[2],NOTABLE)) 
    encptr->tabletofile[0]='\0';
  else 
    sprintf(encptr->tabletofile,"%s%s",tabledir,arr[2]);
  return ERR_NOCONFERROR;

}

static int
addEmailCharset(tok_array arr)
{
  encoding *encptr = config.enc;
  while(encptr)
    {
      if(!strcasecmp(arr[0],encptr->name))
	{
	  encptr->servercharset=strdup(arr[1]);
	  encptr->clientcharset=strdup(arr[2]);
	  encptr->changecharset = 1;
	  return ERR_NOCONFERROR;
	}
      encptr=encptr->chain;
    }
  return ERR_CONFIG;
}

int
InitTables(encoding *encptr) 
{
  int file;
  DebugLog("InitTables: filefrom:%s fileto: %s name:%s client:%s server:%s",
	 *(encptr->tablefromfile)?encptr->tablefromfile:"NONE",
	 *(encptr->tabletofile)?encptr->tabletofile:"NONE",
	 encptr->name,encptr->clientcharset,encptr->servercharset);
  bzero(encptr->tablefrom,TABLESIZE);
  bzero(encptr->tableto,TABLESIZE);
  if(!*encptr->tablefromfile) goto to_table;
  if(-1==(file = open(encptr->tablefromfile,O_RDONLY)))
    {
      syslog(LOG_ERR,"InitTables:%s : %s, assuming empty table",
	     encptr->tablefromfile,strerror(errno));
      goto to_table;
    }
  if(TABLESIZE != read(file,encptr->tablefrom,TABLESIZE))
    syslog(LOG_ERR,"InitTables: Error reading \'from\' table %s:%s",
	   encptr->tablefromfile,strerror(errno));
  if(-1==close(file))
    syslog(LOG_ERR,"InitTables: Error closing file: %s",strerror(errno));
to_table:
  if(!*encptr->tabletofile) goto end_read;
  if(-1==(file = open(encptr->tabletofile,O_RDONLY)))
    {
      syslog(LOG_ERR,"InitTables: %s: %s, assuming empty table",
	     encptr->tabletofile,strerror(errno));
      goto end_read;
    }
  if(TABLESIZE != read(file,encptr->tableto,TABLESIZE))
    syslog(LOG_ERR,"InitTables: Error reading \'to\' table %s :%s",
	   encptr->tabletofile,strerror(errno));
  if(-1==close(file))
    syslog(LOG_ERR,"InitTables: Error closing file: %s",strerror(errno));
end_read:
  return ERR_NOCONFERROR;
}


 encoding * 
EncodingByName (const char *name) 
{
  encoding *encptr = config.enc;
  while(encptr!=NULL) {
    if(!strcmpi(name,encptr->name)) return encptr;
    encptr=encptr->chain;
  }
  configComplain("Unknown encoding %s",name);
  return NULL;
}


static int  
initTableDir(tok_array arr) 
{
  strcpy(tabledir,arr[0]);
  if(tabledir[strlen(tabledir)-1]!='/') strcat(tabledir,"/");
  return ERR_NOCONFERROR;
}

struct keytoken conftokens[] = {
  {"tabledir",TABLEDIR,1},
  {"encoding",ENCODING,3},
  {"emailcharset",EMAILCHARSET,3},
  {"encprecedence",ENCPRECEDENCE,1},
  {"gwprecedence",GWPRECEDENCE,1},
  {"default",DEFAULT,1},
  {"option",OPTION,4},
  {"peer",PEER,2},
  {"virtualhost",VIRTUALHOST,2}
#ifdef TRANSPARENT_HOST
  ,{"transparenthost",TRANSPARENTHOST,1}
#endif
};

static int 
getToken(const char *word) 
{
  int i;
  for(i=0;i<sizeof(conftokens)/sizeof(conftokens[0]);i++)
    if(!strcmpi(word,conftokens[i].token)) return i;
  return NOTOKEN;
}

static enum tokens  
Parse(char *str, tok_array arr) 
{
  char *p;
  int i,tk;
  if(NULL!=(p=strchr(str,'#'))) *p='\0';
  if(NULL==(p=strtok(str,TOKDELIM))) return NOTOKEN;

  if(NOTOKEN==(tk=getToken(p))) {
    configComplain("Unknown keyword %s",p);
    return NOTOKEN;
  }
  for(i=0;i<conftokens[tk].nwords;i++) {
    p=strtok(NULL,TOKDELIM);
    if(p==NULL){
      configComplain("Incorrect number of parameters");
      return PARSEERR;
    }else {
      strncpy(arr[i],p,sizeof(arr[i])-1);
      arr[i][sizeof(arr[i])-1]='\0';
    }
  }
  return conftokens[tk].tok;
}

#if defined (ENCODING_PREC) && defined (GATEWAY_PREC)
#define PRECS_SIZE 10
static char encprec[PRECS_SIZE]=ENCODING_PREC,gwprec[PRECS_SIZE]=GATEWAY_PREC;
#else
#error 'You must define both ENCODING_PREC and GATEWAY_PREC in Makefile'
#endif

static int 
initEncPrecedence(tok_array cline)
{
  int i,c=0;
  bzero(encprec,PRECS_SIZE);
  for (i=0;i<PRECS_SIZE-1 && cline[0][i];i++)
    {
      if(strchr(P_ENC_PREC,cline[0][i]))
	encprec[c++]=cline[0][i];
      else 
	{
	  configComplain("Bad encprecedence string");
	  return ERR_CONFIG;
	}
    }
  if(c<2)
    {
      strcpy(encprec,ENCODING_PREC);
      configComplain("No valid encoding precedences found, defaults restored");
      return ERR_NOCONFERROR;
    }
  return ERR_NOCONFERROR;
}

static int 
initGwPrecedence(tok_array cline)
{
  int i,c=0;
  bzero(gwprec,PRECS_SIZE);
  for (i=0;i<PRECS_SIZE-1 && cline[0][i];i++)
    {
      if(strchr(P_GW_PREC,cline[0][i]))
	gwprec[c++]=cline[0][i];
      else 
	{
	  configComplain("Bad gwprecedence string");
	  return ERR_CONFIG;
	}
    }
  if(c<2)
    {
      strcpy(encprec,ENCODING_PREC);
      configComplain("No valid gateway precedences found, defaults restored");
      return ERR_NOCONFERROR;
    }
  return ERR_NOCONFERROR;
}

 
encoding *DefaultEncoding = NULL;

static int
initDefault(tok_array cline)
{
  DefaultEncoding = EncodingByName(cline[0]);  return ERR_NOCONFERROR;

}


static int 
processOptions(tok_array cline)
{
  char *t,*gname="\0";
  int pn = strtod(cline[1],&t);
  Gateway *g;
  if(strcmpi(cline[2],"protocol")) {
    configComplain("Invalid option parameter: %s",cline[2]);
    return ERR_CONFIG;
  }

  if(!strcmpi(cline[3],"telnet")) {g=TelnetGW; gname = "TelnetGW";}
  else if (!strcmpi(cline[3],"http")) {g = HttpGW;gname = "HttpGW";}
  else if (!strcmpi(cline[3],"gopher")) {g = GopherGW;gname = "GopherGW";}
  else if (!strcmpi(cline[3],"smtp")) {g = SmtpGW;gname = "SmtpGW";}
  else if (!strcmpi(cline[3],"pop")) {g = PopGW;gname = "PopGW";}
  else g=DefaultGW;
  if(!strcmpi("port",cline[0])) {
    port_options *po;
    AddToList(po,config.po,port_options);
    po->port=htons(pn);
    po->gw=g;
    strcpy(po->name,gname);
  } else if (!strcmpi("tos",cline[0])) {
    tos_options *to;
    AddToList(to,config.to,tos_options);
    to->tos=htons(pn);
    to->gw=g;
    strcpy(to->name,gname);
  } else {
    configComplain("Unknown option %s",cline[0]);
    return ERR_CONFIG;
  }
  return ERR_NOCONFERROR;

}

Gateway *
GatewayByName (const char *name) 
{
  if(!strcmpi(name,"telnet")) return TelnetGW;
  else if (!strcmpi(name,"gopher")) return GopherGW;
  else if (!strcmpi(name,"http")) return HttpGW;
  else if (!strcmpi(name,"smtp")) return SmtpGW;
  else if (!strcmpi(name,"pop")) return PopGW;
  
  return NULL;
}


static int 
hookup(char *host, host_addr *hostaddr )
{
  struct hostent *hp = NULL;

  memset(hostaddr, 0, sizeof (*hostaddr));
  hostaddr->s_addr = inet_addr(host);
  if (hostaddr->s_addr == -1) { 
    hp = gethostbyname(host);
    if (hp == NULL) 
      {
#ifndef NO_HSTRERROR
			configComplain("%s: %s", host, hstrerror(h_errno));
#else
			configComplain("%s: error %d", host, h_errno);
#endif

	return ERR_CONFIG;
      }
    memmove(&hostaddr->s_addr,hp->h_addr_list[0],sizeof(hostaddr->s_addr));
  }
  return ERR_NOCONFERROR;
}


static int 
addPeer(tok_array cline)
{
  host_addr a;
  encoding *enc;
  peer *p;

  if(ERR_NOCONFERROR != hookup(cline[0],&a)) return ERR_CONFIG;
  enc = EncodingByName(cline[1]);
  AddToList(p,config.peers,peer);
  p->encptr = enc;
  p->addr.s_addr = a.s_addr;
  return ERR_NOCONFERROR;
}


static int 
addVhost(tok_array cline)
{
  host_addr a;
  encoding *enc;
  virthost *v;

  if(ERR_NOCONFERROR != hookup(cline[0],&a)) return ERR_CONFIG;
  enc = EncodingByName(cline[1]);
  AddToList(v,config.vhosts,virthost);
  v->encptr = enc;
  v->addr.s_addr=a.s_addr;
  return ERR_NOCONFERROR;
}


#ifdef TRANSPARENT_HOST
static int 
addTranspHost(tok_array cline)
{
  host_addr a;
  encoding *enc;
  transphost *t;

  if(ERR_NOCONFERROR != hookup(cline[0],&a)) return ERR_CONFIG;
  enc = (encoding*)1; /* not NULL!. Compatibility only */
  AddToList(t,config.thl,transphost);
  t->encptr = enc;
  t->addr.s_addr=a.s_addr;
  return ERR_NOCONFERROR;
}
#endif

int 
ReadConfig (const char *cfileName) 
{
  char strbuf[256];
  tok_array commandline;
  FILE *cfile;

  if(NULL==(cfile=fopen(cfileName,"rt")))
    {
      syslog(LOG_ERR,"Can't open configfile %s\n",cfileName); 
      return ERR_CONFIG;
    }
  while(!feof(cfile))
    {
      if(NULL==fgets(strbuf,255,cfile)) break;
      cur_string++;
      switch(Parse(strbuf,commandline))
	{
	case NOTOKEN    : 
	  break;
	case PARSEERR   : 
	  configComplain("Bad config token");
	  break;
	case TABLEDIR   : 
	  if(ERR_NOCONFERROR!=initTableDir(commandline)) goto error; 
	  break;
	case ENCODING   : 
	  if(ERR_NOCONFERROR!=addEncoding(commandline)) goto error;  
	  break;
	case EMAILCHARSET   : 
	  if(ERR_NOCONFERROR!=addEmailCharset(commandline)) goto error;  
	  break;
	case ENCPRECEDENCE : 
	  if(ERR_NOCONFERROR!=initEncPrecedence(commandline)) goto error; 
	  break;
	case GWPRECEDENCE : 
	  if(ERR_NOCONFERROR!=initGwPrecedence(commandline)) goto error; 
	  break;
	case DEFAULT    : 
	  if(ERR_NOCONFERROR!=initDefault(commandline)) goto error;   
	  break;
	case OPTION     :
	  if(ERR_NOCONFERROR!=processOptions(commandline)) goto error; 
	  break;
	case PEER       : 
	  if(ERR_NOCONFERROR!=addPeer(commandline)) goto error;       
	  break;
	case VIRTUALHOST : 
	  if(ERR_NOCONFERROR!=addVhost(commandline))
	    goto error;      
	  break;
#ifdef TRANSPARENT_HOST
	case TRANSPARENTHOST : 
	  if(ERR_NOCONFERROR!=addTranspHost(commandline)) goto error; 
	  break;
#endif
	default: configComplain("Some shit happens");break;
	}
    }
  fclose(cfile);
  return ERR_NOCONFERROR;
    error:
  syslog(LOG_ERR,"Fatal: Config file not readed after string %d",cur_string);
  fclose(cfile);
  return ERR_CONFIG;
  
}

void 
ClearConfig(void) 
{
  FreeLIST(encoding,config.enc);
  FreeLIST(peer,config.peers);
  FreeLIST(virthost,config.vhosts);
  FreeLIST(tos_options,config.to);
  FreeLIST(port_options,config.po);
#ifdef TRANSPARENT_HOST
  FreeLIST(transphost,config.thl);
#endif
}


struct sockaddr_in local_addr,peer_addr;
static int llen=sizeof(local_addr),plen=sizeof(peer_addr),peer_found=0,
  local_found=0;

void 
GetAddresses(int s)
{
  if(0==getpeername(s,(struct sockaddr*)&peer_addr,&plen))
    {
#ifdef ADDR_LOGGING
      {
	struct hostent *h;
	if((h=gethostbyaddr((char*)&peer_addr.sin_addr,4,AF_INET))!= 
	   (struct hostent*)0)
	  syslog(ADDR_LOGGING,"Client address: %s",h->h_name);
	else
	  syslog(ADDR_LOGGING,"Client address: %s (address not found in DNS)",
		 inet_ntoa(peer_addr.sin_addr));
      }
#endif
      peer_found=1;
    }
  else
    {
      syslog(LOG_WARNING,"Cannot determine client peer name: %s",strerror(errno));
    }
  if(0==getsockname(s,(struct sockaddr*)&local_addr,&llen))
    {
#ifdef ADDR_LOGGING
      {
	struct hostent *h;
	if((h=gethostbyaddr((char*)&local_addr.sin_addr,4,AF_INET))!= 
	   (struct hostent*)0)
	  syslog(ADDR_LOGGING,"Local address: %s, port %d",
		 h->h_name,ntohs(local_addr.sin_port));
	else
	  syslog(ADDR_LOGGING,"Local address: %s (not in DNS!),port %d",
		 inet_ntoa(local_addr.sin_addr),ntohs(local_addr.sin_port));
      }
#endif
      local_found = 1;
    }
 else
    {
      syslog(LOG_WARNING,"Cannot determine local address: %s",strerror(errno));
    }
}

#define FindEncodingByAddr(type,listvar,addrs,retval)	\
{							\
  type *list = listvar;					\
  retval = NULL;					\
  while(list)						\
    {							\
      if(list->addr.s_addr == addrs.sin_addr.s_addr)	\
	{						\
	  retval = list->encptr;			\
	  break;					\
	}						\
      list = list->chain;				\
    }							\
}


static encoding*
EncodingByPeer(void)
{
  encoding *ret;
  if(peer_found){
    FindEncodingByAddr(peer,config.peers,peer_addr,ret);
    return ret;
  }
  else
    return NULL;
}

static encoding*
EncodingByVhost(void)
{
  encoding *ret;
  if(local_found) 
    {
      FindEncodingByAddr(virthost,config.vhosts,local_addr,ret);
      return ret;
    }
   else
     return NULL;
}

encoding *
EncodingByAddress(void)
{
  encoding *ret=DefaultEncoding;
  int i;
  for(i=0;i<strlen(encprec);i++)
    switch (toupper(encprec[i]))
      {
      case 'V':
	if(NULL!=(ret = EncodingByVhost()))
	  {
	    DebugLog("Encoding %s choosed by vhost address",ret->name);
	    return ret;
	  }
	break;
     case 'P':
	if(NULL!=(ret = EncodingByPeer()))
	  {
	    DebugLog("Encoding %s choosed by peer address",ret->name);
	    return ret;
	  }
	break;
      default:
	DebugLog("Unknown encprecedence value");
	break;
      }
  
  DebugLog("Default encoding choosed");
  return DefaultEncoding;
}


#define FindGateway(type,listvar,fieldn,compval,retval)	\
	{								\
	  type *list = listvar;						\
	  retval=NULL;							\
	  while(list!=NULL)						\
	    {								\
	      if(list->##fieldn == compval)				\
		retval=list;						\
	      list=list->chain;						\
	    }								\
	}

Gateway*
GatewayBySocket (int sock) 
{
  tos_options *tr;
  port_options *pr;
  int ltos,toslen=sizeof(ltos),i;

  for(i=0;i<strlen(gwprec);i++)
    switch(toupper(gwprec[i]))
      {
      case 'P':
	if(local_found)
	  {
	    FindGateway(port_options,config.po,port,local_addr.sin_port,pr);
	    if(pr!=NULL)
	      {
		DebugLog("Gateway %s choosed by port number",
			 *pr->name?pr->name:"defaultGW");
		return pr->gw;
	      }
	  }
	break;
      case 'T':
	  if(0==getsockopt(sock, IPPROTO_IP, IP_TOS, (char*)&ltos, &toslen))
	    {
	      FindGateway(tos_options,config.to,tos,ltos,tr);
	      if(tr!=NULL)
		{   
		  DebugLog("Gateway %s choosed by TOS",*tr->name?tr->name:"defaultGW");
		  return tr->gw;
		}
	    }
	  else
	    syslog(LOG_WARNING,"Cannot determine TOS: %s",strerror(errno));
	  
	  break;
      }
  DebugLog("No optional gateway found, assuming defaultGW");
  return DefaultGW;
}

#ifdef TRANSPARENT_HOST
Gateway*
IsTransparentAddr(int sock)
{
  encoding *enc;
  if(local_found)
    {
      FindEncodingByAddr(transphost,config.thl,local_addr,enc);
      if(enc!=NULL)
	{
	  DebugLog("Transparent Gateway choosed by local address");
	  return transparentGW;
	}
    }

  return NULL;
}
#endif

