/* $Id: decode.c,v 1.79 2004/11/08 06:51:35 onoe Exp $ */

/*-
 * Copyright (c) 1998-2001 Atsushi Onoe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <stddef.h>

#ifdef ZLIB
#include <zlib.h>
#endif /* ZLIB */

#include "cue.h"
#include "decode.h"
#include "utf8.h"

char b64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
char hex[] = "0123456789abcdef0123456789ABCDEF";
char uue[] = "`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_";
char hqx[] = "!\"#$%&'()*+,-012345689@ABCDEFGHIJKLMNPQRSTUVXYZ[`abcdefhijklmpqr";

char rb64[256], rhex[256], ruue[256], rhqx[256];

static struct charset {
	const char *cs_name;
	u_char	cs_enc;
	u_char	cs_code[3];
	u_char	cs_len;
} charset_tbl[] = {
/* XXX The first 7 charsets MUST match the definition of FDB_CS_XXX in cue.h */
	{ "iso-2022-jp",    'B', { CS_KANJI, CS_KANA, CS_HOJO } }, /* default */
	{ "iso-2022-jp",    'B', { CS_KANA } },
	{ "shift_jis",	    'B', { CS_KANJI, CS_KANA, CS_HOJO } },	/*XXX*/
	{ "utf-8",	    'B', { CS_KANJI, CS_KANA, CS_HOJO } },	/*XXX*/
	{ "us-ascii",       'Q', },
	{ "iso-2022-jp-2",  'B', },
	{ "cp943c",	    'Q', { CS_KANJI, CS_KANA, CS_HOJO } },	/*XXX*/

/* additional charsets */
	{ "euc-jp",	    'B', { CS_KANJI, CS_KANA, CS_HOJO } },
	{ "iso-8859-1",	    'Q', { CS_96SET|CS_CODE('A'), CS_96SET|CS_CODE('A'),
				   CS_96SET|CS_CODE('A') } },
	{ "iso-8859-2",	    'Q', { CS_96SET|CS_CODE('B'), CS_96SET|CS_CODE('B'),
				   CS_96SET|CS_CODE('B') } },
	{ "iso-8859-3",	    'Q', { CS_96SET|CS_CODE('C'), CS_96SET|CS_CODE('C'),
				   CS_96SET|CS_CODE('C') } },
	{ "iso-8859-4",	    'Q', { CS_96SET|CS_CODE('D'), CS_96SET|CS_CODE('D'),
				   CS_96SET|CS_CODE('D') } },
	{ "iso-8859-5",	    'Q', { CS_96SET|CS_CODE('L'), CS_96SET|CS_CODE('L'),
				   CS_96SET|CS_CODE('L') } },
	{ "iso-8859-6",	    'Q', { CS_96SET|CS_CODE('G'), CS_96SET|CS_CODE('G'),
				   CS_96SET|CS_CODE('G') } },
	{ "iso-8859-7",	    'Q', { CS_96SET|CS_CODE('F'), CS_96SET|CS_CODE('F'),
				   CS_96SET|CS_CODE('F') } },
	{ "iso-8859-8",	    'Q', { CS_96SET|CS_CODE('H'), CS_96SET|CS_CODE('H'),
				   CS_96SET|CS_CODE('H') } },
	{ "iso-8859-9",	    'Q', { CS_96SET|CS_CODE('M'), CS_96SET|CS_CODE('M'),
				   CS_96SET|CS_CODE('M') } },
	{ "iso-8859-10",    'Q', { CS_96SET|CS_CODE('V'), CS_96SET|CS_CODE('V'),
				   CS_96SET|CS_CODE('V') } },
	{ "iso-8859-15",    'Q', { CS_96SET|CS_CODE('b'), CS_96SET|CS_CODE('b'),
				   CS_96SET|CS_CODE('b') } },
	{ "ks_c_5601-1987", 'B', { CS_DWIDTH|CS_CODE('C') } },
	{ "euc-kr",	    'B', { CS_DWIDTH|CS_CODE('C') } },
	{ NULL, }
};

void
decode_init(void)
{
	int i;

	memset(rb64, -1, sizeof(rb64));
	for (i = 0; i < sizeof(b64); i++)
		rb64[((u_char *)b64)[i]] = i;
	memset(rhex, -1, sizeof(rhex));
	for (i = 0; i < sizeof(hex); i++)
		rhex[((u_char *)hex)[i]] = i & 0x0f;
	memset(ruue, -1, sizeof(ruue));
	for (i = 0; i < sizeof(uue); i++)
		ruue[((u_char *)uue)[i]] = i;
	ruue[' '] = 0;	/* XXX */
	memset(rhqx, -1, sizeof(rhqx));
	for (i = 0; i < sizeof(hqx); i++)
		rhqx[((u_char *)hqx)[i]] = i;

	for (i = 0; charset_tbl[i].cs_name != NULL; i++)
		charset_tbl[i].cs_len = strlen(charset_tbl[i].cs_name);
}

static void
charset_dinit(struct decode *dp, int id)
{
	int i;

	if (id < 0 && id >= sizeof(charset_tbl)/sizeof(charset_tbl[0]))
		id = 0;
	dp->cscode[0] = CS_ASCII;
	for (i = 1; i < 4; i++) {
		dp->cscode[i] = charset_tbl[id].cs_code[i - 1];
		if (dp->cscode[i] == 0)
			dp->cscode[i] = charset_tbl[0].cs_code[i - 1];
	}
	/* special handling */
	if (id == FDB_CS_SJIS || id == FDB_CS_SJIS2)
		dp->sjis = 1;
	else if (id == FDB_CS_UTF8)
		dp->utf8 = 1;
}

static void
DINIT(struct decode *dp, int flags)
{
	char *p;

	if (dp->bufsiz != CHARBLOCK) {
		p = realloc(dp->buf, CHARBLOCK);
		if (!p)
			abort();
		dp->buf = p;
		dp->bufsiz = CHARBLOCK;
	}
	dp->cnt = 0;
	dp->flsbuf = NULL;
	dp->pend = 0;
	dp->err = 0;
	dp->encoded = 0;
	dp->noconv = 0;
	dp->notext = 0;
	dp->sjis = 0;
	dp->utf8 = 0;
	dp->bclen = 0;
	dp->shift = 0;
	charset_dinit(dp, flags & FDB_CHARSET);
	if (flags & FDB_NOCONV)
		dp->noconv = 1;
	if (flags & FDB_NOTEXT)
		dp->notext = 1;
}

static void
OUTERR(struct decode *dp, u_char c)
{

	if (c < ' ' && c != '\n' && c != '\t') {
		OUTC(dp, '^'); OUTC(dp, c + '@');
		dp->encoded = 1;
	} else if (!(c & 0x80)) {
		OUTC(dp, c);
	} else {
		OUTC(dp, '\\'); OUTC(dp, 'x');
		OUTC(dp, hex[c >> 4]); OUTC(dp, hex[c & 0xf]);
		dp->encoded = 1;
	}
	dp->err = 1;
}

static void
OUTPEND(struct decode *dp)
{
	int i;
	u_char c;

	for (i = 0; i < 4; i++) {
		c = dp->pend >> 24;
		dp->pend <<= 8;
		if (c != 0)
			OUTERR(dp, c);
	}
}

static void
DOUTC(struct decode *dp, char c0)
{
	u_char c = (u_char)c0;
	u_char s;
	u_long v;
	int G = 0;

	if (dp->noconv) {
		OUTC(dp, c);
		return;
	}
#if 0	/* try to decode and show */
	if (dp->notext) {
		OUTERR(dp, c);
		return;
	}
#endif
	if (dp->pend == '\r') {
		if (c != '\n') {
			OUTC(dp, '\n');
		}
		dp->pend = 0;
	}

	if (dp->sjis == 1 && (c & 0x80)) {		/* SJIS 1st */
		if (dp->pend)
			OUTPEND(dp);
		if (c >= 0xa1 && c <= 0xdf) {		/* KANA */
			OUTC(dp, CS_SS2);
			OUTC(dp, c);
		} else if (c >= 0x81 && c <= 0xfc) {
			dp->sjis = 2;
			dp->pend = c;
		} else {
			OUTERR(dp, c);
		}
		dp->encoded = 1;
		return;
	}
	if (dp->sjis == 2) {				/* SJIS 2nd */
		s = dp->pend;
		s -= (s <= 0x9f) ? 0x71 : 0xb1;
		s = (s * 2 + 1) | 0x80;
		if (s >= 0xa0 && s <= 0xfd && c >= 0x9f && c <= 0xfc) {
			OUTC(dp, s + 1);
			OUTC(dp, c + 2);
		} else if (s >= 0xa1 && s <= 0xfe && c >= 0x40 && c <= 0x7e) {
			OUTC(dp, s);
			OUTC(dp, c + 0x61);
		} else if (s >= 0xa1 && s <= 0xfe && c >= 0x80 && c <= 0x9e) {
			OUTC(dp, s);
			OUTC(dp, c + 0x60);
		} else {
			OUTERR(dp, dp->pend);
			OUTERR(dp, c);
		}
		dp->pend = 0;
		dp->sjis = 1;
		return;
	}
	if (dp->utf8) {
		if (c & 0x80) {
			dp->pend = (dp->pend << 8) | c;
			if ((dp->pend & 0xfff00000) == 0x00f00000 ||
			    (dp->pend & 0xffffe000) == 0x0000e000 ||
			    (dp->pend & 0xffffffc0) == 0x000000c0)
				return;
			v = utf8_decode(dp->pend);
			dp->encoded = 1;
			dp->pend = 0;
			if (v == 0) {
				OUTPEND(dp);
				return;
			}
			c = v & 0xff;
			s = (v >> 8) & 0xff;
			if ((v >> 24) == CS_SSX) {
				OUTC(dp, CS_SSX);
				OUTC(dp, (v >> 16) & 0xff);
				OUTC(dp, s);
				OUTC(dp, c);
				return;
			}
			if ((v >> 16) == CS_SSX) {
				OUTC(dp, CS_SSX);
				OUTC(dp, s);
				OUTC(dp, c);
				return;
			}
			if ((v >> 16) == CS_SS3) {
				OUTC(dp, CS_SS3);
				OUTC(dp, s);
				OUTC(dp, c);
				return;
			}
			if ((v >> 8) == CS_SS2) {
				OUTC(dp, CS_SS2);
				OUTC(dp, c);
				return;
			}
			dp->pend = s;
		} else {
			if (dp->pend)
				OUTPEND(dp);
		}
	}

	switch (dp->pend) {
	case 0:
		break;

	case 0x1b:
		switch (c) {
		case 0x24:	/* ESC-$- */
		case 0x28:	/* ESC-(- */
		case 0x29:	/* ESC-)- */
		case 0x2a:	/* ESC-*- */
		case 0x2b:	/* ESC-+- */
		case 0x2d:	/* ESC--- */
		case 0x2e:	/* ESC-.- */
		case 0x2f:	/* ESC-/- */
			dp->pend = (dp->pend << 8) | c;
			return;
		case 0x4e:	/* ESC-N */
			dp->pend = CS_SS2;
			return;
		case 0x4f:	/* ESC-O */
			dp->pend = CS_SS3;
			return;
		default:
			OUTPEND(dp);
			break;
		}
		break;

	case 0x1b24:
		switch (c) {
		case 0x28:	/* ESC-$-(- */
		case 0x29:	/* ESC-$-)- */
		case 0x2a:	/* ESC-$-*- */
		case 0x2b:	/* ESC-$-+- */
		case 0x2d:	/* ESC-$--- */
		case 0x2e:	/* ESC-$-.- */
		case 0x2f:	/* ESC-$-/- */
			dp->pend = (dp->pend << 8) | c;
			return;
		case 0x40:	/* ESC-$-@ */
		case 0x42:	/* ESC-$-B */
			dp->cscode[0] = CS_DWIDTH | CS_CODE('B');	/*XXX*/
			dp->pend = 0;
			return;
		default:
			OUTPEND(dp);
			break;
		}
		break;

	case 0x1b2428:		/* ESC-$-(- */
		if (c == '@')
			c = 'B';	/* XXX */
	case 0x1b2429:		/* ESC-$-)- */
	case 0x1b242a:		/* ESC-$-*- */
	case 0x1b242b:		/* ESC-$-+- */
	case 0x1b242d:		/* ESC-$--- */
	case 0x1b242e:		/* ESC-$-.- */
	case 0x1b242f:		/* ESC-$-/- */
		dp->cscode[dp->pend & 0x00000003] = CS_DWIDTH |
		    CS_CODE(c) | ((dp->pend & 0x00000004) ? CS_96SET : 0);
		dp->pend = 0;
		return;

	case 0x1b28:		/* ESC-(- */
		if (c == 'J')
			c = 'B';	/* XXX */
	case 0x1b29:		/* ESC-)- */
	case 0x1b2a:		/* ESC-*- */
	case 0x1b2b:		/* ESC-+- */
	case 0x1b2d:		/* ESC--- */
	case 0x1b2e:		/* ESC-.- */
	case 0x1b2f:		/* ESC-/- */
		dp->cscode[dp->pend & 0x0003] =
		    CS_CODE(c) | ((dp->pend & 0x0004) ? CS_96SET : 0);
		dp->pend = 0;
		return;

	case CS_SS2:
		G = 2;
		dp->pend = 0;
		break;

	case CS_SS3:
		G = 3;
		dp->pend = 0;
		break;

	default:
		if ((dp->pend & 0xff000000) == 0x1b000000 ||
		    (dp->pend & 0xffff0000) == 0x001b0000 ||
		    (dp->pend & 0xffffff00) == 0x00001b00) {
			OUTPEND(dp);
			break;
		}
		if ((G == 0 && (c & 0x80)) || dp->shift)
			G = 1;
		if (!(dp->cscode[G] & CS_DWIDTH)) {
			OUTPEND(dp);
			break;
		}
		if (dp->cscode[G] & CS_96SET) {
			if ((c & 0x7f) < 0x20) {
				OUTPEND(dp);
				break;
			}
		} else {
			if ((c & 0x7f) <= 0x20 || (c & 0x7f) >= 0x7f) {
				OUTPEND(dp);
				break;
			}
		}
		if (dp->cscode[G] == CS_KANJI) {
			if (G != 1 || !(dp->pend & 0x80) || !(c & 0x80))
				dp->encoded = 1;
			OUTC(dp, dp->pend | 0x80);
			OUTC(dp, c | 0x80);
		} else if (dp->cscode[G] == CS_HOJO) {
			dp->encoded = 1;
			OUTC(dp, CS_SS3);
			OUTC(dp, dp->pend | 0x80);
			OUTC(dp, c | 0x80);
		} else {
			dp->encoded = 1;
			OUTC(dp, CS_SSX);
			OUTC(dp, dp->cscode[G]);
			OUTC(dp, dp->pend | 0x80);
			OUTC(dp, c | 0x80);
		}
		dp->pend = 0;
		return;
	}

	if (c == 0x1b) {
		dp->pend = c;
		dp->encoded = 1;
	} else if (c == 0x0e) {
		dp->shift = 1;
		dp->encoded = 1;
	} else if (c == 0x0f) {
		dp->shift = 0;
		dp->encoded = 1;
	} else if (c == '\n' || c == '\t') {
		OUTC(dp, c);
	} else if (c == '\r') {
		dp->encoded = 1;
		dp->pend = '\r';
	} else if (c == '\014' /* ^L */) {
		OUTC(dp, '^'); OUTC(dp, c + '@');
		dp->encoded = 1;
	} else if (c < ' ') {
		dp->encoded = 1;
		OUTERR(dp, c);
	} else {
		if ((G == 0 && (c & 0x80)) || dp->shift)
			G = 1;
		if (dp->cscode[G] & CS_96SET) {
			if ((c & 0x7f) < 0x20) {
				OUTERR(dp, c);
				return;
			}
		} else {
			if (c == ' ') {
				OUTC(dp, c);
				return;
			}
			if ((c & 0x7f) <= 0x20 || (c & 0x7f) >= 0x7f) {
				OUTERR(dp, c);
				return;
			}
		}
		if (dp->cscode[G] & CS_DWIDTH) {
			dp->pend = c;
			return;
		}
		if (dp->cscode[G] == CS_ASCII) {
			if (G != 0 || (dp->pend & 0x80) || (c & 0x80))
				dp->encoded = 1;
			OUTC(dp, c);
		} else if (dp->cscode[G] == CS_KANA) {
			dp->encoded = 1;
			OUTC(dp, CS_SS2);
			OUTC(dp, c | 0x80);
		} else {
			dp->encoded = 1;
			OUTC(dp, CS_SSX);
			OUTC(dp, dp->cscode[G]);
			OUTC(dp, c | 0x80);
		}
	}
}

/* decode 1 header */
int
decode_header(cbuf_t *hdr, int flags)
{
	char c, *p, *b, *ep;
	cbuf_t charset;
	int i, code, skip;
	int qenc;
	static struct decode d;

	DINIT(&d, flags);
	qenc = 0;
	for (p = hdr->ptr, ep = p + hdr->len; p < ep; ) {
		if (*p == '\n') {
			b = p - 1;
			while (++p < ep) {
				if (*p != ' ' && *p != '\t')
					break;
			}
			if (p == ep)
				break;
			if (*b != '=' || *p != '=')
				DOUTC(&d, ' ');
			continue;
		}
		if (qenc) {
			if (*p == '?' && p + 1 < ep && p[1] == '=') {
				qenc = 0;
				p += 2;
				d.cscode[0] = CS_ASCII;
				d.shift = 0;
				d.pend = 0;
				continue;
			}
			if (*p == '=') {
				if (p + 1 < ep && p[1] == '\n') {
					p += 2;	/* continuation */
					continue;
				}
				if (p + 2 < ep) {
					p++;
					c = rhex[*(unsigned char *)p++] << 4;
					c |= rhex[*(unsigned char *)p++];
					DOUTC(&d, c);
					continue;
				}
			}
		}
		if (*p != '=' || p + 7 > ep || p[1] != '?') {
  nodecode:
			DOUTC(&d, *p);
			p++;
			continue;
		}

		for (CSETP(&charset, p + 2); ; CL(&charset)++) {
			if (CE(&charset) + 4 >= ep)
				goto nodecode;
			if (CE(&charset)[0] == '?' &&
			    (CE(&charset)[1]=='B' || CE(&charset)[1]=='Q' ||
			     CE(&charset)[1]=='b' || CE(&charset)[1]=='q') &&
			    CE(&charset)[2] == '?')
				break;
		}
		d.cscode[0] = CS_ASCII;
		code = charset_id(&charset);
		if (code == FDB_CS_DEFAULT)
			goto nodecode;
		charset_dinit(&d, code);
		d.encoded = 1;
		d.shift = 0;
		d.pend = 0;
		p = CE(&charset) + 3;
		if (CE(&charset)[1] == 'Q' || CE(&charset)[1] == 'q') {
			qenc = 1;
			continue;
		}
		for (;;) {
			if (p + 1 < ep && *p == '?' && p[1] == '=') {
				p += 2;
				d.cscode[0] = CS_ASCII;
				d.shift = 0;
				d.pend = 0;
				break;
			}
			for (i = 0, code = 0, skip = 0; i < 4; i++, p++) {
				if (p >= ep)
					goto error;
				code <<= 6;
				if (*p == '=') {
					skip += 6;
				} else {
					c = rb64[*(u_char *)p];
					if (c == -1)
						goto error;
					code |= c;
				}
			}
			for (i = 8; i + skip <= 24; i += 8) {
				c = (code >> 16) & 0xff;
				DOUTC(&d, c);
				code <<= 8;
			}
		}
	}
	if (d.encoded) {
		hdr->ptr = d.buf;
		hdr->len = d.cnt;
		return (d.err ? -1 : 1);
	}
  error:
	return 0;
}

/* decode 1 line body */
static int
decode_qtext(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	static struct decode d;

	DINIT(&d, flags);
	for (p = *pp, c = '\n'; p < ep; ) {
		if ((c = *p++) == '\n')
			break;
		if (c == '=') {
			d.encoded = 1;
			if (p < ep && *p == '\n') {
				p++;	/* continuation */
				continue;
			}
			if (p + 3 >= ep)
				goto error;
			c = rhex[*(unsigned char *)p++] << 4;
			c |= rhex[*(unsigned char *)p++];
		}
		DOUTC(&d, c);
	}
	if (d.encoded) {
		res->ptr = d.buf;
		res->len = d.cnt;
		*pp = p;
		return (d.err ? -1 : 1);
	}
  error:
	if (c != '\n') {
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode base64 text */
static int
decode_base64(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	int i, code, skip;
	static struct decode d;

	DINIT(&d, flags);
	for (p = *pp; p < ep; ) {
		if (!d.noconv && d.cnt > 0 && d.buf[d.cnt-1] == '\n' &&
		    d.sjis <= 1 && d.pend == 0) {
			if (*p == '\n')
				p++;
			break;
		}
		if (*p == '\n') {
			p++;
			if (*p == '-')
				break;
			continue;
		}
		for (i = 0, code = 0, skip = 0; i < 4; p++) {
			if (p >= ep)
				goto error;
			if (*p == '=') {
				i++;
				code <<= 6;
				skip += 6;
				continue;
			}
			c = rb64[*(unsigned char *)p];
			if (c < 0)
				continue;
			i++;
			code = (code << 6) | c;
		}
		for (i = 8; i + skip <= 24; i += 8) {
			c = (code >> 16) & 0xff;
			DOUTC(&d, c);
			code <<= 8;
		}
	}
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);

  error:
	while (p < ep) {
		if (*p++ == '\n')
			break;
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode uuencode text */
static int
decode_uuencode(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p, *s;
	int i, code, cnt;
	static struct decode d;

	DINIT(&d, flags);

	for (p = *pp; p < ep; ) {
		if (!d.noconv && d.cnt > 0 && d.buf[d.cnt-1] == '\n') {
			if (*p == '\n')
				p++;
			break;
		}
		cnt = ruue[*(unsigned char *)p];
		if (cnt < 0) {
			s = p;
			while (p < ep) {
				if (*p++ == '\n')
					break;
			}
			if (s + 3 < p && strncmp(s, "end", 3) == 0) {
				p = ep;
				break;
			}
			continue;
		}
		p++;
		while (cnt > 0) {
			for (i = 0, code = 0; i < 4; i++, p++) {
				if (p >= ep)
					goto error;
				code <<= 6;
				c = ruue[*(unsigned char *)p];
				if (c < 0)
					goto error;
				code |= c;
			}
			for (i = 0; i < 3; i++) {
				if (cnt-- == 0)
					break;
				c = (code >> 16) & 0xff;
				DOUTC(&d, c);
				code <<= 8;
			}
		}
		if (*p != '\n')
			break;
		p++;
	}
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);

  error:
	while (p < ep) {
		if (*p++ == '\n')
			break;
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode binhex text */
/* XXX */
static int
decode_binhex(char **pp, char *ep, cbuf_t *res, int flags)
{
	char *p;

	/* start */
	p = *pp;
	if (*p != ':') {
		res->ptr = p;
		res->len = 0;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		*pp = p;
		return 0;
	}
	/* end */
	while (p < ep) {
		if (*p++ == ':')
			break;
	}

	res->ptr = banner("BINHEX");
	res->len = strlen(res->ptr);
	*pp = p;
	return 1;
}

#ifdef ZLIB
/* decode x-gzip64 text */
static int
decode_gzip64(char **pp, char *ep, cbuf_t *res, int flags)
{
	char *p, *q;
	u_char *u;
	int cc, zflags;
	z_stream z;
	char dbuf[CHARBLOCK];
	static struct decode d;

	DINIT(&d, flags);
	p = *pp;
	if (decode_base64(&p, ep, res, FDB_NOCONV) == 0)
		goto err;

	/* XXX: skip gzip header */
	u = (u_char *)res->ptr;
	if (u[0] != 0x1f || u[1] != 0x8b || u[2] != Z_DEFLATED)
		goto err;
	u += 3;
	zflags = *u++;
	u += 6;			/* time, xflags, oscode */
	if (zflags & 0x04) {	/* extra field */
		cc = (u[1] << 8) | u[0];
		u += cc + 2;
	}
	if (zflags & 0x08) {	/* orig name */
		while (*u++)
			;
	}
	if (zflags & 0x10) {	/* comment */
		while (*u++)
			;
	}
	if (zflags & 0x02) {	/* header crc */
		u += 2;
	}

	memset(&z, 0, sizeof(z));
	z.next_in = (Bytef *)u;
	z.avail_in = res->len - ((char *)u - res->ptr) + 4/*XXX*/;
	z.next_out = (Bytef *)dbuf;
	z.avail_out = sizeof(dbuf);
	if (inflateInit2(&z, -15) != Z_OK)
		goto err;
	for (;;) {
		cc = inflate(&z, Z_SYNC_FLUSH);
		if (cc == Z_STREAM_END)
			break;
		if (cc < 0)
			goto err;
		if (z.avail_out == 0) {
			for (q = dbuf; q < dbuf + sizeof(dbuf); q++) {
				DOUTC(&d, *q);
			}
			z.next_out = (Bytef *)dbuf;
			z.avail_out = sizeof(dbuf);
		}
	}
	inflateEnd(&z);
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);
 err:
	return decode_base64(pp, ep, res, flags);
}
#endif /* ZLIB */

/* decode 1 line body */
int
decode_text(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	static struct decode d;

	switch (flags & FDB_ENCODE) {
	case FDB_ENC_Q:
		return decode_qtext(pp, ep, res, flags);
	case FDB_ENC_B64:
		return decode_base64(pp, ep, res, flags);
#ifdef ZLIB
	case FDB_ENC_GZ64:
		return decode_gzip64(pp, ep, res, flags);
#endif /* ZLIB */
	case FDB_ENC_UU:
		return decode_uuencode(pp, ep, res, flags);
	case FDB_ENC_BINHEX:
		return decode_binhex(pp, ep, res, flags);
	}

	DINIT(&d, flags);
	for (p = *pp; p < ep; ) {
		if ((c = *p++) == '\n')
			break;
		DOUTC(&d, c);
	}
	if (d.encoded) {
		res->ptr = d.buf;
		res->len = d.cnt;
		*pp = p;
		return (d.err ? -1 : 1);
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

int
charset_id(const cbuf_t *charset)
{
	int i;

	for (i = 0; charset_tbl[i].cs_name != NULL; i++) {
		if (i == FDB_CS_DEFAULT)
			continue;
		if (CL(charset) == charset_tbl[i].cs_len &&
		    strncasecmp(CP(charset), charset_tbl[i].cs_name,
		    CL(charset)) == 0)
			return i;
	}
	return FDB_CS_DEFAULT;
}

const char *
charset_name(int id)
{

	if (id < 0 && id >= sizeof(charset_tbl)/sizeof(charset_tbl[0]))
		id = 0;
	return charset_tbl[id].cs_name;
}

int
charset_guess(char *p, char *ep)
{
	int i, code, id;
	static struct decode d;

	DINIT(&d, 0);
	d.cscode[1] = d.cscode[2] = d.cscode[3] = code = CS_ASCII;
	for (; p < ep; p++) {
		if (*p & 0x80) {
			/* XXX assume 8bit is used only for euc-jp */
			if (p + 1 == ep || (p[1] & 0x80) == 0)
				code = CS_96SET|CS_CODE('A');	/* 8859-1 */
			else
				code = CS_KANJI;		/* 2022-jp */
			break;
		}
		DOUTC(&d, *p);
		for (i = 0; i < 4; i++) {
			if (d.cscode[i] != CS_ASCII && d.cscode[i] != code) {
				if (code != CS_ASCII)
					return FDB_CS_JP2;
				code = d.cscode[i];
				break;
			}
		}
	}
	if (code == CS_ASCII)
		return FDB_CS_ASCII;
	for (id = 0; charset_tbl[id].cs_name != NULL; id++) {
		for (i = 0; i < 3; i++) {
			if (code == charset_tbl[id].cs_code[i])
				return id;
		}
	}
	return FDB_CS_JP2;
}

/* decode base64 binary */
static int
save_qdecode(FILE *fp, cbuf_t *buf)
{
	char c, *p, *ep;

	p = buf->ptr;
	ep = p + buf->len;
	while (p < ep) {
		c = *p++;
		if (c == '=') {
			if (p < ep && *p == '\n') {
				p++;	/* continuation */
				continue;
			}
			if (p + 3 >= ep)
				goto error;
			c = rhex[*(unsigned char *)p++] << 4;
			c |= rhex[*(unsigned char *)p++];
		}
		putc((u_char)c, fp);
	}
	return 1;

  error:
	return 0;
}

/* decode base64 binary */
static int
save_base64_decode(FILE *fp, cbuf_t *buf)
{
	int i, code, skip;
	char c, *p, *ep;

	p = buf->ptr;
	ep = p + buf->len;

	code = 0;	/* for debug */
	while (p < ep) {
		for (i = 0, code = 0, skip = 0; i < 4; p++) {
			if (p >= ep)
				goto error;
			if (*p == '=') {
				i++;
				code <<= 6;
				skip += 6;
				continue;
			}
			c = rb64[*(unsigned char *)p];
			if (c < 0)
				continue;
			i++;
			code = (code << 6) | c;
		}
		for (i = 8; i + skip <= 24; i += 8) {
			c = (code >> 16) & 0xff;
			putc((u_char)c, fp);
			code <<= 8;
		}
	}
	return 1;

  error:
	return 0;
}

/* decode uuencode binary */
static int
save_uudecode(FILE *fp, cbuf_t *buf)
{
	int i, code, cnt;
	char c, *p, *s, *ep;

	p = buf->ptr;
	ep = p + buf->len;

	code = 0;	/* for debug */
	/* seek "begin" */
	while (p < ep) {
		s = p;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		if (s + 6 < p && strncmp(s, "begin ", 6) == 0)
			break;
	}
	while (p < ep) {
		s = p;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		cnt = ruue[*(unsigned char *)s];
		if (cnt < 0) {
			if (s + 3 < p && strncmp(s, "end", 3) == 0)
				return 1;
			break;
		}
		s++;
		while (cnt > 0) {
			for (i = 0, code = 0; i < 4; i++, s++) {
				if (s >= p)
					goto error;
				code <<= 6;
				c = ruue[*(unsigned char *)s];
				if (c < 0)
					goto error;
				code |= c;
			}
			for (i = 0; i < 3; i++) {
				if (cnt-- == 0)
					break;
				c = (code >> 16) & 0xff;
				putc((u_char)c, fp);
				code <<= 8;
			}
		}
		if (++s != p)
			break;
	}
  error:
	return 0;
}

/* decode binhex binary */
static int
save_binhex(FILE *fp, cbuf_t *buf)
{
	int i, code;
	int st;
	char c, *p, *ep, prev;
	int rcnt, ccnt, slen;
	static int len[] = {
		1,	/* file name len */
		0,	/* file name (MODIFIED) */
		1 + 4 + 4 + 2,	/* skip: vers, type, creator, flags */
		4,	/* data length */
		4,	/* resource length */
		2,	/* crc */
		0,	/* data (MODIFIED) */
		2,	/* data crc */
		0,	/* resource (MODIFIED) */
		2,	/* resource crc */
		-1,	/* end of format */
	};

	p = buf->ptr;
	ep = p + buf->len;

	/* seek ":" in column 1 */
	while (p < ep) {
		if (*p == ':') {
			p++;
			break;
		}
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
	}

	code = 0;	/* for debug */
	st = 0;
	rcnt = 0;
	ccnt = 0;
	prev = 0;
	slen = 0;
	for (;;) {
		if (rcnt > 0) {
			rcnt--;
			c = prev;
		} else {
			if (ccnt == 0) {
				for (i = 0, code = 0; i < 4; i++, p++) {
					for (;;) {
						if (p >= ep)
							goto error;
						if (*p != '\n')
							break;
						p++;
					}
					code <<= 6;
					c = rhqx[*(unsigned char *)p];
					if (c < 0)
						goto error;
					code |= c;
				}
				ccnt = 3;
			}
			c = (code >> (--ccnt * 8)) & 0xff;
			if (rcnt < 0) {
				if (c) {
					rcnt = c - 1;
					continue;
				}
				rcnt = 0;
				c = 0x90;
			} else if ((u_char)c == 0x90) {
				rcnt = -1;
				continue;
			}
		}
		switch (st) {
		case 0:		/* file name length */
			len[1] = c;
			break;
		case 3:		/* data length */
			len[6] |= (u_char)c << ((3 - slen) * 8);
			break;
		case 4:		/* resource length */
			len[8] |= (u_char)c << ((3 - slen) * 8);
			break;
		case 6:		/* data */
			putc((u_char)c, fp);
			break;
		}
		if (++slen == len[st]) {
			slen = 0;
			while (len[++st] == 0)
				;
			if (len[st] < 0) {
				if (p < ep && *p == ':')
					return 1;
				break;
			}
		}
	}
  error:
	return 0;
}

void
save_part(struct state *state, int all)
{
	struct filedb *fdb;
	FILE *fp;
	char buf[CHARBLOCK];
	char *p, *ep;
	cbuf_t *cb;
	struct stat stbuf;
	int i;

	if ((fdb = state->message) == NULL)
		return;
	message_parseall(fdb);

	p = state->status;
	ep = state->status + sizeof(state->status);
	strlcpy(p, "File", ep - p);
	p += 4;
	if (fdb->filename.ptr) {
		i = snprintf(p, ep - p, " (%.*s)", fdb->filename.len,
		    fdb->filename.ptr);
		if (i <= 0) {
			strlcpy(state->status, "Quit", sizeof(state->status));
			return;
		}
		p += i;
	}
	strlcpy(p, ": ", ep - p);
	strlcpy(buf, "~/", sizeof(buf));
	if (edit_stline(state, buf, sizeof(buf), fname_completion) == NULL) {
		strlcpy(state->status, "Quit", sizeof(state->status));
		return;
	}
	if (buf[0] == '\0')
		strlcpy(buf, ".", sizeof(buf));
	else
		fname_expand(buf, buf);
	if (stat(buf, &stbuf) == 0 && S_ISDIR(stbuf.st_mode)) {
		if (fdb->filename.ptr) {
			sprintf(buf + strlen(buf), "/%.*s",
				fdb->filename.len, fdb->filename.ptr);
		} else {
			errno = EISDIR;
			snprintf(state->status, sizeof(state->status),
			     "%s: %s", buf, strerror(errno));
			return;
		}
	}
	if (stat(buf, &stbuf) == 0) {
		if (!edit_yn(state, "%s: overwrite? ", buf)) {
			strlcpy(state->status, "Quit", sizeof(state->status));
			return;
		}
	}
	if ((fp = fopen(buf, "w")) == NULL) {
		snprintf(state->status, sizeof(state->status),
		    "%s: %s", buf, strerror(errno));
		return;
	}
	if (all) {
		p = CP(&fdb->buf_mhdr);
		ep = CE(&fdb->buf_body);
		fwrite(p, ep - p, 1, fp);
	} else if ((fdb->flags & FDB_NOTEXT) && !(fdb->flags & FDB_INLINE)) {
		switch (fdb->flags & FDB_ENCODE) {
		case FDB_ENC_Q:
			save_qdecode(fp, &fdb->buf_body);
			break;
		case FDB_ENC_B64:
			save_base64_decode(fp, &fdb->buf_body);
			break;
		case FDB_ENC_UU:
			save_uudecode(fp, &fdb->buf_body);
			break;
		case FDB_ENC_BINHEX:
			save_binhex(fp, &fdb->buf_body);
			break;
		default:
			fwrite(CP(&fdb->buf_body), CL(&fdb->buf_body), 1, fp);
			break;
		}
	} else {
		for (i = fdb->skip_lines; (cb = fdb_read(fdb, i)) != NULL; i++) {
			print_jis(fp, "%.*s%s", CL(cb), CP(cb),
				  (fdb_ismore(fdb, i) ? "" : "\n"));
		}
	}
	fclose(fp);
	snprintf(state->status, sizeof(state->status), "Saved to %s", buf);
}

void
print_jis(FILE *fp, char *fmt, ...)
{
	va_list ap;
	u_char c, cs, *p;
	u_char cscode[4];
	u_char buf[CHARBLOCK];

	va_start(ap, fmt);
	vsnprintf((char *)buf, sizeof(buf), fmt, ap);
	va_end(ap);

	cscode[0] = CS_ASCII;
	cscode[1] = cscode[2] = cscode[3] = 0;

	p = buf;
	for (;;) {
		c = *p++;
		switch (c) {
		case CS_SS2:
			c = *p++;
			if (c == '\0')
				goto exit;
			if (cscode[0] != CS_KANA) {
				putc('\033', fp);
				putc('(', fp);
				putc('I', fp);
				cscode[0] = CS_KANA;
			}
			c &= ~0x80;
			putc(c, fp);
			break;

		case CS_SS3:
			c = *p++;
			if (c == '\0' || *p == '\0')
				goto exit;
			if (cscode[0] != CS_HOJO) {
				putc('\033', fp);
				putc('$', fp);
				putc('(', fp);
				putc('D', fp);
				cscode[0] = CS_HOJO;
			}
			c &= ~0x80;
			putc(c, fp);
			c = *p++ & ~0x80;
			putc(c, fp);
			break;

		case CS_SSX:
			cs = *p++;
			if (cs == '\0' || *p == '\0' ||
			    ((cs & CS_DWIDTH) && p[1] == '\0'))
				goto exit;
			if (cs & CS_96SET) {
				if (cs != cscode[2]) {
					putc('\033', fp);
					if (cs & CS_DWIDTH)
						putc('$', fp);
					putc('.', fp);
					putc('@' + CS_CODE(cs), fp);
					cscode[2] = cs;
				}
				putc('\033', fp);
				putc('N', fp);
			} else {
				if (cs != cscode[0]) {
					putc('\033', fp);
					if (cs & CS_DWIDTH)
						putc('$', fp);
					putc('(', fp);
					putc('@' + CS_CODE(cs), fp);
					cscode[0] = cs;
				}
			}
			c = *p++ & ~0x80;
			putc(c, fp);
			if (cs & CS_DWIDTH) {
				c = *p++ & ~0x80;
				putc(c, fp);
			}
			break;

		default:
			if (c & 0x80) {
				if (*p == '\0') {
					c = '\0';
					goto exit;
				}
				if (cscode[0] != CS_KANJI) {
					putc('\033', fp);
					putc('$', fp);
					putc('B', fp);
					cscode[0] = CS_KANJI;
				}
				c &= ~0x80;
				putc(c, fp);
				c = *p++ & ~0x80;
				putc(c, fp);
			} else {
  exit:
				if (cscode[0] != CS_ASCII) {
					putc('\033', fp);
					putc('(', fp);
					putc('B', fp);
					cscode[0] = CS_ASCII;
				}
				if (c == '\0')
					return;
				putc(c, fp);
			}
		}
	}
}


/* XXX should be merged with print_jis() */
static int
encode_hdr(u_char **pp, u_char *ep, cbuf_t *res, int limit)
{
	const u_char *csname;
	u_char *s, *p;
	u_char c;
	int id;
	int elen;
	u_char qbuf[4];
	u_char cs;
	u_char cscode[4];
	static struct decode d;

	cscode[0] = CS_ASCII;
	cscode[1] = cscode[2] = cscode[3] = 0;

	id = charset_guess(*pp, ep);
	if (id == FDB_CS_ASCII) {
		/* should not happen */
		res->ptr = (char *)*pp;
		res->len = ep - *pp;
		*pp = ep;
		return res->len;
	}
	DINIT(&d, id);

	csname = charset_name(id);
	limit -= 2 + strlen((char *)csname) + 3 + 2;
	if (limit < 4 + 4 + 4)	/* esc + word + esc */
		return 0;

	OUTC(&d, '='); OUTC(&d, '?');
	for (; *csname; csname++)
		OUTC(&d, *csname);
	OUTC(&d, '?'); OUTC(&d, charset_tbl[id].cs_enc); OUTC(&d, '?');
	if (charset_tbl[id].cs_enc == 'Q') {
		for (p = *pp; p < ep; p++) {
			if (*p == CS_SSX) {
				c = p[2] | 0x80;
				snprintf(qbuf, sizeof(qbuf), "%02x", c);
				p += 2;
				OUTC(&d, '=');
				OUTC(&d, qbuf[0]);
				OUTC(&d, qbuf[1]);
			} else if (*p == '=') {
				OUTC(&d, '=');
				OUTC(&d, '3');
				OUTC(&d, 'd');
			} else
				OUTC(&d, *p);
		}
		OUTC(&d, '?'); OUTC(&d, '=');
		res->ptr = d.buf;
		res->len = d.cnt;
		elen = p - *pp;
		*pp = p;
		return elen;
	}
	elen = 0;
	limit = limit * 3 / 4;		/* B encoding */
	for (s = p = *pp; p < ep; s = p) {
		c = *p++;

		switch (c) {
		case '\0':
			goto exit;

		case CS_SS2:
			c = *p++;
			if (c == '\0')
				goto exit;
			if (++elen > limit)
				goto exit;
			if (cscode[0] != CS_KANA) {
				if (cscode[0] == CS_ASCII)
					limit -= 3;
				if ((elen += 3) > limit)
					goto exit;
				BOUTC(&d, '\033');
				BOUTC(&d, '(');
				BOUTC(&d, 'I');
				cscode[0] = CS_KANA;
			}
			c &= ~0x80;
			BOUTC(&d, c);
			break;

		case CS_SS3:
			c = *p++;
			if (c == '\0' || *p == '\0')
				goto exit;
			if ((elen += 2) > limit)
				goto exit;
			if (cscode[0] != CS_HOJO) {
				if (cscode[0] == CS_ASCII)
					limit -= 3;
				if ((elen += 4) > limit)
					goto exit;
				BOUTC(&d, '\033');
				BOUTC(&d, '$');
				BOUTC(&d, '(');
				BOUTC(&d, 'D');
				cscode[0] = CS_HOJO;
			}
			c &= ~0x80;
			BOUTC(&d, c);
			c = *p++ & ~0x80;
			BOUTC(&d, c);
			break;

		case CS_SSX:
			cs = *p++;
			if (cs == '\0' || *p == '\0' ||
			    ((cs & CS_DWIDTH) && p[1] == '\0'))
				goto exit;
			if ((elen += (cs & CS_DWIDTH) ? 2 : 1) > limit)
				goto exit;
			if (cs & CS_96SET) {
				if ((elen += 2) > limit)
					goto exit;
				if (cs != cscode[2]) {
					elen += (cs & CS_DWIDTH) ? 4 : 3;
					if (elen > limit)
						goto exit;
					BOUTC(&d, '\033');
					if (cs & CS_DWIDTH)
						BOUTC(&d, '$');
					BOUTC(&d, '.');
					BOUTC(&d, '@' + CS_CODE(cs));
					cscode[2] = cs;
				}
				BOUTC(&d, '\033');
				BOUTC(&d, 'N');
			} else {
				if (cs != cscode[0]) {
					if (cscode[0] == CS_ASCII)
						limit -= 3;
					elen += (cs & CS_DWIDTH) ? 4 : 3;
					if (elen > limit)
						goto exit;
					BOUTC(&d, '\033');
					if (cs & CS_DWIDTH)
						BOUTC(&d, '$');
					BOUTC(&d, '(');
					BOUTC(&d, '@' + CS_CODE(cs));
					cscode[0] = cs;
				}
			}
			c = *p++ & ~0x80;
			BOUTC(&d, c);
			if (cs & CS_DWIDTH) {
				c = *p++ & ~0x80;
				BOUTC(&d, c);
			}
			break;

		default:
			if (c & 0x80) {
				if (*p == '\0')
					goto exit;
				if ((elen += 2) > limit)
					goto exit;
				if (cscode[0] != CS_KANJI) {
					if (cscode[0] == CS_ASCII)
						limit -= 3;
					if ((elen += 3) > limit)
						goto exit;
					BOUTC(&d, '\033');
					BOUTC(&d, '$');
					BOUTC(&d, 'B');
					cscode[0] = CS_KANJI;
				}
				c &= ~0x80;
				BOUTC(&d, c);
				c = *p++ & ~0x80;
				BOUTC(&d, c);
			} else {
				if (++elen > limit)
					goto exit;
				if (cscode[0] != CS_ASCII) {
					limit += 3;
					if ((elen += 3) > limit)
						goto exit;
					BOUTC(&d, '\033');
					BOUTC(&d, '(');
					BOUTC(&d, 'B');
					cscode[0] = CS_ASCII;
				}
				BOUTC(&d, c);
			}
		}
	}
  exit:
	if (cscode[0] != CS_ASCII) {
		BOUTC(&d, '\033');
		BOUTC(&d, '(');
		BOUTC(&d, 'B');
		cscode[0] = CS_ASCII;
	}
	BFLUSH(&d);
	OUTC(&d, '?'); OUTC(&d, '=');
	res->ptr = d.buf;
	res->len = d.cnt;
	elen = s - *pp;
	*pp = s;
	return elen;
}

void
print_hdr(FILE *fp, char *fmt, ...)
{
	va_list ap;
	u_char *p, *s, *t;
	int width, slen;
	int encode;
	cbuf_t cbuf;
	u_char buf[CHARBLOCK];
	static struct decode d;
	int fold = 78;

	va_start(ap, fmt);
	vsnprintf((char *)buf, sizeof(buf), fmt, ap);
	va_end(ap);

#if 0
	if (strncasecmp(buf, "Content-", 8) == 0
	||  strncasecmp(buf, "To:", 3) == 0
	||  strncasecmp(buf, "Cc:", 3) == 0
	||  strncasecmp(buf, "From:", 5) == 0
	||  strncasecmp(buf, "Sender:", 7) == 0
	||  strncasecmp(buf, "Apparently-To:", 14) == 0
	||  strncasecmp(buf, "Resent-To:", 10) == 0
	||  strncasecmp(buf, "Resent-Cc:", 10) == 0
	||  strncasecmp(buf, "Resent-From:", 12) == 0) {
		sflag = 1;
	} else {
		sflag = 0;
	}
#endif

	/* convert euc if jis code */
	for (p = buf; *p; p++) {
		if (*p == '\033') {
			/* would be JIS text -- convert euc */
			DINIT(&d, 0);
			for (p = buf; *p; p++)
				DOUTC(&d, *p);
			memcpy(buf, d.buf, d.cnt);
			buf[d.cnt] = '\0';
			break;
		}
	}

	width = 0;
	p = buf;
	while (*p) {
		/* get space */
		for (s = p; *p; p++) {
			if (!isspace(*p))
				break;
		}
		/* get token */
		encode = 0;
		for (t = p; *p; p++) {
			if (isspace(*p))
				break;
			if (*p & 0x80)
				encode = 1;
		}
		if (!encode) {
			/*
			 * cannot split token
			 * insert "\n " if there is no room
			 */
			if (width > 0 && width + (p - s) > fold) {
				putc('\n', fp);
				putc(' ', fp);
				width = 1;
				s = t;		/* skip leading spaces */
			}
			fwrite(s, p - s, 1, fp);
			width += p - s;
		} else {
			slen = t - s;
			width += slen;
			while (t < p) {
				if (encode_hdr(&t, p, &cbuf, fold - width)) {
					if (slen) {
						/* OK put leading spaces */
						fwrite(s, slen, 1, fp);
						slen = 0;
					}
					fwrite(cbuf.ptr, cbuf.len, 1, fp);
					width += cbuf.len;
				} else {
					if (fold == CHARBLOCK) {
						/* give up */
						break;
					} else if (width == 1) {
						/* cannot fold */
						fold = CHARBLOCK;
					} else {
						putc('\n', fp);
						putc(' ', fp);
						width = 1;
					}
				}
			}
		}
		if (*p == '\n') {
			putc(*p, fp);
			p++;
			width = 0;
		}
	}
}

#define	ENC(c1,c2,c3)	(((c1) << 16) | ((c2) << 8) | (c3))
#define	EOUT(fp, c, n)	putc(b64[((c) >> (6 * (3-n))) & 0x3f], (fp))

void
save_base64_encode(FILE *fp, cbuf_t *buf)
{
	int w;
	unsigned long c;
	u_char *p, *ep;

	for (p = (u_char *)buf->ptr, ep = p + buf->len, w = 0; p + 2 < ep; p += 3) {
		c = ENC(p[0], p[1], p[2]);
		EOUT(fp, c, 0); EOUT(fp, c, 1); EOUT(fp, c, 2); EOUT(fp, c, 3);
		if ((w += 4) >= 72) {
			putc('\n', fp);
			w = 0;
		}
	}
	switch (ep - p) {
	case 1:
		c = ENC(p[0], 0, 0);
		EOUT(fp, c, 0); EOUT(fp, c, 1);
		putc('=', fp); putc('=', fp);
		w += 4;
		break;
	case 2:
		c = ENC(p[0], p[1], 0);
		EOUT(fp, c, 0); EOUT(fp, c, 1); EOUT(fp, c, 2);
		putc('=', fp);
		w += 4;
		break;
	}
	if (w != 0)
		putc('\n', fp);
}

#ifdef TEST_MAIN
main()
{
	char buf[] = "=?iso-2022-jp?B?GyRCTytGLzJKM1g4JjVmPWobKEI=?=";
	char *p;
	cbuf_t res;

	decode_init();
	p = buf;
	if (decode_header(NULL, &p, buf + sizeof(buf)-1, &res))
		printf("%.*s\n", res.len, res.ptr);
	else
		printf("Decode failed\n");
	exit(0);
}
#endif
