/* $Id: decode.c,v 1.58 2003/04/17 07:35:53 itojun Exp $ */

/*-
 * Copyright (c) 1998-2001 Atsushi Onoe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <stddef.h>

#ifdef ZLIB
#include <zlib.h>
#endif /* ZLIB */

#include "cue.h"
#include "decode.h"

char b64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
char hex[] = "0123456789abcdef0123456789ABCDEF";
char uue[] = "`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_";
char hqx[] = "!\"#$%&'()*+,-012345689@ABCDEFGHIJKLMNPQRSTUVXYZ[`abcdefhijklmpqr";

char rb64[256], rhex[256], ruue[256], rhqx[256];

void
decode_init(void)
{
	int i;

	memset(rb64, -1, sizeof(rb64));
	for (i = 0; i < sizeof(b64); i++)
		rb64[((u_char *)b64)[i]] = i;
	memset(rhex, -1, sizeof(rhex));
	for (i = 0; i < sizeof(hex); i++)
		rhex[((u_char *)hex)[i]] = i & 0x0f;
	memset(ruue, -1, sizeof(ruue));
	for (i = 0; i < sizeof(uue); i++)
		ruue[((u_char *)uue)[i]] = i;
	memset(rhqx, -1, sizeof(rhqx));
	for (i = 0; i < sizeof(hqx); i++)
		rhqx[((u_char *)hqx)[i]] = i;
}

static void
DINIT(struct decode *dp, int flags)
{

	if (dp->bufsiz != CHARBLOCK) {
		dp->bufsiz = CHARBLOCK;
		dp->buf = realloc(dp->buf, dp->bufsiz);
	}
	dp->cnt = 0;
	dp->flsbuf = NULL;
	dp->pend = 0;
	dp->esc = 0;
	dp->err = 0;
	dp->encoded = 0;
	dp->noconv = 0;
	dp->notext = 0;
	dp->sjis = 0;
	dp->lang = DL_JAPANESE;
	dp->mode = DM_ASCII;
	dp->bclen = 0;

	if (flags & FDB_NOCONV)
		dp->noconv = 1;
	if (flags & FDB_NOTEXT)
		dp->notext = 1;
	if (flags & FDB_SJIS)
		dp->sjis = 1;

	if (flags & FDB_KOREAN)
		dp->lang = DL_KOREAN;
	else if (flags & FDB_88591)
		dp->lang = DL_88591;
}

static void
OUTERR(struct decode *dp, u_char c)
{

	if (c < ' ' && c != '\n' && c != '\t') {
		OUTC(dp, '^'); OUTC(dp, c + '@');
		dp->encoded = 1;
	} else if (!(c & 0x80)) {
		OUTC(dp, c);
	} else {
		OUTC(dp, '\\'); OUTC(dp, 'x');
		OUTC(dp, hex[c >> 4]); OUTC(dp, hex[c & 0xf]);
		dp->encoded = 1;
	}
	dp->err = 1;
}

static void
OUTPEND(struct decode *dp)
{
	int i;
	u_char c;

	for (i = 0; i < 4; i++) {
		c = dp->pend >> 24;
		dp->pend <<= 8;
		if (c != 0)
			OUTERR(dp, c);
	}
}

static void
DOUTC(struct decode *dp, char c0)
{
	u_char c = (u_char)c0;
	u_char s;

	if (dp->noconv) {
		OUTC(dp, c);
		return;
	}
	if (dp->notext) {
		OUTERR(dp, c);
		return;
	}

	if (dp->sjis == 1 && (c & 0x80)) {		/* SJIS 1st */
		if (dp->pend)
			OUTPEND(dp);
		if (c >= 0xa1 && c <= 0xdf) {		/* KANA */
			OUTC(dp, 0x8e);
			OUTC(dp, c);
		} else if (c >= 0x81 && c <= 0xfc) {
			dp->sjis = 2;
			dp->pend = c;
		} else {
			OUTERR(dp, c);
		}
		dp->encoded = 1;
		return;
	}
	if (dp->sjis == 2) {				/* SJIS 2nd */
		s = dp->pend;
		s -= (s <= 0x9f) ? 0x71 : 0xb1;
		s = (s * 2 + 1) | 0x80;
		if (s >= 0xa0 && s <= 0xfd && c >= 0x9f && c <= 0xfc) {
			OUTC(dp, s + 1);
			OUTC(dp, c + 2);
		} else if (s >= 0xa1 && s <= 0xfe && c >= 0x40 && c <= 0x7e) {
			OUTC(dp, s);
			OUTC(dp, c + 0x61);
		} else if (s >= 0xa1 && s <= 0xfe && c >= 0x80 && c <= 0x9e) {
			OUTC(dp, s);
			OUTC(dp, c + 0x60);
		} else {
			OUTERR(dp, dp->pend);
			OUTERR(dp, c);
		}
		dp->pend = 0;
		dp->sjis = 1;
		return;
	}

	if (dp->esc) {
		dp->pend = (dp->pend << 8) | c;
		dp->esc = 0;
		switch (dp->pend) {
		case 0x1b24:	/* ESC-$ */
		case 0x1b28:	/* ESC-( */
		case 0x1b2e:	/* ESC-. */
		case 0x1b2428:	/* ESC-$-( */
			dp->esc = 1;
			return;
		case 0x1b2e41:	/* ESC-.-A */
			dp->mode = DM_88591;	dp->pend = 0;	return;
		case 0x1b2440:	/* ESC-$-@ */
		case 0x1b2442:	/* ESC-$-B */
			dp->mode = DM_KANJI;	dp->pend = 0;	return;
		case 0x1b2849:	/* ESC-(-I */
			dp->mode = DM_KANA;	dp->pend = 0;	return;
		case 0x1b2842:	/* ESC-(-B */
		case 0x1b284a:	/* ESC-(-J */
			dp->mode = DM_ASCII;	dp->pend = 0;	return;
		case 0x1b242842: /* ESC-$-(-B */
			dp->mode = DM_KANJI;	dp->pend = 0;	return;
		case 0x1b242843: /* ESC-$-(-C */
			dp->mode = DM_KOREAN;	dp->pend = 0;	return;
		case 0x1b4e:	/* ESC-N */
			dp->mode = DM_88591;	dp->pend = 0;	return;
			return;
		default:
			dp->pend >>= 8;	 /* delete last */
			OUTPEND(dp);
			break;
		}
	}

	if (dp->pend == 0x8e) {
		if (dp->lang == DL_JAPANESE && c >= 0xa1 && c < 0xff) {
			OUTC(dp, 0x8e);
			OUTC(dp, c);
		} else {
			dp->pend = (dp->pend << 8) | c;
			OUTPEND(dp);
		}
		dp->pend = 0;
		return;
	}
	if (dp->pend) {
		if ((dp->lang == DL_JAPANESE && c >= 0xa1 && c < 0xff) ||
		    (dp->mode == DM_KANJI && c >= 0x21 && c < 0x7f)) {
			if (!(c & 0x80)) {
				c |= 0x80;
				dp->encoded = 1;
			}
			OUTC(dp, dp->pend);
			OUTC(dp, c);
		} else if ((dp->lang == DL_KOREAN && c >= 0xa1 && c < 0xff) ||
		    (dp->mode == DM_KOREAN && c >= 0x21 && c < 0x7f)) {
			OUTC(dp, 0x8d);
			OUTC(dp, dp->pend);
			OUTC(dp, 0x8d);
			OUTC(dp, 0x80 | c);
		} else {
			dp->pend = (dp->pend << 8) | c;
			OUTPEND(dp);
		}
		dp->pend = 0;
		return;
	}

	if (c == 0x1b) {
		dp->esc = 1;
		dp->pend = c;
		dp->encoded = 1;
	} else if (c == 0x0e && dp->lang == DL_JAPANESE) {
		dp->mode = DM_KANA;
		dp->encoded = 1;
	} else if (c == 0x0f) {
		dp->mode = DM_ASCII;
		dp->encoded = 1;
	} else if (c == '\n' || c == '\t') {
		OUTC(dp, c);
	} else if (c == '\r') {
		dp->encoded = 1;
	} else if (c == '\014') {
		OUTC(dp, '^'); OUTC(dp, c + '@');
		dp->encoded = 1;
	} else if (c < ' ' || (dp->lang != DL_88591 && c == 0x7f)) {
		dp->encoded = 1;
		OUTERR(dp, c);
	} else if (c & 0x80) {
		switch (dp->lang) {
		case DL_JAPANESE:
			if (c == 0x8e || (c >= 0xa1 && c < 0xff))
				dp->pend = c;
			else
				OUTERR(dp, c);
			break;
		case DL_KOREAN:
			dp->encoded = 1;
			if (c >= 0xa1 && c < 0xff)
				dp->pend = c;
			else
				OUTERR(dp, c);
			break;
		case DL_88591:
			if (c >= 0xa0) {
				OUTC(dp, 0x8c);
				OUTC(dp, c);
				dp->encoded = 1;
			} else
				OUTERR(dp, c);
			break;
		default:
			break;
		}
	} else {
		switch (dp->mode) {
		case DM_ASCII:
			OUTC(dp, c);
			break;
		case DM_88591:
			OUTC(dp, 0x8c);
			OUTC(dp, 0x80 | c);
			dp->encoded = 1;
			break;
		case DM_KANA:
			OUTC(dp, 0x8e);
			OUTC(dp, 0x80 | c);
			dp->encoded = 1;
			break;
		case DM_KANJI:
		case DM_KOREAN:
			dp->pend = 0x80 | c;
			dp->encoded = 1;
			break;
		}
	}
}

/* decode 1 header */
int
decode_header(cbuf_t *hdr, int flags)
{
	char c, *p, *b, *ep;
	cbuf_t charset;
	int i, code, skip;
	int qenc;
	static struct decode d;

	DINIT(&d, flags);
	qenc = 0;
	for (p = hdr->ptr, ep = p + hdr->len; p < ep; ) {
		if (*p == '\n') {
			b = p - 1;
			while (++p < ep) {
				if (*p != ' ' && *p != '\t')
					break;
			}
			if (p == ep)
				break;
			if (*b != '=' || *p != '=')
				DOUTC(&d, ' ');
			continue;
		}
		if (qenc) {
			if (*p == '?' && p + 1 < ep && p[1] == '=') {
				qenc = 0;
				p += 2;
				continue;
			}
			if (*p == '=') {
				if (p + 1 < ep && p[1] == '\n') {
					p += 2;	/* continuation */
					continue;
				}
				if (p + 2 < ep) {
					p++;
					c = rhex[*(unsigned char *)p++] << 4;
					c |= rhex[*(unsigned char *)p++];
					DOUTC(&d, c);
					continue;
				}
			}
		}
		if (*p != '=' || p + 7 > ep || p[1] != '?') {
  nodecode:
			DOUTC(&d, *p);
			p++;
			continue;
		}

		for (CSETP(&charset, p + 2); ; CL(&charset)++) {
			if (CE(&charset) + 4 >= ep)
				goto nodecode;
			if (CE(&charset)[0] == '?' &&
			    (CE(&charset)[1]=='B' || CE(&charset)[1]=='Q' ||
			     CE(&charset)[1]=='b' || CE(&charset)[1]=='q') &&
			    CE(&charset)[2] == '?')
				break;
		}
		if (CMATCH("iso-8859-1", &charset))
			d.lang = DL_88591;
		else if (CMATCH("iso-2022-jp", &charset) ||
		    CMATCH("euc-jp", &charset))
			d.lang = DL_JAPANESE;
		else if (CMATCH("shift_jis", &charset)) {
			d.lang = DL_JAPANESE;
			d.sjis = 1;
		} else if (CMATCH("euc-kr", &charset) ||
		    CMATCH("ks_c_5601-1987", &charset))
			d.lang = DL_KOREAN;
		else
			goto nodecode;
		d.encoded = 1;
		p = CE(&charset) + 3;
		if (CE(&charset)[1] == 'Q' || CE(&charset)[1] == 'q') {
			qenc = 1;
			continue;
		}
		for (;;) {
			if (p + 1 < ep && *p == '?' && p[1] == '=') {
				p += 2;
				break;
			}
			for (i = 0, code = 0, skip = 0; i < 4; i++, p++) {
				if (p >= ep)
					goto error;
				code <<= 6;
				if (*p == '=') {
					skip += 6;
				} else {
					c = rb64[*(u_char *)p];
					if (c == -1)
						goto error;
					code |= c;
				}
			}
			for (i = 8; i + skip <= 24; i += 8) {
				c = (code >> 16) & 0xff;
				DOUTC(&d, c);
				code <<= 8;
			}
		}
	}
	if (d.encoded) {
		hdr->ptr = d.buf;
		hdr->len = d.cnt;
		return (d.err ? -1 : 1);
	}
  error:
	return 0;
}

/* decode 1 line body */
static int
decode_qtext(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	static struct decode d;

	DINIT(&d, flags);
	for (p = *pp, c = '\n'; p < ep; ) {
		if ((c = *p++) == '\n')
			break;
		if (c == '=') {
			d.encoded = 1;
			if (p < ep && *p == '\n') {
				p++;	/* continuation */
				continue;
			}
			if (p + 3 >= ep)
				goto error;
			c = rhex[*(unsigned char *)p++] << 4;
			c |= rhex[*(unsigned char *)p++];
		}
		DOUTC(&d, c);
	}
	if (d.encoded) {
		res->ptr = d.buf;
		res->len = d.cnt;
		*pp = p;
		return (d.err ? -1 : 1);
	}
  error:
	if (c != '\n') {
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode base64 text */
static int
decode_base64(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	int i, code, skip;
	static struct decode d;

	DINIT(&d, flags);
	for (p = *pp; p < ep; ) {
		if (!d.noconv && d.cnt > 0 && d.buf[d.cnt-1] == '\n' &&
		    d.sjis <= 1 && d.pend == 0 && d.esc == 0) {
			if (*p == '\n')
				p++;
			break;
		}
		if (*p == '\n') {
			p++;
			if (*p == '-')
				break;
			continue;
		}
		for (i = 0, code = 0, skip = 0; i < 4; p++) {
			if (p >= ep)
				goto error;
			if (isspace(*p))
				continue;
			i++;
			code <<= 6;
			if (*p == '=') {
				skip += 6;
				continue;
			}
			c = rb64[*(unsigned char *)p];
			if (c < 0)
				goto error;
			code |= c;
		}
		for (i = 8; i + skip <= 24; i += 8) {
			c = (code >> 16) & 0xff;
			DOUTC(&d, c);
			code <<= 8;
		}
	}
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);

  error:
	while (p < ep) {
		if (*p++ == '\n')
			break;
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode uuencode text */
static int
decode_uuencode(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p, *s;
	int i, code, cnt;
	static struct decode d;

	DINIT(&d, flags);

	for (p = *pp; p < ep; ) {
		if (!d.noconv && d.cnt > 0 && d.buf[d.cnt-1] == '\n') {
			if (*p == '\n')
				p++;
			break;
		}
		cnt = ruue[*(unsigned char *)p];
		if (cnt < 0) {
			s = p;
			while (p < ep) {
				if (*p++ == '\n')
					break;
			}
			if (s + 3 < p && strncmp(s, "end", 3) == 0) {
				p = ep;
				break;
			}
			continue;
		}
		p++;
		while (cnt > 0) {
			for (i = 0, code = 0; i < 4; i++, p++) {
				if (p >= ep)
					goto error;
				code <<= 6;
				c = ruue[*(unsigned char *)p];
				if (c < 0)
					goto error;
				code |= c;
			}
			for (i = 0; i < 3; i++) {
				if (cnt-- == 0)
					break;
				c = (code >> 16) & 0xff;
				DOUTC(&d, c);
				code <<= 8;
			}
		}
		if (*p != '\n')
			break;
		p++;
	}
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);

  error:
	while (p < ep) {
		if (*p++ == '\n')
			break;
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode binhex text */
/* XXX */
static int
decode_binhex(char **pp, char *ep, cbuf_t *res, int flags)
{
	char *p;

	/* start */
	p = *pp;
	if (*p != ':') {
		res->ptr = p;
		res->len = 0;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		*pp = p;
		return 0;
	}
	/* end */
	while (p < ep) {
		if (*p++ == ':')
			break;
	}

	res->ptr = banner("BINHEX");
	res->len = strlen(res->ptr);
	*pp = p;
	return 1;
}

#ifdef ZLIB
/* decode x-gzip64 text */
static int
decode_gzip64(char **pp, char *ep, cbuf_t *res, int flags)
{
	char *p, *q;
	u_char *u;
	int cc, zflags;
	z_stream z;
	char dbuf[CHARBLOCK];
	static struct decode d;

	DINIT(&d, flags);
	p = *pp;
	if (decode_base64(&p, ep, res, FDB_NOCONV) == 0)
		goto err;

	/* XXX: skip gzip header */
	u = (u_char *)res->ptr;
	if (u[0] != 0x1f || u[1] != 0x8b || u[2] != Z_DEFLATED)
		goto err;
	u += 3;
	zflags = *u++;
	u += 6;			/* time, xflags, oscode */
	if (zflags & 0x04) {	/* extra field */
		cc = (u[1] << 8) | u[0];
		u += cc + 2;
	}
	if (zflags & 0x08) {	/* orig name */
		while (*u++)
			;
	}
	if (zflags & 0x10) {	/* comment */
		while (*u++)
			;
	}
	if (zflags & 0x02) {	/* header crc */
		u += 2;
	}

	memset(&z, 0, sizeof(z));
	z.next_in = (Bytef *)u;
	z.avail_in = res->len - ((char *)u - res->ptr) + 4/*XXX*/;
	z.next_out = (Bytef *)dbuf;
	z.avail_out = sizeof(dbuf);
	if (inflateInit2(&z, -15) != Z_OK)
		goto err;
	for (;;) {
		cc = inflate(&z, Z_SYNC_FLUSH);
		if (cc == Z_STREAM_END)
			break;
		if (cc < 0)
			goto err;
		if (z.avail_out == 0) {
			for (q = dbuf; q < dbuf + sizeof(dbuf); q++) {
				DOUTC(&d, *q);
			}
			z.next_out = (Bytef *)dbuf;
			z.avail_out = sizeof(dbuf);
		}
	}
	inflateEnd(&z);
	res->ptr = d.buf;
	res->len = d.cnt;
	*pp = p;
	return (d.err ? -1 : 1);
 err:
	return decode_base64(pp, ep, res, flags);
}
#endif /* ZLIB */

/* decode 1 line body */
int
decode_text(char **pp, char *ep, cbuf_t *res, int flags)
{
	char c, *p;
	static struct decode d;

	if (flags & FDB_QENCODE)
		return decode_qtext(pp, ep, res, flags);
	if (flags & FDB_BASE64)
		return decode_base64(pp, ep, res, flags);
#ifdef ZLIB
	if (flags & FDB_GZIP64)
		return decode_gzip64(pp, ep, res, flags);
#endif /* ZLIB */
	if (flags & FDB_UUENCODE)
		return decode_uuencode(pp, ep, res, flags);
	if (flags & FDB_BINHEX)
		return decode_binhex(pp, ep, res, flags);

	DINIT(&d, flags);
	for (p = *pp; p < ep; ) {
		if ((c = *p++) == '\n')
			break;
		DOUTC(&d, c);
	}
	if (d.encoded) {
		res->ptr = d.buf;
		res->len = d.cnt;
		*pp = p;
		return (d.err ? -1 : 1);
	}
	res->ptr = *pp;
	res->len = p - *pp;
	*pp = p;
	return 0;
}

/* decode base64 binary */
static int
save_qdecode(FILE *fp, cbuf_t *buf)
{
	char c, *p, *ep;

	p = buf->ptr;
	ep = p + buf->len;
	while (p < ep) {
		c = *p++;
		if (c == '=') {
			if (p < ep && *p == '\n') {
				p++;	/* continuation */
				continue;
			}
			if (p + 3 >= ep)
				goto error;
			c = rhex[*(unsigned char *)p++] << 4;
			c |= rhex[*(unsigned char *)p++];
		}
		putc((u_char)c, fp);
	}
	return 1;

  error:
	return 0;
}

/* decode base64 binary */
static int
save_base64_decode(FILE *fp, cbuf_t *buf)
{
	int i, code, skip;
	char c, *p, *ep;

	p = buf->ptr;
	ep = p + buf->len;

	code = 0;	/* for debug */
	while (p < ep) {
		if (*p == '\n') {
			p++;
			continue;
		}
		for (i = 0, code = 0, skip = 0; i < 4; i++, p++) {
			if (p >= ep)
				goto error;
			code <<= 6;
			if (*p == '=') {
				skip += 6;
				continue;
			}
			c = rb64[*(unsigned char *)p];
			if (c < 0)
				goto error;
			code |= c;
		}
		for (i = 8; i + skip <= 24; i += 8) {
			c = (code >> 16) & 0xff;
			putc((u_char)c, fp);
			code <<= 8;
		}
	}
	return 1;

  error:
	return 0;
}

/* decode uuencode binary */
static int
save_uudecode(FILE *fp, cbuf_t *buf)
{
	int i, code, cnt;
	char c, *p, *s, *ep;

	p = buf->ptr;
	ep = p + buf->len;

	code = 0;	/* for debug */
	/* seek "begin" */
	while (p < ep) {
		s = p;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		if (s + 6 < p && strncmp(s, "begin ", 6) == 0)
			break;
	}
	while (p < ep) {
		s = p;
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
		cnt = ruue[*(unsigned char *)s];
		if (cnt < 0) {
			if (s + 3 < p && strncmp(s, "end", 3) == 0)
				return 1;
			break;
		}
		s++;
		while (cnt > 0) {
			for (i = 0, code = 0; i < 4; i++, s++) {
				if (s >= p)
					goto error;
				code <<= 6;
				c = ruue[*(unsigned char *)s];
				if (c < 0)
					goto error;
				code |= c;
			}
			for (i = 0; i < 3; i++) {
				if (cnt-- == 0)
					break;
				c = (code >> 16) & 0xff;
				putc((u_char)c, fp);
				code <<= 8;
			}
		}
		if (++s != p)
			break;
	}
  error:
	return 0;
}

/* decode binhex binary */
static int
save_binhex(FILE *fp, cbuf_t *buf)
{
	int i, code;
	int st;
	char c, *p, *ep, prev;
	int rcnt, ccnt, slen;
	static int len[] = {
		1,	/* file name len */
		0,	/* file name (MODIFIED) */
		1 + 4 + 4 + 2,	/* skip: vers, type, creator, flags */
		4,	/* data length */
		4,	/* resource length */
		2,	/* crc */
		0,	/* data (MODIFIED) */
		2,	/* data crc */
		0,	/* resource (MODIFIED) */
		2,	/* resource crc */
		-1,	/* end of format */
	};

	p = buf->ptr;
	ep = p + buf->len;

	/* seek ":" in column 1 */
	while (p < ep) {
		if (*p == ':') {
			p++;
			break;
		}
		while (p < ep) {
			if (*p++ == '\n')
				break;
		}
	}

	code = 0;	/* for debug */
	st = 0;
	rcnt = 0;
	ccnt = 0;
	prev = 0;
	slen = 0;
	for (;;) {
		if (rcnt > 0) {
			rcnt--;
			c = prev;
		} else {
			if (ccnt == 0) {
				for (i = 0, code = 0; i < 4; i++, p++) {
					for (;;) {
						if (p >= ep)
							goto error;
						if (*p != '\n')
							break;
						p++;
					}
					code <<= 6;
					c = rhqx[*(unsigned char *)p];
					if (c < 0)
						goto error;
					code |= c;
				}
				ccnt = 3;
			}
			c = (code >> (--ccnt * 8)) & 0xff;
			if (rcnt < 0) {
				if (c) {
					rcnt = c - 1;
					continue;
				}
				rcnt = 0;
				c = 0x90;
			} else if ((u_char)c == 0x90) {
				rcnt = -1;
				continue;
			}
		}
		switch (st) {
		case 0:		/* file name length */
			len[1] = c;
			break;
		case 3:		/* data length */
			len[6] |= (u_char)c << ((3 - slen) * 8);
			break;
		case 4:		/* resource length */
			len[8] |= (u_char)c << ((3 - slen) * 8);
			break;
		case 6:		/* data */
			putc((u_char)c, fp);
			break;
		}
		if (++slen == len[st]) {
			slen = 0;
			while (len[++st] == 0)
				;
			if (len[st] < 0) {
				if (p < ep && *p == ':')
					return 1;
				break;
			}
		}
	}
  error:
	return 0;
}

void
save_part(struct state *state, int all)
{
	struct filedb *fdb;
	FILE *fp;
	char buf[CHARBLOCK];
	char *p, *ep;
	cbuf_t *cb;
	struct stat stbuf;
	int i;

	if ((fdb = state->message) == NULL)
		return;
	message_parseall(fdb);

	p = state->status;
	strcpy(p, "File");
	p += 4;
	if (fdb->filename.ptr)
		p += sprintf(p, " (%.*s)", fdb->filename.len, fdb->filename.ptr);
	strcpy(p, ": ");
	strlcpy(buf, "~/", sizeof(buf));
	if (edit_stline(state, buf, sizeof(buf), fname_completion) == NULL) {
		strlcpy(state->status, "Quit", sizeof(state->status));
		return;
	}
	if (buf[0] == '\0')
		strlcpy(buf, ".", sizeof(buf));
	else
		fname_expand(buf, buf);
	if (stat(buf, &stbuf) == 0 && S_ISDIR(stbuf.st_mode)) {
		if (fdb->filename.ptr) {
			sprintf(buf + strlen(buf), "/%.*s",
				fdb->filename.len, fdb->filename.ptr);
		} else {
			errno = EISDIR;
			snprintf(state->status, sizeof(state->status),
			     "%s: %s", buf, strerror(errno));
			return;
		}
	}
	if (stat(buf, &stbuf) == 0) {
		if (!edit_yn(state, "%s: overwrite? ", buf)) {
			strlcpy(state->status, "Quit", sizeof(state->status));
			return;
		}
	}
	if ((fp = fopen(buf, "w")) == NULL) {
		snprintf(state->status, sizeof(state->status),
		    "%s: %s", buf, strerror(errno));
		return;
	}
	if (all) {
		p = CP(&fdb->buf_mhdr);
		ep = CE(&fdb->buf_body);
		fwrite(p, ep - p, 1, fp);
	} else if ((fdb->flags & FDB_NOTEXT) && !(fdb->flags & FDB_INLINE)) {
		if (fdb->flags & FDB_BASE64)
			save_base64_decode(fp, &fdb->buf_body);
		else if (fdb->flags & FDB_QENCODE)
			save_qdecode(fp, &fdb->buf_body);
		else if (fdb->flags & FDB_UUENCODE)
			save_uudecode(fp, &fdb->buf_body);
		else if (fdb->flags & FDB_BINHEX)
			save_binhex(fp, &fdb->buf_body);
		else
			fwrite(CP(&fdb->buf_body), CL(&fdb->buf_body), 1, fp);
	} else {
		for (i = fdb->skip_lines; (cb = fdb_read(fdb, i)) != NULL; i++) {
			print_jis(fp, "%.*s%s", CL(cb), CP(cb),
				  (fdb_ismore(fdb, i) ? "" : "\n"));
		}
	}
	fclose(fp);
	snprintf(state->status, sizeof(state->status), "Saved to %s", buf);
}

void
print_jis(FILE *fp, char *fmt, ...)
{
	va_list ap;
	u_char c, *p;
	enum { M_ASCII, M_KANJI, M_KANA, M_KOREAN } mode;
	u_char buf[CHARBLOCK];

	va_start(ap, fmt);
	vsnprintf((char *)buf, sizeof(buf), fmt, ap);
	va_end(ap);

	mode = M_ASCII;
	p = buf;

	for (;;) {
		c = *p++;
		if (c == 0x8e) {
			c = *p++;
			if (c == '\0')
				goto exit;
			if (mode != M_KANA) {
				putc('\033', fp);
				putc('(', fp);
				putc('I', fp);
				mode = M_KANA;
			}
			c &= ~0x80;
			putc(c, fp);
		} else if (c == 0x8d) {
			c = *p++;
			if (c == '\0' || *p != 0x8d || p[1] == '\0') {
				c = '\0';
				goto exit;
			}
			if (mode != M_KOREAN) {
				putc('\033', fp);
				putc('$', fp);
				putc('(', fp);
				putc('C', fp);
				mode = M_KOREAN;
			}
			c &= ~0x80;
			putc(c, fp);
			p++;
			c = *p++ & ~0x80;
			putc(c, fp);
		} else if (c & 0x80) {
			if (*p == '\0') {
				c = '\0';
				goto exit;
			}
			if (mode != M_KANJI) {
				putc('\033', fp);
				putc('$', fp);
				putc('B', fp);
				mode = M_KANJI;
			}
			c &= ~0x80;
			putc(c, fp);
			c = *p++ & ~0x80;
			putc(c, fp);
		} else {
  exit:
			if (mode != M_ASCII) {
				putc('\033', fp);
				putc('(', fp);
				putc('B', fp);
				mode = M_ASCII;
			}
			if (c == '\0')
				break;
			putc(c, fp);
		}
	}
}

static int
encode_hdr(u_char **pp, u_char *ep, cbuf_t *res, int limit)
{
	u_char *p;
	u_char c;
	int newmode, elen;
	static struct decode d;

	/* check minimum length */
	if (limit < 16 + 4 + 4 + 4 + 2)
		return 0;
	DINIT(&d, 0);
	/* TODO: KOREAN */
	for (p = "=?ISO-2022-JP?B?"; *p; p++)
		OUTC(&d, *p);
	for (p = *pp; p < ep; p++) {
		c = *p;
		if (c == 0x8e)
			newmode = DM_KANA;
		else if (c & 0x80)
			newmode = DM_KANJI;
		else
			newmode = DM_ASCII;

		if (newmode != DM_ASCII && p + 1 == ep)
			break;

		elen = 0;		/* encoded word */
		if (d.bclen == 0)
			elen += 4;
		if (d.bclen == 2 && newmode == DM_KANJI)
			elen += 8;	/* to encode it */
		if (newmode != d.mode)
			elen += 4;	/* escape sequence */
		if (newmode != DM_ASCII)
			elen += 4;	/* escape sequence */
		elen += 2;		/* ?= */
		if (d.cnt + elen > limit)
			break;

		switch (newmode) {
		case DM_ASCII:
			if (d.mode != DM_ASCII) {
				BOUTC(&d, '\033');
				BOUTC(&d, '(');
				BOUTC(&d, 'B');
				d.mode = DM_ASCII;
			}
			BOUTC(&d, c);
			break;
		case DM_KANA:
			if (d.mode != DM_KANA) {
				BOUTC(&d, '\033');
				BOUTC(&d, '(');
				BOUTC(&d, 'I');
				d.mode = DM_KANA;
			}
			c = *++p & ~0x80;
			BOUTC(&d, c);
			break;
		case DM_KANJI:
			if (d.mode != DM_KANJI) {
				BOUTC(&d, '\033');
				BOUTC(&d, '$');
				BOUTC(&d, 'B');
				d.mode = DM_KANJI;
			}
			c &= ~0x80;
			BOUTC(&d, c);
			c = *++p & ~0x80;
			BOUTC(&d, c);
			break;
		}
	}
	*pp = p;
	if (d.mode != DM_ASCII) {
		BOUTC(&d, '\033');
		BOUTC(&d, '(');
		BOUTC(&d, 'B');
		d.mode = DM_ASCII;
	}
	BFLUSH(&d);
	OUTC(&d, '?'); OUTC(&d, '=');
	res->ptr = d.buf;
	res->len = d.cnt;
	return 1;
}

void
print_hdr(FILE *fp, char *fmt, ...)
{
	va_list ap;
	u_char *p, *s, *t;
	int width, slen;
	int encode;
	cbuf_t cbuf;
	u_char buf[CHARBLOCK];
	static struct decode d;
	const int fold = 78;

	va_start(ap, fmt);
	vsnprintf((char *)buf, sizeof(buf), fmt, ap);
	va_end(ap);

#if 0
	if (strncasecmp(buf, "Content-", 8) == 0
	||  strncasecmp(buf, "To:", 3) == 0
	||  strncasecmp(buf, "Cc:", 3) == 0
	||  strncasecmp(buf, "From:", 5) == 0
	||  strncasecmp(buf, "Sender:", 7) == 0
	||  strncasecmp(buf, "Apparently-To:", 14) == 0
	||  strncasecmp(buf, "Resent-To:", 10) == 0
	||  strncasecmp(buf, "Resent-Cc:", 10) == 0
	||  strncasecmp(buf, "Resent-From:", 12) == 0) {
		sflag = 1;
	} else {
		sflag = 0;
	}
#endif

	/* convert euc if jis code */
	for (p = buf; *p; p++) {
		if (*p == '\033') {
			/* would be JIS text -- convert euc */
			DINIT(&d, 0);
			for (p = buf; *p; p++)
				DOUTC(&d, *p);
			memcpy(buf, d.buf, d.cnt);
			buf[d.cnt] = '\0';
			break;
		}
	}

	width = 0;
	p = buf;
	while (*p) {
		/* get space */
		for (s = p; *p; p++) {
			if (!isspace(*p))
				break;
		}
		/* get token */
		encode = 0;
		for (t = p; *p; p++) {
			if (isspace(*p))
				break;
			if (*p & 0x80)
				encode = 1;
		}
		if (!encode) {
			/*
			 * cannot split token
			 * insert "\n " if there is no room
			 */
			if (width > 0 && width + (p - s) > fold) {
				putc('\n', fp);
				putc(' ', fp);
				width = 1;
				s = t;		/* skip leading spaces */
			}
			fwrite(s, p - s, 1, fp);
			width += p - s;
		} else {
			slen = t - s;
			if (encode_hdr(&t, p, &cbuf, fold - (width + slen))) {
				/* OK put leading spaces */
				fwrite(s, slen, 1, fp);
				fwrite(cbuf.ptr, cbuf.len, 1, fp);
				width += slen + cbuf.len;
			}
			while (t < p) {
				putc('\n', fp);
				putc(' ', fp);
				width = 1;
				encode_hdr(&t, p, &cbuf, fold - width);
				fwrite(cbuf.ptr, cbuf.len, 1, fp);
				width += cbuf.len;
			}
		}
		if (*p == '\n') {
			putc(*p, fp);
			p++;
			width = 0;
		}
	}
}

#define	ENC(c1,c2,c3)	(((c1) << 16) | ((c2) << 8) | (c3))
#define	EOUT(fp, c, n)	putc(b64[((c) >> (6 * (3-n))) & 0x3f], (fp))

void
save_base64_encode(FILE *fp, cbuf_t *buf)
{
	int w;
	unsigned long c;
	u_char *p, *ep;

	for (p = (u_char *)buf->ptr, ep = p + buf->len, w = 0; p + 2 < ep; p += 3) {
		c = ENC(p[0], p[1], p[2]);
		EOUT(fp, c, 0); EOUT(fp, c, 1); EOUT(fp, c, 2); EOUT(fp, c, 3);
		if ((w += 4) >= 72) {
			putc('\n', fp);
			w = 0;
		}
	}
	switch (ep - p) {
	case 1:
		c = ENC(p[0], 0, 0);
		EOUT(fp, c, 0); EOUT(fp, c, 1);
		putc('=', fp); putc('=', fp);
		w += 4;
		break;
	case 2:
		c = ENC(p[0], p[1], 0);
		EOUT(fp, c, 0); EOUT(fp, c, 1); EOUT(fp, c, 2);
		putc('=', fp);
		w += 4;
		break;
	}
	if (w != 0)
		putc('\n', fp);
}

#ifdef TEST_MAIN
main()
{
	char buf[] = "=?iso-2022-jp?B?GyRCTytGLzJKM1g4JjVmPWobKEI=?=";
	char *p;
	cbuf_t res;

	decode_init();
	p = buf;
	if (decode_header(NULL, &p, buf + sizeof(buf)-1, &res))
		printf("%.*s\n", res.len, res.ptr);
	else
		printf("Decode failed\n");
	exit(0);
}
#endif
