#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/in_systm.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>

#include <openssl/evp.h>
#include <openssl/md5.h>

#define KEYFILE "/.passkey"
#define PACKET_SIZE  1024 

#define SOFTWARE_NAME "ctunnel"
#define VERSION_MAJOR 0
#define VERSION_MINOR 2
#define COPYRIGHT "Copyright (C) 2008 Jess Mahan"

struct crypt {
    unsigned char *data;
    int len;
};
struct keys {
    unsigned char key[18];
    unsigned char iv[18];
    char cipher[255];
};
struct options {
    char host[255];
    int port_listen;
    int port_forward;
    int server;
    int daemon;
    struct keys key;
    char cipher[255];
};
int bin_size(unsigned char *s)
{   
    unsigned char *p = s;
    
    while (*p) {
        p++;
    }
    return p - s;
}
struct crypt do_encrypt(EVP_CIPHER_CTX *ctx, unsigned char *intext, int size, unsigned char *key, unsigned char *iv)
{
    unsigned char *outbuf = NULL;
    int outlen = 0;
    int tmplen = 0;
    struct crypt crypto;

    crypto.len = 0;
    outbuf = calloc(1, size + EVP_CIPHER_CTX_block_size(ctx) + 1 * PACKET_SIZE);
    if (!EVP_EncryptUpdate(ctx, outbuf, &outlen, intext, size))
        fprintf(stdout, "EVP_EncryptUpdate: Error\n");

    outlen += tmplen;
    crypto.len = outlen;

    crypto.data = outbuf;

    return crypto;
}
struct crypt do_decrypt(EVP_CIPHER_CTX *ctx, unsigned char *intext, int len, unsigned char *key, unsigned char *iv)
{
    unsigned char *outbuf = NULL;
    int outlen = 0;
    int tmplen = 0;
    struct crypt crypto;

    crypto.len = 0;

    outbuf = calloc(1, len + (EVP_CIPHER_CTX_block_size(ctx)) * sizeof(char *) * PACKET_SIZE);
    memset(outbuf, 0x00, len + (EVP_CIPHER_CTX_block_size(ctx)) + 1);

    if (!EVP_DecryptUpdate(ctx, outbuf, &outlen, intext, len))
        fprintf(stdout, "EVP_DecryptUpdate: Error\n");

    outlen += tmplen;
    crypto.len = outlen;
    crypto.data = outbuf;

    return crypto;
}
int tcp_listen(int port)
{
    struct sockaddr_in sin;
    int sockfd;
    int x = 1;

    sockfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &x, sizeof(x));
    if (sockfd == -1) {
	perror("socket()");
	exit(1);
    }

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    sin.sin_port = htons(port);

    if (bind(sockfd, (struct sockaddr *) &sin, sizeof(sin)) == -1) {
	perror("bind()");
	exit(1);
    }
    if (listen(sockfd, 20) == -1) {
	perror("listen()");
	exit(1);
    }
    return sockfd;
}
int tcp_connect(char *ip, int port)
{
    int clifd, servfd;
    struct sockaddr_in address;
    struct hostent *ht;
    char host[255] = "";

    clifd = socket(AF_INET, SOCK_STREAM, 0);
    if (clifd < 0) {
	perror("socket");
    }

    if (!isdigit(ip[0])) {
	ht = gethostbyname(ip);
	memset(host, 0x00, 255);
	snprintf(host, 32, "%s", inet_ntoa(*(struct in_addr *)ht->h_addr_list[0]));
	fprintf(stdout, "Connecting to %s -> %s\n", host, ht->h_name);
    } else {
	snprintf(host, 254, "%s", ip);
    }

    address.sin_family = AF_INET;
    address.sin_addr.s_addr = inet_addr(host);
    address.sin_port = htons(port);

    servfd = connect(clifd, (struct sockaddr *) &address, sizeof(address));
    if (servfd < 0) {
	perror("connect");
    }

    return clifd;
}
void version(void)
{
    fprintf(stdout, "%s %d.%d %s\n", SOFTWARE_NAME, VERSION_MAJOR,
                    VERSION_MINOR, COPYRIGHT);
}
void usage(void)
{
    version();
    fprintf(stdout, "Usage is as follows:\n" \
		    "./ctunnel [-p][-v][-h] -c|-s -f -H -C cipher\n" \
		    "  -n\t(optional) Do not fork into Background\n" \
		    "  -p\t(optional) Print Stored Key, IV, and Cipher then exit\n" \
		    "  -v\t(optional) Print Version information then exit\n" \
		    "  -h\t(optional) Print the screen you are reading then exit\n" \
		    "  -c\t(manditory) Operate in Client Mode (do not use with -s)\n" \
		    "  -s\t(manditory) Operate in Server Mode (do not use with -c)\n" \
		    "  -l\t(manditory) Listen for TCP connections on this port\n" \
		    "  -f\t(manditory) Forward TCP connection from -l to this port\n" \
		    "  -H\t(manditory) Host to forward TCP connections to\n" \
		    "  -C\t(manditory) Encrypt TCP traffic with this cipher\n\n" \
		    "Example: (Tunnel Mysql)\n" \
		    "  Client Machine:\n" \
		    "    ./ctunnel -c -l 2020 -f 2021 -h remote.ip -C aes-256-cfb\n" \
		    "  Remote Machine:\n" \
		    "    ./ctunnel -s -l 2021 -f 3306 -h 127.0.0.1 -C aes-256-cfb\n" \
		    "  Client Machine:\n" \
		    "    mysql -u root -p -h 127.0.0.1 -P 2020\n");
}
struct options get_options(int argc, char *argv[])
{
    int i = 0, x = 0;
    struct options opt;
    char *homedir;
    char *passkey;
    struct stat st;
    int fd;

    if (argc <= 1) {
        usage();
	exit(0);
    }

    opt.port_listen = -1;
    opt.port_forward = -1;
    opt.server = -1;
    opt.daemon = -1;
    memset(opt.host, 0x00, sizeof(opt.host));
    memset(opt.key.key, 0x00, sizeof(opt.key.key));
    memset(opt.key.iv, 0x00, sizeof(opt.key.iv));
    memset(opt.cipher, 0x00, sizeof(opt.cipher));

    homedir = getenv("HOME");
    passkey = calloc(1, strlen(homedir) + strlen(KEYFILE) + 2);
    snprintf(passkey, strlen(homedir) + strlen(KEYFILE) + 1, "%s%s",
             homedir, KEYFILE);
    if (stat(passkey, &st) < 0) {
        fprintf(stdout, "No Passkey, will prompt\n");
	strcpy(opt.key.cipher, opt.cipher);
    } else {
        fd = open(passkey, O_RDONLY,
                  S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
        read(fd, &opt.key, sizeof(struct keys));
        fprintf(stdout, "Found Stored Key/Iv and Cipher[%s]: Remove %s and re-run to change\n",
	                opt.key.cipher, passkey);
	close(fd);
    }

    for (i = 0; argv[i] != NULL; i++) {
        if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
		case 'p':
		    fprintf(stdout, "Key:    %s\nIV:     %s\nCipher: %s\n",
		                    opt.key.key, opt.key.iv, opt.key.cipher);
		    exit(0);
		    break;
		case 's':
		    opt.server = 1;
		    break;
		case 'c':
		    opt.server = 0;
		    break;
		case 'n':
		    opt.daemon = 1;
		    break;
		case 'h':
		    usage();
		    exit(0);
		    break;
		case 'v':
		    version();
		    break;
	    }
	}
    }
    for (i = 0; argv[i] != NULL; i++) {
	if (argv[i][0] == '-' && argv[i+1]) {
	    switch (argv[i][1]) {
		case 'l':
		    if (argv[i+1][0] != '-')
		        opt.port_listen = atoi(argv[i+1]);
		    break;
		case 'f':
		    if (argv[i+1][0] != '-')
		        opt.port_forward = atoi(argv[i+1]);
		    break;
		case 'C':
		    if (argv[i+1][0] != '-') 
		        strcpy(opt.cipher, argv[i+1]);
		    break;
		case 'H':
		    if (argv[i+1][0] != '-')
		        strcpy(opt.host, argv[i+1]);
		    break;
	    }
	}
    }

    if ((bin_size(opt.key.key)) < 16) {
        fprintf(stdout, "Enter Key [16 Characters]: ");
	for (x = 0; x < 16; x++) {
	    opt.key.key[x] = fgetc(stdin);
	}
        fgetc(stdin);
    }
    if ((bin_size(opt.key.iv)) < 16) {
        fprintf(stdout, "Enter IV  [16 Characters]: ");
	for (x = 0; x < 16; x++) {
	    opt.key.iv[x] = fgetc(stdin);
	}
        fgetc(stdin);
    }

    if ((strlen(opt.cipher)) > 0) {
        strcpy(opt.key.cipher, opt.cipher);
    }

#ifdef _WIN32
    fd = open(passkey, O_RDWR | O_BINARY);
#else
    fd = open(passkey, O_RDWR | O_TRUNC | O_CREAT,
                       S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
#endif
    write(fd, &opt.key, sizeof(struct keys));
    close(fd);

    if ((bin_size((unsigned char *) opt.key.cipher)) <= 0) {
        fprintf(stdout, "ERROR: Please specify Stream Cipher with -C\n");
	exit(0);
    }
    if (opt.server < 0) {
        fprintf(stdout, "ERROR: Please specify Client of Server mode with -c or -s\n");
	exit(0);
    }
    if ((bin_size((unsigned char *) opt.host)) <= 0) {
        fprintf(stdout, "ERROR: Please specify Host with -H\n");
	exit(0);
    }
    if (opt.port_listen < 0) {
        fprintf(stdout, "ERROR: Please specify Listen Port with -l\n");
	exit(0);
    }
    if (opt.port_forward < 0) {
        fprintf(stdout, "ERROR: Please specify Forawrd Port with -f\n");
	exit(0);
    }

    free(passkey);

    return opt;
}
void crypto_shutdown(int signal)
{
    fprintf(stdout, "[ctunnel] Shutting Down\n");
    EVP_cleanup();
//    EVP_CIPHER_CTX_cleanup(ectx);
//    EVP_CIPHER_CTX_cleanup(dctx);
    exit(0);
}
int main(int argc, char *argv[])
{
    int inc_serv, out_cli;
    int clifd;
    fd_set rfd;
    int ret = 0;
    struct sockaddr_in pin;
    socklen_t addrsize = 0;
    void *data;
    pid_t pid;
    struct crypt crypto;
    struct options opt;
    EVP_CIPHER_CTX ectx;
    EVP_CIPHER_CTX dctx;
    const EVP_CIPHER *c = NULL;

    memset(opt.host, 0x00, sizeof(opt.host));
    opt = get_options(argc, argv);

    /* STREAM CIPHERS ONLY */
    OpenSSL_add_all_ciphers();
    c = EVP_get_cipherbyname(opt.key.cipher);
    if (!c) {
        fprintf(stdout, "Cipher \"%s\" not found!\n", opt.key.cipher);
	exit(1);
    }
    EVP_CIPHER_CTX_init(&ectx);
    EVP_EncryptInit_ex(&ectx, EVP_get_cipherbyname(opt.key.cipher), NULL, opt.key.key, opt.key.iv);
    EVP_CIPHER_CTX_set_padding(&ectx, 1);

    EVP_CIPHER_CTX_init(&dctx);
    EVP_DecryptInit_ex(&dctx, EVP_get_cipherbyname(opt.key.cipher), NULL, opt.key.key, opt.key.iv);
    EVP_CIPHER_CTX_set_padding(&dctx, 1);

    signal(SIGCHLD, SIG_IGN);
    signal(SIGINT, crypto_shutdown); 
    signal(SIGTERM, crypto_shutdown); 

    if (opt.daemon != 1) {
        fprintf(stdout, "Going into background\n");
	pid = fork();
	if (pid == -1) {
	    fprintf(stdout, "ERROR: %s\n", strerror(errno));
	    exit(1);
	}
	if (pid > 0) {
	    exit(0);
	}
    }

    fprintf(stdout, "Listening on %d using %s\n", opt.port_listen, opt.key.cipher);
    inc_serv = tcp_listen(opt.port_listen);

    while (1) {
	clifd = accept(inc_serv, (struct sockaddr *) &pin, &addrsize);

	pid = fork();
	if (pid == 0) {
	    close(inc_serv);
	    out_cli = tcp_connect(opt.host, opt.port_forward);

	    data = calloc(1, sizeof(char *) * PACKET_SIZE);

	    while (1) {
	        FD_ZERO(&rfd);
	        FD_SET(clifd, &rfd);
	        FD_SET(out_cli, &rfd);
	        crypto.len = 0;
		ret = select(clifd + out_cli + 1, &rfd, NULL, NULL, NULL);
		if (FD_ISSET(clifd, &rfd)) {
		    ret = read(clifd, data, PACKET_SIZE * 2);
		    if (ret <= 0) {
			break;
		    } 
		    if (opt.server == 0) {
                        crypto = do_encrypt(&ectx, (unsigned char *) data, ret, opt.key.key, opt.key.iv);
		        ret = write(out_cli, crypto.data, crypto.len);
		        free(crypto.data);
			if (ret <= 0)
			    break;
		    }
		    if (opt.server == 1) {
			crypto = do_decrypt(&dctx, (unsigned char *) data, ret, opt.key.key, opt.key.iv);
		        ret = write(out_cli, crypto.data, crypto.len);
		        free(crypto.data);
			if (ret <= 0)
			    break;
		    }
		    memset(data, 0x00, sizeof(char *) * PACKET_SIZE);
		}
		if (FD_ISSET(out_cli, &rfd)) {
		    ret = read(out_cli, data, PACKET_SIZE * 2);
		    if (ret <= 0) {
			break;
		    } 
		    if (opt.server == 0) {
			crypto = do_decrypt(&dctx, (unsigned char *) data, ret, opt.key.key, opt.key.iv);
		        ret = write(clifd, crypto.data, crypto.len);
			free(crypto.data);
			if (ret <= 0)
			    break;
		    }
		    if (opt.server == 1) {
                        crypto = do_encrypt(&ectx, (unsigned char *) data, ret, opt.key.key, opt.key.iv);
		        ret = write(clifd, crypto.data, crypto.len);
			free(crypto.data);
			if (ret <= 0)
			    break;
		    }
		    memset(data, 0x00, sizeof(char *) * PACKET_SIZE);
		}
	    }
	    fprintf(stdout, "Connection Closed\n");
	    FD_CLR(out_cli, &rfd);
	    FD_CLR(clifd, &rfd);
	    close(out_cli);
	    close(clifd);
	    free(data);
            EVP_CIPHER_CTX_cleanup(&ectx);
            EVP_CIPHER_CTX_cleanup(&dctx);
            EVP_cleanup();
	    _exit(0);
	} else {
	    close(clifd);
	}
    }
    return 0;
}
