-----------------------------------------------------------------------------
-- |
-- Module      :  Coded.Encryption.PKCS8
-- Copyright   :  (c) Dominic Steinitz 2003
-- License     :  BSD-style (see the file libraries/base/LICENSE)
-- 
-- Maintainer  :  dominic.steinitz@blueyonder.co.uk
-- Stability   :  experimental
-- Portability :  non-portable
--
-- Provide Haskell types associated with the ASN.1 types defined in the 
-- PKCS8 standard <http://www.rsasecurity.com/rsalabs/pkcs/pkcs-8/> and
-- functions to convert them to and from an abstract representation of ASN.1.
-- This representation can be encoded and decoded from BER.
-- It currently only handles the unencrypted PrivateKeyInfo format.
--
-----------------------------------------------------------------------------

module Codec.Encryption.PKCS8 
   (
-- * Type declarations
   RSAPrivateKey(..),
   Version(..),
   Algorithm,
   Parameters,
   AlgorithmIdentifier(..),
   PrivateKeyInfo(..)
-- * Example
   
{- | 
Generate a key pair.

>openssl genrsa -out private.pem

And convert the private key to PKCS8 format without encryption.

>openssl pkcs8 -topk8 -inform PEM -outform DER -in private.pem -nocrypt -out private.der

Now read it with the Haskell program.

>module Main(main) where
>
>import IO
>import Char
>import Control.Monad.State
>import Codec.ASN1.ASN1
>import Codec.Encryption.Utils
>import Codec.Encryption.PKCS8
>
>-- Generate a key pair.
>-- 
>-- > openssl genrsa -out private.pem
>--
>-- And convert the private key to PKCS8 format without encryption.
>--
>-- > openssl pkcs8 -topk8 -inform PEM -outform DER -in private.pem -nocrypt -out private.der
>--
>-- Now read it with the Haskell program.
>
>main = 
>   do ifh <- openFile "private.der" ReadMode
>      (x,y) <- decode ifh
>      let z::PrivateKeyInfo = fromASN NoTag y 
>      putStrLn $ show z
>      -- Decoding can be done using a state monad as an alternative.
>      -- stdin is a dummy file handle so that the overloaded function decode can be used.
>      let test = 
>             runState (decode stdin) (map (chr . fromIntegral) 
>                                          (encode (toASN NoTag z)))
>          test' :: PrivateKeyInfo
>          test' = let ((x,y),z) = test in fromASN NoTag y
>      putStrLn $ show test'
-}
   ) where

import IO
import Char
import Codec.ASN1.ASN1
import Codec.Encryption.Utils
import Control.Monad.Error
import Control.Monad.State

data RSAPrivateKey =
   MkRSAPrivateKey {
      version         :: Version,
      modulus         :: Integer, -- n
      publicExponent  :: Integer, -- e
      privateExponent :: Integer, -- d
      prime1          :: Integer, -- p
      prime2          :: Integer, -- q
      exponent1       :: Integer, -- d mod (p-1)
      exponent2       :: Integer, -- d mod (q-1)
      coefficient     :: Integer  -- (inverse of q) mod p
      }
   deriving Show

instance ASNable RSAPrivateKey where
   toASN t a =
      case t of
         NoTag ->
	    Constructed' (Universal,16) as
	 Implicit tag ->
	    Constructed' tag as
      where
         as = [toASN NoTag (version a),
	       toASN NoTag (modulus a),
	       toASN NoTag (publicExponent a),
	       toASN NoTag (privateExponent a),
	       toASN NoTag (prime1 a),
	       toASN NoTag (prime2 a),
	       toASN NoTag (exponent1 a),
	       toASN NoTag (exponent2 a),
	       toASN NoTag (coefficient a)]
   fromASN NoTag x =
      case x of
	 Constructed' t' [v,m,pu,pr,p,q,e,f,c] ->
	    case t' of
	       (Universal,16) ->  
	          let version         = fromASN NoTag v
	              modulus         = fromASN NoTag m
		      publicExponent  = fromASN NoTag pu
		      privateExponent = fromASN NoTag pr
		      prime1          = fromASN NoTag p
		      prime2          = fromASN NoTag q
		      exponent1       = fromASN NoTag e
		      exponent2       = fromASN NoTag f
		      coefficient     = fromASN NoTag c
		         in MkRSAPrivateKey {version         = version,
		                             modulus         = modulus,
			                     publicExponent  = publicExponent,
					     privateExponent = privateExponent,
					     prime1          = prime1,
					     prime2          = prime2,
					     exponent1       = exponent1,
					     exponent2       = exponent2,
					     coefficient     = coefficient}
	       otherwise -> 
	          error "fromASN: invalid universal constructed tag for RSAPrivateKey"
	 otherwise ->
	    error "fromASN: invalid primitve tag or invalid number of components for RSAPrivateKey"

data Version = V1
   deriving (Show, Enum)

instance ASNable Version where
   toASN t v = toASN t (fromEnum v)
   fromASN t v = 
      let (x::Integer) = fromASN t v
         in toEnum $ fromIntegral x

type Algorithm = OID

-- | This will do for now. DSA has some parameters which are more complicated
-- than this but since we plan to do RSA initially and this has NULL parameters
-- then anything will do to get us going.

type Parameters = Int

-- | The parameters will only ever be Nothing as this implementation
-- only supports RSA and this has no parameters. So even if the parameters
-- are non-NULL, fromASN will not fail but will ignore them.

data AlgorithmIdentifier =
   MkAlgorithmIdentifier {
      algorithm :: Algorithm,
      parameters :: Maybe Parameters } 
   deriving Show

instance ASNable AlgorithmIdentifier where
   toASN t a =
      case t of
         NoTag ->
	    Constructed' (Universal,16) as
	 Implicit tag ->
	    Constructed' tag as
      where
         as = [toASN NoTag (algorithm a), p]
	 p  = case parameters a of
	         Nothing -> toASN NoTag NULL
		 Just qs -> toASN NoTag qs
   fromASN t v = 
      case v of
         Constructed' t' [a,p] ->
	    case t' of
	       (Universal,16) ->
	          let algorithm = fromASN NoTag a
		     in MkAlgorithmIdentifier {algorithm  = algorithm,
		                               parameters = Nothing}
	       otherwise ->
	          error "fromASN: invalid universal constructed tag for AlgorithmIdentifier"
	 otherwise ->
	    error "fromASN: invalid primitve tag or invalid number of components for AlgorithmIdentifier"

data PrivateKeyInfo =
   MkPrivateKeyInfo {
      version1 :: Version,
      privateKeyAlgorithm :: AlgorithmIdentifier,
      privateKey :: RSAPrivateKey }
   deriving Show

instance ASNable PrivateKeyInfo where
   toASN t a =
      case t of
         NoTag ->
	    Constructed' (Universal,16) as
	 Implicit tag ->
	    Constructed' tag as
      where
         as = [toASN NoTag (version1 a),
	       toASN NoTag (privateKeyAlgorithm a),
	       toASN NoTag $ 
               MkOctetString $
               map MkOctet $ 
	       encode $ 
	       toASN NoTag (privateKey a)]
   fromASN NoTag x =
      case x of
	 Constructed' t' [v,a,k] ->
	    case t' of
	       (Universal,16) ->  
	          let version1            = fromASN NoTag v
		      privateKeyAlgorithm = fromASN NoTag a
		      (xs::OctetString)   = fromASN NoTag k
		      (MkOctetString ys) = xs
		      zs = map (\y -> let (MkOctet z) = y in chr $ fromIntegral z) ys
		      ((_,b),_)       =  runState (decode stdin) zs
		      privateKey          = fromASN NoTag b
		         in MkPrivateKeyInfo 
		               {version1 = version1,
		                privateKeyAlgorithm = privateKeyAlgorithm,
			        privateKey = privateKey}
	       otherwise -> 
	          error "fromASN: invalid universal constructed tag for PrivateKeyInfo"
	 otherwise ->
	    error "fromASN: invalid primitve tag or invalid number of components for PrivateKeyInfo"
