// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// event.cpp
////////////////////////////////////////////////////////////////////////

#include "event.h"
#include "game.h"
#include "extwindow.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Event
// DESCRIPTION: Load an event from a file.
// PARAMETERS : file    - file descriptor
// RETURNS    : identifier of the event owner, -1 on error
////////////////////////////////////////////////////////////////////////

short Event::Load( File &file ) {
  int i;

  e_id = file.Read8();
  e_type = file.Read8();
  e_trigger = file.Read8();
  e_depend = file.Read8();
  for ( i = 0; i < 3; ++i ) e_tdata[i] = file.Read16();
  for ( i = 0; i < 3; ++i ) e_data[i] = file.Read16();
  e_title = file.Read16();
  e_message = file.Read16();
  e_flags = file.Read16();
  e_player = 0;

  return file.Read8();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Save
// DESCRIPTION: Save the event data to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Event::Save( File &file ) const {
  int i;

  file.Write8( e_id );
  file.Write8( e_type );
  file.Write8( e_trigger );
  file.Write8( e_depend );

  for ( i = 0; i < 3; ++i ) file.Write16( e_tdata[i] );
  for ( i = 0; i < 3; ++i ) file.Write16( e_data[i] );

  file.Write16( e_title );
  file.Write16( e_message );
  file.Write16( e_flags );
  file.Write8( e_player->ID() );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Check
// DESCRIPTION: Check whether the event can be executed. The trigger
//              conditions and event dependencies must be met.
// PARAMETERS : -
// RETURNS    : 0 if the event cannot be activated, 1 if it can, and 2
//              if it can be activated and requires a message window to
//              pop up
////////////////////////////////////////////////////////////////////////

short Event::Check( void ) {
  short rc = CheckTrigger();
 
  if ( rc != 0 ) {
    TLWList deps;
    deps.AddHead( new TLWNode( "", this, ID() ) );

    if ( !CheckDependencies( deps ) ) rc = 0;
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::CheckTrigger
// DESCRIPTION: Check whether the event trigger conditions are met.
// PARAMETERS : -
// RETURNS    : 0 if the trigger is inactivate, 1 if active, and 2
//              if activate and execution requires a message window to
//              pop up
////////////////////////////////////////////////////////////////////////

short Event::CheckTrigger( void ) {
  short rc = 0;
  Mission *mis = Gam->GetMission();

  if ( !Disabled() ) {
    switch ( e_trigger ) {
    case ETRIGGER_TURN:
      if ( (mis->GetTurn() >= e_tdata[0]) &&
           (&mis->GetPlayer() == e_player) ) rc = 1;
      break;
    case ETRIGGER_UNIT_DESTROYED:
      if ( e_tdata[0] == -1 ) {                               // destroy all enemy units to score
        if ( mis->GetPlayer( e_tdata[1] ).Units(0) == 0 ) rc = 1;
      } else {                                                // trigger if
        Unit *u = mis->GetUnit( e_tdata[0] );                 //  * unit not found
        if ( !u || !u->IsAlive() ||                           //  * unit found, but already dead
           (u->Owner() && (u->Owner()->ID() != e_tdata[1])) ) //  * owner != original owner (captured)
          rc = 1;
      }
      break;
    case ETRIGGER_HAVE_BUILDING:
      if ( (e_tdata[2] == -1) || (e_tdata[2] == mis->GetTurn()) ) {
        Building *b = mis->GetShop( e_tdata[0] );
        if ( b && b->Owner() && (b->Owner()->ID() == e_tdata[1]) ) rc = 1;
      }
      break;
    case ETRIGGER_HAVE_UNIT:
      if ( (e_tdata[2] == -1) || (e_tdata[2] == mis->GetTurn()) ) {
        Unit *u = mis->GetUnit( e_tdata[0] );
        if ( u && u->Owner() && (u->Owner()->ID() == e_tdata[1]) ) rc = 1;
      }
      break;
    case ETRIGGER_UNIT_POSITION:
      if ( e_tdata[0] == -1 ) {
        Unit *u = mis->GetMap().GetUnit( Point(e_tdata[1], e_tdata[2]) );
        if ( u && (u->Owner() == e_player) ) rc = 1;
      } else {
        Unit *u = mis->GetUnit( e_tdata[0] );
        if ( u && (u->Owner() == e_player) &&
           (u->Position() == Point(e_tdata[1], e_tdata[2])) ) rc = 1;
      }
      break;
    case ETRIGGER_HANDICAP:
      if ( mis->GetHandicap() & e_tdata[0] ) rc = 1;
      break;
    }

    if ( rc ) {
      if ( e_player->IsHuman() &&
           (e_player == &mis->GetPlayer()) && (e_message != -1) ) rc = 2;

      // special rule for score events which can also carry
      // a message for the _other_ player
      else if ( (e_type == EVENT_SCORE) && (e_data[1] != -1) &&
           (e_player != &mis->GetPlayer()) && mis->GetPlayer().IsHuman() )
        rc = 2;
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::CheckDependencies
// DESCRIPTION: Check whether event dependencies are met.
// PARAMETERS : deps - list of dependent events already checked (with
//                     positive results)
// RETURNS    : TRUE if all dependencies are met, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Event::CheckDependencies( TLWList &deps ) {
  bool rc = true;

  if ( e_depend != -1 ) {

    // if the event does not exist anymore it has already been executed
    Event *dep = Gam->GetMission()->GetEvent( e_depend );
    if ( dep ) {

      // did we already check this event earlier in the cycle?
      if ( !deps.GetNodeByID( dep->ID() ) ) {
        if ( dep->CheckTrigger() ) {
          deps.AddTail( new TLWNode( "", dep, dep->ID() ) );
          rc = dep->CheckDependencies( deps );
        } else rc = false;
      }
    }

  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Execute
// DESCRIPTION: Execute this event. What this means depends on the event
//              type.
// PARAMETERS : msgwin  - pointer to a message window to show messages
//                        (may be NULL for events with a return code of
//                        0 or 1 from Event::Check())
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Event::Execute( MessageWindow *msgwin ) {
  Mission *mis = Gam->GetMission();

  bool show_msg = true;

  switch ( e_type ) {
  case EVENT_MESSAGE:
    break;
  case EVENT_CREATE_UNIT: {
    Point p( e_data[1], e_data[2] );
    MapObject *mo = mis->GetMap().GetMapObject( p );

    if ( mo ) {
      const UnitType *type = mis->GetUnitSet().GetUnitInfo(e_data[0]);
      show_msg = (mo->Owner() == e_player)
        && ((mo->IsUnit() && static_cast<Unit *>(mo)->IsTransport())
             || !mo->IsUnit())
        && dynamic_cast<UnitContainer *>(mo)->Allow( type );
    }

    if ( show_msg ) {
      Unit *u = mis->CreateUnit( e_data[0], *e_player, p );
      if ( u ) u->UnsetFlags( U_DONE );
    }
    break; }
  case EVENT_MINING: {
    Building *b = mis->GetShop( e_data[0] );
    if ( b ) {
      if ( e_data[2] == 0 ) b->SetCrystals( e_data[1] );
      else if ( e_data[2] == 1 ) b->ModifyCrystals( e_data[1] );
      else if ( e_data[2] == 2 ) b->SetCrystalProduction( e_data[1] );
      else if ( e_data[2] == 3 ) b->ModifyCrystalProduction( e_data[1] );
    } else show_msg = false;
    break; }
  case EVENT_RESEARCH: {
    Building *b = mis->GetShop( e_data[0] );
    if ( b && (b->Owner() == e_player) ) b->SetUnitProduction( 1 << e_data[1] );
    else show_msg = false;
    break; }
  case EVENT_SCORE:
    e_player->Success( (signed char)e_data[0] );
    DisplayMessage( &mis->GetOtherPlayer(*e_player), e_data[1], e_data[2],
                    mis, true, msgwin );
    break;
  case EVENT_NEXT_MAP:
    mis->SetSequel( e_data[0] );
    break;
  case EVENT_MANIPULATE_EVENT: {
    Event *e = mis->GetEvent( e_data[0] );
    if ( e ) {
      if ( e_data[2] == 0 ) e->SetFlags( e_data[1] );
      else if ( e_data[2] == 1 ) e->UnsetFlags( e_data[1] );
      else if ( e_data[2] == 2 ) e->ToggleFlags( e_data[1] );
    }
    break; }
  }

  DisplayMessage( e_player, e_message, e_title, mis,
                  show_msg, msgwin );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::DisplayMessage
// DESCRIPTION: Check whether the message is to be displayed, and pop up
//              a MessageWindow if it is.
// PARAMETERS : p     - player to show message to
//              msg   - message index
//              title - title index (if -1 use player name)
//              m     - pointer to mission object
//              show  - if FALSE don't show (and don't record) message
//              win   - message window (must not be NULL if a message is
//                      displayed)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Event::DisplayMessage( Player *p, short msg, short title, Mission *m,
                            bool show, MessageWindow *win ) const {
  if ( (msg == -1) || !p->IsHuman() ) show = false;

  if ( show ) {
    if ( &m->GetPlayer() == p ) {
      const char *tstr;
      if ( title == -1 ) tstr = p->Name();
      else tstr = m->GetMessage(title);

      win->SetTitle( tstr );
      win->textscroll->SetText( m->GetMessage(msg) );
      win->Draw();
      win->Show();
    } else {
      // if the other player is human (ie. we have a history) put the message
      // into the turn replay queue
      History *hist = m->GetHistory();
      if ( hist ) hist->RecordMsgEvent( title, msg );
    }
  }
}

