// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// mapwindow.cpp
////////////////////////////////////////////////////////////////////////

#include "mapwindow.h"
#include "game.h"

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MapWindow
// DESCRIPTION: Set the basic map window variables.
// PARAMETERS : x     - left edge of window
//              y     - top edge of window
//              w     - window width
//              h     - window height
//              flags - window flags (see window.h for details)
//              view  - pointer to the view the window will be put on
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

MapWindow::MapWindow( short x, short y, unsigned short w, unsigned short h,
                      unsigned short flags, View *view ) :
    Window( x, y, w, h, flags, view ),
    speed_icon( view->GetSystemIcons(), 0, 0, ICON_WIDTH, ICON_HEIGHT ),
    def_icon( view->GetSystemIcons(), 32, 0, ICON_WIDTH, ICON_HEIGHT ),
    ground_icon( view->GetSystemIcons(), 64, 0, ICON_WIDTH, ICON_HEIGHT ),
    air_icon( view->GetSystemIcons(), 96, 0, ICON_WIDTH, ICON_HEIGHT ),
    ship_icon( view->GetSystemIcons(), 128, 0, ICON_WIDTH, ICON_HEIGHT )
  {
  panel = new Panel( this, view );
  mview = new MapView( this, *this,
              MV_AUTOSCROLL|MV_DISABLE|MV_DISABLE_CURSOR|MV_DISABLE_FOG );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::HandleEvent
// DESCRIPTION: React to user input.
// PARAMETERS : event - SDL_Event
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

GUI_Status MapWindow::HandleEvent( const SDL_Event &event ) {
  return Gam->HandleEvent(event);
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::Draw
// DESCRIPTION: Draw the map window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::Draw( void ) {
  if ( mview->Enabled() ) mview->Draw();
  else Window::DrawBack();
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawBack
// DESCRIPTION: Draw the map window background into the internal buffer.
// PARAMETERS : x - left edge of area to paint
//              y - top edge of area to paint
//              w - width of area to paint
//              h - height of area to paint
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawBack( short x, short y, unsigned short w, unsigned short h ) {
  FillRect( x, y, w, h, Color(CF_COLOR_SHADOW) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MoveHex
// DESCRIPTION: Smoothly move a hex image (usually a unit or cursor)
//              from one hex to another (adjacent) one.
// PARAMETERS : img     - map tile identifier for the hex image
//              surface - surface containing the image
//              hex1    - source hex position
//              hex2    - destination hex position
//              speed   - total time (ms) for the animation
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::MoveHex( unsigned short img, Surface *surface,
                const Point &hex1, const Point &hex2, unsigned short speed ) {
  // allocate a new surface where we can store the "background" graphics
  Point psrc = mview->Hex2Pixel( hex1 );
  Point pdst = mview->Hex2Pixel( hex2 );

  Rect bg;
  bg.x = MIN( psrc.x, pdst.x );
  bg.y = MIN( psrc.y, pdst.y );
  bg.w = GFX_WIDTH + ABS(psrc.x-pdst.x);
  bg.h = GFX_HEIGHT + ABS(psrc.y-pdst.y);
  bg.Clip( *this );

  Surface *bgs = new Surface;
  if ( !bgs->Create( bg.w, bg.h, DISPLAY_BPP, 0 ) ) {
    unsigned long oldticks, ticks = 0;

    Blit( bgs, bg, 0, 0 );

    oldticks = SDL_GetTicks();
    do {
      short cx, cy;

      // don't modify the following three lines; some versions of gcc
      // (e.g. 2.95.3) seem to produce bogus code when all the calculations
      // are done in one line
      cx = (short)((pdst.x - psrc.x) * ticks);
      cy = (short)((pdst.y - psrc.y) * ticks);
      mview->DrawHex( img, surface, this,
                      psrc.x + cx / speed, psrc.y + cy / speed, bg );

      Show( bg );
      bgs->Blit( this, Rect( 0, 0, bg.w, bg.h ), bg.x, bg.y );
      ticks = SDL_GetTicks() - oldticks;
    } while ( ticks < speed );

    Show( bg );
  }
  delete bgs;
}


////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::BoxAvoidHexes
// DESCRIPTION: Try to place a box on the screen so that it does not
//              cover the two given hexes.
// PARAMETERS : box  - the box; left and top edge will be modified
//              hex1 - first hex position
//              hex2 - second hex position
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::BoxAvoidHexes( Rect &box, const Point &hex1, const Point &hex2 ) const {
  // define some preferred positions; we'll try to take one of these
  Point defs[4];
  defs[0] = Point( (w - box.w) / 2, (h - box.h) / 2 );
  defs[1] = Point( (w - box.w) / 2, 40 );
  defs[2] = Point( 40, (h - box.h) / 2 );
  defs[3] = Point( w - box.w - 40, (h - box.h) / 2 );

  Point p1 = mview->Hex2Pixel( hex1 );
  Point p2 = mview->Hex2Pixel( hex2 );

  for ( int i = 0; i < 4; ++i ) {
    if ( (((defs[i].x > p1.x + GFX_WIDTH) || (defs[i].x + box.w <= p1.x)) ||
         ((defs[i].y > p1.y + GFX_HEIGHT) || (defs[i].y + box.h <= p1.y))) &&
         (((defs[i].x > p2.x + GFX_WIDTH) || (defs[i].x + box.w <= p2.x)) ||
         ((defs[i].y > p2.y + GFX_HEIGHT) || (defs[i].y + box.h <= p2.y))) ) {
      box.x = defs[i].x;
      box.y = defs[i].y;
      return;
    }
  }
} 


////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::MoveCursor
// DESCRIPTION: Move the cursor into the given direction if possible.
// PARAMETERS : dir - requested direction
// RETURNS    : new cursor position
////////////////////////////////////////////////////////////////////////

Point MapWindow::MoveCursor( Direction dir ) {
  Point dest, cursor = mview->Cursor();

  if ( mview->GetMap()->Dir2Hex( cursor, dir, dest ) ) return cursor;

  mview->DisableCursor();
  mview->UpdateHex( cursor );
  MoveHex( mview->GetCursorImage(),
           mview->GetMap()->GetTerrainSet()->GetTiles(),
           cursor, dest, ANIM_SPEED_CURSOR );
  mview->EnableCursor();
  return dest;
}


////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::DrawUnitInfo
// DESCRIPTION: Display unit type information.
// PARAMETERS : type - unit type (may be NULL)
//              dest - destination surface
//              rect - rectangle in which to display
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::DrawUnitInfo( const UnitType *type, Surface *dest,
                              const Rect &rect ) {
  char buf[12];
  short xpos, ypos, xspacing, yspacing, terbreak;
  const TerrainSet *tset = mview->GetMap()->GetTerrainSet();

  dest->DrawBox( rect, BOX_CARVED );
  if ( type == NULL ) return;

  yspacing = rect.h - GFX_HEIGHT - (ICON_HEIGHT + sfont->Width()) * 2;
  if ( rect.w >= rect.h ) {        // draw all terrain on the same level
    terbreak = 7;
  } else {                         // draw two terrain levels
    terbreak = 4;
    yspacing -= GFX_HEIGHT;
  }
  yspacing /= 4;

  sfont->Write( type->Name(), dest,
                rect.x + (rect.w - sfont->TextWidth(type->Name())) / 2,
                rect.y + yspacing, rect );

  // show unit specs
  xspacing = (rect.w - 2 * ICON_WIDTH - 3 * sfont->Width()) / 3;
  xpos = rect.x + xspacing;
  ypos = rect.y + sfont->Height() + 2 * yspacing;
  itoa( type->Speed(), buf );
  sfont->Write( buf, dest, xpos + ICON_WIDTH + 4,
                ypos + (ICON_HEIGHT - sfont->Height())/2 );
  speed_icon.Draw( dest, xpos, ypos );

  xpos += ICON_WIDTH + xspacing;
  itoa( type->Armour(), buf );
  sfont->Write( buf, dest, xpos + ICON_WIDTH + 4,
                ypos + (ICON_HEIGHT - sfont->Height())/2 );
  def_icon.Draw( dest, xpos, ypos );

  // show unit combat strength
  xspacing = (rect.w - 3 * ICON_WIDTH) / 4;
  xpos = rect.x + xspacing;
  ypos += ICON_HEIGHT - 5;
  ground_icon.Draw( dest, xpos, ypos );
  if ( type->Firepower(U_GROUND) == 0 ) strcpy( buf, "-" );
  else if ( (type->MinFOF(U_GROUND) == type->MaxFOF(U_GROUND))
         && (type->MaxFOF(U_GROUND) <= 1) ) itoa( type->Firepower(U_GROUND), buf );
  else sprintf( buf, "%2d (%d-%d)", type->Firepower(U_GROUND),
                type->MinFOF(U_GROUND), type->MaxFOF(U_GROUND) );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );

  xpos += ICON_WIDTH + xspacing;
  air_icon.Draw( dest, xpos, ypos );
  if ( type->Firepower(U_AIR) == 0 ) strcpy( buf, "-" );
  else if ( (type->MinFOF(U_AIR) == type->MaxFOF(U_AIR))
         && (type->MaxFOF(U_AIR) <= 1) ) itoa( type->Firepower(U_AIR), buf );
  else sprintf( buf, "%2d (%d-%d)", type->Firepower(U_AIR),
                type->MinFOF(U_AIR), type->MaxFOF(U_AIR) );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );

  xpos += ICON_WIDTH + xspacing;
  ship_icon.Draw( dest, xpos, ypos );
  if ( type->Firepower(U_SHIP) == 0 ) strcpy( buf, "-" );
  else if ( (type->MinFOF(U_SHIP) == type->MaxFOF(U_SHIP))
         && (type->MaxFOF(U_SHIP) <= 1) ) itoa( type->Firepower(U_SHIP), buf );
  else sprintf( buf, "%2d (%d-%d)", type->Firepower(U_SHIP),
                type->MinFOF(U_SHIP), type->MaxFOF(U_SHIP) );
  sfont->Write( buf, dest, xpos + (ICON_WIDTH - sfont->TextWidth(buf))/2,
                           ypos + ICON_HEIGHT );


  ypos += ICON_HEIGHT + sfont->Height() + yspacing;
  // show terrain access
  Rect terborder( rect.x + 10, ypos, rect.w - 20, rect.h - ypos );
  dest->FillRect( terborder.x, terborder.y - yspacing/2, terborder.w, 1, Color(CF_COLOR_SHADOW) );
  dest->FillRect( terborder.x, terborder.y + 1 - yspacing/2, terborder.w, 1, Color(CF_COLOR_HIGHLIGHT) );

  xpos = terborder.x + (terborder.w - GFX_WIDTH * terbreak) / 2;
  ypos = terborder.y;
  for ( int i = 0; i < 7; ++i ) {
    if ( i == terbreak ) {
      xpos = terborder.x + (terborder.w - GFX_WIDTH * (7 - terbreak)) / 2;
      ypos += GFX_HEIGHT;
    }

    const TerrainType *tt = tset->GetTerrainInfo( tset->GetTerrainClassID(i) );
    mview->DrawTerrain( tt->tt_image, dest, xpos, ypos, rect );
    if ( !(type->Terrain() & tt->tt_type) ) mview->DrawFog( dest, xpos, ypos, rect );
    xpos += GFX_WIDTH;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : MapWindow::VideoModeChange
// DESCRIPTION: This method is called by the view whenever the video
//              resolution changes. The window can then adjust itself
//              to the new dimensions. Afterwards a View::Refresh() is
//              performed, i. e. the window just needs to reblit its
//              contents to its internal buffer, and NOT call a view
//              update itself.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MapWindow::VideoModeChange( void ) {
  if ( (view->Width() != w) || (view->Height() != h) ) {
    SetSize( view->Width(), view->Height() );

    mview->Resize( *this );
    Draw();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Panel
// DESCRIPTION: Create the panel window.
// PARAMETERS : mw   - pointer to map window
//              view - pointer to view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Panel::Panel( MapWindow *mw, View *view ) :
    Window( 0, mw->Height() - DEFAULT_PANEL_HEIGHT, 0, 0, 0, view ),
    workshop_icon( view->GetSystemIcons(), 148, 32, XP_ICON_WIDTH, XP_ICON_HEIGHT ),
    factory_icon( view->GetSystemIcons(), 160, 32, XP_ICON_WIDTH, XP_ICON_HEIGHT )
{
  obj = NULL;
  mapwin = mw;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Draw
// DESCRIPTION: Draw the panel window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Panel::Draw( void ) {
  if ( obj ) {
    short namex = 5;

    Window::Draw();

    if ( obj->IsUnit() ) {
      Unit *u = static_cast<Unit *>(obj);
      char buf[4];
      sprintf( buf, "(%d)", u->GroupSize() );

      namex += 5 + XP_ICON_WIDTH;
      Images[ICON_XP_BASE + u->XPLevel()]->Draw( this, 5, (h - XP_ICON_HEIGHT)/2 );
      sfont->Write( buf, this, namex + 5 + sfont->TextWidth( u->Name() ),
                  (h - sfont->Height())/2 );
    } else {
      Building *b = static_cast<Building *>(obj);
      if ( b->IsWorkshop() ) {
        workshop_icon.Draw( this, 5, (h - XP_ICON_HEIGHT)/2 );
        namex += 5 + XP_ICON_WIDTH;
      }
      if ( b->IsFactory() ) {
        factory_icon.Draw( this, namex, (h - XP_ICON_HEIGHT)/2 );
        namex += 5 + XP_ICON_WIDTH;
      }
    }
    sfont->Write( obj->Name(), this, namex, (h - sfont->Height())/2 );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::Update
// DESCRIPTION: Set the information displayed on the panel and update
//              the display.
// PARAMETERS : obj - map object to display information about (may be
//                    NULL, in which case the panel will be cleared)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Panel::Update( MapObject *obj ) {
  Rect refresh = *this;

  if ( !obj ) {
    if ( this->obj ) {
      w = h = 0;
      y = mapwin->Height() - DEFAULT_PANEL_HEIGHT;
      view->Refresh( refresh );
      this->obj = NULL;
    }
  } else {
    this->obj = obj;

    // resize and reposition if necessary
    w = sfont->TextWidth( obj->Name() ) + 10;
    if ( obj->IsUnit() ) w += 15 + XP_ICON_WIDTH + sfont->Width() * 3;
    else {
      Building *b = static_cast<Building *>(obj);
      if ( b->IsWorkshop() ) w += 5 + XP_ICON_WIDTH;
      if ( b->IsFactory() ) w += 5 + XP_ICON_WIDTH;
    }
    h = refresh.h = DEFAULT_PANEL_HEIGHT;

    MapView *mv = mapwin->GetMapView();
    Point pcursor = mv->Hex2Pixel( mv->Cursor() );
    if ( pcursor.x <= w + GFX_WIDTH ) {
      if ( y == 0 ) {
        if ( pcursor.y <= h + GFX_HEIGHT ) {
          y = mapwin->Height() - h;
        }
      } else if ( pcursor.y > y - GFX_HEIGHT ) y = 0;
    }
    SetSize( w, h );   // allocate surface
    Draw();

    // update the old window position and show the new
    if ( refresh.y == y ) {
      if ( w > refresh.w ) refresh.w = w;
    } else Show();
    view->Refresh( refresh );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::HandleEvent
// DESCRIPTION: The window must pass all incoming events to the map
//              window which does proper processing.
// PARAMETERS : event - event as received from the event handler
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status Panel::HandleEvent( const SDL_Event &event ) {
  return mapwin->HandleEvent( event );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Panel::VideoModeChange
// DESCRIPTION: This method is called by the view whenever the video
//              resolution changes. The window can then adjust itself
//              to the new dimensions. Afterwards a View::Refresh() is
//              performed, i. e. the window just needs to reblit its
//              contents to its internal buffer, and NOT call a view
//              update itself.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Panel::VideoModeChange( void ) {
  Window::VideoModeChange();

  if ( TopEdge() + DEFAULT_PANEL_HEIGHT < view->Height() )
    y = view->Height() - DEFAULT_PANEL_HEIGHT;
}


////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::TacticalWindow
// DESCRIPTION: Create a window and show an overview map of the level.
// PARAMETERS : mapview - pointer to map viewport
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

TacticalWindow::TacticalWindow( MapView *mapview, View *view ) :
        Window( WIN_CLOSE_ESC|WIN_CENTER, view ), mv(mapview)  {
  Map *pmap = mv->GetMap();

  Rect win, map, viewport;

  unsigned char magnify = MIN( (view->Width() - 20) / pmap->Width(),
                               (view->Height() * 3 / 4) / pmap->Height() );
  if ( magnify < 2 ) magnify = 2;
  else if ( magnify > 6 ) magnify = 6;

  map.w = MIN( pmap->Width() * magnify + 2, view->Width() - 10 );
  map.h = MIN( pmap->Height() * magnify + magnify/2 + 2,
               view->Height() - sfont->Height() * 3 - 40 );

  // calculate window dimensions
  win.w = MAX( map.w + 20, sfont->Width() * 35 + 10 );
  win.h = MIN( map.h + sfont->Height() * 3 + 40, view->h );
  win.Center( *view );
  win.Align( *view );

  map.x = (win.w - map.w) / 2;
  map.y = MAX( 5, win.h - 3 * sfont->Height() - map.h - 35 );

  SetSize( win );
  CalcViewPort( viewport );

  new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 9,
                    w - 2, sfont->Height() + 8, 'o',
                    WIDGET_DEFAULT, "OK", this );

  mw = new MapWidget( 0, map.x, map.y, map.w, map.h,
                      WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY, this );
  mw->SetPlayerColors( Gam->GetPlayer(PLAYER_ONE)->LightColor(),
                       Gam->GetPlayer(PLAYER_TWO)->LightColor() );
  mw->SetMap( pmap, viewport, magnify );
  mw->SetHook( this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::WidgetActivated
// DESCRIPTION: Handle events from the MapWidget.
// PARAMETERS : widget - widget sending the message (unused)
//              win    - window the widget belongs to (unused)
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status TacticalWindow::WidgetActivated( Widget *widget, Window *win ) {
  Point hex = mw->GetLastHex();
  Rect viewport;

  view->DisableUpdates();
  if ( !mv->HexVisible( hex ) ) mv->CenterOnHex( hex );
  Gam->SetCursor( hex );

  CalcViewPort( viewport );
  mw->SetViewPort( viewport );
  mw->Draw();
  view->EnableUpdates();
  view->Refresh();

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void TacticalWindow::Draw( void ) {
  Window::Draw();

  // show unit and building count
  Color col, colbg;
  unsigned short linewidth = sfont->Width() * 20, lx, l1y, l2y, lh;
  char numbuf[4];

  unsigned short bs1 = 0, bs2 = 0, bstotal = 0;
  Building *b = static_cast<Building *>( Gam->BuildingsList()->Head() );
  while ( b ) {
    if ( b->Owner() ) {
      if ( b->Owner()->ID() == PLAYER_ONE ) ++bs1;
      else if ( b->Owner()->ID() == PLAYER_TWO ) ++bs2;
    }
    ++bstotal;
    b = static_cast<Building *>( b->Next() );
  }


  lh = (h - mw->y - mw->h - sfont->Height() * 3 - 9) / 5;
  lx = MIN( (w - linewidth - 10) / 2, mw->x + 5 );
  l1y = mw->y + mw->h + lh * 2;
  l2y = h - (sfont->Height() + lh) * 2 - 9;
  DrawBox( Rect( lx - 5, l1y - lh, w - 2*lx + 10, (sfont->Height() * 2 + lh * 3) ),
           BOX_RECESSED );

  lx = (w - linewidth - 10) / 2;
  sfont->Write( "Units", this, lx, l1y );
  sfont->Write( "Buildings", this, lx, l2y );

  lx += sfont->Width() * 10;
  col = Gam->GetPlayer(PLAYER_ONE)->LightColor();
  colbg = Gam->GetPlayer(PLAYER_ONE)->DarkColor();
  sfont->Write( itoa( Gam->GetPlayer(PLAYER_ONE)->Units(0), numbuf ), this,
                lx, l1y, col );
  sfont->Write( itoa( bs1, numbuf ), this, lx, l2y, col );

  lx += sfont->Width() * 4;
  col = Gam->GetPlayer(PLAYER_TWO)->LightColor();
  colbg = Gam->GetPlayer(PLAYER_TWO)->DarkColor();
  sfont->Write( itoa( Gam->GetPlayer(PLAYER_TWO)->Units(0), numbuf ), this,
                lx, l1y, col );
  sfont->Write( itoa( bs2, numbuf ), this, lx, l2y, col );

  lx += sfont->Width() * 4;
  sfont->Write( itoa( Gam->UnitsList()->CountNodes(), numbuf ), this, lx, l1y );
  sfont->Write( itoa( bstotal, numbuf ), this, lx, l2y );
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::CalcViewPort
// DESCRIPTION: Determine which part of the map is currently displayed
//              on the map window and set the viewport rect accordingly.
// PARAMETERS : vp - Rect buffer to hold the view port coordinates
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void TacticalWindow::CalcViewPort( Rect &vp ) const {
  vp.x = mv->MinXHex(-1);
  vp.y = mv->MinYHex(-1);
  vp.w = (mv->MaxXHex(-1,0) + 1) - vp.x;
  vp.h = (mv->MaxYHex(-1,0) + 1) - vp.y;
}

