// Crimson Fields -- a hex-based game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

/////////////////////////////////////////////////////////////////////
// main.cpp -- Crimson Fields
/////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <string.h>

#ifndef _WIN32_WCE
# include <time.h>
# include <fstream.h>
#endif

#include "SDL.h"
#include "SDL_endian.h"

#include "view.h"
#include "game.h"
#include "misc.h"
#include "fileio.h"
#include "initwindow.h"
#include "globals.h"
#include "sound.h"
#include "options.h"

// global vars
Game *Gam;
Image *Images[NUM_IMAGES] = { NULL };

// global options, can be accessed from everywhere
Options CFOptions;

struct GUIOptions {
  short px_width;
  short px_height;
  short bpp;
  bool sfx;
  bool music;
  unsigned char sfx_vol;
  unsigned char music_vol;
  unsigned long sdl_flags;
  const char *level;
};

// local vars
static View *display;

// local function prototypes
static void parse_options( int argc, char **argv, GUIOptions &opts );
static void print_usage( char *prog );
static View *init( GUIOptions &opts );
static void load_settings( GUIOptions &opts );
static void save_settings( View *display );
static void set_icon( const Surface &s, const Rect &icon );
static GUI_Status event_filter( SDL_Event &event, Window *window );
static void do_exit( void );

#ifdef _WIN32_WCE
static bool init_wince( struct GUIOptions &opts );
# define time(n) GetTickCount()
#endif

// Start of program functions //////////////////////////////////////////

#ifdef _WIN32_WCE
int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrev, LPWSTR szCmdLine, int sw) {
#else
int main( int argc, char **argv ) {
#endif
  struct GUIOptions guiopts = { 800, 600, DISPLAY_BPP, true, true,
                    MIX_MAX_VOLUME*3/4, MIX_MAX_VOLUME/2, SDL_HWSURFACE, NULL };

#ifdef _WIN32_WCE
  if ( !init_wince( guiopts ) ) return 0;
#else
  parse_options( argc, argv, guiopts );
#endif

  display = init( guiopts );
  if ( display ) {

    // only open intro screen if the user didn't supply a level on the command line
    int intro = 1;
    if ( guiopts.level ) {
      Gam = new Game( display );
      intro = Gam->Load( guiopts.level );
      if ( !intro &&
           (!Gam->GetLevelPassword() ||
             Gam->CheckPassword( "Access restricted", "Enter access code",
                                 Gam->GetLevelPassword(), false )) ) Gam->StartTurn();
      else {
        delete Gam;
        Gam = NULL;
        intro = 1;
      }
    }

    GUI_Status status;
    do {
      if ( intro ) {
        display->Refresh();
        TitleWindow *twin = new TitleWindow( display );
        new InitWindow( display, twin );
      } else intro = 1;

      do {
        status = display->HandleEvents();
      } while ( (status != GUI_QUIT) && (status != GUI_RESTART) );

      delete Gam;
      Gam = NULL;
      display->CloseAllWindows();
    } while ( status != GUI_QUIT );
  }

  do_exit();
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : parse_options
// DESCRIPTION: Process any options given to the program on the command
//              line.
// PARAMETERS : argc - argument count
//              argv - pointer to array of arguments
//              opts - buffer to store GUI and audio options; should be
//                     initialized with defaults before calling this
//                     method
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

// parse command line arguments
void parse_options( int argc, char **argv, GUIOptions &opts ) {
  load_settings( opts );

  while ( argc > 1 ) {
    --argc;

    if (strcmp(argv[argc-1], "--width") == 0) {
      opts.px_width = atoi(argv[argc]);
    } else if (strcmp(argv[argc-1], "--height") == 0) {
      opts.px_height = atoi(argv[argc]);
    } else if (strcmp(argv[argc-1], "--level") == 0) {
      opts.level = argv[argc];
    } else if (strcmp(argv[argc-1], "--fullscreen") == 0) {
      if ( atoi( argv[argc] ) ) opts.sdl_flags |= SDL_FULLSCREEN;
      else opts.sdl_flags &= ~SDL_FULLSCREEN;
    } else if (strcmp(argv[argc-1], "--sound") == 0) {
      if ( atoi( argv[argc] ) ) opts.sfx = opts.music = true;
      else opts.sfx = opts.music = false;
    } else {
      if (strcmp(argv[argc], "--version") == 0)
        fprintf( stdout, PROGRAMNAME" "VERSION"\n" );
      else print_usage( argv[0] );
      exit ( 0 );
    }
    --argc;
  }
  if ( opts.px_width < MIN_XRES ) opts.px_width = MIN_XRES;
  if ( opts.px_height < MIN_YRES ) opts.px_height = MIN_YRES;
}

////////////////////////////////////////////////////////////////////////
// NAME       : print_usage
// DESCRIPTION: Print a usage message to stdout.
// PARAMETERS : prog - program name as given on the command line
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void print_usage( char *prog ) {
  fprintf( stdout, "Usage: %s [options]\n\n"
                   "Available options:\n"
                   "  --level <level>      load level or save file\n"
                   "  --width <width>      set screen width\n"
                   "  --height <height>    set screen height\n"
                   "  --fullscreen <1|0>   enable/disable fullscreen mode\n"
#ifndef DISABLE_SOUND
                   "  --sound <1|0>        enable/disable sound\n"
#endif
                   "  --help               display this help and exit\n"
                   "  --version            output version information and exit\n",
        prog );
}

////////////////////////////////////////////////////////////////////////
// NAME       : init
// DESCRIPTION: Initialize the display, the system icons surface and the
//              game fonts.
// PARAMETERS : opts  - contains the user settings for the sound and
//                      graphics subsystems
// RETURNS    : a pointer to the display surface on success, NULL
//              otherwise
////////////////////////////////////////////////////////////////////////

View *init( GUIOptions &opts ) {
  View *view = NULL;
  Font *f1 = NULL, *f2 = NULL;
  Surface *icons = NULL;
  bool ok = false;

  if ( SDL_Init( SDL_INIT_VIDEO ) >= 0 ) {
    string datpath( get_data_dir() );
    datpath.append( CF_DATFILE );
    SDL_RWops *datfile = SDL_RWFromFile( datpath.c_str(), "rb" );
    if ( datfile ) {

      icons = new Surface;		// load icons surface
      if ( !icons->LoadImageData( datfile ) ) {

        set_icon( *icons, Rect(64, 0, 32, 32) );

        view = new View( opts.px_width, opts.px_height, opts.bpp,
                         opts.sdl_flags );
        if ( view->s_surface ) {
          f1 = new Font( datfile );	// load fonts
          f2 = new Font( datfile );
          ok = true;
        } else fprintf( stderr, "Couldn't set video mode: %s\n", SDL_GetError() );

      } else fprintf( stderr, "Error reading data file\n" );

      SDL_RWclose( datfile );

    } else fprintf( stderr, "Error: Couldn't open '%s'\n", datpath.c_str() );
  } else fprintf( stderr, "Couldn't initialize: %s\n", SDL_GetError() );

  if ( ok ) {
    // set main window title
    SDL_WM_SetCaption( PROGRAMNAME, PROGRAMNAME );
    SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL );

    srand( time(0) );				// initialize random number generator

    Audio::InitSfx( opts.sfx, opts.sfx_vol );	// load sound effects
    Audio::InitMusic( opts.music, opts.music_vol );

    for ( int i = 0; i <= XP_MAX_LEVEL; ++i )
      Images[ICON_XP_BASE+i] = new Image( icons, 64 + i * XP_ICON_WIDTH, 32, XP_ICON_WIDTH, XP_ICON_HEIGHT );

    icons->DisplayFormat();

    view->SetSmallFont( f1 );
    view->SetLargeFont( f2 );
    view->SetFGPen( Color(CF_COLOR_HIGHLIGHT) );
    view->SetBGPen( Color(CF_COLOR_SHADOW) );
    view->SetEventFilter( event_filter );
    view->SetSystemIcons( icons );

    // create options and save games directories if possible
    create_config_dir();

    return view;
  } else {
    delete icons;
    delete view;
    return NULL;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : load_settings
// DESCRIPTION: Read default display settings from the crimsonrc file.
// PARAMETERS : opts - buffer to store the settings. These should
//                     already be initialized with some defaults in
//                     case the rc file doesn't exist or this function
//                     fails.
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void load_settings( GUIOptions &opts ) {
  string crimsonrc( get_config_dir() );
  crimsonrc.append( CRIMSONRC );

  FILE *fp = fopen( crimsonrc.c_str(), "r" );
  if ( fp ) {
    char linebuf[256], *val;
    unsigned short linecnt = 0;

    while ( fgets( linebuf, 255, fp ) ) {
      ++linecnt;
      if ( (linebuf[0] != '#') && (linebuf[0] != '\n') ) {
        val = strchr( linebuf, ' ' );
        if ( val ) {
          while ( *val == ' ' ) ++val;

          if ( !strncmp( linebuf, "width", 5 ) ) opts.px_width = atoi(val);
          else if ( !strncmp( linebuf, "height", 6 ) ) opts.px_height = atoi(val);
          else if ( !strncmp( linebuf, "fullscreen", 10 ) ) {
            if ( atoi(val) != 0 ) opts.sdl_flags |= SDL_FULLSCREEN;
          } else if ( !strncmp( linebuf, "sfxvol", 6 ) ) opts.sfx_vol = atoi(val);
          else if ( !strncmp( linebuf, "musicvol", 8 ) ) opts.music_vol = atoi(val);
          else if ( !strncmp( linebuf, "sfx", 3 ) ) opts.sfx = (atoi(val) != 0);
          else if ( !strncmp( linebuf, "music", 5 ) ) opts.music = (atoi(val) != 0);
          else if ( !strncmp( linebuf, "showdamage", 10 ) ) CFOptions.SetDamageIndicator( atoi(val) != 0 );
          else if ( !strncmp( linebuf, "showreplay", 10 ) ) CFOptions.SetTurnReplay( atoi(val) != 0 );
          else fprintf( stderr, "warning: unrecognized config option in line %d\n", linecnt );
        }
      }
    }
    fclose( fp );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : save_settings
// DESCRIPTION: Save current display settings to the crimsonrc file.
// PARAMETERS : display - pointer to display
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void save_settings( View *display ) {
  string crimsonrc( get_config_dir() );
  crimsonrc.append( CRIMSONRC );

  FILE *fp = fopen( crimsonrc.c_str(), "w" );
  if ( fp ) {
    fprintf( fp, "width %d\n", display->Width() );
    fprintf( fp, "height %d\n", display->Height() );
    fprintf( fp, "fullscreen %d\n", display->IsFullScreen() );
    fprintf( fp, "sfx %d\n", Audio::GetSfxState() );
    fprintf( fp, "music %d\n", Audio::GetMusicState() );
    fprintf( fp, "sfxvol %d\n", Audio::GetSfxVolume() );
    fprintf( fp, "musicvol %d\n", Audio::GetMusicVolume() );
    fprintf( fp, "showdamage %d\n", CFOptions.GetDamageIndicator() );
    fprintf( fp, "showreplay %d\n", CFOptions.GetTurnReplay() );
    fclose( fp );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : set_icon
// DESCRIPTION: Set the application icon.
// PARAMETERS : s    - icon surface
//              icon - icon position and size on the surface
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void set_icon( const Surface &s, const Rect &icon ) {
  Surface is;
  is.Create( icon.Width(), icon.Height(), DISPLAY_BPP, 0 );
  is.SetColorKey( s.GetColorKey() );
  is.Flood( is.GetColorKey() );
  s.Blit( &is, icon, 0, 0 );

  SDL_WM_SetIcon( is.s_surface, NULL );
}

////////////////////////////////////////////////////////////////////////
// NAME       : event_filter
// DESCRIPTION: This is the global event filter function. It is hooked
//              to the display and called everytime the event handler
//              receives an event.
// PARAMETERS : event  - event received by the event handler
//              window - pointer to the currently active window
// RETURNS    : GUI_Status; if the filter returns GUI_NONE the event
//              handler will not pass the event to its windows, but
//              silently drop it.
////////////////////////////////////////////////////////////////////////

GUI_Status event_filter( SDL_Event &event, Window *window ) {
  GUI_Status rc = GUI_OK;

  if ( event.type == SDL_KEYDOWN ) {
    rc = GUI_NONE;

    switch ( event.key.keysym.sym ) {
    case SDLK_F11:            // toggle sound
      Audio::ToggleSfxState();
      break;
    default:
      rc = GUI_OK;            // send to windows
    }
  } else if ( event.type == SDL_QUIT ) do_exit();

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : init_wince
// DESCRIPTION: Initialize on Windows CE.
// PARAMETERS : opts - program options buffer (containing defaults)
// RETURNS    : TRUE if initialization succeeded, FALSE otherwise
////////////////////////////////////////////////////////////////////////

#ifdef _WIN32_WCE
bool init_wince( struct GUIOptions &opts ) {
  HWND hwnd = FindWindow( _T("SDL_app"), _T(PROGRAMNAME) );
  if ( hwnd ) {
    SetForegroundWindow( hwnd );
    ShowWindow( hwnd, SW_SHOW );
    return false;
  }

  load_settings( opts );

  unsigned short maxw = GetSystemMetrics(SM_CXSCREEN); 
  unsigned short maxh = GetSystemMetrics(SM_CYSCREEN); 

  if( opts.px_width > maxw ) opts.px_width = maxw;
  if( opts.px_height > maxh ) opts.px_height = maxh;

  opts.sdl_flags |= SDL_FULLSCREEN;
  return true;
}
#endif

////////////////////////////////////////////////////////////////////////
// NAME       : do_exit
// DESCRIPTION: Free all resources and exit the program.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void do_exit( void ) {
  Audio::ShutdownSfx();
  Audio::ShutdownMusic();

  if ( display ) {
    save_settings( display );
    delete display;
  }

  for ( int i = 0; i < NUM_IMAGES; ++i ) delete Images[i];

  SDL_Quit();
  exit( 0 );
}

