// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// game.cpp
////////////////////////////////////////////////////////////////////////

#include <string.h>
#include "SDL_endian.h"

#include "game.h"
#include "path.h"
#include "combat.h"
#include "history.h"
#include "event.h"
#include "unitwindow.h"
#include "filewindow.h"
#include "initwindow.h"
#include "fileio.h"
#include "sound.h"
#include "ai.h"
#include "options.h"

extern Options CFOptions;

// button identifiers used by the game hook function dispatcher
#define G_BUTTON_END_TURN	 10
#define G_BUTTON_MAP		 11
#define G_BUTTON_BRIEFING	 12
#define G_BUTTON_SAVE		 13
#define G_BUTTON_LEV_INFO	 14
#define G_BUTTON_GENERAL_OPTIONS 15
#define G_BUTTON_VIDEO_OPTIONS	 16
#define G_BUTTON_SOUND_OPTIONS	 17
#define G_BUTTON_ABORT		 18
#define G_BUTTON_QUIT		 19

#define G_BUTTON_UNIT_INFO	 20
#define G_BUTTON_UNIT_CONTENT	 21
#define G_BUTTON_UNIT_SWEEP	 22


////////////////////////////////////////////////////////////////////////
// NAME       : Game::Game
// DESCRIPTION: Create a new game.
// PARAMETERS : view - the display surface
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Game::Game( View *view ) {
  this->view = view;
  flags = 0;
  players[PLAYER_ONE] = players[PLAYER_TWO] = NULL;
  player = NULL;
  unit = NULL;
  messages = NULL;

  map = NULL;
  history = NULL;
  unit_set = NULL;
  terrain_set = NULL;
  shader = NULL;

  mwin = new MapWindow( 0, 0, view->Width(), view->Height(), 0, view );
  if ( CFOptions.GetDamageIndicator() ) mwin->GetMapView()->EnableUnitStats();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::~Game
// DESCRIPTION: Destroy the game.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Game::~Game( void ) {
  view->CloseWindow( mwin );
  delete players[PLAYER_ONE];
  delete players[PLAYER_TWO];
  delete [] messages;
  delete history;
  delete unit_set;
  delete terrain_set;
  delete shader;
  delete map;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::Quit
// DESCRIPTION: Ask the user for confirmation and exit.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::Quit( void ) const {
  Audio::PlaySfx( Audio::SND_GUI_ASK, 0 );
  DialogWindow *req = new DialogWindow( NULL, "Do you really want to quit?",
                      "yYes|nNo", 1, 0, view );
  req->SetButtonID( 0, GUI_QUIT );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CreateSaveFileName
// DESCRIPTION: Get the name of the file to save to. Pop up a file
//              requester if necessary.
// PARAMETERS : filename - default filename
// RETURNS    : file name or empty string on error
////////////////////////////////////////////////////////////////////////

string Game::CreateSaveFileName( const char *filename ) const {
  string file;
  if ( filename ) file.append( filename );

  bool filesel;
  do {
    GUI_Status rc;
    DialogWindow *dw;
    filesel = false;

    if ( file.length() == 0 ) filesel = true;
    else if ( exists( file.c_str() ) ) {
      // if file exists let user confirm the write
      char *conmsg = new char[ file.length() + 20 ];
      strcpy( conmsg, file.c_str() );
      strcat( conmsg, " exists. Overwrite?" );
      dw = new DialogWindow( NULL, conmsg, "yYes|nNo", 1, 0, view );
      dw->SetButtonID( 0, 1 );
      dw->SetButtonID( 1, 0 );
      rc = dw->EventLoop();
      if ( rc == 0 ) filesel = true;
      view->CloseWindow( dw );
      delete [] conmsg;
    }

    if ( filesel ) {
      bool done = false;
      FileWindow *fw = new FileWindow( get_save_dir().c_str(), last_file_name.c_str(),
                                       ".sav", WIN_FILE_SAVE, view );
      fw->ok->SetID( 1 );
      fw->cancel->SetID( 0 );

      do {
        rc = fw->EventLoop();

        if ( rc == 1 ) {
          file = fw->GetFile();
          if ( file.length() != 0 ) {
            view->CloseWindow( fw );
            done = true;
          }
        } else if ( rc == 0 ) {
          if ( flags & GI_PBEM ) {
            // if saving a PBeM game is aborted the game data is lost...
            dw = new DialogWindow( "Warning", "Really abort? Current game will be lost!",
                                   "wSo what?|oOops!", 1, WIN_FONT_BIG, view );
            Audio::PlaySfx( Audio::SND_GUI_ASK, 0 );
            dw->SetButtonID( 0, 1 );
            dw->SetButtonID( 1, 0 );
            rc = dw->EventLoop();
            view->CloseWindow( dw );

            if ( rc != 0 ) {
              view->CloseWindow( fw );
              file = "";
              done = true;
              filesel = false;
            }
          } else {
            view->CloseWindow( fw );
            file.assign( "" );
            done = true;
            filesel = false;
          }
        }

      } while ( !done );
    }
  } while ( filesel );

  return file;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::Save
// DESCRIPTION: Save the current game to a file.
// PARAMETERS : filename - data file name; if NULL pops up a file
//                         selection window
// RETURNS    : 0 on successful write, non-zero otherwise
////////////////////////////////////////////////////////////////////////

int Game::Save( const char *filename ) {
  string fname( CreateSaveFileName( filename ) );
  if ( fname.length() == 0 ) return -1;
 
  SDL_RWops *file = SDL_RWFromFile( fname.c_str(), "wb" );
  if ( !file ) {
    new NoteWindow( "Error!", "Couldn't open file for writing", 0, view );
    return -1;
  }

  unsigned char version = FILE_VERSION, current_player = player->ID();
  unsigned short len, num;

  // set last filename
  last_file_name = file_part( fname );
  num = last_file_name.rfind( '.' );
  last_file_name.erase( num );

  // save game info
  SDL_WriteLE32( file, FID_MISSION );
  SDL_RWwrite( file, &version, 1, 1 );

  SDL_WriteLE16( file, flags );
  SDL_WriteLE16( file, turn );

  SDL_RWwrite( file, &level_info, 1, 1 );
  SDL_RWwrite( file, &next_map, 1, 1 );
  SDL_RWwrite( file, &current_player, 1, 1 );
  SDL_RWwrite( file, &turn_phase, 1, 1 );

  len = unit_set->GetName().length();            // save mission set info
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, unit_set->GetName().c_str(), 1, len );
  len = terrain_set->GetName().length();
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, terrain_set->GetName().c_str(), 1, len );

  map->Save( file );     // save map

  players[PLAYER_ONE]->Save( file );              // save player data
  players[PLAYER_TWO]->Save( file );

  num = buildings.CountNodes();                 // save buildings
  SDL_WriteLE16( file, num );
  for ( Building *b = static_cast<Building *>( buildings.Head() );
        b; b = static_cast<Building *>( b->Next() ) ) b->Save( file );

  // save transports; basically, transports are not much different
  // from other units but we MUST make sure that transports having
  // other units on board are loaded before those units; we need to
  // make two passes through the list, first saving all unsheltered
  // units (which are possibly transports carrying other units), and
  // all sheltered units in the second run
  num = units.CountNodes();
  SDL_WriteLE16( file, num ); 
  Unit *u;
  for ( u = static_cast<Unit *>( units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {
    if ( !u->IsSheltered() ) u->Save( file );
  }

  for ( u = static_cast<Unit *>( units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {
    if ( u->IsSheltered() ) u->Save( file );
  }

  num = combat.CountNodes();                           // save combat data
  SDL_WriteLE16( file, num );
  for ( Combat *c = static_cast<Combat *>( combat.Head() );
        c; c = static_cast<Combat *>( c->Next() ) )
    c->Save( file );

  num = events.CountNodes();                           // save events
  SDL_WriteLE16( file, num );
  for ( Event *e = static_cast<Event *>( events.Head() );
        e; e = static_cast<Event *>( e->Next() ) )
    e->Save( file );

  SDL_WriteLE16( file, msg_num );                     // save text messages
  SDL_WriteLE32( file, msg_len );
  for ( num = 0; num < msg_num; ++num ) crypt( messages[num] );
  SDL_RWwrite( file, messages[0], 1, msg_len );
  for ( num = 0; num < msg_num; ++num ) crypt( messages[num] );

  if ( history ) history->Save( file );               // save turn history
  else SDL_WriteLE16( file, 0 );

  SDL_RWclose( file );

  if ( flags & GI_PBEM ) {
    fname.insert( 0, "Current PBeM game was saved to\n" );
    NoteWindow *nw = new NoteWindow( "Game saved", fname.c_str(), 0, view );
    nw->SetButtonID( 0, GUI_RESTART );
  }
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : Game::Load
// DESCRIPTION: Load a game from a mission file (start a new game) or
//              a save file (resume game).
// PARAMETERS : filename - data file name
// RETURNS    : 0 on success, non-zero otherwise
////////////////////////////////////////////////////////////////////////

int Game::Load( const char *filename ) {
  int rc = -1;
  SDL_RWops *file = SDL_RWFromFile( filename, "rb" );
  if ( !file ) return -1;

  // read game info
  unsigned char version;
  unsigned long id = SDL_ReadLE32( file );
  SDL_RWread( file, &version, 1, 1 );
  if ( (version == FILE_VERSION) && (id == FID_MISSION) ) {
    unsigned short len, i;
    unsigned char current_player;
    char *uset, *tset;

    // set last filename
    last_file_name = file_part( filename );
    i = last_file_name.rfind( '.' );
    last_file_name.erase( i );

    flags = SDL_ReadLE16( file );   // if this is a new map (GI_SAVEFILE not set)
                                    // GI_AI only has informational purposes
    if ( !(flags & GI_SAVEFILE) ) {
      flags &= ~GI_AI;
      flags |= (GI_SAVEFILE|
               (CFOptions.GetComputerPlayer() ? GI_AI :
               (CFOptions.GetPBEM() ? GI_PBEM : 0)));
    }
    turn = SDL_ReadLE16( file );

    if ( flags & GI_PASSWORD ) {
      SDL_RWread( file, password, 1, 8 );
      crypt( password );
      flags ^= GI_PASSWORD;  // level password is never saved
    } else password[0] = '\0';

    SDL_RWread( file, &level_info, 1, 1 );
    SDL_RWread( file, &next_map, 1, 1 );
    SDL_RWread( file, &current_player, 1, 1 );
    SDL_RWread( file, &turn_phase, 1, 1 );

    len = SDL_ReadLE16( file );        // load name of unit set
    uset = new char [len + 1];
    SDL_RWread( file, uset, 1, len );
    uset[len] = '\0';

    len = SDL_ReadLE16( file );        // load name of terrain set
    tset = new char [len + 1];
    SDL_RWread( file, tset, 1, len );
    tset[len] = '\0';

    map = new Map();
    map->Load( file );			// load map

    unit_set = new UnitSet;
    if ( unit_set->Load( uset ) ) {
      fprintf( stderr, "Error: Unit set '%s' not available.\n", uset );
      delete [] uset;
      return -1;
    }
    delete [] uset;

    terrain_set = new TerrainSet;
    if ( terrain_set->Load( tset ) ) {
      fprintf( stderr, "Error: Terrain set '%s' not available.\n", tset );
      delete [] tset;
      return -1;
    }
    delete [] tset;

    map->SetUnitSet( unit_set );
    map->SetTerrainSet( terrain_set );
    mwin->GetMapView()->SetMap(map);	// attach map to map window
    shader = new MoveShader( map, units, mwin->GetMapView()->GetFogBuffer() );

    players[PLAYER_ONE] = new Player( file, HUMAN );
    players[PLAYER_TWO] = new Player( file, (flags & GI_AI) ? COMPUTER : HUMAN );
    player = players[current_player];

    len = SDL_ReadLE16( file );		// load buildings
    for ( i = 0; i < len; ++i ) {
      Building *b = new Building( file, players );
      buildings.AddTail( b );
      map->SetBuilding( b, b->Position() );
    }

    len = SDL_ReadLE16( file );		// load units
    for ( i = 0; i < len; ++i ) {
      Unit *u = LoadUnit( file );
      if ( u ) {
        units.AddTail( u );
        map->SetUnit( u, u->Position() );
      }
    }

    len = SDL_ReadLE16( file );		// load combats
    for ( i = 0; i < len; ++i ) {
      combat.AddTail( new Combat( file, this ) );
    }

    len = SDL_ReadLE16( file );		// load events
    for ( i = 0; i < len; ++i ) {
      events.AddTail( new Event( file, players ) );
    }

    // load text messages
    msg_num = SDL_ReadLE16( file );		// number of messages
    msg_len = SDL_ReadLE32( file );
    messages = (char **)new char[msg_len + (msg_num + 1) * sizeof(char *)];
    messages[msg_num] = 0;
    if ( msg_num ) {                         // load messages
      char *ptr;
      int j;

      ptr = (char *)&messages[msg_num + 1];
      SDL_RWread( file, ptr, 1, msg_len );
      messages[0] = ptr;
      crypt( ptr );

      i = 1; j = 0;
      while ( i < msg_num ) {
        if ( ptr[j++] == '\0' ) {
          messages[i] = &ptr[j];
          crypt( messages[i++] );
        }
      }
    }

    len = SDL_ReadLE16( file );
    if ( len ) history = new History( file, len );
    else if ( NextPlayer()->Type() == COMPUTER ) history = NULL;
    else history = new History();

    rc = 0;
  } else
    fprintf( stderr, "Error: File '%s' is not of the required type\n", filename );
 
  SDL_RWclose( file );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::LoadUnit
// DESCRIPTION: Load a unit from a file. This is a separate function
//              because some parameters must be supplied which are only
//              accessible to the Game class itself.
// PARAMETERS : file  - file descriptor
//              force - override automatic unit type detection and
//                      ignore transports (required for turn history);
//                      defaults to FALSE
// RETURNS    : pointer to loaded unit or NULL on error
////////////////////////////////////////////////////////////////////////

Unit *Game::LoadUnit( SDL_RWops *file, bool force ) {
  unsigned char tid, pid;
  const UnitType *type;
  Player *p = NULL;
  Unit *u;

  SDL_RWread( file, &tid, 1, 1 );
  SDL_RWread( file, &pid, 1, 1 );

  if ( pid != PLAYER_NONE ) p = players[pid];
  type = unit_set->GetUnitInfo( tid );

  if ( !force && (type->Flags() & U_TRANSPORT) )
    u = new Transport( file, type, p );
  else u = new Unit( file, type, p );
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SwitchMap
// DESCRIPTION: Dump the current map and move on to another one.
// PARAMETERS : mission - map file name (excluding path and suffix)
// RETURNS    : -1 on error, 0 otherwise
////////////////////////////////////////////////////////////////////////

int Game::SwitchMap( const char *mission ) {
  string mapname = get_levels_dir();
  mapname.append( mission );
  mapname.append( ".lev" );

  mwin->GetMapView()->Disable();
  mwin->Draw();
  mwin->Show();

  // remove current game data
  delete players[PLAYER_ONE];
  delete players[PLAYER_TWO];
  players[PLAYER_ONE] = players[PLAYER_TWO] = NULL;

  delete [] messages;
  messages = NULL;
  delete history;
  history = NULL;
  delete unit_set;
  unit_set = NULL;
  delete terrain_set;
  terrain_set = NULL;
  delete shader;
  shader = NULL;
  delete map;
  map = NULL;

  while ( !units.IsEmpty() ) delete units.RemHead();
  while ( !buildings.IsEmpty() ) delete buildings.RemHead();
  while ( !events.IsEmpty() ) delete events.RemHead();
  while ( !combat.IsEmpty() ) delete combat.RemHead();

  // reset variables
  player = NULL;
  unit = NULL;
  flags = 0;

  // load new map
  return Load( mapname.c_str() );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::StartTurn
// DESCRIPTION: Start a turn, i.e. draw the map to the display.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::StartTurn( void ) {
  MapView *mv = mwin->GetMapView();

  view->SetFGPen( player->LightColor() );   // set player color scheme
  view->SetBGPen( player->DarkColor() );

  if ( player->Type() == HUMAN ) {
    char buf[10];
    NoteWindow *nw;

    // show turn information and greeting dialog
    sprintf( buf, "Turn %d", Turn() );

    nw = new NoteWindow( player->Name(), buf, WIN_FONT_BIG|WIN_CENTER, view );
    nw->EventLoop();
    view->CloseWindow( nw );
  }

  // in email games set up or check player passwords
  if ( flags & GI_PBEM ) {
    if ( !player->Password() ) SetPlayerPassword( player );
    else CheckPassword( player->Name(), "Enter password",
                        player->Password(), true );
  }

  if ( turn_phase == TURN_START ) {
    // replay
    if ( history ) {
      history->Replay( mwin );
      delete history;
    }

    // begin new turn
    if ( NextPlayer()->Type() == COMPUTER ) history = NULL;
    else history = new History();

    // check for victory conditions
    if ( flags & GI_GAME_OVER ) {
      ShowDebriefing( player, true );
      return;
    }

    turn_phase = TURN_IN_PROGRESS;
    mv->SetCursorImage( IMG_CURSOR_IDLE );
    CheckEvents();
  }

  if ( player->Type() == COMPUTER ) {
    AI ai( player, mv->GetMap() );
    ai.Play();
    EndTurn();
  } else {
    // set the cursor to one of the player's units
    Point startcursor( 0, 0 );
    for ( Unit *u = static_cast<Unit *>( units.Head() );
          u; u = static_cast<Unit *>( u->Next() ) ) {
      if ( u->Owner() == player ) {
        startcursor = u->Position();
        break;
      }
    }

    view->DisableUpdates();
    mv->Enable();
    mv->CenterOnHex( startcursor );
    SetCursor( startcursor );
    view->EnableUpdates();
    view->Refresh();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SetPlayerPassword
// DESCRIPTION: Set a password for a player in an email game.
// PARAMETERS : player - player to set password for
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::SetPlayerPassword( Player *player ) const {
  bool pw_ok = false;
  PasswordWindow *pwin;
  const char *pw, *msg1 = "Choose a password";

  pwin = new PasswordWindow( player->Name(), msg1,
                             player->Password(), view );

  do {
    pwin->EventLoop();
    pw = pwin->string->String();

    // only accept if player has entered a password
    if ( pw ) {
      pwin->NewPassword( pw );
      pwin->string->SetTitle( "Confirm password" );
      pwin->string->SetString( NULL );
      pwin->Draw();
      pwin->Show();
      pwin->string->SetFocus();

      pwin->EventLoop();
      pw_ok = pwin->PasswordOk();

      if ( !pw_ok ) {
        Audio::PlaySfx( Audio::SND_GUI_ERROR, 0 );
        pwin->NewPassword( NULL );
        pwin->string->SetTitle( msg1 );
        pwin->string->SetString( NULL );
        pwin->Draw();
        pwin->Show();
        pwin->string->SetFocus();
      }
    } else pwin->string->SetFocus();
  } while ( !pw_ok );

  player->SetPassword( pw );
  view->CloseWindow( pwin );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CheckPassword
// DESCRIPTION: Request a password from the player and compare it to a
//              given string.
// PARAMETERS : title - window title
//              msg   - message
//              pw    - password to check against
//              retry - what to do if player enters a wrong password;
//                      if TRUE retry until password is correct; if
//                      FALSE return immediately
// RETURNS    : TRUE if pw and player's password match, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Game::CheckPassword( const char *title, const char *msg,
                          const char *pw, bool retry ) const {
  PasswordWindow *pwin = new PasswordWindow( title, msg, pw, view );
  bool pw_ok = false;

  do {
    pwin->EventLoop();
    pw_ok = pwin->PasswordOk();

    if ( !pw_ok ) {
      Audio::PlaySfx( Audio::SND_GUI_ERROR, 0 );
      if ( retry ) {
        pwin->string->SetString( NULL, true );
        pwin->string->SetFocus();
      }
    }
  } while ( retry && !pw_ok );

  view->CloseWindow( pwin );
  return pw_ok;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CheckEvents
// DESCRIPTION: Check for pending game events.
// PARAMETERS : -
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status Game::CheckEvents( void ) {
  Event *e = static_cast<Event *>( events.Head() ), *e2;
  MessageWindow *win = NULL;

  while ( e ) {
    e2 = static_cast<Event *>( e->Next() );
    short check = e->Check();
    if ( check > 0 ) {
      // event will be executed and can be taken out of the queue
      e->Remove();

      // check == 2 means we need a window to display a message
      if ( (check == 2) && !win ) win = new MessageWindow( "Event", NULL, view );
      e->Execute( win );
      delete e;

      if ( check == 2 ) win->EventLoop();
    }
    e = e2;
  }
  if ( win ) view->CloseWindow( win );
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::HaveWinner
// DESCRIPTION: Check whether one of the players has won completed his
//              mission. Notify players and quit the current game if so.
// PARAMETERS : -
// RETURNS    : true if mission is completed by any player AND both
//              players have been informed so, false otherwise
////////////////////////////////////////////////////////////////////////

bool Game::HaveWinner( void ) {
  bool quit = false;

  if ( (players[PLAYER_ONE]->Success( 0 ) >= 100) ||
       (players[PLAYER_TWO]->Success( 0 ) >= 100) ) {
    flags |= GI_GAME_OVER;

    if ( NextPlayer()->Type() != HUMAN ) quit = true;

    ShowDebriefing( player, quit );
  }
  return quit;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::EndTurn
// DESCRIPTION: End turn for the current player. Execute combat orders
//              and prepare everything for the next player.
// PARAMETERS : -
// RETURNS    : GUI_Status (GUI_QUIT if mission is complete)
////////////////////////////////////////////////////////////////////////

GUI_Status Game::EndTurn( void ) {
  GUI_Status rc = GUI_OK;

  if ( unit ) DeselectUnit();
  mwin->GetMapView()->DisableCursor();
  mwin->GetPanel()->Update(NULL);


  // calculate modifiers for combat
  Combat *com = static_cast<Combat *>( combat.Head() );
  while ( com ) {
    com->CalcModifiers( mwin->GetMapView()->GetMap() );
    com = static_cast<Combat *>( com->Next() );
  }

  // execute combat orders
  while ( !combat.IsEmpty() ) {
    Combat *com = static_cast<Combat *>(combat.RemHead());
    CombatWindow *cwin = com->Resolve( mwin, view );
    delete com;

    if ( cwin ) {
      rc = cwin->EventLoop();
      view->CloseWindow( cwin );
      if ( rc == GUI_CLOSE ) rc = GUI_OK;
    }
  }

  // destroyed units may have triggered events...
  if ( rc == GUI_OK ) rc = CheckEvents();

  // check for mission completion
  if ( !HaveWinner() ) {
    // set new player
    player = NextPlayer();
    if ( player->ID() == PLAYER_ONE ) ++turn;
    turn_phase = TURN_START;

    // remove all destroyed units from the list,
    // restore movement points, reset status
    Unit *next, *u = static_cast<Unit *>(units.Head());
    while ( u ) {
      next = static_cast<Unit *>(u->Next());
      if ( !u->IsAlive() ) {
        u->Remove();
        delete u;
      } else {
        if ( u->Owner() == player ) u->RefreshMoves();
        else u->UnsetFlags( U_MOVED|U_ATTACKED|U_DONE|U_BUSY );
      }
      u = next;
    }

    // produce crystals in mines
    Building *b = static_cast<Building *>(buildings.Head());
    while ( b ) {
      if ( (b->Owner() == player) && b->IsMine() )
        b->SetCrystals( b->Crystals() + b->CrystalProduction() );
      b = static_cast<Building *>(b->Next());
    }

    // check if we're playing an email game. if so, save and exit
    if ( flags & GI_PBEM ) {
      string filebuf( get_save_dir() );
      filebuf.append( last_file_name );
      filebuf.append( ".sav" );

      int err = Save( filebuf.c_str() );
      if ( err ) {
        NoteWindow *nw = new NoteWindow( "Error", "Game was not saved!", WIN_CLOSE_ESC, view );
        nw->SetButtonID( 0, GUI_RESTART );
      }
    } else {
      mwin->GetMapView()->Disable();
      mwin->Draw();
      mwin->Show();

      StartTurn();
    }
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GetBuildingByID
// DESCRIPTION: Find the building corresponding to the given identifier.
// PARAMETERS : id - identifier of the building to be searched for
// RETURNS    : pointer to the building, or NULL if no building with
//              that ID exists
////////////////////////////////////////////////////////////////////////

Building *Game::GetBuildingByID( unsigned short id ) const {
  Building *b = static_cast<Building *>( buildings.Head() );
  while ( b ) {
    if ( b->ID() == id ) return b;
    b = static_cast<Building *>( b->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GetUnitByID
// DESCRIPTION: Find the unit corresponding to the given identifier.
// PARAMETERS : id - identifier of the unit to be searched for
// RETURNS    : pointer to the unit, or NULL if no unit with that ID
//              exists
////////////////////////////////////////////////////////////////////////

Unit *Game::GetUnitByID( unsigned short id ) const {
  Unit *u = static_cast<Unit *>( units.Head() );
  while ( u ) {
    if ( u->ID() == id ) return u;
    u = static_cast<Unit *>( u->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GetEventByID
// DESCRIPTION: Find the event corresponding to the given identifier.
// PARAMETERS : id - identifier of the event to be searched for
// RETURNS    : pointer to the event, or NULL if no event with that ID
//              exists
////////////////////////////////////////////////////////////////////////

Event *Game::GetEventByID( unsigned char id ) const {
  Event *e = static_cast<Event *>( events.Head() );
  while ( e ) {
    if ( e->ID() == id ) return e;
    e = static_cast<Event *>( e->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SetCursor
// DESCRIPTION: Set the cursor to a new hex on the map. Contrary to the
//              low-level function in MapView this updates the display
//              at the old and new position if necessary.
// PARAMETERS : cursor - new cursor position
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::SetCursor( const Point &cursor ) const {
  MapView *mv = mwin->GetMapView();
  MapObject *mobj = NULL;
  Unit *u;
  Rect upd;

  if ( mv->CursorEnabled() ) {
    Point old = mv->Cursor();

    upd = mv->SetCursor( Point(-1,-1) ); // disable cursor for hex update
    mwin->Show( upd );                   // update previous cursor position

    if ( player->Mode() == MODE_IDLE ) {
      // if we had highlighted a unit's target we need to remove that mark
      const Point *target;
      u = map->GetUnit( old );
      if ( u && (u->Owner() == player) && (target = u->Target()) ) {
        upd = mv->UpdateHex( *target );
        mwin->Show( upd );
      }
    } else if ( player->Mode() == MODE_BUSY ) mv->SetCursorImage( IMG_CURSOR_SELECT );
  }

  if ( cursor.x != -1 ) {
    u = map->GetUnit( cursor );
    if ( u ) {
      if ( u->Owner() != player ) {
        if ( (player->Mode() == MODE_BUSY) && unit->CanHit( u ) )
          mv->SetCursorImage( IMG_CURSOR_ATTACK );
      } else if ( player->Mode() == MODE_IDLE ) {
        // if it's a unit of the active player, highlight its target if it has one
        const Point *target = u->Target();
        if ( target ) {
          Point p = mv->Hex2Pixel( *target );
          mv->DrawTerrain( IMG_CURSOR_HIGHLIGHT, mwin, p.x, p.y, *mwin);
          mwin->Show( Rect(p.x, p.y, GFX_WIDTH, GFX_HEIGHT) );
        }
      }
    }

    mobj = map->GetMapObject( cursor );
  }
  upd = mv->SetCursor( cursor );
  mwin->GetPanel()->Update( mobj );
  mwin->Show( upd );
}


////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveUnit
// DESCRIPTION: Move a unit to another hex.
// PARAMETERS : u  - unit to be moved
//              hx - destination hex x
//              hy - destination hex y
// RETURNS    : the unit if it's still available, or NULL if it
//              cannot be selected this turn (read: deselect it)
////////////////////////////////////////////////////////////////////////

Unit *Game::MoveUnit( Unit *u, const Point &dest ) {
  if ( (player->Type() == HUMAN) &&
       (shader->GetStep(dest) == -1) ) return u;

  if ( u->Position() != dest ) {
    MapView *mv = mwin->GetMapView();
    const Point &pos = u->Position();

    Path path( map );
    if ( path.Find( u, pos, dest ) == -1 ) {
      fprintf( stderr, "Internal error: FindPath() failed!\n" );
      return u;
    }

    SoundEffect *sfx = u->MoveSound();
    if ( sfx ) sfx->Play( Audio::SFX_LOOP );

    if ( !u->IsSheltered() ) {
      map->SetUnit( NULL, pos );
      mv->UpdateHex( pos );
    } else {
      Transport *t = static_cast<Transport *>( map->GetUnit( pos ) );
      if ( t ) t->RemoveUnit( u );
      else map->GetBuilding( pos )->RemoveUnit( u );
    }

    short step, oldhex = map->HexTypeID( dest ); 
    while ( (step = path.GetStep( u->Position() )) != -1)
      MoveUnit( u, (Direction)step );

    u->SetFlags( U_MOVED );
    u->SetMoves( 0 );

    if ( sfx ) sfx->Stop();

    const Point &posnew = u->Position();
    int conquer = map->SetUnit( u, posnew );
    mwin->Show( mv->UpdateHex( posnew ) );

    if ( history && (conquer == 1) )                     // a building was conquered
      history->RecordTileEvent( map->HexTypeID( posnew ), oldhex, posnew.x, posnew.y );

    if ( u->IsSlow() || !UnitTargets( u ) ) {
      u->SetFlags( U_DONE );
      if ( u == unit ) DeselectUnit();
      CheckEvents();
      return NULL;
    }

    shader->Clear();
    shader->ShadeMap( u );
    if ( mv->CursorEnabled() ) mwin->GetPanel()->Update( u );
    mwin->Draw();
    mwin->Show();
    CheckEvents();
  }
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveUnit
// DESCRIPTION: Move a unit one hex in a given direction.
// PARAMETERS : u   - unit to be moved
//              dir - direction
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Game::MoveUnit( Unit *u, Direction dir ) {
  MapView *mv = mwin->GetMapView();
  const Point &pos = u->Position();
  Point posnew;
  if ( map->Dir2Hex( pos, dir, posnew ) ) return -1;

  short cost, moves = u->Moves();
  if ( u->IsAircraft() || u->IsMine() || map->GetUnit( posnew ) ) cost = MCOST_MIN;
  else cost = map->MoveCost( posnew );
  if ( cost > moves ) return -1;

  u->Face( dir );
  u->SetMoves( moves - cost );

  if ( mv->Enabled() &&
       (mv->HexVisible(pos) || mv->HexVisible(posnew)) )
    mwin->MoveHex( u->Image(), unit_set->GetTiles(),
                   pos, posnew, ANIM_SPEED_UNIT );

  u->SetPosition( posnew.x, posnew.y );
  if ( !u->IsDummy() && history ) history->RecordMoveEvent( *u, dir );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CreateUnit
// DESCRIPTION: Create a new unit for the currently active player.
// PARAMETERS : type   - unit type
//              hex    - position on map
//              player - proud owner of a freshman unit
// RETURNS    : unit or NULL on error
////////////////////////////////////////////////////////////////////////

Unit *Game::CreateUnit( const UnitType *type, const Point &hex, Player *player ) {
  // find an unused unit ID; start from the back of the range to avoid
  // potential conflicts with IDs of destroyed units
  unsigned short id = 32000;
  Unit *u = static_cast<Unit *>( units.Head() );
  while ( u ) {
    if ( u->ID() == id ) {
      --id;
      u = static_cast<Unit *>( units.Head() );
    } else u = static_cast<Unit *>( u->Next() );
  }

  if ( type->Flags() & U_TRANSPORT )
    u = new Transport( type, player, id, hex.x, hex.y );
  else u = new Unit( type, player, id, hex.x, hex.y );
  if ( u ) {
    u->SetFlags( U_DONE );   // can't move on the first turn
    units.AddTail( u );
    map->SetUnit( u, hex );
  }
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitTargets
// DESCRIPTION: Find out whether the unit still has things to do, i.e.
//              enemies in range, mines to clear etc.
// PARAMETERS : u - unit to check for
// RETURNS    : TRUE if there are options left, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Game::UnitTargets( Unit *u ) const {
  Unit *tg = static_cast<Unit *>( units.Head() );
  bool rc = false;
  while ( tg && !rc ) {
    if ( u->CanHit( tg ) ) rc = true;
    else if ( u->IsMinesweeper() && tg->IsMine() &&
              !tg->IsSheltered() && NextTo( u->Position(), tg->Position() ) ) {
      bool enemy = false;

      if ( tg->Owner() != u->Owner() ) {
        Point adj[6];
        map->GetNeighbors( tg->Position(), adj );
        for ( int i = NORTH; (i <= NORTHWEST) && !enemy; ++i ) {
          if ( adj[i].x != -1 ) {
            Unit *e = map->GetUnit( adj[i] );
            if ( e && (e->Owner() == tg->Owner()) && !e->IsMine() ) enemy = true;
          }
        }
      }

      if ( !enemy ) rc = true;
    }
    tg = static_cast<Unit *>( tg->Next() );
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SelectUnit
// DESCRIPTION: Select a unit to move/attack.
// PARAMETERS : u - unit to be selected
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::SelectUnit( Unit *u ) {
  if ( u->IsReady() ) {
    MapView *mv = mwin->GetMapView();

    if ( unit ) DeselectUnit( false );

    unit = u;
    player->SetMode( MODE_BUSY );

    if ( mv->Enabled() ) {
      Audio::PlaySfx( Audio::SND_GAM_SELECT, 0 );
      mv->EnableFog();
      shader->Clear();
      shader->ShadeMap( unit );
      mwin->Draw();
      mwin->Show();

      mv->SetCursorImage( IMG_CURSOR_SELECT );
      SetCursor( u->Position() );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::DeselectUnit
// DESCRIPTION: Deselect the currently active unit.
// PARAMETERS : update - whether to update the display or not (default
//                       value is "true")
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::DeselectUnit( bool display /* = true */ ) {
  MapView *mv = mwin->GetMapView();
  player->SetMode( MODE_IDLE );

  if ( mv->Enabled() ) {
    mv->SetCursorImage( IMG_CURSOR_IDLE );

    mv->DisableFog();
    if ( display ) {
      mwin->Draw();
      mwin->Show();

      SetCursor( unit->Position() );
    }
  }
  unit = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SelectNextUnit
// DESCRIPTION: Select the current player's next available unit.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::SelectNextUnit( void ) {
  Unit *start, *u;

  if ( unit ) start = unit;
  else start = static_cast<Unit *>( units.Head() );

  u = static_cast<Unit *>( units.NextNode(start) );

  while ( u && (u != start) ) {

    if ( (u->Owner() == player) && u->IsReady() && !u->IsSheltered() ) {
      // we ignore SHELTERED units because it wouldn't be obvious which
      // of the units inside the transport was selected
      SelectUnit( u );
      break;
    }

    u = static_cast<Unit *>( units.NextNode(u) );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::EnterSpecialMode
// DESCRIPTION: Activate one of the special game modes (for pioneers,
//              mine-sweepers, or depot builders)
// PARAMETERS : mode - mode to enter (see player.h for definitions)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::EnterSpecialMode( unsigned char mode ) {
  MapView *mv = mwin->GetMapView();

  player->SetMode( mode );

  if ( mv->Enabled() ) {
    MinesweeperShader mss( map, units, mv->GetFogBuffer() );

    mv->EnableFog();
    mss.Clear();
    mss.ShadeMap( unit );
    mwin->Draw();
    mwin->Show();

    mv->SetCursorImage( IMG_CURSOR_SPECIAL );
    SetCursor( unit->Position() );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ClearMine
// DESCRIPTION: Try to move a mine from the map into an adjacent mine
//              sweeper unit.
// PARAMETERS : sweeper - mine sweeper unit (currently selected unit)
//              mine    - mine to be cleared
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::ClearMine( Transport *sweeper, Unit *mine ) {
  const Point &spos = sweeper->Position();
  bool allow;

  if ( mine->Owner() != sweeper->Owner() ) {
    Player *mowner = mine->Owner();
    mine->SetOwner( sweeper->Owner() );
    allow = sweeper->Allow( mine );
    if ( !allow ) mine->SetOwner( mowner );
  } else allow = sweeper->Allow( mine );

  if ( allow ) {
    mine->SetMoves( MCOST_MIN );
    MoveUnit( mine, spos );

    if ( !UnitTargets(sweeper) ) DeselectUnit();

  } else new NoteWindow( "Error", "No space for mine.", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::HandleEvent
// DESCRIPTION: Handle key and mouse button events special to the map
//              window.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status Game::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = GUI_OK;
  MapView *mv = mwin->GetMapView();

  // check for keyboard commands
  if ( event.type == SDL_KEYDOWN ) {

    switch ( event.key.keysym.sym ) {
    case SDLK_KP1: case SDLK_KP2: case SDLK_KP3:
    case SDLK_KP4: case SDLK_KP6: case SDLK_KP7:
    case SDLK_KP8: case SDLK_KP9:
    case SDLK_LEFT: case SDLK_RIGHT: case SDLK_UP: case SDLK_DOWN:
      MoveCommand( event.key.keysym.sym );
      break;
    case SDLK_SPACE:      // select the unit underneath the cursor
      SelectCommand();
      break;
    case SDLK_c: {        // show transport or building contents
      MapObject *mo = map->GetMapObject( mv->Cursor() );
      if ( mo ) {
        if ( mo->Type() == MO_BUILDING ) 
          ContainerContent( static_cast<UnitContainer *>(
                            static_cast<Building *>(mo)) );
        else if ( static_cast<Unit *>(mo)->IsTransport() )
          ContainerContent( static_cast<UnitContainer *>(
                            static_cast<Transport *>(mo)) );
      }
      break; }
    case SDLK_e:          // end turn
      rc = EndTurn();
      break;
    case SDLK_i: {        // display unit information
      Unit *u = map->GetUnit( mv->Cursor() );
      if ( u ) UnitInfo( u );
      break; }
    case SDLK_m:          // open tactical map window
      new TacticalWindow( mv, view );
      break;
    case SDLK_n:          // select next unit
      SelectNextUnit();
      break;
    case SDLK_s:          // sweep mine
      if ( unit && unit->IsMinesweeper() && !unit->IsSheltered() )
        EnterSpecialMode( MODE_SWEEP );
      break;
    case SDLK_q:         // quit
      Quit();
      break;

    case SDLK_ESCAPE:
      if ( unit ) {
        DeselectUnit();
        break;
      }                  // fall through...
    case SDLK_g:         // game menu
      GameMenu();
      break;
    default:
      break;
    }
  } else if ( event.type == SDL_MOUSEBUTTONDOWN ) {
    Point pos;
    if ( event.button.button == SDL_BUTTON_LEFT ) {
      if ( !mv->Pixel2Hex( event.button.x - mwin->x, event.button.y - mwin->y, pos ) ) {
        if ( pos == mv->Cursor() ) SelectCommand();
        else SetCursor( pos );
      }
    } else if ( !mv->Pixel2Hex( event.button.x - mwin->x, event.button.y - mwin->y, pos ) ) {
      Unit *u = map->GetUnit( pos );
      if ( event.button.button == SDL_BUTTON_RIGHT ) {
        if ( u ) UnitMenu( u );
        else GameMenu();
      } else {		// middle mouse button
        if ( u ) {
          if ( u->IsTransport() ) ContainerContent( static_cast<Transport *>(u) );
        } else if ( map->GetBuilding( pos ) )
          ContainerContent( map->GetBuilding( pos ) );
      }
    } else GameMenu();
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveCommand
// DESCRIPTION: Got a move command from the user. See what he wants to
//              do. Move the cursor or a selected unit.
// PARAMETERS : key - the key code used to give the order
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::MoveCommand( int key ) {
  Direction dir;

  switch ( key ) {
  case SDLK_KP1:  dir = SOUTHWEST; break;
  case SDLK_DOWN:
  case SDLK_KP2:  dir = SOUTH;     break;
  case SDLK_KP3:  dir = SOUTHEAST; break;
  case SDLK_KP7:  dir = NORTHWEST; break;
  case SDLK_UP:
  case SDLK_KP8:  dir = NORTH;     break;
  case SDLK_KP9:  dir = NORTHEAST; break;
  case SDLK_LEFT:
  case SDLK_KP4:  dir = WEST;      break;
  case SDLK_RIGHT:
  case SDLK_KP6:  dir = EAST;      break;
  default: return;
  }

  Point cursor = mwin->MoveCursor( dir );
  if ( cursor != mwin->GetMapView()->Cursor() ) SetCursor( cursor );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SelectCommand
// DESCRIPTION: Got a select command from the user. See what he wants to
//              do. Select/deselect a unit, enter a building, or attack
//              an enemy unit.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::SelectCommand( void ) {
  MapView *mv = mwin->GetMapView();
  Point cursor = mv->Cursor();

  Unit *u = map->GetUnit( cursor );

  if ( player->Mode() == MODE_BUSY ) {
    if ( u ) {
      if ( u->Owner() == player ) {
        if ( cursor == unit->Position() ) DeselectUnit();
        else if ( shader->GetStep(cursor) != -1 )
          MoveUnit( unit, cursor );    // try to move into transport
        else SelectUnit( u );
      } else if ( unit->CanHit( u ) ) {   // attack the unit
        RegisterCombat( unit, u );
        DeselectUnit();
      }
    } else if ( unit->Position() == cursor ) DeselectUnit();
    else MoveUnit( unit, cursor );  // try to move there
  } else if ( player->Mode() == MODE_IDLE ) {
    if ( u && (u->Owner() == player) ) SelectUnit( u );
    else {
      Building *b = map->GetBuilding( cursor );
      if ( b ) ContainerContent( b );
    }
  } else if ( player->Mode() == MODE_SWEEP ) {
    if ( (shader->GetStep( cursor ) != -1) &&
         (cursor != unit->Position()) )
      ClearMine( static_cast<Transport *>(unit), u );
    else DeselectUnit();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ContainerContent
// DESCRIPTION: Open a window to display the content of a transport or
//              building.
// PARAMETERS : c - container to look into
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::ContainerContent( UnitContainer *c ) {
  const char *name;

  if ( c->CType() == MO_UNIT ) name = static_cast<Transport *>(c)->Name();
  else name = static_cast<Building *>(c)->Name();

  if ( unit ) DeselectUnit();

  if ( (c->Owner() == player) || !c->Owner() ) new ContainerWindow( c, view );
  else new NoteWindow( name, "Access denied!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitInfo
// DESCRIPTION: Display a window with information about a unit. If the
//              current player is not authorised to peek at the unit
//              specifications (e.g. because it's a hostile unit) the
//              information request will fail.
// PARAMETERS : unit - unit to show information about
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::UnitInfo( Unit *unit ) {
  if ( unit->Owner() == player )
    new UnitInfoWindow( unit->Type(), mwin, view );
  else new NoteWindow( unit->Name(), "Access denied!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowLevelInfo
// DESCRIPTION: Display level information supplied by the creator, if
//              any. If no info was supplied, say so.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::ShowLevelInfo( void ) const {
  const char *msg;

  if ( level_info != -1 ) msg = messages[level_info];
  else msg = "No level information available!";

  new NoteWindow( "Level Information", msg, WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowBriefing
// DESCRIPTION: Display a window with the mission objectives for the
//              current player. If the mission creator did not supply a
//              briefing, pop up an error.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::ShowBriefing( void ) const {
  if ( player->Briefing() != -1 )
    new MessageWindow( player->Name(), messages[player->Briefing()], view );
  else new NoteWindow( player->Name(), "No mission briefing available!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowDebriefing
// DESCRIPTION: When the mission is over, display a message telling
//              the players whether they won or lost and optionally
//              return to the main menu.
// PARAMETERS : player  - player to show debriefing for
//              restart - whether to return to the main menu or load
//                        the next map
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::ShowDebriefing( Player *player, bool restart ) {

  if ( player->Type() == HUMAN ) {
    Player *winner = NULL;
    bool draw = false;
    const char *msg;

    if ( players[PLAYER_ONE]->Success( 0 ) >= 100 ) winner = players[PLAYER_ONE];
    if ( players[PLAYER_TWO]->Success( 0 ) >= 100 ) {
      if ( winner ) draw = true;
      else winner = players[PLAYER_TWO];
    }

    if ( draw ) msg = "This game was a draw!";
    else if ( player == winner ) msg = "Congratulations! You have won!";
    else msg = "You have been defeated!";

    NoteWindow *note = new NoteWindow( "Mission Debriefing", msg, WIN_CENTER, view );

    if ( restart ) {
      if ( (next_map == -1) || draw || (winner->Type() != HUMAN) ) note->SetButtonID( 0, GUI_RESTART );
      else {
        note->EventLoop();
        view->CloseWindow( note );

        int err = SwitchMap( messages[next_map] );
        if ( err == -1 ) {
          note = new NoteWindow( "Error", "Requested map not available!", 0, view );
          note->SetButtonID( 0, GUI_RESTART );
        } else {
          if ( GetLevelPassword() ) {
            char msgbuf[36] = "Access code for this map: ";
            strcat( msgbuf, GetLevelPassword() );
            note = new NoteWindow( last_file_name.c_str(), msgbuf, WIN_CENTER, view );
            note->EventLoop();
            view->CloseWindow( note );
          }
          StartTurn();
        }
      }
    } else {
      note->EventLoop();
      view->CloseWindow( note );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GameMenu
// DESCRIPTION: Pop up a MenuWindow with general game options like
//              "End Turn" or "Quit".
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::GameMenu( void ) {
  MenuWindow *menu = new MenuWindow( PROGRAMNAME, this, view );

  menu->AddItem( 0, G_BUTTON_END_TURN, 0, "End Turn", 'e' );
  menu->AddItem( 0, G_BUTTON_MAP,      0, "Map", 'm' );
  menu->AddItem( 0, G_BUTTON_BRIEFING, 0, "Briefing", 'b' );
  menu->AddBar( 0 );
  menu->AddItem( 0, G_BUTTON_LEV_INFO, 0, "Level Info", 'l' );
  menu->AddMenu( 0, 0, "Options", 'o' );
  menu->AddItem( 1, G_BUTTON_GENERAL_OPTIONS, 0, "General", 'g' );
  menu->AddItem( 1, G_BUTTON_VIDEO_OPTIONS, 0, "Video", 'v' );
#ifndef DISABLE_SOUND
  menu->AddItem( 1, G_BUTTON_SOUND_OPTIONS, 0, "Sound", 's' );
#endif
  menu->AddBar( 0 );
  menu->AddItem( 0, G_BUTTON_SAVE, 
                 (flags & GI_PBEM) ? WIDGET_DISABLED : 0,
                 "Save...", 's' );
  menu->AddItem( 0, G_BUTTON_ABORT,    0, "Abort...", 'a' );
  menu->AddItem( 0, G_BUTTON_QUIT,     0, "Quit...", 'q' );

  menu->Layout();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitMenu
// DESCRIPTION: Pop up a MenuWindow with possible actions for a selected
//              unit.
// PARAMETERS : u - unit to open window for
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::UnitMenu( Unit *u ) {
  bool transport = (u->IsTransport() && (u->Owner() == player));
  bool sweeper = (u->IsMinesweeper() && (u == unit));

  if ( transport || sweeper ) {
    MenuWindow *menu = new MenuWindow( u->Name(), this, view );

    menu->AddItem( 0, G_BUTTON_UNIT_INFO, 0, "Info", 'i' );

    if ( transport )
      menu->AddItem( 0, G_BUTTON_UNIT_CONTENT, 0, "Content", 'c' );

    if ( sweeper )
      menu->AddItem( 0, G_BUTTON_UNIT_SWEEP,
                     (u->IsReady() ? 0 : WIDGET_DISABLED), "Sweep", 's' );

    menu->Layout();
    g_tmp_prv_unit = u;
  } else UnitInfo( u );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::RegisterCombat
// DESCRIPTION: File a new combat. Record it for historical replay.
// PARAMETERS : att - attacking unit
//              def - defending unit
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Game::RegisterCombat( Unit *att, Unit *def ) {
  combat.AddTail( new Combat( att, def ) );
  att->Attack( def );
  if ( history ) history->RecordAttackEvent( *att, *def );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::WidgetActivated
// DESCRIPTION: The WidgetActivated() method gets called whenever a
//              widget from the game menu or another associated window
//              (e.g. password confirmation) is activated.
// PARAMETERS : button - pointer to the widget that called the function
//              win    - pointer to the window the widget belongs to
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status Game::WidgetActivated( Widget *button, Window *win ) {
  GUI_Status rc = GUI_OK;

  switch ( button->ID() ) {
  case G_BUTTON_END_TURN:
    view->CloseWindow( win );
    rc = EndTurn();
    break;
  case G_BUTTON_MAP:
    view->CloseWindow( win );
    new TacticalWindow( mwin->GetMapView(), view );
    break;
  case G_BUTTON_BRIEFING:
    view->CloseWindow( win );
    ShowBriefing();
    break;
  case G_BUTTON_SAVE:
    view->CloseWindow( win );
    Save( NULL );
    break;
  case G_BUTTON_LEV_INFO:
    view->CloseWindow( win );
    ShowLevelInfo();
    break;
  case G_BUTTON_GENERAL_OPTIONS:
    static_cast<MenuWindow *>(win)->CloseParent();
    view->CloseWindow( win );
    new GeneralOptionsWindow( *mwin->GetMapView(), view );
    break;
  case G_BUTTON_VIDEO_OPTIONS:
    static_cast<MenuWindow *>(win)->CloseParent();
    view->CloseWindow( win );
    new VideoOptionsWindow( view );
    break;
#ifndef DISABLE_SOUND
  case G_BUTTON_SOUND_OPTIONS:
    static_cast<MenuWindow *>(win)->CloseParent();
    view->CloseWindow( win );
    new SoundOptionsWindow( view );
    break;
#endif
  case G_BUTTON_ABORT: {
    view->CloseWindow( win );
    Audio::PlaySfx( Audio::SND_GUI_ASK, 0 );
    DialogWindow *req = new DialogWindow( NULL, "Return to main menu?",
                        "yYes|nNo", 1, 0, view );
    req->SetButtonID( 0, GUI_RESTART );
    break; }
  case G_BUTTON_QUIT:
    view->CloseWindow( win );
    Quit();
    break;

  case G_BUTTON_UNIT_INFO:
    view->CloseWindow( win );
    UnitInfo( g_tmp_prv_unit );
    break;
  case G_BUTTON_UNIT_CONTENT:
    view->CloseWindow( win );
    ContainerContent( static_cast<Transport *>(g_tmp_prv_unit) );
    break;
  case G_BUTTON_UNIT_SWEEP:
    view->CloseWindow( win );
    EnterSpecialMode( MODE_SWEEP );
    break;
  }

  return rc;
}

