/* fitness.word-count.c - fitness score on words found in dictionary
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"

#define MAX_WORD_LENGTH 12
#define MIN_WORD_LENGTH 3
#ifndef MIN
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif

/* Global variables */
static const char DICTIONARY_NAME[] = DATA_DIR "/word-list.dat";
static char **dict_std;
static int number_of_words;

/* Fn Declarations */
int load_dict(const char *filename, char ***dict);
int dictionary_lookup(char **dict, char *word);

/* Plugin Interface */
/* ---------------- */

const char name[] = "fitness.word-count";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "Fitness score based on words found in text matching a dictionary.";
const int interface_version = 1;
const int plugin_type = PLUGIN_FITNESS;

int boot(void) {
    number_of_words = load_dict(DICTIONARY_NAME, &dict_std);
    if (!dict_std) 
	return PLUGIN_BOOT_FAIL;
    return PLUGIN_BOOT_OK;
}

float fitness(char *text) {
    int pos, word_size, length, match_count = 0;
    char buf[BUFFER_SIZE];
    length = strlen(text);
    for(pos = 0; pos < length; pos++) {
	word_size = MAX_WORD_LENGTH;
	while (word_size >= MIN_WORD_LENGTH) {
	    if (pos + word_size <= length) {
		strncpy(buf, (text + pos), word_size);
		buf[word_size] = '\0';
		if (dictionary_lookup(dict_std, buf)) {
		    match_count += word_size;
		    pos += word_size - 1; /* Advance to next word */
		    continue;
		}
	    }
	    word_size--;
	}
    }
    if (match_count == 0)
	match_count++;
    return ((float) length / (float) match_count);
}


int load_dict(const char *filename, char ***dict) {
    FILE *inf;
    char buf[BUFFER_SIZE];
    int words = 0, number_of_words;
        
    inf = fopen(filename, "r");
    if (!inf) {
	g_warning("Error opening dictionary file: %s", filename);
	*dict = NULL;
	return 0;
    } 
    
    /* Count words in dictionary */
    while(fscanf(inf, "%s", buf) != EOF)
	words++;
    rewind(inf);

    *dict = malloc(sizeof(char *) * words);
    number_of_words = words;
    words = 0;

    while(fscanf(inf, "%s", buf) != EOF) {
	(*dict)[words] = malloc(sizeof(char) * (strlen(buf) + 1));
	strcpy((*dict)[words], buf);
	words++;
    }
    if (words != number_of_words) {
	g_warning("Error reading dictionary file: %s", filename);
    }
    return number_of_words;
}

int dictionary_lookup(char **dict, char *word) {
    int max, min = 0, c, p;
    max = number_of_words - 1;
    while (max - min > 1) {
	p = (max + min) / 2;
	c = strcmp(word, dict[p]);
	if (c > 0) 
	    min = p;
	else if (c < 0)
	    max = p;
	else return TRUE;
    }
    return FALSE;

}

int main(void) {
    char buf[BUFFER_SIZE];
    boot();
    while(-1) {
	printf("Word: ");
	scanf("%s", buf);
	printf("%f\n", fitness(buf));
	
    }
}
