/* plugins.c - plugin handling code
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* External variables */
extern char *text;

/* Global variables */
plugin_data *plugin_list = NULL;
plugin_data *current_plugin; /* The current (mode) plugin */
int number_of_plugins = 0;

/* Null Widget for default start */
static void null_init(void) {;}
static int null_boot(void) {return PLUGIN_BOOT_OK;}
static GtkWidget *null_widget(char *text) {return gtk_hbox_new(TRUE, 0);}
plugin_data null_plugin = {"", "", "", "", INTERFACE_VERSION, PLUGIN_MODE, 
                           null_boot, null_init, null_init, null_widget, (char *(*)(char *)) g_strdup, null_init, NULL};

/* Load a symbol from the plugin into the identically named element */
/* of a plugin_data structure casting it appropriately along the way. */
/* If an error is encountered, the loop is continued to the next directory entry */
#define LOAD_SYMBOL(SYMBOL, CAST)     if(!g_module_symbol(plugin, #SYMBOL, plugin_symbol)) { \
                         	         g_warning("Error loading plugin: %s - '" #SYMBOL "' not found", d->d_name); \
                                         free(new_plugin); \
				         continue; \
                                      } \
                                      new_plugin->SYMBOL = CAST (*plugin_symbol); \

/* Initial finding and 'parsing' plugins */
void load_plugins(void) {
    DIR *plugin_dir;
    struct dirent *d;
    GModule *plugin;
    char buf[1024]; 
    plugin_data *new_plugin;  /* LOAD_SYMBOL uses this to build up a new plugin record */
    gpointer *plugin_symbol;  /* temp var for LOAD_SYMBOL */

    plugin_symbol = malloc(sizeof(gpointer));

    if (!g_module_supported())
	g_error("Plugins not supported on this platform.");

    if (!(plugin_dir = opendir("plugins")))
	g_error("Could not open plugin directory - %s", PLUGIN_DIR);

    /* Loop over each file in the plugin directory */
    while ((d = readdir(plugin_dir))) {

	if (d->d_name[0] == '.') /* Skip dotfiles, dirup etc. */
	    continue;

	sprintf(buf, "%s%c%s", PLUGIN_DIR, G_DIR_SEPARATOR, d->d_name);

	if (!(plugin = g_module_open(buf, 0))) {
	    g_warning("Error loading plugin: %s", g_module_error());
	    continue;
	}

	new_plugin = malloc(sizeof(plugin_data));

	/*          Symbol              Type to cast into (or dereference)*/
        /*          ------              ----------------- */
	LOAD_SYMBOL(plugin_type,        * (int *)                );
	LOAD_SYMBOL(name,               (char *)                 );
	LOAD_SYMBOL(version,            (char *)                 );
	LOAD_SYMBOL(author,             (char *)                 );
	LOAD_SYMBOL(description,        (char *)                 );
	LOAD_SYMBOL(interface_version,  * (int *)                );
	LOAD_SYMBOL(make_widget,        (GtkWidget *(*)(char *)) );
	LOAD_SYMBOL(boot,               (int (*)())              );

	/* Symbols needed only by mode plugins */
      if (new_plugin->plugin_type == PLUGIN_MODE) {
	LOAD_SYMBOL(init_plugin,        (void (*)())             );
	LOAD_SYMBOL(deactivate_plugin,  (void (*)())             );
	LOAD_SYMBOL(transform,          (char *(*)(char *))      );
	LOAD_SYMBOL(reset,              (void (*)())             );
      }
	/* Check for incompatible version */ 
	if (new_plugin->interface_version != INTERFACE_VERSION)
	    g_warning("%s - plugin uses newer interface than Crank.", d->d_name);
	
	/* Run the plugin's boot function */
	if(new_plugin->boot() == PLUGIN_BOOT_FAIL) {
	    g_warning("Error loading plugin: %s - plugin boot failed.", d->d_name);
	    continue;
	}
	
	/* Add the plugin to the list */
	new_plugin->next = plugin_list;
	plugin_list = new_plugin;
	
	g_message("Plugin loaded: %s v%s", new_plugin->name, new_plugin->version);
	number_of_plugins++;
    }

    if (!number_of_plugins)
	g_warning("Warning: No valid plugins found.");
    else
	g_message("%d plugin%s loaded.", number_of_plugins, (number_of_plugins == 1) ? "" : "s");

    free(plugin_symbol);
}

/* Return source text (should this be strcpy'd?) */
char *get_text(void) {
    return text;
}
