/* monoalphabetic.hillclimb-cracker.c - automatic cracker using n-gram stats
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"
#include "common.statistics.n-grams.h"

/* Global data */
static key global_key;
static GtkWidget *global_key_labels[26];  /* Pointers to the labels for the global key */
static int abort_computation = FALSE;     /* Whether the user has signalled to abort a computation */ 
static float *slft_std;
static float *bift_std;
static float *trift_std;
static stats *text_stats;

/* fn decls */
static void do_break(void);
static void cb_break(GtkWidget *widget, gpointer gdata);

#define TRY_MAX 1000
#define MAX_SEARCH_DEPTH 3

/* Plugin Interface */
/* ---------------- */

const char name[] = "monoalphabetic.hillclimb-cracker";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "An automatic cracker using randomised hillclimbing for monoalphabetic ciphers, using a fitness function derived from the error between the encountered n-gram frequencies, and the expected for a standard language.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;

int boot(void) {
    /* Load the standard frequency tables */
    /* Should use global resource when implemented */
    slft_std = load_slft_std(DEFAULT_SLFT);
    bift_std = load_bift_std(DEFAULT_BIFT);
    trift_std = load_trift_std(DEFAULT_TRIFT);
    key_identity(&global_key);
    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *plugin_controls;
    GtkWidget *key_display;
    GtkWidget *break_button;

    plugin_controls = gtk_hbox_new(FALSE, 0); 
    
    key_display = make_key_display(&global_key, global_key_labels);
    gtk_box_pack_start(GTK_BOX(plugin_controls), key_display, FALSE, TRUE, 0);
    gtk_widget_show(key_display);

    break_button = gtk_button_new_with_label("Break");
    gtk_signal_connect(GTK_OBJECT(break_button), "clicked", GTK_SIGNAL_FUNC(cb_break), NULL);
    gtk_box_pack_start(GTK_BOX(plugin_controls), break_button, FALSE, TRUE, 0);
    gtk_widget_show(break_button);
    return plugin_controls;
}

char *transform(char *text) {
    update_key_labels(&global_key, global_key_labels);
    return apply_key_text(&global_key, text);
}

void reset(void) {
    key_identity(&global_key);
}

/* Plugin implementation */
/* --------------------- */

static void cb_break(GtkWidget *widget, gpointer gdata) {
    /* Ensures bijective key - do_break should be able to handle otherwise...*/
    /* Redundant until global resource control is put in */
    key_invert(&global_key); key_complete(&global_key);
    display_text();

    do_break();
} 

static gint cb_abort(GtkWidget *widget, gpointer gdata) {
    abort_computation = TRUE;
    while (gtk_grab_get_current())
	gtk_grab_remove(gtk_grab_get_current());
    return (FALSE);
}

static void make_break_progress(GtkWidget **dialog_p, GtkWidget **progress_bar_p) {
    GtkWidget *button;
    GtkWidget *message;
    GtkWidget *dialog;
    GtkWidget *progress_bar;
    dialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dialog), "Automatic Cryptanalysis");
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_abort), NULL);
    progress_bar = gtk_progress_bar_new();
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), progress_bar, TRUE, TRUE, 0);
    button = gtk_button_new_with_label("Finish");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
		              (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    message = gtk_label_new("Processing...");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), message, TRUE, TRUE, 0);
    gtk_widget_show(message);
    gtk_widget_show(progress_bar);
    gtk_grab_add(dialog);
    *dialog_p = dialog;
    *progress_bar_p = progress_bar;
}

/* Basic random hill climbing... */
static void do_break(void) {
    GtkWidget *progress_box, *progress_bar;
    key best_key;
    int try_counter = 0, search_depth, swap_count, swap_a, swap_b, reset_search_depth;
    long try_max;
    stats *under_key_stats;
    float best_error, current_error;
    
    abort_computation = 0;

    /* Calculate the stats */
    text_stats = make_stats(get_text(), slft_std, bift_std, trift_std);
    
    /* Check for zero length text */
    if (!text_stats->letter_count) {
	free(text_stats);
	return;
    }

    /* Display the processing progress */
    make_break_progress(&progress_box, &progress_bar);
    gtk_widget_show(progress_box);

    /* Now text_stats contains the frequencies for the untransformed text */
    /* And under text_stats contain the frequencies under the key permutation */
    under_key_stats = transform_stats_with_key(text_stats, &global_key, slft_std, bift_std, trift_std);
    
    /* Init best scores and key to current */
    best_error = current_error = under_key_stats->total_error;
    key_copy(&global_key, &best_key);
    
    search_depth = 1; reset_search_depth = FALSE;

 depth_loop:
    do {

	if (reset_search_depth) {
	    search_depth = 1;
	    reset_search_depth = FALSE;
	}

	try_max = TRY_MAX * (long) pow(26.0, (float) (search_depth - 1));//search_depth * search_depth;
	try_counter = 0;
//	printf("Depth: %d, Best Error: %f Try_max: %ld\n", search_depth, best_error, try_max);

	while (try_counter < try_max && !abort_computation) {

	    try_counter = 0;
	    while (current_error >= best_error && try_counter++ < try_max && !abort_computation) {
		
		/* Retrieve copy of best key and randomly mutate it */
		key_copy(&best_key, &global_key);
		for (swap_count = 1; swap_count <= search_depth; swap_count++) 
		    key_swap(&global_key, swap_a = randnum('A', 'Z'), swap_b = randnum('A', 'Z'));
		
		/* Calculate stats for this new transformation */
		free_stats(under_key_stats);
		under_key_stats = transform_stats_with_key(text_stats, &global_key, slft_std, bift_std, trift_std);
		current_error = under_key_stats->total_error;
		gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), ((float) try_counter) / ((float) try_max));
		yield_to_ui();
	    }
	    
	    /* Update the GUI with progress so far if a successful error reduce has occurred */
	    if (current_error < best_error) {
		display_text();
		yield_to_ui();
		/* Also, reset search depth */
		if (search_depth > 1)
		    reset_search_depth = TRUE;
		best_error = current_error;
		key_copy(&global_key, &best_key);
		goto depth_loop;
	    }
	}
	if (abort_computation)
	  break;
    } while (search_depth++ <= MAX_SEARCH_DEPTH);
	
    /* Clean up */
    if (!abort_computation)
	gtk_widget_destroy(GTK_WIDGET(progress_box));
    if (under_key_stats) 
	free_stats(under_key_stats);

    key_copy(&best_key, &global_key);

    free_stats(text_stats);
    display_text();
}
