/* ui_stats.c - stats displays
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* external variables and functions */
float *slft_std;
float *bift_std;
float *trift_std;

/* callbacks */
void cb_save_slft(GtkWidget *widget, gpointer gdata);
void cb_save_bift(GtkWidget *widget, gpointer gdata);
void cb_save_trift(GtkWidget *widget, gpointer gdata);
void cb_sort_ft_display(GtkWidget *widget, gint col, gpointer data);

static char *slft_display_titles[] = {"Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};
static char *bift_display_titles[] = {"First Letter", "Second Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};
static char *trift_display_titles[] = {"First Letter", "Second Letter", "Third Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};

/* Make a tabular display of slft using a GtkClist */
/* Displays zero values */
GtkWidget *make_slft_display(float *slft) {
    GtkWidget *clist;
    int i;
    char *row_data[3];
    char letter[2], value[30], value2[30];
    
    clist = gtk_clist_new_with_titles(3, slft_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {
	sprintf(letter, "%c", (char) i);
	sprintf(value, "%.8f", slft[i]);
	sprintf(value2, "%.8f", slft_std[i]);
	row_data[0] = letter; row_data[1] = value; row_data[2] = value2;
	gtk_clist_append(GTK_CLIST(clist), row_data);
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Make a tabular display of bift using a GtkClist */
GtkWidget *make_bift_display(float *bift) {
    GtkWidget *clist;
    int i, j;
    char *row_data[4];
    char letter1[2], letter2[2], value[30], value2[30];
    float fvalue;
    clist = gtk_clist_new_with_titles(4, bift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    fvalue = (bift + 26 * i)[j];
	    if (fvalue > 0.0) {
		sprintf(letter1, "%c", (char) i);
		sprintf(letter2, "%c", (char) j);
		sprintf(value, "%.8f", fvalue);
		sprintf(value2, "%.8f", (bift_std + 26 * i)[j]);
		row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = value; row_data[3] = value2;
		gtk_clist_append(GTK_CLIST(clist), row_data);
	    }
	}
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Make a tabular display of trift using a GtkClist */
GtkWidget *make_trift_display(float *trift) {
    GtkWidget *clist;
    float fvalue;
    int i, j, k;
    char *row_data[5];
    char letter1[2], letter2[2], letter3[2], value[30], value2[30];

    clist = gtk_clist_new_with_titles(5, trift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    for (k = 'A'; k <= 'Z'; k++) {
		fvalue = (trift + 26 * 26 * i + 26 * j)[k];
		if (fvalue > 0.0) {
		    sprintf(letter1, "%c", (char) i);
		    sprintf(letter2, "%c", (char) j);
		    sprintf(letter3, "%c", (char) k);
		    sprintf(value, "%.8f", fvalue);
		    sprintf(value2, "%.8f", (trift_std + 26 * 26 * i + 26 * j)[k]);
		    row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = letter3; row_data[3] = value; row_data[4] = value2;
		    gtk_clist_append(GTK_CLIST(clist), row_data);
		}
	    }
	}
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Add a floating point entry to the stats summary table */
void add_table_entry_f(GtkWidget *table, char *name, float value, int pos) {
    char buf[256];
    GtkWidget *label_name, *label_value;
    label_name = gtk_label_new(name);
    gtk_table_attach(GTK_TABLE(table), label_name, 0, 1, pos, pos + 1, 0, 0, 0, 0);
    gtk_widget_show(label_name);

    sprintf(buf, "%f", value);
    label_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_value, 1, 2, pos, pos + 1, 0, 0, 0, 0);
    gtk_widget_show(label_value);
}

/* Add an integer entry to the stats summary table */
void add_table_entry_i(GtkWidget *table, char *name, int value, int pos) {
    char buf[256];
    GtkWidget *label_name, *label_value;
    label_name = gtk_label_new(name);
    gtk_table_attach(GTK_TABLE(table), label_name, 0, 1, pos, pos + 1, 0, 0, 0, 0);
    gtk_widget_show(label_name);

    sprintf(buf, "%d", value);
    label_value = gtk_label_new(buf);
    gtk_table_attach(GTK_TABLE(table), label_value, 1, 2, pos, pos + 1, 0, 0, 0, 0);
    gtk_widget_show(label_value);
}

GtkWidget *make_stats_summary(stats *the_stats) {
    GtkWidget *table;
    GtkWidget *save_button_slft;
    GtkWidget *save_button_bift;
    GtkWidget *save_button_trift;

    table = gtk_table_new(2, 9, FALSE);
    
    save_button_slft = gtk_button_new_with_label("Save as Default Unigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_slft), "pressed", GTK_SIGNAL_FUNC(cb_save_slft), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_slft, 0, 1, 0, 1, 0, 0, 0, 0);
    gtk_widget_show(save_button_slft);

    save_button_bift = gtk_button_new_with_label("Save as Default Bigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_bift), "pressed", GTK_SIGNAL_FUNC(cb_save_bift), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_bift, 0, 1, 1, 2, 0, 0, 0, 0);
    gtk_widget_show(save_button_bift);

    save_button_trift = gtk_button_new_with_label("Save as Default Trigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_trift), "pressed", GTK_SIGNAL_FUNC(cb_save_trift), NULL);
    gtk_table_attach(GTK_TABLE(table), save_button_trift, 0, 1, 2, 3, 0, 0, 0, 0);
    gtk_widget_show(save_button_trift);

    add_table_entry_f(table, "Unigram Error:", the_stats->slft_error, 3);
    add_table_entry_f(table, "Bigram Error:", the_stats->bift_error, 4);
    add_table_entry_f(table, "Trigram Error:", the_stats->trift_error, 5);
    add_table_entry_f(table, "Total Error:", the_stats->total_error, 6);
    add_table_entry_i(table, "Letter Count:", the_stats->letter_count, 7);
    add_table_entry_f(table, "IC:", the_stats->ic, 8);
    add_table_entry_f(table, "Entropy:", the_stats->entropy, 9);
    add_table_entry_f(table, "Efficiency:", the_stats->efficiency, 10);
    add_table_entry_f(table, "Redundancy:", the_stats->redundancy, 11);

    return table;
}
