/* ui_boxes.c - functions for generating pop-up windows
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* Global variables */
extern preprocess_options preprocess_opts;
extern char *text;

/* External funcs and variables */
void cb_preprocess(GtkWidget *widget, gpointer gdata);
gint cb_abort(GtkWidget *widget, gpointer gdata);
void cb_prepro_check_toggle(GtkWidget *widget, gpointer gdata);
void cb_ok_edit_source(GtkWidget *widget, gpointer gdata);
GtkWidget *make_slft_display(float *slft);
GtkWidget *make_bift_display(float *slft);
GtkWidget *make_trift_display(float *trift);
GtkWidget *make_stats_summary(stats *the_stats);

static void cb_close_dialog(GtkWidget *widget, gpointer data) {
    while (gtk_grab_get_current())
	gtk_grab_remove(gtk_grab_get_current());
}

void make_break_progress(GtkWidget **dialog_p, GtkWidget **progress_bar_p) {
    GtkWidget *button;
    GtkWidget *message;
    GtkWidget *dialog;
    GtkWidget *progress_bar;
    dialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dialog), "Automatic Cryptanalysis");
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_abort), NULL);
    progress_bar = gtk_progress_bar_new();
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), progress_bar, TRUE, TRUE, 0);
    button = gtk_button_new_with_label("Stop");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
		              (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    message = gtk_label_new("Processing...");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), message, TRUE, TRUE, 0);
    gtk_widget_show(message);
    gtk_widget_show(progress_bar);
    gtk_grab_add(dialog);
    *dialog_p = dialog;
    *progress_bar_p = progress_bar;
}

/* An info box with text and an OK button */
GtkWidget *make_OK_box(const char *text) {
    GtkWidget *dialog;
    GtkWidget *button;
    GtkWidget *message;
    dialog  = gtk_dialog_new();
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_close_dialog), NULL);
    button = gtk_button_new_with_label("OK");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    message = gtk_label_new(text);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), message, FALSE, FALSE, 0);
    gtk_widget_show(message);
    gtk_grab_add(dialog);
    return dialog;
}

static GtkWidget *add_opt(char *txt, int type, GtkWidget *vbox) {
    GtkWidget *opt;
    opt = gtk_check_button_new_with_label(txt);
    gtk_box_pack_start(GTK_BOX(vbox), opt, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(opt), "toggled", cb_prepro_check_toggle, GINT_TO_POINTER(type));
    gtk_widget_show(opt);
    return opt;
}

/* Source text editing box */
GtkWidget *make_source_edit_dialog(void) {
    GtkWidget *dialog;
    GtkWidget *ok_button;
    GtkWidget *cancel_button;
    GtkWidget *text_area;
    GtkWidget *vscrollbar;
    GtkWidget *text_box;

    dialog  = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dialog), "Edit Source");
    gtk_window_set_policy(GTK_WINDOW(dialog), TRUE, TRUE, FALSE);
    gtk_widget_set_usize(dialog, -1, -1);
    gtk_widget_set_usize(dialog, WINDOW_WIDTH, WINDOW_HEIGHT);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_close_dialog), NULL);

    /* Text area */
    text_box = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), text_box, TRUE, TRUE, 0);
    
    text_area = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(text_area), TRUE);
    gtk_box_pack_start(GTK_BOX(text_box), text_area, TRUE, TRUE, 0);
    
    vscrollbar = gtk_vscrollbar_new (GTK_TEXT(text_area)->vadj);
    gtk_box_pack_start(GTK_BOX(text_box), vscrollbar, FALSE, FALSE, 0);

    gtk_widget_show(vscrollbar);
    gtk_widget_show(text_area);
    gtk_widget_show(text_box);

    gtk_text_insert(GTK_TEXT(text_area), NULL, NULL, NULL, text, strlen(text));

    /* OK / Cancel */
    ok_button = gtk_button_new_with_label("OK");
    cancel_button = gtk_button_new_with_label("Discard Changes");
    gtk_signal_connect_object(GTK_OBJECT(cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(ok_button),
		      "clicked",
		       GTK_SIGNAL_FUNC(cb_ok_edit_source),
		       text_area);
    gtk_signal_connect_object(GTK_OBJECT(ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), ok_button, TRUE, TRUE, 5);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), cancel_button, TRUE, TRUE, 5);
    gtk_widget_show(ok_button);
    gtk_widget_show(cancel_button);
    gtk_grab_add(dialog);
    return dialog; 
}

/* Source text preprocessing dialog box */
GtkWidget *make_preprocess_dialog(void) {
    GtkWidget *dialog;
    GtkWidget *ok_button;
    GtkWidget *cancel_button;
    GtkWidget *header;

    preprocess_opts.remove_whitespace = FALSE;
    preprocess_opts.remove_punctuation = FALSE;
    preprocess_opts.remove_numbers = FALSE;

    dialog  = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dialog), "Preprocess");
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_close_dialog), NULL);

    ok_button = gtk_button_new_with_label("OK");
    cancel_button = gtk_button_new_with_label("Cancel");
    gtk_signal_connect_object(GTK_OBJECT(cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_signal_connect(GTK_OBJECT(ok_button),
		      "clicked",
		       GTK_SIGNAL_FUNC(cb_preprocess),
		       NULL);
    gtk_signal_connect_object(GTK_OBJECT(ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), ok_button, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), cancel_button, TRUE, TRUE, 0);
    gtk_widget_show(ok_button);
    gtk_widget_show(cancel_button);

    header = gtk_label_new("Source Text Preprocessing Options");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), header, TRUE, TRUE, 0);
    gtk_widget_show(header);

    add_opt("Remove whitespace", PREPRO_WHITESPACE, GTK_DIALOG(dialog)->vbox);
    add_opt("Remove punctuation", PREPRO_PUNCTUATION, GTK_DIALOG(dialog)->vbox);
    add_opt("Remove numbers", PREPRO_NUMBERS, GTK_DIALOG(dialog)->vbox);
    
    gtk_grab_add(dialog);

    return dialog; 
}

GtkWidget *make_stats_display_window(stats *s) {
    float *slft = s->slft;
    float *bift = s->bift;
    float *trift = s->trift;
    GtkWidget *dialog;
    GtkWidget *summary_page = make_stats_summary(s);
    GtkWidget *clist_slft = make_slft_display(slft);
    GtkWidget *clist_bift = make_bift_display(bift);
    GtkWidget *clist_trift = make_trift_display(trift);
    GtkWidget *window_main;
    GtkWidget *window_slft;
    GtkWidget *window_bift;
    GtkWidget *window_trift;
    GtkWidget *button;
    GtkWidget *notebook;
    window_main = gtk_scrolled_window_new(NULL, NULL);
    window_slft = gtk_scrolled_window_new(NULL, NULL);
    window_bift = gtk_scrolled_window_new(NULL, NULL);
    window_trift = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_main),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_slft),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);    
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_bift),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_trift), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_main), summary_page);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_slft), clist_slft);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_bift), clist_bift);    
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_trift), clist_trift);    
    gtk_widget_show(window_main);
    gtk_widget_show(window_slft);
    gtk_widget_show(window_bift);
    gtk_widget_show(window_trift);
    dialog  = gtk_dialog_new();
    gtk_widget_set_usize(dialog, 440, 500);
    
    gtk_window_set_title(GTK_WINDOW(dialog), "Text Statistics");
    button = gtk_button_new_with_label("Dismiss");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    gtk_widget_show(button);
    notebook = gtk_notebook_new();
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_main, gtk_label_new("Summary"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_slft, gtk_label_new("Unigrams"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_bift, gtk_label_new("Bigrams"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_trift, gtk_label_new("Trigrams"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show(clist_slft);
    gtk_widget_show(clist_bift);
    gtk_widget_show(clist_trift);
    gtk_widget_show(summary_page);
    gtk_widget_show(notebook);
    return dialog;
}
