/* autocrack.c - Routines to automatically break a cipher
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* Global Variables */
int abort_computation = 0;

/* External functions and variables */
extern key global_key;
extern char *text;
extern stats *text_stats;
GtkWidget *textarea;
void update_key_labels(void);
void apply_global_key_text(void);
void free_stats(stats *the_stats);
stats *make_stats(char *text);
int randnum (int a, int b);
void key_swap(key *k, int index1, int index2);
void key_copy(key *k1, key *k2);
stats *transform_stats_with_key(stats *s, key *key);
void init_global_key(void);
void make_break_progress(GtkWidget **dialog, GtkWidget **progress_bar);
void yield_to_ui(void);

#define TRY_MAX 1000
#define MAX_SEARCH_DEPTH 3

/* Basic random hill climbing... */
void do_break(void) {
    GtkWidget *progress_box, *progress_bar;
    key best_key;
    int try_counter = 0, search_depth, swap_count, swap_a, swap_b, reset_search_depth;
    long try_max;
    char *visible_text;
    stats *under_key_stats;
    float best_error, current_error;
    
    abort_computation = 0;

    /* Calculate the stats */
    if (text_stats)
	free_stats(text_stats);
	text_stats = make_stats(text);

    /* Check for zero length text */
    if (!text_stats->letter_count)
      return;

    /* Display the processing progress */
    make_break_progress(&progress_box, &progress_bar);
    gtk_widget_show(progress_box);

    /* Now text_stats contains the frequencies for the untransformed text */
    /* And under text_stats contain the frequencies under the key permutation */
    under_key_stats = transform_stats_with_key(text_stats, &global_key);
    
    /* Init best scores and key to current */
    best_error = current_error = under_key_stats->total_error;
    key_copy(&global_key, &best_key);
    
    search_depth = 1; reset_search_depth = FALSE;

 depth_loop:
    do {

	if (reset_search_depth) {
	    search_depth = 1;
	    reset_search_depth = FALSE;
	}

	try_max = TRY_MAX * (long) pow(26.0, (float) (search_depth - 1));//search_depth * search_depth;
	try_counter = 0;
//	printf("Depth: %d, Best Error: %f Try_max: %ld\n", search_depth, best_error, try_max);

	while (try_counter < try_max && !abort_computation) {

	    try_counter = 0;
	    while (current_error >= best_error && try_counter++ < try_max && !abort_computation) {
		
		/* Retrieve copy of best key and randomly mutate it */
		key_copy(&best_key, &global_key);
		for (swap_count = 1; swap_count <= search_depth; swap_count++) 
		    key_swap(&global_key, swap_a = randnum('A', 'Z'), swap_b = randnum('A', 'Z'));
		
		/* Calculate stats for this new transformation */
		free_stats(under_key_stats);
		under_key_stats = transform_stats_with_key(text_stats, &global_key);
		current_error = under_key_stats->total_error;
		gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), ((float) try_counter) / ((float) try_max));
		yield_to_ui();
	    }
	    
	    /* Update the GUI with progress so far if a successful error reduce has occurred */
	    if (current_error < best_error) {
		update_key_labels();
		apply_global_key_text();
		yield_to_ui();
		/* Also, reset search depth */
		if (search_depth > 1)
		    reset_search_depth = TRUE;
		best_error = current_error;
		key_copy(&global_key, &best_key);
		goto depth_loop;
	    }
	}
	if (abort_computation)
	  break;
    } while (search_depth++ <= MAX_SEARCH_DEPTH);
	
    /* Clean up */
    if (!abort_computation)
	gtk_widget_destroy(GTK_WIDGET(progress_box));
    if (under_key_stats) 
	free_stats(under_key_stats);

    key_copy(&best_key, &global_key);
    apply_global_key_text();
    update_key_labels();
    
    free_stats(text_stats);
    visible_text = GLOBAL_TRANSFORMED_TEXT;
    text_stats = make_stats(visible_text); free(visible_text);
//  printf("\nUp error: %f\n", text_stats->total_error);
}

