/* ui_callbacks.c - Definition of UI callbacks
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank.h"

/* External functions and variables */
extern float *slft_std;
extern float *bift_std;
extern float *trift_std;
extern key global_key;
extern char fromchar;
extern char tochar;
extern char *text;
extern GtkWidget *key_controls;
extern GtkWidget *textarea;
extern preprocess_options preprocess_opts;
extern stats *text_stats;
extern int abort_computation;

void key_shift_L(key *k);
void key_shift_R(key *k);
void key_reverse(key *k);
void key_invert(key *k);
void key_clear(key *k);
void key_complete(key *k);
void update_key_labels(void);
void apply_global_key_text(void);
GtkWidget *make_key_controls(void);
GtkWidget *make_OK_box(const char *text);
GtkWidget *make_preprocess_dialog(void);
GtkWidget *make_stats_display_window(stats *s);
stats *make_stats(char *text);
void free_stats(stats *the_stats);
void do_save_slft(float *slft);
void do_save_bift(float *bift);
void do_save_trift(float *trift);
int do_load (char *filename);
int do_save (char *filename);
char *do_preprocess(char *text);
void prepro_check_toggle(int option_number);
void do_break(void);
float *load_slft_std(char *filename);
float *load_bift_std(char *filename);
float *load_trift_std(char *filename);
GtkWidget *make_source_edit_dialog(void);
void init_global_key(void);

/* Make source = view */
void cb_set_source_from_view(GtkWidget *widget, gpointer gdata) {
    free(text);
    text = gtk_editable_get_chars(GTK_EDITABLE(textarea), 0, -1);
    init_global_key();
    update_key_labels(); 
}

/* Edit source */
void cb_open_edit_source_dialog(GtkWidget *widget, gpointer gdata) {
    gtk_widget_show(make_source_edit_dialog());
}

void cb_ok_edit_source(GtkWidget *widget, gpointer gdata) {
    free(text);
    text = gtk_editable_get_chars(GTK_EDITABLE(gdata), 0, -1);
    apply_global_key_text();
}

/* Help - about */
void cb_about(GtkWidget *widget, gpointer gdata) {
    gtk_widget_show(make_OK_box(STR_ABOUT));
}

/* Automatic breaking */
void cb_break(GtkWidget *widget, gpointer gdata) {
    /* Ensures bijective key - do_break should be able to handle otherwise...*/
    key_invert(&global_key); key_complete(&global_key);
    update_key_labels();
    
    do_break();
} 

gint cb_abort(GtkWidget *widget, gpointer gdata) {
    abort_computation = TRUE;
    while (gtk_grab_get_current())
	gtk_grab_remove(gtk_grab_get_current());
    return (FALSE);
}

/* Stats dialog */
void cb_open_stats_dialog(GtkWidget *widget, gpointer gdata) {
    GtkWidget *dialog;
    char *visible_text;
    /* Have stats recalculated here (calculate on demand) */
    visible_text = GLOBAL_TRANSFORMED_TEXT;
    free_stats(text_stats);
    text_stats = make_stats(visible_text);
    g_free(visible_text);
    dialog = make_stats_display_window(text_stats);
    gtk_widget_show(dialog);
}

void cb_save_slft(GtkWidget *widget, gpointer gdata) {
    do_save_slft(text_stats->slft);
    free(slft_std);
    slft_std = load_slft_std(DEFAULT_SLFT);
}

void cb_save_bift(GtkWidget *widget, gpointer gdata) {
    do_save_bift(text_stats->bift);
    free(bift_std);
    bift_std = load_bift_std(DEFAULT_BIFT);
}

void cb_save_trift(GtkWidget *widget, gpointer gdata) {
    do_save_trift(text_stats->trift);
    free(trift_std);
    trift_std = load_trift_std(DEFAULT_TRIFT);
}

void cb_sort_ft_display(GtkWidget *widget, gint col, gpointer gdata) {
    gtk_clist_set_sort_column(GTK_CLIST(widget), col);
    /* Sort up for the frequencies and down for the letters */
    if (!strcmp(STR_FREQUENCY, gtk_clist_get_column_title(GTK_CLIST(widget), col)) ||
        !strcmp(STR_STANDARD_FREQUENCY, gtk_clist_get_column_title(GTK_CLIST(widget), col)))
	gtk_clist_set_sort_type(GTK_CLIST(widget), GTK_SORT_DESCENDING);
    else
	gtk_clist_set_sort_type(GTK_CLIST(widget), GTK_SORT_ASCENDING);
    gtk_clist_sort(GTK_CLIST(widget));
}

/* Preprocessing dialog */
void cb_open_prepro_dialog (GtkWidget *widget, gpointer gdata) {
    GtkWidget *dialog = make_preprocess_dialog();
    gtk_widget_show(dialog);
}

void cb_preprocess(GtkWidget *widget, gpointer gdata) {
    text = do_preprocess(text);
    apply_global_key_text();
}

void cb_prepro_check_toggle(GtkWidget *widget, gpointer gdata) {
    prepro_check_toggle(GPOINTER_TO_INT(gdata));
}

/* Key control window */
gint cb_close_key_controls (GtkWidget *widget, gpointer gdata) {
    gtk_widget_hide(key_controls);
    return(TRUE);
}

void cb_open_key_controls (GtkWidget *widget, gpointer gdata) {
    gtk_widget_show(key_controls);
}

/* Key control button callbacks */
void cb_invert (GtkWidget *widget, gpointer gdata) {
    key_invert(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_shift_L (GtkWidget *widget, gpointer gdata) {
    key_shift_L(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_shift_R (GtkWidget *widget, gpointer gdata) {
    key_shift_R(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_reverse (GtkWidget *widget, gpointer gdata) {
    key_reverse(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_complete (GtkWidget *widget, gpointer gdata) {
    key_complete(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_clear (GtkWidget *widget, gpointer gdata) {
    key_clear(&global_key);
    update_key_labels();
    apply_global_key_text();
}

void cb_change (GtkWidget *widget, gpointer gdata) {
    global_key[(int)fromchar] = tochar;
    update_key_labels();
    apply_global_key_text();
}

void cb_change_from_select(GtkWidget *widget, gpointer gdata) {
    fromchar = GPOINTER_TO_INT(gdata); 
}

void cb_change_to_select(GtkWidget *widget, gpointer gdata) {
    tochar = GPOINTER_TO_INT(gdata);
}

/*  File Request Dialog Callbacks */
void cb_open(GtkWidget *widget, GtkFileSelection *fselection) {
    if (!do_load(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fselection))))
	gtk_widget_show(make_OK_box("ERROR: Unable to find or open file."));
}

void cb_save(GtkWidget *widget, GtkFileSelection *fselection) {
    if (!do_save(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fselection))))
	gtk_widget_show(make_OK_box("ERROR: Unable to save to file."));
}

/* Menu callbacks */
void cb_start_open (GtkWidget *widget, gpointer gdata) {
    GtkWidget *fopener;
    fopener = gtk_file_selection_new("Get source text");
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION (fopener)->cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(fopener));
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fopener)->ok_button),
		       "clicked",
		       (GtkSignalFunc) cb_open,
		       GTK_OBJECT(fopener));
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION (fopener)->ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(fopener));
    gtk_widget_show(fopener);
    
}

void cb_start_save (GtkWidget *widget, gpointer gdata) {
    GtkWidget *fopener;
    fopener = gtk_file_selection_new("Save viewed text");
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION (fopener)->cancel_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(fopener));
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fopener)->ok_button),
		       "clicked",
		       (GtkSignalFunc) cb_save,
		       GTK_OBJECT(fopener));
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION (fopener)->ok_button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(fopener));
    gtk_widget_show(fopener);
}

void cb_exit_crank (GtkWidget *widget, gpointer gdata) {
      gtk_main_quit();
}
