/*
** Copyright 1998 - 2003 Double Precision, Inc.
** See COPYING for distribution information.
**
** $Id: pop3dserver.c,v 1.20 2003/04/28 18:40:19 mrsam Exp $
*/

#if	HAVE_CONFIG_H
#include	"config.h"
#endif
#include	<sys/types.h>
#if HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#if HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#if HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#if HAVE_NDIR_H
#include <ndir.h>
#endif
#endif
#include	<sys/types.h>
#if	HAVE_SYS_STAT_H
#include	<sys/stat.h>
#endif
#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	<signal.h>
#include	<errno.h>
#include	"authlib/authmod.h"
#include	"numlib/numlib.h"
#include	"maildir/config.h"
#include	"maildir/maildirmisc.h"
#include	"maildir/maildirquota.h"
#include	"maildir/maildirgetquota.h"
#include	"maildir/maildircreate.h"

#define POP3DLIST "courierpop3dsizelist"

extern void pop3dcapa();

static const char *authaddr, *remoteip;

struct msglist {
	struct msglist *next;
	char *filename;
	int isdeleted;
	off_t size;
	} ;

static struct msglist *msglist_l;
static struct msglist **msglist_a;
static unsigned msglist_cnt;

static unsigned long top_count=0;
static unsigned long retr_count=0;

/*
** The RFC is pretty strict in stating that octet size must count the CR
** in the CRLF endofline.
*/

static void calcsize(struct msglist *m)
{
FILE	*f=fopen(m->filename, "r");
int	c, lastc;

	m->size=0;
	if (f == 0)
	{
		perror("pop3d");
		return;
	}
	lastc='\n';
	while ((c=getc(f)) >= 0)
	{
		if (c == '\n')	++m->size;
		++m->size;
		lastc=c;
	}
	if (lastc != '\n')	m->size += 2;

	if (ferror(f))
	{
		perror("pop3d");
		return;
	}
	fclose(f);
}


/*
** Read courierpop3dsizelist
*/

static int cmpmsgs(const void *a, const void *b);

static struct msglist **readpop3dlist()
{
	struct msglist **a;
	struct msglist *list=NULL;
	size_t mcnt=0;

	char linebuf[1024];

	FILE *fp=fopen(POP3DLIST, "r");

	size_t i;

	if (fp)
	{
		struct msglist *m;

		char *p;

		size_t n=0;
		int ch;

		while ((ch=getc(fp)) != EOF)
		{
			if (ch != '\n')
			{
				if (n < sizeof(linebuf)-1)
					linebuf[n++]=ch;
				continue;
			}
			linebuf[n]=0;
			n=0;

			if ((p=strrchr(linebuf, ' ')) == NULL)
				continue;
			*p++=0;
			if (linebuf[0] == 0)
				continue;

			if ((m=(struct msglist *)malloc(sizeof(struct
							       msglist))) == 0)
			{
				perror("malloc");
				exit(1);
			}

			if ((m->filename=strdup(linebuf)) == NULL)
			{
				perror("malloc");
				exit(1);
			}
			m->size=atol(p);
			m->next=list;
			list=m;
			++mcnt;
		}
		fclose(fp);
	}
	if ((a=(struct msglist **)malloc((mcnt+1)
					 *sizeof(struct msglist *))) == 0)
	{
		perror("malloc");
		exit(1);
	}

	for (i=0; list; list=list->next)
		a[i++]=list;

	a[i]=NULL;
	qsort(a, i, sizeof(*a), cmpmsgs);

	return a;
}

static void savepop3dlist(struct msglist **a, size_t cnt)
{
	FILE *fp;
	size_t i;

	struct maildir_tmpcreate_info createInfo;

	maildir_tmpcreate_init(&createInfo);

	createInfo.uniq="pop3";
	createInfo.doordie=1;

	if ((fp=maildir_tmpcreate_fp(&createInfo)) == NULL)
	{
		maildir_tmpcreate_free(&createInfo);
		return;
	}

	for (i=0; i<cnt; i++)
	{
		char *p=a[i]->filename;
		char *q;

		if ((q=strrchr(p, '/')) != NULL)
			p=q+1;

		fprintf(fp, "%s %lu\n", p, (unsigned long)a[i]->size);
	}

	if (fflush(fp) || ferror(fp))
	{
		fclose(fp);
		unlink(createInfo.tmpname);
		maildir_tmpcreate_free(&createInfo);
		return;
	}

	if (fclose(fp))
	{
		unlink(createInfo.tmpname);
		maildir_tmpcreate_free(&createInfo);
		return;
	}

	rename(createInfo.tmpname, POP3DLIST);
	maildir_tmpcreate_free(&createInfo);
}

/* Scan cur, and pick up all messages contained therein */

static void scancur()
{
DIR	*dirp;
struct	dirent *de;
struct	msglist *m;

	if ((dirp=opendir("cur")) == 0)
	{
		perror("pop3d");
		return;
	}

	while ((de=readdir(dirp)) != 0)
	{
		if ( de->d_name[0] == '.' )	continue;

		if ((m=(struct msglist *)malloc(sizeof(struct msglist))) == 0)
		{
			perror("malloc");
			exit(1);
		}
		if ((m->filename=(char *)malloc(strlen(de->d_name)+5)) == 0)
		{
			free( (char *)m);
			perror("malloc");
			exit(1);
		}
		strcat(strcpy(m->filename, "cur/"), de->d_name);
		m->isdeleted=0;
		m->next=msglist_l;
		msglist_l=m;
		msglist_cnt++;
	}
	closedir(dirp);
}

/*
** When sorting messages, sort on the arrival date - the first part of the
** name of the file in the maildir is the timestamp.
*/

static int cmpmsgs(const void *a, const void *b)
{
	const char *aname=(*(struct msglist **)a)->filename;
	const char *bname=(*(struct msglist **)b)->filename;
	const char *ap=strrchr(aname, '/');
	const char *bp=strrchr(bname, '/');
	long	na, nb;

	if (ap)
		++ap;
	else
		ap=aname;

	if (bp)
		++bp;
	else
		bp=bname;
 
	na=atol(ap);
	nb=atol(bp);

	if (na < nb)	return (-1);
	if (na > nb)	return (1);

	return strcmp(ap, bp);
}

static void sortmsgs()
{
	size_t i, n;
	struct msglist *m;
	struct msglist **prev_list;
	int savesizes=0;

	if (msglist_cnt == 0)	return;

	if ((msglist_a=(struct msglist **)malloc(
			msglist_cnt*sizeof(struct msglist *))) == 0)
	{
		perror("pop3d");
		msglist_cnt=0;
		return;
	}

	for (i=0, m=msglist_l; m; m=m->next, i++)
		msglist_a[i]=m;
	qsort(msglist_a, msglist_cnt, sizeof(*msglist_a), cmpmsgs);

	prev_list=readpop3dlist();

	n=0;

	for (i=0; i<msglist_cnt; i++)
	{
		while (prev_list[n] &&
		       cmpmsgs(&prev_list[n], &msglist_a[i]) < 0)
		{
			++n;
			savesizes=1;
		}

		if (prev_list[n] &&
		    cmpmsgs(&prev_list[n], &msglist_a[i]) == 0)
		{
			msglist_a[i]->size=prev_list[n]->size;
			n++;
		}
		else
		{
			calcsize(msglist_a[i]);
			savesizes=1;
		}      
	}

	if (prev_list[n])
		savesizes=1;

	for (i=0; prev_list[i]; i++)
	{
		free(prev_list[i]->filename);
		free(prev_list[i]);
	}

	free(prev_list);

	if (savesizes)
		savepop3dlist(msglist_a, msglist_cnt);
}

static void mkupper(char *p)
{
	while (*p)
	{
		*p=toupper(*p);
		p++;
	}
}

static void cleanup()
{
	unsigned i;

	long deleted_bytes=0;
	long deleted_messages=0;

	for (i=0; i<msglist_cnt; i++)
		if (msglist_a[i]->isdeleted)
		{
			unsigned long un=0;

			const char *filename=msglist_a[i]->filename;

			if (maildirquota_countfile(filename))
			{
				if (maildir_parsequota(filename, &un))
				{
					struct stat stat_buf;

					if (stat(filename, &stat_buf) == 0)
						un=stat_buf.st_size;
				}
			}

			if (unlink(msglist_a[i]->filename))
				un=0;

			if (un)
			{
				deleted_bytes -= un;
				deleted_messages -= 1;
			}
		}

	if (deleted_messages < 0)
		maildir_quota_deleted(".", deleted_bytes, deleted_messages);

	return;
}

/* POP3 STAT */

static void do_stat()
{
off_t	n=0;
unsigned i, j;
char	buf[NUMBUFSIZE];

	j=0;
	for (i=0; i<msglist_cnt; i++)
	{
		if (msglist_a[i]->isdeleted)	continue;
		n += msglist_a[i]->size;
		++j;
	}

	printf("+OK %u %s\r\n", j, libmail_str_off_t(n, buf));
	fflush(stdout);
}

static unsigned getmsgnum(const char *p)
{
unsigned i;

	if (!p || (i=atoi(p)) > msglist_cnt || i == 0 ||
		msglist_a[i-1]->isdeleted)
	{
		printf("-ERR Invalid message number.\r\n");
		fflush(stdout);
		return (0);
	}
	return (i);
}

/* POP3 LIST */

static void do_list(const char *msgnum)
{
unsigned i;
char	buf[NUMBUFSIZE];

	if (msgnum)
	{
		if ((i=getmsgnum(msgnum)) != 0)
		{
			printf("+OK %u %s\r\n", i,
				libmail_str_off_t(msglist_a[i-1]->size, buf));
			fflush(stdout);
		}
		return;
	}

	printf("+OK POP3 clients that break here, they violate STD53.\r\n");
	for (i=0; i<msglist_cnt; i++)
	{
		if (msglist_a[i]->isdeleted)	continue;
		printf("%u %s\r\n", i+1, libmail_str_off_t(msglist_a[i]->size, buf));
	}
	printf(".\r\n");
	fflush(stdout);
}

/* RETR and TOP POP3 commands */

static void do_retr(unsigned i, unsigned *lptr)
{
FILE	*f=fopen(msglist_a[i]->filename, "r");
char	*p;
int	c, lastc='\n';
int	inheader=1;
char	buf[NUMBUFSIZE];
unsigned long *cntr;

	if (!f)
	{
		printf("-ERR Can't open the message file - it's gone!\r\n");
		fflush(stdout);
		return;
	}
	printf( (lptr ? "+OK headers follow.\r\n":"+OK %s octets follow.\r\n"),
		libmail_str_off_t(msglist_a[i]->size, buf));

	cntr= &retr_count;
	if (lptr)
		cntr= &top_count;

	for (lastc=0; (c=getc(f)) >= 0; lastc=c)
	{
		if (lastc == '\n')
		{
			if (lptr)
			{
				if (inheader)
				{
					if (c == '\n')	inheader=0;
				}
				else if ( (*lptr)-- == 0)	break;
			}

			if (c == '.')
				putchar('.');
		}
		if (c == '\n')	putchar('\r');
		putchar(c);
		++*cntr;
	}
	if (lastc != '\n')	printf("\r\n");
	printf(".\r\n");
	fflush(stdout);
	fclose(f);
	if (lptr)	return;

	if ((p=strchr(msglist_a[i]->filename, MDIRSEP[0])) != 0 &&
		(p[1] != '2' || p[2] != ',' || strchr(p, 'S') != 0))
		return;

	if ((p=malloc(strlen(msglist_a[i]->filename)+5)) == 0)
		return;

	strcpy(p, msglist_a[i]->filename);
	if (strchr(p, MDIRSEP[0]) == 0)	strcat(p, MDIRSEP "2,");
	strcat(p, "S");

	if (lptr	/* Don't mark as seen for TOP */
	    || rename(msglist_a[i]->filename, p))
	{
		free(p);
		return;
	}
	free(msglist_a[i]->filename);
	msglist_a[i]->filename=p;
}

/*
** The UIDL of the message is really just its filename, up to the first MDIRSEP character
*/

static void print_uidl(unsigned i)
{
const char *p=strchr(msglist_a[i]->filename, '/')+1;

	while (*p && *p != MDIRSEP[0])
	{
		if (*p < 0x21 || *p > 0x7E || *p == '\'' || *p == '"' ||
			*p == '+')
			printf("+%02X", (int)(unsigned char)*p);
		else
			putchar(*p);
		++p;
	}
	printf("\r\n");
}

static void do_uidl(const char *msgnum)
{
unsigned i;

	if (msgnum)
	{
		if ((i=getmsgnum(msgnum)) != 0)
		{
			printf("+OK %u ", i);
			print_uidl(i-1);
			fflush(stdout);
		}
		return;
	}
	printf("+OK\r\n");
	for (i=0; i<msglist_cnt; i++)
	{
		if (msglist_a[i]->isdeleted)	continue;
		printf("%u ", i+1);
		print_uidl(i);
	}
	printf(".\r\n");
	fflush(stdout);
}

static void acctout(const char *disc)
{
	static const char msg2[]=", user=";
	static const char msg3[]=", ip=[";
	static const char msg4[]="], top=";
	static const char msg5[]=", retr=";
	char num1[NUMBUFSIZE];
	char num2[NUMBUFSIZE];
	char *p;

	libmail_str_size_t(top_count, num1);
	libmail_str_size_t(retr_count, num2);

	p=malloc(strlen(authaddr)+strlen(remoteip)+strlen(disc)+
		 strlen(num1)+strlen(num2)+200);	/* Should be enough */

	strcpy(p, disc);
	strcat(p, msg2);
	strcat(p, authaddr);
	strcat(p, msg3);
	strcat(p, remoteip);
	strcat(p, msg4);
	strcat(p, num1);
	strcat(p, msg5);
	strcat(p, num2);
	strcat(p, "\n");
	write(2, p, strlen(p));
	free(p);
}

static RETSIGTYPE bye(int signum)
{
	acctout("ERR: TIMEOUT");
	exit(0);
#if	RETSIGTYPE != void
	return (0);
#endif
}

static void loop()
{
char	buf[BUFSIZ];
char	*p;
int	c;

	signal(SIGALRM, bye);
	while (alarm(300), fgets(buf, sizeof(buf), stdin))
	{
		alarm(0);
		if ((p=strchr(buf, '\n')) != 0)
			*p=0;
		else while ((c=getc(stdin)) >= 0 && c != '\n')
			;
		p=strtok(buf, " \t\r");
		if (!p)	p="";

		mkupper(p);
		if (strcmp(p, "QUIT") == 0)
		{
			printf("+OK Bye-bye.\r\n");
			fflush(stdout);
			cleanup();
			acctout("INFO: LOGOUT");
			return;
		}

		if (strcmp(p, "STAT") == 0)
		{
			do_stat();
			continue;
		}

		if (strcmp(p, "LIST") == 0)
		{
			do_list(strtok(NULL, " \t\r"));
			continue;
		}

		if (strcmp(p, "RETR") == 0)
		{
		unsigned	i;

			if ((i=getmsgnum(strtok(NULL, " \t\r"))) == 0)
				continue;

			do_retr(i-1, 0);
			continue;
		}

		if (strcmp(p, "CAPA") == 0)
		{
			pop3dcapa();
			continue;
		}

		if (strcmp(p, "DELE") == 0)
		{
		unsigned	i;

			if ((i=getmsgnum(strtok(NULL, " \t\r"))) == 0)
				continue;

			msglist_a[i-1]->isdeleted=1;
			printf("+OK Deleted.\r\n");
			fflush(stdout);
			continue;
		}

		if (strcmp(p, "NOOP") == 0)
		{
			printf("+OK Yup.\r\n");
			fflush(stdout);
			continue;
		}

		if (strcmp(p, "RSET") == 0)
		{
		unsigned i;

			for (i=0; i<msglist_cnt; i++)
				msglist_a[i]->isdeleted=0;
			printf("+OK Resurrected.\r\n");
			fflush(stdout);
			continue;
		}

		if (strcmp(p, "TOP") == 0)
		{
		unsigned	i, j;
		const	char *q;

			if ((i=getmsgnum(strtok(NULL, " \t\r"))) == 0)
				continue;

			q=strtok(NULL, " \t\r");

			if (!q)	goto error;

			j=atoi(q);
			do_retr(i-1, &j);
			continue;
		}

		if (strcmp(p, "UIDL") == 0)
		{
			do_uidl(strtok(NULL, " \t\r"));
			continue;
		}

error:
		printf("-ERR Invalid command.\r\n");
		fflush(stdout);
	}
	acctout("ERR: DISCONNECTED");
}

/* Like every good Maildir reader, we purge the tmp subdirectory */

static void purgetmp()
{
DIR	*p=opendir("tmp");
time_t	t;
struct	dirent *de;
struct	stat	stat_buf;
char	*n;

	if (!p)	return;
	time (&t);
	t -= 48L * 60L * 60L;

	while ((de=readdir(p)) != 0)
	{
		if (de->d_name[0] == '.')	continue;
		n=malloc(strlen(de->d_name)+5);
		if (!n)	continue;
		strcat(strcpy(n, "tmp/"), de->d_name);
		if (stat(n, &stat_buf) == 0 && stat_buf.st_mtime < t)
			unlink(n);
		free(n);
	}
	closedir(p);
}

static void scannew()
{
DIR	*p=opendir("new");
struct	dirent *de;
char	*n, *o;

	if (!p)	return;

	while ((de=readdir(p)) != 0)
	{
		if (de->d_name[0] == '.')	continue;
		n=malloc(strlen(de->d_name)+5);
		if (!n)	continue;
		o=malloc(strlen(de->d_name)+8);
		if (!o)	continue;
		strcat(strcpy(n, "new/"), de->d_name);
		strcat(strcpy(o, "cur/"), de->d_name);

		if (strchr(o, MDIRSEP[0]) == NULL)
			strcat(o, MDIRSEP "2,");
		rename(n, o);
		free(n);
		free(o);
	}
	closedir(p);
}

#include	<unistd.h>

int main(int argc, char **argv)
{
char	*p;

	authmodclient();

	if ((authaddr=getenv("AUTHADDR")) == 0 ||
		(remoteip=getenv("TCPREMOTEIP")) == 0)
	{
		fprintf(stderr, "ERROR: Required environment variables not initialized.\n");
		fflush(stderr);
		exit (1);
	}

	{
	struct	stat	buf;

		if ( stat(".", &buf) < 0 || buf.st_mode & S_ISVTX)
		{
			fprintf(stderr, "INFO: LOCKED, user=%s, ip=[%s]\n",
				authaddr, remoteip);
			printf("-ERR Your account is temporarily unavailable (+t bit set on home directory).\r\n");
			exit(0);
		}
	}

	if ((p=getenv("MAILDIR")) != 0 && *p)
	{
		if (chdir(p))
		{
			printf("-ERR Maildir: %s\r\n",strerror(errno));
			exit(1);
		}
	}
	else if (argc > 1 && chdir(argv[1]))
	{
		printf("-ERR Maildir: %s\r\n",strerror(errno));
		exit(1);
	}

	fprintf(stderr, "INFO: LOGIN, user=%s, ip=[%s]\n",
			authaddr,
			remoteip);
	fflush(stderr);

	msglist_cnt=0;
	msglist_l=0;
	msglist_a=0;
	purgetmp();
	scannew();
	scancur();
	sortmsgs();
	printf("+OK logged in.\r\n");
	fflush(stdout);
	loop();
	return (0);
}
